.\"
.\" The .zZ macro is used to define whether or not the mesh-specification
.\" is printed out.  If .zZ=1 skip mention of the mesh, otherwise, print
.\" the mesh specification.
.\"
.nr zZ 1
.TL
THE MTV Plot Data Format
.AU
Kenny K.H. Toh
.AI
Visualization Working Group,
TCAD Department,
Intel Corporation
3/17/93
.SH
.ce
INTRODUCTION
.LP
The MTV plot data format, or MTVDAT format, is used to specify data 
to be plotted in the MTV
visualization program.  The most important feature of this format is that
multiple types of data may be specified in a single file or data-stream, 
thus avoiding the
proliferation of data-files of various formats. 
As of this date, the MTVDAT format includes the following types of data:
.RS
.IP "\fB\(bu 2D Curves\fP"
Connected sets of (x,y) points in 2D space
.IP "\fB\(bu 3D Curves\fP"
Connected sets of (x,y,z) points in 3D space
.IP "\fB\(bu 3D Contour Curves\fP"
3D Triangles/rectangles to be contoured
.IP "\fB\(bu 3D rectangular mesh data\fP"
Z-values on a rectangular x-y grid, used primarily for contours
.IP "\fB\(bu 4D volumetric data\fP"
T-values on a rectangular x-y-z grid
.IP "\fB\(bu 2D/3D Vector data\fP"
Vectors in 2D/3D space
.if \n(zZ=1 .ig zA
.IP "\fB\(bu Hierarchical mesh data\fP"
Hierarchical collection of points, nodes, triangles and quantities used to
represent quantities on a mesh structure.
.zA
.RE
.LP
The MTV format also allows the inclusion of plot-options with the data,
so that the properties of the plot may be customized.
.LP
The MTV data-format is presently limited to ASCII and machine-dependent 
BINARY data.
.bp
.SH
.ce 
The MTVDAT Format
.SH
Overview of the MTVDAT format 
.LP
Data in the MTVDAT format is organized into sets of data, or \fBdatasets\fP.
There are different types of datasets, corresponding to the various types of
data that can be represented in the MTVDAT format, e.g. CURVE2D datasets, 
CONTOUR datasets.  
.LP
Datasets are identified using \fBcommand\fP lines.  An \fBcommand\fP
line is of the format "$ DATA=<TYPE>"; 
the "$" character is the first non-blank character 
and is followed by a keyword specifying the type of data to be read.
The \fBcommand\fP line is followed by the plot data, written in the
appropriate format.
The dataset ends with the next \fBcommand\fP line, 
or when a "$ END"
line is encountered in the data-stream.  An end-of-file (EOF) will also
signify the end of the dataset, if reading from a data-file.
Thus, using \fBcommand\fP lines, multiple datasets and multiple types of
datasets can be included in a single file or data-stream.
.LP
Each dataset is assigned a number of properties, which instruct the plotting
program on how to treat the data mathematically, and on how to construct the
plot of the dataset.  For example, in order to plot contours of a
rectangular mesh, the plot program needs to know the contour intervals as
well as labeling information for the plot.  
These plot resources are by default
automatically assigned by the MTVDAT reader.
However, the MTVDAT format also
allows these resources to be set using optional \fBinstruction\fP lines.
\fBInstruction\fP lines are identified
by a percent "%" sign as the first non-blank character in the line.
.LP
Comments can also be inserted in the data-stream; comments are preceded by
the pound "#" character.
Plot annotations can also be included - see Appendix B for more details.
.LP
.if \n(zZ=0 .so mtv_ident1.tms
.if \n(zZ=1 .so mtv_ident2.tms
.IP "\fBTable 1:\fP" 9
The MTVDAT format uses special \fBcommand\fP and
\fBinstruction\fP lines to specify datasets and their properties.
.sp
.LP
Both \fBcommand\fP and \fBinstruction\fP lines consist of <argument>=<value>
pairs, that is, an argument or keyword separated from a value by an optional 
equal sign "=".  In general, the argument of the <argument>=<value> pair is
case-insensitive while the value string is case-sensitive.
Multiple <argument>=<value> pairs are can be specified on a single line,
provided that they are separated by spaces or tabs.
Both types of lines are terminated 
either by a "#" character or by the
end-of-line (EOLN) character. 
Furthermore, \fBcommand\fP and \fBinstruction\fP lines can be continued on a 
new line using the backslash "\e" character.
.bp
.SH
COMMANDS: Identifying Datasets
.LP
Several types of datasets are currently supported, identified in the following
manner:
.KS
.DS
$ DATA = CURVE2D
$ DATA = CURVE3D
$ DATA = COLUMN
$ DATA = CONTCURVE
$ DATA = CONTOUR
$ DATA = GRID4D
$ DATA = VECTOR
.if \n(zZ=0 \{\
$ DATA = MESH
\}
.DE
.KE
Detailed descriptions and formatting information for each type of dataset 
are provided in the following sections.
.LP
As described earlier, a single file or data-stream 
can contain one or more datasets, 
each preceded by a \fBcommand\fP line. 
Thus, a data-file in the MTVDAT
format has the following structure:
.KS
.DS
$ DATA = CURVE2D
   [optional properties]
   [2D curve data]
$ DATA = CURVE3D
   [optional properties]
   [3D curve data]
$ DATA = COLUMN
   [optional properties]
   [2D curve data in multiple columns]
$ DATA = CONTCURVE
   [optional properties]
   [triangles/rectangles specification]
$ DATA = CONTOUR 
   [optional properties]
   [rectangular grid specification]
$ DATA = GRID4D
   [optional properties]
   [4D rectangular grid specification]
$ DATA = VECTOR
   [optional properties]
   [vector specification]
.if \n(zZ=0 \{\
$ DATA = MESH
   [optional properties]
   [hierarchical list of points, nodes, triangles, etc]
\}
$ END
.DE
.KE
.LP
The \fBcommand\fP line forces the MTV reader to invoke a special
reader for each type of data.  Thus, a "$ DATA=CURVE2D" \fBcommand\fP tells the
MTV reader to expect data in a CURVE2D format, which is essentially 2
columns of data.  The data read in is saved in a dataset structure for later
plotting.
.KS
.LP
Datasets can also be identified using unique character strings, by including
the "NAME=<label>" argument in the \fBcommand\fP line.  The following example 
defines two CURVE2D datasets named "Curve A" and "Curve B" respectively.
.TS
box, center;
l.
$ DATA=CURVE2D NAME="Curve A"
  0 0
  1 1 
  2 2
$ DATA=CURVE2D NAME="Curve B"
  1 0
  2 0 
  3 0
$ END
.TE
.KE
.LP
Note that with the exception of the string used for the dataset name, the
\fBcommand\fP line is case-insensitive. Furthermore, as mentioned
previously,
a \fBcommand\fP can be
continued on a new line using the backslash "\e" character.  
Thus the following \fBcommands\fP are identical:
.KS
.DS
$ DATA=CURVE2D NAME="Curve A"
$ Data = Curve2d name = "Curve A"
$ data = curve2d \e
  name = "Curve A"
.DE
.KE
.SH
INSTRUCTIONS: Customizing the Plot
.LP
The properties of the data or the intended plot
can also be customized by including \fBinstruction\fP lines 
along with each dataset specification.
A single \fBinstruction\fP line contains one or more <argument>=<value> pairs,
separated by spaces or tabs.
The list of acceptable <argument>=<value> keywords depends on the type of
data being read; refer to \fBAppendix A\fP for a complete list of \fBPlot
Resources\fP.
.LP
The example below demonstrates the use of \fBinstruction\fP 
lines to set line-types
and labels for curves.
.KS
.TS
box, center;
l.
$ DATA=CURVE2D NAME="Curve A"
#
# First curve
#
% linetype=1 linewidth=2 linelabel="Cv 1"
  0 0           # The data starts here
  1 1 
  2 2
$ DATA=CURVE2D NAME="Curve B"
#
# First curve
#
% linetype=1 linewidth=3 linelabel="Cv 2"
% markertype=3  # Draw the markers too
  1 0           # Data
  2 0 
  3 0
$ END
.TE
.KE
.LP
Note in the above that the \fBcommand\fP line is similar in
form to the \fBinstruction\fP line, 
the primary difference being that the 
\fBcommand\fP line begins with a "$" character and the 
\fBinstruction\fP line begins
with a "%" character.  
.bp
.SH
.ce
CURVE2D FORMAT
.SH
Overview
.LP
The CURVE2D format is used to specify a list of joined points in 2D (x,y)
space.  A CURVE2D dataset may contain one or more curves, each of which
contain one or more points.
.SH
Resources
.LP
The CURVE2D format accepts the standard list of \fBPlot Resources\fP, which
are listed in \fBAppendix A\fP.  By default, each curve in the dataset 
is plotted in a continuous solid line and individual points are not marked;
use \fBCurve Resources\fP for changing these and other curve properties.
In addition, the CURVE2D format accepts the following resources:
.KS
.TS
box, tab(|), center;
cfB s s s
lfB | lfB | lfB | lfB
l   | l   | l   | l.
CURVE2D Property Argument List
_
Name|Type|Default|Description
=
binary|Boolean|False|Machine-dependent binary
_
npts|int|-|Binary array size
.TE
.KE
.IP "\fBbinary\fP" 12
Specifies whether the curve data is written in ASCII or machine-dependent
binary format.  If \fBTrue\fP, the program tries to read 2 double-precision
binary arrays of size \fBnpts\fP 
containing the x and y coordinates of the points in the curve.
The arrays are assumed to be on the line following the \fIbinary=True\fP
\fBinstruction\fP.  Default is \fBFalse\fP.
.IP "\fBnpts\fP" 12
Specifies the number of points in a curve.  Used only for reading data in
binary format.
.SH
ASCII Format Specification
.LP
Each curve consists of a number of (x,y) points.
The coordinates and optional integer identifier (ID) of the point must be
specified on the same line as follows:
.DS
x-coordinate y-coordinate [point-ID]
.DE
Note in the above that the various fields are space or tab-separated.
Each field is format-free, meaning that the numbers may be specified
using decimals, exponential or even integer formats.
.LP
The points on each curve are specified on adjacent lines.
Multiple curves are separated by one or more blank lines.
The CURVE2D format thus has the following form:
.DS
$ DATA=CURVE2D
 
% [optional instructions]
 
# ASCII Curve
% [optional instructions]
x1 y1   #[point 1, curve 1]
x2 y2   #[point 2, curve 1]
 ...
xn yn   #[point n, curve 1]

# ASCII Curve
 ...
$ END
.DE
.KS
.LP
The following shows an example file containing two CURVE2D datasets; 
the first dataset has three curves while the second has two curves.
.DS 
.B1
.sp
.so ex1.mtvdat 
.sp
.B2
.DE
.KE
.bp
.SH
Binary Format Specification
.LP
Curves may also be specified in a machine-dependent binary format;
this format will work only for files produced and read on the same
computing platform.  The machine-dependence is a disadvantage, but it is
balanced by the fact that reading binary data is can be up to 10 times
faster than reading ASCII data.  Note also that when programs are exchanging
data using pipes, machine-dependence becomes a moot issue, since the
pipe-linked programs must be on the same machine in order for pipes to
work.
.LP
Curve binary data is identified by the following \fBinstruction\fP line:
.DS
% binary=True npts=<value>
.DE
In the above, \fInpts\fP is the number of points in the curve.
The binary data-stream must begin on the line immediately following the 
\fBinstruction\fP line above.  The binary stream consists of 2 arrays of
type double-precision which contain x and y coordinate data
respectively.
Each array is assumed to be of size \fInpts\fP. Note that the binary format
does not provide a means of specifying the point IDs.
.LP
Binary and ASCII curves may be mixed in the same file or data-stream.
However, a single curve must be either totally ASCII or totally binary;
ASCII and binary formats may not be mixed in a single curve.
The CURVE2D format combining both ASCII and binary curves 
thus has the following form:
.DS
$ DATA=CURVE2D

% [optional instructions]

# ASCII Curve
% [optional instructions]
x1 y1   #[point 1, curve 1]
x2 y2   #[point 2, curve 1]
 ...
xn yn   #[point n, curve 1]
 
# Binary Curve
% [optional instructions]
% binary=True npts=<number>
[x-coordinate binary double-precision array]\(dg
[y-coordinate binary double-precision array]

# ASCII Curve
 ...

# Binary Curve
 ...
$ END
.DE
.FS \(dg
The square brackets in this case denote an array; 
the brackets are not to be written out
to the data-stream.
.FE

.KS
.SH
Programming Example
.LP
The following short program illustrates writing binary data in the CURVE2D
format.
.LP
.TS
box center;
l.
#include <stdio.h>
#include <math.h>
main()
{
#define BINARYFILE "data.mtvdat"
   FILE   *fp;
   double xarr[100], yarr[100];
   int    i, npts=100;
 
   /* Open up a file */
   if ((fp=fopen(BINARYFILE,"w")) == NULL) {
      (void) fprintf(stderr,"cat: Couldn't open file %s\en",BINARYFILE); return;
   }
  
   /* Fill arrays */
   for (i=0; i<npts; i++) {
      xarr[i] = 0.1*i; 
      yarr[i] = cos(xarr[i]); 
   }

   /* Write out CURVE2D header */
   (void) fprintf(fp,"$ DATA=CURVE2D\en");
 
   /* Write out binary data */
   (void) fprintf(fp,"%% BINARY npts=%d\en",npts);
   if (fwrite((char *)xarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return;
   }
   if (fwrite((char *)yarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return;
   }
   (void) fprintf(fp,"$ END\en");
 
   /* Close the file */
   (void) fclose(fp);
}
.TE
.KE
.bp
.SH
.ce
CURVE3D FORMAT
.SH
Overview
.LP
The CURVE3D format is used to specify a list of joined points in 3D (x,y,z)
space.  A CURVE3D dataset may contain one or more curves, each of which
contain one or more points.  Note that the CURVE3D 
format is similar to the CURVE2D
format except for the addition of an extra coordinate.
.SH
Resources
.LP
The CURVE3D format accepts the standard list of \fBPlot Resources\fP, which
are listed in \fBAppendix A\fP.  By default, each curve in the dataset 
is plotted in a continuous solid line and individual points are not marked;
use \fBCurve Resources\fP for changing these and other curve properties.
In addition, the CURVE3D format accepts the following resources:
.KS
.TS
box, tab(|), center;
cfB s s s
lfB | lfB | lfB | lfB
l   | l   | l   | l.
CURVE3D Property Argument List
_
Name|Type|Default|Description
=
binary|Boolean|False|Machine-dependent binary
_
npts|int|-|Binary array size
.TE
.KE
.IP "\fBbinary\fP" 12
Specifies whether the curve data is written in ASCII or machine-dependent
binary format.  If \fBTrue\fP, the program tries to read 3 double-precision
binary arrays of size \fBnpts\fP
containing the x, y and z coordinates of the points in the curve.
The arrays are assumed to be on the line following the \fIbinary=True\fP
\fBinstruction\fP.  Default is \fBFalse\fP.
.IP "\fBnpts\fP" 12
Specifies the number of points in a curve.  Used only for reading data in
binary format.
.SH
ASCII Format Specification
.LP
Each curve consists of a number of (x,y,z) points.
The coordinates and optional integer identifier (ID) of the point must be
specified on the same line as follows:
.DS
x-coordinate y-coordinate z-coordinate [point-ID]
.DE
Note in the above that the various fields are space or tab-separated.
Each field is format-free, meaning that the numbers may be specified
using decimals, exponential or even integer formats.
.LP
The points on each curve are specified on adjacent lines.
Multiple curves are separated by one or more blank lines.
The ASCII CURVE3D format thus has the following form:
.DS
$ DATA=CURVE3D
 
# ASCII Curve
% [optional instructions]
x1 y1 z1  #[point 1, curve 1]
x2 y2 z2  #[point 2, curve 1]
 ...
xn yn zn  #[point n, curve 1]
 
# ASCII Curve
 ...
$ END
.DE
.KS
.LP
The following shows an example file containing a CURVE3D dataset.
This dataset has five filled curves which make up the surfaces of a pyramid.
.DS 
.B1
.sp
.so ex2.mtvdat 
.sp
.B2
.DE
.KE
.bp
.SH
Binary Format Specification
.LP
Curves may also be specified in a machine-dependent binary format;
this format will work only for files produced and read on the same
computing platform.  The machine-dependence is a disadvantage, but it is
balanced by the fact that reading binary data is can be up to 10 times
faster than reading ASCII data.  Note also that when programs are exchanging
data using pipes, machine-dependence becomes a moot issue, since the
pipe-linked programs must be on the same machine in order for pipes to
work.
.LP
Curve binary data is identified by the following \fBinstruction\fP line:
.DS
% binary=True npts=<number-of-points>
.DE
In the above, \fInpts\fP is the number of points in the curve.
The binary data-stream must begin on the line immediately following the 
\fBinstruction\fP line above.  The binary stream consists of 3 arrays of
type double-precision which contain x, y, and z coordinate data
respectively.
Each array is assumed to be of size \fInpts\fP. Note that the binary format
does not provide a means of specifying the point IDs.
.LP
Binary and ASCII curves may be mixed in the same file or data-stream.
However, a single curve must be either totally ASCII or totally binary;
ASCII and binary formats may not be mixed in a single curve.
The CURVE3D format thus has the following form:
.DS
$ DATA=CURVE3D

% [optional instructions]

# ASCII Curve
% [optional instructions]
x1 y1 z1  #[point 1, curve 1]
x2 y2 z2  #[point 2, curve 1]
 ...
xn yn zn  #[point n, curve 1]

# Binary Curve
% [optional instructions]
% binary=True npts=<number>
[x-coordinate binary double-precision array]\(dg
[y-coordinate binary double-precision array]
[z-coordinate binary double-precision array]

# ASCII Curve
 ...

# Binary Curve
 ...
$ END
.DE
.FS \(dg
The square brackets in this case denote an array; 
the brackets are not to be written out
to the data-stream.
.FE
.KS
.SH
Programming Example
.LP
The following short program illustrates writing binary data in the CURVE3D
format.
.LP
.TS
box center;
l.
#include <stdio.h>
#include <math.h>
main()
{
#define BINARYFILE "data.mtvdat"
   FILE   *fp;
   double xarr[100], yarr[100], zarr[100];
   int    i, npts=100;
 
   /* Open up a file */
   if ((fp=fopen(BINARYFILE,"w")) == NULL) {
      (void) fprintf(stderr,"cat: Couldn't open file %s\en",BINARYFILE); return;
   }
  
   /* Fill arrays */
   for (i=0; i<npts; i++) {
      xarr[i] = 0.1*i; 
      yarr[i] = cos(xarr[i]); 
      zarr[i] = sin(xarr[i]);
   }

   /* Write out CURVE3D header */
   (void) fprintf(fp,"$ DATA=CURVE3D\en");
 
   /* Write out binary data */
   (void) fprintf(fp,"%% BINARY npts=%d\en",npts);
   if (fwrite((char *)xarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return;
   }
   if (fwrite((char *)yarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return;
   }
   if (fwrite((char *)zarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return;
   }
   (void) fprintf(fp,"$ END\en");
 
   /* Close the file */
   (void) fclose(fp);
}
.TE
.KE
.bp
.SH
.ce
COLUMN FORMAT
.SH
Overview
.LP
The COLUMN format specifies 2D curves in multiple columns.  The format can
be considered a superset of the CURVE2D format; each column represents a
single CURVE2D dataset, so multiple CURVE2D datasets
can be constructed from the COLUMN data. 
.SH
Resources
.LP
The COLUMN format accepts the standard list of \fBPlot Resources\fP, which
are listed in \fBAppendix A\fP.  By default, each curve in the dataset 
is plotted in a continuous solid line and individual points are not marked;
use \fBCurve Resources\fP for changing these and other curve properties.
In addition, the COLUMN format accepts the following resources:
.KS
.TS
box, tab(|), center;
cfB s s s
lfB | lfB | lfB | lfB
l   | l   | l   | l.
COLUMN Property Argument List
_
Name|Type|Default|Description
=
xcolumn|String|NULL|x-coordinate column
.TE
.KE
.IP "\fBxcolumn\fP" 12
Specifies the column containing the x-coordinate data. If not specified,
the x-column is assumed to be the first column. See the Format Specification
for more details.
.SH
ASCII Format Specification
.LP
The COLUMN format has the following form:
.DS
$ DATA=COLUMN
 
% [optional instructions]
 
% xcolumn = "column i"
"column 1" "column 2" ... "column i" ... "column m"  # column labels

# 1st set of curves (ASCII)
% [optional instructions]
    y1.1       y2.1           yi.1           yn.1
    y1.2       y2.2           yi.2           yn.2
    ...
    y1.n       y2.n           yi.n           yn.n

# 2nd set of curves (ASCII)
% [optional instructions]
    y1.1       y2.1           yi.1           yn.1
    y1.2       y2.2           yi.2           yn.2
    ...
    y1.n       y2.n           yi.n           yn.n

# More curves (ASCII)
    ...
.DE
.LP
As shown above, the column format consists of multiple columns of data.
Each column is identified with a string label.  One of the columns is the
x-ordinate; the other columns are the various y-ordinates. 
Each data-point in all the columns is specified on the
same line as follows (assuming the 1st column is the x-column):
.DS
x-coordinate y1-coordinate y2-coordinate ... yn-coordinate
.DE
Note in the above that the various fields are space or tab-separated.
Each field is format-free, meaning that the numbers may be specified
using decimals, exponential or even integer formats.
.LP
Each column, when paired with the x-column, forms a set of curves which 
is stored in a CURVE2D dataset. Thus, given N columns, N-1 CURVE2D datasets
are generated.  
As in the CURVE2D format, 
the points on each curve are specified on adjacent lines.
Multiple curves are separated by one or more blank lines.
.LP
The following shows an example file containing a COLUMN dataset. The dataset
contains 3 columns, so 2 CURVE2D datasets are generated when the data is
read in.
.DS 
.B1
.sp
.so ex6a.mtvdat 
.sp
.B2
.DE
.KE
.KS
.LP
The next example shows a similar column dataset. In this case, the x-column
is the third column, labeled "DataC". 
.DS
.B1
.sp
.so ex6b.mtvdat
.sp
.B2
.DE
.KE
.KE
.bp
.SH
.ce
CONTCURVE FORMAT
.SH
Overview
.LP
The CONTCURVE format specifies a 3-dimensional surface as a collection of 3D
triangles. The format is 
identical to that of the CURVE3D format; refer to the section of the CURVE3D
format for full details on the format specification. 
.LP
Curves specified in the CONTCURVE format are converted into triangles 
which are then sliced to obtain contours of the surface.
discarded.
Curves with 3 points become triangles, while curves with greater then 3
points are triangulated.\(dd
.FS \(dg
Note that the triangulation routine has not been perfected yet; the routine
fails when passed 2 or more data-points sharing the same x-y coordinates.
Furthermore, the triangles generated are sometimes obtuse, resulting in
strange plots. 
.FE
.LP
Once the data has been read in, the 3D mesh is sliced to obtain contours;
each contour is a curve of iso-z-values.  By default, approximately
10 contour lines are
plotted at constant z-intervals; the number of contours, their intervals and
plot style (i.e. lines, gradated colors, mesh) may be set using \fBDataset
Properties\fP (see Appendix A).
.SH
Resources
.LP
The CONTCURVE format accepts the standard list of \fBPlot Resources\fP, 
as well as the following:
.KS
.TS
box, tab(|), center;
cfB s s s
lfB | lfB | lfB | lfB
l   | l   | l   | l.
CONTCURVE Property Argument List
_
Name|Type|Default|Description
=
binary|Boolean|False|Machine-dependent binary
_
npts|int|-|Binary array size
.TE
.KE
.IP "\fBbinary\fP" 12
Specifies whether the curve data is written in ASCII or machine-dependent
binary format.  If \fBTrue\fP, the program tries to read 3 double-precision
binary arrays of size \fBnpts\fP
containing the x, y and z coordinates of the points in the curve.
The arrays are assumed to be on the line following the \fIbinary=True\fP
\fBinstruction\fP.  Default is \fBFalse\fP.
.IP "\fBnpts\fP" 12
Specifies the number of points in a curve.  Used only for reading data in
binary format.
.SH
Format Specification
.LP
Each curve consists of a number of (x,y,z) points.
The points on a curve may be specified either in ASCII or in binary.
The ASCII format requires the coordinates of the points in the curve
to be listed one-per-line as follows:
.DS
x-coordinate y-coordinate z-coordinate [point-ID]  # point 0
x-coordinate y-coordinate z-coordinate [point-ID]  # point 1
 ...
x-coordinate y-coordinate z-coordinate [point-ID]  # point n
.DE
In contrast, the binary format has the following form:
.DS
% binary=True npts=<number-of-points>
[x-coordinate binary double-precision array]\(dg
[y-coordinate binary double-precision array]
[z-coordinate binary double-precision array]
.DE
.FS \(dg
The square brackets in this case denote an array; 
the brackets are not to be written out
to the data-stream.
.FE
where the \fBinstruction\fP line containing the \fIbinary\fP flag is
immediately followed by arrays containing x, y, and z coordinate data.
.KS
.LP
Multiple curves are separated by one or more blank lines.
The CONTCURVE format thus has the following form:
.DS
$ DATA=CONTCURVE
 
# ASCII Curve
% [optional instructions]
x1 y1 z1  #[point 1, curve 1]
x2 y2 z2  #[point 2, curve 1]
 ...
xn yn zn  #[point n, curve 1]
 
# Binary Curve
% [optional instructions]
% binary=True npts=<number-of-points>
[x-coordinate binary double-precision array]\(dg
[y-coordinate binary double-precision array]
[z-coordinate binary double-precision array]
 
# ASCII Curve
 ...

# Binary Curve
 ...
$ END
.DE
.KE
.FS \(dg
The square brackets in this case denote an array; 
the brackets are not to be written out
to the data-stream.
.FE
.KS
.LP
The following shows an example file containing a CONTCURVE dataset.
This dataset has five filled curves which make up the surfaces of a pyramid.
The curves are stored as 6 triangles.
.DS 
.B1
.sp
.so ex3.mtvdat 
.sp
.B2
.DE
.KE
.bp
.SH
.ce
CONTOUR FORMAT
.SH
Overview
.LP
The CONTOUR format specifies a 3-dimensional surface on a rectangular grid.
The data consists of a grid-specification, denoting the 
points on the x-y grid, followed by the main body, consisting 
of z-values at the nodes of the x-y grid.
The surface z-values are assumed to be single-valued functions of x and y.
The x-y rectangular grid may be a uniform or nonuniform grid, i.e. the
spacing of the grid in x and y does not necessarily have to be uniform.
A single dataset may contain only one 3D mesh.
.LP
Once the data has been read in, the 3D mesh is sliced to obtain contours;
each contour is a curve of iso-z-values.  By default, approximately
10 contour lines are
plotted at constant z-intervals; the number of contours, their intervals and
plot style (i.e. lines, gradated colors, mesh) may be set using \fBDataset
Properties\fP (see Appendix A).
.SH
Resources
.LP
The CONTOUR format accepts all of the \fBPlot Resources\fP listed in
\fBAppendix A\fP, except for the \fBCurve Resources\fP and the \fBGlobal
Curve Resources\fP.\(dg
.FS \(dg
At present, it is not possible to manually set 
curve properties (e.g. linetype, linewidth) of
the contour curves derived from a CONTOUR dataset.
.FE
In addition, the CONTOUR
dataset accepts the following resources (specified in \fBinstruction\fP
lines):
.KS
.TS
box, tab(|), center;
cfB s s s
lfB | lfB | lfB | lfB
l   | l   | l   | l.
CONTOUR Property Argument List
_
Name|Type|Default|Description
=
nx|int|-|Number of Grid Points in x (required)
_
xmin|double|0.0|Uniform Grid boundary
_
xmax|double|nx|Uniform Grid boundary
_
xgrid|Boolean|False|Non-Uniform Grid
_
ny|int|-|Number of Grid Points in y (required)
_
ymin|double|0.0|Uniform Grid boundary 
_
ymax|double|ny|Uniform Grid boundary 
_
ygrid|Boolean|False|Non-Uniform Grid
_
binary|Boolean|False|Machine-dependent binary
_
joincurve|integer|0|Join contours at boundary
      |||0=Don't join
      |||1=Join with boundary at zmin
      |||2=Join with boundary at zmax
_
interp|integer|0|Contour interpolation style
      |||0=Interpolate from rectangle
      |||1=Interpolate from 2 triangles
      |||2=Interpolate from 4 triangles
      |||3=Don't Interpolate 
.TE
.KE
.IP "\fBnx\fP" 12
Specifies the number of grid-points on the x-axis.  Minimum value is 2.
Required for both uniform and non-uniform grid specification.
.IP "\fBxmin\fP" 12
.IP "\fBxmax\fP" 12
Specifies the limits of the grid in x. For a uniform grid, the size of a
grid-division is \fI(xmax - xmin)/(nx - 1)\fP. \fIxmin\fP defaults to 0,
while \fIxmax\fP defaults to \fInx\fP.
.IP "\fBxgrid\fP" 12
Specifies whether the grid is uniform or nonuniform on the x-axis.
If \fBTrue\fP, the x-coordinates of the grid must be specified following
this \fBinstruction\fP.
.IP "\fBny\fP" 12
Specifies the number of grid-points on the y-axis.  Minimum value is 2.
Required for both uniform and non-uniform grid specification.
.IP "\fBymin\fP" 12
.IP "\fBymax\fP" 12
Specifies the limits of the grid in y. For a uniform grid, the size of a
grid-division is \fI(ymax - ymin)/(ny - 1)\fP. \fIymin\fP defaults to 0,
while \fIymax\fP defaults to \fIny\fP.
.IP "\fBygrid\fP" 12
Specifies whether the grid is uniform or nonuniform on the y-axis.
If \fBTrue\fP, the y-coordinates of the grid must be specified following
this \fBinstruction\fP.
.IP "\fBbinary\fP" 12
Specifies whether the following data is written in ASCII or
machine-dependent binary.  Must be specified individually for x-grid
coordinates, y-grid coordinates and surface z-values.
If \fBTrue\fP, a double-precision binary array of appropriate size 
is expected to follow this \fBinstruction\fP.
.IP "\fBjoincurve\fP" 12
Causes contour curves to be joined at the data-boundaries. This is done
by defining a boundary layer around the rectangular border, 
and setting the z-value of that boundary layer at either the maximum 
or minimum z-value. 
\fIJoincurve="HIGH"\fP or \fIjoincurve=2\fP sets the border z-value 
to its maximum value: this is useful for plots
which have high average z-values. 
\fIJoincurve="LOW"\fP or \fIjoincurve=1\fP sets the border z-value 
to its minimum value: this
is useful for plots which have low average z-values.
The default value is 0, in which case the contours are not joined at the
boundaries.
.IP "\fBinterp\fP" 12
Specifies the interpolation method used to determined contours of the
rectangular mesh.  \fIInterp=0\fP determines the contours from the
rectangles.  \fIInterp=1\fP and \fIinterp=2\fP break up each rectangle into
2 or 4 triangles respectively; contours are calculated based on these
triangles.  \fIInterp=3\fP actually means no interpolation; each data-point
is expanded into a rectangle parallel to the z-plane, which results in
digital or bitmap-like contours.  The default value is 0.
.SH
Uniform Grid Format Specification
.LP
The uniform grid 
assumes that the x-y points are arranged on a regular rectangular grid 
bounded by \fIxmin\fP, \fIxmax\fP, \fIymin\fP and \fIymax\fP.  
Within this grid, there are \fInx\fP and \fIny\fP
grid points in x and y. 
The CONTOUR format for specifying a uniform grid is as follows:
.DS
$ DATA=CONTOUR
% nx=<value> [xmin=<value> xmax=<value>]
% ny=<value> [ymin=<value> ymax=<value>]
% [binary=<T/F>]
 z[0,0] z[1,0] z[2,0] ... z[i,0] 
 z[0,1] z[1,1] z[2,1] ... z[i,1]
 ...
 z[0,j] z[1,j] z[2,j] ... z[i,j] 
$ END
.DE
The boundary limits and number of grid-points are specified using
\fBinstructions\fP.
Note that the boundary limits are optional; if \fIxmin\fP, \fIxmax\fP, etc.
are omitted, \fIxmin=0\fP and \fIxmax=nx\fP values are substituted.
The uniform grid boundary specification is followed by \fInx \(mu ny\fP
z-values on the grid, arranged by column and row.
In C programming terms, this format corresponds to printing a 2D
array in the following manner:
.DS
/* j is the outer-most loop */
for (j=0; j<ny; j++)
for (i=0; i<nx; i++)
   (void) printf("%g\en", array[i][j]);
.DE
The z-values do not all have to be on the same line; they can be specified
one value per line as in the above example, or several values per line. 
Note however that the CONTOUR
reader has is a 1000-character limit per line - if too many z-values are
printed on the same line, only the first 1000 characters (approximately 100
floating-point numbers) will be read.
.KS
.LP
As an example, the data-set shown below defines a rectangular grid bounded
by \fIxmin\fP=0.0, \fIxmax\fP=2.0, \fIymin\fP=0.0 and \fIymax\fP=1.0.
There are 5 grid-points in x (\fInx\fP=5) and 3 in y (\fPny\fP=3), 
corresponding to a uniform square grid 0.5 units in width.  
.TS
box center;
l.
$ DATA = CONTOUR
% XMIN=0.0   XMAX=2.0    NX=5
% YMIN=0.0   YMAX=1.0    NY=3
1.0 1.0 1.0 1.0 1.0  
1.0 2.0 1.0 2.0 1.0
1.0 2.0 3.0 2.0 5.0 
$ END
.TE
The dataset above produces the following mapping:
.TS
allbox, tab(|), center;
l c c c c c.
z(x,y)|x=0.0|x=0.5|x=1.0|x=1.5|x=2.0
y=0.0 |1.0  |1.0  |1.0  |1.0  |1.0
y=0.5 |1.0  |2.0  |1.0  |2.0  |1.0
y=1.0 |1.0  |2.0  |3.0  |2.0  |5.0
.TE
.KE
.LP
The z-values can also be specified in a machine-dependent binary format.
The machine-dependent binary format is defined by the presence of the
a "Binary=True" string in an \fBinstruction\fP line.
The binary data-stream must begin on the line immediately following the
above \fBinstruction\fP line. The stream consists of a 1-dimensional
array of type double-precision. The array-size is \fInx \(mu ny\fP.
The following code-fragment illustrates writing out binary data in the
CONTOUR format:
.TS
box center;
l.
   double xarr[20], yarr[20], zarr[400];
   int nx=20, ny=20, i,j;

   /* Fill x,y arrays */
   for (i=0; i<nx; i++) xarr[i] = 0.1*i;
   for (j=0; j<ny; j++) yarr[j] = 0.1*j;

   /* Fill 1D z-array: note that z[i,j] = zarr[i+nx*j] */
   for (i=0; i<nx; i++)
   for (j=0; j<ny; j++)
      zarr[i+nx*j] = sqrt(xarr[i]*xarr[i] + yarr[j]*yarr[j]);

   /* Write out contour header */
   (void) fprintf(fp,"$ DATA=CONTOUR\en");
   (void) fprintf(fp,"%% xmin=0.0 xmax=2.0 ymin=0.0 ymax=2.0\en");
   (void) fprintf(fp,"%% nx=%d ny=%d\en",nx,ny);

   /* Write out contour data */
   (void) fprintf(fp,"%% BINARY\en");
   if (fwrite((char *)zarr,sizeof(double),nx*ny,fp) != nx*ny) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return; 
   }
   (void) fprintf(fp,"$ END\en");
.TE
.KS
This code fragment yields the following:
.TS
box, center;
l.
$ DATA=CONTOUR
% xmin=0.0 xmax=2.0 ymin=0.0 ymax=2.0
% nx=20 ny=20
[binary double-precision array]\(dg
$ END
.TE
.KE
.FS \(dg
The square brackets in this case denote an array; 
the brackets are not to be written out
to the data-stream.
.FE
.SH
Non-Uniform Grid Specification
.LP
The x-y grid may also be composed of nonuniform rectangles. Nonuniformity is
specified using "XGRID=True" and "YGRID=True", together with \fInx\fP and 
\fIny\fP for the number of grid-points. 
The data then consists of \fInx\fP points for the
x-grid, \fIny\fP points for the y-grid, followed by \fInx \(mu ny\fP
points for the z-values on the grid.
The non-uniform CONTOUR format has the following form:
.DS
$ DATA=CONTOUR

% nx=<value> xgrid=True [binary=<T/F>]
 x[0] x[1] ... x[i] ... x[nx-1]

% ny=<value> ygrid=True [binary=<T/F>]
 y[0] y[1] ... y[j] ... y[ny-1]

% [binary=<T/F>]
 z[0,0] z[1,0] z[2,0] ... z[i,0] 
 z[0,1] z[1,1] z[2,1] ... z[i,1]
 ...
 z[0,j] z[1,j] z[2,j] ... z[i,j] 
$ END
.DE
In the above, the x-grid values \fBmust\fP be specified before the y-grid
values which in turn \fBmust\fP come before the z-values.
Furthermore, the x, y and z-grid arrays can be specified in binary, provided
that a "BINARY=True" \fBinstruction\fP is included before each occurrence of
a binary array.
.LP
The example below defines a non-uniform rectangular grid bounded
by \fIxmin=\fP0.0, \fIxmax=\fP2.0, \fIymin=\fP0.0 and \fIymax=\fP1.0.
There are 5 grid-points in x (\fInx=\fP5) and 3 in y (\fPny=\fP3).
.TS
box, center;
l.
$ DATA=CONTOUR

% NX=5 XGRID
0.0 0.2 1.0 1.7 2.0

% NY=3 YGRID
1.0 0.6 0.0

# The z-data
1.0 1.0 1.0 1.0 1.0
1.0 2.0 1.0 2.0 1.0
1.0 2.0 3.0 2.0 5.0
$ END
.TE
.KS
.LP
The dataset above results in this mapping:
.TS
allbox, tab(|), center;
l c c c c c.
z(x,y)|x=0.0|x=0.2|x=1.0|x=1.7|x=2.0
y=1.0 |1.0  |1.0  |1.0  |1.0  |1.0
y=0.6 |1.0  |2.0  |1.0  |2.0  |1.0
y=0.0 |1.0  |2.0  |3.0  |2.0  |5.0
.TE
.KE
.KS
.LP
The non-uniform grid specification may also be mixed with the uniform
format, that is, either of the x or y grid-values can be non-uniform.
The example below
shows a grid nonuniform in x and uniform in y.
.TS
box, center;
l.
$ DATA=CONTOUR

% NX=5 XGRID
0.0 0.2 1.0 1.7 2.0

% NY=3 YMIN=0 YMAX=2 

# The z-data
1.0 1.0 1.0 1.0 1.0
1.0 2.0 1.0 2.0 1.0
1.0 2.0 3.0 2.0 5.0
$ END
.TE
.LP
As might be expected, the following mapping is obtained.
.TS
allbox, tab(|), center;
l c c c c c.
z(x,y)|x=0.0|x=0.2|x=1.0|x=1.7|x=2.0
y=0.0 |1.0  |1.0  |1.0  |1.0  |1.0
y=1.0 |1.0  |2.0  |1.0  |2.0  |1.0
y=2.0 |1.0  |2.0  |3.0  |2.0  |5.0
.TE
.KE
.bp
.SH
.ce
4D GRID FORMAT
.SH
Overview
.LP
The GRID4D format specifies 4-dimensional data as values on a rectangular
volumetric x-y-z grid.  The format is an extension of the CONTOUR format
from 3D to 4D.  As with the CONTOUR format, the data consists of a
grid-specification, denoting the points on the x-y-z grid, followed by the
main body of t-values at the nodes of the x-y-z grid.  The t(x,y,z) values
are assumed to be single-valued functions of x, y and z.  The rectangular
volumetric grid may be uniform or nonuniform.  Furthermore, the
grid-points as well as the t-values on the grid may be specified in ASCII
or machine-dependent binary formats.
.LP
Once the data has been read in, the 4D volume is sliced along each of its
six orthogonal surfaces; contours of the t-values are plotted on each surface. 
By default, the contours are plotted in gradated colors with
approximately 10 contour intervals between tmin and tmax.
The number of contours and their intervals
may be set using \fBDataset
Properties\fP (see Appendix A).
.SH
Resources
.LP
The GRID4D format accepts all of the \fBPlot Resources\fP listed in
\fBAppendix A\fP, except for the \fBCurve Resources\fP and the \fBGlobal
Curve Resources\fP.\(dg
.FS \(dg
At present, it is not possible to manually set 
curve properties (e.g. linetype, linewidth) of
the contour curves derived from a GRID4D dataset.
.FE
In addition, the GRID4D
dataset accepts the following resources (specified in \fBinstruction\fP
lines):
.KS
.TS
box, tab(|), center;
cfB s s s
lfB | lfB | lfB | lfB
l   | l   | l   | l.
GRID4D Property Argument List
_
Name|Type|Default|Description
=
nx|int|-|Grid Divisions in x (required)
_
xmin|double|0.0|Uniform Grid boundary
_
xmax|double|nx|Uniform Grid boundary
_
xgrid|Boolean|False|Non-Uniform Grid
_
ny|int|-|Grid Divisions in y (required)
_
ymin|double|0.0|Uniform Grid boundary 
_
ymax|double|ny|Uniform Grid boundary 
_
ygrid|Boolean|False|Non-Uniform Grid
_
nz|int|-|Grid Divisions in z (required)
_
zmin|double|0.0|Uniform Grid boundary 
_
zmax|double|nz|Uniform Grid boundary 
_
zgrid|Boolean|False|Non-Uniform Grid
_
binary|Boolean|False|Machine-dependent binary
.TE
.KE
.IP "\fBnx\fP" 12
Specifies the number of grid-points on the x-axis.  Minimum value is 2.
Required for both uniform and non-uniform grid specification.
.IP "\fBxmin\fP" 12
.IP "\fBxmax\fP" 12
Specifies the limits of the grid in x. For a uniform grid, the size of a
grid-division is \fI(xmax - xmin)/(nx - 1)\fP. \fIxmin\fP defaults to 0,
while \fIxmax\fP defaults to \fInx\fP.
.IP "\fBxgrid\fP" 12
Specifies whether the grid is uniform or nonuniform on the x-axis.
If \fBTrue\fP, the x-coordinates of the grid must be specified following
this \fBinstruction\fP.
.IP "\fBny\fP" 12
Specifies the number of grid-points on the y-axis.  Minimum value is 2.
Required for both uniform and non-uniform grid specification.
.IP "\fBymin\fP" 12
.IP "\fBymax\fP" 12
Specifies the limits of the grid in y. For a uniform grid, the size of a
grid-division is \fI(ymax - ymin)/(ny - 1)\fP. \fIymin\fP defaults to 0,
while \fIymax\fP defaults to \fIny\fP.
.IP "\fBygrid\fP" 12
Specifies whether the grid is uniform or nonuniform on the y-axis.
If \fBTrue\fP, the y-coordinates of the grid must be specified following
this \fBinstruction\fP.
.IP "\fBnz\fP" 12
Specifies the number of grid-points on the z-axis.  Minimum value is 2.
Required for both uniform and non-uniform grid specification.
.IP "\fBzmin\fP" 12
.IP "\fBzmax\fP" 12
Specifies the limits of the grid in z. For a uniform grid, the size of a
grid-division is \fI(zmax - zmin)/(nz - 1)\fP. \fIzmin\fP defaults to 0,
while \fIzmax\fP defaults to \fInz\fP.
.IP "\fBzgrid\fP" 12
Specifies whether the grid is uniform or nonuniform on the z-axis.
If \fBTrue\fP, the z-coordinates of the grid must be specified following
this \fBinstruction\fP.
.IP "\fBbinary\fP" 12
Specifies whether the following data is written in ASCII or
machine-dependent binary.  Must be specified individually for x-grid
coordinates, y-grid coordinates, z-grid coordinates and volume t-values.
If \fBTrue\fP, a double-precision binary array of appropriate size
is expected to follow this \fBinstruction\fP.
.SH
Format Specification
.LP
The grid divisions in x, y, and z may be specified uniformly or
non-uniformly.  The following \fBinstruction\fP line is used to 
specify a uniform distribution in the x-axis.
.DS
% nx=<value> [xmin=<value> xmax=<value>]
.DE
To specify a non-uniform grid in x, specify the "XGRID=True" in an
instruction line, followed by the x-grid values.  The x-grid values can be
specified in binary, provided a "BINARY=True" \fBinstruction\fP is included
before the binary data-stream.
.DS
% nx=<value> xgrid=True [binary=<T/F>]
 x[0] x[1] ... x[i] ... x[nx-1]
.DE
The y and z grid values are specified in a similar fashion.
.KS
.LP
The t-values on the volumetric grid are specified next, either in ASCII or
in binary.  The GRID4D format for a uniform grid has the following form:
.DS
$ DATA=GRID4D
% nx=<value> [xmin=<value> xmax=<value>]
% ny=<value> [ymin=<value> ymax=<value>]
% nz=<value> [zmin=<value> zmax=<value>]
 
 t[0,0,0] t[1,0,0] t[2,0,0] ... t[i,0,0]
 t[0,1,0] t[1,1,0] t[2,1,0] ... t[i,1,0]
 ...
 t[0,j,0] t[1,j,0] t[2,j,0] ... t[i,j,0]
 
 t[0,0,1] t[1,0,1] t[2,0,1] ... t[i,0,1]
 t[0,1,1] t[1,1,1] t[2,1,1] ... t[i,1,1]
 ...
 t[0,j,1] t[1,j,1] t[2,j,1] ... t[i,j,1]
 
 ...
 ...
 
 t[0,0,k] t[1,0,k] t[2,0,k] ... t[i,0,k]
 t[0,1,k] t[1,1,k] t[2,1,k] ... t[i,1,k]
 ...
 t[0,j,k] t[1,j,k] t[2,j,k] ... t[i,j,k]
$ END
.DE
.KE
.KS
.LP
In C programming terms, this format corresponds to printing a 3D array in
the following manner:
.DS
/* k is the outer-most loop */
for (k=0; k<nz; k++)
for (j=0; j<ny; j++)
for (i=0; i<nx; i++)
   (void) printf("%g\en", array[i][j][k]);
.DE
.KE
.KS
The following example defines a GRID4D dataset bounded by 
\fIxmin=\fP0.0, \fIxmax=\fP2.0, 
\fIymin=\fP0.0, \fIymax=\fP1.0, 
\fIzmin=\fP0.0 and \fIzmax=\fP0.5.
.TS
box, center;
l.
$ DATA = GRID4D

# Regular grid in x
% NX=5 XMIN=0.0 XMAX=2.0 

# Irregular grid in y
% NY=3 YGRID=True 
1.0 0.6 0.0

# Regular grid in z
% NZ=2 ZMIN=0.0 ZMAX=0.5
 
# The t-data for z=0.0
0.0 0.0 0.0 0.0 0.0
0.0 2.0 1.0 2.0 0.0
0.0 2.0 1.0 2.0 0.0
 
# The t-data for z=0.5
1.0 1.0 1.0 1.0 1.0
1.0 2.0 1.0 2.0 1.0
1.0 2.0 3.0 2.0 5.0
$ END
.TE
.KE
.KS
The above results in the following mappings:
.TS
allbox, tab(|), center;
l c c c c c.
t(x,y,z), z=0.0|x=0.0|x=0.5|x=1.0|x=1.5|x=2.0
y=1.0 |0.0  |0.0  |0.0  |0.0  |0.0
y=0.6 |0.0  |2.0  |1.0  |2.0  |0.0
y=0.0 |0.0  |2.0  |1.0  |2.0  |0.0
.TE
.TS
allbox, tab(|), center;
l c c c c c.
t(x,y,z), z=0.5|x=0.0|x=0.5|x=1.0|x=1.5|x=2.0
y=1.0 |1.0  |1.0  |1.0  |1.0  |1.0
y=0.6 |1.0  |2.0  |1.0  |2.0  |1.0
y=0.0 |1.0  |2.0  |3.0  |2.0  |5.0
.TE
.KE
.bp
.SH
.ce
VECTOR FORMAT
.SH
Overview
.LP
The VECTOR format specifies a set of 3D vectors.
Each vector is specified as a physical point, and a direction vector attached
to that point. Both the point and the direction vector are specified in 3D.
.LP
Once the vectors have been read in, a scale-factor is calculated based on
the magnitude of the largest vector in the vector dataset, and the boundary
limits of the dataset; the vectors in
the plot are drawn using this scale-factor.  By default all vectors
of sufficient length (as determined by dimensions on the plot)
are plotted with arrow-heads.  The vector scale-factor and 
the shape of the vector can be controlled using 
\fBDataset Properties\fP (see Appendix A).
.SH
Resources
.LP
The VECTOR format accepts all of the \fBPlot Resources\fP listed in
\fBAppendix A\fP, except for the \fBGlobal
Curve Resources\fP.  Use \fBCurve Resources\fP to set the line-type, color,
and width of the vector arrows.
In addition, the VECTOR
dataset accepts the following resources (specified in \fBinstruction\fP
lines):
.KS
.TS
box, tab(|), center;
cfB s s s
lfB | lfB | lfB | lfB
l   | l   | l   | l.
VECTOR Property Argument List
_
Name|Type|Default|Description
=
binary|Boolean|False|Machine-dependent binary
_
npts|int|-|Binary array size
.TE
.KE
.IP "\fBbinary\fP" 12
Specifies whether the curve data is written in ASCII or machine-dependent
BINARY format.  If \fBTrue\fP, the program tries to read 6 double-precision
binary arrays of size \fBnpts\fP
containing the x, y and z coordinates and vx, vy and vz components of the
vectors.
The arrays are assumed to be on the line following the \fIbinary\fP
\fBinstruction\fP.  Default is \fBFalse\fP.
.IP "\fBnpts\fP" 12
Specifies the number of vectors to be read.  Used only for reading data in
BINARY format.
.SH
Format Specification
.LP
Each vector consists of a physical point (x,y,z) and a direction vector
(vx, vy, vz).  The vectors can be specified either in ASCII or in binary.
The ASCII format requires the coordinates and direction-vector each vector
to be specified one per line as follows:
.DS
x-coord y-coord z-coord vx-component vy-component vz-component # Vector 0
x-coord y-coord z-coord vx-component vy-component vz-component # Vector 1
 ...
x-coord y-coord z-coord vx-component vy-component vz-component # Vector n
.DE
In contrast, the binary format requires the coordinates and vector
components of all the vectors to be written out separately in 6 different
arrays:
.DS
% binary=True npts=<number-of-vectors>
[x-coordinate binary double-precision array]\(dg
[y-coordinate binary double-precision array]
[z-coordinate binary double-precision array]
[vx-component binary double-precision array]
[vy-component binary double-precision array]
[vz-component binary double-precision array]
.DE
The arrays must be preceded by an \fBinstruction\fP line containing the
\fIbinary\fP flag and the number of vectors contained in the arrays.
.FS \(dg
The square brackets in this case denote an array; 
the brackets are not to be written out
to the data-stream.
.FE
.KS
.LP
The VECTOR format thus has the following format:
.DS
$ DATA=VECTOR

# ASCII format
% [optional instructions]
x1 y1 z1 vx1 vy1 vz1  # vector 1
x2 y2 z1 vx2 vy2 vz2  # vector 2
 ...
xn yn zn vxn vyn vzn  # vector n

# BINARY format
% [optional instructions]
% binary=True npts=<number-of-vectors>
[x-coordinate binary double-precision array]\(dg
[y-coordinate binary double-precision array]
[z-coordinate binary double-precision array]
[vx-component binary double-precision array]
[vy-component binary double-precision array]
[vz-component binary double-precision array]
$ END
.DE
.KE
.FS \(dg
The square brackets in this case denote an array; 
the brackets are not to be written out
to the data-stream.
.FE
.LP
Note that it is possible to mix ASCII and binary formats; 
some vectors can be specified in ASCII while others can be specified in
binary.
.KS
.LP
The following shows an example file containing a VECTOR dataset.
.DS
.B1
.sp
.so ex4.mtvdat
.sp
.B2
.DE
.KE
.KS
.SH
Programming Example
.LP
The following short program illustrates writing binary data in the VECTOR
format.
.LP
.TS
box center;
l.
#include <stdio.h>
#include <math.h>
main()
{
#define BINARYFILE "data.mtvdat"
   FILE   *fp;
   double xarr[100], yarr[100], zarr[100];
   double vxarr[100], vyarr[100], vzarr[100];
   int    i, npts=100;
 
   /* Open up a file */
   if ((fp=fopen(BINARYFILE,"w")) == NULL) {
      (void) fprintf(stderr,"cat: Couldn't open file %s\en",BINARYFILE); return;
   }
  
   /* Fill arrays */
   for (i=0; i<npts; i++) {
      xarr[i] = 0.1*i;          vxarr[i] = xarr[i];
      yarr[i] = cos(xarr[i]);   vyarr[i] = yarr[i];
      zarr[i] = sin(xarr[i]);   vzarr[i] = zarr[i];
   }

   /* Write out vector header */
   (void) fprintf(fp,"$ DATA=VECTOR\en");
 
   /* Write out binary data */
   (void) fprintf(fp,"%% BINARY npts=%d\en",npts);
   if (fwrite((char *)xarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return; }
   if (fwrite((char *)yarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return; }
   if (fwrite((char *)zarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return; }
   if (fwrite((char *)vxarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return; }
   if (fwrite((char *)vyarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return; }
   if (fwrite((char *)vzarr,sizeof(double),npts,fp) != npts) {
      (void) fprintf(stderr,"   ***Binary write error of data array!\en"); return; }
   (void) fprintf(fp,"$ END\en");
 
   /* Close the file */
   (void) fclose(fp);
}
.TE
.KE
.if \n(zZ=1 .ig zB
.bp
.SH
.ce
MESH FORMAT
.SH
Overview
.LP
The mesh format is a hierarchical list of points, nodes, triangles and
quantities which specifies a mesh structure and quantities associated with
that structure.  
The format is essentially an adaptation of the mesh
structure used in SUPREM4, in which triangles consist of nodes connected to
physical points in a device.  Quantities are solutions or values of a
parameter on the mesh structure.  Unlimited numbers of quantities can be
associated with a mesh; such quantities can be plotted either as a 3D
surface or as iso-value contours.
.LP
The data is read in as a mesh structure and quantities attached to the
structure.  The mesh structure and quantities are all saved as separate
datasets; the mesh structure is saved as a MESH dataset, while
quantities are saved either as CONTOUR or VECTOR datasets.
If the z-range of a quantity is greater than the x-range or y-range 
by 8 orders of magnitude, contours of the quantity are automatically plotted
on a logarithmic scale.
.LP
The MESH format accepts all of the \fBPlot Resources\fP listed in
\fBAppendix A\fP, except for the \fBGlobal Curve Resources\fP.
In addition, the following defaults are used for MESH-based
datasets.
.KS
.TS
box, tab(|), center;
cfB s s s s
lfB | lfB | lfB | lfB | lfB
l   | l   | l   | l   | l.
Plotset/Dataset Property Argument List
_
Name|Type|Default|Range|Description
=
equalscale|Boolean|True|T/F|Equal scaling in X and Y
_
yflip|Boolean|True|T/F|Flip Y-Scale
_
contlabel|Boolean|False|T/F|Plot contour labels
_
linetypes|integer|1|1 \(em 3|No of contour linetypes
_
meshplot|Boolean|False|T/F|Plot mesh
_
boundary|Boolean|True|T/F|Plot boundary
_
fillbound|Boolean|True|T/F|Fill boundary
.TE
.KE
.SH
ASCII Format Description 
.LP
The mesh format consists of data-values preceded by specific characters.
The leading characters must be the first non-blank or non-tabbed characters
on a line.
.KS
.ta 0.2i 3i
.DS
c	point-ID x y z 	- specifies points
n	node-ID point-ID 	- specifies a node
t	tria-ID node1-ID node2-ID node3-ID region-ID 	- specifies a triangle in a specific region
r	region-ID region-label 	- specifies a region or material
q	quantity-label nfields	- specifies a quantity
v	node-ID soln-field1 soln-field2 ... soln-field-n	- specifies values (solutions) at the given nodes
.DE
.KE
Note that the mesh and associated quantities are specified in 
terms of nodes instead of points.  The basic idea here is that a given
triangle in the mesh will contain 3 nodes, each of which will have a value
and the address of a point.  The value of the node is given by the
quantity, e.g. potential, to be plotted.
.LP
As an example, a single triangle with 3 points and 3 nodes is
specified thus:
.DS
c 1  0.0 0.0 0.0
c 2  1.0 0.0 0.0
c 3  0.0 1.0 0.0
n 11 1
n 12 2
n 13 3
t 1  11 12 13  1
.DE
At this point, the triangle does not contain any solutions or values; it is,
however, a complete specification of the physical mesh. The next step is to
specify some quantities:
.DS
q "Oxide" 1
v 11 0.2
v 12 0.4
v 13 0.6
q "Boron" 1
v 11 0.8
v 12 0.8
v 13 0.6
.DE
Then if "Oxide" is to be plotted, the data in the "Oxide" values are applied
to the appropriate nodes, e.g. node 11 is filled with a value of 0.2, and so
on.  Similarly, "Boron" can be plotted by applying the quantity to the mesh
before plotting the result.
.SH
ASCII Format Specification
.RS
.IP "\fBPoint\fP:" 11
Define coordinates of a point in the mesh
.LP
.TS 
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i).
Syntax:|c|ptID   x y z
.sp
       |ptID|= integer identifier for a point
       |x,y,z|= cartesian coordinates
.sp 
Example:| |
        |c|1  0.0 0.5 -0.2
.TE

.IP "\fBNode\fP:" 11
Define a node, i.e. the solution in a particular material
at a physical point.
.LP
.TS 
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) sw(3.5i) .
Syntax:|n|ndID ptID
.sp
       |ndID|= integer identifier for a node
       |ptID|= integer identifier for a point
.sp 
Example:| |
        |n|2 1
.sp
Limitations:|T{
.ll 4i
The points must be previously defined using the point (p) statement.
Not doing so generates an error.
T}
.TE

.IP "\fBTriangle\fP:" 11
Define a triangular element, connected to 3 nodes.
.LP
.TS
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) sw(3.5i).
Syntax:|t|trID   ndID1 ndID2 ndID3   regID
.sp
       |trID   |= integer identifier for a triangle
       |ndID1-3|= node identifiers
       |regID  |= region to which the triangle belongs
.sp 
Example:| |
        |t|1   1 2 3   1
.sp 
Limitations:|T{
.ll 4i
The nodes must be previously defined using the node (n) statement.
Not doing so generates an error.
T}
.TE

.KS
.IP "\fBRegion\fP:" 11
A region or area containing several triangles
.LP
.TS
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) sw(3.5i).
Syntax:|r|regID matname
.sp
       |regID  |= integer identifier for a region
       |matname|= character string (optionally) delimited by quotes ("")/('')
.sp
Example:| |
        |r|1 "oxide"
.sp
Limitation:|T{
.ll 4i
The regions specified must match those specified in the triangle (t)
statements.
T}
.TE
.KE

.KS
.IP "\fBQuantity\fP:" 11
Define the quantity and number of fields in the quantity
.IP "\fBNode-Value\fP:" 11
Values of solutions at particular nodes
.LP
.TS
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) sw(3.5i).
Syntax:|q|label nfields
       |v|ndID field1 field2 ... field_n
.sp
       |label  |= character string (optionally) delimited by quotes ("")/('')
       |nfields|= number of fields to be read in subsequent 'v' statements
       |ndID   |= node identifiers
       |field_n|= value of n-th component of field at a mesh node
.sp
Example:|(Scalar)|
        |q|boron 1  
        |v|1 1.0e9
        |v|2 2.4e10
.sp
Example:|(Vector)|
        |q|efield 3 
        |v|1 0.2 0.4 0.2
        |v|2 0.0 0.4 -0.1
.sp
Limitations:|T{
.ll 4i
Only a single quantity may be specified for each "q-v" block.
.br
The nodes must be previously defined using
the node (n) statement.  Not doing so generates an error.
T}
.TE
.KE
.RE
.KS
.SH
ASCII Format Example
.LP
The following example defines a MESH dataset containing 
a simple mesh-structure. 
At this point, no quantities are associated with this mesh.
.RS
.DS 
.B1
.sp
.so ex5a.mtvdat
.sp
.B2
.DE
.RE
.KE
.KS
.LP
The next example defines a second MESH dataset containing only quantities.
The quantities are associated implicitly with the mesh-structure in the 
first example.
Assuming that the two sets of data (or files) are read in order, 
the MESH reader will
automatically assign the quantities to the mesh-structure.
Note that the MESH reader associates quantities to the \fIlatest\fP
mesh read in, so if a different mesh is specified after the first mesh is
read in, then the quantities in this example will be associated with that last
mesh and thus give incorrect results.
.RS
.DS
.B1
.sp
.so ex5b.mtvdat
.sp
.B2
.DE
.RE
.KE
.bp
.SH
BINARY Format Description 
.LP
The binary mesh format is very similar to the ASCII mesh format.
The binary format also consists of data-values preceded by specific characters.
The leading characters must be the first non-blank or non-tabbed characters
on a line.
The general flow of the binary mesh format is as follows:
.KS
.ta 0.2i 3i
.DS
bc	npoints 	- specifies list of points
	[point-ID array]\(dg
	[point x-coordinate array]
	[point y-coordinate array]
	[point z-coordinate array]
bn	nnodes 	- specifies list of nodes
	[node-ID array]
	[point-ID array]
bt	ntrias 	- specifies list of triangles 
	[tria-ID array]
	[node1-ID array]
	[node2-ID array]
	[node3-ID array]
	[region-ID array]	
r	region-ID region-label 	- specifies a region or material
q	quantity-label nfields	- specifies a quantity
bv	nvalues nfields	- specifies values (solutions) at the given nodes
	[node-ID array]
	[soln-field1 array]
	...
	[soln-field-n array]
.DE
.KE
.FS \(dg
The square brackets denote an array; the brackets are not to be written out
to the data-stream.
.FE
As can be seen from the above, the primary difference between the ASCII and
binary formats is that in the binary format, 
lists of elements such as points and nodes are written in binary arrays.
To illustrate, the ASCII format for points is:
.ta 0.2i
.DS
c	point-ID1 x1 y1 z1
c	point-ID2 x2 y2 z2
 ...
c	point-IDn xn yn zn
.DE
while the binary format is:
.ta 0.2i
.DS
bc	npoints
	[point-ID array]
	[point x-coordinate array]
	[point y-coordinate array]
	[point z-coordinate array]
.DE
In the ASCII format, each point is written one to a line.  In contrast, the
binary format writes the points as a group; each field of the point (e.g.
x-coordinate, y-coordinate) is
placed in a separate array, and the arrays are written in
binary format. The size of the array is 
equal to the number of points in the mesh; this number is 
written as a field of the "bc" line.  Note also that the "b" as the
leading character in a line identifies the
data as machine-dependent binary - thus "c" identifies a single point in
ASCII, while "bc" identifies a group of points in binary.
The same applies to nodes, triangles and node-values.
.bp
.SH
BINARY Format Specification
.RS
.IP "\fBPoint\fP:" 11
Define coordinates of points in the mesh
.LP
.TS 
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) sw(3.5i).
Syntax:|bc|npoints
       |  |[ptID binary integer array]\(dg
       |  |[x-coordinate binary double-precision array]
       |  |[y-coordinate binary double-precision array]
       |  |[z-coordinate binary double-precision array]
.sp
       |npoints|= array size (number of points) 
       |ptID|= integer identifier for a point
       |x,y,z|= cartesian coordinates
.sp 
Example:| |
        |c|20
        | |pt[0].ID pt[1].ID ... pt[19].ID
        | |pt[0].x  pt[1].x  ... pt[19].x 
        | |pt[0].y  pt[1].y  ... pt[19].y 
        | |pt[0].z  pt[1].z  ... pt[19].z 
.sp
Note:|T{
.ll 4i
The binary arrays are written out on the line following the "bc"
line.  See programming example in the following section.
T}
.TE
.FS \(dg
The square brackets denote an array; the brackets are not to be written out
to the data-stream.
.FE

.IP "\fBNode\fP:" 11
Define nodes, i.e. the solutions in a particular material
at physical points.
.LP
.TS 
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) lw(3.5i) 
lw(0.8i) lw(0.5i) sw(3.5i) .
Syntax:|bn|nnodes
       |  |[ndID binary integer array]\(dg
       |  |[ptID binary integer array]
.sp
       |nnodes|= array size (number of nodes) 
       |ndID|= integer identifier for a node
       |ptID|= integer identifier for a point
.sp 
Example:| |
        |bn|20
        | |nd[0].ID nd[1].ID ... nd[19].ID
        | |nd[0].pt.ID nd[1].pt.ID ... nd[19].pt.ID
.sp
Note:|T{
.ll 4i
The binary arrays are written out on the line following the "bn"
line.  See programming example in the following section.
T}
.sp
Limitations:|T{
.ll 4i
The points must be previously defined using the point (p/bp) statement.
Not doing so generates an error.
T}
.TE

.KS
.IP "\fBTriangle\fP:" 11
Define triangular elements, each connected to 3 nodes.
.LP
.TS
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) sw(3.5i).
Syntax:|t|ntrias
       |  |[trID binary integer array]
       |  |[ndID1 binary integer array]
       |  |[ndID2 binary integer array]
       |  |[ndID3 binary integer array]
       |  |[regID binary integer array]
.sp
       |ntrias|= array size (number of trias) 
       |trID   |= integer identifier for a triangle
       |ndID1-3|= node identifiers
       |regID  |= region to which the triangle belongs
.sp 
Example:| |
        |t|20
        | |tr[0].ID tr[1].ID ... tr[19].ID
        | |tr[0].n1.ID tr[1].n1.ID ... tr[19].n1.ID
        | |tr[0].n2.ID tr[1].n2.ID ... tr[19].n2.ID
        | |tr[0].n3.ID tr[1].n3.ID ... tr[19].n3.ID
        | |tr[0].rg.ID tr[1].rg.ID ... tr[19].rg.ID
.sp
Note:|T{
.ll 4i
The binary arrays are written out on the line following the "bn"
line.  See programming example in the following section.
T}
.sp 
Limitations:|T{
.ll 4i
The nodes must be previously defined using the node (n/bn) statement.
Not doing so generates an error.
T}
.TE
.KE

.KS
.IP "\fBRegion\fP:" 11
A region or area containing several triangles
.LP
.TS
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) sw(3.5i).
Syntax:|r|regID matname
.sp
       |regID  |= integer identifier for a region
       |matname|= character string (optionally) delimited by quotes ("")/('')
.sp
Example:| |
        |r|1 "oxide"
.sp
Limitation:|T{
.ll 4i
The regions specified must match those specified in the triangle (t/bt)
statements.
T}
.TE
.KE

.KS
.IP "\fBQuantity\fP:" 11
Define the quantity and number of fields in the quantity
.IP "\fBNode-Value\fP:" 11
Values of solutions at particular nodes
.LP
.TS
tab(|), box;
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) lw(3.5i)
lw(0.8i) lw(0.5i) sw(3.5i).
Syntax:|q|label nfields
       |nv|nvalues nfields
       |  |[ndID binary integer array]
       |  |[field1 binary double-precision array]
       |  |[field2 binary double-precision array]
       |  |...
       |  |[fieldn binary double-precision array]
.sp
       |label  |= character string (optionally) delimited by quotes ("")/('')
       |nfields|= number of fields to be read in subsequent 'nv' statements
       |nvalues|= array size (number of values) 
       |ndID   |= node identifiers
       |fieldn |= value of n-th component of field at a mesh node
.sp
Example:|(Scalar)|
        |q|boron 1  
        |nv|20 1
        | |nv[0].ID nv[1].ID ... nv[19].ID
        | |nv[0].field[0] nv[1].field[0] ... nv[19].field[0]
.sp
Example:|(Vector)|
        |q|efield 3 
        |nv|20 3
        | |nv[0].ID nv[1].ID ... nv[19].ID
        | |nv[0].field[0] nv[1].field[0] ... nv[19].field[0]
        | |nv[0].field[1] nv[1].field[1] ... nv[19].field[1]
        | |nv[0].field[2] nv[1].field[2] ... nv[19].field[2]
.sp
Note:|T{
.ll 4i
The binary arrays are written out on the line following the "bv"
line.  See programming example in the following section.
T}
.sp 
Limitations:|T{
.ll 4i
Only a single quantity may be specified for each "q-bv" block.
.br
The nodes must be previously defined using
the node (n/bn) statement.  Not doing so generates an error.
T}
.TE
.KE
.RE
.bp
.SH
Programming Example
.LP
The following short program shows an example of writing data in the
binary MESH format. The example assumes that points are initially stored in
an array of point-structures.  In writing the data, the 
point-fields are saved in separate arrays then printed in binary format.  
This example writes only the points in a mesh; nodes, triangles and
quantities should be written in a similar fashion.
.KS
.TS
box center;
l.
#include <stdio.h>

/* Structure declaration */
typedef struct point_str {
   int ID;
   double x, y, z;
} point;

main()
{
#define BINARYFILE "data.mtvdat"
   FILE   *fp;
   point pt[100];
   int   npoints=100, i, *parr;
   double *xarr, *yarr, *zarr;

   /* Open up a file */
   if ((fp=fopen(BINARYFILE,"w")) == NULL) {
      (void) fprintf(stderr,"cat: Couldn't open file %s\en",BINARYFILE); return; }

   /* Fill the points */
   for (i=0; i<npoints; i++) { pt[i].ID = i; pt[i].x  = 0.1*i; pt[i].y  = 0.2*i; pt[i].z  = 0.3*i; }

   /* Put the point elements in 4 separate arrays */
   if ((parr=(int *)malloc((unsigned int)(npoints*sizeof(int))))==NULL) exit(-1);
   if ((xarr=(double *)malloc((unsigned int)(npoints*sizeof(double))))==NULL) exit(-1);
   if ((yarr=(double *)malloc((unsigned int)(npoints*sizeof(double))))==NULL) exit(-1);
   if ((zarr=(double *)malloc((unsigned int)(npoints*sizeof(double))))==NULL) exit(-1);

   /* Fill arrays */
   for (i=0; i<npoints; i++) { parr[i] = pt[i].ID; xarr[i] = pt[i].x; yarr[i] = pt[i].y; zarr[i] = pt[i].z; }

   /* Write out MESH header & binary data */
   (void) fprintf(fp,"$ DATA=MESH\en");
   (void) fprintf(fp,"bc %d\en",npoints);
   if (fwrite((char *)parr,sizeof(int),npoints,fp) != npoints) { (void) fprintf(stderr,"***Error!\en"); exit(-1); }
   if (fwrite((char *)xarr,sizeof(double),npoints,fp) != npoints) { (void) fprintf(stderr,"***Error!\en"); exit(-1); }
   if (fwrite((char *)yarr,sizeof(double),npoints,fp) != npoints) { (void) fprintf(stderr,"***Error!\en"); exit(-1); }
   if (fwrite((char *)zarr,sizeof(double),npoints,fp) != npoints) { (void) fprintf(stderr,"***Error!\en"); exit(-1); }
   (void) fprintf(fp,"$ END\en");

   /* Close the file */
   (void) fclose(fp);

   /* Free the arrays */
   free((char *) parr); free((char *) xarr); free((char *) yarr); free((char *) zarr);
}
.TE
.KE
.zB
