
##############################################################################
#
# Copyright (c) 2003-2018 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2018 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
Test suite for the util.py module.

The tests must be linked with a function space class object in the setUp method:

to run the use:

   from esys.bruce import Brick
   class Test_utilOnBruce(Test_util_no_tagged_data):
       def setUp(self):
           self.domain = Brick(10,10,13)
           self.functionspace = ContinuousFunction(self.domain)
   suite = unittest.TestSuite()
   suite.addTest(unittest.makeSuite(Test_utilOnBruce))
   unittest.TextTestRunner(verbosity=2).run(suite)

This test assumes that samples with x_0 coordinate 0 are tagged with 1 and all samples tagged with 1 have x_0
coordinate 0.

:note: at this stage this test will not pass as it tests for functionlity that has not been implemented yet. It also 
  does not test the full functionalitu of util.py yet.

:var __author__: name of author
:var __copyright__: copyrights
:var __license__: licence agreement
:var __url__: url entry point on documentation
:var __version__: version
:var __date__: date of the version
"""

__author__="Lutz Gross, l.gross@uq.edu.au"

import esys.escriptcore.utestselect as unittest
import numpy
from esys.escript import *
from test_util_base import Test_util_base, Test_util_values

from test_util_reduction_new import Test_util_reduction_new
from test_util_unary_new import Test_util_unary_new
from test_util_binary_new import Test_util_binary_new
from test_util_binary_leftover import Test_util_binary_leftover

## these aspects are test in the _new tests
#from test_util_overloaded_binary_no_tagged_data import Test_util_overloaded_binary_no_tagged_data
#from test_util_overloaded_binary_with_tagged_data import Test_util_overloaded_binary_with_tagged_data
#from test_util_unary_no_tagged_data import Test_util_unary_no_tagged_data
#from test_util_unary_with_tagged_data import Test_util_unary_with_tagged_data
#from test_util_binary_no_tagged_data import Test_util_binary_no_tagged_data
#from test_util_binary_with_tagged_data import Test_util_binary_with_tagged_data

from test_util_spatial_functions1 import Test_Util_SpatialFunctions_noGradOnBoundary_noContact
from test_util_spatial_functions2 import Test_Util_SpatialFunctions_noGradOnBoundary
from test_util_spatial_functions3 import Test_Util_SpatialFunctions
from test_util_slicing_no_tagged_data import Test_util_slicing_no_tagged_data
from test_util_slicing_with_tagged_data import Test_util_slicing_with_tagged_data


class Test_util_reduction(Test_util_reduction_new):
   """ test for reduction operation Lsup,sup,inf for all data types"""
   pass 
class Test_util_unary(Test_util_unary_new):
   """ all unary tests """
   pass
class Test_util_binary(Test_util_binary_new, Test_util_binary_leftover):
   """
   test for all binary operation
   """
   pass

## Testing of these ops is now in Test_util_binary
#class Test_util_overloaded_binary(Test_util_overloaded_binary_no_tagged_data,Test_util_overloaded_binary_with_tagged_data):
   #"""test for all overloaded operation"""
   #pass

class Test_util(Test_util_unary_new,Test_util_reduction_new, Test_util_binary):
   """all tests"""
   pass


class Test_util_overloaded_binary_still_failing(Test_util_base):
   """
   these overloaded operations still fail!

        - wrong return value of Data binaries (Mantis 0000054) 
   """
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_Symbol_rank1(self):
      arg0=Data(-4.93686078973,self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0+arg1
      s1=numpy.array([0.51662736235119944, 2.8171396846123073])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-4.4202334273802917, -2.1197211051191838]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_Symbol_rank2(self):
      arg0=Data(-2.22764991169,self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0+arg1
      s1=numpy.array([[2.0746979587719538, 0.99992890307042437, -2.3128078094931848, -4.0103712739722654, 
4.8853529531011013], 
          [0.09856857946648212, 0.73520899085847624, -3.6585265509750844, 3.0095320582437939, 3.4125902906059444], 
[1.4894150898632059, 
          -1.4124339049368793, 1.5397397961722188, 4.8841402613336111, 1.1241155288598881], [2.8283598865494408, 
1.5980765295723476, 
          -1.0022373011497274, -2.0622178471715067, 4.9699555072046042]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-0.15295195292152819, -1.2277210086230577, -4.5404577211866668, -6.2380211856657475, 
2.6577030414076193], 
           [-2.1290813322269999, -1.4924409208350058, -5.8861764626685664, 0.78188214655031185, 1.1849403789124624], 
[-0.73823482183027611, 
           -3.6400838166303613, -0.68791011552126324, 2.6564903496401291, -1.103534382833594], [0.60070997485595878, 
-0.62957338212113445, 
           -3.2298872128432095, -4.2898677588649887, 2.7423055955111222]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_Symbol_rank3(self):
      arg0=Data(-4.67318656609,self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0+arg1
      s1=numpy.array([[[3.9409337165894076, 1.6101568824796857], [1.2441782896909706, 1.2872758759353298]], 
[[4.022494973005406, 
         -2.758155583474049], [1.8311643900357311, 4.0940647266277157]], [[2.5378127449303243, 0.063283784588161751], 
[4.5495644157820809, 
         2.8673770080506742]], [[-0.93484143473477577, 4.914438575705228], [-1.951066895455166, -1.2021165219313259]], 
         [[-0.4220608661301819, -4.9682501775464418], [0.98338081352961559, 3.4054674805751066]], [[3.9967556325744127, 
-4.7659141789100659], 
         [0.34265275409881024, -0.25226631819007572]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-0.73225284950136693, -3.0630296836110888], [-3.429008276399804, -3.3859106901554448]], 
         [[-0.6506915930853685, -7.4313421495648235], [-2.8420221760550435, -0.57912183946305884]], 
[[-2.1353738211604503, 
         -4.6099027815026128], [-0.12362215030869361, -1.8058095580401003]], [[-5.6080280008255503, 
0.24125200961445348], 
         [-6.6242534615459405, -5.8753030880221004]], [[-5.0952474322209564, -9.6414367436372164], 
[-3.6898057525611589, 
         -1.2677190855156679]], [[-0.67643093351636185, -9.4391007450008395], [-4.3305338119919643, 
-4.9254528842808503]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_Symbol_rank4(self):
      arg0=Data(4.16645075056,self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0+arg1
      s1=numpy.array([[[[1.5917180025121436, -0.50082927718401749, 0.71261274386013618, 2.4216324938382936], 
[2.5988764746053095, 
         0.15985324844397741, -2.1952754277135025, -2.1102730593254035], [4.7816092243808672, -3.1240954141765496, 
         4.0831220997721331, 2.4301203557965216]], [[3.4691826046114969, -2.4961081730013177, -4.9623977358253111, 
2.2652744558918698], 
         [0.41830032681767193, -3.2186897293959649, -4.1590967541108324, -1.7789994379155196], [-0.17901184206486764, 
         -0.85223673399918809, 1.2515459884606104, -4.530305999148645]]], [[[-4.9028671865135838, 3.9106181278983012, 
         0.69716765577825246, 4.8537569187159395], [-2.8912890367657318, -4.8177854256421764, -4.3303142092509415, 
         -0.99481907472179198], [-1.2640734452454305, 4.8028129765204639, -2.5491771511234962, 3.2550469051981921]], 
         [[2.0572417475748761, 3.7392706991121187, 4.5778678295843704, 3.6658188498258486], [-2.7069743698567206, 
         -2.684769111460461, -3.0941141983763156, -2.1180719361316589], [-1.4744678905986119, 1.926687036555828, 
         2.2206999030392947, 0.72956973127168734]]], [[[-2.8290294475300151, -3.1467788245496631, 3.6471044178360348, 
         3.5237454065241209], [-1.6165850845596652, 1.2437746199742081, -2.8022357261752004, -1.9652183524467781], 
         [-2.3842126490032092, 3.7068998814751613, -1.389546865398994, -1.7153758702474589]], [[-1.0746517242894815, 
         -4.3575382718398723, 0.93160793707280121, 1.4002531109392731], [-1.5745690740270168, -3.4394046042905124, 
         4.2641517580348793, -1.7620679696550843], [-4.2559205627171135, 2.1912319337278863, 1.1987265764805723, 
-3.2957352772592809]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[5.7581687530761378, 3.6656214733799768, 4.8790634944241305, 6.5880832444022879], 
          [6.7653272251693037, 4.3263039990079717, 1.9711753228504918, 2.0561776912385907], [8.9480599749448615, 
          1.0423553363874447, 8.2495728503361274, 6.5965711063605159]], [[7.6356333551754911, 1.6703425775626766, 
          -0.7959469852613168, 6.4317252064558641], [4.5847510773816662, 0.94776102116802941, 0.0073539964531619262, 
          2.3874513126484747], [3.9874389084991266, 3.3142140165648062, 5.4179967390246047, -0.36385524858465068]]],
          [[[-0.7364164359495895, 8.0770688784622955, 4.8636184063422467, 9.0202076692799338], [1.2751617137982625,
          -0.6513346750781821, -0.16386345868694718, 3.1716316758422023], [2.9023773053185637, 8.9692637270844582, 
          1.6172735994404981, 7.4214976557621863]], [[6.2236924981388704, 7.905721449676113, 8.7443185801483647, 
          7.8322696003898429], [1.4594763807072737, 1.4816816391035332, 1.0723365521876786, 2.0483788144323354], 
          [2.6919828599653823, 6.0931377871198222, 6.3871506536032889, 4.8960204818356816]]], [[[1.3374213030339792, 
          1.0196719260143312, 7.8135551684000291, 7.6901961570881152], [2.5498656660043291, 5.4102253705382024, 
          1.3642150243887938, 2.2012323981172162], [1.7822381015607851, 7.8733506320391555, 2.7769038851650003, 
          2.4510748803165354]], [[3.0917990262745128, -0.19108752127587803, 5.0980586876367955, 5.5667038615032673], 
          [2.5918816765369774, 0.72704614627348185, 8.4306025085988736, 2.40438278090891], [-0.089469812153119221, 
          6.3576826842918805, 5.3651773270445666, 0.87071547330471333]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([3.8454947431609945, 3.4801848055393254]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(0.181985677208)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([4.0274804203691783, 3.6621704827475092]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([2.6719646801005306, 4.0262173014652003]),self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0+arg1
      s1=numpy.array([3.7355891147806837, -3.0309968912239551])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([6.4075537948812142, 0.99522041024124519]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[2.209887477038702, 2.087043312051243, 3.7254247294014622,
      -3.7510652436671732, 0.70343608099575317], [4.1654611738215745, 1.5418518980850271, 
      2.7730022594684423, 3.386030420596251, 1.2758288509710365], [2.2174938185138764, 
      -1.244837837360393, 2.2331288285078887, -1.1442348969501834, 1.9394801392868004], 
      [0.68612447219195705, 0.7127527031233436, -3.6346644102130776, 2.0671128943191714, 
      3.7445028703597156]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(4.82316401579)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[7.0330514928326018, 6.9102073278451428, 8.5485887451953619, 
      1.0720987721267266, 5.5266000967896529], [8.9886251896154743, 6.3650159138789268, 
      7.596166275262342, 8.2091944363901508, 6.0989928667649362], [7.0406578343077761, 
      3.5783261784335068, 7.0562928443017885, 3.6789291188437163, 6.7626441550807002], 
      [5.5092884879858568, 5.5359167189172434, 1.1884996055808221, 6.8902769101130712, 
      8.5676668861536154]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[-3.62961836797558, 4.0323249470469893, -2.4833229912823516, 
      -0.0081902035785272886, -0.26448613257378906], [2.0867535529248489, 0.049446344294963751, 
      4.4906317789174501, 2.6121865600043499, 1.3687146632565392], [4.2509170325103511, 
      2.9845191554148567, -0.9329820582137387, -0.58236994049271118, -3.4448732067194388], 
      [-2.3231599587033402, 1.6550934434842866, -4.5990521452319584, -2.1470268566500152, 
      -3.9698084155531008]]),self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0+arg1
      s1=numpy.array([[3.3234017918244003, 3.3386199217996175, -2.5928786077225316, 
         -4.1429140632213803, 0.42204291369978719], [3.4123580113357495, -3.9076190537235664, 
         1.8779298531672159, 0.98377543853039562, -4.9365820051249267], [4.5252395032935961, 
         -4.8193051910732096, 1.060979071451845, -3.2927325266544871, -3.3828356655691971], 
         [-4.6411804903406182, -0.42921544747540707, -2.4541073523344323, -0.70845691989162329, 
         -1.2357505826155588]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-0.3062165761511797, 7.3709448688466068, -5.0762015990048832, 
          -4.1511042667999076, 0.15755678112599814], [5.4991115642605983, -3.8581727094286027, 
          6.3685616320846661, 3.5959619985347455, -3.5678673418683875], [8.7761565358039473, 
          -1.834786035658353, 0.12799701323810631, -3.8751024671471983, -6.8277088722886354], 
          [-6.9643404490439584, 1.2258779960088795, -7.0531594975663907, -2.8554837765416385, 
          -5.2055589981686596]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[-2.0819775543023136, 4.4438294149957258], [1.203494127071604, 
         1.3934659764012478]], [[-1.7207192546012995, 1.128687542370864], [1.013953229943537, 
         2.0535582502969056]], [[-1.8482126685735398, 0.64499519705235819], 
         [-4.1200947648310313, 3.8041018736261574]], [[-0.12876390427677542, -0.26859118353213773], 
         [-2.8945993824974847, -3.3476923883525944]], [[3.1332107854705562, -4.6334666373330595], 
         [3.0499420638074994, -2.7959034777693104]], [[4.726734207260332, -1.3724501610660034], 
         [3.3499737674080023, -2.515294322458935]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(0.860178486532)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-1.2217990677700952, 5.3040079015279442], [2.0636726136038224, 
          2.2536444629334662]], [[-0.86054076806908109, 1.9888660289030824], [1.8741317164757554, 
          2.913736736829124]], [[-0.98803418204132143, 1.5051736835845766], [-3.2599162782988129, 
          4.6642803601583758]], [[0.73141458225544298, 0.59158730300008067], [-2.0344208959652663, 
          -2.487513901820376]], [[3.9933892720027746, -3.7732881508008411], [3.9101205503397178, 
          -1.935724991237092]], [[5.5869126937925504, -0.51227167453378497], [4.2101522539402207, 
          -1.6551158359267166]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[-1.849788129717993, 0.64693319038907493], [3.0379670344950327, 
           0.80277076526299229]], [[2.4995340022105639, -4.3955703049125949], [0.58331276679079203, 
           0.044119077451267863]], [[2.2979922792046947, 1.6054844683234073], [0.50524258350986084, 
           -3.5539312710422779]], [[-1.1980433912188793, -2.6450000406046001], [-2.4128326188310121, 
           0.80678465051263526]], [[-2.9963692865064209, -1.0152803020104519], [-0.21931259441936035, 
           -1.153119362615751]], [[-4.2927186206837717, 0.4561872009236847], [3.0860876046130041, 
           -0.78568544768378068]]]),self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0+arg1
      s1=numpy.array([[[-3.4985389035935222, 1.8888458641158987], [-4.2891085749380489, 
         2.8296217607019845]], [[-0.8200921678141917, 4.4359194831012676], 
         [-4.6185751325042244, 0.16520675598470014]], [[-2.801157092531934, 3.6231020804204928], 
         [1.5439760747845899, 2.0378140868272894]], [[0.99864930993784018, 3.369884315459073], 
         [4.399815205976239, -4.9546136700941936]], [[1.6240932313892289, -3.4517363344048615], 
         [2.8668483027947236, 1.1624090061600336]], [[2.6364367974081624, 2.628371373764919], 
         [-2.5877409052653833, -1.29236451403668]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-5.3483270333115147, 2.5357790545049737], [-1.2511415404430162, 
          3.6323925259649767]], [[1.6794418343963722, 0.040349178188672674], 
          [-4.0352623657134323, 0.209325833435968]], [[-0.50316481332723928, 5.2285865487439001], 
          [2.0492186582944507, -1.5161171842149885]], [[-0.19939408128103908, 0.72488427485447282], 
          [1.9869825871452269, -4.1478290195815584]], [[-1.372276055117192, -4.4670166364153134], 
          [2.6475357083753632, 0.0092896435442826331]], [[-1.6562818232756094, 
          3.0845585746886037], [0.49834669934762088, -2.0780499617204606]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[-0.026017904532606551, -0.80192450547405958, 
           0.93785799257835656, -4.4900007911078319], [-1.8444162073720949, 
           1.2059856695600812, 1.8326324480310756, 3.3745782356451564], 
           [3.0929324433706693, -0.94197156488767142, -2.3469684397851207, 
           -4.8976052662192613]], [[1.2658444546015346, 3.0389250549456399, 
           -2.567254770133963, 3.7513728753285314], [-0.10225306211433605, 
           -0.34121316520335299, -2.8745573331597321, -0.73976781968982142], 
           [4.6114590072566681, 3.5325642767850063, 2.1587079910040661, 
           3.8644723652636905]]], [[[-2.5953113243103623, 0.6437882672443429, 
           4.5677362343759853, 3.4108524985046262], [2.9904338528780352, 
           0.73113299006492127, 2.4253724263400445, 3.8646536702562031], 
           [-1.2545053686514152, -4.2675706218911706, -3.6576679389702105, 
           -0.29502287354943402]], [[0.9550527228483654, 2.9537233833481267, 
           -2.6904009310953283, 1.5998857010519698], [-3.7171702199982004, 
           -1.1578306702024044, 1.764070139728485, -1.1506068782808967], 
           [1.5727320181060982, 0.18468074769418674, 3.3262967055395372, 
           -1.2208265816075849]]], [[[-0.25003967903418278, -2.603663543909648, 
           4.6824047463125531, 1.0968919539473987], [1.3471700099604398, 
           -3.8321880437450218, -4.2809409903460676, 1.2933005361204906], 
           [-2.857251250328674, 3.6768205829450178, -2.7999953058490643, 
           2.1117422072666692]], [[-2.1994223710236427, 3.7669030216280923, 
           -3.5232105054852991, -3.7071480752824462], [-0.35952695279389246, 
           2.5451704526750873, -4.2842310996736144, -1.3813503044378783], 
           [-2.5647173415905145, 4.7437501634141572, -4.2234318870342245, 
           2.1862042652792866]]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(0.33323555487)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[0.30721765033724147, -0.46868895060421156, 
          1.2710935474482046, -4.1567652362379839], [-1.5111806525022469, 
          1.5392212244299293, 2.1658680029009236, 3.7078137905150044], 
          [3.4261679982405173, -0.6087360100178234, -2.0137328849152727, 
          -4.5643697113494133]], [[1.5990800094713826, 3.3721606098154879, 
          -2.234019215264115, 4.0846084301983794], [0.23098249275551197, 
          -0.0079776103335049697, -2.541321778289884, -0.4065322648199734], 
          [4.9446945621265161, 3.8657998316548543, 2.4919435458739141, 
          4.1977079201335386]]], [[[-2.2620757694405143, 0.97702382211419092, 
          4.9009717892458333, 3.7440880533744743], [3.3236694077478832, 
          1.0643685449347693, 2.7586079812098925, 4.1978892251260511], 
          [-0.92126981378156714, -3.9343350670213226, -3.3244323841003625, 
          0.038212681320413999]], [[1.2882882777182134, 3.2869589382179747, 
          -2.3571653762254803, 1.9331212559218178], [-3.3839346651283524, 
         -0.82459511533255636, 2.097305694598333, -0.81737132341104868], 
         [1.9059675729759462, 0.51791630256403476, 3.6595322604093852, 
         -0.88759102673773693]]], [[[0.083195875835665234, -2.2704279890398, 
         5.0156403011824011, 1.4301275088172467], [1.6804055648302878, 
         -3.4989524888751737, -3.9477054354762195, 1.6265360909903386], 
         [-2.524015695458826, 4.0100561378148658, -2.4667597509792163, 
         2.4449777621365172]], [[-1.8661868161537947, 4.1001385764979403, 
         -3.1899749506154511, -3.3739125204125981], [-0.026291397924044446, 
         2.8784060075449354, -3.9509955448037664, -1.0481147495680303], 
         [-2.2314817867206664, 5.0769857182840052, -3.8901963321643764, 
         2.5194398201491346]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[1.6204760394819004, -0.95393695229398112, -1.221681223499369, 2.6618713903411937], 
[-1.5387523541807724, 4.6220978399651482, -2.1795716817360713, -3.776821154104939], [1.4330066566763016, 
3.7880327985429378, -0.65902727001966976, -4.29506128665055]], [[-4.0199255222547103, -3.644811287300751, 
3.6508998060332054, -3.569704984460552], [-3.8429890733645489, -2.9119635791576437, 2.3183698092323652, 
1.3643661323778851], [2.9328022056563725, -0.080129403375118535, 0.15566128013433289, 2.344258136058456]]], 
[[[3.03272210358924, 2.8841814084596393, -4.059068204445289, -0.091640986980607408], [-4.2591024547151859, 
-0.36305436045316863, 0.19284537915686428, 4.5041324479849649], [1.2988816365062537, -1.6778808169453416, 
-3.5496975707176146, 4.314356820196215]], [[-1.4533462849506518, -1.003910808707118, 3.8948057966291092, 
1.266066103629278], [-4.4119138102620346, -2.1246183047037603, -2.4610566322999161, -3.5862383252945271], 
[2.9290698526446066, -0.26093763373887136, 0.87809331627623344, -0.47993365832407076]]], [[[2.1717793325666745, 
0.83592896851733212, -2.2538107669063279, 1.6303402530881517], [-0.53207705017646578, -4.5214994998308979, 
-3.6999121226789988, 3.5355643886671686], [3.3936340080223193, -2.1140030580705247, 1.821327452830638, 
-1.6123768640462668]], [[2.3105165926895497, -3.0414367260786292, -1.5788704194425076, 1.0377969965556915], 
[1.3575822980511116, 4.3465002873169833, 0.55678010189701688, 4.99079375906609], [4.2819911907361128, 
4.9615031124625322, 2.7964852390480104, 0.029646894001982282]]]]),self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0+arg1
      s1=numpy.array([[[[3.779495003239937, -4.7840877608643506, 2.651273004571375, -2.179381582597685], 
[-0.27370078331190673, -3.6151069379138887, -0.6880481455894909, 4.4373993248198644], [-1.6276288613086387, 
-1.6376839670015721, -3.1138607609774835, -2.7809800576738719]], [[0.85446276622548556, -4.3676040003341114, 
-4.0083595770538496, -3.915065868011578], [1.6989039436984452, 3.5347026474299419, -1.8748410832866327, 
-4.6526613314583045], [1.9480513434936046, 4.7386182205273322, -1.2001630607496541, 1.8094726084650006]]], 
[[[4.9996435011863589, 0.60285036470010045, 1.457536438507919, 2.7443970579013879], [4.131864622110669, 
0.20996245110639133, 3.3652305004680549, 3.1437873739212119], [-3.0818670302029405, -2.461603163946088, 
-0.56609916674720218, -4.1186964404844861]], [[-2.7183232427482262, -2.1509712746053999, -2.281087666097271, 
-2.4094567126275344], [-3.4723848022755091, -1.563218902128277, -4.7598832341275878, 1.8751725484288029], 
[-4.0474621098792882, 0.59894943914858167, 1.0736279895120182, 4.5015525072725033]]], [[[-3.0082200796749703, 
0.23283074563588535, 2.5230303985659734, 4.8262414779000231], [3.3772486493634837, 1.8234317033464915, 
-1.7905158376185746, -2.9990918311449244], [-3.6765085717620041, 2.0057610304617572, -2.1487273241068525, 
-4.1965541804451352]], [[0.26210933249566715, -2.9167787158271663, -0.89589477578380539, -0.41427249402553912], 
[-3.1708181836677332, 4.3890602408555726, -1.1754542095914857, 4.8422639037274919], [-3.0044937138520034, 
-4.1626528668210083, 0.20385989364778467, -0.016309737359709864]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[5.3999710427218375, -5.7380247131583317, 1.429591781072006, 0.48248980774350869], 
[-1.8124531374926791, 1.0069909020512595, -2.8676198273255622, 0.66057817071492542], [-0.19462220463233715, 
2.1503488315413657, -3.7728880309971533, -7.0760413443244214]], [[-3.1654627560292248, -8.0124152876348624, 
-0.35745977102064419, -7.4847708524721295], [-2.1440851296661037, 0.62273906827229819, 0.44352872594573256, 
-3.2882951990804195], [4.8808535491499772, 4.6584888171522136, -1.0445017806153212, 4.1537307445234566]]], 
[[[8.0323656047755989, 3.4870317731597398, -2.60153176593737, 2.6527560709207805], [-0.12723783260451693, 
-0.1530919093467773, 3.5580758796249192, 7.6479198219061768], [-1.7829853936966868, -4.1394839808914297, 
-4.1157967374648168, 0.19566037971172889]], [[-4.171669527698878, -3.154882083312518, 1.6137181305318382, 
-1.1433906089982564], [-7.8842986125375436, -3.6878372068320373, -7.2209398664275035, -1.7110657768657243], 
[-1.1183922572346816, 0.33801180540971032, 1.9517213057882516, 4.0216188489484326]]], [[[-0.83644074710829575, 
1.0687597141532175, 0.26921963165964558, 6.4565817309881748], [2.8451715991870179, -2.6980677964844064, 
-5.4904279602975734, 0.53647255752224421], [-0.28287456373968478, -0.10824202760876744, -0.3273998712762145, 
-5.808931044491402]], [[2.5726259251852168, -5.9582154419057956, -2.474765195226313, 0.62352450253015235], 
[-1.8132358856166215, 8.7355605281725559, -0.61867410769446884, 9.833057662793582], [1.2774974768841094, 
0.79885024564152385, 3.0003451326957951, 0.013337156642272419]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank0_Symbol_rank0(self):
      arg0=Data(3.50668349593,self.functionspace)
      arg0.setTaggedValue(1,-3.09146650776)
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(-4.32369560802)
      sub=res.substitute({arg1:s1})
      ref=Data(-0.81701211209,self.functionspace)
      ref.setTaggedValue(1,-7.41516211578)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank0_Symbol_rank1(self):
      arg0=Data(3.83444600418,self.functionspace)
      arg0.setTaggedValue(1,-0.266863397142)
      arg1=Symbol(shape=(2,))
      res=arg0+arg1
      s1=numpy.array([3.6938635924807581, -2.3199399928130826])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([7.5283095966592981, 1.5145060113654574]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([3.4270001953384694, -2.5868033899553713]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank0_Symbol_rank2(self):
      arg0=Data(-2.85642807584,self.functionspace)
      arg0.setTaggedValue(1,-0.357260114938)
      arg1=Symbol(shape=(4, 5))
      res=arg0+arg1
      s1=numpy.array([[4.4124412590911621, 1.732298167196193, 1.8228166076040306, -3.9853565905277355, 
3.3793508288079881], [-1.5339512663354116, -2.8915144317379058, -3.6493591659102464, 1.4243106283527815, 
-0.6931246781623841], [4.7714119110273394, 0.45700055229079606, 1.2539528503924027, -1.4029360809413403, 
2.8915917074007416], [4.2546657221847255, 3.2639891865967527, -0.4712967898993945, -3.9077971138749112, 
-3.5655383189938084]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[1.5560131832472779, -1.1241299086476912, -1.0336114682398536, -6.8417846663716197, 
0.52292275296410384], [-4.3903793421792958, -5.74794250758179, -6.5057872417541311, -1.4321174474911027, 
-3.5495527540062684], [1.9149838351834552, -2.3994275235530882, -1.6024752254514816, -4.2593641567852245, 
0.035163631556857311], [1.3982376463408412, 0.40756111075286849, -3.3277248657432787, -6.7642251897187951, 
-6.4219663948376926]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[4.0551811441529519, 1.3750380522579828, 1.4655564926658204, 
-4.3426167054659457, 3.0220907138697779], [-1.8912113812736218, -3.248774546676116, -4.0066192808484562, 
1.0670505134145714, -1.0503847931005943], [4.4141517960891292, 0.099740437352585865, 0.89669273545419248, 
-1.7601961958795505, 2.5343315924625314], [3.8974056072465153, 2.9067290716585426, -0.82855690483760469, 
-4.2650572288131219, -3.9227984339320185]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank0_Symbol_rank3(self):
      arg0=Data(-2.98759917871,self.functionspace)
      arg0.setTaggedValue(1,-4.26584239637)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0+arg1
      s1=numpy.array([[[0.65736935684204045, 1.4685807994312459], [0.99740155640158257, -2.8001282911414127]], 
[[-0.80947613326718226, -4.0270117786915378], [1.1564198209626229, -4.917538904347448]], [[-1.0488230155998202, 
4.0958534641909754], [-4.9502522108275002, -0.19486641488505008]], [[-4.507307254914509, -0.98539101308887389], 
[-4.5909807035957675, 2.4265853650826985]], [[-4.252924691613126, 0.42394291278212481], [3.4198717705842103, 
-4.6000003047031024]], [[4.9609535782609235, 3.1625779529060711], [0.26834958946896492, 3.0941570460788874]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-2.3302298218695272, -1.5190183792803218], [-1.9901976223099851, -5.7877274698529799]], 
[[-3.7970753119787499, -7.0146109574031055], [-1.8311793577489448, -7.9051380830590157]], [[-4.0364221943113883, 
1.1082542854794077], [-7.9378513895390679, -3.1824655935966177]], [[-7.4949064336260767, -3.9729901918004416], 
[-7.5785798823073351, -0.56101381362886915]], [[-7.2405238703246937, -2.5636562659294428], [0.43227259187264266, 
-7.5875994834146701]], [[1.9733543995493559, 0.17497877419450347], [-2.7192495892426027, 
0.10655786736731976]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-3.6084730395261495, -2.7972615969369441], [-3.2684408399666074, 
-7.0659706875096031]], [[-5.0753185296353722, -8.2928541750597269], [-3.1094225754055671, -9.183381300715638]], 
[[-5.3146654119680097, -0.16998893217721456], [-9.2160946071956893, -4.46070881125324]], [[-8.773149651282699, 
-5.2512334094570638], [-8.8568230999639574, -1.8392570312854915]], [[-8.5187670879813169, -3.8418994835860651], 
[-0.84597062578397964, -8.8658427010712924]], [[0.69511118189273358, -1.1032644434621188], [-3.997492806899225, 
-1.1716853502893025]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank0_Symbol_rank4(self):
      arg0=Data(-3.36894529378,self.functionspace)
      arg0.setTaggedValue(1,-4.62956527999)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0+arg1
      s1=numpy.array([[[[-4.6824549992604805, 0.17860523484039881, -3.9939994980255102, -0.36579022311332743], 
[-2.0003582573358858, 3.3436256968249793, -1.5671485178714373, 3.9554829351801821], [4.0499415739210693, 
-3.1796189569360358, 0.28181611699077536, 1.4851321313182684]], [[4.9608073066477267, 2.1353944107091136, 
3.2103965744924743, 0.36273874746876089], [0.33193515801312934, -1.8768462949087295, -3.5968753845201462, 
-1.9342255010038101], [-0.98845968068423407, -2.6505467151645048, -3.9269883741621214, -1.2671783073823359]]], 
[[[4.0296290320262234, 0.094183089334959114, -1.6548527114390654, 1.1815006848827636], [4.4205350333429578, 
1.0602877007979998, -2.7207610093848364, 2.5749353581909009], [2.368743673752042, 0.36879117257479166, 
3.1294699111463196, 3.8766421343643209]], [[-4.2994052301352443, -4.4665347726615128, -4.9654257982784813, 
1.4010627781386145], [-0.49010647980719568, 1.1149343027340697, 3.8533389980231654, -1.4762647122950145], 
[-2.4078638813490985, 4.4431147205208923, 3.0392301612263246, -2.3032611338556377]]], [[[1.1388924488325571, 
4.4978561941078308, -3.3123851704811691, 1.3453478111463726], [4.1779635175178385, 3.1786527767023234, 
-2.8109803623964669, 4.7217176158252876], [0.26914741902392958, -1.6630169842885789, -3.6267544687045641, 
-4.7016327677304943]], [[0.44478691577550755, 2.9451130426961889, -1.0836274217802466, -4.8754431681482586], 
[1.6457024072282014, -1.106310648992209, -3.2732924796145912, 4.7940609535301668], [-4.2482158844391957, 
2.2391243759174451, 4.6408645091714327, 4.1449515947243611]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-8.0514002930449351, -3.1903400589440558, -7.3629447918099649, -3.734735516897782], 
[-5.3693035511203409, -0.025319596959475277, -4.9360938116558923, 0.5865376413957275], [0.68099628013661473, 
-6.5485642507204904, -3.0871291767936793, -1.8838131624661862]], [[1.5918620128632721, -1.233550883075341, 
-0.15854871929198033, -3.0062065463156937], [-3.0370101357713253, -5.2457915886931836, -6.9658206783046008, 
-5.3031707947882651], [-4.3574049744686887, -6.0194920089489594, -7.2959336679465761, -4.6361236011667906]]], 
[[[0.66068373824176874, -3.2747622044494955, -5.0237980052235205, -2.187444608901691], [1.0515897395585032, 
-2.3086575929864548, -6.0897063031692911, -0.79400993559355371], [-1.0002016200324126, -3.000154121209663, 
-0.23947538263813506, 0.5076968405798663]], [[-7.668350523919699, -7.8354800664459674, -8.3343710920629359, 
-1.9678825156458402], [-3.8590517735916503, -2.2540109910503849, 0.48439370423871075, -4.8452100060794692], 
[-5.7768091751335531, 1.0741694267364377, -0.32971513255813001, -5.6722064276400923]]], [[[-2.2300528449518975, 
1.1289109003233762, -6.6813304642656242, -2.023597482638082], [0.80901822373338383, -0.19029251708213124, 
-6.1799256561809219, 1.352772322040833], [-3.099797874760525, -5.0319622780730331, -6.9956997624890187, 
-8.0705780615149489]], [[-2.9241583780089471, -0.42383225108826572, -4.4525727155647008, -8.2443884619327132], 
[-1.7232428865562532, -4.4752559427766636, -6.6422377733990459, 1.4251156597457122], [-7.6171611782236504, 
-1.1298209178670096, 1.2719192153869781, 0.77600630093990652]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-9.3120202792456048, -4.4509600451447255, -8.6235647780106355, 
-4.9953555030984518], [-6.6299235373210106, -1.285939583160145, -6.1967137978565621, -0.67408234480494222], 
[-0.579623706064055, -7.8091842369211601, -4.347749162994349, -3.144433148666856]], [[0.33124202666260238, 
-2.4941708692760107, -1.4191687054926501, -4.2668265325163635], [-4.297630121971995, -6.5064115748938534, 
-8.2264406645052706, -6.5637907809889349], [-5.6180249606693584, -7.2801119951496291, -8.5565536541472458, 
-5.8967435873674603]]], [[[-0.59993624795890099, -4.5353821906501652, -6.2844179914241902, -3.4480645951023607], 
[-0.20903024664216652, -3.5692775791871245, -7.3503262893699608, -2.0546299217942234], [-2.2608216062330824, 
-4.2607741074103327, -1.5000953688388048, -0.75292314562080342]], [[-8.9289705101203687, -9.0961000526466371, 
-9.5949910782636056, -3.2285025018465099], [-5.11967175979232, -3.5146309772510547, -0.77622628196195897, 
-6.1058299922801389], [-7.0374291613342228, -0.18645055946423206, -1.5903351187587997, -6.932826413840762]]], 
[[[-3.4906728311525672, -0.13170908587729357, -7.9419504504662939, -3.2842174688387518], [-0.45160176246728589, 
-1.450912503282801, -7.4405456423815917, 0.092152335840163246], [-4.3604178609611948, -6.2925822642737028, 
-8.2563197486896875, -9.3311980477156187]], [[-4.1847783642096168, -1.6844522372889355, -5.7131927017653705, 
-9.505008448133383], [-2.983862872756923, -5.7358759289773333, -7.9028577595997156, 0.16449567354504246], 
[-8.8777811644243201, -2.3904409040676793, 0.011299229186308324, -0.48461368526076321]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([-4.9434811071655114, 1.7588416724781917]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([3.0524482361043965, -0.58828792238396233]))
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(-4.86003727467)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-9.8035183818403411, -3.1011956021966389]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-1.8075890385704341, -5.4483251970587929]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([0.47124983588436109, 3.3842142103059487]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([4.4506172428158504, -1.5976912605342894]))
      arg1=Symbol(shape=(2,))
      res=arg0+arg1
      s1=numpy.array([2.7380372395241483, -1.2414970456241372])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([3.2092870754085094, 2.1427171646818115]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([7.1886544823399987, -2.8391883061584267]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[3.7123556177495072, -1.2322724929891438, -4.3196981967098704, 4.5149190397092358, 
-3.4294461596271342], [-0.32526237821140569, 4.906418518064358, 1.6782843293160443, -4.5452294423093242, 
-3.4252951962126454], [4.7623389482797158, 4.8957853100883888, 2.4605965522735644, -3.3235939770772349, 
-3.6622677868193731], [3.7849671492059009, -3.7965523255405484, -0.98706292680421903, -2.9575953641431996, 
3.7235194699440495]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[3.846235478086534, -2.9152984736534773, 2.1299170235868692, 
1.4194093106373815, -1.9728564928751369], [0.12730504885223404, -2.4537968289763077, 1.8352652361138375, 
-1.1054616749639532, -0.67553225283567997], [-4.6542627767136047, 0.014905560429250286, 0.84138572626791408, 
-1.4074784720342515, -3.3322631066777983], [-0.64893500421415951, 4.4524265176475826, -3.5204114624144456, 
3.5239615703390363, 2.3718443568961201]]))
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(3.4845259086)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[7.1968815263516515, 2.2522534156130005, -0.83517228810772615, 7.9994449483113801, 
0.055079748975010112], [3.1592635303907386, 8.3909444266665023, 5.1628102379181886, -1.06070353370718, 
0.059230712389498841], [8.2468648568818601, 8.3803112186905331, 5.9451224608757087, 0.16093193152490937, 
-0.17774187821722887], [7.2694930578080452, -0.31202641693840416, 2.4974629817979253, 0.52693054445894472, 
7.2080453785461938]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[7.3307613866886783, 0.56922743494866701, 5.6144429321890135, 
4.9039352192395258, 1.5116694157270074], [3.6118309574543783, 1.0307290796258366, 5.3197911447159818, 
2.3790642336381911, 2.8089936557664643], [-1.1697368681114604, 3.4994314690313946, 4.3259116348700584, 
2.0770474365678928, 0.15226280192434594], [2.8355909043879848, 7.9369524262497269, -0.035885553812301296, 
7.0084874789411806, 5.8563702654982643]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[2.7675952117994296, 0.98431175880226363, -1.8309000840442566, 2.0351166910383416, 
2.1718600084175153], [0.64718493825654111, 3.0274641310077364, 4.6031246235215555, -0.072830522019846633, 
-3.436466903373192], [-2.7989895712459734, 3.2804563231391093, 3.1416998470123456, 0.25702028842752966, 
-3.1553411419958821], [-4.5620989116806543, -0.23300222673645532, -2.3978689464069101, 0.41391436589174457, 
-3.7252639362836382]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-2.1509506437818238, -2.5007519800405218, 0.30616207266744233, 
-0.46790716227581797, 0.6454558125610621], [1.9589653025955753, -4.9059174981425437, -4.7107956989445992, 
2.6150016745692826, -3.3329567586885211], [1.1850451086308738, 3.8781029980110997, -4.7104324292639133, 
-4.8362413881812492, 4.9066980390674555], [-1.2440311634968171, -1.6429522113717008, 4.0547225056117124, 
-0.33314796054153195, -2.6143781039708855]]))
      arg1=Symbol(shape=(4, 5))
      res=arg0+arg1
      s1=numpy.array([[-0.0104190624259477, 3.439083370835446, -1.7585221913131677, 3.8784501968475897, 
0.08088556648108991], [0.53276272310770789, -1.3171951284400176, -0.841014288686317, 2.4350359443944622, 
0.55796159262639922], [-3.3985580423616479, 0.73804937880111687, 0.84641655693241269, -2.0376479444757822, 
-0.094456394031885438], [0.8829252865168975, 0.84170422580042903, -1.9539396350167637, -4.8054718599517194, 
-0.37594711864698205]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[2.7571761493734819, 4.4233951296377096, -3.5894222753574243, 5.9135668878859313, 
2.2527455748986052], [1.179947661364249, 1.7102690025677187, 3.7621103348352385, 2.3622054223746156, 
-2.8785053107467928], [-6.1975476136076217, 4.0185057019402262, 3.9881164039447583, -1.7806276560482526, 
-3.2497975360277676], [-3.6791736251637568, 0.60870199906397371, -4.3518085814236738, -4.3915574940599749, 
-4.1012110549306202]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-2.1613697062077715, 0.93833139079492422, -1.4523601186457253, 
3.4105430345717718, 0.72634137904215201], [2.4917280257032832, -6.2231126265825614, -5.5518099876309162, 
5.0500376189637448, -2.7749951660621219], [-2.2135129337307742, 4.6161523768122166, -3.8640158723315006, 
-6.8738893326570309, 4.8122416450355701], [-0.36110587697991958, -0.80124798557127175, 2.1007828705949487, 
-5.1386198204932514, -2.9903252226178676]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[1.6094791339338048, 4.27222307751477], [4.9486531857239697, -4.5552975586923292]], 
[[-0.12032729123703056, -4.1413061177629231], [-2.7473350985925316, 4.7319188820310991]], [[0.13107637034429231, 
-3.2138415379490204], [-3.9942457581718696, 1.3262496008026838]], [[2.56850905863657, 1.8321753808437329], 
[4.5176482730823331, 4.4664637318837137]], [[0.50860355331966556, 0.55279434819439199], [3.1688695988617859, 
-2.6740526298455016]], [[4.4977965557520072, 3.6422271944652209], [3.7948343945899445, 
-3.0377990068633332]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-2.9548694146760557, 3.1101651017467038], [-0.31006440672923752, 
0.74616091042484989]], [[-3.1016477433464864, 2.9532816390640111], [-2.0494474684559894, -1.1448583599993354]], 
[[4.2052724347365604, -1.8157003708847643], [4.8073133555422327, -2.7045312989764492]], [[-2.3803833325202763, 
0.19928505008920272], [-2.8622812030202094, 3.9488692362256081]], [[-4.1266217915470236, 4.8461083576413735], 
[-3.1895474177762351, 4.4625154514412237]], [[-0.65350755924337811, 2.8015786665738105], [0.94103003425367859, 
0.27556367440023166]]]))
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(4.49324308458)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[6.1027222185118468, 8.765466162092812], [9.4418962703020117, -0.062054474114287217]], 
[[4.3729157933410114, 0.35193696681511888], [1.7459079859855104, 9.2251619666091411]], [[4.6243194549223343, 
1.2794015466290216], [0.49899732640617245, 5.8194926853807258]], [[7.061752143214612, 6.3254184654217749], 
[9.0108913576603751, 8.9597068164617557]], [[5.0018466378977076, 5.046037432772434], [7.6621126834398279, 
1.8191904547325404]], [[8.9910396403300492, 8.1354702790432629], [8.2880774791679865, 
1.4554440777147089]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[1.5383736699019863, 7.6034081863247458], [4.1831786778488045, 
5.2394039950028919]], [[1.3915953412315556, 7.4465247236420531], [2.4437956161220526, 3.3483847245787066]], 
[[8.6985155193146024, 2.6775427136932777], [9.3005564401202747, 1.7887117856015928]], [[2.1128597520577657, 
4.6925281346672447], [1.6309618815578326, 8.4421123208036501]], [[0.36662129303101842, 9.3393514422194155], 
[1.3036956668018069, 8.9557585360192657]], [[3.8397355253346639, 7.2948217511518525], [5.4342731188317206, 
4.7688067589782737]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[-2.7345315461324993, 4.5316724428402377], [-1.2207000383039999, -2.1651454481686692]], 
[[-2.5222456135735638, 3.1325113872519896], [0.54140311786327011, -1.6266115642059011]], [[4.3999274072752783, 
-0.64510581732829841], [-3.3878893926233533, -0.14783111107246061]], [[2.4816188811184228, 1.505965932327137], 
[-2.8128544405052458, 3.2460332510852936]], [[1.5649806120186849, 1.1768584297160487], [-3.3133262672401544, 
-2.5740884272652789]], [[2.936076596237732, -0.80694051724477056], [1.6382059835800931, 
-0.059174653042079584]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[4.107948776768561, 4.79459166600315], [-0.070211802843057391, 
-2.3000592273671394]], [[1.53142006950028, 0.5983353676488381], [4.2000369856633419, -3.7326077043834074]], 
[[-3.6852528003303684, -0.40061815593309014], [4.849947657932514, 3.2046322763443698]], [[4.6824735127774275, 
-2.3356975272114679], [-1.4284737023138216, -0.96863966970867921]], [[4.4306883649430571, 0.16250464015770305], 
[4.7866411719098583, -1.6949698779239197]], [[-4.9624929004021014, -0.4120760567738655], [-3.510925072784119, 
-0.26388846668772636]]]))
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0+arg1
      s1=numpy.array([[[3.7560333190798687, 0.63030183757017788], [-3.8821224320935288, 4.3508142113739634]], 
[[4.3548667192676795, -3.4709315123037445], [-0.19540447292770935, -1.1720138856956916]], [[3.7993994701980398, 
-4.5475458462287497], [-0.20650310401114513, -2.7802894344079201]], [[-0.46867874332271242, 0.82685022383334505], 
[-3.5357776147305264, 0.7633420403065605]], [[-0.19578164461526359, -4.1370261640670458], [-1.2073883253186946, 
0.74664652191646397]], [[-0.697880661399644, -0.46932885527321488], [2.4087818009804716, -1.8245102799854829]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[1.0215017729473694, 5.1619742804104156], [-5.1028224703975287, 2.1856687632052942]], 
[[1.8326211056941157, -0.33842012505175489], [0.34599864493556076, -2.7986254499015928]], [[8.1993268774733181, 
-5.1926516635570481], [-3.5943924966344984, -2.9281205454803807]], [[2.0129401377957103, 2.3328161561604821], 
[-6.3486320552357718, 4.0093752913918541]], [[1.3691989674034213, -2.9601677343509971], [-4.520714592558849, 
-1.8274419053488149]], [[2.238195934838088, -1.2762693725179854], [4.0469877845605646, 
-1.8836849330275625]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[7.8639820958484297, 5.4248935035733279], [-3.9523342349365862, 
2.050754984006824]], [[5.8862867887679595, -2.8725961446549064], [4.0046325127356326, -4.904621590079099]], 
[[0.11414666986767141, -4.9481640021618398], [4.6434445539213689, 0.42434284193644967]], [[4.2137947694547151, 
-1.5088473033781229], [-4.9642513170443481, -0.20529762940211871]], [[4.2349067203277935, -3.9745215239093428], 
[3.5792528465911637, -0.94832335600745576]], [[-5.6603735618017454, -0.88140491204708038], [-1.1021432718036475, 
-2.0883987466732092]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[1.2403285479679145, -0.65355746314869823, 0.23507371305026048, 2.9495208917061202], 
[-4.4153187452600653, -1.0271324152128747, 3.6087985228033794, 1.587633224392107], [1.5882989512534262, 
-2.3766989521547401, -4.6462509853387939, 1.1425676014861166]], [[-4.8469447836806694, -1.4338245370809863, 
-4.8441809139347694, 0.082128480181090424], [4.2695412477206585, -2.0376229192188622, -2.685821131586259, 
-4.5654361329152717], [3.5226403567783482, -4.9633770210253347, 4.1637469549065127, -3.5898874968684167]]], 
[[[2.7439089503129228, 0.81346375693975492, -2.576882111469688, 4.758878084101946], [0.098363354586225249, 
-4.314913184354209, -1.1821682575010484, 4.9687115939178916], [-2.5414207769554564, 1.9836872846103208, 
-1.5982744174212127, 4.5509211096426121]], [[4.759533396882766, -4.550347299113696, 4.9394743649799153, 
-3.9692445921595421], [1.5755016838325195, 2.6599597206311305, -0.59545966103916648, -1.308464088815966], 
[1.7018715016873482, 0.31781368103450536, -0.91184792887657995, -0.60566457689943931]]], [[[-0.365764084374395, 
-0.75878286483821444, -3.1104661623240091, -3.7302303444372109], [0.58052395594970907, 0.14085590954626337, 
4.6712439745076182, 0.65991412045590181], [-4.5675491076195733, -3.3042112830144132, -2.6719400309110553, 
-3.8520603991598765]], [[3.4260488825099618, -1.2789319515430164, 1.8435112511824903, 1.0773214658952854], 
[-4.0772283149901236, 1.0211433275718873, -2.015430043082814, 0.1376630245430368], [1.3249956905172624, 
3.1987247807146968, 1.0304156332749459, 3.785256475561086]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-3.8774766185796605, 3.1521364883779448, -4.9233158714840091, 
3.7988193665209522], [4.8244393256113263, 2.4688683468745563, -4.5044275072582254, 1.1107496985072052], 
[-2.9980383766650376, -4.2922660982517158, 3.4924104659712771, -0.5135964311738892]], [[1.9573144047865201, 
-2.2686101409008961, -2.907052414660404, -4.0582253229051144], [-2.0281877168409657, 1.7867206317317663, 
0.018511114285918673, -4.0475974398672498], [1.3023403490307315, 1.9932255873687215, -4.6698465653310688, 
-4.5630845029599421]]], [[[-1.9525649263627876, -0.72040110769848908, -3.6987029249472769, -3.3184217891099999], 
[-4.0519149413902857, 4.1195877398536549, -3.8261874289376463, 3.423780007792768], [0.11768639970294359, 
-1.4898880703788131, -1.1746648112150213, -0.28493737967147226]], [[-2.0138403307539932, 3.9987186392010816, 
-1.0125535260055338, 0.57376641241565363], [4.213727608092972, 0.51388058678005066, -4.4106027756910908, 
-1.9979423050108283], [1.5708368447511347, -1.6270284297780933, -0.55277364435139376, -1.7748804647831715]]], 
[[[2.7639070541103061, 2.7303808332951629, 0.41148416591473591, -1.9337000414572802], [-2.7585163378482456, 
2.2319457297797207, 3.7988668025967804, 3.6103374331669471], [-4.5925114196923271, -2.1274746711435997, 
3.3094547630756779, -4.1386856959210352]], [[-2.1348423629137692, 3.539794593057783, 4.8265405725541157, 
4.9426398297282788], [4.5757071915543417, -4.0433372993763399, -0.84096548582416997, 2.0567811910343226], 
[4.5367596882428671, -4.9139510999364404, 1.1342166543217944, 1.4859311895053571]]]]))
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(4.83582066753)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[6.0761492154946453, 4.1822632043780326, 5.0708943805769913, 7.785341559232851], 
[0.4205019222666655, 3.8086882523138561, 8.4446191903301102, 6.4234538919188378], [6.424119618780157, 
2.4591217153719906, 0.18956968218793691, 5.9783882690128474]], [[-0.011124116153938601, 3.4019961304457444, 
-0.008360246408038563, 4.9179491477078212], [9.1053619152473892, 2.7981977483078686, 2.1499995359404718, 
0.27038453461145906], [8.358461024305079, -0.12755635349860395, 8.9995676224332435, 1.2459331706583141]]], 
[[[7.5797296178396536, 5.6492844244664857, 2.2589385560570427, 9.5946987516286768], [4.934184022112956, 
0.52090748317252178, 3.6536524100256824, 9.8045322614446224], [2.2943998905712744, 6.8195079521370516, 
3.2375462501055181, 9.3867417771693429]], [[9.5953540644094968, 0.28547336841303483, 9.7752950325066461, 
0.86657607536718873], [6.4113223513592503, 7.4957803881578613, 4.2403610064875643, 3.5273565787107648], 
[6.537692169214079, 5.1536343485612361, 3.9239727386501508, 4.2301560906272915]]], [[[4.4700565831523358, 
4.0770378026885163, 1.7253545052027217, 1.1055903230895199], [5.4163446234764399, 4.9766765770729942, 
9.507064642034349, 5.4957347879826326], [0.26827155990715745, 1.5316093845123175, 2.1638806366156755, 
0.98376026836685426]], [[8.2618695500366925, 3.5568887159837144, 6.679331918709221, 5.9131421334220162], 
[0.75859235253660717, 5.8569639950986181, 2.8203906244439167, 4.9734836920697676], [6.1608163580439932, 
8.0345454482414276, 5.8662363008016767, 8.6210771430878168]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.95834404894707026, 7.9879571559046756, -0.087495203957278278, 
8.634640034047683], [9.6602599931380571, 7.304689014401287, 0.3313931602685054, 5.946570366033936], 
[1.8377822908616932, 0.543554569275015, 8.3282311334980079, 4.3222242363528416]], [[6.7931350723132509, 
2.5672105266258347, 1.9287682528663268, 0.77759534462161639], [2.8076329506857651, 6.6225412992584971, 
4.8543317818126495, 0.78822322765948094], [6.1381610165574623, 6.8290462548954523, 0.165974102195662, 
0.27273616456678873]]], [[[2.8832557411639432, 4.1154195598282417, 1.1371177425794539, 1.5173988784167309], 
[0.78390572613644505, 8.9554084073803857, 1.0096332385890845, 8.2596006753194988], [4.9535070672296744, 
3.3459325971479177, 3.6611558563117095, 4.5508832878552585]], [[2.8219803367727376, 8.8345393067278124, 
3.823267141521197, 5.4095870799423844], [9.0495482756197028, 5.3497012543067815, 0.42521789183563996, 
2.8378783625159025], [6.4066575122778655, 3.2087922377486375, 4.283047023175337, 3.0609402027435593]]], 
[[[7.5997277216370369, 7.5662015008218937, 5.2473048334414667, 2.9021206260694505], [2.0773043296784852, 
7.0677663973064515, 8.6346874701235112, 8.4461581006936779], [0.24330924783440366, 2.7083459963831311, 
8.1452754306024087, 0.6971349716056956]], [[2.7009783046129616, 8.3756152605845138, 9.6623612400808465, 
9.7784604972550095], [9.4115278590810725, 0.79248336815039089, 3.9948551817025608, 6.8926018585610533], 
[9.3725803557695979, -0.078130432409709627, 5.9700373218485252, 6.3217518570320879]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_taggedData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[-3.1509236523814286, 1.680234058442708, -1.7187977550532416, 3.9846453843972913], 
[-1.6754979614332322, -3.8450074807346901, -1.5740330789137689, -4.4201074343218751], [2.276529915966389, 
-0.80235747833916982, 4.571247045598767, -3.4093255486695617]], [[-4.0166628667791446, -1.3933240066153738, 
-1.215071574667598, -3.4706735067142258], [-3.0960303329082572, 4.3009033191704589, 4.4065883064621634, 
4.8965445768019009], [-4.4443460968929758, 3.8975314333052253, -4.4153045047286144, 1.7496820405056166]]], 
[[[1.634274247051799, -2.4623052709302771, 1.4279180811059975, 0.92544783745377668], [-4.4862942162658106, 
-0.17080151547727951, 0.52532922395695625, -0.11419327223481623], [-1.1603038628614835, -2.5757515035829472, 
1.9959550719114718, -1.7953240768392242]], [[4.9309159450812103, 3.2298165897638906, -0.075208625571880461, 
-1.1899071115534432], [1.6545058865005409, -1.9426363189361773, 1.620629502101667, -4.2257681218133687], 
[-0.24689686416986767, 2.1247379677905815, -0.022501917990521925, -1.9988138278359822]]], [[[-2.16170138942825, 
1.2184335532362125, 1.1509535832826323, 2.2195238124001797], [2.7455643566460015, 4.6453581322389361, 
-4.1082447076462643, -4.0639146315693067], [-4.96116105494092, -3.6915142795866762, -1.2186796693827917, 
4.7933913234222967]], [[2.0022553772723217, -0.96891528014022654, -2.5457411370843142, -3.3574915783043058], 
[0.10326637441549735, 2.2065594442944327, 3.4159550457557479, -0.71182719653128945], [-1.5473005591196651, 
-1.8237704422942014, 3.7660184612895105, -2.1565964302540372]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-2.7644812297675436, 0.40931971763703956, 3.611075059192606, 
0.50972765741910564], [-4.2130726841282584, -1.1190277433669751, -0.71203745760782766, -3.152956525368753], 
[-1.6186056313723087, 1.1274726343098616, 4.4133392834898437, 1.5220424195160689]], [[0.16147933294385375, 
2.4654462130650998, -2.2315133839410328, -4.5248215067907562], [2.2226933853289026, 3.7083490689582508, 
1.6042940030913613, 0.26178935291219929], [2.4033332562872989, 2.6116613010273229, -3.5340848426974594, 
-4.3871506552920767]]], [[[-2.5011422414749243, -2.9785737952530678, -4.0632268435384287, -2.9061747268645899], 
[-3.4361922491984487, 0.92512310228203631, -3.7591410062368915, -0.10199113857196274], [1.4370716393838645, 
0.71874746237537668, -4.5480615526025323, -3.9385610102938093]], [[-3.5039474073115562, 1.4740925776889409, 
-0.06403798877318323, -3.3828440686373753], [-1.9590119108809123, -0.13446729158123816, -2.4360152863347251, 
0.81375486060557112], [2.4638296949211451, 0.84554464160795018, 1.0770605717668191, 0.90311465710515648]]], 
[[[-3.0365259446312756, -2.1113062138954444, 3.190598106141481, 4.7146234105400531], [4.7073713389281071, 
2.0949812753843036, 1.902801485931489, -0.4384294077249864], [-4.4341512258710214, 4.114619941421422, 
4.1663347911930675, -0.082374028629738305]], [[-0.58950965471106098, -1.9744112566224792, -0.0098348725084971278, 
2.3871548847218813], [-1.1861224380121662, -3.8703032573387253, 0.2332725218101972, 2.7881117501797101], 
[-4.3313677243610327, 2.5428749523942127, 3.9018944633638419, -0.49408732338659789]]]]))
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0+arg1
      s1=numpy.array([[[[1.8433628252117984, 1.5322432245117268, 0.55363793461945665, 4.6657626927783653], 
[-0.94710403494804751, 3.9800168829397649, 3.0366988370600794, 2.8875431155604332], [-1.188024345098996, 
1.0665386751463011, 4.7835901054797993, 2.5969696632689807]], [[-1.99850752062535, 1.1333681341555639, 
-0.49718999089842697, 1.1440753369804515], [0.26294280812698378, -3.8684363170040701, 0.061030108864615684, 
-4.1179127492349608], [-4.67031644465197, 4.9054510497550492, -0.2640662442281041, 1.363134852748785]]], 
[[[-1.4621905107325697, -2.8811881835070574, -2.0127263016810106, 3.9187151372775499], [4.0559843147336121, 
3.8748150284806506, -4.7195991819934049, 1.6441241199343715], [1.1018797372155733, 1.5720711461020827, 
-2.8718182782954003, -2.4926472889456743]], [[2.1583981297206112, -2.7029142786449709, -4.0306810999276212, 
-0.041927417439557857], [2.5297094316362001, 3.2023688131127575, -0.87830172094753056, 1.5087811969314782], 
[0.94040146920827272, 1.8042467131134678, 2.6306472495122346, 0.16819275341523543]]], [[[0.15798239523545377, 
2.4104584738150319, 2.3850248364278386, 3.2174938931658534], [4.8575582926065533, 0.30772922316230389, 
-4.4397211951638047, 0.39063821497748741], [-2.3146321369181688, -3.0703095447217885, 1.7397877979741549, 
4.033153568325778]], [[-1.7935270727714037, -3.9682025038313595, -3.4065483616803141, 2.1844510922893523], 
[-4.2449404804537032, 1.9572337718531996, -4.6593011375931308, 0.98236210083608633], [4.8624542464851288, 
0.5657266529616205, 0.50114562982511135, -3.2736237576584317]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-1.3075608271696302, 3.2124772829544348, -1.165159820433785, 8.6504080771756566], 
[-2.6226019963812797, 0.13500940220507474, 1.4626657581463105, -1.5325643187614419], [1.088505570867393, 
0.26418119680713126, 9.3548371510785664, -0.81235588540058101]], [[-6.0151703874044946, -0.25995587245980989, 
-1.7122615655660249, -2.3265981697337743], [-2.8330875247812735, 0.43246700216638878, 4.4676184153267791, 
0.77863182756694016], [-9.1146625415449449, 8.8029824830602745, -4.6793707489567185, 3.1128168932544016]]], 
[[[0.17208373631922935, -5.3434934544373345, -0.58480822057501314, 4.8441629747313266], [-0.4303099015321985, 
3.7040135130033711, -4.1942699580364486, 1.5299308476995552], [-0.058424125645910152, -1.0036803574808646, 
-0.87586320638392845, -4.2879713657848981]], [[7.0893140748018215, 0.52690231111891972, -4.1058897254995017, 
-1.2318345289930011], [4.184215318136741, 1.2597324941765802, 0.74232778115413645, -2.7169869248818905], 
[0.69350460503840505, 3.9289846809040494, 2.6081453315217127, -1.8306210744207467]]], [[[-2.0037189941927962, 
3.6288920270512444, 3.5359784197104709, 5.4370177055660331], [7.6031226492525548, 4.95308735540124, 
-8.5479659028100698, -3.6732764165918192], [-7.2757931918590888, -6.7618238243084647, 0.52110812859136324, 
8.8265448917480747]], [[0.20872830450091806, -4.9371177839715861, -5.9522894987646282, -1.1730404860149535], 
[-4.1416741060382058, 4.1637932161476323, -1.2433460918373829, 0.27053490430479687], [3.3151536873654637, 
-1.2580437893325809, 4.2671640911146218, -5.430220187912469]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.92111840455574523, 1.9415629421487663, 4.1647129938120626, 
5.1754903501974709], [-5.1601767190763059, 2.8609891395727898, 2.3246613794522517, -0.26541340980831984], 
[-2.8066299764713047, 2.1940113094561626, 9.1969293889696431, 4.1190120827850496]], [[-1.8370281876814962, 
3.5988143472206637, -2.7287033748394598, -3.3807461698103047], [2.4856361934558864, -0.16008724804581931, 
1.665324111955977, -3.8561233963227615], [-2.2669831883646712, 7.5171123507823721, -3.7981510869255635, 
-3.0240158025432917]]], [[[-3.9633327522074939, -5.8597619787601252, -6.0759531452194393, 1.0125404104129601], 
[0.61979206553516342, 4.7999381307626869, -8.4787401882302973, 1.5421329813624087], [2.5389513765994378, 
2.2908186084774593, -7.4198798308979326, -6.4312082992394837]], [[-1.345549277590945, -1.22882170095603, 
-4.0947190887008045, -3.4247714860769332], [0.57069752075528779, 3.0679015215315193, -3.3143170072822556, 
2.3225360575370493], [3.4042311641294178, 2.649791354721418, 3.7077078212790537, 1.0713074105203919]]], 
[[[-2.8785435493958218, 0.29915225991958749, 5.5756229425693196, 7.9321173037059065], [9.5649296315346604, 
2.4027104985466075, -2.5369197092323157, -0.047791192747498989], [-6.7487833627891902, 1.0443103966996334, 
5.9061225891672224, 3.9507795396960397]], [[-2.3830367274824646, -5.9426137604538383, -3.4163832341888112, 
4.5716059770112336], [-5.4310629184658694, -1.9130694854855257, -4.4260286157829336, 3.7704738510157965], 
[0.53108652212409613, 3.1086016053558332, 4.4030400931889533, -3.7677110810450296]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.481249850026)+(1.-msk_arg0)*(-1.48465416864)
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(-2.65110429185)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(-3.13235414188)+(1.-msk_ref)*(-4.13575846049)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.13411439983)+(1.-msk_arg0)*(-0.629637549331)
      arg1=Symbol(shape=(2,))
      res=arg0+arg1
      s1=numpy.array([-0.62992419613163175, 4.55886114005793])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([0.50419020369403444, 5.6929755398835962])+(1.-msk_ref)*numpy.array([-1.259561745462479, 
3.9292235907270827])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(3.01809294358)+(1.-msk_arg0)*(0.889743657807)
      arg1=Symbol(shape=(4, 5))
      res=arg0+arg1
      s1=numpy.array([[-2.793178683106079, -2.6222774715493582, 1.0142792223620747, -3.0640922264732984, 
-2.3554298671206055], [0.088775964219395043, 3.4441381957619619, 3.3892189758872853, 2.7423767697866088, 
3.977644321141641], [1.4526982641352157, 2.2184052986969505, -3.952710218879385, -4.7169576073736375, 
-0.7937042808225101], [2.2686916098744314, -1.553248315886353, -2.7367045745859819, 3.7958840729585344, 
1.4548199443717298]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.22491426047411567, 0.39581547203083645, 4.0323721659422693, -0.045999282893103732, 
0.66266307645958911], [3.1068689077995897, 6.4622311393421565, 6.4073119194674799, 5.7604697133668035, 
6.9957372647218357], [4.4707912077154104, 5.2364982422771451, -0.93461727529919036, -1.6988646637934428, 
2.2243886627576845], [5.2867845534546261, 1.4648446276938416, 0.28138836899421271, 6.813977016538729, 
4.4729128879519244]])+(1.-msk_ref)*numpy.array([[-1.9034350252987218, -1.732533813742001, 1.9040228801694319, 
-2.1743485686659412, -1.4656862093132483], [0.97851962202675224, 4.3338818535693191, 4.2789626336946425, 
3.632120427593966, 4.8673879789489982], [2.3424419219425729, 3.1081489565043077, -3.0629665610720278, 
-3.8272139495662802, 0.096039376984847102], [3.1584352676817886, -0.66350465807899583, -1.8469609167786247, 
4.6856277307658916, 2.344563602179087]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.98444562132)+(1.-msk_arg0)*(4.30756765987)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0+arg1
      s1=numpy.array([[[1.9993822405268356, -3.1230808428690615], [4.9036400439562815, -4.8838867997176525]], 
[[0.42763250705520939, 1.7579324334230453], [-3.7242679708963458, 1.8833596506298056]], [[-3.5481907533254931, 
0.2040318933875751], [-2.5124574767604746, -4.1576503017979416]], [[2.4187154671810562, -0.51775884222858526], 
[-1.722028671225063, 4.8177194310600537]], [[3.5460779618762999, 3.7426721831596925], [-3.14876579453641, 
-1.8491069265603413]], [[-2.0602497125201733, 1.8445672729830882], [2.6289048953955998, -2.1171625740448654]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-2.9850633807947604, -8.1075264641906575], [-0.080805577365314463, 
-9.8683324210392485]], [[-4.5568131142663866, -3.2265131878985507], [-8.7087135922179417, -3.1010859706917904]], 
[[-8.5326363746470886, -4.7804137279340209], [-7.4969030980820701, -9.1420959231195376]], [[-2.5657301541405397, 
-5.5022044635501812], [-6.7064742925466589, -0.16672619026154223]], [[-1.4383676594452961, -1.2417734381619034], 
[-8.1332114158580069, -6.8335525478819372]], [[-7.0446953338417693, -3.1398783483385078], [-2.3555407259259962, 
-7.1016081953664614]]])+(1.-msk_ref)*numpy.array([[[6.3069499004015404, 1.1844868170056433], [9.2112077038309863, 
-0.57631913984294769]], [[4.7352001669299142, 6.0655000932977501], [0.58329968897835904, 6.1909273105045104]], 
[[0.75937690654921175, 4.5115995532622799], [1.7951101831142302, 0.14991735807676321]], [[6.726283127055761, 
3.7898088176461195], [2.5855389886496418, 9.1252870909347585]], [[7.8536456217510047, 8.0502398430343973], 
[1.1588018653382948, 2.4584607333143635]], [[2.2473179473545315, 6.152134932857793], [6.9364725552703046, 
2.1904050858298394]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank0_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-2.9697925334)+(1.-msk_arg0)*(-4.26135335725)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0+arg1
      s1=numpy.array([[[[3.9689996783063126, 2.6024749301521517, -2.8657897182202263, 3.4523361907793202], 
[1.0646468808240472, 2.2809214673673006, 1.9110441510817342, 3.6637536830808415], [-4.8161620946685977, 
1.1260192950202335, -1.5444099528131283, 4.5856953227320361]], [[3.4807853259935388, 1.0632821522370133, 
-1.7813251042294, 0.96803702807832348], [-2.2395880868316476, 4.8919502166960243, 3.0915081953974273, 
-0.85921425228962178], [-0.24500754865585961, -3.000069805276242, -2.3285433357124861, -3.7526812827715004]]], 
[[[-2.6148866735769314, -2.9426881222754986, -2.1105189060422127, -1.718323686970705], [0.38236683235255065, 
4.8146833101999391, -0.69724678041282662, -3.674837501299455], [-1.1217878757973345, 1.9457797122429064, 
4.3330454272287042, 1.2870165165330079]], [[0.90390350707926448, 4.0932246664578322, 4.0170833493811937, 
2.3057200276883218], [-4.1149618340720506, 4.3206785552080422, 4.5478406361616468, 3.4270491303459689], 
[-3.2122582790653578, -0.051138136931458078, 2.847106348954056, -2.0922906343243097]]], [[[-3.8470709835005801, 
0.79389346854249432, 1.9702586564654192, -1.230993932131331], [0.52027641197917784, 4.1606002966489264, 
-4.1240899145057277, 3.0855602864655047], [1.2434749670286918, 1.9421106344042691, -4.7997149299258455, 
-3.1016051858236517]], [[-4.0158867307020536, -1.2810983979769732, 4.1806447574751786, 2.4159993753375488], 
[3.8210591526688589, 2.9170696329659753, 0.212629682453775, -3.6791629346607402], [-0.52709663403725493, 
-2.0893727810689953, -1.7473644406170976, -4.1869442335699976]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.99920714490574225, -0.36731760324841867, -5.8355822516207967, 0.48254365737874982], 
[-1.9051456525765231, -0.68887106603326975, -1.0587483823188362, 0.69396114968027112], [-7.7859546280691685, 
-1.8437732383803369, -4.5142024862136987, 1.6159027893314657]], [[0.51099279259296848, -1.9065103811635571, 
-4.7511176376299709, -2.0017555053222469], [-5.2093806202322179, 1.9221576832954539, 0.12171566199685691, 
-3.8290067856901921], [-3.21480008205643, -5.9698623386768119, -5.2983358691130569, -6.7224738161720712]]], 
[[[-5.5846792069775013, -5.9124806556760685, -5.0803114394427826, -4.6881162203712758], [-2.5874257010480197, 
1.8448907767993687, -3.667039313813397, -6.6446300347000253], [-4.0915804091979044, -1.024012821157664, 
1.3632528938281339, -1.6827760168675625]], [[-2.0658890263213059, 1.1234321330572619, 1.0472908159806233, 
-0.66407250571224852], [-7.0847543674726214, 1.3508860218074719, 1.5780481027610764, 0.45725659694539855], 
[-6.1820508124659277, -3.0209306703320284, -0.12268618444651436, -5.0620831677248805]]], [[[-6.8168635169011509, 
-2.175899064858076, -0.99953387693515117, -4.2007864655319018], [-2.4495161214213925, 1.190807763248356, 
-7.0938824479062976, 0.11576775306493436], [-1.7263175663718786, -1.0276818989963012, -7.7695074633264163, 
-6.0713977192242226]], [[-6.9856792641026235, -4.250890931377544, 1.2108522240746082, -0.55379315806302154], 
[0.8512666192682885, -0.052722900434595044, -2.7571628509467954, -6.6489554680613105], [-3.4968891674378253, 
-5.0591653144695652, -4.7171569740176675, -7.1567367669705675]]]])+(1.-msk_ref)*numpy.array([[[[-0.29235367894345909, 
-1.65887842709762, -7.1271430754699985, -0.80901716647045152], [-3.1967064764257245, -1.9804318898824711, 
-2.3503092061680375, -0.59759967416893023], [-9.0775154519183694, -3.1353340622295383, -5.8057633100629005, 
0.32434196548226435]], [[-0.78056803125623286, -3.1980712050127584, -6.0426784614791718, -3.2933163291714482], 
[-6.5009414440814197, 0.63059685944625254, -1.1698451618523444, -5.1205676095393935], [-4.5063609059056313, 
-7.2614231625260137, -6.5898966929622578, -8.0140346400212721]]], [[[-6.8762400308267031, -7.2040414795252703, 
-6.3718722632919844, -5.9796770442204767], [-3.8789865248972211, 0.5533299529501674, -4.9586001376625983, 
-7.9361908585492262], [-5.3831412330471062, -2.3155736450068654, 0.071692069978932516, -2.9743368407167639]], 
[[-3.3574498501705072, -0.16812869079193948, -0.244270007868578, -1.9556333295614499], [-8.3763151913218223, 
0.059325197958270515, 0.28648727891187509, -0.83430422690380279], [-7.4736116363151295, -4.3124914941812298, 
-1.4142470082957157, -6.3536439915740814]]], [[[-8.1084243407503518, -3.4674598887072774, -2.2910947007843525, 
-5.4923472893811027], [-3.7410769452705939, -0.10075306060084532, -8.3854432717554985, -1.175793070784267], 
[-3.01787839022108, -2.3192427228455026, -9.0610682871756172, -7.3629585430734235]], [[-8.2772400879518244, 
-5.5424517552267449, -0.080708599774593104, -1.8453539819122229], [-0.44029420458091284, -1.3442837242837964, 
-4.0487236747959967, -7.9405162919105123], [-4.7884499912870266, -6.350726138318767, -6.0087177978668693, 
-8.4482975908197702]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank1_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([2.1945719955206853, 
-3.4851810549539852])+(1.-msk_arg0)*numpy.array([-3.159460740559509, 1.0507096466806898])
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(2.92811762582)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([5.1226896213358133, 
-0.5570634291388572])+(1.-msk_ref)*numpy.array([-0.23134311474438096, 3.9788272724958178])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank1_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([1.9387192390641195, 
-2.294788495198282])+(1.-msk_arg0)*numpy.array([-3.9950296964046816, -4.9584579002903517])
      arg1=Symbol(shape=(2,))
      res=arg0+arg1
      s1=numpy.array([0.68148355985483988, 0.33396702170122339])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([2.6202027989189594, -1.9608214734970586])+(1.-msk_ref)*numpy.array([-3.3135461365498418, 
-4.6244908785891283])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank2_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[1.9335525790389809, 4.8876884032830024, -3.6794048434152948, -2.9337672885330814, 
0.5880232587543972], [1.2731441866942719, 4.8021715240969982, 2.9871285060348427, 4.3674026791776921, 
2.3324101078324144], [3.257367767879968, 3.614481137699638, -4.0465097244122443, -3.3712543524462166, 
0.83424572698980626], [-4.7734011845397317, -1.1918316514932537, -2.641576771310632, -3.7441723823507447, 
2.5792398168240602]])+(1.-msk_arg0)*numpy.array([[0.51038147587387783, -3.548018657118809, 3.7494118465432393, 
3.6729170048063136, -2.9522974158811746], [3.2109365766033289, -1.7347320393345091, -0.9996429948297223, 
-0.75500884718678307, 1.5928790967815267], [-4.1174844249701259, 4.2030131668606234, -4.8484509001230229, 
2.7032344298767921, 4.3009935101668333], [-1.4527019870327429, 3.9347061378002781, 1.21415230923688, 
-3.666838308237784, -3.8400590973123858]])
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(3.22997214356)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[5.1635247225953336, 8.117660546839355, -0.44943269985894219, 0.29620485502327121, 
3.8179954023107499], [4.5031163302506245, 8.0321436676533509, 6.2171006495911953, 7.5973748227340447, 
5.5623822513887671], [6.4873399114363206, 6.8444532812559906, -0.81653758085589168, -0.14128220888986398, 
4.0642178705461589], [-1.5434290409833791, 2.038140492063099, 0.58839537224572069, -0.51420023879439203, 
5.8092119603804129]])+(1.-msk_ref)*numpy.array([[3.7403536194302305, -0.31804651356245639, 6.979383990099592, 
6.9028891483626662, 0.27767472767517809], [6.4409087201596815, 1.4952401042218435, 2.2303291487266304, 
2.4749632963695696, 4.8228512403378794], [-0.88751228141377325, 7.4329853104169761, -1.6184787565666703, 
5.9332065734331447, 7.5309656537231859], [1.7772701565236098, 7.1646782813566308, 4.4441244527932326, 
-0.43686616468143136, -0.61008695375603317]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank2_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.074742989914646785, -1.8482493880577588, 1.0926262448311599, 4.5158483202643716, 
-3.0805669333005561], [0.0085606966159099684, -2.9696862086974996, 3.3024460854167597, 1.5088165460119427, 
-3.6452065491857266], [0.18694035412066512, -4.6738922180085147, 3.9551045875071438, 4.0084174115638724, 
-0.63332177275981749], [2.5093858800842108, -0.36171911019222946, 0.19138395375626427, -3.1795621861527734, 
-2.6267949144535008]])+(1.-msk_arg0)*numpy.array([[-3.5942187686631524, -3.7060821431133406, 0.9533196788857623, 
-4.8840044000628744, 0.3938790125214453], [4.0652979493208985, 4.5325841421496644, -0.4281905049316661, 
-1.742508580451184, 2.7120740894023898], [0.56888661640784566, -2.4569299021956068, 3.568568120069024, 
-2.0793352745659766, -1.7689628659930126], [-4.8632954420706014, -2.8828667280653364, 3.4090243893802246, 
3.0651732601260697, 4.6463764755640256]])
      arg1=Symbol(shape=(4, 5))
      res=arg0+arg1
      s1=numpy.array([[-1.4953863183942318, -3.5127993001524969, 2.9138150805794103, -1.6144165168200519, 
-0.65062618022498242], [-4.9181569250500168, -2.6971927119277908, 4.2365880197149934, -4.2036145824282496, 
2.2260090531531453], [4.0868409931398002, -3.3893548967194032, 2.9012650531553019, -2.2355683566643378, 
2.9627609193479501], [4.9921359000605019, 0.6569024014440803, 3.3639734573108839, 0.89356331435440595, 
-4.0709626638242327]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[-1.5701293083088785, -5.3610486882102553, 4.0064413254105702, 2.9014318034443196, 
-3.7311931135255385], [-4.9095962284341068, -5.6668789206252903, 7.5390341051317531, -2.6947980364163069, 
-1.4191974960325813], [4.2737813472604653, -8.0632471147279183, 6.8563696406624457, 1.7728490548995346, 
2.3294391465881326], [7.5015217801447127, 0.29518329125185083, 3.5553574110671482, -2.2859988717983675, 
-6.6977575782777334]])+(1.-msk_ref)*numpy.array([[-5.0896050870573841, -7.2188814432658379, 3.8671347594651726, 
-6.4984209168829263, -0.25674716770353712], [-0.85285897572911828, 1.8353914302218737, 3.8083975147833273, 
-5.9461231628794335, 4.9380831425555352], [4.6557276095476459, -5.8462847989150095, 6.4698331732243259, 
-4.3149036312303144, 1.1937980533549375], [0.12884045798990051, -2.2259643266212561, 6.7729978466911085, 
3.9587365744804757, 0.57541381173979289]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank3_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-2.1957568090391955, 0.56747277575122101], [-1.4226171578539604, 
-3.1174336379255854]], [[1.9150168705353749, 0.46771483389240665], [-0.73261624542450932, 1.4533109165427449]], 
[[-4.3700026677098416, -4.4121889510507675], [-4.2432470132589684, -4.6365817911825937]], [[4.3712760608754326, 
0.48815678812850649], [-4.2919585871561221, 2.8753619236403747]], [[4.7410827225779482, -3.2941488290580354], 
[3.5834613437014919, 0.53477849558006074]], [[-2.2697241902980902, 1.4839036193452078], [4.3514574228344109, 
2.0334834769049763]]])+(1.-msk_arg0)*numpy.array([[[1.9065956016010119, 3.8011536401496766], [4.2481111431072272, 
0.7657337986451509]], [[1.7488690210709832, 4.5064595133713876], [-1.261534521038973, -1.5095749568667172]], 
[[1.2010203264269057, 0.055494332510111377], [4.3269730839285749, -0.54412407243328076]], [[-2.6257140205956175, 
-3.4462245120816002], [1.3451771798822101, 2.462398203439907]], [[-2.5713124204289493, 1.9356323962441504], 
[1.8879658089499234, 3.1212800001648091]], [[1.942043508304808, 0.80539011514164471], [-0.3765200612428643, 
0.73339801844715691]]])
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(2.24723235412)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.05147554507665264, 2.8147051298670691], [0.82461519626188773, 
-0.87020128380973727]], [[4.162249224651223, 2.7149471880082547], [1.5146161086913388, 3.700543270658593]], 
[[-2.1227703135939935, -2.1649565969349194], [-1.9960146591431203, -2.3893494370667456]], [[6.6185084149912807, 
2.7353891422443546], [-2.044726233040274, 5.1225942777562228]], [[6.9883150766937963, -1.0469164749421873], 
[5.83069369781734, 2.7820108496959088]], [[-0.022491836182242153, 3.7311359734610559], [6.598689776950259, 
4.2807158310208244]]])+(1.-msk_ref)*numpy.array([[[4.15382795571686, 6.0483859942655247], [6.4953434972230752, 
3.012966152760999]], [[3.9961013751868313, 6.7536918674872357], [0.98569783307687509, 0.73765739724913093]], 
[[3.4482526805427538, 2.3027266866259595], [6.574205438044423, 1.7031082816825673]], [[-0.37848166647976944, 
-1.1989921579657521], [3.5924095339980582, 4.7096305575557551]], [[-0.32408006631310116, 4.1828647503599985], 
[4.1351981630657715, 5.3685123542806572]], [[4.1892758624206561, 3.0526224692574928], [1.8707122928729838, 
2.980630372563005]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank3_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-3.6330041831896742, 1.9011276595647058], [4.0527837903730326, 3.7453216540822218]], 
[[1.1423057067323032, -4.6191355501663702], [-0.19479401086936399, 3.6518312558771875]], [[-0.78164127432320996, 
-0.0025588788834731702], [-2.5155059876978534, -2.7853664238124578]], [[-2.4557560474662496, -1.7001261418483038], 
[2.2437567320884249, -4.5528490181464578]], [[3.3965240991344601, 2.7531638892344281], [-1.0182649859279858, 
0.37879180372082377]], [[-2.2634040587587356, -3.6908761533687482], [-2.6652399154901509, 
-2.0159814304593739]]])+(1.-msk_arg0)*numpy.array([[[4.9981907924797788, 4.277720751221235], [-4.4785446333946686, 
-3.8140270519701982]], [[1.4517149340948965, 1.9122847710945834], [-1.0984824997077558, 4.9260526287710995]], 
[[3.0231870187238314, -4.426803554802202], [-0.1009215503507912, -2.4226611633877337]], [[3.1439947236211125, 
-2.7156096061802728], [-0.27949941006709977, 0.15562912547547469]], [[-1.6704879956646712, -0.87822202800174587], 
[-4.0968204088950708, -4.8812474874399072]], [[-3.0876637956180186, 0.42808604578959475], [-0.76617423765119153, 
1.4811418969805343]]])
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0+arg1
      s1=numpy.array([[[-3.655791939954395, 1.9082625611635287], [2.0305234873740705, -3.9575879711347337]], 
[[0.58883813376680294, -0.44253502109642717], [-0.50659655202841058, 4.7262250303753071]], [[2.3551049262619417, 
-2.7472704728416062], [-4.2131185370897501, 1.1560716927603512]], [[-1.8521430501234626, -2.8126771236453196], 
[-1.6116964851382032, 4.3144406033510982]], [[-4.4005771771028979, -3.8795508309654512], [0.95903540985898683, 
-0.84559016177598512]], [[-2.6007509769442674, -0.13151235868250399], [-1.5038936232862978, -3.9733280592961249]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-7.2887961231440688, 3.8093902207282344], [6.0833072777471031, -0.21226631705251187]], 
[[1.7311438404991062, -5.0616705712627974], [-0.70139056289777457, 8.3780562862524945]], [[1.5734636519387317, 
-2.7498293517250794], [-6.7286245247876035, -1.6292947310521066]], [[-4.3078990975897122, -4.5128032654936234], 
[0.63206024695022167, -0.23840841479535957]], [[-1.0040530779684378, -1.1263869417310231], [-0.059229576068998924, 
-0.46679835805516134]], [[-4.8641550357030034, -3.8223885120512522], [-4.1691335387764488, 
-5.9893094897554988]]])+(1.-msk_ref)*numpy.array([[[1.3423988525253838, 6.1859833123847636], [-2.4480211460205981, 
-7.7716150231049319]], [[2.0405530678616994, 1.4697497499981562], [-1.6050790517361664, 9.6522776591464066]], 
[[5.3782919449857731, -7.1740740276438082], [-4.3140400874405413, -1.2665894706273826]], [[1.29185167349765, 
-5.5282867298255924], [-1.891195895205303, 4.4700697288265729]], [[-6.0710651727675691, -4.757772858967197], 
[-3.137784999036084, -5.7268376492158923]], [[-5.688414772562286, 0.29657368710709076], [-2.2700678609374894, 
-2.4921861623155905]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank4_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[4.965007128412612, 3.4584141019026564, -1.0391619896304451, 4.5542963326499351], 
[-0.0016792172679549466, -2.9053441565334981, 0.025786108583792711, -0.89554847161554374], [4.4904084527351209, 
-0.89553646258473307, 3.8929449623498495, -2.8715607346304415]], [[-3.727374719009604, 2.2555823384608908, 
0.53380019017552272, -0.29480940480144113], [-3.6344667828862445, -4.8499559892732567, 3.5342171405331317, 
1.9875915936023327], [3.0643486049591804, -2.9482947381564806, 1.257296440825332, -4.4599817600046716]]], 
[[[-3.7989993001254971, 4.2006768317373879, -1.9340842456373886, 0.25295780568139836], [0.15305381262779072, 
2.184447614622945, -2.0595806484522039, 1.6196719151709491], [-1.550459702477788, 2.2328097059995393, 
-3.2648987061947632, -1.7698524550474004]], [[-3.1067614393264673, 3.6490340896776274, 4.2948603770463407, 
-3.4382940099694084], [-1.765073080880275, 2.5928931740693892, 2.2530590640640069, 2.7653349815108443], 
[-0.88766895991026384, 3.8444038125137965, 3.8283329993863564, 1.6961545196727537]]], [[[-1.6941819291782823, 
-4.3507603532160344, 0.58625398426930175, -4.9534370199923137], [4.3258398610183271, 4.7398172498630355, 
-0.27425006429631082, -0.80958052389792012], [0.27800145594245151, -0.70646630926925713, -1.3619199397032533, 
-0.22712536683851958]], [[-3.7307177958823781, -0.17135910311966995, -1.2454260400370809, 1.8499155339141273], 
[0.7652733563966283, -4.2318891899847593, 4.1390775019993704, 2.1086112655335079], [-4.4480501135282662, 
4.3290513315610166, -4.1098101623830443, -2.8839598970399614]]]])+(1.-msk_arg0)*numpy.array([[[[3.9323713317642746, 
4.4527426387356446, 1.8489227456459432, 2.295838413561385], [-1.5932231826477694, -0.043483214358698064, 
2.6866561252017789, -1.3064680912144833], [-4.563955043071191, -4.5294274892608124, 1.1139333008427865, 
-3.356095173880258]], [[-0.39784058429088365, 1.3572530126249651, 0.73921609667405086, -2.8036097598039502], 
[-1.6466307808609693, -3.6730522383966999, -4.2815488732075613, -3.0943250956889665], [0.84471742986867238, 
3.3304241697775492, -2.7207357502431542, -1.8257126717947059]]], [[[0.21030801293033274, 4.6379651350087698, 
4.213456762528347, 4.0550184068364885], [-2.5755175539757227, 2.6713165204428986, 3.2808072440183729, 
2.8475364996882107], [4.8503832880401561, -0.89396576884489498, 4.8726952699950328, 1.8570156992262419]], 
[[-4.6778874236692944, 2.1109769293880465, 0.79097589510131172, -2.1112073984121893], [2.558958067688426, 
2.8307096810380727, 0.012443144332241474, -3.7601222060065065], [-1.3755439053562823, 2.9800220614031678, 
1.6579582033193425, 4.4427116407434362]]], [[[-0.86660146317817688, 1.3032310329697525, 3.0027070238303377, 
-2.9114837729491319], [-3.4567748888099636, 3.3638086688271702, 4.1486162466002519, 2.0749122046757407], 
[0.84439318528796647, -3.6592289308593697, 0.77430002321168345, 1.7927967246699836]], [[-1.1981415218608116, 
2.3445312580391588, -1.5436298697897444, 1.6111465180751141], [1.6230738725320037, -1.3035089800291666, 
-4.6787506207538687, 2.9155460797717678], [3.3315156088599238, -3.5200805068877128, -1.1181004173108544, 
-2.2485916181204857]]]])
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(3.43950171094)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[8.4045088393544027, 6.8979158128444471, 2.4003397213113455, 7.9937980435917257], 
[3.4378224936738357, 0.5341575544082926, 3.4652878195255834, 2.543953239326247], [7.9299101636769116, 
2.5439652483570576, 7.3324466732916402, 0.56794097631134921]], [[-0.28787300806781335, 5.6950840494026815, 
3.9733019011173134, 3.1446923061403496], [-0.19496507194445378, -1.410454278331466, 6.9737188514749224, 
5.4270933045441234], [6.5038503159009711, 0.49120697278531011, 4.6967981517671227, -1.0204800490628809]]], 
[[[-0.35949758918370645, 7.6401785426791786, 1.5054174653044021, 3.6924595166231891], [3.5925555235695814, 
5.6239493255647357, 1.3799210624895868, 5.0591736261127398], [1.8890420084640027, 5.67231141694133, 0.1746030047470275, 
1.6696492558943903]], [[0.33274027161532338, 7.0885358006194181, 7.7343620879881314, 0.0012077009723823195], 
[1.6744286300615157, 6.0323948850111799, 5.6925607750057976, 6.204836692452635], [2.5518327510315268, 
7.2839055234555872, 7.2678347103281471, 5.1356562306145443]]], [[[1.7453197817635084, -0.91125864227424369, 
4.0257556952110924, -1.513935309050523], [7.7653415719601178, 8.1793189608048262, 3.1652516466454799, 
2.6299211870438706], [3.7175031668842422, 2.7330354016725336, 2.0775817712385374, 3.2123763441032711]], 
[[-0.29121608494058737, 3.2681426078221207, 2.1940756709047098, 5.289417244855918], [4.204775067338419, 
-0.79238747904296858, 7.5785792129411611, 5.5481129764752986], [-1.0085484025864755, 7.7685530425028073, 
-0.67030845144125362, 0.55554181390182933]]]])+(1.-msk_ref)*numpy.array([[[[7.3718730427060652, 7.8922443496774353, 
5.2884244565877339, 5.7353401245031757], [1.8462785282940213, 3.3960184965830926, 6.1261578361435696, 
2.1330336197273074], [-1.1244533321294004, -1.0899257783190217, 4.5534350117845772, 0.083406537061532671]], 
[[3.041661126650907, 4.7967547235667558, 4.1787178076158416, 0.63589195113784047], [1.7928709300808214, 
-0.23355052745490923, -0.84204716226577059, 0.34517661525282417], [4.2842191408104631, 6.7699258807193399, 
0.71876596069863652, 1.6137890391470848]]], [[[3.6498097238721234, 8.0774668459505605, 7.6529584734701377, 
7.4945201177782792], [0.86398415696606801, 6.1108182313846893, 6.7203089549601636, 6.2870382106300013], 
[8.2898849989819468, 2.5455359420968957, 8.3121969809368235, 5.2965174101680326]], [[-1.2383857127275038, 
5.5504786403298372, 4.2304776060431024, 1.3282943125296014], [5.9984597786302167, 6.2702113919798634, 
3.4519448552740322, -0.32062049506471579], [2.0639578055855083, 6.4195237723449585, 5.0974599142611332, 
7.8822133516852269]]], [[[2.5729002477636138, 4.7427327439115432, 6.4422087347721284, 0.52801793799265884], 
[-0.017273177868172951, 6.8033103797689609, 7.5881179575420425, 5.5144139156175314], [4.2838948962297572, 
-0.21972721991757904, 4.2138017341534741, 5.2322984356117743]], [[2.2413601890809791, 5.7840329689809495, 
1.8958718411520463, 5.0506482290169048], [5.0625755834737944, 2.1359927309126241, -1.239248909812078, 
6.3550477907135585], [6.7710173198017145, -0.080578795945922099, 2.3214012936309363, 1.190910092821305]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_expandedData_rank4_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[3.2510674404409041, 2.1171696862303406, 2.9610258759664267, -3.8373977579450456], 
[0.75383244276133166, 2.4077943881602728, 3.873284406870285, 3.7937584009819574], [-4.6069898901399364, 
-2.5452970249895754, 3.650830786457707, -0.56630176651201847]], [[3.6738989513815135, -1.1553536380556686, 
4.303352195803182, 2.0201689947921695], [2.5110280594242029, 1.1178178456135743, 3.5722095880572251, 
-3.0495901167648221], [-1.8161969765914288, -3.850369287459924, 1.8305771607495833, 3.8129356009276751]]], 
[[[4.8159492177547296, -2.7259760165966638, -0.056119891503465524, 3.2320437499651025], [4.1412540490540568, 
2.3145635424798332, 4.2298625240821792, -4.9326174629443722], [1.2505234798682396, 4.1728981653768358, 
-1.4526511101284445, -0.73865645812869563]], [[-2.5027203270038956, -0.75821705726011146, -2.0074201432570495, 
-0.20166798891695503], [1.7962444938241209, 4.9186635916785164, -3.3612255674731486, -3.1402103698143327], 
[4.8100127068213077, -3.7003932729639377, -2.3809463861562454, 2.6337296431542621]]], [[[0.8461884816413443, 
2.2850095300693116, 3.1039351776827235, 2.7358221987272575], [-1.331100327658973, -2.4718869003284438, 
3.8392116060077814, 3.7886003252177218], [-2.740692362699221, -1.1104811343803189, 1.065443269317063, 
-1.604926521206449]], [[3.1359320207935291, 2.4159415877072101, -2.9781841648177654, 0.4457695581762291], 
[1.4022534028069558, 3.2181877465159641, 4.1561033889739196, -4.5314636502141923], [2.4896032954770373, 
-1.6749755107952033, -4.2977752660345292, 4.3862296692093636]]]])+(1.-msk_arg0)*numpy.array([[[[3.8098232095134126, 
-2.0180524002497693, 4.420784171182504, -2.4324750966542674], [2.4681882567616125, 3.0279649104786941, 
2.2383665512055266, -0.091420157761364251], [4.7846856391630048, 0.45001495814867454, 2.8428137570111911, 
3.6542996408716562]], [[-3.3832925941075711, -4.6684050424331947, 2.7145812310865534, 0.57489640415196952], 
[3.2363298539062395, -0.28076205609599914, -2.1610563710523598, -3.9600308036480381], [4.1445091213012599, 
0.23464603550937735, -4.9214532841127738, 3.7601288072640866]]], [[[4.5878923885513938, -2.7602444517968006, 
-2.4823493575559641, -1.1998619544811917], [-1.0165322624110429, 4.8743114304602564, 3.0069704689379755, 
2.0086372739622043], [-1.7482883016273565, 4.5233781656491008, 1.0481669308330579, 3.3780108680134457]], 
[[-4.5351514069636076, -4.760484108729206, -1.7334568308716203, -4.3080131499917833], [4.0321976091043883, 
-2.6576000312675063, 1.3372423488299923, -3.8949616711167625], [3.5793384711817051, 2.60693067621275, 
1.8056256765125287, -3.9915454170699869]]], [[[0.39851532295995273, 2.2465287291059273, 0.64170560779626662, 
-4.7331314705888738], [3.5329039709028898, -2.5311269573107662, 2.8367974744858193, -4.3457969220676684], 
[-1.526677955424999, -2.5983211468943357, -1.3293797580217093, -3.1887378668078279]], [[3.1416335105809505, 
0.35146012646543134, 2.428390004415637, 2.7813900205500861], [3.5228217461650111, -0.012304332300811183, 
-3.1395042313107369, 4.8647351561551702], [2.2570133784920099, -1.7535240218446777, 0.38792070998653028, 
-0.21839923153693785]]]])
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0+arg1
      s1=numpy.array([[[[-0.55399336747432937, -3.6468486902030306, 2.4533567494215669, 4.8267547347789659], 
[1.1480960590338416, 3.5599245920968787, -2.8247534868419724, -2.2031349101131505], [1.7520095897646017, 
4.4293583295521266, -3.2046920932014888, -3.8760923163847472]], [[3.9288042477427645, 1.103593535294765, 
0.62546922225950485, 2.5431633219905123], [2.5483588394973191, -0.82358610517599207, -0.47010674146441023, 
2.7635563586840011], [3.5616440522317419, 2.2995934729430481, -3.501591556463012, 1.3778428754586027]]], 
[[[-4.3918539920661051, 0.24976043236636869, -2.4847081470778463, 4.8636790550226792], [-4.2172400078729559, 
-2.0316184192507647, -0.53464794178739794, -0.035422588600630966], [1.7049703562375615, 4.2019750499164399, 
-3.7430217705554858, -3.4952387702082346]], [[-0.39925876875124189, 1.4505137462439404, -4.1941814051173072, 
-1.844757872605356], [-3.4448187389632414, -3.5340944666273377, -3.178247383159305, -1.7824872241435519], 
[-3.6843631882800798, -4.1186208792142187, 2.0636953370355959, -0.18717114434561122]]], [[[-2.4316812831173742, 
0.39582208925882689, 1.4893695917228467, -3.1232026180567773], [2.1122901499636226, 4.9884613457151978, 
-4.7793541216702149, -3.9541373136233391], [-4.8256481088328194, -0.10764491664526066, 2.9970513787255895, 
-1.0443943611478437]], [[3.6491162738908258, 3.4225261399204765, -2.9600723325757849, 3.3422667802452324], 
[-3.763493116056098, 4.6894908619506595, 2.532040050484988, 0.99028387045053101], [2.5962274887920085, 
-0.2721955960411897, -4.7946284910477441, -0.96141278632713245]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[2.6970740729665748, -1.52967900397269, 5.4143826253879936, 0.98935697683392032], 
[1.9019285017951733, 5.9677189802571515, 1.0485309200283126, 1.5906234908688068], [-2.8549803003753347, 
1.8840613045625512, 0.44613869325621813, -4.4423940828967652]], [[7.6027031991242779, -0.051760102760903592, 
4.9288214180626868, 4.5633323167826818], [5.059386898921522, 0.29423174043758227, 3.1021028465928149, 
-0.28603375808082099], [1.7454470756403131, -1.550775814516876, -1.6710143957134287, 5.1907784763862779]]], 
[[[0.42409522568862457, -2.4762155842302951, -2.5408280385813118, 8.0957228049877816], [-0.075985958818899135, 
0.28294512322906851, 3.6952145822947813, -4.9680400515450032], [2.9554938361058012, 8.3748732152932757, 
-5.1956728806839303, -4.2338952283369302]], [[-2.9019790957551375, 0.6922966889838289, -6.2016015483743567, 
-2.046425861522311], [-1.6485742451391205, 1.3845691250511787, -6.5394729506324536, -4.922697593957885], 
[1.1256495185412279, -7.8190141521781564, -0.3172510491206495, 2.4465584988086508]]], [[[-1.5854928014760299, 
2.6808316193281385, 4.5933047694055702, -0.38738041932951983], [0.78118982230464962, 2.516574445386754, 
-0.94014251566243345, -0.16553698840561726], [-7.5663404715320404, -1.2181260510255796, 4.0624946480426525, 
-2.6493208823542926]], [[6.7850482946843549, 5.8384677276276866, -5.9382564973935503, 3.7880363384214615], 
[-2.3612397132491423, 7.9076786084666235, 6.6881434394589077, -3.5411797797636613], [5.0858307842690458, 
-1.9471711068363931, -9.0924037570822733, 3.4248168828822312]]]])+(1.-msk_ref)*numpy.array([[[[3.2558298420390832, 
-5.6649010904527994, 6.8741409206040709, 2.3942796381246985], [3.6162843157954541, 6.5878895025755728, 
-0.58638693563644573, -2.2945550678745148], [6.5366952289276066, 4.8793732877008011, -0.36187833619029774, 
-0.22179267551309101]], [[0.54551165363519338, -3.5648115071384296, 3.3400504533460582, 3.1180597261424818], 
[5.7846886934035586, -1.1043481612719912, -2.63116311251677, -1.196474444964037], [7.7061531735330018, 
2.5342395084524254, -8.4230448405757858, 5.1379716827226893]]], [[[0.19603839648528876, -2.5104840194304319, 
-4.9670575046338108, 3.6638171005414875], [-5.2337722702839988, 2.8426930112094917, 2.4723225271505775, 
1.9732146853615733], [-0.043317945389794943, 8.7253532155655407, -2.6948548397224279, -0.11722790219478885]], 
[[-4.9344101757148495, -3.3099703624852657, -5.9276382359889279, -6.1527710225971397], [0.58737887014114687, 
-6.1916944978948436, -1.8410050343293127, -5.6774488952603139], [-0.10502471709837469, -1.5116902030014687, 
3.8693210135481246, -4.1787165614155981]]], [[[-2.0331659601574215, 2.6423508183647542, 2.1310751995191133, 
-7.8563340886456512], [5.6451941208665124, 2.4573343884044316, -1.9425566471843956, -8.2999342356910084], 
[-6.3523260642578183, -2.7059660635395963, 1.6676716207038802, -4.2331322279556716]], [[6.7907497844717764, 
3.7739862663859078, -0.53168232816014793, 6.1236568007953185], [-0.24067136989108695, 4.6771865296498483, 
-0.60746418082574882, 5.8550190266057012], [4.8532408672840184, -2.0257196178858674, -4.4067077810612139, 
-1.1798120178640703]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_Symbol_rank0(self):
      arg0=Data(1.30830371112,self.functionspace)
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(0.0412291309402)
      sub=res.substitute({arg1:s1})
      ref=Data(1.26707458018,self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_Symbol_rank1(self):
      arg0=Data(-4.2604726935,self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0-arg1
      s1=numpy.array([-3.8546037299533653, -1.305392606117024])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-0.4058689635493371, -2.9550800873856784]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_Symbol_rank2(self):
      arg0=Data(0.902009664206,self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0-arg1
      s1=numpy.array([[-3.117681444740418, -3.2512793024980069, -3.7762244881344218, -0.50644943812549315, 
3.066726444630655], [-2.6348956508380805, -0.90372740616696667, 0.5252271533586752, 2.0132741900533446, 
2.0837322808099037], [0.088376617597372586, 0.67864487020517306, 3.7057383001711681, 1.0445042366908988, 
-2.1093161712985955], [4.328915747720707, -0.73501622742024342, -0.088412628376807412, -3.0414953794209754, 
1.610361274316344]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[4.0196911089468177, 4.1532889667044071, 4.6782341523408215, 1.4084591023318929, 
-2.1647167804242553], [3.5369053150444802, 1.8057370703733664, 0.37678251084772452, -1.1112645258469449, 
-1.181722616603504], [0.81363304660902713, 0.22336479400122666, -2.8037286359647684, -0.14249457248449904, 
3.0113258355049952], [-3.4269060835143073, 1.6370258916266431, 0.99042229258320713, 3.9435050436273751, 
-0.7083516101099443]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_Symbol_rank3(self):
      arg0=Data(4.30012329043,self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0-arg1
      s1=numpy.array([[[-2.4328051948060772, 1.3096803933228829], [-1.9201038070201615, 2.2529209930562519]], 
[[4.4911763191005498, -0.0070408039855616167], [-4.5070979412665588, 0.23394826644475319]], [[-2.0679275681214171, 
4.7260141882743518], [-1.9530690972223672, 4.2165911161948344]], [[4.2340594486013217, 0.31531838157863668], 
[1.2102543060708451, 4.5768051588147358]], [[4.9016533619135778, 1.0237157761801843], [-1.6198381225390657, 
1.509534129406096]], [[-2.8351524725878399, -0.8712771035569391], [-1.2500793307427105, 0.52784760832550681]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[6.732928485237343, 2.990442897108383], [6.2202270974514278, 2.0472022973750139]], 
[[-0.19105302866928398, 4.3071640944168275], [8.8072212316978238, 4.0661750239865126]], [[6.3680508585526834, 
-0.42589089784308598], [6.2531923876536331, 0.083532174236431445]], [[0.066063841829944181, 3.9848049088526292], 
[3.0898689843604208, -0.27668186838346998]], [[-0.60153007148231197, 3.2764075142510816], [5.9199614129703315, 
2.7905891610251699]], [[7.1352757630191057, 5.1714003939882049], [5.5502026211739768, 
3.772275682105759]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank0_Symbol_rank4(self):
      arg0=Data(-3.5839426267,self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0-arg1
      s1=numpy.array([[[[-2.9729696451374421, 2.7845056200381855, 0.070436437102223692, 0.66836223796868044], 
[0.40381761203578836, -1.7869220467261826, -4.3681167712065552, 1.0762008553734699], [-3.4293067325266744, 
-3.8959384230092855, -4.2869773308861872, -3.5982581222849266]], [[3.8085384127848325, -4.9902013750126919, 
1.7025140755302903, -1.8585391591273237], [-1.8948326373524536, 2.0874520505745666, -1.8647114753321095, 
3.9665649921657007], [-2.6617432109425376, -0.043781338271665859, -4.3924469058705498, -4.6038566089651081]]], 
[[[4.1612414942039617, -0.24691459950937489, 1.8801077349311939, -4.0607604598486082], [-0.48975931816079132, 
4.776651055544292, 2.5892649853139229, 2.6300466396994988], [-0.6331493645323949, -4.8747858313906498, 
2.5714462579440713, -0.12625615907892662]], [[1.8766405716198298, 0.97931619405259518, -1.2333119307639082, 
3.632140408148242], [0.96979041799351151, -4.0819837173164526, 3.4625138677193164, -1.7431511130821575], 
[-2.7530992377422381, -3.1495479306859906, 1.3466227111831488, -2.3016323722421128]]], [[[-2.8378224290103491, 
-0.7230057223129247, 0.95865498114414649, 0.14297561114879365], [2.3319242484901492, 4.9972541799736234, 
-1.7121650896762564, 1.6097551517446558], [2.7133813837524077, -3.1913323682416994, -0.39896207531318861, 
-3.2753783571190107]], [[1.3158800827274399, -0.034075573686918936, 3.2707189112070392, -2.9118211235462041], 
[4.362994678434946, -3.2771781302292515, 3.4919565479064456, 1.6061522420425254], [-1.8973785117347788, 
-4.4461539342202174, -3.8132486661529263, -0.74231592463494511]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-0.61097298156602342, -6.368448246741651, -3.6543790638056892, -4.252304864672146], 
[-3.9877602387392539, -1.7970205799772829, 0.78417414450308964, -4.6601434820769354], [-0.15463589417679113, 
0.31199579630581997, 0.70303470418272163, 0.014315495581461057]], [[-7.392481039488298, 1.4062587483092264, 
-5.2864567022337559, -1.7254034675761418], [-1.689109989351012, -5.6713946772780321, -1.7192311513713561, 
-7.5505076188691662], [-0.9221994157609279, -3.5401612884317997, 0.80850427916708423, 1.0199139822616425]]], 
[[[-7.7451841209074272, -3.3370280271940906, -5.4640503616346594, 0.4768178331451427], [-3.0941833085426742, 
-8.3605936822477567, -6.1732076120173884, -6.2139892664029643], [-2.9507932621710706, 1.2908432046871843, 
-6.1553888846475369, -3.4576864676245389]], [[-5.4605831983232953, -4.5632588207560607, -2.3506306959395573, 
-7.2160830348517075], [-4.553733044696977, 0.49804109061298707, -7.0464564944227819, -1.840791513621308], 
[-0.83084338896122745, -0.43439469601747493, -4.9305653378866143, -1.2823102544613527]]], [[[-0.74612019769311644, 
-2.8609369043905408, -4.542597607847612, -3.7269182378522592], [-5.9158668751936148, -8.5811968066770881, 
-1.8717775370272092, -5.1936977784481213], [-6.2973240104558732, -0.39261025846176612, -3.1849805513902769, 
-0.30856426958445482]], [[-4.8998227094309055, -3.5498670530165466, -6.8546615379105047, -0.67212150315726138], 
[-7.9469373051384116, -0.306764496474214, -7.0758991746099111, -5.1900948687459909], [-1.6865641149686867, 
0.8622113075167519, 0.22930603944946082, -2.8416267020685204]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([2.6649927252905226, 0.29496968217893382]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(1.03366663195)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([1.6313260933372291, -0.73869694977435962]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([3.9090880537794526, -3.9706193840215942]),self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0-arg1
      s1=numpy.array([-3.7233870114697742, 0.99043840493200186])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([7.6324750652492268, -4.9610577889535961]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[2.8033126273843685, 0.51509190965393792, 3.931306976936968, -3.3823534090429486, 
-2.3486719525293087], [-2.9837425664154784, -2.4457160287299686, 3.8981965382683743, -0.89609359902144714, 
4.1620406111464288], [3.6868893591462246, -2.9993029597001462, 1.8283120616948665, -2.0195573949932277, 
-2.1640627499057361], [-2.9723279323425489, -4.8559061533246624, -1.0130455282709172, -3.7833351321644395, 
3.514692525422209]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(4.86937457463)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-2.0660619472497519, -4.3542826649801825, -0.93806759769715242, -8.2517279836770694, 
-7.2180465271634286], [-7.8531171410495988, -7.315090603364089, -0.97117803636574607, -5.7654681736555675, 
-0.70733396348769162], [-1.1824852154878958, -7.8686775343342665, -3.0410625129392539, -6.8889319696273486, 
-7.0334373245398565], [-7.8417025069766693, -9.7252807279587827, -5.8824201029050371, -8.6527097067985608, 
-1.3546820492119114]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[-1.1140360715186182, -1.5235600156934481, 4.3075103934286023, 4.6800377743432158, 
-3.2505150436972521], [0.39123458636258768, 0.41088806870879768, -2.9614108446790501, 1.1049238977643405, 
0.92166667279843395], [0.54565864417397059, -4.8476249672143004, 4.9444652981547943, 4.0252126389168215, 
-3.9123423425216322], [-3.6777596228844844, -3.4408972758983558, 2.7718180074050611, -0.3997152204895924, 
-0.16573647825956073]]),self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0-arg1
      s1=numpy.array([[-2.4209487163246299, 1.3152643083131128, -0.71046464711788015, 0.21557543046364458, 
-2.202065459251934], [-3.9101544501984198, -2.8682151089642827, 2.7125251197023488, 1.4173123031722534, 
2.7246295240806209], [-1.5744991442525436, 3.0598215212654001, 0.63494427405471487, -4.906149376046594, 
-1.6839564426436748], [4.0729555430880922, -0.83371622418680769, 0.46337987461630981, 4.0014755703742395, 
-2.1103899940006032]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[1.3069126448060118, -2.8388243240065609, 5.0179750405464825, 4.4644623438795712, 
-1.0484495844453181], [4.301389036561007, 3.2791031776730803, -5.6739359643813989, -0.31238840540791291, 
-1.8029628512821869], [2.1201577884265141, -7.9074464884797004, 4.3095210241000794, 8.9313620149634154, 
-2.2283858998779573], [-7.7507151659725766, -2.6071810517115481, 2.3084381327887513, -4.4011907908638319, 
1.9446535157410425]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[-2.6064326776506652, 4.9989076052590633], [-3.0068821433777249, -3.1193113732509516]], 
[[-1.3190483681618739, 3.9479827067009108], [1.0954417889014865, 4.6359051697534426]], [[-2.9778493741722056, 
3.4845430816156977], [1.7569072943914552, 1.1616150547614428]], [[-0.91210869485198565, -1.3406976214361355], 
[3.2217649968914159, -2.662260898242006]], [[4.1697693146337542, -1.1741423631833072], [-4.9803850608859115, 
1.2700647554700222]], [[4.6074170359664368, 1.453706456526124], [0.20949339688511692, 
3.0091215511346796]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(-1.04145599079)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-1.5649766868561219, 6.0403635960536066], [-1.9654261525831815, -2.0778553824564083]], 
[[-0.27759237736733056, 4.9894386974954541], [2.1368977796960298, 5.6773611605479859]], [[-1.9363933833776623, 
4.525999072410241], [2.7983632851859985, 2.2030710455559861]], [[0.12934729594255767, -0.29924163064159215], 
[4.2632209876859593, -1.6208049074474626]], [[5.2112253054282975, -0.13268637238876391], [-3.9389290700913682, 
2.3115207462645655]], [[5.6488730267609801, 2.4951624473206673], [1.2509493876796602, 
4.0505775419292229]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[2.0075159970537113, 4.417162011434554], [0.71949384400506577, 1.0783048900035652]], 
[[4.7614254606302335, -2.0888542276996978], [-3.5997702799671547, 4.2825487871951644]], [[-0.39389734575197544, 
1.3283252585178928], [3.6919455158435834, -0.76277259642421402]], [[-4.4972180700076887, -3.7983795355307128], 
[-0.26779668046970784, -0.79380221724008582]], [[-2.0572521505738273, -1.5154686544559368], [4.0972713376059851, 
4.5986089620495108]], [[-1.3971821196462377, 0.16028646761807508], [-0.63755809097850857, 
-3.3787710682197272]]]),self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0-arg1
      s1=numpy.array([[[3.5103565349856751, 0.91526758558677379], [-3.7224124618951135, -0.27931399630195397]], 
[[1.5813622936549105, 3.6172915696233972], [-1.2364412564258132, 0.16417768270487709]], [[0.64050559170122234, 
4.6361361331624593], [-0.47839680540824325, -2.1615310941440589]], [[-0.85667930966756511, 1.669882578368358], 
[0.22343162562157293, 0.80905790542025358]], [[-3.5873387244847543, 3.1163266795230058], [3.5553732672252671, 
-4.6758779472194405]], [[3.6742958529176484, 0.58762359541383802], [1.5778519953325496, -0.39731537378910975]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-1.5028405379319638, 3.5018944258477802], [4.4419063059001793, 1.3576188863055192]], 
[[3.180063166975323, -5.7061457973230949], [-2.3633290235413416, 4.1183711044902873]], [[-1.0344029374531978, 
-3.3078108746445665], [4.1703423212518267, 1.3987584977198448]], [[-3.6405387603401236, -5.4682621138990708], 
[-0.49122830609128076, -1.6028601226603394]], [[1.5300865739109271, -4.6317953339789426], [0.54189807038071791, 
9.2744869092689513]], [[-5.0714779725638861, -0.42733712779576294], [-2.2154100863110582, 
-2.9814556944306174]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[0.66483074145605592, 2.9129070748039982, -1.8655842911981346, -1.098354904466996], 
[1.7426470733136448, -2.4896761957460898, 4.3864323453867851, -4.0781460331955177], [-0.62183708580819008, 
-2.6186592235582786, -1.8750164189422014, -3.9631241880095969]], [[4.0419620323350909, 0.15536839603964836, 
1.9771157591398101, -2.6101097405194453], [-4.7364297803535704, 1.8318126417179714, 3.2354822684907454, 
2.2507758179659376], [-4.8699934080808029, -0.35744120243411981, 4.0908957400805122, -3.8440017446794084]]], 
[[[4.5466344627836612, -2.8174576749848423, -0.32339288977492142, -3.3368918944053516], [3.3311423168153738, 
-1.2448667289851647, -0.66737673743075376, -3.9953617725851598], [-4.8878412407428931, 3.1347720870691358, 
-2.4390985397355847, -3.5615840737730475]], [[-3.7978882365989697, 4.345238312451805, 2.8310129832366435, 
2.8564779239624674], [-0.85025481289091864, -4.3757742754757345, 3.5451710843902031, -2.5068001174158816], 
[2.6943798866386315, 2.2746017608025317, -4.2655778273063607, 0.97165631163417387]]], [[[-2.9330039029788955, 
4.3910413333213238, 2.5513441899802833, -3.8678703253194402], [-2.6748516851594308, -3.8887038302549062, 
1.2485088138696518, -3.9629424578182251], [-0.38166273681210328, 3.82781593241344, -4.1817331752844087, 
4.682478964767725]], [[-0.85849290617372809, -0.49338756563096275, -1.0480256440941615, -0.51008618582467946], 
[-0.26820315453886501, 4.8354933917592806, 2.9555158912003154, -2.4766421456452479], [2.5098219987182944, 
3.6215601735655589, -4.4497307132070123, -3.9295385075107028]]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(-2.59361652138)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[3.2584472628375467, 5.506523596185489, 0.72803223018335617, 1.4952616169144948], 
[4.3362635946951356, 0.10394032563540101, 6.9800488667682759, -1.4845295118140269], [1.9717794355733007, 
-0.025042702176787834, 0.7186001024392894, -1.3695076666281061]], [[6.6355785537165817, 2.7489849174211392, 
4.5707322805213009, -0.01649321913795454], [-2.1428132589720796, 4.4254291630994622, 5.8290987898722362, 
4.8443923393474284], [-2.2763768866993122, 2.236175318947371, 6.6845122614620029, -1.2503852232979176]]], 
[[[7.140250984165152, -0.22384115360335155, 2.2702236316065694, -0.74327537302386082], [5.9247588381968646, 
1.3487497923963261, 1.926239783950737, -1.401745251203669], [-2.2942247193614023, 5.7283886084506266, 
0.15451798164590613, -0.96796755239155674]], [[-1.2042717152174789, 6.9388548338332958, 5.4246295046181343, 
5.4500944453439581], [1.7433617084905721, -1.7821577540942437, 6.1387876057716939, 0.08681640396560919], 
[5.2879964080201223, 4.8682182821840225, -1.6719613059248699, 3.5652728330156647]]], [[[-0.33938738159740467, 
6.9846578547028146, 5.1449607113617741, -1.2742538039379494], [-0.081235163777940045, -1.2950873088734154, 
3.8421253352511426, -1.3693259364367343], [2.2119537845693875, 6.4214324537949308, -1.5881166539029179, 
7.2760954861492158]], [[1.7351236152077627, 2.100228955750528, 1.5455908772873292, 2.0835303355568113], 
[2.3254133668426258, 7.4291099131407714, 5.5491324125818062, 0.11697437573624292], [5.1034385200997852, 
6.2151766949470497, -1.8561141918255215, -1.335921986129212]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_constData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[2.140332416756844, -4.5756565160935745, 1.0268217328307561, 1.594533973931731], 
[4.1426026647673879, 0.1548614651600202, 3.351820863446946, 0.54777524679756073], [-4.6470169243406527, 
-3.4101935702258368, 1.3604597013400213, -4.3236653508957374]], [[2.3543066928954612, 1.6355558219698443, 
3.8590758340122093, 0.055467084597328409], [1.3949738751098479, -2.9042097100731445, 2.1331143130237962, 
-0.45715627400394165], [3.9505052117900146, -4.8644226435153097, 0.13641466419900183, 0.92434447564323374]]], 
[[[-4.2036478385109302, -2.2096856472681958, -3.309442061812593, -0.17761420723311439], [-4.5417481392819026, 
3.354117107537796, 2.9925164896060084, 4.231145636082223], [-4.3165407391400308, -0.16204594013147311, 
-1.5308101185053733, 3.7017204822457384]], [[2.4648028362561725, 0.43817614121240833, -4.4908194091317366, 
-0.081928750874263656], [-3.4087689978816016, 4.259133980931324, -4.2850896710829334, 4.6395735766216326], 
[-1.3584480043808989, -4.7738821023855085, -1.2617431337636842, -1.2598313032270116]]], [[[2.2708892792624855, 
1.9132737394453327, -0.50215367058696003, 0.19108419265161469], [-2.0796597802531669, 1.1505151966811367, 
1.2957662425378791, -1.5883201097665802], [-1.7035021892623838, 4.8639671345493021, 3.1243484697100534, 
0.47610495992410051]], [[-4.0444287366693015, -1.3614006776767349, -0.18268931922481002, 4.8063591217845332], 
[3.1407426206783704, 2.8940879164962441, -4.9664997014592807, 1.6951588068340158], [-3.895479459710558, 
1.7220903215355694, -3.7165673657855267, 3.1903385713544257]]]]),self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0-arg1
      s1=numpy.array([[[[-4.3482304868754991, -1.2480666735558845, 0.43538858115159051, -2.0858236027245205], 
[-2.442305699452354, 2.0213192586154003, -2.5262404161243679, -4.458062700052194], [0.26228138879138641, 
-2.6430658161459242, -4.7246503759525602, 4.2538788761081854]], [[-1.6124403577544308, -1.8284497197976037, 
-3.0160374139385002, 2.7523938918136759], [1.4437250527651582, -2.7814473787336489, 3.5116683735594361, 
-3.9808640616716562], [1.7054962689298705, 4.7974185413341068, 1.9447068850818283, -1.2797130952071156]]], 
[[[3.7642823106611107, 0.11145650212965919, -0.096799862214571597, 2.0215787533002523], [0.26390717935294816, 
0.12612295721321498, 4.0275730341758482, -1.2268861937462172], [-2.947926663434548, -1.4514539315574626, 
2.4550945474164232, -2.7897655841602651]], [[-1.5947829088079746, 0.80620330852535815, -4.5614285986030234, 
-1.9102368071164841], [2.0807019362652692, -4.099640999530064, -1.8395330667711352, -4.6367501410986929], 
[-2.5162327168837786, 4.6954385782651951, -2.1576821461704854, -1.62194811763983]]], [[[0.06729391952569852, 
-0.57919376543293488, -3.1838952254737416, 1.7056529660452817], [3.6116233555564143, 0.81964000588296315, 
-0.16440769780998377, 0.079355513141521783], [2.9805073823987431, 1.3188532056435962, 3.4153481616516537, 
-2.5138710663982189]], [[2.8884594089569315, 1.1351683507610142, -0.68804270946144719, -4.7325886514124882], 
[1.1204800401276476, 0.55566378590737031, 0.94240513232859335, 2.9610440134171334], [-2.6222587774463815, 
-4.4048348584786705, -0.29650368246657699, -1.0078523107846902]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[6.4885629036323431, -3.32758984253769, 0.59143315167916555, 3.6803575766562515], 
[6.5849083642197419, -1.8664577934553801, 5.8780612795713143, 5.0058379468497547], [-4.9092983131320391, 
-0.76712775407991263, 6.0851100772925815, -8.5775442270039228]], [[3.9667470506498921, 3.464005541767448, 
6.8751132479507095, -2.6969268072163475], [-0.048751177655310229, -0.12276233133949566, -1.3785540605356399, 
3.5237077876677145], [2.2450089428601441, -9.6618411848494166, -1.8082922208828265, 2.2040575708503494]]], 
[[[-7.9679301491720409, -2.321142149397855, -3.2126421995980214, -2.1991929605333667], [-4.8056553186348507, 
3.227994150324581, -1.0350565445698399, 5.4580318298284407], [-1.3686140757054828, 1.2894079914259895, 
-3.9859046659217965, 6.4914860664060035]], [[4.0595857450641475, -0.36802716731294982, 0.070609189471286804, 
1.8283080562422205], [-5.4894709341468708, 8.3587749804613871, -2.4455566043117982, 9.2763237177203255], 
[1.1577847125028797, -9.4693206806507035, 0.89593901240680118, 0.3621168144128184]]], [[[2.203595359736787, 
2.4924675048782676, 2.6817415548867816, -1.514568773393667], [-5.6912831358095808, 0.33087519079817351, 
1.4601739403478629, -1.667675622908102], [-4.684009571661127, 3.5451139289057059, -0.29099969194160025, 
2.9899760263223194]], [[-6.932888145626233, -2.4965690284377491, 0.50535339023663717, 9.5389477731970214], 
[2.0202625805507228, 2.3384241305888738, -5.908904833787874, -1.2658852065831177], [-1.2732206822641765, 
6.1269251800142399, -3.4200636833189497, 4.1981908821391158]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank0_Symbol_rank0(self):
      arg0=Data(-2.29417952191,self.functionspace)
      arg0.setTaggedValue(1,-4.27612309963)
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(-2.86386679086)
      sub=res.substitute({arg1:s1})
      ref=Data(0.569687268944,self.functionspace)
      ref.setTaggedValue(1,-1.41225630877)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank0_Symbol_rank1(self):
      arg0=Data(-4.72691427991,self.functionspace)
      arg0.setTaggedValue(1,0.483106242273)
      arg1=Symbol(shape=(2,))
      res=arg0-arg1
      s1=numpy.array([-0.58516003749737244, 2.93231182282255])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-4.1417542424175267, -7.6592261027374491]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([1.0682662797700972, -2.4492055805498252]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank0_Symbol_rank2(self):
      arg0=Data(4.84060376911,self.functionspace)
      arg0.setTaggedValue(1,-3.32867505476)
      arg1=Symbol(shape=(4, 5))
      res=arg0-arg1
      s1=numpy.array([[3.5332516865172998, 4.2256878903288939, -4.6404295927681405, 4.9721874322243114, 
-1.5545932240349902], [0.40603544670242542, -2.879718425724147, -2.1385047584627337, 4.6127992237598132, 
0.57646645021785048], [-2.6334801212800754, -2.3655947826469701, 0.48086858542515643, 1.0360291664664301, 
-3.4378490059536082], [-0.23853194944872236, -2.0363663305583768, -2.3289186751171798, 3.5102407359843486, 
4.1303419895739388]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[1.3073520825884426, 0.6149158787768485, 9.4810333618738838, -0.13158366311856895, 
6.3951969931407326], [4.434568322403317, 7.7203221948298895, 6.9791085275684761, 0.2278045453459292, 
4.2641373188878919], [7.4740838903858178, 7.2061985517527125, 4.359735183680586, 3.8045746026393124, 
8.2784527750593497], [5.0791357185544648, 6.8769700996641188, 7.1695224442229222, 1.3303630331213938, 
0.71026177953180358]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-6.8619267412736988, -7.5543629450852929, 1.3117545380117415, 
-8.3008624869807104, -1.7740818307214088], [-3.7347105014588244, -0.44895662903225197, -1.1901702962936653, 
-7.9414742785162122, -3.9051415049742495], [-0.69519493347632366, -0.96308027210942893, -3.8095436401815554, 
-4.3647042212228291, 0.10917395119720918], [-3.0901431053076767, -1.2923087241980222, -0.99975637963921926, 
-6.8389157907407476, -7.4590170443303379]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank0_Symbol_rank3(self):
      arg0=Data(-3.20552188916,self.functionspace)
      arg0.setTaggedValue(1,-0.473083670166)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0-arg1
      s1=numpy.array([[[0.71230320805011704, -3.008236723891188], [0.81066003773158002, -3.6043239509733382]], 
[[3.691034498943317, -3.3919882986743777], [0.84551364067512935, 3.3207859438709946]], [[0.41963337446652105, 
-3.6038224020133991], [-2.3537235378574151, -3.7120927558232997]], [[-3.4588851001838727, -0.31880183563871789], 
[-1.3379489058063267, -3.9118810181560226]], [[4.4984539881701195, -3.2158956295350851], [1.5013508852420685, 
2.8717656529358955]], [[-0.13701019263353231, -3.1176264463626078], [-1.67955120335195, 4.317481449568719]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-3.917825097207726, -0.19728516526642093], [-4.016181926889189, 0.3988020618157293]], 
[[-6.896556388100926, 0.18646640951676874], [-4.0510355298327383, -6.5263078330286035]], [[-3.62515526362413, 
0.39830051285579016], [-0.85179835130019388, 0.50657086666569073]], [[0.2533632110262638, -2.886720053518891], 
[-1.8675729833512822, 0.70635912899841369]], [[-7.7039758773277285, 0.010373740377476182], [-4.7068727743996774, 
-6.0772875420935044]], [[-3.0685116965240766, -0.087895442795001166], [-1.525970685805659, 
-7.523003338726328]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-1.1853868782160886, 2.5351530537252165], [-1.2837437078975515, 
3.1312402808073667]], [[-4.1641181691092886, 2.9189046285084062], [-1.3185973108411009, -3.7938696140369661]], 
[[-0.89271704463249257, 3.1307387318474276], [1.8806398676914435, 3.2390090856573281]], [[2.9858014300179012, 
-0.15428183452725364], [0.86486523564035522, 3.4387973479900511]], [[-4.9715376583360911, 2.7428119593691136], 
[-1.97443455540804, -3.344849323101867]], [[-0.33607347753243921, 2.6445427761966362], [1.2064675331859784, 
-4.7905651197346906]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank0_Symbol_rank4(self):
      arg0=Data(-0.215341183726,self.functionspace)
      arg0.setTaggedValue(1,-3.01917111711)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0-arg1
      s1=numpy.array([[[[3.1718058337950783, -4.3218518167555349, 4.7360170033398816, 2.6415781893387447], 
[1.7953624357215787, 0.37239845986582054, 0.85595953231170441, -4.2093909477304852], [-4.0724848735753412, 
-2.3789549933876364, 3.8266481046469991, -4.4686983670793881]], [[-1.3807814097985793, -0.9345570079736385, 
3.2111606830229267, 2.5248569160832579], [-0.19847478717542089, 3.6200277417416071, -1.3367301493578787, 
-1.9914051287776093], [4.2384277387383236, -3.1625190831895669, -4.8267032630177118, -3.7590986361039294]]], 
[[[-0.96721285038350846, 0.23717549644533698, -2.0558971771798862, -2.1889488119398925], [2.1163450477817447, 
-4.308535473047935, 0.96468545582662735, 0.58036767508710252], [-0.26889479983427034, -4.6749066439752021, 
-2.6908936581627731, 3.3090528029139286]], [[1.0683391958055246, -4.3705975019062535, 4.6959723711804546, 
-0.58815635047014858], [-1.7921642772643898, 2.8079866307247423, 4.5837878995413348, -3.6656523242301429], 
[2.1083853748587442, -0.44280454111162726, -2.5427523262585563, 3.9551312168955626]]], [[[4.0479839543530591, 
1.694708528108122, -1.8081650371476021, 2.5627212563151982], [2.9443513555348222, -3.4330381296191126, 
-2.3471872352829837, 2.9291777099369405], [0.92208424820838264, -1.7857214370413055, 3.2638247404414695, 
3.3713981402987798]], [[-2.3853121535462418, 2.1417428055374232, 3.1558224539661612, -4.4802179321245248], 
[-3.0197245205703069, 2.7624146301708477, -4.6790033997765104, -4.0453165901737584], [4.8295161047601614, 
-3.5764718373510842, 4.356981591617421, -4.7034098127513264]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-3.3871470175211567, 4.1065106330294565, -4.95135818706596, -2.856919373064823], 
[-2.0107036194476571, -0.5877396435918989, -1.0713007160377828, 3.9940497640044068], [3.8571436898492628, 
2.163613809661558, -4.0419892883730775, 4.2533571833533097]], [[1.165440226072501, 0.71921582424756014, 
-3.426501866749005, -2.7401980998093363], [-0.01686639655065747, -3.8353689254676855, 1.1213889656318003, 
1.776063945051531], [-4.4537689224644019, 2.9471778994634885, 4.6113620792916334, 3.543757452377851]]], 
[[[0.7518716666574301, -0.45251668017141533, 1.8405559934538078, 1.9736076282138142], [-2.3316862315078231, 
4.0931942893218567, -1.1800266395527057, -0.79570885881318087], [0.053553616108191981, 4.4595654602491237, 
2.4755524744366948, -3.5243939866400069]], [[-1.283680379531603, 4.1552563181801752, -4.911313554906533, 
0.37281516674407023], [1.5768230935383114, -3.0233278144508207, -4.7991290832674132, 3.4503111405040645], 
[-2.3237265585848226, 0.2274633573855489, 2.3274111425324779, -4.1704724006216409]]], [[[-4.2633251380791375, 
-1.9100497118342004, 1.5928238534215238, -2.7780624400412766], [-3.1596925392609005, 3.2176969458930342, 
2.1318460515569053, -3.1445188936630188], [-1.137425431934461, 1.5703802533152271, -3.4791659241675479, 
-3.5867393240248582]], [[2.1699709698201635, -2.3570839892635016, -3.3711636376922396, 4.2648767483984464], 
[2.8043833368442286, -2.977755813896926, 4.463662216050432, 3.8299754064476801], [-5.0448572884862397, 
3.3611306536250058, -4.5723227753434994, 4.4880686290252481]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-6.1909769509085075, 1.3026806996421056, -7.7551881204533109, 
-5.6607493064521739], [-4.8145335528350079, -3.3915695769792498, -3.8751306494251336, 1.1902198306170559], 
[1.0533137564619119, -0.64021612372579284, -6.8458192217604283, 1.4495272499659588]], [[-1.6383897073148499, 
-2.0846141091397907, -6.2303318001363559, -5.5440280331966871], [-2.8206963299380083, -6.6391988588550364, 
-1.6824409677555505, -1.0277659883358199], [-7.2575988558517528, 0.14334796607613765, 1.8075321459042826, 
0.73992751899050013]]], [[[-2.0519582667299208, -3.2563466135587662, -0.96327393993354304, -0.83022230517353668], 
[-5.1355161648951739, 1.2893643559345058, -3.9838565729400566, -3.5995387922005317], [-2.7502763172791589, 
1.6557355268617728, -0.32827745895065608, -6.3282239200273578]], [[-4.0875103129189538, 1.3514263847928243, 
-7.7151434882938839, -2.4310147666432806], [-1.2270068398490395, -5.8271577478381715, -7.602959016654764, 
0.64648120711671364], [-5.1275564919721734, -2.576366576001802, -0.47641879085487293, -6.9743023340089918]]], 
[[[-7.0671550714664884, -4.7138796452215512, -1.2110060799658271, -5.5818923734286274], [-5.9635224726482514, 
0.41386701250568336, -0.67198388183044555, -5.9483488270503697], [-3.9412553653218119, -1.2334496800721237, 
-6.2829958575548988, -6.390569257412209]], [[-0.63385896356718741, -5.1609139226508525, -6.1749935710795905, 
1.4610468150110956], [0.0005534034568777102, -5.7815857472842769, 1.6598322826630811, 1.0261454730603292], 
[-7.8486872218735906, 0.55730072023765498, -7.3761527087308503, 1.6842386956378972]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([3.3101673523710691, 0.048409361416743124]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.70887806236646611, -0.73932065177372408]))
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(1.15960287006)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([2.1505644823090515, -1.1111935086452744]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.45072480769555145, -1.8989235218357416]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([-2.0708546339036071, 2.2714034647505121]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-0.16265022615439584, -0.29272834777410406]))
      arg1=Symbol(shape=(2,))
      res=arg0-arg1
      s1=numpy.array([1.8495632665872739, -2.2808524667130694])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-3.920417900490881, 4.5522559314635815]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-2.0122134927416697, 1.9881241189389653]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[4.703380807076492, -4.2567944639019304, -2.0784707905046593, 0.18023637488621791, 
1.1164321428411501], [3.3809585074696322, 1.5795463086222137, 1.5300027430790495, -1.6695215658775489, 
-4.9671698822372887], [-0.56875186129757704, -0.88988163011215704, 1.0953422249288387, 1.2629450835517639, 
1.9829321534877584], [-2.3470243950738103, -1.5345245349366401, 1.7913793425402638, 3.2778179482022125, 
3.2743088989127749]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[2.1331140495285128, 4.902243346193929, -3.8569193535703947, 
-1.2051025219030698, 4.8526791592750644], [-1.9285295160668192, -2.2715983725035862, -1.6280809153232632, 
0.63571110979312273, -4.5616322454088643], [1.1933837591252878, -2.4657544917793928, 3.8511059475300904, 
-3.0018611957635444, 3.560382804940847], [-4.284584247208282, -4.3366343606789348, 3.6048395763720524, 
-2.2301793774115106, 4.6397261587379131]]))
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(0.0560012612314)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[4.6473795458450571, -4.3127957251333653, -2.1344720517360942, 0.12423511365478301, 
1.0604308816097152], [3.3249572462381973, 1.5235450473907788, 1.4740014818476146, -1.7255228271089837, 
-5.0231711434687236], [-0.62475312252901194, -0.94588289134359194, 1.0393409636974038, 1.206943822320329, 
1.9269308922563235], [-2.4030256563052452, -1.590525796168075, 1.7353780813088289, 3.2218166869707776, 
3.21830763768134]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[2.0771127882970779, 4.8462420849624941, -3.9129206148018296, 
-1.2611037831345047, 4.7966778980436295], [-1.9845307772982541, -2.3275996337350211, -1.6840821765546981, 
0.57970984856168783, -4.6176335066402991], [1.1373824978938529, -2.5217557530108277, 3.7951046862986555, 
-3.0578624569949793, 3.5043815437094121], [-4.3405855084397169, -4.3926356219103697, 3.5488383151406175, 
-2.2861806386429455, 4.5837248975064782]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[0.044613582775737015, -0.22965054883260905, -3.3954728255423361, -0.043404784226975579, 
-0.81018025865095922], [4.0980455142640473, 3.3299876326958326, 4.4694158188546833, 0.047800124529065791, 
-4.1128886475115927], [-0.86793714814288414, 3.7852706993586231, 2.8168181178475837, -2.6081900317073039, 
1.795227525921204], [-2.7964436060814792, 2.46599228887926, -4.3894587372918519, -3.0809581135280197, 
4.5629513161933648]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[0.18467263707487369, -2.906541382403959, -4.2471361917218733, 
1.7478696798949915, -2.0555035204044225], [-4.1703824796767011, -0.58145273211245829, -1.3034416354534684, 
-4.4238643252257699, -3.0019960418182654], [-0.011560599410600503, 4.5614736908410478, -4.1865499712522745, 
0.41611035316936196, 1.4719370557053075], [3.3285499812876207, 4.2147545548351992, 3.8796865015190463, 
-2.8665673368928459, 3.8754754018195001]]))
      arg1=Symbol(shape=(4, 5))
      res=arg0-arg1
      s1=numpy.array([[-0.34040680852948757, 0.51480179015857086, 2.6579250902566542, -3.8908104282358877, 
-1.0766494604779266], [-1.7785348143550985, 1.7875285221080928, -0.26464821727786259, 3.7856697734154743, 
0.14935084548977784], [1.6454427368239299, -3.0878902261983701, 2.1577262475041596, -3.540342914142153, 
2.8529020416879671], [2.8849125795379305, -3.1409630887157123, -0.30215664293811351, 3.5493007526176896, 
0.27226779139430857]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[0.38502039130522459, -0.74445233899117991, -6.0533979157989908, 3.8474056440089122, 
0.26646920182696743], [5.8765803286191458, 1.5424591105877399, 4.7340640361325459, -3.7378696488864085, 
-4.2622394930013705], [-2.5133798849668141, 6.8731609255569932, 0.65909187034342409, 0.93215288243484906, 
-1.0576745157667631], [-5.6813561856194097, 5.6069553775949723, -4.0873020943537384, -6.6302588661457094, 
4.2906835247990562]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.52507944560436126, -3.4213431725625298, -6.9050612819785275, 
5.6386801081308793, -0.97885405992649588], [-2.3918476653216025, -2.3689812542205511, -1.0387934181756058, 
-8.2095340986412442, -3.1513468873080432], [-1.6570033362345304, 7.6493639170394179, -6.3442762187564341, 
3.9564532673115149, -1.3809649859826596], [0.44363740174969024, 7.3557176435509115, 4.1818431444571598, 
-6.4158680895105356, 3.6032076104251916]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[-0.70323441272603926, -1.4205742401701604], [-3.6004008923276585, 4.1739347100888349]], 
[[-2.7687391296703767, -0.96114141211843496], [0.45711266950319906, 0.36713165606152121]], [[3.8726070188081287, 
2.6611494194452137], [-0.28060302358441547, 1.0399275995737964]], [[2.5912385881777, -0.12172669528696911], 
[1.831517522951442, -4.9891623764024926]], [[3.8572507842255241, 2.9719918728052663], [0.42882676434271261, 
-1.4826468418372341]], [[0.16110396579090835, 4.8052378752678955], [2.4890225545274554, 
-1.4594734254395068]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[3.4601998637619467, 3.5105292543746671], [-1.9715134513187751, 
1.6897677346566677]], [[0.99895689216195205, 3.7908023259957879], [-2.9811497902134496, 0.46336396583979944]], 
[[-2.0979181014824011, 0.68992077008736707], [4.5817275596392033, 3.1112543881649586]], [[-1.0666850119171398, 
-3.7136243224538679], [-2.1842168128700248, -0.60998709362389292]], [[-1.0817587775668578, 1.1357523207967555], 
[0.72114300996433212, 2.0871085948686607]], [[2.6196090777455074, -4.8403131105182826], [4.4462612480444346, 
2.6275786734235638]]]))
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(3.40075496466)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-4.1039893773891789, -4.8213292048333001], [-7.0011558569907981, 0.77317974542569523]], 
[[-6.1694940943335164, -4.3618963767815746], [-2.9436422951599406, -3.0336233086016184]], [[0.4718520541449891, 
-0.73960554521792599], [-3.6813579882475551, -2.3608273650893432]], [[-0.80951637648543961, -3.5224816599501088], 
[-1.5692374417116977, -8.3899173410656331]], [[0.4564958195623845, -0.42876309185787331], [-2.971928200320427, 
-4.8834018065003733]], [[-3.2396509988722313, 1.4044829106047558], [-0.91173241013568429, 
-4.8602283901026464]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.059444899098807014, 0.10977428971152747], [-5.3722684159819147, 
-1.7109872300064719]], [[-2.4017980725011876, 0.39004736133264828], [-6.3819047548765893, -2.9373909988233402]], 
[[-5.4986730661455407, -2.7108341945757726], [1.1809725949760637, -0.28950057649818106]], [[-4.4674399765802795, 
-7.1143792871170071], [-5.5849717775331644, -4.0107420582870326]], [[-4.4825137422299974, -2.2650026438663842], 
[-2.6796119546988075, -1.3136463697944789]], [[-0.7811458869176322, -8.2410680751814223], [1.0455062833812949, 
-0.77317629123957587]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[-2.8893927498914151, -3.9495986710021471], [2.0674301637688552, -4.9323681378020368]], 
[[-3.9365223323164567, -3.9166796931279513], [-2.1295831296849688, 0.049270642730291137]], [[1.1604521699930164, 
-4.7263968957110194], [0.18403419227820805, -3.9919770732677948]], [[-4.4683480884742268, 3.1077188243660192], 
[0.090355977211302729, -0.013539049772621325]], [[1.2239143556433882, 4.66468811676115], [4.6443599318212119, 
2.902664355759085]], [[3.1499666861977964, 3.5678517696258449], [0.73557701807290599, 
-4.1703133219986768]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.62745401025262382, 0.69024538347902542], [4.3685303267738433, 
2.2109723240557235]], [[-0.7348498808881363, -2.7513236139357309], [2.5887407011037489, 4.1931952710033542]], 
[[2.1336250254996258, -2.1610465999144091], [-4.054796877122568, 0.054975312915938268]], [[2.8778982280083021, 
0.031841424972327559], [-1.6040852288365626, -0.14653197703489251]], [[1.0241081083490533, 2.0236436389548764], 
[-4.7683548819587331, 0.81201234013234735]], [[-3.2923450240347405, 2.2531528995219965], [-3.594199051432386, 
-1.9523442452177875]]]))
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0-arg1
      s1=numpy.array([[[0.67454553417657603, 2.9833990689244789], [-3.9375622829117427, 0.0094498156860893801]], 
[[2.1574617938010734, -0.48892733726965609], [0.62118276066421352, 0.99065918564407696]], [[1.7968244154456219, 
-1.6314349433046926], [1.8612952961850224, 4.6630470176393288]], [[0.43763307675500052, 4.0271951272236688], 
[-1.1711764825930993, -4.5547560714878275]], [[2.514477748308436, 3.7600620047710827], [1.5805136896170069, 
2.4948517124974012]], [[-0.74781838229224817, -2.9876928953003903], [4.1339271192034222, 4.4719827170790509]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-3.5639382840679912, -6.932997739926626], [6.004992446680598, -4.9418179534881261]], 
[[-6.0939841261175296, -3.4277523558582952], [-2.7507658903491823, -0.94138854291378582]], [[-0.63637224545260551, 
-3.0949619524063268], [-1.6772611039068144, -8.6550240909071228]], [[-4.9059811652292273, -0.91947630285764959], 
[1.261532459804402, 4.5412170217152061]], [[-1.2905633926650477, 0.90462611199006737], [3.063846242204205, 
0.40781264326168376]], [[3.8977850684900446, 6.5555446649262352], [-3.3983501011305162, 
-8.6422960390777277]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-0.047091523923952217, -2.2931536854454535], [8.3060926096855852, 
2.2015225083696341]], [[-2.8923116746892097, -2.2623962766660748], [1.9675579404395354, 3.2025360853592773]], 
[[0.33680061005400397, -0.52961165660971643], [-5.9160921733075904, -4.6080717047233906]], [[2.4402651512533016, 
-3.9953537022513412], [-0.43290874624346332, 4.4082240944529349]], [[-1.4903696399593827, -1.7364183658162062], 
[-6.34886857157574, -1.6828393723650539]], [[-2.5445266417424923, 5.2408457948223868], [-7.7281261706358082, 
-6.4243269622968384]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[3.1002455029763922, 2.6515488300516923, -0.77582358496211956, -3.4443694355246803], 
[-2.6599091620789581, -0.70044327546902529, -4.3223485855396966, 4.9338402947088049], [-4.5546987200991147, 
-4.159833516760548, -1.2113818643763619, 1.341501344402797]], [[-0.99132126989665803, -3.81966827017445, 
-1.5631671743562592, -2.9170370396917167], [0.94015514336519956, -4.5328623228274036, 2.5469993786586862, 
4.5298447080413311], [-1.8826808741220304, -0.21100480137345734, -1.7750931594239239, -3.5343470478632764]]], 
[[[-3.4624410933639691, 3.7419877938482422, -4.1641241285521557, -2.8763768520849711], [4.3838179808162643, 
-0.076650368742670949, -2.2790272387608601, 1.4407514353417152], [-0.58059366739859364, 3.0282179950037378, 
4.3946428646333242, -3.9361840734571896]], [[-0.40769305246403231, -0.93123230765280152, -3.5500981163613665, 
-1.4382421516555786], [0.18862577968690264, 3.8234595158976035, 1.2783334948832605, -0.84599833008897818], 
[-1.5452449895609535, -2.1285283532469434, 2.9517034908101669, -1.043778516582341]]], [[[2.5188074736534176, 
4.926760464276164, -1.2494158315784532, -4.1847607799981805], [1.764772573553314, 4.6090994448443769, 
-3.7864884573437072, 2.5743244083963681], [-0.44624416686502322, -0.44288726525437028, -2.5180469174818598, 
-4.8009656021603]], [[-1.0967276921708047, -1.5639987059537273, -3.3122649580537331, -3.947879272385495], 
[4.1267460589959857, -4.5801997177900287, 0.85366271506547697, -3.5573421152778972], [-4.7127368302025108, 
-4.5592524679039892, -1.8586387462495613, -3.2614675219884837]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-3.6140016210408508, -4.1545999292001445, 4.9863169403898908, 
-2.2007289242442383], [-2.3634275248295822, 1.4929955627211893, 1.1905831175627091, -3.1298255396253936], 
[-0.78867439130174599, -2.5664248245819756, -1.882393556334109, -2.3300345925878529]], [[3.7578772846055983, 
-1.9632657478837121, -1.3792653830852455, -0.23840250166856869], [-1.650781665029756, -3.2744446113480907, 
-1.2541229166086589, -2.3471598629273149], [-1.939332795628903, 0.81542234976851624, 0.52422540705571663, 
0.91808367692950554]]], [[[-3.0689349511345867, -4.8032602579819264, 3.769084882991141, -1.5864959564378189], 
[-3.2063200431555905, -0.3347729502698602, 1.763270929850381, 0.65936335478094321], [-3.6143633139881959, 
0.15424644431103118, 3.7156782910709154, -3.2826914978804203]], [[-0.091940996157960697, 2.5331247115220021, 
3.4383904670893202, 0.77887041122794898], [4.2850997491436988, 3.3877021574758341, 3.9303516193668084, 
0.97217787674818279], [-1.8219977615256742, 3.7582967180633755, -3.967674705101544, 3.2183851949652524]]], 
[[[3.8000102844693906, -2.9266220460152672, 0.11901081743168795, -0.70455205529677301], [4.6787843021952913, 
-3.2637583894745239, 4.6693989140352041, 2.042172937625808], [-2.9445501417858964, 0.36254085518902812, 
2.8333171427728354, -2.7757509476245721]], [[3.8180860212706147, -3.4817247466262815, -3.2683613783585006, 
-2.0706219843820262], [4.8065072235822566, 2.2788211866672707, 3.8562835841415382, -1.1633706258500731], 
[2.652336823163191, -2.6060953909144513, 0.62089818312127321, -1.6242126976534612]]]]))
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(-4.55573857649)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[7.6559840794689205, 7.2072874065442205, 3.7799149915304087, 1.1113691409678479], 
[1.8958294144135701, 3.8552953010235029, 0.23338999095283164, 9.4895788712013331], [0.0010398563934135296, 
0.3959050597319802, 3.3443567121161664, 5.8972399208953252]], [[3.5644173065958702, 0.73607030631807824, 
2.992571402136269, 1.6387015368008115], [5.4958937198577278, 0.02287625366512458, 7.1027379551512144, 
9.0855832845338593], [2.6730577023704978, 4.3447337751190709, 2.7806454170686044, 1.0213915286292519]]], 
[[[1.0932974831285591, 8.2977263703407704, 0.39161444794037248, 1.6793617244075572], [8.9395565573087925, 
4.4790882077498573, 2.2767113377316681, 5.9964900118342435], [3.9751449090939346, 7.583956571496266, 
8.9503814411258524, 0.61955450303533866]], [[4.1480455240284959, 3.6245062688397267, 1.0056404601311617, 
3.1174964248369497], [4.7443643561794309, 8.3791980923901317, 5.8340720713757888, 3.70974024640355], 
[3.0104935869315748, 2.4272102232455848, 7.5074420673026951, 3.5119600599101872]]], [[[7.0745460501459458, 
9.4824990407686922, 3.3063227449140751, 0.3709777964943477], [6.3205111500458422, 9.1648380213369052, 
0.76925011914882102, 7.1300629848888963], [4.109494409627505, 4.1128513112381579, 2.0376916590106684, 
-0.24522702566777177]], [[3.4590108843217235, 2.991739870538801, 1.2434736184387951, 0.60785930410703326], 
[8.6824846354885139, -0.024461141297500433, 5.4094012915580052, 0.99839646121463099], [-0.15699825370998255, 
-0.0035138914114609676, 2.697099830242967, 1.2942710545040446]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[0.94173695545167746, 0.40113864729238369, 9.542055516882419, 
2.35500965224829], [2.192311051662946, 6.0487341392137175, 5.7463216940552373, 1.4259130368671347], 
[3.7670641851907822, 1.9893137519105526, 2.6733450201584192, 2.2257039839046753]], [[8.3136158610981266, 
2.5924728286088161, 3.1764731934072827, 4.3173360748239595], [2.9049569114627722, 1.2812939651444375, 
3.3016156598838693, 2.2085787135652133], [2.6164057808636252, 5.3711609262610445, 5.0799639835482449, 
5.4738222534220338]]], [[[1.4868036253579415, -0.24752168148939813, 8.3248234594836692, 2.9692426200547093], 
[1.3494185333369377, 4.220965626222668, 6.3190095063429093, 5.2151019312734714], [0.94137526250433234, 
4.7099850208035594, 8.2714168675634436, 1.273047078612108]], [[4.4637975803345675, 7.0888632880145304, 
7.9941290435818484, 5.3346089877204772], [8.8408383256362271, 7.9434407339683624, 8.4860901958593367, 
5.527916453240711], [2.7337408149668541, 8.3140352945559037, 0.58806387139098426, 7.7741237714577807]]], 
[[[8.3557488609619188, 1.629116530477261, 4.6747493939242162, 3.8511865211957552], [9.2345228786878195, 
1.2919801870180043, 9.2251374905277324, 6.5979115141183362], [1.6111884347066319, 4.9182794316815563, 
7.3890557192653636, 1.7799876288679561]], [[8.3738245977631429, 1.0740138298662467, 1.2873771981340276, 
2.4851165921105021], [9.3622458000747848, 6.834559763159799, 8.4120221606340664, 3.3923679506424551], 
[7.2080753996557192, 1.9496431855780769, 5.1766367596138014, 2.931525878839067]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_taggedData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[2.1869721643026576, 0.35542091272423715, 2.5099944114031967, 4.7276012581949995], 
[-0.23596027111215712, 3.2557128306673206, -2.4174678213407566, 4.9025765849007588], [3.4987602616867228, 
-2.3969967727517094, 2.614715035832643, -3.9538109091356577]], [[0.54151166641114745, 4.3433313907072311, 
-3.9824411189395126, 0.11193040884063787], [-4.3326960505433521, -2.6555021449849603, -1.6650005107909016, 
-0.21278258756168267], [2.9438726263016104, 4.614591333740627, -1.4283352855346321, 4.195747529596801]]], 
[[[0.4129039465707498, 0.25218586208094607, 4.2227877593235625, -3.8395686827717723], [-4.246422814789943, 
-4.2708029152046789, -4.4791253262093615, 2.3703854064691221], [-0.32074671911367325, -4.0633264555676574, 
-4.8034904727622223, 0.101245496731595]], [[3.3860052077100544, 4.4048456672981686, 3.3258905421337257, 
-0.60591078242426555], [2.9574702297232829, 2.9390786518156196, 3.0627580449874809, -2.1902821038190523], 
[1.2765769390449559, 4.5442832941192819, 0.47031486471564055, -3.2094801674304509]]], [[[1.4972627407797212, 
-2.7514173987810633, 0.19744444113354387, 1.3720920976100972], [-3.147124860705004, -3.6707691951555885, 
1.1521564952279704, -0.12493802519996233], [1.3717811158015873, -1.737983464544548, -2.5919544001996897, 
-4.4195022009129206]], [[-3.5078213357756582, 1.5909514876001909, 3.932618549290213, 0.32844467348406869], 
[-0.037083415286228494, 2.358949404615915, -3.7082781631298478, -4.9441324919087766], [1.219588665287433, 
-2.1155364750524797, 2.3443039764677165, 4.1618790582351313]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[3.8216987557975131, -0.59039813916696193, -1.9474433412604117, 
4.1666345075852202], [1.0033840403657788, -1.8365638623400207, -1.1472895447555285, 0.49043998461267968], 
[1.525782098623524, 0.98710575843395354, 1.9521603305269073, 1.4982217977497818]], [[4.8105014981222372, 
0.18255767851204219, 0.10092997041413909, 2.3610713615733667], [3.8639541584797801, 1.8455276769077198, 
3.9278199867001007, 2.5501176762845867], [3.2925051662999447, 0.78129602184334157, -0.73105877010655362, 
2.9378923845982694]]], [[[1.3162347911484948, -1.7534583809398363, -4.4745574675152744, 0.84388146264593455], 
[-2.1398633576757309, 1.6224556269216279, 4.0151064679341637, 0.81646760002277574], [0.95506629968888479, 
-3.384786519820715, 2.08961451298733, 1.4802214615087061]], [[2.5752388025402837, -2.7094797245847468, 
-2.6808155024703106, -1.7780191613070642], [-0.58755728186204248, -4.3097624692690948, 3.6757907841395685, 
-1.8312242243207608], [-3.7229135985460826, -1.5786991892133564, 2.6894504757052617, -0.48567336902160463]]], 
[[[3.4562176552233623, -1.5291903913231595, 4.9276217294297595, -1.4641622460496571], [-3.9633150641051529, 
-1.3895475276782743, -2.0928641563143735, 4.286214622292805], [-0.016872120519226819, -0.86571000346058913, 
4.2635805792181465, 4.0351866281897113]], [[-1.973695982407413, -4.452260246087465, -2.5681734906597109, 
3.0954829513656215], [2.6526834215550927, -4.3976717675273207, 2.0111485813735106, 2.7969396373439324], 
[-0.72100288848623784, 1.4868693846138363, 2.3876845459322045, -3.759851286518614]]]]))
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0-arg1
      s1=numpy.array([[[[-1.2326165508314046, 0.019536700697927678, 3.3313535404093759, -2.4782775769684271], 
[3.9342491756801525, 1.2904741959913864, -2.7701975380199206, 2.4757520771582744], [2.5202328466158281, 
-1.3683915774027189, 3.4678638218372768, -2.2884507446983129]], [[-4.9275394706777931, 4.7975831194456333, 
1.7829898690658723, -0.96339421834763073], [-2.7923805247323799, -0.026981154987572253, 2.5136604629187271, 
0.14658337947380495], [1.1254475424349959, 4.8000437885357261, 3.3479331374253167, 1.6298765760037002]]], 
[[[-0.46473842692243572, 1.2430212762010644, -0.23618382206216726, -1.2230171932711418], [2.0127498669810855, 
-0.31475870950595031, -0.20645609212011973, -4.9825089187683691], [-4.6108703987985988, -0.47963035537661725, 
-3.1919702863790422, -3.9993603357626117]], [[3.8402219409685951, 3.04406815317755, 4.7640360318949195, 
1.5279973254325983], [-4.9716807317737235, -3.4706635767559693, -1.2581696190523903, -2.591452040312936], 
[1.6191001515432157, -3.5419762128533741, 0.92904425652178801, 4.6966930122512043]]], [[[-2.4787875268428614, 
4.8717538415307775, 3.6264063974305554, 2.0645154974740256], [-4.5070489852671329, 2.3540394703493703, 
3.2007816723140134, -0.44359603196672026], [2.5406621078154732, 3.6651768892659895, -2.7039262200534422, 
-1.9309627063916244]], [[-0.037762488646412962, -4.6825147640959859, -3.1180187992817956, -0.3407644296025687], 
[-1.6601757648009907, -1.0174825465103088, 0.060955158106047236, 1.2341204474061849], [-0.24621306712976931, 
-1.3620636349151272, -0.12322079758969373, 2.3717593913603183]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[3.4195887151340623, 0.33588421202630947, -0.82135912900617924, 7.2058788351634266], 
[-4.1702094467923096, 1.9652386346759343, 0.35272971667916408, 2.4268245077424844], [0.97852741507089469, 
-1.0286051953489905, -0.85314878600463384, -1.6653601644373448]], [[5.4690511370889405, -0.45425172873840225, 
-5.7654309880053844, 1.0753246271882686], [-1.5403155258109722, -2.628520989997388, -4.1786609737096292, 
-0.35936596703548762], [1.8184250838666145, -0.18545245479509909, -4.7762684229599488, 2.5658709535931008]]], 
[[[0.87764237349318552, -0.99083541412011833, 4.4589715813857298, -2.6165514895006305], [-6.2591726817710285, 
-3.9560442056987286, -4.2726692340892418, 7.3528943252374912], [4.2901236796849256, -3.5836961001910401, 
-1.6115201863831801, 4.1006058324942067]], [[-0.45421673325854073, 1.3607775141206186, -1.4381454897611938, 
-2.1339081078568638], [7.9291509614970064, 6.4097422285715888, 4.3209276640398713, 0.40116993649388366], 
[-0.34252321249825979, 8.0862595069726559, -0.45872939180614747, -7.9061731796816552]]], [[[3.9760502676225826, 
-7.6231712403118408, -3.4289619562970115, -0.69242339986392842], [1.359924124562129, -6.0248086655049589, 
-2.0486251770860431, 0.31865800676675793], [-1.1688809920138858, -5.4031603538105379, 0.11197181985375249, 
-2.4885394945212962]], [[-3.4700588471292453, 6.2734662516961768, 7.0506373485720086, 0.66920910308663739], 
[1.6230923495147622, 3.3764319511262237, -3.7692333212358951, -6.1782529393149614], [1.4658017324172024, 
-0.7534728401373525, 2.4675247740574102, 1.7901196668748129]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[5.0543153066289177, -0.60993483986488961, -5.2787968816697877, 
6.6449120845536473], [-2.9308651353143738, -3.127038058331407, 1.6229079932643922, -1.9853120925455947], 
[-0.99445074799230415, 2.3554973358366724, -1.5157034913103695, 3.7866725424480947]], [[9.7380409688000302, 
-4.6150254409335911, -1.6820598986517332, 3.3244655799209974], [6.6563346832121599, 1.872508831895292, 
1.4141595237813736, 2.4035342968107818], [2.1670576238649488, -4.0187477666923845, -4.0789919075318704, 
1.3080158085945692]]], [[[1.7809732180709306, -2.9964796571409007, -4.2383736454531071, 2.0668986559170763], 
[-4.1526132246568164, 1.9372143364275782, 4.2215625600542834, 5.7989765187911448], [5.5659366984874836, 
-2.9051561644440977, 5.2815847993663727, 5.4795817972713179]], [[-1.2649831384283114, -5.7535478777622968, 
-7.4448515343652302, -3.3060164867396624], [4.384123449911681, -0.83909889251312553, 4.9339604031919588, 
0.76022781599217515], [-5.3420137500892988, 1.9632770236400177, 1.7604062191834737, -5.1823663812728089]]], 
[[[5.9350051820662237, -6.400944232853937, 1.3012153319992041, -3.5286777435236827], [0.54373392116198005, 
-3.7435869980276446, -5.293645828628387, 4.7298106542595253], [-2.5575342283347, -4.5308868927265786, 
6.9675067992715887, 5.9661493345813357]], [[-1.935933493761, 0.23025451800852093, 0.54984530862208469, 
3.4362473809681902], [4.3128591863560839, -3.3801892210170119, 1.9501934232674634, 1.5628191899377475], 
[-0.47478982135646852, 2.8489330195289635, 2.5109053435218982, -6.1316106778789319]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(2.42413566075)+(1.-msk_arg0)*(2.73592046896)
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(0.0730314190245)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(2.35110424173)+(1.-msk_ref)*(2.66288904994)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-2.38585027921)+(1.-msk_arg0)*(-2.14546935212)
      arg1=Symbol(shape=(2,))
      res=arg0-arg1
      s1=numpy.array([1.0449404678521192, -2.9654578889240057])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-3.4307907470591283, 
0.57960760971699665])+(1.-msk_ref)*numpy.array([-3.1904098199744872, 0.81998853680163775])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(2.15276640076)+(1.-msk_arg0)*(-2.04284766814)
      arg1=Symbol(shape=(4, 5))
      res=arg0-arg1
      s1=numpy.array([[-2.5429314638433684, 2.0318827224945402, -2.3636856893688076, 3.4855417570765717, 
0.44952339669472341], [2.5403509140391156, 2.3524971436536095, 3.9461465487262188, 2.6955339698780154, 
-0.45702899742654868], [-1.0602022717036155, 0.74771157767510843, 1.6452939357358289, -3.0322095528230921, 
1.6787335078454735], [-4.263078102519902, 3.2046384335109863, 4.0147512257312048, 3.3998288702285713, 
-0.56118778404289138]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[4.6956978646047602, 0.12088367826685165, 4.5164520901301994, -1.3327753563151798, 
1.7032430040666684], [-0.38758451327772381, -0.19973074289221771, -1.793380147964827, -0.5427675691166236, 
2.6097953981879405], [3.2129686724650073, 1.4050548230862834, 0.50747246502556287, 5.1849759535844839, 
0.47403289291591832], [6.4158445032812939, -1.0518720327495945, -1.861984824969813, -1.2470624694671795, 
2.7139541848042832]])+(1.-msk_ref)*numpy.array([[0.50008379570506278, -4.0747303906328458, 0.32083802123050198, 
-5.5283894252148773, -2.4923710648330291], [-4.5831985821774213, -4.3953448117919152, -5.9889942168645245, 
-4.7383816380163211, -1.585818670711757], [-0.98264539643469018, -2.7905592458134141, -3.6881416038741346, 
0.98936188468478647, -3.7215811759837791], [2.2202304343815964, -5.247486101649292, -6.0575988938695104, 
-5.4426765383668769, -1.4816598840954143]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(3.30825297654)+(1.-msk_arg0)*(-3.92076322418)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0-arg1
      s1=numpy.array([[[-0.52002332126128437, 4.3478222442071139], [3.3434922005534364, 2.8013302606159396]], 
[[-2.3200079969586795, -3.0556917667690642], [-2.7103276420969582, 4.1511200748037105]], [[-0.92404095393396624, 
2.6484690327098859], [-2.1529217611726503, 4.4602897709717144]], [[0.58271708006920253, 1.9322598870751975], 
[-3.5184596230462182, -4.4222029485403436]], [[-4.3953168785776278, -4.450145776704125], [4.2137072146995536, 
3.8966485797913304]], [[3.1838339108927798, -3.6438064267677328], [1.3789445362861974, -2.9975552731311272]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[3.8282762978010325, -1.0395692676673658], [-0.035239224013688286, 
0.50692271592380855]], [[5.6282609734984277, 6.3639447433088119], [6.0185806186367063, -0.84286709826396233]], 
[[4.2322939304737144, 0.65978394382986227], [5.4611747377123985, -1.1520367944319663]], [[2.7255358964705456, 
1.3759930894645507], [6.8267125995859663, 7.7304559250800917]], [[7.7035698551173759, 7.7583987532438732], 
[-0.90545423815980541, -0.58839560325158224]], [[0.12441906564696836, 6.952059403307481], [1.9293084402535507, 
6.3058082496708749]]])+(1.-msk_ref)*numpy.array([[[-3.4007399029178136, -8.2685854683862114], [-7.2642554247325339, 
-6.7220934847950371]], [[-1.6007552272204184, -0.86507145741003377], [-1.2104355820821397, -8.071883298982808]], 
[[-2.9967222702451317, -6.5692322568889843], [-1.7678414630064476, -8.3810529951508119]], [[-4.5034803042483009, 
-5.853023111254295], [-0.40230360113287977, 0.50143972436124562]], [[0.47455365439852981, 0.52938255252502708], 
[-8.1344704388786511, -7.8174118039704279]], [[-7.1045971350718773, -0.27695679741136514], [-5.2997077604652958, 
-0.92320795104797071]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank0_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.28115160685)+(1.-msk_arg0)*(-2.99624588284)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0-arg1
      s1=numpy.array([[[[-3.7034845683259832, -3.2006988280486115, 4.6850337347787345, -1.5431340070704103], 
[-3.9508001556883876, 4.7231128873762902, -3.0051096732527691, -0.071944916970104522], [3.2109725637398565, 
4.0910170733379978, -3.7166755556626772, -4.402146700420734]], [[-1.5273991623031669, -1.4865381526416344, 
3.902360473786171, -1.3538484671917517], [0.38707743115008331, 4.3855048056490773, 1.9022231675241139, 
1.397387379628614], [1.0431068102446126, 3.0934379513218886, 2.0138255231319624, 4.2870052231295865]]], 
[[[-4.2737086360299941, 4.2752748398653857, -3.7092106416006629, 1.417380944080846], [-2.4275128587779737, 
-2.879911926405645, -4.23153844815229, -0.30555854124221682], [-2.6571106905165331, 2.6754859746804112, 
-4.5544081791240201, -0.020082609244357563]], [[1.0570642052363857, -1.7647078574502792, 2.6330635742775668, 
3.717540829723692], [4.9220552078075279, -3.9060168420798869, 1.4799017868437296, 2.7842835488914588], 
[-2.0839669385912343, -4.8850626605172867, 1.7595980725429907, 3.0026383083452117]]], [[[-0.83195539201513036, 
-1.2109400306251725, 2.0638657571201078, -0.86905066581365009], [-0.54092453152611775, 3.4954317917180884, 
3.7826658876966359, -2.5779636206330894], [1.6720368874738147, 0.42564364358069096, -4.9027760864384096, 
0.66861897918883617]], [[-4.1302737255553801, -3.2949127465748109, 1.5706320204575341, -2.2912291830881903], 
[-2.19574275564025, 3.983182476523945, 2.032922034582441, -2.7459308093848711], [4.6025690264891459, 
3.7012963844874829, 0.1748188819614116, 4.2002322255258893]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[7.984636175171131, 7.4818504348937598, -0.40388212793358669, 5.8242856139155581], 
[8.2319517625335354, -0.44196128053114236, 7.286261280097917, 4.3530965238152524], [1.0701790431052913, 
0.19013453350715004, 7.997827162507825, 8.6832983072658827]], [[5.8085507691483151, 5.7676897594867818, 
0.37879113305897683, 5.6350000740368991], [3.8940741756950645, -0.10435319880392946, 2.3789284393210339, 
2.8837642272165338], [3.2380447966005352, 1.1877136555232592, 2.2673260837131854, -0.0058536162844387007]]], 
[[[8.5548602428751419, 0.005876766979762138, 7.9903622484458108, 2.8637706627643018], [6.7086644656231211, 
7.1610635332507933, 8.5126900549974387, 4.5867101480873647], [6.938262297361681, 1.6056656321647367, 8.835559785969167, 
4.3012342160895054]], [[3.2240874016087622, 6.0458594642954271, 1.6480880325675811, 0.5636107771214558], 
[-0.64090360096238008, 8.1871684489250356, 2.8012498200014182, 1.496868057953689], [6.3651185454363821, 
9.1662142673624345, 2.5215535343021571, 1.2785132984999361]]], [[[5.1131069988602782, 5.4920916374703204, 
2.2172858497250401, 5.1502022726587979], [4.8220761383712656, 0.78571981512705946, 0.49848571914851192, 
6.8591152274782372], [2.6091147193713331, 3.8555079632644569, 9.1839276932835574, 3.6125326276563117]], 
[[8.4114253324005279, 7.5760643534199588, 2.7105195863876137, 6.5723807899333382], [6.4768943624853978, 
0.29796913032120287, 2.2482295722627068, 7.0270824162300194], [-0.32141741964399806, 0.57985522235766496, 
4.1063327248837362, 0.080919381319258576]]]])+(1.-msk_ref)*numpy.array([[[[0.70723868548106505, 0.20445294520369339, 
-7.6812796176236526, -1.4531118757745078], [0.9545542728434695, -7.7193587702212083, 0.0088637904078510132, 
-2.9243009658748136], [-6.2072184465847746, -7.0872629561829159, 0.72042967281775905, 1.4059008175758159]], 
[[-1.4688467205417512, -1.5097077302032837, -6.8986063566310891, -1.6423974156531664], [-3.3833233139950014, 
-7.3817506884939954, -4.898469050369032, -4.3936332624735321], [-4.0393526930895307, -6.0896838341668067, 
-5.0100714059768805, -7.2832511059745046]]], [[[1.277462753185076, -7.2715207227103038, 0.71296475875574483, 
-4.4136268269257641], [-0.56873302406694437, -0.11633395643927313, 1.2352925653073719, -2.6906873416027013], 
[-0.33913519232838496, -5.6717318575253293, 1.558162296279102, -2.9761632736005605]], [[-4.0533100880813038, 
-1.2315380253946389, -5.6293094571224849, -6.7137867125686101], [-7.918301090652446, 0.90977095923496876, 
-4.4761476696886477, -5.7805294317363769], [-0.91227894425368383, 1.8888167776723686, -4.7558439553879088, 
-5.9988841911901298]]], [[[-2.1642904908297877, -1.7853058522197456, -5.0601116399650259, -2.127195217031268], 
[-2.4553213513188004, -6.4916776745630065, -6.778911770541554, -0.41828226221182874], [-4.6682827703187328, 
-3.4218895264256091, 1.9065302035934915, -3.6648648620337543]], [[1.134027842710462, 0.29866686372989282, 
-4.5668779033024522, -0.70501669975672776], [-0.8005031272046681, -6.9794283593688631, -5.0291679174273591, 
-0.250315073460047], [-7.598814909334064, -6.697542267332401, -3.1710647648063297, -7.1964781083708074]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank1_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.57185536765716005, 
-4.5016440600070959])+(1.-msk_arg0)*numpy.array([-0.4418100919929735, 1.7838290839713755])
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(4.01685432532)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-3.4449989576654145, 
-8.5184983853296714])+(1.-msk_ref)*numpy.array([-4.4586644173155481, -2.2330252413511991])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank1_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-4.1734209340603439, 
4.5527582003296185])+(1.-msk_arg0)*numpy.array([-1.7000682822887789, 0.76683988376374757])
      arg1=Symbol(shape=(2,))
      res=arg0-arg1
      s1=numpy.array([-1.5016152385157842, 0.80809700227400683])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-2.6718056955445597, 
3.7446611980556117])+(1.-msk_ref)*numpy.array([-0.19845304377299477, -0.041257118510259261])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank2_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[3.3500126396534871, 3.4943903203535527, -1.7005861531401179, 1.4952347206139418, 
-4.5979578172283739], [-2.3055331093587372, -3.6474162865795225, -3.0632961186256935, 4.7258384683418715, 
-0.58388337502415943], [4.7641302227265427, -0.11182220465882864, 2.8628458472454756, 1.6967713595739653, 
2.8474759788446562], [2.5863322473986914, 1.6349340161801535, -2.9934700314340712, 3.4068691472223609, 
-0.97913156666695667]])+(1.-msk_arg0)*numpy.array([[-0.34407378508566389, 2.6789454460601672, -3.3795587578901665, 
-4.1659261688389009, 2.3147542825953309], [-2.0615148857755603, -2.1181768528675784, 4.7855957803525566, 
2.4248630846228734, 4.4597452365342818], [4.5985091304874572, 2.9992334161018466, 0.73974708846994552, 
-0.24440017509511858, -0.49166350583553875], [1.5878740787090537, 3.0210382196579779, 3.6343442933400869, 
1.5494651243470852, -3.3635312675197349]])
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(-3.53998589595)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[6.8899985356048354, 7.0343762163049011, 1.8393997428112305, 5.0352206165652902, 
-1.0579719212770256], [1.2344527865926112, -0.10743039062817417, 0.47668977732565487, 8.2658243642932199, 
2.9561025209271889], [8.3041161186778911, 3.4281636912925197, 6.402831743196824, 5.2367572555253137, 
6.3874618747960046], [6.1263181433500398, 5.1749199121315019, 0.54651586451727718, 6.9468550431737093, 
2.5608543292843917]])+(1.-msk_ref)*numpy.array([[3.1959121108656845, 6.2189313420115155, 0.16042713806118192, 
-0.6259402728875525, 5.8547401785466793], [1.4784710101757881, 1.42180904308377, 8.325581676303905, 5.9648489805742217, 
7.9997311324856302], [8.1384950264388056, 6.539219312053195, 4.2797329844212939, 3.2955857208562298, 
3.0483223901158096], [5.1278599746604021, 6.5610241156093263, 7.1743301892914353, 5.0894510202984335, 
0.17645462843161352]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank2_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[2.0867998826855514, 2.311725929216629, -3.4719596731221403, 1.7817832811577139, 
1.5141982978301929], [3.1010865709749673, -2.1704923524391537, 3.7204405507466163, 4.629811066660821, 
1.6635344950905893], [-2.574527711983543, -1.6203338172344193, 3.7119433126415871, -4.2495237660622687, 
-2.1154248806831588], [0.14708606411584846, -4.3739162090051034, 0.28212084215683131, -3.2454357930486841, 
4.0490170686662843]])+(1.-msk_arg0)*numpy.array([[2.5664289274057825, -3.8876267007915413, 3.0606117231617365, 
0.45673258502894409, -2.4098041248367421], [2.4831763479900646, -4.4003484897067748, -3.6829879581883054, 
-4.4939546625771341, 0.095882545889256932], [4.8837698588887477, 4.688682977288769, -1.7221295814057069, 
-2.8466915452782313, -1.7320653753684723], [0.9117963691890596, -0.77307239329958843, -4.179217925450148, 
-2.8549317288801146, 2.3840070557997581]])
      arg1=Symbol(shape=(4, 5))
      res=arg0-arg1
      s1=numpy.array([[3.235357796595256, -1.480870361715898, 4.6623764990451662, -0.75539625063053251, 
1.764045470290668], [0.076611613190003425, -4.3375283365611939, -0.16726979393963415, 3.2199460507232871, 
-3.1622960810628884], [0.33570324662007156, 1.8340432936621101, 3.3105489280357343, -4.5476113052695135, 
1.6510039686145541], [1.9731991965232831, -1.2055959073743616, 3.1802938969891557, 1.6969195493915894, 
4.7202410276309497]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[-1.1485579139097046, 3.792596290932527, -8.1343361721673055, 2.5371795317882464, 
-0.24984717246047516], [3.0244749577849639, 2.1670359841220401, 3.8877103446862504, 1.4098650159375339, 
4.8258305761534777], [-2.9102309586036146, -3.4543771108965293, 0.40139438460585275, 0.29808753920724484, 
-3.7664288492977129], [-1.8261131324074347, -3.1683203016307417, -2.8981730548323243, -4.9423553424402735, 
-0.67122395896466536]])+(1.-msk_ref)*numpy.array([[-0.66892886918947347, -2.4067563390756432, -1.6017647758834297, 
1.2121288356594766, -4.1738495951274102], [2.4065647348000612, -0.062820153145580981, -3.5157181642486712, 
-7.7139007133004212, 3.2581786269521453], [4.5480666122686761, 2.854639683626659, -5.0326785094414408, 
1.7009197599912822, -3.3830693439830264], [-1.0614028273342235, 0.43252351407477319, -7.3595118224393037, 
-4.5518512782717035, -2.3362339718311915]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank3_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-4.4842044121065818, 4.4928465492805714], [-0.8488426741388917, 
-4.6899835951885827]], [[-3.8520901512994188, -4.2130527577972394], [-1.9136871018997637, -1.6373131250552273]], 
[[-4.193784678832186, -0.65999502399047039], [-0.57202165743856082, -2.5346604397115016]], [[-3.7518661140733869, 
-4.2333127035460327], [-1.185325910878734, 0.27013359391463077]], [[4.7175093778110231, -2.5123249429723304], 
[3.8200721640724424, 4.5930030128887935]], [[-4.7065467532244636, -1.7055937731234607], [-1.331474083968188, 
-4.3474657122786562]]])+(1.-msk_arg0)*numpy.array([[[3.7753100517410338, 3.7909180149825001], [0.99488027773324017, 
-3.0286171370440904]], [[3.6374742681576677, 1.145681069564648], [-0.002113196116643401, -0.3884450840387661]], 
[[-2.5595107128502024, -3.4455619811895488], [-4.5771852308072871, 1.3642969267838581]], [[-0.18445690526205638, 
0.49675060587106934], [-3.8670923300147821, -4.1547783162827496]], [[-3.9389053222472947, 1.3272580810242838], 
[3.5990473792265725, 2.2029039321052881]], [[-4.5403710311302428, -0.87738273585574866], [3.383530992594503, 
3.353845130538776]]])
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(-4.30522721091)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-0.17897720120150851, 8.7980737601856447], [3.4563845367661816, 
-0.38475638428350933]], [[0.45313705960565454, 0.092174453107833898], [2.3915401090053097, 2.667914085849846]], 
[[0.11144253207288735, 3.6452321869146029], [3.7332055534665125, 1.7705667711935718]], [[0.55336109683168644, 
0.071914507359040591], [3.1199013000263394, 4.5753608048197041]], [[9.0227365887160964, 1.7929022679327429], 
[8.1252993749775158, 8.8982302237938669]], [[-0.40131954231939027, 2.5996334377816126], [2.9737531269368853, 
-0.042238501373582871]]])+(1.-msk_ref)*numpy.array([[[8.0805372626461072, 8.0961452258875735], [5.3001074886383135, 
1.2766100738609829]], [[7.942701479062741, 5.4509082804697213], [4.3031140147884299, 3.9167821268663072]], 
[[1.7457164980548709, 0.8596652297155245], [-0.2719580199022138, 5.6695241376889314]], [[4.1207703056430169, 
4.8019778167761427], [0.43813488089029118, 0.15044889462232369]], [[0.36632188865777859, 5.6324852919293571], 
[7.9042745901316458, 6.5081311430103614]], [[-0.23514382022516944, 3.4278444750493247], [7.6887582034995763, 
7.6590723414438493]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank3_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[2.0747863514072602, 3.5162387004014164], [0.60187542749819123, 
-1.4464372144424988]], [[-0.79722414010768361, -3.2668034589151995], [3.6997621801618905, -2.3676159172001379]], 
[[-4.2988464154229238, 2.9969641407457024], [2.6364527127840898, 1.1108844479920323]], [[-1.1644293703177389, 
4.7657018637717314], [3.954999187366635, -3.1405571932690379]], [[1.9169748083366978, 3.5980880196573022], 
[4.6505164496107305, -3.4906561986190576]], [[-4.799933282554286, 3.4274402953401566], [-3.2690807817244751, 
3.3152263479112811]]])+(1.-msk_arg0)*numpy.array([[[-0.19327777478115316, 1.1303927856512574], [0.070720562815962396, 
0.2691348667587512]], [[-2.2293468476711373, 4.4261476420776269], [-2.1677478226049174, 3.9963032240053238]], 
[[-3.0163006244468358, 0.039611843610902753], [-1.7062014469652445, -0.85393662791937786]], [[2.3270289055995583, 
1.3694144393292564], [1.7400166142803206, 1.0276232159123833]], [[0.34573570990013103, -3.0575470737366093], 
[-0.16902625990476849, -2.0541180978179363]], [[1.4322948650591076, 1.3766358910177399], [-1.2248059444270067, 
3.8606015627846109]]])
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0-arg1
      s1=numpy.array([[[-0.065448649070245501, -1.0082288113108073], [2.940556380714975, 2.6943538184190166]], 
[[4.4012174975531551, 2.1536331083101858], [0.42620647125632161, -3.3632985397458972]], [[0.7816306898353016, 
-4.1519556164933835], [4.587159382474109, 3.7392943755961223]], [[1.0899221203445091, -3.3294088440228276], 
[4.0864323956812836, -1.4550107947057112]], [[3.7465493005246273, 3.1852429656365171], [-3.8082443104157484, 
-2.7860725384289688]], [[-4.7267420836233232, 0.9545260667209563], [-4.6866116848499395, 0.18931611034152862]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[2.1402350004775057, 4.5244675117122242], [-2.3386809532167838, -4.1407910328615154]], 
[[-5.1984416376608387, -5.4204365672253854], [3.2735557089055689, 0.99568262254575934]], [[-5.0804771052582254, 
7.1489197572390859], [-1.9507066696900193, -2.62840992760409]], [[-2.2543514906622479, 8.095110707794559], 
[-0.13143320831464855, -1.6855463985633268]], [[-1.8295744921879296, 0.41284505402078508], [8.4587607600264789, 
-0.70458366019008878]], [[-0.07319119893096282, 2.4729142286192003], [1.4175309031254644, 
3.1259102375697525]]])+(1.-msk_ref)*numpy.array([[[-0.12782912571090765, 2.1386215969620648], [-2.8698358178990127, 
-2.4252189516602654]], [[-6.6305643452242924, 2.2725145337674411], [-2.593954293861239, 7.359601763751221]], 
[[-3.7979313142821374, 4.1915674601042863], [-6.2933608294393535, -4.5932310035155002]], [[1.2371067852550492, 
4.698823283352084], [-2.346415781400963, 2.4826340106180944]], [[-3.4008135906244963, -6.2427900393731264], 
[3.6392180505109799, 0.73195444061103254]], [[6.1590369486824308, 0.42210982429678356], [3.4618057404229328, 
3.6712854524430822]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank4_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-2.3915780501354655, -1.5713847750997409, -2.0965030994469647, 
-2.1551686098873679], [3.3436095870548073, 3.0694003316976595, -0.81504636610563619, -2.4693923873770118], 
[-2.319226534159692, -3.4872612814422221, -0.52829274680925575, 0.30144897282215677]], [[-4.8403842893778295, 
2.5434758702608882, -0.65369410807785577, 2.38720324090216], [-4.436083252026509, -1.9729315005424199, 
2.1429634466708958, -4.4793507786187812], [-4.6789538924900809, -1.2258958461641156, 4.6704546977809684, 
-4.1766942543918724]]], [[[-0.99705682132654605, 0.63139391938552247, 3.648090293364568, 0.9177900253507385], 
[-2.6400178360936186, -4.9693099498708015, -2.0831389743339668, -4.8827387011810046], [0.92728596167706101, 
-0.70240094092455596, -4.2968994749977032, -1.8662843872126853]], [[4.4680529533242801, -0.62442225881816338, 
3.3149308543703011, -4.9724764458198436], [-2.5558868654124858, 0.54392866523319405, -4.0956874823606899, 
-3.1974475782258649], [-1.1195425378333166, 1.4289844369376148, -1.2628376022909036, -2.0331849636301413]]], 
[[[0.15561155815826755, 1.0566531457111576, -1.6429274914523804, -3.1609988875067607], [-3.9169786589191657, 
-0.0862972748194899, -2.2849500782464229, -3.593674280376046], [4.5374768908066745, 1.9914470295786906, 
-2.6711697877422749, -4.6476473568202383]], [[4.7467779338409635, 1.4258515187294085, 1.3579719892502204, 
3.8840821581627232], [-0.74410865296963191, -4.1369032807050301, 0.10505268333553275, 0.20183690831668066], 
[3.2258506139241625, -0.19145900822910011, -1.9876999864961387, 
-1.118642852906222]]]])+(1.-msk_arg0)*numpy.array([[[[-4.1006205365267823, 0.98436034016399709, -2.0659912052394889, 
-4.829130675843226], [-3.2904658009023189, 1.8958877020661227, -4.4668040074389035, 4.7329845357071445], 
[-1.7781671949524789, -0.77841298536212911, -2.7909279205249824, -3.9295744077558559]], [[-1.3638768752460773, 
4.4638784666445588, 2.5119633402011221, 3.4658456524464523], [-2.3171958056993116, -2.6568253529176222, 
2.2419660036743707, -1.9675169728612851], [-1.1143935824519682, 1.186871291556292, 1.8459225649295181, 
3.4239497313955702]]], [[[3.2057908235968178, 1.0054526017670806, 2.8530443499731035, 2.9117538931912392], 
[3.7690204040343271, -0.2740720613800427, -1.6755030321386211, -4.0975186439069002], [1.5575983081276483, 
-1.4138189638564769, -4.5799310621318954, -2.1831715499767892]], [[4.2307751851678876, -4.574159194957832, 
3.6142529396906227, 2.9695212799242778], [1.1073212833019772, 3.5565535386979832, 4.1163170572300576, 
-2.6051576587226011], [-2.4885332002171778, 2.7193644049811461, 3.6891018981647203, 2.2279362582877482]]], 
[[[3.8371440925068896, 0.098907118846149444, -4.8484985466419248, -3.2646614116360695], [-1.3166337092696869, 
3.8989945382792683, 2.4522596196795661, 4.8579102565531542], [2.8065577922030176, -2.6140964300168221, 
0.26485552888380326, -3.2455906809923696]], [[3.4179005303801677, -4.9507538637080364, -3.9286015574556798, 
0.67686821786057827], [1.2296342635912527, -1.0149250475477691, 1.9729311750755993, -4.6474538783990385], 
[-1.2276432760256037, 2.5170369379074629, 0.97453825295943464, 3.8596939709877667]]]])
      arg1=Symbol(shape=())
      res=arg0-arg1
      s1=numpy.array(3.09580908291)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-5.4873871330441668, -4.6671938580084422, -5.1923121823556659, -5.2509776927960692], 
[0.24780050414610599, -0.026408751211041803, -3.9108554490143375, -5.5652014702857127], [-5.4150356170683933, 
-6.5830703643509239, -3.624101829717957, -2.7943601100865445]], [[-7.9361933722865308, -0.5523332126478131, 
-3.749503190986557, -0.70860584200654131], [-7.5318923349352103, -5.0687405834511212, -0.95284563623780549, 
-7.5751598615274824], [-7.7747629753987821, -4.3217049290728173, 1.5746456148722672, -7.2725033373005736]]], 
[[[-4.0928659042352473, -2.4644151635231788, 0.55228121045586676, -2.1780190575579628], [-5.7358269190023199, 
-8.0651190327795028, -5.1789480572426676, -7.9785477840897059], [-2.1685231212316403, -3.7982100238332572, 
-7.3927085579064045, -4.9620934701213866]], [[1.3722438704155788, -3.7202313417268646, 0.21912177146159983, 
-8.0682855287285449], [-5.651695948321187, -2.5518804176755072, -7.1914965652693912, -6.2932566611345662], 
[-4.2153516207420179, -1.6668246459710865, -4.3586466851996049, -5.1289940465388426]]], [[[-2.9401975247504337, 
-2.0391559371975436, -4.7387365743610816, -6.256807970415462], [-7.012787741827867, -3.1821063577281912, 
-5.3807591611551242, -6.6894833632847472], [1.4416678078979732, -1.1043620533300107, -5.7669788706509761, 
-7.7434564397289396]], [[1.6509688509322622, -1.6699575641792928, -1.7378370936584808, 0.7882730752540219], 
[-3.8399177358783332, -7.2327123636137314, -2.9907563995731685, -2.8939721745920206], [0.13004153101546123, 
-3.2872680911378014, -5.0835090694048404, -4.2144519358149228]]]])+(1.-msk_ref)*numpy.array([[[[-7.1964296194354835, 
-2.1114487427447042, -5.1618002881481901, -7.9249397587519272], [-6.3862748838110202, -1.1999213808425786, 
-7.5626130903476048, 1.6371754527984432], [-4.8739762778611802, -3.8742220682708304, -5.8867370034336837, 
-7.0253834906645576]], [[-4.459685958154779, 1.3680693837358575, -0.58384574270757916, 0.37003656953775099], 
[-5.4130048886080129, -5.7526344358263231, -0.85384307923433056, -5.063326055769986], [-4.2102026653606694, 
-1.9089377913524093, -1.2498865179791832, 0.32814064848686897]]], [[[0.1099817406881165, -2.0903564811416206, 
-0.24276473293559775, -0.18405518971746204], [0.67321132112562587, -3.369881144288744, -4.7713121150473228, 
-7.1933277268156015], [-1.5382107747810529, -4.5096280467651777, -7.6757401450405967, -5.2789806328854905]], 
[[1.1349661022591864, -7.6699682778665332, 0.5184438567819214, -0.12628780298442344], [-1.9884877996067241, 
0.46074445578928191, 1.0205079743213563, -5.7009667416313023], [-5.5843422831258795, -0.3764446779275552, 
0.59329281525601907, -0.86787282462095305]]], [[[0.74133500959818832, -2.9969019640625518, -7.9443076295506261, 
-6.3604704945447708], [-4.4124427921783882, 0.80318545537056707, -0.6435494632291352, 1.762101173644453], 
[-0.28925129070568367, -5.7099055129255234, -2.830953554024898, -6.3413997639010713]], [[0.32209144747146645, 
-8.0465629466167385, -7.0244106403643816, -2.418940865048123], [-1.8661748193174486, -4.1107341304564704, 
-1.1228779078331019, -7.7432629613077397], [-4.323452358934305, -0.57877214500123841, -2.1212708299492666, 
0.76388488807906541]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_sub_overloaded_expandedData_rank4_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[4.7227008802641297, 2.8863279383915197, -2.1896969198324809, 4.4302312776864046], 
[-2.0197165879770251, -4.9702841021278301, 4.7902292515365534, 4.5956841306054539], [2.93934146754248, 
1.646059013328439, 0.99083597111362476, 3.3910212199665182]], [[1.2787313823171234, 1.3787912835332463, 
-0.071420926539018659, -3.2388329800378415], [3.349975825307892, -1.7816480803027677, 2.2965490165680036, 
-1.9018094630451774], [-1.1200280993595113, -0.66137198885848481, 0.21843601647543931, -1.4087126883569634]]], 
[[[-1.9454924644698748, 0.7634003631723596, 4.1323447887625271, 1.443298202567032], [-3.7655189569283309, 
0.99421618018940361, 4.6046246816957677, 0.74421638400845325], [-3.5622774770791485, 1.2397714154717114, 
-1.0582462113989424, 1.2971002709503896]], [[1.3379013865706302, -0.39532127021252883, 4.7111583084640927, 
-0.88268548014728054], [-4.8134271904014936, -3.1339427094271266, 2.1759548242117912, -1.5082214108365442], 
[-3.2684776044041417, -1.1500337944886265, -4.7387724227769104, -0.87161000308836023]]], [[[-0.77862807704750203, 
4.0600646358304022, -0.60854424079166769, -2.1879855865994302], [-3.1756330451114421, 2.3997197078889831, 
3.48971886859092, 1.3336201149028941], [-3.1489389998309738, -3.126335897832373, 4.4951625025311497, 
4.8070472911835367]], [[-0.48575660505239959, 2.6019068715787999, 2.0846243088005867, -4.1696787529733781], 
[4.548381761398181, 2.8671236182352331, -2.4623488980316131, -2.7420314450035965], [-2.6608024266156169, 
1.9100775920001078, -4.7415568715485605, 4.335606352714283]]]])+(1.-msk_arg0)*numpy.array([[[[-0.066813883761502879, 
2.3352842907581648, 1.9822821047204409, 2.2488083940398429], [-3.5461097023132315, 3.9394596449218593, 
-3.1392993492194812, -1.195903516669877], [-2.4035485197244641, -1.2564828050011667, 4.2913267573861269, 
3.3717435374804552]], [[2.0596631449037996, 1.766648199074103, 3.7051853214349446, 2.3190764926321314], 
[2.0765914923551589, -3.0883600344375619, 3.3829116246346622, -0.77543432130851642], [-1.0720358718139797, 
1.757742252759785, 0.37504357425332913, 2.5917331896567894]]], [[[-4.2390464570087687, 0.22513794491672012, 
2.4848124269039893, 0.40807584912367822], [3.4683686407549388, 0.16480295091112396, 2.9378502257107231, 
-4.5549662855752659], [-3.9215318218115671, -0.029245135802563382, -2.553738608483358, 2.451028936659565]], 
[[-3.607109515954888, -3.993893382807582, 0.073725334847121182, -2.1709804092290019], [0.33344114744651687, 
-0.58055248058879805, -2.3870396661749513, 4.2208499406342153], [2.5521039977169968, 0.99728976648084799, 
-2.2262756306598854, -1.1032924331956737]]], [[[4.966005471135805, -2.2223123277417969, 3.9211131181159882, 
1.2020059492898092], [-1.9322416802383922, 1.9213246599314813, -4.9244981020217091, 3.9198631295221293], 
[0.441626539964318, -4.8403835386932315, 0.067761940060105275, -3.3198352912434692]], [[3.7289362256010783, 
4.4966066758919183, 4.7606849535179343, -4.2473703353049705], [-4.1395576165491477, 1.394341710895656, 
1.1616039705755199, 3.875922408907579], [3.2845443210135308, 4.7114056834057489, 1.3775615753886044, 
-3.9913085015805105]]]])
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0-arg1
      s1=numpy.array([[[[-1.3801587913318047, -1.0654264899468902, 0.63456255053258914, 0.019001978473544234], 
[2.9683175293448718, 3.6853848061948291, -3.378954186340446, 0.041808892524110242], [-0.43021730340079323, 
3.8311628728062068, -2.1793361115232388, 3.1444277139479642]], [[4.3924534836876852, 4.6258828194696662, 
-1.337885612990688, 4.1259618924749724], [-4.4897704263159222, 0.15369009386017485, 2.6962829695946411, 
2.0341337355139855], [-2.3651800760864128, 2.5627722574362446, 4.595047097795911, -4.3290029697233141]]], 
[[[4.178160967558556, -4.7760981608958755, -1.9186656311572339, 1.2206352911034513], [-0.085772738154105355, 
-1.3074976307569877, -1.956712770838227, 3.2327542991796854], [1.1038448851539249, 3.3037436446982742, 
-0.27900134960503031, -0.91401651221828928]], [[-2.3182986051061674, -1.849076104179066, 1.7255801749968178, 
-4.800626013966669], [-1.065938515473257, -1.7027357609246607, -4.5450782032387052, 4.2110109611691939], 
[0.42483398312984555, -2.0300778021420198, -1.8600131931716146, -0.65565335271777681]]], [[[1.5409796879232402, 
3.8526060616448135, 4.5046749363240419, -2.3425879056067043], [-2.0020629040094438, 4.0201378578644089, 
-0.91649358299543415, -4.6048272916989044], [2.9771669103984353, -0.18712141889004474, 4.7604346415721732, 
3.1208517541066438]], [[-2.0397258049700016, 4.8476984980922424, -0.83670507076672873, -0.65429392708528322], 
[1.8648243434781451, -3.0663012456143979, -0.46932892583376429, 1.7337690374943984], [-2.830192293285001, 
2.4237706377557764, -4.5545542759999593, -1.4623880986500994]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[6.1028596715959349, 3.9517544283384098, -2.8242594703650701, 4.4112292992128603], 
[-4.9880341173218969, -8.6556689083226601, 8.169183437876999, 4.5538752380813436], [3.3695587709432733, 
-2.1851038594777679, 3.1701720826368636, 0.24659350601855401]], [[-3.1137221013705618, -3.2470915359364199, 
1.2664646864516693, -7.3647948725128138], [7.8397462516238141, -1.9353381741629425, -0.3997339530266375, 
-3.9359431985591629], [1.2451519767269015, -3.2241442462947294, -4.3766110813204717, 2.9202902813663507]]], 
[[[-6.1236534320284308, 5.5394985240682351, 6.051010419919761, 0.22266291146358075], [-3.6797462187742256, 
2.3017138109463913, 6.5613374525339943, -2.4885379151712321], [-4.6661223622330734, -2.0639722292265628, 
-0.77924486179391206, 2.2111167831686789]], [[3.6561999916767975, 1.4537548339665372, 2.9855781334672749, 
3.9179405338193884], [-3.7474886749282366, -1.4312069485024659, 6.7210330274504964, -5.7192323720057381], 
[-3.6933115875339873, 0.8800440076533933, -2.8787592296052957, -0.21595665037058343]]], [[[-2.3196077649707423, 
0.20745857418558877, -5.1132191771157096, 0.15460231900727406], [-1.1735701411019983, -1.6204181499754258, 
4.4062124515863541, 5.9384474066017985], [-6.1261059102294091, -2.9392144789423282, -0.26527213904102354, 
1.6861955370768928]], [[1.553969199917602, -2.2457916265134426, 2.9213293795673154, -3.5153848258880949], 
[2.6835574179200359, 5.9334248638496305, -1.9930199721978488, -4.4758004824979949], [0.16938986666938405, 
-0.51369304575566854, -0.18700259554860121, 5.7979944513643824]]]])+(1.-msk_ref)*numpy.array([[[[1.3133449075703019, 
3.4007107807050549, 1.3477195541878517, 2.2298064155662987], [-6.5144272316581038, 0.25407483872703018, 
0.2396548371209648, -1.2377124091939873], [-1.9733312163236709, -5.0876456778073731, 6.4706628689093657, 
0.22731582353249102]], [[-2.3327903387838855, -2.8592346203955632, 5.0430709344256321, -1.8068853998428409], 
[6.5663619186710811, -3.2420501282977368, 0.68662865504002113, -2.8095680568225019], [1.293144204272433, 
-0.80503000467645958, -4.2200035235425819, 6.9207361593801036]]], [[[-8.4172074245673247, 5.0012361058125956, 
4.4034780580612232, -0.81255944197977303], [3.5541413789090441, 1.4723005816681116, 4.8945629965489506, 
-7.7877205847549513], [-5.025376706965492, -3.3329887805008376, -2.2747372588783277, 3.3650454488778543]], 
[[-1.2888109108487207, -2.144817278628516, -1.6518548401496966, 2.6296456047376671], [1.3993796629197739, 
1.1221832803358627, 2.1580385370637538, 0.0098389794650213247], [2.1272700145871513, 3.0273675686228678, 
-0.36626243748827081, -0.44763908047789691]]], [[[3.4250257832125648, -6.0749183893866103, -0.58356181820805375, 
3.5445938548965135], [0.069821223771051599, -2.0988131979329276, -4.008004519026275, 8.5246904212210346], 
[-2.5355403704341173, -4.6532621198031867, -4.6926727015120679, -6.4406870453501135]], [[5.7686620305710798, 
-0.35109182220032409, 5.597390024284663, -3.5930764082196873], [-6.0043819600272927, 4.4606429565100534, 
1.6309328964092842, 2.1421533714131806], [6.1147366142985318, 2.2876350456499726, 5.9321158513885637, 
-2.5289204029304111]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_Symbol_rank0(self):
      arg0=Data(4.62465145684,self.functionspace)
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(-4.83968289402)
      sub=res.substitute({arg1:s1})
      ref=Data(-22.3818465465,self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_Symbol_rank1(self):
      arg0=Data(3.61282962415,self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0*arg1
      s1=numpy.array([-1.5808963858957537, 2.6509696096021864])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-5.7115092956749542, 9.5775015382906385]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_Symbol_rank2(self):
      arg0=Data(1.49025589385,self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0*arg1
      s1=numpy.array([[4.1668790679299175, -2.8153908003971773, 3.9030969110369043, -0.17962827030748763, 
2.1351246321738504], [-3.6496148227091672, 1.387147904486504, -1.1823464175604426, 1.1752089388333173, 
-1.7608073918316345], [1.1775740286573697, 1.5139307880954602, -4.5926484431964676, -1.3565377974967943, 
2.5163302873137585], [-4.9066894975281929, -0.71152046468118701, 3.1503673258369584, -1.582645414456433, 
4.0919813724626888]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[6.2097160899280084, -4.1956527337729987, 5.8166131759266637, -0.26769208852717874, 
3.1818820671938379], [-5.4388599998117435, 2.0672053402977806, -1.7619987173376985, 1.7513620475973966, 
-2.6240535936055083], [1.7548866366471667, 2.2561442798348774, -6.8442214108383101, -2.0215884479450947, 
3.7499760415336878], [-7.3122229429659207, -1.0603475660835118, 4.6948534751098423, -2.3585466567627744, 
6.0980993578224529]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_Symbol_rank3(self):
      arg0=Data(-4.8189372207,self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0*arg1
      s1=numpy.array([[[-2.1082618280451539, -1.3955382894516588], [-3.4395290563777836, 4.8101517504101992]], 
[[-2.0990702249254642, -0.19805771431177988], [3.5892783984855807, 2.3466998684539329]], [[1.0198837335276068, 
1.3499161645518498], [2.1694219720539518, -4.6399343230104897]], [[0.39865646301668001, -4.6889954108117458], 
[-1.8077039797673278, -2.0729006873515732]], [[-1.4351307436857752, -0.27135779329323384], [-0.11640854123168598, 
3.5462285443877146]], [[-4.7284739075521287, -2.9968208233684992], [-0.63458679250348737, -0.3430241153835798]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[10.159581394141497, 6.7250114059464261], [16.574874591447742, -23.179819307252547]], 
[[10.11528763575015, 0.95442769134320948], [-17.296507269905895, -11.308599341897418]], [[-4.9147556842796076, 
-6.5051612501794489], [-10.454308288528681, 22.359552210744798]], [[-1.9211004679024974, 22.595974512838154], 
[8.7112119921028786, 9.9891782770868964]], [[6.9158049573139522, 1.307656170226968], [0.56096545214841331, 
-17.089052725648113]], [[22.786218910197551, 14.441491409490299], [3.058033914157781, 
1.6530116772185961]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank0_Symbol_rank4(self):
      arg0=Data(-0.0961090764765,self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0*arg1
      s1=numpy.array([[[[-1.3508458008250446, -1.7001494994927766, -1.1781435318758104, 0.94544141796815939], 
[0.48819490611616523, -1.7587019774587032, -3.7925425828521075, 4.7932646338579481], [0.58005604545753187, 
-1.1056764808385546, -0.51753380988709718, 0.78769257503235046]], [[2.2158543714442409, 2.8083791437737293, 
4.2821497276046312, 3.557157624087198], [1.3939326777613186, 1.1701609581270382, 0.2011137853628675, 
2.0268107124192767], [4.9657490768337222, -0.36506867019444478, 1.9579179042357691, 2.3678413818675184]]], 
[[[4.6392916421533705, 4.4042259724661363, -0.16399061756687949, -2.3651374585729892], [-3.2439831086523521, 
-0.75751658111162179, 0.90117784493079078, 3.4349562957397168], [2.5698732564560274, 2.3597838208145365, 
-4.9703254679114597, 3.3611480514859409]], [[3.5875371307296735, 3.7296897049486226, 0.57748787491421005, 
-3.5628770219331427], [-2.1948129576712505, 4.1072354163079758, 3.4869674239178412, 1.087096416125072], 
[-1.8854495462953502, -0.47642374646276409, -4.0405955013288795, 2.2924875059879755]]], [[[4.5285297494026793, 
-2.5139281223909449, 2.208483948920561, -2.7970423581490991], [-1.7523437586040416, 1.8001610169726279, 
-2.0370614441012389, -1.7571467006380495], [-4.7379091083725093, -1.7199833882948159, -1.9229770651511577, 
1.983783387985067]], [[3.6886437141901496, 1.3947955105400336, -1.5553692753039892, -3.1732704417249646], 
[-3.1233013224428783, -0.18349036348379588, -2.6033177681431141, -4.8475794865627444], [0.44359684613617745, 
2.4458348840139719, -4.4593547045866213, 2.8568083531481321]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[0.12982854237943392, 0.16339979826821033, 0.11323028680532936, -0.090865501543539021], 
[-0.046919961567349337, 0.16902722285092533, 0.36449776513566223, -0.4606762372674883], [-0.055748650833525863, 
0.10626554545516434, 0.049739696513606095, -0.075704405933744209]], [[-0.21296371724588994, -0.26991072590391707, 
-0.41155345565411638, -0.34187513413231108], [-0.13396958233003523, -0.11246308901442943, -0.019328860177915387, 
-0.19479490576326494], [-0.47725355778845102, 0.035086412742886867, -0.18817368159287634, -0.22757104845409318]]], 
[[[-0.44587803523243974, -0.42328609080747298, 0.015760986805161359, 0.2273111768833927], [0.31177622067789706, 
0.072804219026262926, -0.086611370417368036, -0.33013047732063483], [-0.24698814533960797, -0.22679664370263819, 
0.4776933905085276, -0.32303683512905357]], [[-0.34479488045953066, -0.35845703308646898, -0.055501826334373089, 
0.34242482017728643], [0.21094144640040824, -0.39474260273287426, -0.33512921881633451, -0.10447983259467819], 
[0.18120881463745528, 0.045788646284003677, 0.38833790204776181, -0.22032885703438657]]], [[[-0.43523281201138325, 
0.2416113101712597, -0.21225535274389759, 0.26882115790732197], [0.16841614030876842, -0.17301181285021075, 
0.19578009411842659, 0.16887774663202701], [0.45535606883521218, 0.16530601500391165, 0.18481554981714085, 
-0.19065958934863894]], [[-0.35451214082160976, -0.13405250839155117, 0.1494851046293674, 0.30498009156431655], 
[0.30017760565777185, 0.017635089376762319, 0.25020246647106092, 0.46589638759990259], [-0.042633683210029791, 
-0.23506693191655562, 0.42858446233889236, -0.27456521249137733]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([0.27659058494851418, 4.1541405281225892]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(3.8719303416)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([1.0709394780619239, 16.08454275409127]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([3.7426337922420245, 3.9291817340183464]),self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0*arg1
      s1=numpy.array([0.24137850302835329, -1.7566967446888535])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([0.90339134215470884, -6.9023807614409334]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[-4.5985565783651303, 1.9500350746471993, 4.9713294753194841, -4.1975843272554876, 
-2.7203349447146974], [0.6454726334120906, 4.0369461924736108, -0.05835204361358759, -4.4624467830414529, 
-3.7659400185729366], [2.0977327530498648, 0.21380148281940414, -2.0069737126558609, 2.3624658088422841, 
4.2025541873980536], [3.414052245462365, -0.88493614463302261, -1.2061553540091854, 4.49473416391168, 
2.5601548526811913]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(1.33330352327)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-6.1312716879061586, 2.5999886355337587, 6.6282911047962196, -5.5966539727668181, 
-3.6270321662717726], [0.86061093630488039, 5.3824745816900785, -0.077800985340197651, -5.9497960182490823, 
-5.021141095199475], [2.7969144705273297, 0.2850622703241823, -2.6759051222010943, 3.1498839865422879, 
5.603280304805077], [4.5519678875143299, -1.179888479511161, -1.608171183115485, 5.9928448969206256, 
3.4134634852052463]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[-0.80021427988594684, -1.1537754790930244, 2.8892977581295964, 3.7382841198075329, 
2.3764815828521835], [0.29503112693550992, -2.1167484120073929, -3.8919266120092, 1.3224523748721193, 
2.9631976530881987], [-2.8104707825838995, -2.0632038633268568, -3.0589018293825343, -1.678440560478224, 
-4.5543583893018766], [-0.82692491278534597, 3.2768573560743448, 3.9160327845735843, 2.3191381875549339, 
4.4585962073227758]]),self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0*arg1
      s1=numpy.array([[0.84019739061084842, -0.26041818053762533, -4.1862385457734819, 2.5897362481178323, 
1.3358121377857461], [-2.9413855411668366, -0.72774497299829122, -3.5296778480396993, -2.4701281082949977, 
1.8361878086654535], [2.766871279727253, -4.8815007323834605, -2.5917825490500368, -4.9262737607479554, 
-2.1516904953173679], [1.8198771262398861, -3.8256760862124253, 4.2065485224111754, 2.1262330288097564, 
0.59770635028954899]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-0.67233794988971163, 0.3004641110143324, -12.095289645299022, 9.6811698908288335, 
3.174532943598229], [-0.86780029096226652, 1.5404530159404959, 13.737247148605071, -3.2666267830530953, 
5.4409874052666343], [-7.7762108908439682, 10.071531169886436, 7.9280083806508852, 8.2684576920589663, 
9.799569658529764], [-1.5049017338959638, -12.536194825062895, 16.472981923661731, 4.9310282127532963, 
2.6649312664937215]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[-0.93968623179119604, 3.5541982898616844], [-4.4072293043240576, -3.6076482066197646]], 
[[1.7484917853830488, 4.9003868861327007], [4.7568448118670297, -2.9711396711199276]], [[3.8141950057192862, 
1.8792050658033421], [-3.9338669245816726, 1.8697569231208337]], [[2.7319382834348493, 4.8419678958442489], 
[1.6265368297291136, 2.7028283208639312]], [[-2.4918167983349662, 0.66893189862375824], [-0.98140319773957252, 
-4.1758241797866216]], [[2.3073753559238792, -1.9496085178777891], [-0.68687199404203181, 
-1.8466377636332689]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(-4.7863046684)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[4.497624598049291, -17.011475867168713], [21.094342193976221, 17.267303453273268]], 
[[-8.3688143950302809, -23.454744630041624], [-22.767708529872653, 14.22077967833674]], [[-18.255899362045483, 
-8.9944479793272372], [18.828685625972383, -8.9492262898982791]], [[-13.075888959772939, -23.175133544101051], 
[-7.7851008214498698, -12.936559810022931]], [[11.926594374657277, -3.2017118692216404], [4.6972947069193065, 
19.986766766112023]], [[-11.04380143779947, 9.331420350662329], [3.2875786316735858, 
8.8385709489135635]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[-3.7703901363085679, -1.2567782536949124], [-4.6867702962008151, -4.0264117118029379]], 
[[4.9065305059289148, -2.9124024019871833], [2.1575041693932544, -2.447359134212459]], [[-4.2014241124001597, 
2.1567970243772496], [-0.21332683825097298, -4.1649402688134085]], [[2.4269573018600399, 1.862212593201459], 
[-0.8497946023648062, -0.85421311729337468]], [[-3.9387175807922148, 2.8138330952953154], [-3.5224045840887532, 
2.3328843983658345]], [[-4.1835084349073455, 1.5103476384019734], [3.5299839973193272, 
0.52130047189201001]]]),self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0*arg1
      s1=numpy.array([[[3.7025504227899226, 0.94126871779523835], [0.2494407260336402, -1.4801201322289135]], 
[[2.2922758002791603, -0.52083778427225891], [-3.8446937528722511, 4.7302904730413342]], [[4.5317616448777596, 
4.7075623194640848], [3.6913207941396351, -0.94890681756314521]], [[0.10066689821055874, -0.2379214158604368], 
[3.5670839439496831, 1.12875998069849]], [[-3.2904800400470879, 0.6205159319494804], [3.0069046365957437, 
-0.47921917980699202]], [[-2.06878107963567, -2.1195493051668968], [-1.6911454119040039, 1.930100136733504]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-13.960059593272241, -1.182966055408349], [-1.1690713854372303, 5.9595730352818101]], 
[[11.247121142072317, 1.5168892139602093], [-8.2949428018620797, -11.576719596675883]], [[-19.039852646439627, 
10.153256402690602], [-0.78745779398387905, 3.952140215820322]], [[0.24431426366771691, -0.44306025680762678], 
[-3.0312886817506057, -0.9642015817884666]], [[12.960271582979336, 1.7460282654774637], [-10.591534675862574, 
-1.1179629479694033]], [[8.6547630966325499, -3.2012562875353661], [-5.9697162411611355, 
1.0061621120780087]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[0.62243391611320931, 2.0870921150140633, -1.966483871868502, 2.0460162713266978], 
[-4.0678614422698969, -1.0483170813789053, -3.3875965478845105, -0.56278682367596566], [-3.1888119343249111, 
1.1466037700100813, -0.08422750542817603, 1.334817640609244]], [[0.94229536045356443, -0.46607285357294259, 
3.2913807414193705, 4.4571709007297127], [-4.8728928412169248, 1.2037128407211304, 1.7363887141819356, 
-1.8638393114984897], [0.2688971714658468, -1.0619718642775746, -1.5898347593643845, 4.0497830306245604]]], 
[[[-3.0176256202980776, 4.6481698965552418, 2.9546667018974437, -0.80203271850496982], [0.920018392214903, 
2.5683755474820238, -0.39881454964042007, -1.8042576364273657], [-2.3945875160036332, 4.1111523243175156, 
0.78082260002804471, -4.7775799631740909]], [[2.8847051100624723, 3.801654962351078, 1.7703173227715148, 
4.1273910274214991], [-0.89823810002448035, -0.13134866838888204, -2.9139897570290261, 3.0266778096414111], 
[-4.6565287368895367, 4.2608689122302597, 4.4884714229987637, 0.39909756290562726]]], [[[-3.0170682335317931, 
3.5429625779103553, -1.5481240066779933, 2.2741378666795491], [0.99402869902853208, -2.7818018223223495, 
-4.1871147291249109, -3.2256430704428585], [-1.1555881857999983, -1.3659488684092769, -3.6647624059065964, 
-4.1746014766224038]], [[-0.16412546559365726, 2.0500472712024926, 0.70515501953914139, 4.9173519977774696], 
[4.8984165381421718, 2.4090796502395673, -4.3641606786463338, -2.9169347489558826], [-2.5705157992905558, 
-2.1873999149378887, 0.30099797008132612, -0.40586512460845547]]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(-4.38997505647)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-2.7324693660399006, -9.1622823254735639, 8.6328151464594995, -8.9819603962624388], 
[17.857810264753809, 4.6020858385281453, 14.87146434660761, 2.470620118049252], [13.998804851470311, 
-5.033561950002353, 0.36975664789865109, -5.8598161472149455]], [[-4.13665312822154, 2.046048201684489, 
-14.449079356187196, -19.566869076641478], [21.39187802580896, -5.2842693459222208, -7.6227031436002131, 
8.1822080867524907], [-1.1804518754912556, 4.6620299948548602, 6.9793349375236682, -17.778446488570129]]], 
[[[13.247301202882928, -20.405349904126975, -12.970913121521617, 3.5209036287121829], [-4.0388577933199716, 
-11.2751045891017, 1.7507859250800237, 7.9206460193673678], [10.51217946579799, -18.047856157115579, 
-3.4277917376536409, 20.973456868640252]], [[-12.663783478454937, -16.689170458038671, -7.7716488890093069, 
-18.119143658691538], [3.9432428538813187, 0.5766173779281577, 12.79234234817573, -13.287040088306636], 
[20.442045004695569, -18.705108243592797, -19.704277588657195, -1.7520283462549344]]], [[[13.244854288882133, 
-15.55351734304433, 6.7962257736436893, -9.9834085097043221], [-4.3637611941537289, 12.212040612046716, 
18.381329219449761, 14.160492620329668], [5.0730033112170725, 5.9964814607344739, 16.088215549830551, 
18.326396353088406]], [[0.72050670008820039, -8.9996563851698248, -3.0956129467236795, -21.587052614141587], 
[-21.503926418659777, -10.575799573608803, 19.158556521698415, 12.805270789276145], [11.284500241155726, 
9.6026310651088593, -1.3213735807060782, 1.7817377733234934]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_constData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[0.92475193690629176, -2.8751640037436377, 2.3253579475929946, -1.4883270341411139], 
[0.99308590109939665, -4.571048057187971, 0.29582761355193554, 0.13789701080785477], [-4.8513899543422578, 
-3.6257472257559966, -0.35978952486470916, 3.2517553292871213]], [[1.4829500707237964, -2.4100404084555849, 
-1.3007935048236199, -1.9592333033911915], [-3.2571300156448268, 3.5809336502177818, 4.572845912197149, 
0.11594932874247199], [4.1148495529245857, -4.948814678027361, 0.20924193900359267, -1.1213985895158896]]], 
[[[-2.5470666325037494, 3.6152403556802923, 3.973428980280806, -2.5374743265932209], [-1.5104931521079923, 
1.9242528516494346, 1.2141691608491234, 4.0577747879052453], [3.8342739616484334, -3.0938013228729275, 
-0.013278763548063566, -0.42321566529779808]], [[2.6294282632341996, 3.3952741627917256, 2.4458864889971466, 
4.2648559736340506], [-3.7590978982471257, 1.9398108969769856, -0.72053544714256734, -2.2852607861733709], 
[-3.2806396751883415, 0.51560443280332802, 3.3391590838269085, -2.5163633463539634]]], [[[4.01150830599126, 
-0.63639401929707073, -3.1327871719882161, 3.2114179123865725], [2.4499611723235173, 2.7753873236361759, 
-4.685579988073755, -2.2124483061230249], [-1.592101861637012, -4.6991157531893437, 1.8862972499478818, 
-2.4723033101868044]], [[1.6629673331527997, -3.797885972234567, -2.7734235709739954, -0.64462102158809298], 
[-0.88743325197640388, 2.5092803298320216, 1.525489125478364, 2.2445055975567962], [-1.160115769838137, 
-1.3355312013577905, -1.0067006671660481, 3.6058946626271364]]]]),self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0*arg1
      s1=numpy.array([[[[1.1454670721774276, -0.011324883951625786, -3.6679325603974888, 0.094693748129400568], 
[-0.94166879518392843, -4.1783274612573384, -0.26913364710175003, -3.1060599945156477], [2.1413905443235972, 
3.4882436840724846, 2.2658723445074678, 2.1819015356595139]], [[-1.9800856525493749, 2.7480580905119893, 
-3.1758715478375219, 4.1124829257304629], [2.0216531202224841, 4.0515397505305888, 4.044920456333422, 
-2.3883049937748946], [1.5444958016374528, -1.2594632181245204, 3.5895692628257976, -1.8412492131846401]]], 
[[[-0.79280116164120962, -1.2215336421582137, -4.9961696612945055, -2.7749169673973384], [-0.23664202961414915, 
-3.5042031492270143, -4.0093512501017203, 0.23193992235689542], [3.1633334993556197, 1.6180559532355963, 
2.5015418045872986, 4.5068540065663516]], [[-4.1132185738946214, 4.1318631987586905, -4.2991786205876377, 
0.23269781735957817], [1.6926075754432439, -3.007700180988361, 3.1039787293459948, 3.355203306995362], 
[-4.1457163793408522, -1.2887202396242636, -2.7544192723911931, -4.0173861352686675]]], [[[1.4375090967135709, 
0.047062319770922123, -0.75634196908244178, -0.17805460405137996], [2.5278812537929483, -3.9412959574596886, 
-0.1516720332141368, -2.385490768218621], [2.0634420046107511, 3.7303717046166387, -3.7967716979561583, 
-0.36579638919369373]], [[-0.89677692789005903, -0.33159623329473931, -2.0783805922287799, 3.3237758621528677], 
[1.8764406996924805, 3.8567013916314448, 2.4876054261100879, -3.122046411865298], [-3.9505368448428069, 
-3.9474451391708176, 0.76222063661286921, 2.065165407462576]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[1.0592728936584552, 0.032560898684288464, -8.5292561305554226, -0.14093526530513639], 
[-0.93515800400241478, 19.099335624075504, -0.079617164548639499, -0.42831638863356958], [-10.388720575054998, 
-12.647489860286688, -0.81523713423442645, 7.0950099465605785]], [[-2.9363681584872698, -6.6229310429171893, 
4.1311530815811848, -8.0573135077187672], [-6.584787059098673, 14.508295027869941, 18.496797973906915, 
-0.27692236086049277], [6.3553678588617721, 6.2328500602902022, 0.75108843274136661, 2.0647742706124967]]], 
[[[2.0193173850265365, -4.4161377189515036, -19.851925322587327, 7.0412805631986641], [0.35744616523310901, 
-6.7429729026590115, -4.8680306428853894, 0.94115996924851042], [12.129087268589474, -5.0059436486027034, 
-0.033217382128710969, -1.9073712167890253]], [[-10.815413171058387, 14.028808362935354, -10.515302901880693, 
0.99242267641760229], [-6.3626775794058616, -5.8343695859208742, -2.2365267016703343, -7.6675145471157151], 
[13.600601636143761, -0.6644698681936374, -9.1974441340729562, 10.10920321894068]]], [[[5.7665796814044832, 
-0.029950178836461127, 2.3694584183777816, -0.57180774483350039], [6.1932109200372141, -10.938622839032124, 
0.71067144357861733, 5.2777750094174012], [-3.2852098569207846, -17.529448442415831, -7.1618400125346513, 
0.90435962385794955]], [[-1.4913107362062921, 1.2593646828759113, 5.76422972394219, -2.1425757917908261], 
[-1.6652158722689767, 9.6775449400565687, 3.7948150260119111, -7.0074506472637719], [4.5830800930287374, 
5.2719361490107728, -0.76732802340590534, 7.4467689202114986]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank0_Symbol_rank0(self):
      arg0=Data(3.142013671,self.functionspace)
      arg0.setTaggedValue(1,-2.04077395087)
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(-0.905206509275)
      sub=res.substitute({arg1:s1})
      ref=Data(-2.84417122722,self.functionspace)
      ref.setTaggedValue(1,1.84732186428)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank0_Symbol_rank1(self):
      arg0=Data(1.54368119889,self.functionspace)
      arg0.setTaggedValue(1,-0.973182859739)
      arg1=Symbol(shape=(2,))
      res=arg0*arg1
      s1=numpy.array([-0.97051483006179051, -4.8243289242685101])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-1.4981654964063673, -7.4472258576349226]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.94448839773843207, 4.6949542188401541]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank0_Symbol_rank2(self):
      arg0=Data(0.576275206322,self.functionspace)
      arg0.setTaggedValue(1,-0.446417285252)
      arg1=Symbol(shape=(4, 5))
      res=arg0*arg1
      s1=numpy.array([[-3.7798020411794298, 2.8125776443752777, -0.74593700484018655, 4.9042983986909512, 
-1.1062378936297144], [1.3747147971013396, 1.7312150406230939, -2.4865059810459189, -3.9444781957615138, 
-4.8713070674060148], [4.7677542872819085, -0.65669250050514094, -2.2966507465733335, 4.6331137703181184, 
-4.2587467390331817], [-2.2122452558031123, -0.89486317692759698, -2.7263171047505361, 1.4136050574112167, 
1.5057522304514919]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-2.1782062011384205, 1.6208187623100703, -0.42986500136777495, 2.8262255715721345, 
-0.63749747039309623], [0.79221405333400008, 0.99765630472347699, -1.4329117472490631, -2.2731049860965866, 
-2.8072134853290578], [2.7475385855977827, -0.37843560621895822, -1.3235028828319906, 2.6699485939051231, 
-2.4542101557111002], [-1.2748620912236397, -0.5156874619142493, -1.5711089520403427, 0.81462554611800553, 
0.86772767727381395]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[1.6873689660148408, -1.255583276563631, 0.33299917267007023, 
-2.189363577211358, 0.49384371731752186], [-0.61369644771829823, -0.77284431862310299, 1.1100192498224006, 
1.7608832478891809, 2.1746356766622736], [-2.1284079256789306, 0.29315888332112217, 1.0252645914581893, 
-2.0683020716109741, 1.9011781578167486], [0.98758452140814967, 0.39948239011636527, 1.2170750806399457, 
-0.63105773214859318, -0.6721938229809169]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank0_Symbol_rank3(self):
      arg0=Data(4.6954921918,self.functionspace)
      arg0.setTaggedValue(1,3.80656545201)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0*arg1
      s1=numpy.array([[[3.3283491776040073, -3.9068878738203718], [-2.1695978858355423, -2.2223735496995447]], 
[[-2.3211489651914921, 4.272069872491878], [1.636342469753175, -4.2787938517786497]], [[2.7410635950334186, 
-3.5668158773147507], [-1.0064480730166228, 1.389332564769]], [[0.77463712529690731, -0.94041585508240466], 
[3.6978341544417166, -2.6780892355753592]], [[-4.4954676727861065, -4.2409706282499835], [2.3785224394198679, 
-4.1039517994892138]], [[-2.0175257524312817, 0.85038925666007348], [-3.2277420742959917, -3.855794844823607]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[15.628237575034362, -18.344761505774329], [-10.187329932293613, -10.43513764988427]], 
[[-10.89893684206883, 20.05947072912349], [7.6834332898420676, -20.091043121362375]], [[12.870642707715545, 
-16.747956101531248], [-4.7257690683049729, 6.5236002096908132]], [[3.6373025733125388, -4.4157153045874011], 
[17.36315139876443, -12.574947094596416]], [[-21.10843335607106, -19.913444470614692], [11.168333542324792, 
-19.270073630038475]], [[-9.4732764173030475, 3.9929961146407393], [-15.155837707011633, 
-18.104854587064445]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[12.669578991701924, -14.871824405371543], [-8.258716356981127, 
-8.4596103757528684]], [[-8.8356054598727027, 16.261913585211406], [6.2288447130233893, -16.28750885246448]], 
[[10.434037982623636, -13.5773180922763], [-3.8311104639897189, 5.2885853424057174]], [[2.948706919001554, 
-3.5797545044815582], [14.07604773957002, -10.194321961547963]], [[-17.112291933867024, -16.143532276496508], 
[9.0540013447323737, -15.621961136660534]], [[-7.6798438277506191, 3.2370623651649013], [-12.286611468022571, 
-14.677335446353782]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank0_Symbol_rank4(self):
      arg0=Data(-1.72281700023,self.functionspace)
      arg0.setTaggedValue(1,1.23448641864)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0*arg1
      s1=numpy.array([[[[-4.3265612601968471, 4.9346403281714331, 2.8024200919119817, -3.9056671936664311], 
[0.98069732637570617, -2.2852413242790757, 3.2075463029671312, -2.6966078512789116], [-2.1601129611240619, 
-2.6503532815304762, -4.675750160645002, 3.2350739199006568]], [[-4.1760984392537086, 4.3345400423125842, 
2.3141216950646779, 0.60673873055732486], [3.6895192429599195, -1.0430965175426432, 0.33936966744652075, 
-4.9652989404647769], [0.016939166262534222, -3.5215478761207564, 0.96881594277756378, 2.4707123930500092]]], 
[[[-4.0598585401879825, 0.32726568454206451, -3.8317591404661555, -4.8432615549786364], [-1.8707032325346216, 
0.11786029243200069, -1.2644962697725761, 4.5016381310909193], [1.0052891428203132, 3.5573702004465542, 
0.94853515124922705, -3.266716026917611]], [[4.4268917686602247, 1.7045644573811822, -4.2672635941058026, 
4.4735466129490451], [-3.3659634968161098, -3.7740307778271154, -0.23936175808445981, 1.638694221507726], 
[-2.6562820856857803, -1.8386899346245853, -3.8721446565337256, 2.2142808663189424]]], [[[-4.9689140219050429, 
3.0036100506068504, 1.7161971518176031, 1.2296325439044953], [-4.2017528414854652, -1.8394187611478952, 
-4.4722717389932569, -2.3151891625454821], [1.0583223957426515, 4.9808003293003509, -0.20896133794562566, 
-3.9944246041361611]], [[-3.3354149131160451, 1.5689046088326091, 1.0657585673339192, -2.4003243575280555], 
[0.12124021598431511, -1.1303400850693057, -1.9271523374197388, -1.7678094654193863], [1.3900959283471721, 
1.5973269294693555, 3.1820328180383193, 0.020208485606988624]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[7.4538732915852082, -8.5014822473732448, -4.8280569761201289, 6.7287498384724653], 
[-1.6895620259559927, 3.9370526030863635, -5.5260152997629799, 4.6457618491254991], [3.7214793318324864, 
4.5660736900247683, 8.0554618655674215, -5.5734403461917594]], [[7.1946533857624484, -7.4676192730552966, 
-3.9868081968486213, -1.045299799699541], [-6.3563664744313337, 1.7970644132987275, -0.58467183243782017, 
8.554301425835547], [-0.029183083606744185, 6.06698254808967, -1.6690925763069098, -4.2565853134149645]]], 
[[[6.9943933115474914, -0.56381888491958132, 6.601419787965443, 8.344053343456924], [3.2228793313878823, 
-0.20305171545342732, 2.1784956702862215, -7.7554987011078413], [-1.7319292253931926, -6.1286978574257542, 
-1.6341524838838573, 5.6279539060835297]], [[-7.626724397207207, -2.9366526251568561, 7.3517142643698472, 
-7.7071021560908708], [5.7989391344540557, 6.5019643834157081, 0.41237650603182713, -2.8231702629851867], 
[4.5762879346145349, 3.1677262775151811, 6.6709966416095421, -3.8148007197688423]]], [[[8.5605295495980389, 
-5.1746704572343623, -2.9566936288903536, -2.1184318506694821], [7.2388512260579478, 3.1689819122397545, 
7.7049057815666941, 3.988647247971739], [-1.823295815105068, -8.5810074820485838, 0.36000214540263886, 
6.8816626141257302]], [[5.7463095151227659, -2.702935531829326, -1.8361069779390997, 4.1353196092052489], 
[-0.20887470520881798, 1.9473691145940035, 3.3201308089314878, 3.0456122001844874], [-2.3948808973010833, 
-2.7519019890081795, -5.4820602341926197, -0.034815522552537087]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-5.3410811151064834, 6.0917464659774998, 3.4595495427759313, 
-4.8214931062905171], [1.2106575302027247, -2.8210993781265854, 3.9596723481566771, -3.3289257687890399], 
[-2.66663011322566, -3.2718251306347788, -5.7721500702479123, 3.9936548173985087]], [[-5.1553368061426292, 
5.3509308132656104, 2.8567518036265622, 0.74901072253301881], [4.554661396727484, -1.2876884842321341, 
0.41894724535949618, -6.1295941064678363], [0.020911170694103174, -4.3473030256449352, 1.1959901235162222, 
3.0500608935740701]]], [[[-5.0118402294424564, 0.40400504285255151, -4.7302546183870042, -5.9789406115194401], 
[-2.3093577338610776, 0.14549693030368552, -1.5610034714491989, 5.5572111344423751], [1.2410157936131636, 
4.39152519850908, 1.1709537618153174, -4.0327165687679427]], [[5.4649377651792932, 2.104261672325459, 
-5.2678789516603199, 5.5225325368173852], [-4.1552362224415518, -4.6589897387390575, -0.29548883949592541, 
2.0229457607473758], [-3.2791441588432502, -2.2698377523754134, -4.7801099894819963, 2.7334996565146623]]], 
[[[-6.1340568754081053, 3.7079158143505366, 2.1186220756193546, 1.5179646753620302], [-5.1870068172759387, 
-2.2707374788199806, -5.5209587222334928, -2.85806957773395], [1.3064846240818391, 6.1487303604553372, 
-0.25795993371372866, -4.9310629240686339]], [[-4.1175244107552791, 1.9367914317381283, 1.3156644769179004, 
-2.963167819687802], [0.14966940002504314, -1.3953894834570877, -2.3790433871858725, -2.1823367757951], 
[1.7160545441447639, 1.9718784005502887, 3.9281762975200034, 0.024947101023013681]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([-0.099233059085104713, 4.771977048069223]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([3.9729085267773208, 4.512809517509826]))
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(0.26176969234)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-0.025976207346631468, 1.249158963725014]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([1.0399870427481233, 1.1813167589860505]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([0.84702689091359229, -3.3372769586299422]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([-2.152707415414048, 1.9005183627662312]))
      arg1=Symbol(shape=(2,))
      res=arg0*arg1
      s1=numpy.array([0.22148437875716098, 4.0581595354793194])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([0.18760322472460655, -13.543202312199522]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([-0.47679106454891407, 7.7126067162133252]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[-4.9957974390113735, 4.9127660926149055, -0.033400607153987849, -4.5745875540405283, 
-3.40034229393263], [4.6311740546030773, 4.0795994583149682, 4.8540687237153293, 4.9306788508967045, 
3.1060981817064288], [-1.3242874361820456, -3.3839454855009707, 0.088505407790738566, -4.4328915815516297, 
-3.0958370529970693], [-2.3333608177639089, -4.3040231210385214, 4.1339174077369343, -4.5703847879440351, 
-3.3406709387044389]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.78277733029185015, 4.3160080804422201, -3.0818130197239957, 
1.7055665928942068, -3.3364799279772583], [4.5669273627829092, 3.6704066897391243, 2.6983979089447621, 
-1.2237350853460538, -0.1257348607090929], [2.1891872096029914, -0.7503980382583979, 1.9746042593444724, 
-2.0584330310875232, -0.7673935307397155], [-0.23746062225782705, 2.8663010003293437, 3.8185722602896526, 
4.8671017855990222, -1.9042813962136051]]))
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(-0.123633480243)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[0.61764782397173934, -0.60738236964752024, 0.0041294333046613622, 0.56557217998023424, 
0.4203961518147894], [-0.57256816597952853, -0.50437507902705625, -0.60012540964938543, -0.60959698629464243, 
-0.38401772817938795], [0.16372626457665548, 0.41836895732351315, -0.01094223158545413, 0.54805381376508067, 
0.38274910912583321], [0.28848151856172877, 0.53212135749834688, -0.51109059615373209, 0.56505257738107939, 
0.41301877449713931]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[0.096777485598944848, -0.53360309973999165, 0.38101526908524591, 
-0.21086513366492038, 0.41250062525520415], [-0.56462512387573127, -0.4537851529579448, -0.33361232456203066, 
0.15129462749623307, 0.015545038417276023], [-0.27065683362567039, 0.092774321037059626, -0.24412719668449809, 
0.25449123947954888, 0.09487553292096082], [0.029358083150294065, -0.35437076809338752, -0.47210337809722808, 
-0.60173673244828274, 0.23543293637500923]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[-2.5363493699555439, 1.9315826441262116, -4.797663921800063, -2.3131658171459835, 
-1.4174075473244754], [-0.4937783451212221, -4.7652740781432534, 1.5781017135273068, -4.2362357361072114, 
-3.918073606586808], [2.094919785395116, 1.3684348598821918, -4.2376402301126852, -1.6050592311847534, 
3.151025223042982], [-2.6417620339476366, 0.27296872857386667, -1.4323869283247213, -4.6402797342361799, 
-3.9199666124863741]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[-0.69802384928264605, 0.87453186333400357, 0.81014803662176149, 
-4.214756633799734, 0.78685864942817574], [0.48682400140861759, -2.7486583171634758, 0.40554914153896249, 
-1.7609786982015061, -0.39145780725801416], [1.2068445571926318, 0.18236245525374706, 4.017808328373075, 
-1.0950851034750277, 0.12173582687690843], [0.22180579191198468, -4.2110674925319236, 2.9122016067639365, 
4.5406571257464297, 3.0637655540581346]]))
      arg1=Symbol(shape=(4, 5))
      res=arg0*arg1
      s1=numpy.array([[-2.7914057802277092, -3.2955192040483841, -0.4909883356152811, -4.414815259808397, 
1.1535659837090115], [0.30062418712185313, 4.6879078677821262, 2.641934893458421, -4.6213986057014331, 
2.2307025160830776], [4.0559589148649486, -0.33010334091372151, -1.5019795108463163, 1.5894091005782052, 
4.3064711265533191], [2.9888346593766579, -4.5884630123207506, 2.4921626108815289, -3.5186629218511625, 
-1.0861727773454932]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[7.0799802919708137, -6.3655676979244857, 2.3555970238060953, 10.212199748003247, 
-1.6350731316459357], [-0.14844171362044126, -22.339165843065977, 4.1692419823943165, 19.577333924268451, 
-8.7400566524118908], [8.4969085795002854, -0.45172491906991191, 6.364848799967322, -2.5510957490121045, 
13.569799142075835], [-7.895789928888072, -1.2525069145814096, -3.5697411470863107, 16.327580247874213, 
4.2577610225859299]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[1.9484678076243735, -2.8820365501694258, -0.39777323610290649, 
18.607371903277738, 0.90769337196755784], [0.14635106969487355, -12.885456950875437, 1.071434428043893, 
8.1381845005383653, -0.87322591559081664], [4.8949119406016965, -0.060198455736490949, -6.0346657877240473, 
-1.7405382293708347, 0.52425182351249966], [0.66294083851702656, 19.32232743186902, 7.2576799597261958, 
-15.977041869203234, -3.3277787409867781]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[0.20286433747766175, 2.066180723397264], [-3.1327754856901735, -1.1194293005675835]], 
[[-1.7914925359739922, 1.275772548969373], [-2.3842819867614953, -3.1968139299234077]], [[-3.5171630188865488, 
-1.4300055611015186], [4.2854751694367756, 1.7799374077309524]], [[-4.2108803597952917, -1.5964309596888695], 
[2.7414856168787471, 1.1873651110226469]], [[-3.5913507733928229, -1.3017853623346696], [-0.13258097661378798, 
-3.1022689591044426]], [[4.9076894073951749, 2.8964781538465161], [2.969217301725779, 
1.8197050412291595]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[3.7913962593423918, 4.4765688935993317], [1.4500345756448763, 
4.8443295010220275]], [[1.7391435441206662, 4.4187921026078829], [-4.2876409136689784, -4.1427096121048841]], 
[[0.11488507950500271, 2.1339510129273167], [-2.8155795121378926, -4.6369329094888991]], [[0.67434242728218052, 
4.9095299484356563], [0.94463745614236405, 3.557499141589803]], [[0.038621679734069048, -0.10332111066950311], 
[-2.0403842705827979, 1.0573287011436552]], [[-2.1400599935190945, 4.2642563454671869], [3.9163707220927186, 
-2.5416950274474726]]]))
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(3.22032001333)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[0.65328808596966281, 6.6537631347069341], [-10.088539593828681, -3.6049205801225788]], 
[[-5.7691792673232181, 4.108395871899436], [-7.6781509993834112, -10.294763877415299]], [[-11.326390459854416, 
-4.6050755275843107], [13.800601454753718, 5.7319680565855959]], [[-13.560382296374918, -5.1410185693811012], 
[8.8284609982830986, 3.8236956301526179]], [[-11.565298770434771, -4.1921654553826446], [-0.42695317237583746, 
-9.9902988157275558]], [[15.804330417828238, 9.3275865669967502], [9.5618299006647014, 
5.8600325626225302]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[12.209509252413918, 14.41598439909561], [4.6695753639655058, 
15.60029124329226]], [[5.6005987611804535, 14.229924642760029], [-13.807575844248428, -13.340850673264026]], 
[[0.36996672076263859, 6.8720051543895364], [-9.0670670520514882, -14.932407848882294]], [[2.1715984144123999, 
15.810257548976251], [3.0420349053536828, 11.456285683055723]], [[0.12437416819593333, -0.33272704048818802], 
[-6.5706903014356595, 3.4049367769580914]], [[-6.8916780268502658, 13.732270051265186], [12.6119670159636, 
-8.1850713646631412]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[-3.0362430129414495, -1.6103362752021533], [3.0322276873410949, 2.889681860828258]], 
[[3.0356323227798061, 3.8849951306068178], [3.1682099248345352, 1.252560279633526]], [[-2.422448371330125, 
-4.3051891736441767], [3.2099062879412248, 3.0454833071508354]], [[-1.1376898513557334, 0.97676409380038631], 
[1.0009530341765513, -3.085670706338802]], [[3.7338110619145226, -3.4624334476005911], [-1.9009045069833541, 
0.020021974502883566]], [[2.2281987737323306, -2.9210437430011229], [-1.3860392623437132, 
0.463839486811219]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[4.1305451146381422, 2.3814888644055499], [-0.89220558616836776, 
3.8418880701208664]], [[3.3492033714884197, 2.1415021426686387], [1.4787086763681101, 0.38743271004052637]], 
[[-4.99502836998282, 4.4948912080047858], [-3.7653670133834769, -4.0800035996907447]], [[-3.1718016142315641, 
2.320405525041048], [-2.8237839197556944, 3.5858545025811086]], [[3.0016852702625556, -2.8784349824436584], 
[-3.7323763876968008, 0.63313826152617381]], [[1.7585155020491481, -3.655987828892715], [0.54081193002197825, 
4.9685421412273278]]]))
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0*arg1
      s1=numpy.array([[[-0.93345150384204523, -3.5420417816351479], [-3.5444663998891226, 2.3971219306415996]], 
[[0.8384895676298747, -4.3543886014540583], [1.9437605664303446, -3.0820690979415186]], [[4.9690708795309, 
-2.1112283314766356], [2.37080840790198, 4.7216280449696395]], [[-3.3735779321675219, 3.0438054138693342], 
[-0.47550686063032277, 2.7155331110677619]], [[-1.3378665576363566, -2.6741065199531286], [-0.22177834028631249, 
0.61204525154245371]], [[4.0531432724462295, -4.0695297515588145], [3.928681336032259, -4.8729434946660577]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[2.8341856064600983, 5.7038783692487431], [-10.74762915459401, 6.9269197611686435]], 
[[2.5453460338109117, -16.916778513418848], [6.1582415180666157, -3.8604773311674778]], [[-12.03731765914358, 
9.0892373557640713], [7.6100728160284898, 14.379639393530271]], [[3.8380853761846505, 2.97307983678279], 
[-0.47596003491968808, -8.3792409729148645]], [[-4.9953409522681316, 9.2589158571325303], [0.4215794466015394, 
0.012254354420993967]], [[9.0312088694261341, 11.88727441774779], [-5.4453065809776664, 
-2.2602636098259725]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[-3.8556635489463869, -8.4353330602232983], [3.162392721967159, 
9.2094741479570601]], [[2.8082720868638438, -9.3249325200257633], [2.8742556143627427, -1.1940943831476432]], 
[[-24.820650015712328, -9.4897416652449422], [-8.9269637741663139, -19.264259419876904]], [[10.700319930984929, 
7.0628628994922567], [1.3427286267814176, 9.737506633230419]], [[-4.0158543396339219, 7.6972417538137563], 
[0.82776024058721887, 0.3875092665369389]], [[7.1275152766229084, 14.87815124101582], [2.1246777357809301, 
-24.211425105067871]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[0.083529595185620309, 4.372823949648037, 4.0009796310147365, 0.68196650279571092], 
[-3.6991633277760947, -2.7548332509536966, -1.0695165519831527, -4.9760591951223629], [2.4321680119150511, 
3.8212450023110254, 0.8004705884830452, -2.0444757685121964]], [[-3.4279885095627494, 3.5526615517407674, 
-0.37278949561560104, 1.8208812052515633], [-1.3542092169449638, 4.6164533745863388, 0.70466827998486359, 
4.8882907830017537], [-1.0014606132552197, 0.027094091914280583, -0.50731223435619732, 1.3858925949581025]]], 
[[[-0.92819420758339621, -0.63572501286400218, 1.5822275384230347, -4.1213856389411898], [0.019204126745418826, 
1.2601369851282858, 3.1282675256554278, -1.2005085063042245], [0.31394545189514567, 4.3373088139034053, 
-4.2967881792088045, -3.2133421015779429]], [[-4.6394850485838566, 2.7899856599199682, 3.4214279590576524, 
0.75816457866836107], [2.6716538166314328, -0.78329465377730312, 0.9411007620209233, -4.3621528303217216], 
[-0.21060811641931387, -1.0884058600082236, 3.3643361086045402, -0.59658223764974405]]], [[[-2.7722966748649869, 
1.3359537198967564, 3.3994221388292836, 0.89269410005117322], [-2.5434807639867083, -2.2312407449400631, 
2.1964509494368221, -3.8483462591031992], [-4.617770174657271, -4.0164566401957895, -2.0915606068178807, 
1.3098480489351907]], [[-4.000475916402392, 3.4797401237531425, 4.727298203954307, -1.3658950385993265], 
[4.3822054513768176, 4.7641649434095044, 2.2480529159500593, -3.370947660818576], [-0.12763750951483388, 
-0.56331578609421484, 1.1108900947641267, 2.3086655633422826]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-0.99642554217400114, -3.1989491729058739, -0.40653827121759534, 
-3.7494299986207311], [-4.962891998625155, 0.65199311462416087, -1.5646394746401904, -0.58561931277306201], 
[-1.9266349397638294, -2.1350741612611923, -1.3141253220586746, 3.0489459946325113]], [[-2.4730670458068271, 
-1.0264935833023214, -2.4514436230760808, -2.4564197681665032], [-0.48365426443540827, -0.92587219714034585, 
-4.1751007096042496, -4.5526966630634966], [-2.3782862353872791, -2.5275939067663735, -2.6938709700385766, 
-4.8403251647207153]]], [[[3.2867188889910004, -1.4242104085047957, 2.0936224809849646, -1.4601757832869966], 
[-0.21399139868108641, -0.98887005738367506, -2.261387321435977, -3.6513388135428149], [2.9334655428664806, 
-3.9524701563078288, 3.4584296338361185, 4.5958550113660674]], [[0.37781815561656451, -3.0593937474827717, 
-2.0739947527751279, -2.4165789597896401], [4.5330104520530448, 4.9794431912053145, 1.7661478112868867, 
3.078941742057026], [4.9858586211966696, -3.1080213069928195, -4.2716128061474183, -1.5876111863601041]]], 
[[[0.90451414172461853, 0.1162545327223361, 3.2911315914907693, -1.4337863404739979], [2.0405912462551932, 
4.8936580709384394, -1.1291930809589745, 2.111861338433255], [-2.0913683111797732, -1.55247331778124, 
4.9769696268492716, -0.24856367420835213]], [[-2.1381113867577026, 1.6110287228762354, 0.19273167692598125, 
-1.1320874579780638], [1.2876584378472149, 0.79519349199575018, -3.7974642196600819, -4.2341641389677163], 
[-4.3323767453858073, -0.80301234928598664, 4.344905698376083, -0.27642913101571054]]]]))
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(-1.40149736096)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-0.11706650721432915, -6.1285012253547855, -5.6073623941039141, -0.955774253928249], 
[5.1843676416216082, 3.8608915310841012, 1.4989246251026278, 6.9739338299224034], [-3.4086770500993597, 
-5.3554647863033438, -1.1218574172814939, 2.8653273941073367]], [[4.8043168495380142, -4.9790457891328019, 
0.52246349429720151, -2.5519602037735591], [1.897920643729998, -6.4699472214569242, -0.98759073474784653, 
-6.8509266319601139], [1.4035444065780798, -0.037972298315350331, 0.71099675763065506, -1.9423248144015677]]], 
[[[1.3008617323823231, 0.89096692782230835, -2.2174877195310199, 5.7761110964560265], [-0.02691453295315966, 
-1.7660786590997284, -4.3842586815689133, 1.6825095033900148], [-0.43999372231503248, -6.0787268563347068, 
6.0219372937459816, 4.5034904752087277]], [[6.5022260517828334, -3.9101575394815598, -4.7951222553187058, 
-1.0625656561736561], [-3.7443157733956758, 1.0977853901194523, -1.3189502343655459, 6.113545679780513], 
[0.29516671935747973, 1.5253979404500404, -4.7151081775766022, 0.83610843165905069]]], [[[3.8853664736090217, 
-1.8723356127940041, -4.7642811563429852, -1.2511084253622755], [3.5646815783684827, 3.1270780156899103, 
-3.0783202091039361, 5.3934471261761532], [6.4717927132812418, 5.6290533816266439, 2.9313166707337701, 
-1.8357485838353969]], [[5.6066564394040377, -4.8768466002510378, -6.6252959572916961, 1.914298291939184], 
[-6.1416493752694707, -6.676964595345189, -3.1506402289923527, 4.7243742505563926], [0.17888363274397381, 
0.78948558759562215, -1.5569095361235428, -3.2355886943530807]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[1.3964877677455285, 4.4833188236584016, 0.56976231423887624, 
5.2548162481543859], [6.9554800387803839, -0.91376662950692644, 2.1928380945550465, 0.82074392137602725], 
[2.7001737836036899, 2.9923008024518656, 1.8417431708300456, -4.2730897651733448]], [[3.4659969381643139, 
1.438628048035971, 3.4356917682723473, 3.4426658224843063], [0.67784017522110429, 1.2976074408742768, 
5.851392626233844, 6.3805923585144591], [3.3331618824920928, 3.5424161899003064, 3.7754530552637315, 
6.7837029445226396]]], [[[-4.6063278491235211, 1.9960271289648437, -2.9342063819375608, 2.0464325068078741], 
[0.29990838051872154, 1.3858987757511469, 3.1693283630907771, 5.1173417111346877], [-4.1112442167813024, 
5.5393764933208445, -4.8469800048716598, -6.4410786697637112]], [[-0.52951114801769439, 4.2877322632209003, 
2.9066981726499224, 3.3868290346857983], [-6.3530021857360977, -6.978676491502144, -2.4752514965759262, 
-4.3151287260286901], [-6.9876676997044234, 4.3558836595439399, 5.9866540748393779, 2.2250328879071364]]], 
[[[-1.2676741825739934, -0.16293042080947032, -4.612512240031629, 2.0094477723483699], [-2.8598832464155737, 
-6.858448871838867, 1.5825611229732273, -2.9597680925181833], [2.9310471689044393, 2.1757872578242567, 
-6.9752097975849967, 0.34836133343241132]], [[2.9965574659698517, -2.2578525035345738, -0.2701129365842932, 
1.5866175847270911], [-1.8046499024549696, -1.1144615804810425, 5.3221360821765895, 5.9341698666157248], 
[6.0718145753232413, 1.1254196883389966, -6.0893738698746489, 0.38741469760974395]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_taggedData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[0.94895783352211716, -0.46506050687609246, -3.4973315286311935, -3.8305662019128084], 
[-0.067011552625391602, -2.0053313702353224, -4.2991620654185292, 1.0093244189118113], [0.87798040910337161, 
-3.1557405362751467, -2.7931609613797503, -4.9110293393501543]], [[-0.43377674574157687, 2.704453693295501, 
-2.5592027474361601, 2.6750943602438779], [3.4305357783522936, 1.1310936272574317, 2.5896451684195103, 
-2.5539402760310126], [4.5085098664567802, 0.11349764632152315, -1.3234736581612081, 2.9677015132450526]]], 
[[[-1.1465164843195783, 2.7375150894033169, -0.10912903641971639, -3.2522173068497242], [4.7360277602148102, 
-1.6692169008049049, 1.233494102169499, -1.3392318230236588], [4.3885022964524385, 4.7515889636089952, 
2.7117463429070652, 2.1998096737486339]], [[-2.5131848249547706, 0.77921677989624794, -3.8934505794744254, 
0.24453982436947097], [-3.4599112442822841, 0.95455762656251686, -2.8118915894259002, 0.78333576699382768], 
[3.9748218483200848, 2.0642803168106862, -3.4033833226648325, 1.1533845398716842]]], [[[0.49409552443598681, 
4.2995599265190414, 1.6569894487618955, -3.0328627613000969], [-4.8050094221736783, -1.8954288851770262, 
-0.65657001508748358, -4.2826450941652627], [-0.74170635020795395, -0.84409250409421333, 3.6365860981881823, 
1.6100686197509306]], [[2.2396538156436314, 4.3413750637744482, -4.4044991729772871, -2.290449250092359], 
[3.1707282757549784, 3.5745559429744826, 1.7312931654177177, 3.6613171235792787], [-0.8687074400525221, 
-3.7812204587391882, 0.42585417549408167, -2.1167068471762196]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-1.6280796155619202, 3.2372833297254093, 1.0760799245326114, 
-0.6791544053399603], [-4.7288229986341772, 3.0364132365453695, -2.7881857643052821, -4.280550845421411], 
[-3.2543576320948486, -0.55086025606976641, 2.7326721039171273, -4.2151748027883897]], [[-0.090786768277823171, 
2.6835858500949072, -4.3650061828137812, 3.7514613040027918], [4.6631405077073573, -1.4292251728385805, 
-1.3449901199338612, 1.8541088776332506], [-0.75859253853348552, 3.0321616898786914, -0.78197759748368689, 
-0.88789890229887547]]], [[[-2.062426285540877, 0.30001445848584307, -0.83075755338807511, 3.1138362566950235], 
[-2.3733111917258274, -2.9574688279506556, -1.7570853300314857, 4.3659538409493486], [-1.018783824996695, 
0.89420512031921273, -1.8230865992410106, 3.1994892045316963]], [[0.3233781997620504, -1.3905319280411477, 
4.9629769739530829, 1.1091529164244776], [2.7351448192501895, 2.0660484456167358, -4.0171925239278465, 
4.3911761581077471], [2.1258515256453734, 1.5150132919784713, 2.3610249459029262, -4.5111733078663327]]], 
[[[-2.4852355463920439, 2.8830764355289409, 1.862859073381494, -2.1509811646634249], [-2.6325170393160802, 
-1.3580306364602079, 3.8152036772116542, -1.8416692569043969], [-3.1659556710428727, -2.074597485798928, 
0.35812962834444573, 4.8712028428189775]], [[-3.0119309329000288, 0.64957909215262699, -1.0255988706704269, 
4.4011917781031578], [3.4155148745532635, 0.92333362783500483, -4.4110812159393742, 3.9318582439553591], 
[-0.47136877672690947, 2.9648508900721211, 4.2958677743999178, -3.5373956917354246]]]]))
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0*arg1
      s1=numpy.array([[[[-0.89769175763822151, 3.1729441493671864, -4.6066637580891037, 4.9743040690579683], 
[-2.6788549386226723, 4.5100260370904159, -1.4037388326608391, -1.3338419564469453], [-1.1381168569459419, 
-3.1452061025455782, -1.5160789869753852, 4.1456610012343731]], [[-2.5938250596030543, -0.93202167450854922, 
-0.77103273701479047, -2.7062810722150674], [-1.105689822011354, -1.8461034185063041, -2.0068461224928411, 
-4.7319124228070422], [-4.5218149557274483, 4.8904398898663093, -3.825654173472147, 0.88914553957591913]]], 
[[[2.5862303105862106, -2.0807587439433863, 2.3052217753786719, 4.1346828351021188], [0.68224921552396101, 
-2.8879458749312326, 0.54057865399082239, -0.0023583593457887631], [1.9450000139899339, 2.3035289835776487, 
0.43188049217124025, 2.7210621624224993]], [[-0.54240954949206355, 2.4426743469776238, -0.12851500556720108, 
2.8020215668130284], [0.52130307899069095, -2.6755785781541075, 0.43102203402703765, 2.8334377967823654], 
[-0.76961602631277248, -0.42105873518056569, -2.6486982753852963, 2.2122858679665658]]], [[[-3.480147294108944, 
3.6715573223019664, 1.7695636383465487, 4.490486044765845], [-2.4274939795683825, 4.5761414724005753, 
-3.0218037175811974, 3.7714386247368186], [1.1400240440382881, -1.8970197325492588, 3.6790698826844643, 
0.066641356768850635]], [[-2.2050658457593322, 2.6656171339669479, -3.3829610334860827, -3.4782484246680303], 
[-1.0395696415089963, 2.3867408856698438, -0.23958153915918956, 4.5415496456746567], [-4.7338880436477284, 
0.94198626201791846, 4.0511716726168139, -3.3273461069241539]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-0.85187162549902817, -1.4756110143942358, 16.111030402967682, -19.05440104497081], 
[0.1795142286953034, -9.0440966927555042, 6.034900739130368, -1.3462792576110065], [-0.99924430366884154, 
9.9254543927430472, 4.2346526407878047, -20.359462808061743]], [[1.1251409933775647, -2.5206094598561033, 
1.973229098931474, -7.2395572335172815], [-3.7931084941699296, -2.0881158119306398, -5.1970193648750147, 
12.085021719258394], [-20.386647342189029, 0.55505341697671495, 5.0631525238248756, 2.6387185632945438]]], 
[[[-2.9651556833340331, -5.6961084589529127, -0.25156663108082233, -13.446887074653596], [3.2311512241062563, 
4.8206080630450217, 0.66680058145640564, 0.0031583898860055686], [8.5356370279948504, 10.945422895721002, 
1.1711503452182641, 5.9858188677683906]], [[1.3631754486940078, 1.9033728389870741, 0.50036682289677803, 
0.68520586182792786], [-1.8036623846788673, -2.5539939372442984, -1.2119872323378713, 2.2195331697718155], 
[-3.0590865962052933, -0.86918325925444495, 9.0145355372174212, 2.551616317889247]]], [[[-1.7195252023972392, 
15.786080730887091, 2.9321482776529417, -13.619027905308091], [11.664131444095956, -8.6737507294445777, 
1.9840257124437006, -16.15173312417452], [-0.8455630728529504, 1.6012601363636387, 13.37925438963315, 
0.10729715731115268]], [[-4.9385841352003395, 11.572443754974021, 14.90024907420384, 7.966751495915819], 
[-3.2961928569490411, 8.5315388172113202, -0.41478588130656219, 16.628053485294025], [4.1123637638924597, 
-3.561857725793407, 1.7252083724272129, 7.0430162874514943]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[1.4615136516687401, 10.271719200896161, -4.9571383891516385, 
-3.3783205220012094], [12.667830843763641, 13.694302756185596, 3.9138846300274661, 5.7095783143275201], 
[3.703839279617827, 1.7325690390404491, -4.1429467550425727, -17.474685793305618]], [[0.23548499463939337, 
-2.501160177692904, 3.3655626642213927, -10.15250872017001], [-5.1559869979808832, 2.6384974773925669, 
2.6991882069804505, -8.7734808313096], [3.4302150860439653, 14.82860448070719, 2.9915758593751893, 
-0.78947134857339996]]], [[[-5.3339093730155467, -0.62425770780385803, -1.9150804021305003, 12.874725321875548], 
[-1.6191896987491827, 8.5410099019178034, -0.94984282265544051, -0.010296488044085244], [-1.9815345538712901, 
2.0598274119188451, -0.78735553775100031, 8.7060090135304602]], [[-0.17540342364848829, -3.3966166692794464, 
-0.63781701343747121, 3.1078703927149545], [1.4258394157605607, -5.5278749625207295, -1.7314984927415895, 
12.442124498712067], [-1.6360894036981373, -0.63790958050220026, -6.2536427023547434, -9.9800049569406735]]], 
[[[8.6489857619996346, 10.585380397622536, 3.296447679619837, -9.6589509024752935], [6.3904192640509674, 
-6.2145403162961061, -11.528796655127632, -6.9457425694795969], [-3.6092655873482475, 3.9355523676576474, 
1.3175839297390308, 0.32462356654173891]], [[6.6415060299238959, 1.7315291579087375, 3.4695610154653864, 
-15.308438368849195], [-3.5506655737079806, 2.2037581206676693, 1.0568136270709447, 17.856729414678441], 
[2.2314070162963722, 2.7928488073795359, 17.403297836956384, 11.770139783546139]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.0430204652553)+(1.-msk_arg0)*(1.78425217281)
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(2.5646949317)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(-0.1103343692)+(1.-msk_ref)*(4.57606250448)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(2.57326660208)+(1.-msk_arg0)*(3.29535894632)
      arg1=Symbol(shape=(2,))
      res=arg0*arg1
      s1=numpy.array([4.0270481495052639, 2.7564226252917825])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([10.362668508083143, 7.0930102828763504])+(1.-msk_ref)*numpy.array([13.270569146744505, 
9.0834019581016197])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.02551699988)+(1.-msk_arg0)*(0.626618362726)
      arg1=Symbol(shape=(4, 5))
      res=arg0*arg1
      s1=numpy.array([[-0.87644218701217369, -3.616282438617354, 0.79667729938717624, 3.9811287325297613, 
0.065796525107652215], [-1.4217337078130887, 4.8515183472866479, -0.78214368357519071, 3.1809506063985502, 
0.95738137909039533], [0.39346667195906182, 4.3285617066713939, -4.5910865330304649, -4.7275376371854012, 
-0.90249772270981055], [1.0196798908676152, -4.5635511009909653, -4.4978908227402012, 1.164740898313557, 
-1.7685582007909115]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[2.6516907362186282, 10.941123994409631, -2.4103607127155628, -12.044972658985373, 
-0.19906850524632991], [4.3014795022930183, -14.678351234952618, 2.3663890110066546, -9.6240201354420911, 
-2.8965736378079594], [-1.1904401048989282, -13.096137028570261, 13.890410353610562, 14.30324548888392, 
2.730522202412847], [-3.0850588442572544, 13.807101435875975, 13.608445147811342, -3.5239433883048839, 
5.3508029017726937]])+(1.-msk_ref)*numpy.array([[-0.54919476824925706, -2.2660289808399305, 0.49921262496268565, 
2.4946483681778449, 0.041229310835994216], [-0.89088444822170665, 3.0400504835102153, -0.49010559441809565, 
1.9932420608926262, 0.59991275226964935], [0.24655344177009811, 2.7123562495913784, -2.8768591264593351, 
-2.9623618939370084, -0.56552164536805005], [0.63895014371973546, -2.8596049191178006, -2.8184609830641962, 
0.72984803470084503, -1.108211044164624]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-3.56361070359)+(1.-msk_arg0)*(4.70518403887)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0*arg1
      s1=numpy.array([[[4.2317874979104992, -1.1989677806745727], [1.3134595205642725, 4.4478340118436144]], 
[[-0.017162439223159964, -0.41900720330017371], [-4.4667032138911269, 2.0617117365888351]], [[2.9794518983997751, 
0.52772381685170533], [-2.6894168529203224, 0.41500536076126604]], [[-4.6733566211583097, 2.4911601334345299], 
[-4.7318467145182375, -0.81771569841489722]], [[1.3136083167944523, 0.82585873482530836], [0.296465998582784, 
-1.7304343680929755]], [[-1.3066203047314175, 2.65896658854032], [3.9719908108129438, -2.8680613980844938]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-15.080443222871585, 4.2726544164712799], [-4.6806584062148398, -15.850348892396909]], 
[[0.061160252115363196, 1.4931785545617493], [15.917591382781623, -7.3471380122247991]], [[-10.617606675968551, 
-1.8806022422720292], [9.5840346834818018, -1.4789175456560166]], [[16.654023676852269, -8.8775249158636207], 
[16.862459599603678, 2.9140204153647811]], [[-4.6811886580533626, -2.9430390270766442], [-1.0564894058000638, 
6.166594435995874]], [[4.6562861034687169, -9.475521795410085], [-14.154628967973551, 
10.220654296766785]]])+(1.-msk_ref)*numpy.array([[[19.91133899106331, -5.6413640647508636], [6.1800687718624765, 
20.927877600075174]], [[-0.080752435100909861, -1.971506005140051], [-21.016660668375366, 9.7007331557512781]], 
[[14.018869516935215, 2.4830376799828495], [-12.654201250232003, 1.9526765994999067]], [[-21.989002981827273, 
11.72136689810848], [-22.264209635536492, -3.8475028525162167]], [[6.1807688855097629, 3.8858173374624307], 
[1.3949270845997368, -8.1420121690652945]], [[-6.1478890026873314, 12.510927152291805], [18.68894776558027, 
-13.494756712769872]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank0_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(2.51859660817)+(1.-msk_arg0)*(3.69262914568)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0*arg1
      s1=numpy.array([[[[0.67581666027729526, 2.8108253236352603, -3.2853142689252701, 4.9303752145005504], 
[-1.3657702337917108, 4.3904435823590919, 3.3571499478699067, 0.015664424638796248], [-3.1193276566121941, 
2.4614116251071474, 4.8836311348501411, 4.5339873643248207]], [[0.92904130460985179, -4.0289182883108268, 
1.1622216990935978, -3.453322460119399], [-4.9809628080889743, -3.091966318347751, -0.82579662341565552, 
0.19293706069789529], [-4.1200195632312617, -4.252279729557471, -4.2626018577200098, 2.8104051978786853]]], 
[[[4.553470024096578, -4.1556230388171853, -4.8792826237543903, -0.084660297479757673], [-1.3940013078624869, 
-4.391326436454821, 1.8581067287379591, -4.9713750706033011], [1.1291380851369173, 0.083960164404878412, 
1.6804470605141759, -4.9497828740616505]], [[1.5580688980525261, -0.37196449698866818, 2.6938341003534481, 
-2.2283064145681664], [0.4008724735809448, 2.2505971863148471, -1.6587271003862281, -2.7339584931448382], 
[-4.8561801785490113, 2.7658580526343668, 2.6247408695536976, 1.2921000397940583]]], [[[-3.378532765195783, 
-4.7136195887628478, -2.2928079312088725, -3.2689147861576906], [2.1255291713840734, -2.9248168929356853, 
-4.2298230449258032, 0.73722628474508767], [-4.0600914026090829, 4.8210888962888614, -1.5990379796232492, 
-1.3511964372595688]], [[-2.5350622751408225, 3.9568471756339907, -2.5691428388833124, 3.4094397075929948], 
[0.80244721075126435, -1.8355785194219432, -4.4146407757484631, 4.8328210906719811], [-4.920522191727871, 
-2.4717933373462273, -4.5300856839069414, -2.5250375128130722]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[1.7021095483176161, 7.0793351262596333, -8.2743813744801198, 12.417626292235164], 
[-3.4398242783642043, 11.057756314881239, 8.455306471815506, 0.039452366764170722], [-7.8563280557071602, 
6.1993029702993985, 12.299896811775723, 11.419285197263687]], [[2.3398802786380641, -10.147219935524449, 
2.9271676292760325, -8.6975262349660429], [-12.545036033862335, -7.7874158819594053, -2.0798485747710065, 
0.48593062666356401], [-10.376667297538809, -10.709777303843696, -10.735774580822937, 7.0782769989541201]]], 
[[[11.468354158082919, -10.466338090388497, -12.288944666479384, -0.21322513807898585], [-3.510926965763792, 
-11.059979868212249, 4.6798213046129975, -12.520888390750915], [2.8438433513788075, 0.21146178529132892, 
4.2323682668163789, -12.466506357778224]], [[3.9241470419266724, -0.93682852047446297, 6.7846814281166719, 
-5.6122049776897045], [1.009636052268762, 5.6683464398043348, -4.1776644489085921, -6.8857385877058555], 
[-12.230758926344738, 6.9660807100382254, 6.6106634513770723, 3.2542787776386608]]], [[[-8.5091611630055262, 
-11.87170630845092, -5.7746582787226588, -8.2330776928059901], [5.353350561609421, -7.3664339060593971, 
-10.653217974099684, 1.8567756202110501], [-10.22573243546206, 12.142378141868068, -4.0273316318104415, 
-3.4031187638502254]], [[-6.3847992476635591, 9.9657018755896978, -6.470634439909837, 8.587003283295978], 
[2.0210408232317629, -4.6230818330415877, -11.118699284078886, 12.171926806847758], [-12.392810502499694, 
-6.2254503155317185, -11.409458438197062, -6.3595509152671994]]]])+(1.-msk_ref)*numpy.array([[[[2.495540296879188, 
10.379335513483992, -12.131447222167042, 18.206047216225837], [-5.0432829716077814, 16.212279934703215, 
12.39670974393805, 0.057842910971599444], [-11.518520219746321, 9.0890803063976193, 18.033438665320531, 
16.742333887671666]], [[3.43060499894721, -14.877301096998385, 4.2916537198201299, -12.751839165684238], 
[-18.392848438720499, -11.417484944606104, -3.0493606800326032, 0.7124450136157725], [-15.213704319978612, 
-15.70209206496787, -15.740207856266352, 10.377784144870411]]], [[[16.814276124980314, -15.345174751614865, 
-18.017381226508029, -0.31261908195608412], [-5.1475298585355107, -16.215539987468492, 6.8612990623305103, 
-18.357444479839728], [4.1694882026791129, 0.31003375015792717, 6.2052677934347074, -18.27771250557068]], 
[[5.753370623933491, -1.3735269427402785, 9.947330312604274, -8.2283092119504264], [1.4802733796476881, 
8.3106207653820228, -6.1250640356231392, -10.095494814878659], [-17.932072464006065, 10.213288057984194, 
9.6921946347836005, 4.7712462660838089]]], [[[-12.475668558412432, -17.405649075135688, -8.4664893922387598, 
-12.070890014125325], [7.8487909682557282, -10.80026410464507, -15.619167856781525, 2.7223032660145061], 
[-14.992411847417884, 17.802493372372791, -5.9046542486134737, -4.9894673457699161]], [[-9.3610448433105873, 
14.611169205765981, -9.4868917262874675, 12.589796434712371], [2.9631399582934548, -6.7781107400101064, 
-16.301631196256576, 17.845816015294734], [-18.169663657162346, -9.1274161195937573, -16.727926428843457, 
-9.3240271137605752]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank1_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([1.8003466734301279, 
3.110968541428603])+(1.-msk_arg0)*numpy.array([-0.057900815820612905, 0.54416620499792501])
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(-1.23860498141)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-2.2299183579697544, 
-3.8532611324133779])+(1.-msk_ref)*numpy.array([0.071716238902929969, -0.67400697222367489])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank1_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.79032144715843966, 
1.1989835373509852])+(1.-msk_arg0)*numpy.array([3.3888677818436879, 2.0461382220071824])
      arg1=Symbol(shape=(2,))
      res=arg0*arg1
      s1=numpy.array([-0.92949574400567592, -1.2179599998896897])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([0.73460042153017635, 
-1.4603139890197456])+(1.-msk_ref)*numpy.array([-3.1499381802216635, -2.4921145086501579])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank2_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[2.9968239240536168, -0.22744917461103942, -3.368130646004337, 2.5739305354263671, 
-3.5520990533562147], [4.2529038750041668, 2.5566407005371827, -0.59374756248383598, 0.27167900571967607, 
3.5828984462512263], [-3.506627820649574, 2.8671976480545798, -4.502344160444026, -3.4451554059919767, 
-0.53368053099846069], [3.3068381259500921, 0.9313565649084623, 2.680662417641706, 0.49678621396386813, 
-4.6856891442661137]])+(1.-msk_arg0)*numpy.array([[-0.85596724799322921, -0.13058599556778994, -0.39878828275994316, 
-4.0930080594310017, -4.4366277751460883], [2.6284949729177862, -0.28953336259360274, 4.6575690245651824, 
-0.75846368797438046, 2.8728013263404817], [3.9714952839792605, 4.7929536309489222, 1.8212316872876864, 
2.7021824250061401, 3.4917451398394661], [-3.9528673242547518, -0.39076547693401587, 4.8178679551326944, 
-4.372708435816083, -4.6827874768603746]])
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(2.10607695069)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[6.3115417917256531, -0.4790254641017731, -7.0935423204623218, 5.4208957733386178, 
-7.4809939428412591], [8.9569428246464202, 5.3844820505972715, -1.2504780558755719, 0.57217689193258392, 
7.5458598343126875], [-7.385228027718342, 6.0385388796403019, -9.4822832603848397, -7.2557623921047183, 
-1.123972265367853], [6.9644555567263726, 1.9615085942275186, 5.6456813303761022, 1.0462699946498486, 
-9.8684219048371684]])+(1.-msk_ref)*numpy.array([[-1.8027328915440832, -0.27502415534822766, -0.83987881052595881, 
-8.6201899329560003, -9.3438794960261902], [5.5358126774666605, -0.60977954141415425, 9.8091987688843929, 
-1.5973828911781676, 6.0503406573173217], [8.3642746773627188, 10.094329167867425, 3.835654078462837, 
5.6910041218650154, 7.3538839566996979], [-8.3250427607485502, -0.82298216409611202, 10.146800651772885, 
-9.2092604487599345, -9.8623107699953731]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank2_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[1.328567726345117, -2.6681448855288812, 3.2894830838559965, 2.8760842998618212, 
2.8603604495641051], [4.0060721329760867, 2.9201617632829659, 2.1721464751857216, 3.3782152232970972, 
-0.56533591383663317], [2.812933639561118, -0.55158819240545931, -4.3300172419246827, 4.4161226276280559, 
1.5663796584927505], [2.1573608096632757, 0.40478964915762283, -2.783180507622979, -1.5186696606391514, 
-3.5877444638062372]])+(1.-msk_arg0)*numpy.array([[1.9794332133759385, -4.6552748863866809, 3.6163220859074414, 
4.9277135493827373, 4.7113876120438363], [2.5672843138256862, -3.4663513236737495, 0.70662767196765763, 
0.51073573024771424, 0.77820357093096604], [-4.9550804650749072, -1.458958922255027, -0.59095436462229589, 
1.6503086087766805, -0.60037112049808439], [2.0892995102876348, 4.3232658919201121, -0.56155298698416622, 
2.2070902567073523, 1.8732215024796837]])
      arg1=Symbol(shape=(4, 5))
      res=arg0*arg1
      s1=numpy.array([[2.6575119197184351, 0.798780755830788, -4.7246974545048115, -3.594253030805985, 
1.5756436650435841], [0.65656829496808733, 0.22663298200663462, 4.5542984252490282, 2.3375554718292921, 
2.7227466581468063], [-1.0177447572471587, 1.532657082535188, 2.6217986442668364, 4.1467735088972546, 
1.5155361075503855], [-2.6181643161762471, 0.56415045659076313, 3.6099145181301697, -1.4000081350932101, 
0.95267578177457679]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[3.5306845689153685, -2.1312627883288111, -15.541812352931064, -10.33737471163186, 
4.5069088220969], [2.6302599498672778, 0.66180496835457081, 9.892603271348559, 7.8967654802351435, 
-1.5392664701290637], [-2.8628484641474969, -0.84539554973300923, -11.35243333453016, 18.312660324289759, 
2.3739049305782052], [-5.6483250889774848, 0.22836226539548773, -10.047043721045087, 2.1261498794140565, 
-3.417957261864017]])+(1.-msk_ref)*numpy.array([[5.260367358833121, -3.7185439923480388, -17.086027753956419, 
-17.711449359812622, 7.4234680444816901], [1.6855974846268469, -0.785589537166827, 3.2181932936796902, 
1.1938731008992738, 2.1188511721101988], [5.0430071650677997, -2.2360837253220716, -1.5493633519903052, 
6.8434560203802217, -0.90988411104533029], [-5.4701294236395936, 2.4389724268900039, -2.027158280413504, 
-3.0899443142752547, 1.7845727593117799]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank3_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.26449277327340059, 2.709494171087842], [0.49156618056888668, 
1.1557293908754884]], [[2.9426463555847633, -2.2855487493843087], [-0.049194778717681409, 0.65490813497401845]], 
[[-2.5662834716418259, -0.6481573871774593], [-4.7039097421222928, -3.6194060480671544]], [[-1.2083023577871419, 
3.399741798797292], [2.1299564643809319, -0.094074363510819659]], [[-4.6384172111976261, -4.5399938401719275], 
[0.25600785550817573, -2.5059486809113416]], [[0.78159444871409178, -3.8859384118790743], [-2.9903682045869107, 
0.31191296888410847]]])+(1.-msk_arg0)*numpy.array([[[1.4148748264368773, 2.6806498611711493], [-4.0435092298014874, 
4.2267476915751718]], [[3.125690778072709, -4.9216068996123754], [-0.39858451763556069, -4.7718632732732615]], 
[[-1.5015409161151947, -4.3392117282980625], [-4.3901880811233127, -2.8392130815499939]], [[1.5905393663877287, 
4.6064612161951075], [-3.1305192294513895, 2.10516821140763]], [[-1.310377796147919, -2.266123007043912], 
[-3.9633905374201128, 2.6610704495417856]], [[-3.3823523217509011, 1.9759866351748991], [3.6164091081703162, 
4.7947187843172081]]])
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(-0.422116691453)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.11164681436744017, -1.1437227150112377], [-0.2074982897720079, 
-0.48785266669154376]], [[-1.2421401437361563, 0.96476827624502515], [0.020765937229078289, -0.27644765514099601]], 
[[1.0832710883803918, 0.27359805181627767], [1.9855988172389765, 1.5278117060356851]], [[0.51004459354416765, 
-1.4350877599033474], [-0.89909017568376171, 0.039710359075749616]], [[1.9579533267701619, 1.9164071790311268], 
[-0.10806518895313126, 1.0578027661377205]], [[-0.32992406274935349, 1.6403194656131641], [1.2622843327469702, 
-0.13166367044669411]]])+(1.-msk_ref)*numpy.array([[[-0.59724228055590678, -1.1315470503419538], [1.7068327379441424, 
-1.7841807511750174]], [[-1.3194062497457233, 2.0774924210974488], [0.16824917784877888, 2.0142831369809828]], 
[[0.63382548359210156, 1.8316536982639513], [1.8531716676608954, 1.1984792323144307]], [[-0.67139321496560267, 
-1.9444641678876049], [1.3214444196665345, -0.88862664035176742]], [[0.55313233986365051, 0.95656834615927522], 
[1.6730133005925627, -1.123282253884367]], [[1.4277473713864239, -0.83409694079570218], [-1.5265466476819478, 
-2.0239308296843248]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank3_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[3.8465426754336836, 0.6522330066175801], [1.1213564084832672, 2.2991484731703036]], 
[[-3.146805277366008, -2.9092773950846063], [-0.7507570800585075, -4.3691574327961602]], [[2.6060037046129327, 
-3.4798742840305676], [-2.2473704638275493, 0.50713868800830042]], [[-1.9023463196744839, -1.0711886388204297], 
[4.0144718529996783, -1.1676181449431819]], [[2.4659181377476571, -3.3224615922640091], [0.19896825249062911, 
-0.68785028874854248]], [[2.1633557767317377, 1.5521876431045971], [4.2492954867619375, 
-4.7770400998496516]]])+(1.-msk_arg0)*numpy.array([[[3.5872129335758469, -1.4939601669977023], [2.6202996623479304, 
-3.4258031677768743]], [[0.71987276551639834, 3.6960268147998683], [1.885662175199954, -0.80213652437160476]], 
[[2.0323879500914588, -3.054629290525348], [0.64860207732057251, -4.2079981974529375]], [[1.2861885512240159, 
0.023168662849575306], [-2.5160035334539876, 2.6814734060795251]], [[-1.257956004629972, -4.3314818687128174], 
[-2.8595244901616734, -2.299231449232094]], [[-0.19237535961527463, -1.5019438253752684], [4.7223884225278852, 
-0.33487628154856086]]])
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0*arg1
      s1=numpy.array([[[3.7703715218725122, 1.8825293253796147], [-3.5569112589449747, 3.7865091325165103]], 
[[1.3234025154170119, -1.0706127854081848], [-0.91030140341539401, -4.0779906813336346]], [[2.4564691769773113, 
-4.2202902733308969], [-1.8831049422439405, 0.41091633330696098]], [[2.5279194148611168, 0.406501938106687], 
[-0.33008527781400687, -3.1142412235060757]], [[-2.6625713201810175, 2.3132871914260473], [-4.863897694906, 
4.1604892545389323]], [[-0.92121754594451399, -3.0027046624253173], [2.5790377788725909, -2.626459339743711]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[14.502894961122463, 1.2278477619381107], [-3.9885652346242333, 8.705746690670745]], 
[[-4.1644900195937034, 3.1147095754765983], [0.68341522360130269, 17.817383296222328]], [[6.4015677754703555, 
14.686079593308524], [4.232034427486715, 0.20839157015447368]], [[-4.8089781952947206, -0.43544025775836859], 
[-1.3251180568739098, 3.6362445602957494]], [[-6.5656829114810957, -7.6858078453893226], [-0.96776122464864589, 
-2.8617937350698126]], [[-1.9929212996456993, -4.6607610729091382], [10.959093593951833, 
12.546701586580348]]])+(1.-msk_ref)*numpy.array([[[13.525125487647125, -2.8124238253222011], [-9.3201733708150698, 
-12.971834980991126]], [[0.95268142866460237, -3.9570135631362282], [-1.7165209244518427, 3.2711052715447542]], 
[[4.9924983550597704, 12.891422283435785], [-1.2213857773520564, -1.7291351898596623]], [[3.2513810098112819, 
0.009418106351692759], [0.83049572532118243, -8.350755020948105]], [[3.3493975799772628, -10.019961526787521], 
[13.908434576224618, -9.5659277382281029]], [[0.17721955668497666, 4.5098937271552346], [12.179218148209955, 
0.87953893733186228]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank4_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.71566615256646759, 3.7745591573508115, -1.8813261752186392, 
-4.2234672474302322], [-4.5183089034463952, 0.90004730988224502, -4.3059651797874992, 2.1824530065115431], 
[-3.2409114328950315, 1.425264269049185, -1.0363956947212296, -4.6538419691451374]], [[-0.021160056002790029, 
-0.67358505413946013, -0.23398482533746634, -3.2333582167367658], [-4.3340836442891923, 1.8672903891253885, 
3.3960333447618343, 1.2033623796151725], [-4.0570605641925273, -0.94056101372721734, -1.7965062790901385, 
1.9686802501306877]]], [[[1.5047353385318765, 0.5056728068922558, 2.409658435923566, -2.1000430599662478], 
[-4.064887735516467, -2.1275738111117271, -0.62603398290270729, 2.4810787925563362], [3.254883998023141, 
3.9921703272954172, 1.7173288579737775, -2.2709735667109623]], [[-1.0333361509972105, 4.6923051335910078, 
-4.2187417938772569, -0.26001229923624436], [4.0524307618320652, 3.1529491855521723, 3.8891703598304161, 
4.9134254224440408], [-2.2621317317299292, -1.4221931139149491, 2.8628397330646003, 2.486038796267632]]], 
[[[1.5100124197717211, -2.2988431057193734, -2.9336531752144488, -2.6131669437218363], [-2.8974794045596344, 
-0.85417257745919439, 4.3573210605245887, 4.2307931931872957], [-0.12423354634863593, -4.6765195903139816, 
-1.5778735854722301, 0.1126115177128213]], [[-4.1436398962257615, -0.76648956746585917, 2.1159176768980608, 
-3.2536216330349363], [-2.2856244596599904, 3.705073565316475, 2.2220301188746312, 0.98343571363908122], 
[2.4860129265223163, -3.1262241511848043, -3.8305865546189102, 
3.1860325494154687]]]])+(1.-msk_arg0)*numpy.array([[[[2.4089227637145711, -2.6908075329445138, -0.26014840610314227, 
-3.9215812800091445], [4.009158789021523, 2.7447580688873892, 4.7131775334041954, -4.4290013383016644], 
[-1.7082394800006173, -0.49659550318463985, 3.9472706938122357, -1.6162682425555799]], [[-3.8246426595124583, 
-2.6800405903617683, 4.0143006613192842, 2.2139414823010402], [-2.5044774188651475, -1.6949103444811264, 
0.41642158825732167, 3.2060207528109235], [-3.5084114918483778, -2.1345709077268706, 1.4430246021304205, 
1.2294065151503952]]], [[[3.5126626365828901, 3.8348878052528992, -1.5523954598502341, -1.2410335237730994], 
[-2.1674583781546621, 3.3708587151697049, 0.56105884392031768, 2.2747443626383266], [-2.0719480048860914, 
1.733990254553726, -0.050490856134826956, -3.7318225551204542]], [[1.4746408782374925, 4.0027176844782097, 
2.8083560519562889, 2.9394382256580087], [-4.6069170484521784, -4.8378490055724885, 1.636720298987302, 
-1.5376228376365288], [-2.6819306123720734, 1.1516574074480923, -3.6392748736610949, 3.6118499182970254]]], 
[[[1.9375861833692003, 2.9438381921906851, 0.57660731138154819, 0.51102545141293199], [-3.0370737652416144, 
4.5219314530432726, -2.0670940828628037, 2.8140667234989287], [-1.3644739955281162, -2.7108624230624456, 
-2.1786469944211206, 2.8077953462368104]], [[4.022901264470141, 4.7164854778488916, -0.17421287538805963, 
-3.832102145875953], [-4.8385573319832922, -4.2618026800252213, -4.5775319725945369, -2.3564870023253039], 
[-4.2941599674913693, -3.8000625353557038, 1.1131824136314092, 1.8132425713432623]]]])
      arg1=Symbol(shape=())
      res=arg0*arg1
      s1=numpy.array(0.717965207903)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-0.51382339801631816, 2.7100021501484619, -1.350726738523665, -3.0323025403715427], 
[-3.2439885912315676, 0.64620265396188303, -3.0915331855279722, 1.5669253265579597], [-2.3268616507127589, 
1.0232901572442055, -0.7440960504300036, -3.3412966169236569]], [[-0.015192184007276179, -0.48361063343539723, 
-0.16799296376949385, -2.3214387043033553], [-3.1117212647398351, 1.3406495324431462, 2.438233786416474, 
0.86397232106270994], [-2.9128283314443832, -0.67529008376584743, -1.2898290041654794, 1.4134439250790429]]], 
[[[1.080347620167597, 0.36305548193114656, 1.7300509199223877, -1.5077578521533153], [-2.9184479681312658, 
-1.5275239736231971, -0.44947061868890531, 1.7813282511207205], [2.3368934663398959, 2.8662393990196935, 
1.2329823705524707, -1.6304800089652003]], [[-0.74189940448410086, 3.3689118307816321, -3.0289098291289465, 
-0.18667978447841244], [2.9095042944301057, 2.2637078175116527, 2.7922890059647112, 3.5276685049395078], 
[-1.6241318790748014, -1.0210851747097522, 2.0554193241418712, 1.7848893612164991]]], [[[1.0841363808970814, 
-1.6504893683335269, -2.1062609118572939, -1.8761629480337436], [-2.0802894030884849, -0.61326619216028622, 
3.1284049211184035, 3.037562314540097], [-0.089195363932689647, -3.3575783599208848, -1.1328583368377678, 
0.08085115172692571]], [[-2.9749892795677022, -0.55031284166088568, 1.5191552747991404, -2.3359871321986905], 
[-1.6409988403673095, 2.6601139126172866, 1.5953403162639133, 0.70607262660183523], [1.7848707876394252, 
-2.2445201726558786, -2.7502278720763003, 2.2874605217258854]]]])+(1.-msk_ref)*numpy.array([[[[1.7295227328719085, 
-1.9319061898166923, -0.18677750447340177, -2.8155589190091503], [2.8784365234748241, 1.9706407975713816, 
3.383897487652936, -3.1798688666551458], [-1.2264565134062644, -0.356538293687512, 2.8340030243311856, 
-1.1604243647929684]], [[-2.7459603621904449, -1.924175899646795, 2.8821282088880955, 1.5895329566247054], 
[-1.7981276507231641, -1.2168866578518498, 0.29897621218834564, 2.3018113563323048], [-2.5189173861531855, 
-1.532547645549204, 1.0360414584772961, 0.88267110424690254]]], [[[2.5219695601663243, 2.7533160203819742, 
-1.1145659290786001, -0.89101889190998895], [-1.5561597050922877, 2.4201592782474872, 0.40282072952090631, 
1.6331873092471489], [-1.4875865800916528, 1.2449446736119429, -0.036250678022026972, -2.6793187566430881]], 
[[1.0587408447256197, 2.8738120345122611, 2.0163019367076376, 2.1104143768017321], [-3.3076061564825183, 
-3.473407267087782, 1.1751082297410078, -1.1039597002996688], [-1.9255328696923644, 0.82684994997116834, 
-2.6128727412832053, 2.5931825775035184]]], [[[1.3911194669720908, 2.1135733996881303, 0.41398398819427723, 
0.36689849446726314], [-2.1805132972775692, 3.2465894558060273, -1.4841016329570593, 2.0204020001890131], 
[-0.97964485587718853, -1.9463049031696793, -1.5641927422961792, 2.0158993695091798]], [[2.8883031427173891, 
3.386272476673903, -0.12507878329731756, -2.7513160138682586], [-3.4739158208065777, -3.0598260472046448, 
-3.2865086943851498, -1.6918756805445263], [-3.0830574538274456, -2.7283126882399662, 0.7992262430365179, 
1.3018450797125141]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_mult_overloaded_expandedData_rank4_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[2.9251174606479946, -4.9654026519140189, 0.30085621199683832, 
-0.30061228591591238], [-2.513978591770071, -3.8844686856929558, -2.1891972059679632, 4.5141042306447474], 
[-2.236512896721278, 1.8541668723658136, 0.26092160236246542, -1.1184332157429031]], [[1.8254032188716751, 
3.8918830207374366, -3.02810273755754, 3.0709862855640271], [-3.2428568515005187, 2.0131925931621293, 
1.9979305988685745, 4.1731959115852639], [-0.67035244538272032, -2.00829150675079, -4.7874365846650164, 
4.1003641988688031]]], [[[-1.175586228847092, -1.8341477845829668, 2.4832839618398461, 0.89234831426067096], 
[-1.7369861501359631, 1.8284772242403307, -0.27353996503704359, 4.8231831031252348], [0.099524325829133176, 
1.2133893916939442, -4.6692295366623551, -4.4818711372137141]], [[3.8966804335291769, -0.47007955430217407, 
1.9640723963394606, 4.8351918103493343], [2.1596571322083662, 3.4394328531876912, 2.846261179352954, 
-1.8012535813987718], [0.41896290835312833, -4.2874267756908147, -0.37745703724418522, -2.6740921738817813]]], 
[[[0.48734325359850583, 4.9758075524770824, -2.053696707710202, -1.2492068717010851], [-0.81009221983657476, 
-0.032340552500626174, -2.7423954288910823, -4.1769441535542455], [-4.1686249915574001, 3.0106427920402847, 
-3.5225347400306015, 3.9203298909772801]], [[-3.7843028776929879, 4.7534900290748308, 4.7905989355194496, 
4.9295960701557782], [0.2236860439332089, 1.1116309427796969, -4.6113096924535757, 4.4416902722827007], 
[-0.78621657417830626, 1.5380907655682377, 4.5160456196047676, 
-3.7405412441349561]]]])+(1.-msk_arg0)*numpy.array([[[[4.3843342392813938, -1.6479745021651206, 1.0928655318928948, 
-0.27867216892680435], [-1.8534416490446235, -0.60606977099904125, 3.7896814633860565, 1.6774705245027342], 
[-2.1712907977922269, -0.9731023323951602, -2.2363323585193129, 3.0179076547691572]], [[3.5949000961561293, 
-4.4826056152295042, 4.4507316837878168, -0.14917077276962942], [2.865875878248751, 0.65613479495564864, 
-3.5114334284145352, -2.2657040605205969], [-3.3885348683628269, -4.1955144800795416, 0.19874114120673436, 
0.77237878029658802]]], [[[3.6482369572987956, 4.1192696915374132, 4.7177908974828782, 3.267869957103418], 
[-3.9817588302026605, 3.9472020063214721, -2.2489247733823725, 0.35633644710087964], [0.30626607261116678, 
1.1534627023800335, 2.6743275744867319, 2.9389573367973432]], [[1.9302124685612068, -3.5741196455396427, 
0.86533807292409826, 3.713340528874534], [-0.75399962875646054, -2.6842573138405124, -3.0230491895158327, 
1.9153830313206104], [1.1778385254967532, 0.11792305960293703, 4.5139117168026992, 2.8139119131046897]]], 
[[[-4.417619341991335, -4.8560564315233137, 4.580720581716939, -0.80247960909749949], [-3.4934853556245651, 
1.0770893472402108, 3.378803127896246, 2.2030520956731934], [3.9240010232090405, -0.580967720625301, 
-1.2967054817990578, 1.780069501914376]], [[-0.19831352998526164, -3.5200058893139854, 0.76880268223223958, 
-3.5461945196094549], [2.6005628531204348, 4.7941479926695827, 4.9792519595550839, -2.3841553080010258], 
[0.78378064155146721, 0.72888520432430148, -0.39840096436977745, 2.3135172058068862]]]])
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0*arg1
      s1=numpy.array([[[[0.6868381582863794, 0.54627711634776066, -3.2186962400562225, -1.0774604195563131], 
[-1.8950462499252776, -3.9532997580259446, 0.94174368416758636, -1.45380459144517], [2.0866672501914945, 
4.3633150084832497, -4.2883759710766967, 1.1324613195015356]], [[-1.6473475256931112, -2.4240842974365195, 
3.0575003656751942, -0.88291292348152517], [-0.26167034327218541, -0.11708643329371515, 2.69348684719085, 
-4.9379016274841749], [0.9686171163405044, 4.8295378077612252, 3.7003121978510229, 4.056897346655898]]], 
[[[-1.6908031207422525, 4.8230189206260157, 0.82952788955472112, 3.4446585748455014], [3.3176619789094879, 
3.0037317679930418, 1.4804656022305664, 1.0055708904902172], [0.88162105416707792, -0.50313800229601746, 
3.9994903610131427, 4.5365056130602301]], [[-4.8084467422206547, -0.19193357971699321, -3.9848748508964316, 
-3.0854097037447001], [-1.4115589742338477, 1.453415069972718, 3.991034445603626, -4.9809560423233554], 
[0.17116217177604565, 3.3177075206574909, 1.7537041099136621, -1.9103533234598826]]], [[[2.5308867325681046, 
-1.042247247233361, -1.1846149285407979, 3.7060742981010737], [-1.297359352173022, 4.2498337462445868, 
1.493118867126439, 3.1157447558047586], [0.15917981571003992, -4.2811882943532051, -2.892893263308518, 
-0.15525299846753171]], [[0.70528939883089059, -3.5415574610175469, 0.91408910363181572, -4.9096466754450141], 
[3.8808985862028376, -2.4998339203436348, -0.7493422147604818, -2.7415281675633221], [-2.6740604084639994, 
1.5889649415442406, -3.0729275401598812, -3.7916787182652412]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[2.0090822894427993, -2.7124858421931139, -0.96836475835178126, 0.3238978397067413], 
[4.7641057027263036, 15.356469115209322, -2.061662642117656, -6.562625456773401], [-4.6668582162192029, 
8.0903141424262, -1.1189299299060254, -1.2665823552745537]], [[-3.0070734760004947, -9.4342525180294281, 
-9.2584252273842349, -2.7114134793590052], [0.84855946551469918, -0.23571754026667907, 5.3813997896526438, 
-20.606830883627179], [-0.64931485257841604, -9.6991197608586983, -17.715009990674201, 16.634756638713682]]], 
[[[1.9876848644362788, -8.8461294682679377, 2.0599533040300941, 3.0738352724669484], [-5.7627329081984522, 
5.4922551255024175, -0.40496650907269482, 4.8500525280070113], [0.087742741052748138, -0.61050231454407089, 
-18.674538525238951, -20.332033570982652]], [[-18.736980336078339, 0.090224051608984959, -7.8265826975130048, 
-14.918547731118739], [-3.0484834062368544, 4.9989235409822532, 11.35952640798204, 8.9719649100247967], 
[0.071710601287329825, -14.224428057977713, -0.66194795753096181, 5.108460871613123]]], [[[1.2334105747390316, 
-5.1860217243322069, 2.4328397786485927, -4.629653480222637], [1.050980717527584, -0.13744197138935588, 
-4.0947223559985781, -13.014291841725987], [-0.66356095792037362, -12.889128679761718, 10.190317019204748, 
-0.60864297055611438]], [[-2.6690287016020968, -16.834758078342482, 4.3790342868285039, -24.202574957127123], 
[0.86810285165369627, -2.7788927376642603, 3.4554490178896384, -12.177018993055025], [2.1023906134884078, 
2.4439723034008711, -13.877480957101886, 14.182930630179902]]]])+(1.-msk_ref)*numpy.array([[[[3.0113280542199465, 
-0.90025075885739858, -3.5176021783907041, 0.30025823205054236], [3.5123576464773363, 2.3959754790373493, 
3.5689085831507947, -2.4387143505360127], [-4.5307613983952022, -4.2459520117298588, 9.5902339496154969, 
3.4176636848536646]], [[-5.922049778516727, 10.86621388347861, 13.608113750703422, 0.13170480308403179], 
[-0.74991472483682653, -0.076824482901260024, -9.4579997542208236, 11.187823767842159], [-3.282192872812852, 
-20.262395804553826, 0.73540426902211176, 3.1334614243985466]]], [[[-6.1684504326080232, 19.867315661446234, 
3.9135391265494457, 11.25669626921629], [-13.210129880150486, 11.856336061073677, -3.3294557689967741, 
0.35832155842535174], [0.27001061779106766, -0.58035091979845577, 10.695947356351342, 13.332596454925692]], 
[[-9.2813238562468232, 0.6859935779052545, -3.4482639243184217, -11.457176901097965], [1.0643149425401714, 
-3.9013400316202884, -12.065093446111812, -9.5404386832200192], [0.20160140002551957, 0.39123422170360572, 
7.9160655295443281, -5.3755659751229006]]], [[[-11.180494182182111, 5.0612114481650323, -5.426389984575974, 
-2.9740490540264393], [4.5323058977990254, 4.5774506556220018, 5.0449546985677118, 6.8641480138584354], 
[0.62462175972042311, 2.4872322049381017, 3.7512305527917205, -0.27636112765281257]], [[-0.13986843034333699, 
12.46630312012565, 0.70275415467140356, 17.410562133681889], [10.092520700006713, -11.98457377122277, 
-3.7311636912234754, 6.5362289327304204], [-2.095876782493292, 1.1581730360816256, 1.2242572954381448, 
-8.7721139535984367]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_Symbol_rank0(self):
      arg0=Data(3.43231999068,self.functionspace)
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(-1.4540852929)
      sub=res.substitute({arg1:s1})
      ref=Data(-2.36046675352,self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_Symbol_rank1(self):
      arg0=Data(-1.99888672106,self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0/arg1
      s1=numpy.array([-1.0166253896000677, 4.6638233509673803])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([1.966197914693572, -0.42859400338170606]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_Symbol_rank2(self):
      arg0=Data(1.75603219348,self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0/arg1
      s1=numpy.array([[-1.9661246795683085, -1.3570192095877909, -1.8523576170966627, 0.73214584413366701, 
4.0725024727547723], [1.8898911955656281, 3.704542947914863, -3.7748480598358505, -2.2414248653554889, 
4.3999361819787648], [-3.981378759357713, 0.34597746243312777, 2.0496740598398322, -2.7919624238592222, 
3.7661073743193256], [-2.3423350580789757, -0.53058380320099552, -0.56646103984835783, 1.7717941659157441, 
-4.6013674001294715]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-0.89314386403468715, -1.294036356358591, -0.94799847355391309, 2.3984732107050015, 
0.43119241921436163], [0.92917105365848052, 0.47402128094425233, -0.46519281455793515, -0.78344459393917454, 
0.39910401443456101], [-0.44106132564159456, 5.0755681631228065, 0.85673728710838137, -0.6289598235553272, 
0.46627247153328438], [-0.74969299862837668, -3.3096226889880782, -3.1000052429972795, 0.99110394833928261, 
-0.38163268454377053]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_Symbol_rank3(self):
      arg0=Data(2.14054149252,self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0/arg1
      s1=numpy.array([[[-1.3253901188041395, -2.78253495112182], [-4.300977060645458, 1.3835241184212173]], 
[[-4.4915480214339922, 2.4570745420262323], [0.3669594834630967, 2.2625027935147521]], [[3.5861676332312769, 
-0.23287586841902996], [-0.090748090278614413, 3.6431882097617247]], [[-2.6410132226145535, -3.1196251341148207], 
[1.761434899422647, -2.1643804473513208]], [[1.5009114368679546, 3.1354061944070661], [4.4611923331014278, 
-2.1935651672343202]], [[-0.031809360087940775, -2.6612631796034103], [-4.8635727802285782, -4.7669299996057299]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-1.6150275018307814, -0.76927748622185721], [-0.49768726090397136, 1.5471660117975328]], 
[[-0.47657099118358148, 0.8711748284031503], [5.8331821058896702, 0.94609451915777054]], [[0.59688829732556048, 
-9.1917703068822263], [-23.587730451975027, 0.58754622854454441]], [[-0.81050010435175179, -0.68615343206318313], 
[1.2152260030870663, -0.98898578350352317]], [[1.4261610911502236, 0.68269989908859252], [0.47981376562515166, 
-0.97582762732427908]], [[-67.292818422177291, -0.80433288557445193], [-0.44011708866066374, 
-0.44903984172213457]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank0_Symbol_rank4(self):
      arg0=Data(-3.54974920415,self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0/arg1
      s1=numpy.array([[[[-4.6125564005877369, -0.69438643429163349, -1.2362005816145647, -0.30244075141731752], 
[-1.1937784959185995, -0.0033389287251761601, -1.6183053769253828, -3.6545432537969038], [-4.6417500711286301, 
1.7794083989940725, -4.5916358946296647, -4.2537071645246352]], [[0.99190852437837673, 4.9497457601229851, 
2.1159345761594448, 4.5666165220115857], [3.2506291129568652, 0.42673325762989656, 2.0122314690593255, 
4.7533864244368562], [2.9256051481580858, -2.1428797577186054, 1.0170937184162421, -0.51756150989596517]]], 
[[[2.7697508281062682, 3.2216955050655187, 4.1307453643983649, -0.6109132252804681], [1.1663708850914603, 
-1.2817940773962224, -0.0804200958300747, -0.059735834319028847], [-1.7817187949510171, -0.93987586758695407, 
4.4482533799407129, 2.9112844484768345]], [[-1.2926950097177148, 4.8418522299498115, 4.081508190552551, 
4.5792771623348134], [3.4344906345489221, 3.5115707226638531, -1.2336261997761913, -4.9918490638148807], 
[2.9678262816222762, -1.7062580143959103, 4.1797990849162208, 3.392276837834407]]], [[[-3.2392133388383293, 
-4.3004082192842166, 4.5627762584330434, -0.079623098707676476], [-1.9104374497872612, 4.7603569268729746, 
-1.5974727715246164, 2.6969700051872891], [0.14884820101978846, -1.787020268808436, -2.4205499696318755, 
2.1558951788757899]], [[-2.1125147886516817, 1.5310662008239184, -1.5322830724155088, 2.2222574246117084], 
[1.1894588535421828, 3.3001015528617934, 1.1758339668590718, -0.75094141518856361], [-1.3255264932208934, 
-0.45235303559717899, -2.4004740260439119, 3.7531870834047236]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[0.76958391309789775, 5.1120658884563053, 2.8714993803941131, 11.737006959260613], 
[2.9735409175867016, 1063.1401555185284, 2.1934977506490343, 0.97132499402245065], [0.76474371729496249, 
-1.994904152501356, 0.77309030716063942, 0.83450718793092105]], [[-3.5787062182711575, -0.7171578857134806, 
-1.6776271082029788, -0.77732587946434517], [-1.0920191386953129, -8.3184264190342141, -1.7640859208948241, 
-0.74678321667685821], [-1.213338446025066, 1.6565321462218994, -3.4900905785521159, 6.858603540403533]]], 
[[[-1.2816131935507342, -1.1018264136283962, -0.85934834781714486, 5.8105620524410613], [-3.043413762742126, 
2.7693599672108506, 44.140076774461704, 59.4241169411154], [1.992317314162168, 3.776827692429984, -0.79800966827936781, 
-1.2193068959669231]], [[2.7460067358999258, -0.73313869064233306, -0.8697150755119526, -0.77517675351608806], 
[-1.0335591451147692, -1.0108721949521922, 2.8774917432793443, 0.71110908177906096], [-1.1960771511898256, 
2.0804293220599646, -0.84926311816258404, -1.046420847661474]]], [[[1.0958676792255069, 0.82544470737243281, 
-0.77798011629176977, 44.581902259059319], [1.8580818778151313, -0.74568971585096155, 2.2221031039929464, 
-1.3161989926923772], [-23.848116267640666, 1.9864067946559978, 1.4665052358695663, -1.6465314450028652]], 
[[1.6803428895353212, -2.3184818541737942, 2.3166406182071824, -1.597361837937973], [-2.9843396377924201, 
-1.0756484754448787, -3.0189204464226163, 4.7270654306071433], [2.6779918940160439, 7.8472982931634876, 
1.478770095254565, -0.94579596627218787]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([3.9883962094707375, 1.3750251611114059]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(3.45871616654)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([1.1531435415409816, 0.39755362825459506]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([-1.2334048137186624, 1.377686460692952]),self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0/arg1
      s1=numpy.array([1.4844200870995552, 1.9547802909252443])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-0.83090011004138475, 0.70477816207204547]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[-2.2020384928131884, -4.3740172257969014, 3.3041701150346512, -2.2405366435075358, 
4.0852398702255694], [0.83572753098331365, -1.3774478891852282, -3.7695866211336106, 3.7407792775265367, 
-3.8975656101620961], [3.4429597608670139, -4.1953097446124099, -3.4659921023383311, -1.6700250836404607, 
1.8624190364059556], [1.5519146057464788, -3.9581877480458152, 1.023175615935096, 0.66728877091972283, 
-2.159854941490873]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(0.937332179697)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-2.3492615963795287, -4.6664537082363049, 3.5250791412093454, -2.3903336426916342, 
4.3583693793011333], [0.8916023039487998, -1.4695408085000394, -4.0216122979493001, 3.9908789632457138, 
-4.1581476605459811], [3.6731479356437879, -4.4757982660618616, -3.6977201651801295, -1.7816790245909124, 
1.9869359835781315], [1.6556719585237416, -4.2228228516861241, 1.0915827260573738, 0.71190212538668518, 
-2.3042577522397214]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[3.5835048539727268, 3.8571075798310233, 3.9190114681880495, 2.1261827038253269, 
3.3617327213332029], [0.75329476297362508, 4.5554315239695242, -4.0352341285409521, -0.2623106802883246, 
1.1321991848119719], [3.601642681322538, 0.35254643689157916, 4.411383875853911, -4.8746220027908125, 
-2.380461998817911], [3.3624236591334427, 3.869796930431475, -3.0974921018602899, 3.9594635826095743, 
-2.2054920280196821]]),self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0/arg1
      s1=numpy.array([[-0.59646319056279751, -3.6970890099277609, -1.8190897467441278, 1.7073115435483661, 
-0.87921199151292839], [3.1618107337278349, -3.9520052671591479, -4.9928921272070461, -2.192093820543024, 
4.5495573770272415], [1.8431133075959449, 3.1217428492223966, 4.9262597144433933, 3.4883151619796173, 
-3.9635009262862084], [1.1267839216226863, -1.8588755230890319, -1.9192169107766244, -2.7631759225358712, 
-2.8239106079831644]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-6.0079228872304471, -1.0432823146734, -2.1543804945316398, 1.2453396170486881, 
-3.8235746939124522], [0.23824789856585635, -1.1526886266637346, 0.80819573620514118, 0.11966215945234733, 
0.24885919463923109], [1.9541081204716173, 0.11293256809394019, 0.89548341572818257, -1.3974144469287193, 
0.60059579727369927], [2.9840891359997417, -2.0817945485670526, 1.613935394414, -1.4329393761421547, 
0.78100631860823788]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[4.0802610727283497, -2.8024539989346686], [1.8071267457266043, -0.13841814957606147]], 
[[-1.7988661169618148, 0.027850264656436252], [-0.50642743316112071, 1.8452982570747869]], [[2.7405547448954763, 
-3.706784288688544], [-3.7888483700042452, -0.58588961440206511]], [[0.7192047060638771, 3.626558913441384], 
[-3.4305483052647547, 0.32455136413554975]], [[-4.74732548737156, -3.237460084387326], [3.9516124747522348, 
2.0449010242900272]], [[2.5313399910511016, -4.0403085459172958], [0.34682792129693585, 
4.8072404359213934]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(-3.09807610149)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-1.3170306148280686, 0.9045788118592859], [-0.58330611854821657, 0.0446787441759516]], 
[[0.58063974480716518, -0.0089895353581049327], [0.16346513661097703, -0.59562715589453463]], [[-0.8845989107815041, 
1.1964794173089723], [1.222968140835246, 0.1891140163149812]], [[-0.2321455905225138, -1.1705841931056031], 
[1.1073156994483895, -0.10475900316959676]], [[1.5323463116641085, 1.044990496789675], [-1.2755052959650881, 
-0.66005513011999017]], [[-0.81706837021672774, 1.3041346995883329], [-0.11194945183245521, 
-1.5516857166971336]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[1.1578386437899475, 4.996126396500328], [-0.91635300388975871, -1.4064554437689711]], 
[[1.9840437948610754, -4.7599852474548303], [3.8573711058777711, 3.2212451543838636]], [[-0.7067030968790311, 
-1.8450312080991926], [-3.4904314203564146, 4.1106376595074803]], [[-1.5537756306237585, -3.2824613533646483], 
[4.7975800514366824, -2.7680700289656945]], [[1.2578485281438132, 3.6884342068649723], [0.30686143258336962, 
-0.1053444702926134]], [[2.8155471322537089, 3.9876357168096845], [-1.6272658547215357, 
-4.1924212908247949]]]),self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0/arg1
      s1=numpy.array([[[-3.55725957171527, -1.1670775665522681], [2.7862539310451577, 1.0863835234493537]], 
[[4.9340858930189082, -1.5697941944512426], [-2.287640013757879, -2.7787914055587528]], [[3.7139854530312846, 
-0.35242947963774185], [-2.1390766886024091, 0.4158275961222877]], [[3.5822790102073565, -2.1203769144295173], 
[-0.76525921225763494, 0.97800877770561545]], [[1.6695611656178597, 2.611002263099742], [0.74990774985468001, 
-0.34139195816695178]], [[-2.9013746777617277, -2.2708713146155524], [-4.3105323025833497, -0.54392296428618003]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-0.3254861278598376, -4.2808863263987469], [-0.32888352123240383, -1.2946214788893002]], 
[[0.40210969932003821, 3.0322352218398869], [-1.6861792426603492, -1.1592252473287548]], [[-0.19028160067300032, 
5.2351784249027027], [1.6317467433282764, 9.8854373731814871]], [[-0.43373942291943912, 1.5480555985244666], 
[-6.2692222120176346, -2.8303120504291575]], [[0.75340068638834046, 1.4126507123307199], [0.40919890832283623, 
0.30857337957883746]], [[-0.97041831716328664, -1.7559936977251271], [0.37750925883244102, 
7.707748277050114]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[0.572941583844365, -2.332247190999662, 1.4999835530459045, 1.5952831694610961], 
[3.2104897575397366, -4.9371677957474267, 2.2024000914918265, 1.4710857974423694], [-0.87463651673330123, 
0.53658487579484948, -1.7854982541271172, 3.5567809651315283]], [[1.0317521940803989, -0.36264412704591908, 
-3.5507235577725971, 1.3909848991573099], [1.5627242518273841, -3.3326845915707382, -4.9229613174667284, 
-4.9895568338934826], [4.1835909015322699, 1.6315284699855113, 0.71636527590831811, -4.6681291758637924]]], 
[[[-0.51377242935229006, -1.8235520070656133, -2.0006476650985006, 0.44472827017699412], [3.9164591306406944, 
-1.6610530974168092, 4.9476376200326353, -1.3836826771010848], [1.4986466482116088, -3.8631080673395788, 
-1.8710856814872723, -1.4352468456143597]], [[1.9960625823999054, -4.210241298814994, 2.6483546110070852, 
-0.86544321226839926], [-0.93317484961197739, -0.44349168612483147, -2.5628503186817273, -3.8695270931809622], 
[-4.9604568128480784, 2.1396464175683629, 1.7039271741506079, -3.8283222528970318]]], [[[-1.441363355020906, 
-3.3811365967582629, -3.351110695562276, 3.7482593597594853], [-2.3746596093071446, -2.8131334479206336, 
-1.2166137380924189, 0.66777797821446239], [-1.3602489200986803, 1.2310997543301729, 4.3550327489506273, 
2.8106898161846896]], [[0.0025102013600246664, -4.1702593487397666, -0.76776964329102171, 1.6998502005468374], 
[-1.2721136787178864, 0.48196474656392496, 3.8101051719028369, 3.2695331075470051], [2.8814364694283192, 
1.768178124165547, 2.7195217125632869, 1.3866661750201548]]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(-3.38430915471)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-0.16929351239873056, 0.68913538461830237, -0.44321705981189319, -0.47137631242701161], 
[-0.94863962208351837, 1.4588406584763218, -0.65076799748788894, -0.43467831400514534], [0.25843871725357914, 
-0.15855078577787907, 0.52758130906604084, -1.0509621912582798]], [[-0.30486345866006281, 0.1071545507422134, 
1.0491723407800817, -0.4110099980732112], [-0.46175576177832739, 0.98474590801808515, 1.4546429100936369, 
1.4743206385109813], [-1.2361727934067384, -0.48208612020977609, -0.21167252847185297, 1.3793447827791936]]], 
[[[0.15181013490950382, 0.53882548068242941, 0.59115393235071689, -0.13140887839924492], [-1.1572403558900153, 
0.49081009490641131, -1.4619342955545513, 0.40885232815524336], [-0.44282202946059263, 1.1414761154314166, 
0.55287079163030139, 0.424088574655207]], [[-0.58979912624731723, 1.2440474869008653, -0.78253920961085932, 
0.25572226788544083], [0.27573569876520421, 0.13104349096103143, 0.75727429189307016, 1.1433728174015798], 
[1.4657221270526268, -0.6322254616098969, -0.5034785819664972, 1.1311975584638549]]], [[[0.4258958886820684, 
0.99906256851586073, 0.99019047680574546, -1.1075404723415438], [0.70166746025578719, 0.83122827121273091, 
0.35948658425561852, -0.19731589157117033], [0.40192809164754195, -0.36376693087158457, -1.286830649879354, 
-0.83050622378023153]], [[-0.00074171751021319377, 1.2322335691269641, 0.22686155672925484, -0.50227391258867338], 
[0.37588577773597637, -0.14241156009429243, -1.1258147520589403, -0.96608582670258014], [-0.85141053541651612, 
-0.52246353490009168, -0.80356775585259854, -0.40973389593840998]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_constData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[-0.16841036162188416, -2.6003560333643625, 3.5545184910572907, 1.5139147914100795], 
[-0.94587340061535929, 0.16930195740283693, 2.4177976471661742, 2.0737743000066882], [4.6768686667116981, 
-3.6339789866072092, 2.5135388182835854, -3.8110207055172829]], [[-3.2003418240897084, 4.104104479479858, 
3.4425399383215289, 0.8930435704987314], [-1.2104858360688286, -0.54602956579953776, 0.96613869262303176, 
-1.1447665449499578], [-0.73618243678854167, -0.75634063494127179, -3.4905332722260507, -1.9294737289274213]]], 
[[[-1.7695622917331444, -4.4618511183236436, 1.8912341467374274, -4.274667165997097], [1.0133441322095447, 
2.2698437785116283, -1.3999716834553122, -0.82282963781866947], [-1.8934325312597222, -1.8347274000416256, 
2.4812263519307525, -0.32095149255687971]], [[1.2294222774888874, -4.7301199904748072, -2.4933570004059069, 
1.739157374421735], [-4.4527415960373729, -0.24978137738396544, -3.5733206736544734, -2.6055729246625714], 
[4.2634437530410096, 0.63371633316309506, 0.35411662272234246, -2.3278320513076158]]], [[[0.57519577741460104, 
-4.1358797158873672, -1.3160752022537814, 3.3667114590653], [4.242351859563227, -4.1169328622744894, 
-1.1012574518657403, -0.017794823889186517], [0.56080815754529567, -0.32077975391617564, 0.1045421322530995, 
-4.5192026724964922]], [[3.7259353163717144, -0.29314631925237755, 0.96174008183977833, 4.8927047445339618], 
[0.19844208500697569, -3.4269276368232626, -1.2296494241640721, 0.74453382811172109], [0.95832215119780528, 
-1.2319028512539818, 1.7600259249758201, 0.51820978084550617]]]]),self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0/arg1
      s1=numpy.array([[[[1.5896505942866401, 0.6891827619618871, 4.6694699071013428, -1.9615140167099101], 
[2.3734917018897859, -3.8358457650850006, -2.4207628518454594, -3.6880303333340336], [-3.955550166922265, 
-4.8219514951045745, 3.0332138878682766, -0.43173075751425927]], [[3.6491269792507204, 3.5362867736239174, 
-3.727772234197988, 0.50203076783315659], [3.0857009585880579, -2.4481643817771017, 3.9495322686066388, 
3.9076676035973037], [-2.5661687086956961, 0.51956619090885425, 0.12707753474037187, -3.8065370519275588]]], 
[[[-2.7343322214319321, 2.1071856335550851, -2.4809811090224079, -3.7158191036172306], [4.1081487764099194, 
-3.9067998805821045, 3.9727737524493261, 0.18115530713208194], [-1.8658727247580997, 1.4129364110321907, 
-0.60367775527123779, -4.9370475380777705]], [[0.85946374601468811, 1.2407326725323173, -0.072139226735624007, 
-2.4294954216721942], [4.3152150682384747, -1.395544165290584, 3.1948762392822694, 3.0548315771279171], 
[1.108458866445222, 4.708696707365803, 3.5605248213884586, -4.016164807819588]]], [[[2.3052524453408676, 
-1.4229694054870725, -0.33634510639310111, 4.763489013156013], [1.6475037257860015, 4.4709813704960855, 
3.9425928470200127, 4.19691117466847], [1.4825715910251436, 2.8996239189360375, -4.7982559296213081, 
-0.63001732867952054]], [[-2.2623123840312722, -3.1391124653197631, -1.6105305432525729, -3.7485403170057396], 
[1.8614704354355895, -1.3878572051796354, -1.1974399594213905, -1.1968610566940283], [-3.0520711866611725, 
0.39756329287125247, 4.6967348695454127, 1.9766807360255321]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-0.10594174734194262, -3.7731008041494896, 0.76122527005722196, -0.77180931592291213], 
[-0.39851557090435591, -0.044136800010019503, -0.9987750949346299, -0.56229860184798586], [-1.1823560489312861, 
0.75363242253609575, 0.82867180199088575, 8.8273087779515276]], [[-0.87701574713271235, 1.160568908067954, 
-0.92348451623203176, 1.7788622286105038], [-0.39228877078960939, 0.22303631645975486, 0.24462104039572191, 
-0.29295392061907044], [0.28687998349209093, -1.4557156492770218, -27.467744628170902, 0.50688426320462898]]], 
[[[0.64716433426163888, -2.1174456807566329, -0.7622928444959578, 1.1503970044816891], [0.24666685345682601, 
-0.58099822051121464, -0.35239149538585995, -4.5421227279791312], [1.0147704643172784, -1.2985208574965552, 
-4.1101835048003634, 0.06500879120192582]], [[1.4304527482278198, -3.8123603054803912, 34.563123466010623, 
-0.71585126644309238], [-1.0318701445059233, 0.17898493189712508, -1.1184535506318147, -0.85293505022370875], 
[3.8462805270471452, 0.13458423265439332, 0.099456299418312022, 0.57961566885284688]]], [[[0.24951530951724002, 
2.9065134499302072, 3.9128715632794941, 0.70677426771993568], [2.5750180671301726, -0.92081190260421242, 
-0.2793231496623142, -0.0042399810595448685], [0.37826716830417445, -0.11062805483887708, -0.021787527340449774, 
7.1731402721389852]], [[-1.6469588119976495, 0.093385096103116669, -0.59715730686949942, -1.3052293241552109], 
[0.10660501570659631, 2.4692220669630798, 1.0268986052196265, -0.62207206421125749], [-0.3139907599095571, 
-3.0986332826580223, 0.37473393194667798, 0.26216159817870188]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank0_Symbol_rank0(self):
      arg0=Data(4.14743871401,self.functionspace)
      arg0.setTaggedValue(1,-1.21319155942)
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(1.68574433632)
      sub=res.substitute({arg1:s1})
      ref=Data(2.46030114096,self.functionspace)
      ref.setTaggedValue(1,-0.719677078714)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank0_Symbol_rank1(self):
      arg0=Data(4.5956316483,self.functionspace)
      arg0.setTaggedValue(1,-2.78021827332)
      arg1=Symbol(shape=(2,))
      res=arg0/arg1
      s1=numpy.array([-3.1585434961186776, -0.83641144105439835])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-1.4549844426550103, -5.4944629194795436]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.88022162010414895, 3.323984030895371]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank0_Symbol_rank2(self):
      arg0=Data(0.921054948699,self.functionspace)
      arg0.setTaggedValue(1,0.0394823576247)
      arg1=Symbol(shape=(4, 5))
      res=arg0/arg1
      s1=numpy.array([[-3.4132432835484785, 1.7587931443963667, -3.0756623852847209, -2.3252904186343857, 
1.7356444015043548], [2.2713204269584573, -0.88162393927914628, 4.5884799751444127, -0.28536103244470201, 
-0.96415679848996039], [-4.9028675399283088, -3.5992342473478436, 2.9789784497386043, -1.4541535741896947, 
-1.7122334747232371], [-3.706048617922951, -1.0579893940366314, -1.4083276675366196, 2.9524896646883523, 
-3.7595222430650335]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-0.2698474360553203, 0.52368577375522674, -0.29946555678715542, -0.39610318836626168, 
0.53067030775444735], [0.40551519625612281, -1.0447254295881458, 0.20073204060776384, -3.2276829839304844, 
-0.95529580887790655], [-0.18786045945513499, -0.25590302975620283, 0.30918483105489369, -0.63339592533191935, 
-0.53792602603303119], [-0.24852748672649719, -0.87057106043797383, -0.6540061449688519, 0.31195873764246379, 
-0.24499255201844189]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.011567402128943166, 0.022448551013809145, -0.012837025875668295, 
-0.016979538258239214, 0.022747953204296133], [0.017382997641416348, -0.044783672341068809, 0.0086046703567616654, 
-0.13835931727073011, -0.040950141809460834], [-0.0080529113428377385, -0.010969654907516195, 0.013253656678242865, 
-0.027151435945629462, -0.023058980102641378], [-0.010653491547244584, -0.037318292458565619, -0.028034922933652444, 
0.013372564211460894, -0.010501961438730782]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank0_Symbol_rank3(self):
      arg0=Data(4.34795846919,self.functionspace)
      arg0.setTaggedValue(1,-3.37152427446)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0/arg1
      s1=numpy.array([[[-0.7964283399534855, -4.3563885775398905], [2.2670305253314744, 3.660342998126211]], 
[[0.98970416783162918, -0.36362420293442987], [-1.5082103582776121, -1.0358411207289953]], [[0.15017273952708887, 
1.5640231629694314], [-1.220547262103826, 1.2530487741570369]], [[-0.17882026193937506, -3.7724811453653215], 
[-1.2436803428756193, -0.22774847919000152]], [[2.699497971812451, 1.4296834805633196], [-4.8957592859528791, 
-2.146219757066874]], [[-0.72865666362676773, 4.5771813067946603], [2.6396990597187395, 4.5439155054068241]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-5.459321637707057, -0.99806488604065802], [1.9179090976533884, 1.1878554745872083]], 
[[4.393190016282583, -11.95728566498973], [-2.8828594402153405, -4.1975148332894978]], [[28.953047556323426, 
2.7799834248849646], [-3.5623024230104048, 3.4699036133818302]], [[-24.314685718698094, -1.1525461100137917], 
[-3.4960418037466843, -19.091053800468391]], [[1.6106544678275652, 3.0412035449119168], [-0.88810707700979552, 
-2.0258682527148579]], [[-5.9670880487799733, 0.94992052482972167], [1.6471417274567848, 
0.95687485033943109]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[4.2333052521197194, 0.77392643343262635], [-1.4871984460681718, 
-0.92109517501174554]], [[-3.4065980361069386, 9.2720018284097723], [2.2354469692890762, 3.2548662212688542]], 
[[-22.450974025508192, -2.1556741321279111], [2.7623053847588577, -2.6906568555004609]], [[18.854263146114384, 
0.89371534132229213], [2.7109251133345817, 14.803718059734445]], [[-1.2489449185243982, -2.358231259085152], 
[0.68866218241893562, 1.5709128868843167]], [[4.6270410232452646, -0.73659399715217178], [-1.2772381238114758, 
-0.74198656873137026]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank0_Symbol_rank4(self):
      arg0=Data(3.7553962803,self.functionspace)
      arg0.setTaggedValue(1,-2.45958468224)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0/arg1
      s1=numpy.array([[[[0.30209630840780743, 2.8852270658365136, -4.3903415927390474, 4.4845847234931409], 
[4.0134940566526396, -2.8333839823358611, 4.6472411139448848, -2.1676319616841253], [1.9906320590044926, 
-4.6737915213906014, -1.1957812424640659, 2.9170526831938375]], [[3.0964720689860776, -4.4876401431960087, 
4.998637079152628, 2.8533787134807831], [3.7729255177559757, -0.73273448472817293, -0.90953567432588045, 
-3.0134320952733864], [1.2174883207447538, -4.969203447194892, 2.3012148245460748, 0.41593532301908454]]], 
[[[4.3069241488554848, 0.71234169023258431, 4.8556620140326565, 1.8082145124352156], [-3.5374500772725535, 
-1.7901952523193723, 4.6526679694301283, -4.4376188651696005], [2.7452568592588422, 0.53836096412756973, 
-2.8204068732647301, -4.8795271769078621]], [[-3.393174985031282, -0.64606900578155368, -4.4624221196219267, 
-2.3987951994472487], [1.4375945710104423, 3.8286850742584093, 1.170192112555676, 4.7719612212908586], 
[-2.396723494110776, -0.28740503184500277, -2.524991852119526, -3.3537880662496655]]], [[[0.024623766163323957, 
4.327465668810266, 0.68752296336131913, -4.9843602135399125], [2.0216870258996078, 3.7344481342322595, 
3.0194814105721566, -1.7317988533225126], [-1.6115633973347188, 4.7297708204656868, -2.137484390559826, 
2.23042822074654]], [[-2.2883581231955787, 0.20442903998765516, -4.1920530024020906, 1.7748267430388616], 
[3.2690041254837414, 1.4972269287755724, -0.53465270543508492, -1.4714871726382519], [-3.1922247493051148, 
-2.1523014413616917, -3.9955444358316869, 3.5074148206456552]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[12.431122710808555, 1.3015947080098942, -0.85537678583155596, 0.8374011222547626], 
[0.93569249817995759, -1.3254102880908314, 0.80809155114224152, -1.7324879622931901], [1.8865346126183558, 
-0.80350102547628632, -3.1405378734334621, 1.28739405425753]], [[1.2127983707372514, -0.83683097585119681, 
0.75128404419717099, 1.3161226242269759], [0.99535394023185464, -5.1251802099813597, -4.1289158702687541, 
-1.2462189827307262], [3.0845439880708629, -0.75573405681742334, 1.6319190369549657, 9.0287986435992362]]], 
[[[0.87194390950620149, 5.2719029811010483, 0.77340561790478524, 2.0768533016816324], [-1.0616111035537803, 
-2.0977579263682209, 0.80714899601135559, -0.84626381724113975], [1.367958071986602, 6.9756102885086664, 
-1.3315086968117844, -0.76962298684838992]], [[-1.1067499603958972, -5.8126860237735665, -0.84156007200365257, 
-1.5655343487285083], [2.612277728386359, 0.98085797276682929, 3.2092134616238446, 0.78697124853915967], 
[-1.5668875819540604, -13.066564131434559, -1.4872904548770602, -1.1197476424022073]]], [[[152.5110438180339, 
0.86780498511318693, 5.462212144798019, -0.75343597160138298], [1.8575557107454101, 1.0056094355349154, 
1.2437222720268766, -2.1684944952439889], [-2.3302814437896617, 0.7939911726906822, -1.7569233707087559, 
1.6837108880566276]], [[-1.6410876611635838, 18.370170307146662, -0.89583702261107834, 2.1159227485324705], 
[1.148789091767819, 2.5082345288638859, -7.0239919149828065, -2.552109423805994], [-1.1764197621477614, 
-1.7448282141760787, -0.93989601182298488, 1.0707020618702912]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-8.14172372777319, -0.85247525623201714, 0.56022626720078217, 
-0.54845316431455637], [-0.61282878397679552, 0.86807319359829038, -0.52925695524084293, 1.1346874034490511], 
[-1.2355797602629599, 0.52625040526085098, 2.0568851516421454, -0.84317458385549982]], [[-0.79431838151277434, 
0.54807974876632271, -0.49205106177732016, -0.86199026810434565], [-0.65190385303429499, 3.3567202492850878, 
2.7042201330470084, 0.8162071035529318], [-2.0202121370100472, 0.49496558319119838, -1.0688201101440833, 
-5.9133825528054009]]], [[[-0.57107685142078468, -3.4528158550336459, -0.50653951513273332, -1.3602283718672756], 
[0.69529876846570882, 1.373919788386156, -0.52863963179770801, 0.55425775781276576], [-0.89593972743977257, 
-4.5686534613860141, 0.87206732672207277, 0.50406209312172789]], [[0.72486231717691918, 3.8069999647501405, 
0.55117705503954395, 1.0253416726877276], [-1.710902873337762, -0.64240976589408783, -2.101864006641482, 
-0.51542428116613925], [1.0262279684244842, 8.55790403684715, 0.97409608675446258, 0.73337510708806808]]], 
[[[-99.88661628455435, -0.5683660762380438, -3.5774582280300198, 0.49346045969051788], [-1.2166001219412634, 
-0.65862065660798974, -0.81457189092959847, 1.4202484760385075], [1.526210316208666, -0.52002195784918936, 
1.1506912953840782, -1.1027410159889803]], [[1.0748250709997458, -12.031483796945189, 0.5867255687910351, 
-1.3858167800791799], [-0.75239571068836386, -1.6427601153607181, 4.6003408516094462, 1.671495836301927], 
[0.77049232914190546, 1.1427696116211459, 0.61558186168052509, -0.70125286229581629]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([-0.24327728711711405, -3.6056280310023405]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([1.2730035138874616, -4.9781403275646703]))
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(3.2936567863)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-0.073862367241458157, -1.0947188079822001]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.38650156846396272, -1.5114326265776077]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([-0.60463512658968011, -0.5520360523301111]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([0.11065887825991361, 3.0681695057704825]))
      arg1=Symbol(shape=(2,))
      res=arg0/arg1
      s1=numpy.array([2.8726661800393423, -0.60963867151620033])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([-0.21047872906047141, 0.90551350844783396]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([0.038521314808112543, -5.032767193294676]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[4.6512814709921759, 2.098614110043922, 1.895016528010288, 2.6977915277647391, 
3.6655404112999896], [-0.48363850300250633, 2.3419850697805096, 2.1662412645385327, -1.9056051572421295, 
2.497215246339004], [4.0815840480205914, -1.2641264348648407, 4.6318841341169641, -3.2591414662330562, 
3.1161047733410481], [1.1685908100777258, -3.1575041951677232, -3.924192536790263, 3.2452072591922736, 
-2.7913771333657267]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[1.6044809956172026, -0.70199805786625014, 2.4770184375160351, 
0.12526292043311571, 2.8453902277172185], [-2.6270109040291056, -2.5010942322759599, 4.1068958534289663, 
-2.6324008007168609, -0.32418555844239805], [3.1579467104423316, 1.086387193462234, 2.7518638232041726, 
1.2174922150915348, -2.6071828689039336], [2.8178725134119729, 2.8056937296183913, 3.6699288938229007, 
3.6037027182613244, 0.085370105743285407]]))
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(-1.90169499965)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[-2.4458609145305559, -1.1035492602308241, -0.99648814786972451, -1.4186247154601728, 
-1.9275122519554746], [0.25431970063157006, -1.2315250711689083, -1.1391107748308005, 1.0020561433862574, 
-1.3131523439899424], [-2.1462874166364903, 0.66473668758670812, -2.4356608893536356, 1.7138087163503437, 
-1.6385933464205495], [-0.61449959656795783, 1.6603630948998709, 2.0635236131561041, -1.7064814598537041, 
1.4678363953659475]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-0.84371100303462776, 0.36914334738060162, -1.302531919144355, 
-0.065869090709308312, -1.4962390016522069], [1.3814049595325335, 1.3151920958627077, -2.1595975454495164, 
1.3842392188058443, 0.17047189928077702], [-1.6605957900874224, -0.57127309777041291, -1.4470584524419186, 
-0.64021423799201049, 1.3709784531112286], [-1.4817689029716252, -1.4753647299597337, -1.9298199209159168, 
-1.8949951064352, -0.044891586589426921]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[-4.6443072321046044, 0.38417837617679673, 3.4279882680004921, 0.40710770023483178, 
2.8688000401894911], [-0.63862918185682371, -1.0613480573447367, 2.5252641176738813, -2.3743507769422569, 
2.2710813572310951], [2.0244163796382626, -0.45931604591878106, -4.2337337609916501, -3.5735365306689681, 
1.5100285975528873], [1.9953319494326704, -1.458603065344283, -2.6914617917833263, 2.8267445415419745, 
-2.1176995723917171]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[2.558345522140046, 3.8739768438957185, -2.5523095986149191, 
-1.1599373064608165, -1.9295185061392859], [2.6353186995223199, -3.9407961905622004, 3.7986447608802525, 
3.1282445710581239, 0.37606431107885818], [4.6848458369965034, -2.5423454969446944, -0.10041273851269938, 
0.39611652325263069, 2.5670890012870329], [2.034125316253621, -4.3475018844119351, -0.69889862294772787, 
-4.8039063806121574, 3.7327807333050149]]))
      arg1=Symbol(shape=(4, 5))
      res=arg0/arg1
      s1=numpy.array([[-1.6000167654125255, -0.94021602909627866, 2.5871609076265356, -4.2349329811309033, 
4.7526174859280115], [-4.0174826369961654, 1.9310149690886282, 1.3522987503107187, 4.9900274807319445, 
-3.1685192513299363], [-2.6079018141064014, 1.653494600107277, 1.6560499898502972, 4.7083848545021212, 
-0.40099662839500461], [-2.6239863890404425, -1.1141605513112127, -0.20010931379470431, 3.50058742814422, 
-0.89214598784888999]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[2.9026616048658607, -0.40860649498399121, 1.325000025276871, -0.096130848362591351, 
0.60362527568938573], [0.15896252443652648, -0.54963222674843171, 1.8673862688208871, -0.47581917857373879, 
-0.71676426023854656], [-0.77626249910483291, -0.27778502928826082, -2.5565253385704674, -0.75897290495529057, 
-3.7656890123909541], [-0.76042008364316904, 1.3091498021786081, 13.449957629381233, 0.80750576855054512, 
2.3737141692446966]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[-1.5989491969357201, -4.1203050405546966, -0.98652912970782747, 
0.27389744102894042, -0.40599070130352011], [-0.65596268550216397, -2.0407900786093434, 2.8090277832523585, 
0.62689926721591294, -0.11868771538030298], [-1.7964042249043672, -1.5375589958260218, -0.060633881300754963, 
0.084130022394806395, -6.4017720337496291], [-0.77520421780749937, 3.9020425550837601, 3.4925841765903027, 
-1.3723143555819919, -4.1840469879883226]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[2.6594060153757653, 3.0985169336341105], [3.3661217116389821, -0.59504905070450942]], 
[[-3.9749579000522637, -4.7342067957921277], [1.1895841648424295, 0.56731685724203196]], [[0.20933515875849551, 
-0.47657164361986748], [3.6005053961204521, -2.7248219369010487]], [[-4.5417601893012947, 4.6881992583085577], 
[0.95219997618841479, 0.045100875229370452]], [[1.6875122232100468, 0.83285808001415162], [-0.98095523355696734, 
1.3721074229557644]], [[1.7539561105027381, -0.13330491004776146], [-3.3240848531330958, 
3.9526032719641933]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[4.6963315268174881, 1.5142976851258796], [4.5399300108016529, 
4.2558918208130496]], [[-3.5130795594275552, -2.2965838545821149], [3.5486672192586823, -0.16210635982120003]], 
[[1.2552491212091041, -0.47844908361541538], [4.446197400169801, 1.9447260435327358]], [[-4.5600746555661678, 
-1.6955914419145826], [-2.0546718268066022, 4.4738577557187309]], [[1.0326084184938722, -1.7805867888057159], 
[2.4947632536764397, 0.81887034400060177]], [[3.2736611846490149, 3.9295086162843287], [-0.83952910002973979, 
-0.69339229981178008]]]))
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(3.40074527236)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[0.78200682567782176, 0.9111287925088859], [0.98981883147769634, -0.17497607231601478]], 
[[-1.1688490556353195, -1.3921086163884109], [0.34980101994451862, 0.16682133232772797]], [[0.061555671475896641, 
-0.14013741267054175], [1.0587401018787967, -0.8012425861615583]], [[-1.3355190776025851, 1.3785799531693437], 
[0.27999744171609176, 0.013262056289824579]], [[0.49621835452551266, 0.244904576295006], [-0.28845301691082309, 
0.40347256647195434]], [[0.51575639162366171, -0.039198734210198434], [-0.97745775908348687, 
1.1622756059065873]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[1.3809712726765795, 0.44528406682920152], [1.3349809077741897, 
1.2514585715682693]], [[-1.0330322556006573, -0.67531781143652669], [1.0434969205434828, -0.047667892428998465]], 
[[0.36911000991810128, -0.14068947989263214], [1.3074185344927436, 0.57185289922762761]], [[-1.3409045048539983, 
-0.49859407456847055], [-0.6041828076649105, 1.3155521503132983]], [[0.30364180078027336, -0.52358722756407616], 
[0.73359309618224211, 0.24079143788171639]], [[0.96263051845069647, 1.1554845487028675], [-0.24686621101955039, 
-0.20389421855490472]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[0.37903407908351117, 4.7562512290710508], [-2.0567133710230481, -2.0597757209890579]], 
[[3.1071466326239037, 3.7378759300098636], [-2.2870602196502565, -3.6754721739328113]], [[3.300935107148554, 
2.1910214216355826], [-2.2941648800534375, -2.0181827356997148]], [[-3.5358995232538684, 0.077598647462692405], 
[1.0254061925828246, 1.3424636637528886]], [[-2.5177374705422064, 3.3774425425435926], [3.7215367528674541, 
-2.5394983441996635]], [[4.9947523199127613, 0.074465686539016751], [1.6903291082153435, 
-1.548003996221603]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[-4.1180675764846963, -3.2588333429017147], [2.3910605684413362, 
-2.5464559979058099]], [[-0.47366800112006935, -2.9943461140704954], [0.68865869503993338, 3.872833966837911]], 
[[2.8476798086608648, -3.2531482133842138], [-2.5572122994418356, 2.4221777629945427]], [[4.3150298961939555, 
3.6112269569743987], [0.73197953589044573, -3.7189842913799733]], [[-2.7850398483841654, -1.3377438559149857], 
[-1.4556441129183781, 3.6956035182385243]], [[2.2019800882262039, 3.8700529933760066], [1.7561308799879001, 
-3.6205253465229568]]]))
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0/arg1
      s1=numpy.array([[[-4.6296995406344124, -2.608733545501265], [-3.2421078749876298, -0.82737098161092959]], 
[[-1.4709282084188935, -3.8180830267668897], [-3.8279718065351354, -2.1824375379963898]], [[3.32588617695232, 
4.0812665637977101], [3.7989606569590251, 3.748273907432111]], [[-2.888051956795227, -2.7465995899159523], 
[4.8196962245729438, 3.6666593093549196]], [[0.56914384990489708, -1.5780278215915988], [-0.33162965001135003, 
3.2519356905168753]], [[1.400833424519293, 1.2066487950482028], [1.140741354057325, -0.80768829487616767]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[-0.08187012477953845, -1.8232031543708858], [0.63437536637515357, 2.4895431031175148]], 
[[-2.1123713685277594, -0.97899283588263286], [0.59746004809799647, 1.6841133411346645]], [[0.99249791830620315, 
0.5368483992373162], [-0.60389277152711029, -0.53842989747842174]], [[1.2243199139594205, -0.028252624717338932], 
[0.21275328253155254, 0.366127188399478]], [[-4.42372779915467, -2.1402934069547039], [-11.221966289021758, 
-0.78091899283408828]], [[3.5655576405358471, 0.061712808933805818], [1.4817812137723205, 
1.9165858983494843]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.8894891645431473, 1.2492013024946684], [-0.73750185392904588, 
3.0777680804658418]], [[0.32201979566984912, 0.78425379780336379], [-0.17990171554143927, -1.7745451585264647]], 
[[0.85621685684695914, -0.79709280502302848], [-0.67313471508515732, 0.64621151570375557]], [[-1.4940970456023919, 
-1.3147992048906205], [0.15187254585849005, -1.0142704782774759]], [[-4.8933847723199335, 0.84773147698101881], 
[4.3893666108219174, 1.1364319193074606]], [[1.5719071587557463, 3.207273739681153], [1.5394645541180672, 
4.482577461492177]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[-0.18987391835747758, 4.5085269431466148, 2.3332490022782091, 3.5140443207840626], 
[-3.8071577551437374, 4.4720793176524474, 1.5612407831467943, -3.8104362786352852], [-4.0532150845521837, 
-0.57280649274337314, -0.56128092769382665, 1.5156021557578434]], [[1.7447218212497937, -2.3056218481816462, 
-3.1637924745560939, -0.30131377200161236], [-2.7752874159121497, 2.6013067905592049, -3.746096460635143, 
-2.9734953480155388], [-3.6161482942867931, 1.9377292214084427, -4.7468533933334172, 3.2118802123097385]]], 
[[[-4.4843124086103083, 2.3827156971613297, 2.1743800991624589, 4.4296694534725685], [-3.4871548142175457, 
2.955544818380722, 3.0587638749887915, -0.51089762690949225], [3.650413980256328, -3.6682136669452814, 
-0.46817705349461569, -0.82910345294884724]], [[1.588186425815735, 1.5765955566881908, -4.3427217875858659, 
-2.0163319014518422], [-1.0220958643238132, 1.9727101787163654, -0.065010798608873266, 1.73315365618957], 
[2.738465567903118, 1.9645854229284678, -2.7935410540423575, 3.0569496206524445]]], [[[-0.21196663843321328, 
-3.1770386067640386, 3.7992231608336073, -4.5172564500621428], [0.98964989530832703, -2.4738974761647881, 
0.0114065763112281, -3.1306195317462948], [-1.7022084431469064, -4.8835685460571892, 3.3149881663472325, 
-1.8527394999613223]], [[-4.688839386407393, -2.2094627892488683, -1.6723044389697153, 3.4908352481955269], 
[-2.4719833755226062, -2.3617449879099341, -4.2339862392091119, 3.802095592491435], [-4.4231212186316329, 
-1.5324562390597976, 3.2464993080986027, 4.0812450056830585]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[0.34365977550907534, 3.4152342295544518, -1.2774419454539219, 
-0.072154050581789697], [-3.2136192243029962, -4.3100480205097416, 0.92063187364718946, 0.65983261161884776], 
[1.836360802359561, 4.7090663804834136, 3.6687227420314681, 0.14984304098941781]], [[1.3240710521065289, 
4.8518181175906356, 4.2513462649612741, 2.7178860490952372], [3.6396852376664448, 3.5569808409457586, 
3.2627252762282897, 2.4127574016450257], [4.0085015954616114, 0.70935755386425026, -1.9722472564280968, 
-4.9450465085805737]]], [[[4.6143116204743464, -1.5718311397775753, -2.8294250621955639, 0.95996522458350686], 
[2.3480451792784409, 2.218190909075064, -4.7734664792996711, 0.0048653152667395005], [4.094925294988629, 
0.098020816683218825, 4.9363224824892118, -2.7422959820113979]], [[1.9859137713910258, -2.1627818319757597, 
-4.0065126947434919, 1.1169402196268106], [0.60687417854650949, 4.7896845328736148, -1.0372629333665175, 
-3.4736146138727317], [-0.77834611882985527, 3.7473488660222962, 4.4949031076591055, 1.1504392368842806]]], 
[[[3.9105694358071688, 0.85300656236961014, -4.4195308596302096, -1.6714210722352862], [-1.7040870582658831, 
4.4180235769979337, 1.0471725443882738, 1.0499359823818599], [-1.2249935138047685, 1.3155129039735147, 
2.6341447922360262, -4.9309444752348908]], [[-4.8684443672971778, -2.2231549054199862, 2.0972135466603978, 
2.2253917338571325], [4.4575262452243205, -1.0296343337965821, 2.3085756197272289, 3.1003426493793196], 
[2.1929744355036931, -1.4027552428126278, -0.58004112631573967, 2.9911401359981511]]]]))
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(-3.1632147744)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[0.060025616943284218, -1.4252990279490751, -0.73761953224333543, -1.1109091767094752], 
[1.2035723233069942, -1.4137766913094203, -0.49356142231691552, 1.2046087763224773], [1.2813594313465078, 
0.18108365495100007, 0.17744003102042699, -0.47913349672732097]], [[-0.55156603192733833, 0.72888564723490989, 
1.0001826307086923, 0.095255552812962635], [0.87736294050361474, -0.82236173515993394, 1.1842687670000487, 
0.94002322323480525], [1.1431877226776648, -0.61258224926441518, 1.500642141580049, -1.0153848035560178]]], 
[[[1.4176439882937615, -0.75325764043739629, -0.6873956573419292, -1.4003694878140707], [1.1024084872260629, 
-0.93434844902089775, -0.96697951076334454, 0.16151215246090378], [-1.1540202738679435, 1.1596473614855805, 
0.14800672318666852, 0.26210785927619656]], [[-0.50207985833555446, -0.49841558956020954, 1.3728823672459796, 
0.6374312353906596], [0.32311933814789817, -0.62364092210289068, 0.020552129161448576, -0.54790894068148144], 
[-0.86572229937262646, -0.62107240988751689, 0.88313353764356695, -0.96640596313356297]]], [[[0.067009878731175476, 
1.0043701845590174, -1.2010639276158939, 1.4280587226072361], [-0.31286206150703538, 0.78208330847024454, 
-0.003606007534974082, 0.98969553287446743], [0.53812610415292728, 1.5438624609305427, -1.0479807419892684, 
0.58571410166503546]], [[1.4823019367366943, 0.69848649137897711, 0.52867242923360025, -1.1035719978443639], 
[0.78147819602019131, 0.74662808451185125, 1.3385073544404151, -1.2019720011621717], [1.3982993676015798, 
0.48446164688593624, -1.0263290796352258, -1.2902206447417288]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[-0.10864256777323951, -1.0796719391910867, 0.40384293718917252, 
0.022810354568944386], [1.0159345645167126, 1.3625530758742546, -0.29104311256316334, -0.20859557718270469], 
[-0.58053623712851588, -1.4886963789477379, -1.1598082974708024, -0.047370492260626021]], [[-0.41858398703187571, 
-1.5338250683628583, -1.3439954502512519, -0.85921641207896848], [-1.1506285526744569, -1.1244828741103539, 
-1.0314586611802425, -0.76275484711676045], [-1.2672239735039237, -0.22425209935319021, 0.62349457659008678, 
1.5632977401979911]]], [[[-1.4587411698434058, 0.49690939499221054, 0.89447769563233703, -0.30347772536740358], 
[-0.74229710808168103, -0.70124574753092495, 1.5090554450906528, -0.001538091977223457], [-1.2945454504477103, 
-0.030987720933937649, -1.5605397782151482, 0.86693322382179105]], [[-0.62781502775689713, 0.68372904978794213, 
1.2665952142005037, -0.35310287137826013], [-0.19185361154036801, -1.5141825245748524, 0.32791416560174402, 
1.0981279684151384], [0.24606173603165085, -1.1846646950278541, -1.4209920692190618, -0.36369305245877914]]], 
[[[-1.2362642800784225, -0.26966444683838514, 1.3971643327533361, 0.52839316690162763], [0.53871999841966511, 
-1.3966878293415304, -0.33104693138861208, -0.33192054832286222], [0.38726220037875464, -0.41587846472517831, 
-0.83274294668644144, 1.5588396068268995]], [[1.5390811925568579, 0.70281503596024564, -0.66300068007811208, 
-0.70352217366561076], [-1.4091759691118937, 0.32550250527704089, -0.72981943509196212, -0.98012397845078236], 
[-0.6932739607981846, 0.44345874145667485, 0.18337076919658227, -0.94560134209205504]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_taggedData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[3.1087373682977262, 1.7288468484290664, -4.7350025546766474, -1.2462687606986202], 
[1.1147661257221184, 1.4226862562785945, -4.8699002989410065, -0.88238981212800027], [-2.5253953718962063, 
-2.7446236822447503, -0.3879205662380496, -3.5863352294770778]], [[-3.8709898597864569, -4.5958808941745852, 
-0.69941717063653019, -4.1924685416353791], [-3.532941685164098, 3.0050044088519066, -0.43457496025027353, 
3.7774083718666578], [-3.4471087299766379, 1.6436932375732649, 2.6583331263073102, 1.6377907641557758]]], 
[[[4.7414386788893221, -3.2160736850371494, 0.96698243106325243, -0.82828848269705979], [3.5200044289955326, 
4.9118207264463223, -4.3585332820909706, -0.77115446561512169], [0.16582160103948596, -0.65994126459526647, 
2.6619397454888905, -2.3437138040847803]], [[1.2215261759413565, -1.7584557798190015, -1.1089650453761957, 
-1.5641210633931846], [2.7598302729800022, 1.8875202967455316, -3.630936106223853, 1.2007447478800914], 
[2.6512287695864964, 1.7707580266874441, 0.1953947241778895, 4.503435455349651]]], [[[4.2589981026451031, 
-3.9277495689426001, -2.8473369657351677, 1.4040000652368345], [0.26972497738921852, 0.60147023681782397, 
-4.4745745609656007, 1.9306805685522557], [3.6376896663673826, 2.8396270545178259, -2.6836138158505385, 
2.9279825131423012]], [[-0.3090571730995233, -0.57453652295428181, -1.6271798513695179, -4.4696813085601139], 
[-2.6653810514006215, -2.3768146409366411, -1.3128180331345818, 4.581635378865748], [0.50247688944640778, 
4.9532451154747754, -2.7018392025036428, 2.1300845231681196]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[-3.4505138582863983, 1.4248257487998686, -3.9524119923215526, 
0.81504856520297952], [-4.3718698276484877, -0.94288708981653535, 4.8282582970579551, -4.6722816105251033], 
[3.6647072271001377, 4.7114077690304565, 0.092514361474126616, -3.8124864196234185]], [[4.165582354632777, 
0.34102300070869251, 0.98343269313930293, -2.6743082425014606], [4.1326181041099233, -0.23439200672333627, 
-1.5849546480857146, -2.0736002379586758], [1.6002261433791851, -1.9446791096418647, -3.7570143378185583, 
1.076047727904327]]], [[[1.2380177953892098, -3.1484579868063811, -0.76442488916600748, -2.1818057802025415], 
[-1.5383818327876098, -3.6057035623329137, 2.163451126002772, 2.188392374984657], [2.3709893728585962, 
2.3471840573448759, -4.4074156468519838, 4.7188550456319831]], [[1.9267506253146127, -1.0414071109895739, 
1.0685570204986226, 4.0226454234187585], [-3.2914506386974205, 0.15234686305474998, 4.7028886509160337, 
-0.054270133945801824], [-2.3311208759539328, 3.6661719497716891, -3.8474697136079583, 1.8341420305410185]]], 
[[[3.3674152480608264, 2.7014266389725128, -1.3030301583873838, -3.2387622104831584], [-4.0951687925040563, 
4.7004938357073254, 4.1950637273295168, 2.0382289901206372], [1.7427744024426053, 1.1790502862054941, 
-2.2495124667079791, -3.0538048367909409]], [[-4.4962463676429856, 1.5822806162106859, -1.8235905534694785, 
-4.4062861037909364], [0.87041939490070597, -4.0113094886293554, -4.6982054748353583, 4.2373051220310582], 
[-0.84614829051970197, 3.3176111131043982, -2.6687460434287589, 4.6770431583761969]]]]))
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0/arg1
      s1=numpy.array([[[[-2.3182995299525277, -1.6619336920343253, -2.9660511871402528, 0.93254822596563614], 
[-4.2541426675441816, -0.30333038370249721, 3.4789554983748676, 0.5140033510602251], [-3.2905733659402281, 
3.7139824585796646, 0.78960211543734715, 2.971514572460773]], [[-0.70540352219700697, -1.3136347308463381, 
0.73924055860583593, -4.9656858822166203], [2.1473770309669362, -2.1549075593317712, 1.8563623226892654, 
-1.3846952158399581], [-2.4903294101424525, -0.88004658956592507, -0.26617618913608698, 3.100112111326629]]], 
[[[-3.3182633250231328, -4.3300560926251794, -0.56051213669826971, -0.70765176831018195], [4.8954497928460068, 
1.0868590622929659, 4.1265642365266171, -1.2873984102989136], [2.4610703924835473, 2.7606204885978327, 
0.74181190652475859, -0.76191236317201927]], [[0.24802432047997591, 4.825490575588006, -1.2970889077321623, 
-1.1984463553405069], [-0.17745571227021895, -1.9770464473025151, 0.90483935921876402, -0.059094745500082979], 
[-0.07302552360214154, -1.2720098622673035, -0.68873120844511426, -2.3715151078623045]]], [[[0.24502664280811803, 
1.2652825479404042, 1.2097368744797734, 3.0828413949333608], [-1.7791129684882536, -3.7837459912887872, 
-0.93771394776597372, -2.4396818013745336], [3.7724216979832228, 1.3354698832167236, 4.4778423096546298, 
3.3178355959010588]], [[4.1570198977003194, -1.4482811362101113, 4.7418722485913811, -1.7134079314039576], 
[-3.9415515512997623, 2.479810994419001, 3.53014195265167, 4.0686466788459725], [-0.0011085503845311706, 
-1.9052407432771523, 4.9068043014278278, -1.9049925369925189]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[-1.3409558722385559, -1.0402622299045123, 1.5963994738883607, -1.3364121296870544], 
[-0.2620424872505856, -4.6902200792187969, -1.3998167844388623, -1.7167005045938151], [0.76746362747472596, 
-0.73899748123591691, -0.49128612835996144, -1.2069048096597954]], [[5.4876247962727875, 3.4985988009114131, 
-0.94612932487848034, 0.84428790726567526], [-1.6452358548201755, -1.3944934184479576, -0.23410029116552836, 
-2.7279709849905682], [1.3841978960443853, -1.8677343416375281, -9.9871184381116578, 0.52830049538270318]]], 
[[[-1.4288916262714828, 0.74273256887241457, -1.7251766157273958, 1.1704746879597985], [0.7190359574598254, 
4.5192802791594398, -1.0562136034406204, 0.59900218879101441], [0.067377837523838524, -0.23905541066619487, 
3.5884295224641907, 3.0760936787104383]], [[4.925025794153826, -0.36440974285909289, 0.85496455853216458, 
1.3051239685641005], [-15.552219974623853, -0.95471722443388563, -4.0127963811817313, -20.318976547219506], 
[-36.305508523717684, -1.3920945734895036, -0.28370243976458448, -1.8989697516239186]]], [[[17.381775523816618, 
-3.1042470121287091, -2.3536828758399353, 0.4554240343159735], [-0.15160643655945524, -0.1589615788698745, 
4.7717905568386882, -0.79136572952443918], [0.96428500247258431, 2.126313060447357, -0.59930958490084085, 
0.88249776955784298]], [[-0.074345848878542772, 0.39670234500032192, -0.34315134741406994, 2.6086498297563501], 
[0.67622635825267541, -0.95846604692286597, -0.37188817071462443, 1.1260833737878855], [-453.27383983445719, 
-2.5998001212984949, -0.55063113108412265, -1.1181589858251946]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[1.4883813820024607, -0.85733008219827367, 1.3325501628083192, 
0.8740015181080979], [1.0276735336128884, 3.1084492041565728, 1.3878470993128227, -9.0899827810220994], 
[-1.1136986839535203, 1.2685595103301153, 0.11716579738756713, -1.283011180546296]], [[-5.905247455610807, 
-0.25960260695070153, 1.330328377807082, 0.53855767479752115], [1.9244958125723541, 0.10877125828823039, 
-0.8537959582101573, 1.4975138313746732], [-0.64257609329187038, 2.2097456347181117, 14.114764923235574, 
0.34709961745346513]]], [[[-0.37309208888073375, 0.72711713646590848, 1.363797211723011, 3.0831630441799445], 
[-0.31424729041971439, -3.317544737333189, 0.52427419082751925, -1.6998563595216394], [0.96339762572412757, 
0.85023786030692383, -5.9414193922821363, -6.1934354575719004]], [[7.7683939284098056, -0.2158137280918167, 
-0.82381170182612518, -3.3565502581680677], [18.54801176355144, -0.07705780674126915, 5.1974846175750971, 
0.91835802805387523], [31.922001527224523, -2.8821883057077038, 5.5863153381622421, -0.77340516383820268]]], 
[[[13.743057528229171, 2.1350382516299056, -1.0771186576814313, -1.0505769825869253], [2.3018036881512924, 
-1.2422857788364075, -4.4737136920314677, -0.83544870030685348], [0.46197762126495856, 0.8828729880194196, 
-0.50236527129546937, -0.92042078292356966]], [[-1.0816032827098874, -1.0925231135380433, -0.38457184375036541, 
2.5716503484260449], [-0.22083166579761651, -1.6175867828867223, -1.3308828760572351, 1.0414532045906044], 
[763.29258672131175, -1.7413080865559627, -0.54388679056390776, -2.4551503838225086]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-4.16921840294)+(1.-msk_arg0)*(4.13078960119)
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(-3.74029681078)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(1.11467581688)+(1.-msk_ref)*(-1.10440155158)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.893893990136)+(1.-msk_arg0)*(0.567432073109)
      arg1=Symbol(shape=(2,))
      res=arg0/arg1
      s1=numpy.array([-0.38687332266391294, 1.9399003858649468])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([-2.3105599113962496, 0.4607937586121677])+(1.-msk_ref)*numpy.array([-1.4667128485410923, 
0.29250577877272954])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.73905345325)+(1.-msk_arg0)*(-4.15656795792)
      arg1=Symbol(shape=(4, 5))
      res=arg0/arg1
      s1=numpy.array([[4.4998386312884939, 0.78446231481643913, -1.6813079285045305, -3.1798436625479907, 
-2.0944868186665833], [4.4953342126454849, 1.6650442515799515, -0.91898434961724451, -1.1670746485185877, 
4.5503544844109296], [1.9273082092830283, 1.6079265632411186, 2.6574428427115047, -3.0448375461618085, 
3.5797303717654643], [-0.075958261118915793, -1.5548980284162739, 3.7360078828652803, -1.3318521748477563, 
2.4342937381441239]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[1.0531607556532867, 6.0411486488736692, -2.8186707341949422, -1.4903416507741973, 
-2.2626322643857333], [1.0542160446968702, 2.8462027052727157, -5.15683804106071, -4.0606258213740682, 
1.0414690700430431], [2.4588975600371681, 2.9473071479664283, 1.783313408317817, -1.5564224302276781, 
1.3238576543708847], [-62.39023094310253, -3.0478226653052989, 1.2684805819015268, -3.5582428311065093, 
1.9467878419876474]])+(1.-msk_ref)*numpy.array([[-0.9237148925791, -5.2986203153641043, 2.472222897099563, 
1.3071611057108705, 1.9845281053476089], [-0.92464047416795137, -2.4963708646060252, 4.5230018984057123, 
3.5615270738680049, -0.91346025285808652], [-2.1566700841637609, -2.5850483802851589, -1.5641231830532274, 
1.3651197789397753, -1.1611399536423701], [54.721736605010705, 2.6732093564730848, -1.1125693757198694, 
3.1208928711619754, -1.7075046831007554]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.78787757499)+(1.-msk_arg0)*(-3.28976848057)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0/arg1
      s1=numpy.array([[[1.740883116607046, -2.2503321522792152], [-1.5061945225102802, 2.4982952052110088]], 
[[-2.7983305802801484, 2.1687543854796365], [1.643320759290833, -1.4653409727441757]], [[3.2175020834207952, 
3.7650334271118187], [1.8014810554950147, 0.12236435291525538]], [[-3.1233989028943911, 0.7696710585002986], 
[-4.2753667310345724, 0.88632267436679513]], [[-4.5134534247108533, 2.7836276968220837], [-1.1959116036265316, 
-0.39370816860750679]], [[1.5027698132905005, -2.504876271197424], [-4.7995705146222427, 0.81056999509517258]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[2.7502579175572066, -2.1276315010390197], [-3.1787909884378931, 1.9164578969705082]], 
[[-1.7109763974029355, 2.2076624292018749], [2.9135380587879744, -3.2674153415800777]], [[1.4880728748121901, 
1.2716693404400943], [2.6577451705005974, 39.128042284553942]], [[-1.532906210139654, 6.2206802790783859], 
[-1.1198752940269061, 5.4019576768818869]], [[-1.0608013696955167, 1.7200136284231737], [-4.003538021097242, 
-12.160981043203339]], [[3.1860352348351633, -1.9114227836496733], [-0.9975637529240533, 
5.9068033654860717]]])+(1.-msk_ref)*numpy.array([[[-1.889712439158538, 1.4619035137730338], [2.1841591052200662, 
-1.3168053453857391]], [[1.1756182431612747, -1.5168930620264134], [-2.0019028311874436, 2.2450532277225537]], 
[[-1.0224604041517171, -0.87376873121071352], [-1.8261465867424562, -26.885023311092727]], [[1.0532655555218566, 
-4.2742525449554005], [0.76947047763019516, -3.7117052014083791]], [[0.72888056461647699, -1.1818277581909278], 
[2.7508458573337604, 8.3558552828834873]], [[-2.1891366538499089, 1.3133457003051041], [0.68542976304877656, 
-4.0585865507976919]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank0_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(1.75692828977)+(1.-msk_arg0)*(-2.14899285708)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0/arg1
      s1=numpy.array([[[[0.53229011632207346, -3.0692416214850793, 4.1551847515535112, 1.9956228948074202], 
[2.6857183881485671, 2.8513187715775192, 2.8150572483190999, 1.8695234954963702], [-0.89878809174293561, 
2.0113600086339014, 1.3054601278816813, 1.3445035108068506]], [[2.3491180092357453, -0.60141647759364947, 
-2.7471836719998368, 2.7627957847036289], [4.5424699426220343, -3.5606662061522512, 4.2638313458612025, 
-2.7733238430403837], [0.74945623900841163, -1.8138683468323045, -4.0112916371182106, -1.0801639118432984]]], 
[[[-3.2970678697848332, -1.0955509383278814, -4.1774865059518689, -0.0073839501556669518], [3.4766781803964601, 
2.9172484094685416, 1.016230261992658, 3.9024189052964413], [-4.0414297442027198, 2.9788951120154019, 
3.2762553730841475, 2.115591154300632]], [[2.9124374952334513, -1.4781274260861421, 4.8479887494343945, 
-4.3582251786045347], [2.0756121461581953, -2.2517472797592286, 2.3516485621871688, 1.7775284781206064], 
[3.4727084749786208, 3.8831969404469557, -4.1994122244787384, -4.1535329385031163]]], [[[3.8977522324001956, 
-4.4938918457123735, 0.012474134731923492, -0.027833866034599808], [0.81441408850560748, 0.26774810790526793, 
-1.1937582797303259, -2.6618906938268347], [4.1680717142998951, -2.6383819679851594, 2.73950628932384, 
1.8380370529340837]], [[0.47796984802075215, -1.5175649986704984, -4.2513663435369313, 3.4215268334350561], 
[-1.6384162411134087, -3.5293107080529076, -4.9362183918195077, 2.0866175314723741], [0.68432046227346, 
2.8697679941819176, 1.3674204155349967, -2.6241647447917482]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[3.300696811560822, -0.57243075210204586, 0.42282795948188318, 0.88039092673325559], 
[0.65417442778904566, 0.61618094310709759, 0.62411813856314535, 0.93977331336139813], [-1.9547747749556408, 
0.87350264608417738, 1.3458306785826581, 1.3067487556914268]], [[0.74790976139219179, -2.9213171823946693, 
-0.63953797763024534, 0.63592405182349598], [0.38677818718938206, -0.49342684431748218, 0.41205388939105542, 
-0.63350996465075404], [2.3442706836281171, -0.96860849511920954, -0.43799565045632483, -1.6265385933616416]]], 
[[[-0.53287598531730074, -1.603693838691949, -0.42057066785647906, -237.93880683514894], [0.50534682780712736, 
0.60225529100200292, 1.7288683042409734, 0.45021519534590931], [-0.43472939058011706, 0.58979192744417874, 
0.53626109374853115, 0.83046683485988038]], [[0.60325012730574579, -1.188617610879251, 0.36240354105082945, 
-0.40312930557024584], [0.84646271367295245, -0.78025109902984713, 0.74710495352909456, 0.9884107688824314], 
[0.50592449738540834, 0.45244377679371173, -0.41837480958128359, -0.42299611337686482]]], [[[0.45075422577276247, 
-0.39095918417482473, 140.84570413315484, -63.121963998303322], [2.1572911306009535, 6.5618700483636143, 
-1.4717621813408515, -0.66003021605811329], [0.42152064796339894, -0.66591127103226522, 0.64133026327282083, 
0.95587207394159879]], [[3.6758140645167345, -1.1577285264939614, -0.41326203102693038, 0.51349247727679848], 
[-1.0723332970476791, -0.49781060243882297, -0.35592596402972293, 0.84199824034348569], [2.5674057501257708, 
0.61221962658011064, 1.2848486608869423, -0.66951905106439324]]]])+(1.-msk_ref)*numpy.array([[[[-4.037258613650617, 
0.70017063565121707, -0.51718346730044429, -1.076853178360514], [-0.80015569263157726, -0.75368383167253838, 
-0.76339223948840329, -1.1494869480160286], [2.3909894632838751, -1.068427754284504, -1.6461574054884893, 
-1.5983542176046435]], [[-0.91480838707695511, 3.5732191204351396, 0.78225306847357656, -0.77783268274130046], 
[-0.47308906480995322, 0.6035367351675005, -0.50400512655556962, 0.77487988374500394], [-2.8674027184371207, 
1.1847567993759462, 0.53573588048219245, 1.9895062531899683]]], [[[0.65178908713899886, 1.9615636132468171, 
0.51442245331507475, 291.03566678779583], [-0.61811670381217776, -0.73665062258920866, -2.1146711896460979, 
-0.55068225867953946], [0.53174074352400413, -0.72140601675245186, -0.65592959411445284, -1.0157883543395652]], 
[[-0.7378674600225622, 1.4538617030959151, -0.44327513287507087, 0.49308899127843331], [-1.0353537683136067, 
0.95436680501382209, -0.91382398358183792, -1.2089780183744372], [-0.61882328233601869, -0.55340815571281099, 
0.51173658174249104, 0.5173891453132069]]], [[[-0.55134157559292896, 0.47820306559730319, -172.27590556503543, 
77.207846528074953], [-2.6386980375371829, -8.0261738314233622, 1.800190954543867, 0.80731822011553245], 
[-0.51558442473760147, 0.81451165265637826, -0.78444530879788221, -1.1691782021760189]], [[-4.4960845668011187, 
1.4160796137002765, 0.50548286913673068, -0.62808008286899497], [1.311628146228581, 0.60889874393290866, 
0.43535206234873525, -1.029893032464839], [-3.1403311395116882, -0.74883853379060061, -1.5715670416121414, 
0.81892452116337022]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank1_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([2.013685465408555, 
-0.23558936453305712])+(1.-msk_arg0)*numpy.array([-3.5034214026844568, 1.5861853637787142])
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(2.67653295039)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([0.75234846823535373, 
-0.088020349048417421])+(1.-msk_ref)*numpy.array([-1.3089401354733579, 0.592626877074932])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank1_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.240783536656898, 
2.4522207164515661])+(1.-msk_arg0)*numpy.array([-2.6143091000360554, -1.7209346138742179])
      arg1=Symbol(shape=(2,))
      res=arg0/arg1
      s1=numpy.array([-0.86125236712465636, -3.638203712883775])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([0.27957372989379237, 
-0.67401962890853218])+(1.-msk_ref)*numpy.array([3.0354739212666386, 0.47301766192474731])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank2_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-3.5524341460375641, 4.7094831286696373, -0.18763500262553379, 0.92078587043782267, 
1.5633164400186814], [-3.2160743627609065, -1.475450853387378, 0.4531551970811698, 1.5869072036359357, 
2.2501989264199054], [-3.0047858395011962, 3.8668179864794041, 4.7640266575428676, -4.5168020830691402, 
-3.9906727379142404], [-4.1304645971682694, -0.88518480435952807, 1.012127262108268, -3.7146974028356574, 
-4.6313643928370212]])+(1.-msk_arg0)*numpy.array([[1.7591260079289945, 1.3879718313395273, 1.5326835858993135, 
4.2409259796446257, 2.655307107561482], [-2.3814537086615495, 3.7333616202275053, 3.1621968222159325, 
4.2220383652526721, 1.5144471593836695], [1.2444878229358283, -2.6197010314766853, -4.3618395583941041, 
-1.238746645541231, -1.9420904898415925], [2.9535211925786955, 0.085799908201006581, -3.3450696662897927, 
-4.0496886845617643, 0.2423218533090532]])
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(-4.12759091887)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.86065557751820976, -1.1409762307454672, 0.045458720671131048, -0.2230806997439248, 
-0.378747911492956], [0.77916499623529623, 0.35746053385346038, -0.10978684806424793, -0.3844632946500936, 
-0.54516035398121099], [0.72797568813457048, -0.93682200161917983, -1.1541906044431842, 1.0942949947926262, 
0.96682854874783597], [1.0006962119929996, 0.21445555573666569, -0.24521016786851502, 0.89996743278379421, 
1.1220502428343315]])+(1.-msk_ref)*numpy.array([[-0.42618710102484819, -0.3362668100161586, -0.37132642648596642, 
-1.0274579198869207, -0.64330675199020826], [0.57695972189861822, -0.90448925138421032, -0.7661119729091026, 
-1.0228819784326422, -0.36690824966706698], [-0.30150464215020217, 0.63468039419829836, 1.056751903018037, 
0.30011371521305036, 0.47051428497038333], [-0.71555569595749791, -0.020786921448245156, 0.81041695556566817, 
0.98112646436194051, -0.058707817240641058]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank2_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[4.32105107438462, -2.7938865728204942, -2.6480612891877806, 4.0962844399213747, 
-1.6936450259444391], [-4.5540373065810638, -3.4735704316216474, 4.0973336660426511, 3.5796783703777653, 
-1.2789555045275067], [1.590790512666854, -3.6030796222704065, -1.5358180525686507, 4.2468820159522522, 
-3.549813920750482], [-0.65580849674863462, -3.0765715955273709, -2.9627491908542303, 0.028934297810717879, 
-3.4006116351470137]])+(1.-msk_arg0)*numpy.array([[0.44461502386601914, 0.042731475461358137, 2.311229454922783, 
4.1960944541714742, -3.5527610506036744], [-0.10889510534494828, 4.1596007072981003, -4.0240613332963262, 
3.5788133245810574, -0.84296801766120399], [-3.6483106190723138, 3.5039136834827218, -1.4706155051518079, 
-2.1562609648378861, -4.3073932462182585], [0.23063830136591879, 4.1274722231354168, 0.13645071183945756, 
-4.5418134155752554, -1.7380106134837989]])
      arg1=Symbol(shape=(4, 5))
      res=arg0/arg1
      s1=numpy.array([[-4.2460288300055975, 1.9723502468412493, 4.5012610797315986, 4.2549986105536082, 
-3.0291017439146586], [-3.3815482186462242, 0.49185174223600381, 3.1342744464723928, 2.5140825751839522, 
4.9827618110943668], [-2.6160153055213375, 0.22203465130806865, -0.21905170509580429, 0.15252026869692958, 
-3.4865345481390886], [-3.7892575047962449, 4.3978652002093614, 1.623750827816151, -1.9970794554891591, 
-0.020308141898515153]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[-1.0176688023992817, -1.4165265917120697, -0.58829320101238813, 0.9626993601740369, 
0.55912450922023427], [1.3467314413763516, -7.0622306141083753, 1.3072670361251153, 1.4238507540333454, 
-0.25667602687325908], [-0.60809679106591863, -16.227555478587014, 7.0112124984233581, 27.844705836384009, 
1.0181496473755489], [0.17307044874056365, -0.69956023103684717, -1.8246329055542057, -0.014488305776312136, 
167.45065364131872]])+(1.-msk_ref)*numpy.array([[-0.10471314295466831, 0.021665257238055607, 0.5134626527952022, 
0.98615648046604887, 1.1728761035316908], [0.032202736233210821, 8.4570213950817141, -1.2838892707131575, 
1.423506673928242, -0.16917686408053739], [1.3946059915521991, 15.780931772766907, 6.713554247425833, 
-14.137537150046303, 1.2354368461707332], [-0.060866357346785971, 0.93851722034111629, 0.084034267759527922, 
2.2742276993995696, 85.581961272925469]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank3_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[3.4818338012749823, 3.4971998906886821], [-2.5239057976796686, 2.8945562475557409]], 
[[-0.41913812937298012, -4.1817511712950708], [-0.9293236620515497, 3.2965731483205314]], [[-3.6083495228355655, 
-2.1861278112872515], [3.5523423560955436, 4.571917845676408]], [[2.5235010801011963, -4.4530426867639079], 
[0.03134464425793837, -3.7225099091273051]], [[3.3642508251751551, -2.7962331811582333], [2.5944511937764982, 
-2.2330126770386505]], [[-1.3859482579701279, -1.4570666635369189], [1.8261086891132727, 
-1.5056570354979151]]])+(1.-msk_arg0)*numpy.array([[[-1.0522384302138366, 4.1034138371598559], [0.079522524263561323, 
-0.023108968432953958]], [[-0.026505249441055589, 1.932139350294455], [4.0042157822936666, 2.2762304926106802]], 
[[1.264288211638612, -1.9946271929128692], [4.2703916328307017, -1.1300899977876777]], [[4.2801783215727394, 
-0.096454065145835877], [4.4865658066576728, 3.9314784386307959]], [[3.1200236527004428, -2.7447976911193561], 
[-0.85723836157011135, -2.8199900333320906]], [[4.5366088574407808, -3.5698521752315759], [1.5597344890251819, 
-1.342225295850489]]])
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(-0.28615667046)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[-12.16757867527485, -12.221276844844011], [8.8200138533333607, -10.115284899367113]], 
[[1.4647155654258333, 14.613502332743044], [3.2476043999188389, -11.520168804812155]], [[12.609699145006596, 
7.6396185620070858], [-12.413977106970515, -15.976974565469046]], [[-8.8185995316688608, 15.56155472317962], 
[-0.10953665419564539, 13.008642793981416]], [[-11.756674481041422, 9.7716861768928052], [-9.0665410301538856, 
7.8034619058510533]], [[4.8433197651583795, 5.0918493746639122], [-6.3814996385676048, 
5.2616527620282927]]])+(1.-msk_ref)*numpy.array([[[3.6771410169222527, -14.339745533681443], [-0.27789855164224664, 
0.080756350693547557]], [[0.092624957504787692, -6.7520332382589583], [-13.993089085985231, -7.9544904158717298]], 
[[-4.4181678854695097, 6.9704025759996995], [-14.923264329180817, 3.9492002614218951]], [[-14.957464785617184, 
0.33706733095137653], [-15.678704254727821, -13.738901952950792]], [[-10.903200850382252, 9.5919402707211265], 
[2.9956958899211616, 9.8547066150873093]], [[-15.853584157763764, 12.475166731199417], [-5.4506312451800714, 
4.6905259754858344]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank3_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[4.3746388890042116, 0.93842526855136477], [1.8706383189312428, 
-3.9786565611021829]], [[-4.8975447110451586, -3.7180783682549334], [2.5316157131902353, -0.25911969310157357]], 
[[4.2345079212590129, 2.6909111977680222], [-4.6351551680329486, -1.6723804666364006]], [[-2.3891891985415548, 
3.1344558904005719], [-4.5767190111429077, 2.4120357644515966]], [[-4.2232569151431667, -3.8926752844093224], 
[2.4150103819552253, -4.5535897724203158]], [[1.3100697627188289, -4.6842849101527513], [-1.1835626225415821, 
-1.6287129504016793]]])+(1.-msk_arg0)*numpy.array([[[-0.53174998140680607, -2.3053360747766796], [-4.7243487743326726, 
4.6049518621447536]], [[-4.0675549321859457, 3.0884984319359248], [-2.3740277209818728, -4.6692510557804932]], 
[[3.1231828125619838, -3.946142358002116], [-3.2294647129609846, 0.76749287499934926]], [[-4.2013021907005701, 
4.7778175384969011], [1.0778106376839505, -3.0244592978197171]], [[0.014195128518603717, -1.7862148338991855], 
[1.5856103610038801, 2.294853232961561]], [[0.84538744044394143, -0.1665108768840593], [4.9401260246627476, 
2.6878593818409549]]])
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0/arg1
      s1=numpy.array([[[2.0746966987933471, -0.038598186515555355], [4.5881839069107979, -4.5221184818631528]], 
[[-2.8952171009517813, 3.2089615497034014], [-2.3810029676646538, -0.20007357454609931]], [[1.5952957217622385, 
3.5074629033411888], [-4.8216610018108206, 1.1790345104067725]], [[1.5767066224494872, -1.604585837573552], 
[3.2217724192087989, 4.7028880210110895]], [[4.9340137982802084, -0.75080095286040205], [-2.5470169322631153, 
-2.2621993393540785]], [[1.0027941556835875, -2.9581897902533969], [-2.2860741942260967, 1.3808737645507074]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[2.1085679133477782, -24.312677699849253], [0.40770778959266579, 0.87982138837347346]], 
[[1.6915984329586637, -1.1586546958154076], [-1.0632560091570595, 1.2951220254320459]], [[2.654371765368603, 
0.7671959110970713], [0.96131917326667549, -1.4184321594279894]], [[-1.5153035856663291, -1.9534360936031212], 
[-1.420559374043824, 0.51288394571066676]], [[-0.85594752828117715, 5.1846967822550107], [-0.94817209550680237, 
2.0129038556436383]], [[1.3064194234616144, 1.5834970851385091], [0.51772712606217608, 
-1.1794799729080312]]])+(1.-msk_ref)*numpy.array([[[-0.25630251482834782, 59.726538547286722], [-1.0296772906632581, 
-1.0183173839017754]], [[1.4049222529283787, 0.96246040474414196], [0.9970704586354957, 23.337669986521096]], 
[[1.9577453696873013, -1.1250703048756534], [0.66978261469400868, 0.65095030571629375]], [[-2.6646061676164279, 
-2.9776017129266807], [0.33453965626431137, -0.64310680677646215]], [[0.0028769940861437288, 2.3790790716155366], 
[-0.62253624658671147, -1.0144345783501139]], [[0.8430318781302184, 0.056288098022877722], [-2.1609648703178355, 
1.9464917437369802]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank4_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.44452618685267975, 3.0663649813827689, -2.1402320946883169, 
0.26287923382873224], [-0.22292293247447503, -1.779872403236904, -0.83539558284346249, -4.8386253696649675], 
[-2.6909471943775531, 1.1035502397379968, 0.93405709802047099, -4.5133518823731986]], [[3.5925478528654295, 
0.52351538108496953, -0.17314676004431107, 1.4384665038816946], [2.2650627980321367, -4.4386411147000606, 
0.38967773187251886, 0.15837556447672352], [2.8076217348733392, 1.2019742108297082, 3.7488885553516198, 
2.8100664184927595]]], [[[4.1664177934939932, -4.7550890791444989, -2.0711507140466989, -2.9208312198338993], 
[4.1895048918264681, -4.592776924342691, -0.74263699511176196, 0.12222597803184154], [-3.4946750822854025, 
-1.571892398029048, 2.6881024097725241, -0.45229879881073565]], [[0.60058700109640029, -4.2888213982767818, 
-2.2458451822005356, 0.91258303405033203], [3.3245528372575315, 2.6854365632371522, 1.3044064696261142, 
3.5732020880561883], [2.7937275003598705, 0.85713829166626887, 0.71698412194859618, -0.043226853713563607]]], 
[[[4.2797940742559852, -0.57449215469944281, 3.5372782003910892, -4.8902142136425928], [3.4633055922696609, 
2.2397500952615088, 0.92632497108839473, 1.611561401548693], [-3.7446826480686668, 0.059061052999116015, 
-1.973290483226485, 1.3807066282723524]], [[-2.2506401919494836, 2.807017817591114, -3.6668928425508698, 
3.3357720442069549], [-2.317215173553099, -1.8521171123736782, 0.9170155613529376, -0.56660297630200507], 
[-2.599117417540401, -0.71112092316441, 3.5146676236929366, 
2.9740235371521369]]]])+(1.-msk_arg0)*numpy.array([[[[1.1035419360879652, -4.9240459106615404, 0.57110407594141677, 
2.5493425213144949], [1.2839073993293217, 2.064480694195943, 2.4864284511610704, -2.2797414911063751], 
[3.4328347507735355, -3.742525310292609, -4.4574196999473639, -2.784337348306626]], [[0.23082397702841462, 
-4.2835955276491635, -4.20264969709529, -4.7450278917265187], [4.3504991743194115, -1.9305238512009049, 
3.4289768271375802, 1.6515283567503367], [0.65954658541804267, 2.0071119325399334, 3.8939804193697718, 
1.7025613989899737]]], [[[-4.4747539140927985, 0.69576514550398461, -0.28643903065104048, -2.5385265745964647], 
[-0.043853132795085514, 2.9368925771712338, 1.7512112730662803, -2.5278476664471796], [-2.9694783445707151, 
1.5583351617541137, -1.5856519335480899, -1.5853506441940759]], [[-2.8774784925906327, 0.18293932340222163, 
-1.4449105065783749, -2.1949936564433181], [1.3493715781295066, -4.2954572018618942, 0.98668403490899248, 
-4.4055318216701655], [-4.6821350103268777, 3.8610980860163053, -3.523887848809939, -3.546193394267144]]], 
[[[2.4380429784019544, 3.4521340144554973, -3.7692521052369967, 4.9812805548611685], [0.84523245498652244, 
3.7857178447844362, -4.170487549201054, -0.54939171272889276], [-3.5433162653492811, -1.4054231603316514, 
0.69989467178326947, 4.6415130978277084]], [[-0.28700657500147031, -0.19881123713218596, -1.6586432383015293, 
-3.4424186977029114], [-4.5813813287646896, 1.7385541775874067, -3.8853770583055702, 0.97367116411611665], 
[0.5470898075374242, -2.534993203695346, -4.9939695365786676, 4.0817080848471043]]]])
      arg1=Symbol(shape=())
      res=arg0/arg1
      s1=numpy.array(2.66573806719)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-0.16675538843211471, 1.1502874266315799, -0.80286661357682243, 0.09861405254483871], 
[-0.083625220053744204, -0.66768465557178724, -0.31338247111570511, -1.8151165822394388], [-1.0094567157580112, 
0.41397549643762593, 0.35039342744007457, -1.6930965340977453]], [[1.3476747385954884, 0.1963866546110643, 
-0.06495265314148993, 0.53961284553270628], [0.84969443393929622, -1.6650702367684398, 0.14618005297240813, 
0.059411525245472156], [1.0532249096146338, 0.45089734269967507, 1.4063229247809164, 1.0541419853204237]]], 
[[[1.5629509308409302, -1.7837795609661897, -0.77695207174998171, -1.0956932550060703], [1.5716116085784111, 
-1.7228913001128441, -0.27858588368149767, 0.045850708115761904], [-1.3109596645285277, -0.5896649852344733, 
1.0083895499175104, -0.16967113325105213]], [[0.22529857996487432, -1.6088682721916423, -0.84248531760932877, 
0.34233784829880937], [1.2471415996099637, 1.0073895092277732, 0.48932282045276504, 1.3404175496588318], 
[1.0480127566711488, 0.32153882716974125, 0.26896270521588078, -0.016215716857401703]]], [[[1.6054818464464042, 
-0.21550960380171494, 1.3269413990545242, -1.8344691377708944], [1.299192007982737, 0.84019886380726794, 
0.34749286979442995, 0.60454604350793251], [-1.4047451601330245, 0.022155610007608559, -0.74024170173101733, 
0.5179453470187918]], [[-0.84428407263630001, 1.0529983617442118, -1.3755638213979053, 1.2513502677786434], 
[-0.86925838741351014, -0.69478585880924792, 0.34400062505739643, -0.21255013134095985], [-0.97500855373963136, 
-0.26676323976338612, 1.3184594791778352, 1.115647322501641]]]])+(1.-msk_ref)*numpy.array([[[[0.41397238148416005, 
-1.8471604435804707, 0.2142386316834391, 0.95633646557068408], [0.48163299130222342, 0.77444994300265702, 
0.93273547081138874, -0.85520086131672679], [1.2877614620220696, -1.4039358766544225, -1.6721146592788225, 
-1.044490223017593]], [[0.086589143873261026, -1.6069078880519039, -1.5765426276573689, -1.7800053013936494], 
[1.6320054951625498, -0.7241986281252718, 1.2863142366988327, 0.61953887258350682], [0.24741612596359824, 
0.7529291633123758, 1.4607513271074251, 0.63868292985895248]]], [[[-1.6786172539481112, 0.26100281721871821, 
-0.10745205396451149, -0.95227907266657619], [-0.016450653323677209, 1.1017183620993041, 0.65693298776039311, 
-0.94827308712706726], [-1.1139422815460562, 0.58457925065244776, -0.59482660845934698, -0.59471358559422671]], 
[[-1.0794303191321972, 0.068626143601199674, -0.54203018832267935, -0.82340935272625038], [0.50619060992421039, 
-1.6113575653696872, 0.37013540341897494, -1.6526499268222394], [-1.7564122551849053, 1.4484161566875804, 
-1.3219182680327803, -1.3302857613479471]]], [[[0.91458459794258318, 1.2950012069615637, -1.4139619160741743, 
1.8686309117048736], [0.31707258315789316, 1.4201387193205603, -1.5644776208624107, -0.20609365919729955], 
[-1.3292064621650026, -0.52721727525641704, 0.26255192901259999, 1.7411737315658249]], [[-0.10766495723412189, 
-0.074580184594690943, -0.62220788258068571, -1.2913566940695587], [-1.7186164631668384, 0.65218492356209323, 
-1.4575239428531019, 0.36525387700306983], [0.20523014405310749, -0.95095359701515858, -1.8733909374079507, 
1.5311737244869104]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_quotient_overloaded_expandedData_rank4_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[0.13107029610522503, -0.47078445251855072, -4.2897888183577919, 
-1.9334992582613539], [0.63937604658959835, -3.33703556551216, 0.72789522858485256, -0.26278387324467101], 
[-3.9668440246732195, -3.671802989288131, 2.0999037661084587, -2.4149526975860147]], [[-4.7632015166069488, 
-0.32819100922220645, 3.9967626932273532, 1.2406587303878966], [1.749127295060128, -0.45247786055816697, 
4.6204416104916426, -2.081666165987246], [-0.68418527302623922, -0.086844242543568484, 4.5968719860538609, 
-0.91155775353879243]]], [[[1.6437881369181344, -2.3806862133298736, -4.4788477060454488, -4.9712582040083095], 
[4.5521169671152215, 3.7737242105731958, 1.1068782587395374, -2.211728710301637], [0.082833519252822008, 
3.0206978008416172, 4.357477209475304, 0.49732738761943907]], [[0.79602598196542029, -1.265855956484395, 
4.8250166190599018, -4.2788131362835058], [-4.383003984209294, 2.5455905334955311, 4.8365355193769055, 
-1.9625703825210605], [-2.6627467883711997, 4.437603110973388, -3.5973501175563993, 0.65182523283167892]]], 
[[[3.0885957484528426, 3.1277669503591561, 1.6769138721837127, 0.98171782970485566], [4.192401912714864, 
-0.47522991390312086, 2.1118225327681861, 2.1921637678324597], [0.3811847761647158, -0.10326953777431402, 
-0.65807832498088992, 2.6865347217403768]], [[-3.1528015361925754, -3.2404126527988009, -1.0365916695998001, 
3.8469436365391338], [-1.8197555129691056, -1.4900674698016427, 0.18151570218425128, -2.8062991438823612], 
[-4.8043393089791238, 3.6114859202585219, 4.5737474506226459, 
-4.6680289702448849]]]])+(1.-msk_arg0)*numpy.array([[[[-4.4610050641633041, -4.2116496086744872, -4.194177401285657, 
4.6667981922187636], [-0.69217835042977516, -0.50405115463934003, 4.9237488959861206, 2.8925023711926752], 
[1.8515933082064082, 3.7230125484719974, -3.019629418821407, 3.8938117509466394]], [[1.4837641305474625, 
-4.8550584858736592, -1.7472809251385635, 3.2307178554322675], [-1.8988735029264605, 0.86622970071825556, 
-4.9469479387882034, 1.4686808631017545], [-4.359409091955369, -2.4483640826280038, -4.9126815838484248, 
3.1077856574793419]]], [[[1.1758972510173491, 0.55388575745847213, 0.89146624927904305, 0.63400471395171998], 
[0.55786297685443831, -3.8299740887244482, -0.11367732885478077, 3.7677830690318999], [-3.5745169669757679, 
-2.4896945451989994, 2.6912852814721315, -3.0192095141877564]], [[-2.6866046448316161, 4.6338119217697056, 
4.5795707054588188, 1.6240797284646193], [3.3889322076803658, 0.29273714098976189, -1.0820300702831531, 
3.1517798425152321], [0.0057643443868427013, 1.3712665201039158, 3.3350408828391025, 4.3948199682885214]]], 
[[[-1.8319354596571156, -0.24146471255168933, 1.2551580386328345, -4.4733596377004439], [1.7721669868368117, 
-1.9145129599168875, -1.1223547031438805, 2.067367827980739], [1.8871564184207337, 1.2753354170198996, 
4.2895243683672284, 3.3249111447243411]], [[1.1893790492598919, -3.0721981828360221, -4.0386799091272056, 
2.7354976215990421], [3.5398142903243972, 3.4570995606739494, -1.4962799143431105, 1.3335418401314989], 
[1.2267226207454929, 3.271100819046973, -4.8430217828867974, 1.5873946155417196]]]])
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0/arg1
      s1=numpy.array([[[[-0.49104290278642271, -2.3674315644822963, 4.0989949383597608, -1.7292635092608224], 
[-4.622988996878866, -4.1541086550089714, -0.037463128987194061, -2.6489483804003844], [-1.3280913255545124, 
3.198305008176586, 4.0433681250873974, -4.8641097044412218]], [[-3.8791417988042811, -1.1645861545593315, 
-2.3888987565565021, -3.5719587761449887], [-2.4965270383303331, 3.9852223086984733, 0.5007169845482391, 
-3.108649685462217], [0.20848826126615183, 4.6794023694682423, 4.6869612020471063, 2.7794546763479264]]], 
[[[3.9549024552919079, -0.48631450907908125, 4.4520002177282301, 4.2795685433565982], [-0.044877637378450963, 
2.5321944239340848, -1.7730261622972501, 1.866409581291264], [-1.5466816416052809, 2.912554476943007, 
-3.3751196293325059, -2.725818247229256]], [[-4.851701238543999, -4.998185592299647, -0.44584459532102105, 
-1.2909219819916928], [0.84509859941086773, -1.8873930915550119, 3.6306196902313079, -0.58336909128228509], 
[-2.247976724721672, -2.3230490248981575, 4.8350209920697722, -0.32527190181639298]]], [[[2.5335174897949022, 
-0.20854461379276934, 3.2780648503024352, -1.6620646994478872], [-2.1379815500652954, -3.2990143123595739, 
2.9742698157590199, -3.0724429738487213], [-1.941430518641484, -3.8959116863238554, -3.0686598489329029, 
4.0511881765933175]], [[0.02000325485725174, 3.2082680619577548, -4.598681116192207, 4.4506904679653321], 
[-1.3308670983956148, -1.4563013194069239, 1.9676557830927131, -1.443387189237999], [3.9560651184617885, 
3.6345685490807202, -3.1473146440504864, 2.5626571964669047]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[-0.26692229001064205, 0.19885873770611004, -1.0465465029518624, 1.1181056258382696], 
[-0.13830360552907706, 0.80330964898773283, -19.429643178861738, 0.099203093268639547], [2.986875938683629, 
-1.1480465371192021, 0.51934518479270775, 0.49648401132503633]], [[1.2279008511818705, 0.28180912845077649, 
-1.6730565421652779, -0.34733288039983173], [-0.70062421444068845, -0.11353892593910048, 9.2276510545379935, 
0.66963678015000538], [-3.2816488989412327, -0.018558832023123775, 0.9807787578967162, -0.32796280554448065]]], 
[[[0.41563304164901532, 4.8953633274033006, -1.0060304328401219, -1.1616260269333594], [-101.43397097149828, 
1.490297970370789, -0.62428760628404523, -1.1850178719997067], [-0.053555636159778923, 1.0371300604863249, 
-1.291058595851039, -0.18245067811287974]], [[-0.16407151694367492, 0.25326309579912565, -10.822193808552843, 
3.3145404571095454], [-5.1863817870065798, -1.348733628879734, 1.3321515146271816, 3.3642001467839115], 
[1.1845081664272488, -1.9102494451953862, -0.74401954478721888, -2.0039395631523571]]], [[[1.2190939122756466, 
-14.998071124806016, 0.51155603954235385, -0.5906616210734561], [-1.9609158519571066, 0.14405209220302512, 
0.71003058349945258, -0.71349209293425142], [-0.19634221905167631, 0.026507155728614104, 0.21445137531607825, 
0.66314735446318107]], [[-157.61442618672615, -1.0100192970849919, 0.22541064348860901, 0.86434760274348943], 
[1.3673457816808718, 1.0231862389635615, 0.09224972362744735, 1.9442455668211094], [-1.2144237177893482, 
0.99364914197916654, -1.4532221807783379, -1.8215580986331779]]]])+(1.-msk_ref)*numpy.array([[[[9.0847562175307548, 
1.778995292561067, -1.0232209271680595, -2.6987201009137105], [0.14972528615081884, 0.12133798041887074, 
-131.42919529410358, -1.0919436530339175], [-1.3941761929913368, 1.164058005397852, -0.74681041285503469, 
-0.80051890017845562]], [[-0.38249803887159334, 4.1689131086319398, 0.73141690092266454, -0.90446672481449997], 
[0.76060602339657379, 0.2173604465747247, -9.879728652007838, -0.47244978099981061], [-20.909614121584703, 
-0.52322153328871157, -1.0481592170429599, 1.1181278413803195]]], [[[0.29732648638247061, -1.1389455735288434, 
0.20023948914673259, 0.14814687684718106], [-12.430756373157672, -1.5125118563266158, 0.064114862641108966, 
2.0187332441924042], [2.3110877318397747, -0.85481475622463277, -0.79738959712203872, 1.107634200210057]], 
[[0.5537448644793036, -0.92709881139841899, -10.271674824635689, -1.2580773672773902], [4.0101027383583956, 
-0.15510131000245292, -0.29802903157125132, -5.402719975423115], [-0.0025642366860165779, -0.59028737896051597, 
0.68976761182818369, -13.511219209980442]]], [[[-0.72307985519587537, 1.1578563845894034, 0.38289603651893378, 
2.6914473541170971], [-0.8288972310274092, 0.5803287826743494, -0.37735470305926527, -0.67287427157387836], 
[-0.97204427369426094, -0.32735223991262846, -1.3978494129477628, 0.82072493298010418]], [[59.459275890228867, 
-0.95758774625624643, 0.87822569277673845, 0.61462320089170264], [-2.65978045034828, -2.3738902894641662, 
-0.76043784039873807, -0.92389751694797129], [0.31008655924816314, 0.89999700786337411, 1.5387790324814787, 
0.61943307038110118]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank0_Symbol_rank0(self):
      arg0=Data(3.78107997963,self.functionspace)
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(0.0800878500517)
      sub=res.substitute({arg1:s1})
      ref=Data(1.11239752186,self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank0_Symbol_rank1(self):
      arg0=Data(4.90258552017,self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0**arg1
      s1=numpy.array([1.2995894043345173, 0.78253444504138814])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([7.8934933252568413, 3.4696165734693301]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank0_Symbol_rank2(self):
      arg0=Data(3.92827709573,self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0**arg1
      s1=numpy.array([[4.1349425587832229, 0.91098413715014492, 2.7789836134951633, 0.12883166700263957, 
3.6410211713020102], [4.0888877726009252, 4.0211549295632993, 4.6892515161560091, 3.4849861795634953, 
4.0032210210582928], [4.299348637922626, 1.7371642461914261, 2.185930497819307, 1.7970091028967969, 
4.5443361828998592], [3.1255490514768072, 2.9012687039592109, 0.33081920581130919, 1.5123642954465053, 
4.3636469435748326]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[286.41226576143146, 3.477832814129735, 44.799992375326617, 1.1927572059187601, 
145.71440254454342], [268.92164649005002, 245.12000850057427, 611.45312320484868, 117.70262752348444, 
239.17864250542837], [358.65926391850468, 10.770271935452501, 19.901441052324959, 11.689248981538858, 
501.48063851421955], [71.979339350865629, 52.959027255929435, 1.5724377868886246, 7.9186302143497267, 
391.64108226882905]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank0_Symbol_rank3(self):
      arg0=Data(3.16126608038,self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0**arg1
      s1=numpy.array([[[2.7962312793949295, 2.0684099882736957], [0.38062016949789762, 3.4476824424756334]], 
[[2.041547872112385, 0.47599085293280569], [4.1441460833378203, 0.81184164478938559]], [[4.3084580838333535, 
4.0366716734419619], [0.46138540150130164, 3.7240656768046549]], [[0.67126378380454188, 2.6436153849274531], 
[3.6341579561428712, 0.79424640609140063]], [[2.8792963151587307, 0.15059024316863731], [1.4454736834885067, 
0.20711228881254098]], [[2.5282275840574844, 1.1701013108471818], [4.7723602197014845, 4.287233089759706]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[24.98774742669173, 10.812287661988025], [1.5497341316804971, 52.888612151720686]], 
[[10.48311297133106, 1.7295347337178362], [117.89549729799067, 2.5457046637850267]], [[142.43948278798089, 
104.17774369773555], [1.7007034469541953, 72.69681517764694]], [[2.1654024349191139, 20.962369213688653], 
[65.550199410739893, 2.4946685023586084]], [[27.494642910796053, 1.1892528402215572], [5.2788095694213668, 
1.2691921182937207]], [[18.355342751374547, 3.8449267600833394], [242.95005261713331, 
139.00193314904641]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank0_Symbol_rank4(self):
      arg0=Data(0.993829389255,self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0**arg1
      s1=numpy.array([[[[0.70582890780189012, 4.459679941127856, 1.7698082654335738, 0.7425015240513263], 
[0.17863451536630379, 0.23071258083531751, 1.1519692954618987, 1.546358793346283], [0.43213129061136418, 
4.7802564121893596, 3.8019765535927128, 1.7733817493669464]], [[4.4818458367006624, 2.8755675026947229, 
1.8170070828708857, 3.689568151497173], [3.2656926466727252, 0.84976244428097047, 0.086645785873186429, 
4.3622867616943184], [2.510711276781572, 1.1909328518939111, 2.7614763003730736, 2.859226851647112]]], 
[[[2.2999398221257867, 3.7396236508286136, 3.6474360642296735, 0.22271721684765292], [2.5248956632409825, 
4.8262190185960856, 0.37713575312925274, 0.15680972420135947], [2.5933818932928228, 4.6152058845108996, 
0.37698282218649021, 0.48371699902902621]], [[1.5772879034133864, 0.010480084060075445, 2.7768860672153597, 
4.4433640193360082], [3.4675949148609839, 1.0817479487868906, 3.3899814682499194, 0.87646369237945143], 
[0.49739388005553137, 1.3463814284307101, 1.0826007333159808, 0.79496282667929863]]], [[[2.4363105443735353, 
3.4203323611137431, 1.8278252124219663, 3.6768660714690418], [3.8007621375663323, 3.2788349807887727, 
4.723911572993079, 4.5936800906463686], [1.8071402427683898, 3.3902207884220079, 0.37997536953424454, 
4.7038449248986476]], [[3.2488654149278418, 1.6048006810659028, 4.9317774433591417, 2.1007443740528178], 
[3.054516223687195, 0.55422473313263965, 3.5194589681498765, 3.326550366831976], [4.3669194535152762, 
0.047451200822730306, 0.70452802796481673, 0.1404011071625294]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[0.99564064100720095, 0.97277331018429902, 0.98910515232215201, 0.99541466269551271], 
[0.99889491206162018, 0.99857297113553989, 0.99289498462376558, 0.99047412161372594], [0.99732879902719118, 
0.97084496836204925, 0.97674154691389037, 0.98908327465377543]], [[0.97263985400820108, 0.98235848653943914, 
0.98881622960918547, 0.9774213780802502], [0.97998918013383751, 0.99475401042354128, 0.99946382997432526, 
0.97335991097124475], [0.98457951330340654, 0.99265555323835852, 0.98305246670030244, 0.98245785140758057]]], 
[[[0.98586485161567172, 0.97711859061328932, 0.97767630927362992, 0.99862239086150251], [0.98449307348906423, 
0.97056880612221985, 0.99766835491976258, 0.99902986140398409], [0.98407582438051522, 0.97183730768595933, 
0.99766929931395654, 0.99701040129524177]], [[0.99028452065604722, 0.99993513323790106, 0.98295870540613939, 
0.97287155660372138], [0.97876523296044249, 0.99332664123569459, 0.97923555070780133, 0.99458961757808051], 
[0.99692600178713375, 0.99170089495067859, 0.99332139797090802, 0.99509148295090599]]], [[[0.98503303654727714, 
0.97905160511359157, 0.98875001942994056, 0.97749822833139455], [0.97674888901436785, 0.97990946373697985, 
0.9711836185354098, 0.9719668027506676], [0.98887662147928368, 0.97923410014112311, 0.99765081960374968, 
0.97130425391482611]], [[0.98009125718466428, 0.99011589292747115, 0.96993486503795368, 0.98708113870220937], 
[0.98127098560076487, 0.99657537729014167, 0.97845107590229674, 0.97962009487115176], [0.9733320001780138, 
0.99970633311899726, 0.99564865802941016, 0.99913133289538214]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([2.866799909147558, 4.876125586700824]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(2.71414665156)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([17.435868015676721, 73.711650960189885]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([2.2084898163228521, 1.4985706335851434]),self.functionspace)
      arg1=Symbol(shape=(2,))
      res=arg0**arg1
      s1=numpy.array([2.9685871945135585, 4.2199180780027188])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([10.506962956516261, 5.5124348187786136]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[0.77922995940155448, 4.9091672288365764, 4.6023034116074584, 4.0585585948105418, 
3.8131836319105985], [0.63921561638913638, 1.9638083205927952, 0.87156249802113606, 1.8813609152354294, 
4.652603835129498], [4.4185672711880182, 4.342637431217093, 2.1485008678971482, 3.5999975099694348, 
3.0608084623767837], [1.5079573236002384, 0.75251090029482415, 1.7564448317741483, 3.6923994643349598, 
1.3401399799253775]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(1.88416909919)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[0.62499961485879429, 20.043593846903835, 17.748329103451528, 14.004750781266964, 
12.452150471142952], [0.43033512176215821, 3.5665495273262708, 0.7718134533153036, 3.2896665757733499, 
18.115582822820418], [16.436880911894658, 15.90873223635964, 4.2247353523330435, 11.17293869306093, 
8.2299551314515949], [2.1682788679762832, 0.5852335625438071, 2.8902325719037805, 11.71940284996929, 
1.7360906938610048]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[1.3915653950314695, 0.72721703282696448, 3.1310378168895978, 2.6348429358346048, 
2.0782491529918197], [1.6459194387429872, 2.562889776578305, 2.0761088251705457, 0.22461806677113369, 
4.206926695670866], [3.2315466420058132, 0.54551438767992344, 2.3962603907765661, 3.5705685721719806, 
1.576440920087347], [4.5957055774395901, 2.6996348087339399, 1.4523409725849665, 2.7420672940204591, 
1.9685543649271546]]),self.functionspace)
      arg1=Symbol(shape=(4, 5))
      res=arg0**arg1
      s1=numpy.array([[2.5855195439682124, 1.0827101090847635, 4.9957557839533946, 0.96832195337783145, 
3.2516725650946836], [2.885435871411993, 3.9749315017293902, 4.653244839630724, 2.3530666812704477, 
2.8630405667066632], [2.0653715617592563, 1.7593252053700164, 0.52661102213923405, 0.82097677205000363, 
2.4546391327367552], [4.7379088202900395, 3.5407454959024718, 1.2463674187947462, 3.0822199358702487, 
1.0765535893703142]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[2.349802789241195, 0.70830818466492951, 299.45926695476669, 2.5552067368444966, 
10.790674928106091], [4.2114637167679394, 42.137956376461631, 29.939356280651772, 0.029778724884887482, 
61.155701008278086], [11.275136610799086, 0.34431546555745196, 1.5844071459062889, 2.8430503327720711, 
3.0565212302888405], [1374.5619596132401, 33.662107628675898, 1.5921977639828369, 22.400261085750593, 
2.0733157810519325]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[3.0660567741514542, 4.3056273316538842], [3.0504750623805625, 3.7871768115665883]], 
[[0.69274312119694637, 4.6538571932210537], [4.8355982528719377, 2.9815267638605385]], [[1.1230458665786869, 
2.1830008754633661], [0.91820982431621256, 0.077119031117770812]], [[2.9614071331788696, 3.5610220293470181], 
[2.6599569889924082, 3.6221415743920904]], [[1.1098062342497441, 2.676585421551215], [2.0821210491789137, 
0.94398503882355267]], [[1.1824417768595235, 1.7231439179679477], [2.2607396485998188, 
3.3720654338288014]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(0.560759226052)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[1.8743657165500538, 2.2674712116369911], [1.8690182094624257, 2.1100648252921164]], 
[[0.81395313946463899, 2.3685487640913623], [2.4199799687700909, 1.8452103631477881]], [[1.0672369661252965, 
1.5492717722289933], [0.95327751240899017, 0.23766519192617833]], [[1.8382175994812311, 2.0384524232966164], 
[1.7308218130238717, 2.0579983247753746]], [[1.0601632912957684, 1.7368809476716751], [1.5087078979960178, 
0.96819192271565813]], [[1.0985300082097622, 1.356811420052328], [1.5799715425927656, 
1.9770722875528537]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[3.3957720535524389, 2.4552077579471865], [4.7087905663535405, 1.2879146157483581]], 
[[0.040773334485048905, 1.338962649449466], [0.15911448294230743, 0.69782664300100028]], [[1.1490297281690691, 
3.6797714112654387], [1.6806152634816163, 2.8654713323448382]], [[3.7833659416351537, 4.1661154468039028], 
[3.469956553077123, 3.9754846411912803]], [[1.8241168951232549, 4.4870775362426878], [1.3424534585235119, 
1.3802296431342149]], [[3.3210730308450875, 2.5390685855193476], [1.1031930645119352, 
2.4082267572037321]]]),self.functionspace)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0**arg1
      s1=numpy.array([[[2.0527495636735842, 1.3882529561124286], [1.3707482038402852, 0.39288536207269026]], 
[[2.3705399572424319, 4.3811434485491185], [3.6465442725413943, 0.81187293080800826]], [[1.5760458701127449, 
2.4821521454352227], [2.2611045908095928, 1.222781185593224]], [[4.0828137815142842, 2.0166310563465872], 
[3.0037225489944039, 3.8035215103515534]], [[1.8270206735321162, 2.0909380886426585], [1.700233947322449, 
0.94341612832048682]], [[3.4512483540336021, 1.9153164559186842], [3.4796412330592381, 4.631860024508482]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[12.299397442261979, 3.4796948710857625], [8.3635241772012918, 1.1045185700581053]], 
[[0.0005079738169757814, 3.592450715967165], [0.0012274367495971861, 0.74669434602170293]], [[1.2447577564334256, 
25.37776750720391], [3.2345019416156076, 3.6228533120009838]], [[228.75452373327897, 17.773353840406003], 
[41.974303066305048, 190.45529976112371]], [[2.9988077003642086, 23.078919356327741], [1.6499038138569069, 
1.3552903973767674]], [[62.959505654838587, 5.9577132902226237], [1.4073820176182781, 
58.609608978624692]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[3.6942258331411808, 1.2537649796273209, 0.42770554254169135, 4.700673631145162], 
[4.4515625216392314, 4.7809283564358287, 3.701290949078563, 2.0902986899469349], [0.95965882113153478, 
2.5044950882367898, 1.5843001497355667, 1.9737450842678197]], [[3.4205167033036328, 1.4681649100887066, 
1.1447801230251771, 3.5317125976738892], [2.925503328877014, 3.6994047548692786, 3.3897537710459931, 
3.2244141062166558], [2.6498609343187218, 1.949698950181703, 2.3949386461700057, 1.6425500663306651]]], 
[[[4.2780973886416191, 4.0618406835817691, 0.76281370649400426, 2.2204933921619303], [0.35215441411200815, 
1.3818360860420937, 2.5239753242436236, 4.5811725609009564], [4.1716817147405427, 0.44398290268201318, 
1.3895634892776816, 0.88264326772164403]], [[0.91724435354033873, 2.1538723799206236, 3.4971391199135713, 
3.3554017029685319], [2.1064185289212674, 2.2856357366084303, 0.51918179267857123, 4.6850674663899552], 
[4.7379107746617768, 1.3230622992714034, 4.6315184230420652, 1.8428337487384241]]], [[[2.4876080017868238, 
2.6810602120023037, 4.5081405094604081, 2.4120441389715914], [4.8306123550480162, 4.2210604925186157, 
4.4800528271783264, 2.9281844533079764], [0.63995448593153881, 1.0111923343576328, 3.3513915338885765, 
3.1411718524458285]], [[1.4289737147392298, 0.45352921851220701, 2.0118969615009972, 3.5980186028034558], 
[3.616620626102065, 2.7726087163304962, 2.3432614104421239, 2.8491353233149006], [0.5320613661271224, 
4.4831794866209096, 1.2664788279071584, 0.28049422891374309]]]]),self.functionspace)
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(1.06920335544)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[4.0438772782241736, 1.2735412636471706, 0.40329131191155071, 5.232097558570211], 
[4.9361882834372706, 5.32766320430086, 4.0521468468151465, 2.1997222137283998], [0.95692805287595428, 
2.6687811343412777, 1.6355615272939708, 2.0688366138549434]], [[3.72436858199117, 1.5077044564832467, 
1.1555422923156822, 3.8539651197510056], [3.1511076657220811, 4.0499389894595748, 3.6885660756269054, 
3.4965304503075489], [2.8347289565408293, 2.0418991380778091, 2.5441507235354024, 1.6999385690588711]]], 
[[[4.7308080931849492, 4.4755720332454247, 0.74865452317740788, 2.3465237379092794], [0.32761633574092269, 
1.4131120282318979, 2.6909816922836431, 5.0900078595749889], [4.6050969731100313, 0.41972303754885415, 
1.4215628284095434, 0.87505100482757892]], [[0.91177753244543847, 2.2713282731526636, 3.8136397625647156, 
3.6486130365880771], [2.2178646706058562, 2.4202014010190394, 0.49615640785649234, 5.2135271144615709], 
[5.2764248443312534, 1.3489444284266585, 5.1498394369137142, 1.9224660983894153]]], [[[2.6495455109781756, 
2.8704290671798018, 5.0032967403347266, 2.5635842415978054], [5.3868816518123941, 4.6634019262660349, 
4.9699739610882974, 3.1541954979564295], [0.62048881052075555, 1.011971499899829, 3.6439508643357619, 
3.4001032908251019]], [[1.4647126684483562, 0.42937939938465319, 2.1116224485244599, 3.9313784965136467], 
[3.9531144908142579, 2.9753493135007223, 2.4854989078477878, 3.063238048859795], [0.50932777150901942, 
4.9736826666605927, 1.2873542029993255, 0.25687297390081976]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_constData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[0.99270068488685292, 2.3472808551478828, 3.1312848033300495, 0.28480210095870712], 
[4.1760257284106208, 1.6103974533434464, 4.5276876502334824, 0.85611009354413747], [3.2374935042795578, 
4.9456771875167229, 4.9802453823156245, 1.2149127690087014]], [[1.8848010664604495, 4.1824957168331602, 
2.4934752197401808, 0.28374659653088119], [4.6634122856431866, 4.948465351830853, 4.5396716570475313, 
3.7492655830226167], [4.2660162764714533, 0.061690267720591635, 4.6305624131357668, 4.183404037793947]]], 
[[[2.7660211396774219, 0.29841308804884997, 2.0624440665165937, 3.4380484119931474], [4.1607961864955918, 
3.0023473648209906, 2.7294567940865302, 4.1979140776413049], [1.1126946263221209, 0.12554655331410369, 
2.1673178358996532, 1.5768419503465658]], [[2.1884187117240974, 2.9636992993533875, 4.0944289364844799, 
0.20431967084707772], [4.9645724723200422, 4.3471545530472664, 3.8082987236588131, 4.9248087663466533], 
[0.94748379921640247, 4.1019196615549953, 2.7446431670111022, 3.2260449250615477]]], [[[4.815971417067975, 
0.55080927709037153, 4.2444312341013362, 2.4110979642875288], [3.8869586759056411, 3.1718207539088712, 
2.2815482728827998, 4.5248091593844997], [1.7555444263914652, 4.8358722724194543, 0.40104556755678317, 
0.72504095284272219]], [[1.6815499163639582, 4.7423275338548976, 4.654236755712521, 4.137431652375871], 
[2.4100531826283573, 2.054034553663679, 4.6472521248761733, 0.98452264700986813], [4.8377266538556594, 
3.082865747001903, 2.5083476832499958, 3.2444587922972503]]]]),self.functionspace)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0**arg1
      s1=numpy.array([[[[0.21753326670751072, 4.1682471517901032, 4.1014560095520087, 4.6700944718391098], 
[3.4121094051365737, 1.1029489552818827, 2.0092241837599003, 0.33422517551710779], [1.1473193657482628, 
2.4557624148843549, 3.4564859349197894, 1.8642853910874539]], [[2.7889107332297205, 0.14971366335904424, 
3.0278221311653697, 3.2041631900025194], [2.9835640858496824, 3.958665469023726, 0.91439051931800353, 
1.7058037073741574], [2.3938649784363508, 3.0154719728696082, 2.6948872188976947, 3.1007570939621973]]], 
[[[2.6265063521385672, 4.5831638786700442, 3.219142414188263, 2.7281296715792824], [3.5791848830985131, 
2.9499608289992416, 0.6944852503432879, 3.4643337801246901], [3.8686503402845167, 2.0190151391117173, 
4.4489402687543533, 0.060998130418435118]], [[2.7466045097612528, 1.6726966829566545, 3.0165307005368542, 
4.277931081219279], [4.5278288671598306, 1.9114000108412157, 2.2131466233201289, 1.7114320346159109], 
[4.0455182963260539, 4.4214997321041718, 1.833260490213896, 1.6277957064140334]]], [[[3.1805485901595181, 
4.6387720949831293, 4.6764390004305776, 0.92658048030955986], [3.6108344682454585, 2.5431872524441639, 
0.36463412919482818, 1.0071871900401932], [2.8537457951315179, 3.6983860534861037, 3.23721845983441, 
0.55328298453643388]], [[1.2875876430466009, 4.3485253099923922, 1.8266284021761314, 0.0290062123030045], 
[3.4490819270809823, 0.22338889791366173, 3.0381024201828888, 1.9556163422370025], [0.44312399416135478, 
0.1273534956738884, 1.2491109040783503, 4.9052871585448905]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[0.99840760191037914, 35.043440814220617, 107.94047254802666, 0.00283573184979817], 
[131.25378495858195, 1.69136219457744, 20.78752777038369, 0.94940103216918048], [3.8492184440675246, 
50.681976498619285, 257.06170337461504, 1.4375275893388262]], [[5.8572122227013041, 1.2389032321759992, 
15.902123865064281, 0.01766446796707654], [98.882782586287178, 561.27606329168657, 3.9881999273730413, 
9.5288296866662598], [32.224695168018663, 0.00022487036760118656, 62.202503506565009, 84.569345856197444]]], 
[[[14.472256933722013, 0.0039174568156997774, 10.281133718280593, 29.048830143480959], [164.49282671202778, 
25.614805564866785, 2.0083921055841238, 144.01116277295671], [1.5115151066807018, 0.01515211425419326, 
31.224795639114653, 1.0281694802522898]], [[8.5941661198499677, 6.1551040178885987, 70.258681959622137, 
0.0011208680343288049], [1415.2525647254004, 16.590718644897333, 19.286031315896192, 15.310076132459979], 
[0.80393467182931233, 513.24239208001916, 6.3658859963948018, 6.7299449373066729]]], [[[148.35743563372318, 
0.06288742447799564, 862.89828006183689, 2.2602309897996125], [134.58102586913591, 18.833096590582731, 
1.3508986287641715, 4.5741688350871659], [4.9829808253152352, 339.97903268955292, 0.051933953881959871, 
0.83702989264405847]], [[1.9526351491620346, 870.09434438616336, 16.592467915656908, 1.0420511218052597], 
[20.779788947462336, 1.174446125966941, 106.41687348793162, 0.96995612336544568], [2.0108561274370049, 
1.1541707905965375, 3.1541298870902299, 321.58780899817299]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank0_Symbol_rank0(self):
      arg0=Data(1.32457126708,self.functionspace)
      arg0.setTaggedValue(1,2.5302736409)
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(0.0639287997913)
      sub=res.substitute({arg1:s1})
      ref=Data(1.01813209787,self.functionspace)
      ref.setTaggedValue(1,1.0611432451)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank0_Symbol_rank1(self):
      arg0=Data(2.2846884006,self.functionspace)
      arg0.setTaggedValue(1,4.92570746786)
      arg1=Symbol(shape=(2,))
      res=arg0**arg1
      s1=numpy.array([2.6855694798894114, 0.16312791921292588])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([9.1971819304828841, 1.1442862992395297]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([72.389000622754097, 1.2970626822826592]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank0_Symbol_rank2(self):
      arg0=Data(2.21652639426,self.functionspace)
      arg0.setTaggedValue(1,4.064025676)
      arg1=Symbol(shape=(4, 5))
      res=arg0**arg1
      s1=numpy.array([[0.2933486938821081, 4.9606241062735927, 2.0417721913214839, 1.2853164260627612, 
1.0499456227730608], [3.7028655976728868, 1.001170934572813, 2.6365100719695787, 0.82410423612955774, 
3.49523244526962], [0.89663456265730868, 2.6778638320352517, 2.7192454387062397, 4.1586374080130568, 
2.9049962967419707], [1.1217652316876965, 2.8499266444045688, 3.9941585553088488, 3.2961533103938754, 
4.8280322509905496]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[1.2629980959890224, 51.850548679528011, 5.0790832260166887, 2.781628104818838, 
2.3064166037170106], [19.053743574603313, 2.2185931490955131, 8.1539886291305006, 1.926950272040735, 
16.151287980578164], [2.0414665862164805, 8.4268444216889868, 8.7090237768086354, 27.385979762644116, 
10.096678215044131], [2.4421022786752205, 9.6636775100828736, 24.025498067528311, 13.784496182053152, 
46.657336659920524]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[1.5088170297129386, 1049.0688973598117, 17.512588025836909, 
6.0631975570649415, 4.3588422505812687], [179.83899364529634, 4.0707036936631003, 40.319932430095271, 
3.1757300928201744, 134.41394191148893], [3.5156966409819765, 42.727003304736336, 45.27954251470117, 
340.74464873701299, 58.751128374701025], [4.8206562660434331, 54.38525791734746, 270.56311294367163, 
101.67490980446708, 871.0866275632236]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank0_Symbol_rank3(self):
      arg0=Data(1.91007445723,self.functionspace)
      arg0.setTaggedValue(1,2.03887271777)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0**arg1
      s1=numpy.array([[[0.80913868611567985, 4.9530927395577313], [2.3778199357445504, 3.0438567271974866]], 
[[4.5320412624387512, 2.9619886850375607], [1.1655661823312315, 2.0467038300183558]], [[4.0312321266790025, 
4.8579928594157327], [2.7986174468536209, 1.9092888523673033]], [[1.8551792128066371, 0.25103319953744346], 
[0.46046274064934245, 0.65517785698260234]], [[4.8533885944970478, 3.8765848236891873], [0.21870605031792009, 
2.7299809081727413]], [[0.1462666195660515, 2.0992108859870315], [4.2223686511204708, 2.6853651998709744]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[1.6881408066763193, 24.664265417521737], [4.6589387298581437, 7.1693013734409963]], 
[[18.781554836108096, 6.7993562971584689], [2.1260952718063053, 3.7603365776127715]], [[13.582477249987816, 
23.192113926087377], [6.1171941494375517, 3.4403782937432652]], [[3.3219926609349035, 1.1763944184475064], 
[1.3471414218010285, 1.5280512879240291]], [[23.123113221966992, 12.288965241534358], [1.1520395805917791, 
5.8514290694100302]], [[1.0992798740910361, 3.8903067155252629], [15.370852789423738, 
5.6848981306905602]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[1.7796701280506677, 34.075162084556929], [5.4409405721417663, 
8.7445845584010726]], [[25.244615802104185, 8.2491655364897767], [2.2941122013099484, 4.2976390039468724]], 
[[17.669463763447823, 31.843072635800556], [7.3428485812037074, 3.8968626324890954]], [[3.7495064865206524, 
1.1958237934315636], [1.3882338189395766, 1.5947975222619479]], [[31.738796403435416, 15.826213597281106], 
[1.1685989451986258, 6.9924457563436029]], [[1.1098222979848305, 4.4614404094987661], [20.246927178426528, 
6.7736920242494039]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank0_Symbol_rank4(self):
      arg0=Data(3.79610015957,self.functionspace)
      arg0.setTaggedValue(1,3.00563767393)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0**arg1
      s1=numpy.array([[[[4.4049017177959024, 1.6295279606965902, 0.34996170852682068, 3.8249897524769225], 
[3.166351042889235, 0.31704330393840041, 0.56944970265842709, 4.0674127332271564], [4.7531815485196223, 
3.9378749832455275, 4.1190804892406074, 1.6490622316576049]], [[0.087540448115054945, 0.47733545759568397, 
3.1046110122624913, 0.55496651223126869], [4.9994026589339793, 1.1539583254393559, 4.3285446977517674, 
3.9074072454108273], [0.39422465478757818, 0.12749582235212634, 3.3198050981771576, 2.7830434367406554]]], 
[[[4.8016423551007534, 1.3182627183354212, 2.5712661793707006, 0.69043427054151507], [4.8923630236769098, 
3.9592101125379666, 3.7599262920168344, 0.91223394177619577], [2.0812255919772435, 3.3045177969311674, 
1.996477332198537, 0.063412148396049153]], [[0.79247002703848735, 4.3525101922793947, 3.1167979206292715, 
2.121356402558547], [1.5149621718504176, 3.5217533956131613, 2.7025468362194096, 1.0269356863150814], 
[0.68822789663005524, 2.7317675329700815, 4.4823426664868622, 3.9053596848348828]]], [[[1.6704709258607828, 
0.38358040317411868, 3.3180843291402025, 0.38642831404405864], [2.1331043853922558, 4.9167900790585284, 
2.2261101298062109, 3.5510258967271158], [0.085366611692996669, 0.85934993459018305, 1.6392862656041329, 
3.3139321252721592]], [[4.2399400668492548, 0.4796559233029315, 4.4601344090922499, 0.59344213261796475], 
[3.3778564736320909, 1.2667412365915995, 0.33178212910170213, 4.7975639267592287], [1.107681780093535, 
4.0673158562596425, 2.0536296572182002, 1.6538443023426441]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[356.38996392138307, 8.7911748930326912, 1.5949460532404072, 164.42226368901473], 
[68.294607355721951, 1.5264237920842607, 2.1374878745830888, 227.19846147554483], [567.14889300380855, 
191.14339207050969, 243.41002621705448, 9.0232680500616969]], [[1.1238684474828207, 1.8903334019477369, 
62.895284718323751, 2.0965875478010414], [787.66627571196375, 4.6615625827727856, 321.87627605217125, 
183.53047169478882], [1.6919564879170577, 1.185395111021144, 83.808511283560932, 40.956409805291528]]], 
[[[605.02352164794058, 5.8038909264970417, 30.876732730682711, 2.5118550633664829], [682.85761872187595, 
196.66158396286718, 150.75332857218464, 3.3766922391691074], [16.059512953026779, 82.116724327381135, 
14.342818783171788, 1.0882709736733585]], [[2.8781122849029743, 332.3327044001154, 63.926131299011949, 
16.94266251167052], [7.5452675423436384, 109.71964173451236, 36.786389714813964, 3.9349799176508915], 
[2.5044729250047846, 38.248624144915659, 395.17542028054856, 183.02986182018722]]], [[[9.2846753307903178, 
1.6681018892605832, 83.616352693779916, 1.6744511276170131], [17.210270999211929, 705.47507179707702, 
19.483610526578207, 114.08880815037078], [1.1206141275171515, 3.1466887145107263, 8.9063605751931156, 
83.154487610286751]], [[285.99403054322283, 1.8961938841923576, 383.63996607747578, 2.2070053099870641], 
[90.556467818241586, 5.4183980619512235, 1.5567321089285786, 601.74081791606875], [4.3824984735781634, 
227.16910220157538, 15.479075041487985, 9.0810128997191981]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[127.42770564150206, 6.009123019727256, 1.4698043152443288, 
67.313581093569127], [32.607322081975042, 1.417511552104789, 1.8713751983679385, 87.895272543141388], 
[186.94745217878133, 76.217496939589864, 93.037799509475548, 6.1397012380871123]], [[1.101130485150756, 
1.6909707601108872, 30.465429935425806, 1.841784536081424], [245.13066163099134, 3.5605652445368658, 
117.15748687077632, 73.704335351859541], [1.543172236683038, 1.150627960139273, 38.606145814941883, 
21.385449809073382]]], [[[197.188119395863, 4.2662473882119789, 16.939566089510787, 2.1378824667292222], 
[217.89109194913004, 78.028188003064429, 62.662322144719482, 2.7289150218921923], [9.8785667055148352, 
37.962087467770061, 8.9989044283769228, 1.0722769941330905]], [[2.3919409318986364, 120.28847935111511, 
30.876771313432304, 10.324616307330889], [5.2973163402393624, 48.214196607451264, 19.572491618101424, 
3.0960657620363805], [2.1326977886693315, 20.212114832315748, 138.7636546194104, 73.538442915757031]]], 
[[[6.2860703585515694, 1.52520114833432, 38.533106971609669, 1.529988774774909], [10.458965062353716, 
223.82781736742263, 11.586161128219478, 49.792660843018716], [1.0984994159263746, 2.5746298970581099, 
6.0740022207983229, 38.357433238090152]], [[106.27292861316032, 1.6952944236467047, 135.41338350999845, 
1.9214439948729383], [41.152983414265528, 4.0310857094584485, 1.4406909605726668, 196.3050693195793], 
[3.3837766744129514, 87.885902342196843, 9.5830737150633496, 6.1720973205785938]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank1_Symbol_rank0(self):
      arg0=Data(numpy.array([4.1741184553454405, 0.519795084994504]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([2.4355964537231616, 2.4239681420315295]))
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(0.189748225901)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([1.3114479666188448, 0.88324188948078752]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([1.1840162859252883, 1.1829415831861259]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank1_Symbol_rank1(self):
      arg0=Data(numpy.array([4.5629531077335415, 4.5226035173374441]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([2.006161984137433, 4.6968944392526355]))
      arg1=Symbol(shape=(2,))
      res=arg0**arg1
      s1=numpy.array([0.54429231596026528, 1.3836466077330671])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([2.2846650350122575, 8.0691185694193059]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([1.4607486327498598, 8.5025409850153419]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank2_Symbol_rank0(self):
      arg0=Data(numpy.array([[2.3230795165079847, 4.8221827824415087, 1.1179672622656109, 0.30841142127845089, 
1.9851413200819312], [1.6096086764732658, 3.9126950072735216, 4.4385256122705776, 0.30619714784702973, 
4.1380665732240614], [2.8117587282802567, 4.9440601240575814, 4.939503112969887, 2.6929128066531005, 
2.231163034758477], [2.0753215133342282, 4.7312248997929389, 3.0044598203814856, 0.45190107531666929, 
3.2469155461925108]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[4.0404281653986827, 3.4163544294665611, 1.5118879543108446, 
0.29437946546373162, 1.0780567867978554], [2.89576783881741, 3.9028769330975206, 0.73348769165477845, 
0.07938039152754843, 2.3339204403596674], [4.4953178182783242, 4.3198063628165961, 1.9723794333418505, 
0.34832800113578116, 1.4109378953192262], [3.8548125014199677, 4.3474411695760482, 3.8370215424540222, 
2.6628300818493198, 3.502967139056163]]))
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(3.37693975863)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[17.225678239419707, 202.89563668419299, 1.4572769866857287, 0.018828872489005506, 
10.130332891848768], [4.9897870882674304, 100.174168790512, 153.35085498135788, 0.018376246897187212, 
121.02857242540207], [32.822513407183344, 220.73895858270379, 220.05264460185529, 28.368394770074936, 
15.030354525693825], [11.770048542533001, 190.25897890651746, 41.057107798125642, 0.068407269728229123, 
53.358883071677532]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[111.65258016089528, 63.359222458920641, 4.0385633334474553, 
0.01608913378675194, 1.2889285546835023], [36.25336587995961, 99.327850404171613, 0.35110716942182546, 
0.00019248543946410841, 17.498644516205651], [160.07833311906862, 139.93443276512832, 9.9120853018333897, 
0.028400263939079649, 3.1980051958954356], [95.257176207793577, 142.98049730958152, 93.780676099539292, 
27.312359649418372, 68.948970836046414]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank2_Symbol_rank2(self):
      arg0=Data(numpy.array([[0.59630963484916111, 4.0780288829250297, 3.9624852325319715, 1.6846092385048308, 
1.5256146923158691], [4.757631930992182, 1.0907398497011196, 4.148123546971151, 4.0562436130675881, 
2.6108552407868424], [1.0074452100035769, 0.99709772494136162, 1.8744273136209786, 3.7094937192733233, 
4.1546089691141521], [0.57260459175440237, 2.3589393541590273, 4.924974054544875, 3.4893242115523364, 
3.1063699363353638]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[4.7034145115588437, 3.1058910294922644, 0.46609292541692238, 
2.559993118201445, 4.0669869979567643], [1.4654501239016176, 0.16850444114669036, 4.3346942370726795, 
3.1553533962062983, 2.9084222061280305], [2.7262610536333303, 3.1922848487892286, 2.1969485452869071, 
2.7956479883470142, 2.5177288459753453], [4.5487599070806581, 4.1540990901306865, 3.0231741775567742, 
0.41122926037550372, 1.9398820473748268]]))
      arg1=Symbol(shape=(4, 5))
      res=arg0**arg1
      s1=numpy.array([[2.7416079864040039, 0.63224112057152015, 0.52397996251523871, 4.8186209452274529, 
4.8197279878883448], [0.56130320504698583, 0.32720461484888569, 1.3969636968032824, 2.3180021064030196, 
2.5834668247760675], [4.1453223702807236, 1.3363923630493038, 3.9507704317368528, 0.16333265279252182, 
2.7972494853881149], [0.86945458388052155, 2.6832538506212087, 4.887818914511139, 3.6271371548546725, 
2.75026190572795]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[0.24234371792801312, 2.4319339752293532, 2.0574205357072062, 12.342799762492653, 
7.6586873023844362], [2.4000575174668723, 1.0288274527220769, 7.296529975340869, 25.682233085798782, 
11.932859269490285], [1.0312260897181611, 0.99612331632723761, 11.968541020974053, 1.2387614177662742, 
53.725832769020172], [0.61583698676696086, 10.002158302968281, 2422.9536603412525, 93.024473666843974, 
22.585263073597048]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[69.741863808230832, 2.0472947074343475, 0.67032626860490152, 
92.714577147033751, 864.03053183050076], [1.239253282133056, 0.5583977571201526, 7.7590376050144778, 
14.348097890590344, 15.770433991167712], [63.909798823560827, 4.7171270918814718, 22.410518069564063, 
1.1828377295495995, 13.234943601604444], [3.732571168689129, 45.659420384423747, 223.05757022915353, 
0.039831784997586613, 6.1866779706398747]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank3_Symbol_rank0(self):
      arg0=Data(numpy.array([[[0.77950308939895752, 4.3805548260973382], [0.88474894743067611, 1.3452014391329203]], 
[[3.7854598959839394, 4.9704991398495943], [3.0458776815645172, 4.8455783000244619]], [[1.2441092116065608, 
4.688538581310401], [1.671806686362181, 3.2325779728211712]], [[0.79967174283782094, 2.6748127679433447], 
[0.041094976369996539, 3.574733763440356]], [[4.4276880614945551, 3.7379180844041517], [1.8782337415444352, 
4.7504197036708504]], [[3.8403312528202007, 1.5779731193016617], [2.9328315539468734, 
0.96308353265894842]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[4.9699559210836597, 4.9746308500577117], [0.022558252522978529, 
0.60650109253941298]], [[1.1539851248785145, 3.8022914672700971], [2.8928894244598164, 0.2171991993210364]], 
[[0.38652771744016162, 0.88729099638551556], [0.56577903066370461, 1.7002692899419745]], [[4.705992629462397, 
1.5282588669485644], [3.21592810975489, 1.1828145472915479]], [[4.9189559445043196, 4.4010564854657321], 
[1.9963853835684535, 4.090557579170131]], [[4.6596884837703936, 3.2681331204800053], [3.8176762647328109, 
0.71084564822671559]]]))
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(1.26846276344)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[0.7290796213289864, 6.5126209312892485], [0.85613690738564474, 1.4566725085520338]], 
[[5.411553791979034, 7.6446476346617729], [4.1074401312964186, 7.4017667246756425]], [[1.3192420395076057, 
7.0988185280236191], [1.9191247382944929, 4.4293902404456409]], [[0.75309050183882864, 3.4834192798972978], 
[0.017443861433090679, 5.0323291050158589]], [[6.6016346226208071, 5.3254897455563261], [2.2245451386508615, 
7.2178743830661016]], [[5.5112475542541164, 1.7835363579929249], [3.9150409749664123, 
0.95340700365462572]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[7.6435878842712057, 7.6527090907181172], [0.0081513452033876062, 
0.53031016942318554]], [[1.1992194154517433, 5.4420934775760763], [3.8475321085627803, 0.14415487873813559]], 
[[0.29947047847650432, 0.85925832691547266], [0.48555869590008138, 1.9606638085017052]], [[7.1323566886507752, 
1.7125645091405592], [4.4004712864667077, 1.2373485807510232]], [[7.5442322969766487, 6.5513080026321733], 
[2.4035259638804325, 5.9706736463740624]], [[7.0434562201188839, 4.4912793394341302], [5.4700398652057585, 
0.64860829551904087]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank3_Symbol_rank3(self):
      arg0=Data(numpy.array([[[3.1169942480278068, 0.61776142231111431], [0.95674772100048089, 0.34421176628441835]], 
[[0.6506945984036081, 4.8830603341828676], [4.0249016382143763, 3.3278663714243883]], [[4.9983574706092258, 
3.2474993968614934], [2.5808769965866158, 4.109991035675316]], [[0.55600005166088318, 3.221761195737145], 
[4.7759470225132832, 2.0771911950652062]], [[4.9104005316452373, 0.95539520734015837], [4.8473936661274379, 
0.90406626611176422]], [[1.8147239674955524, 4.117038144480035], [1.0150523105067939, 
1.5679276757509482]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[0.23370344041087943, 0.21901065352481588], [1.9069456274733536, 
2.1217290550984149]], [[4.2784513533398405, 3.7430970629295257], [2.5930828337136291, 2.6197443766101225]], 
[[0.86365122736275213, 4.4265416957568542], [2.6295017944910084, 1.8927078105813666]], [[0.67336568761223847, 
3.2124588196817956], [3.0857232575673219, 3.2289461138050934]], [[2.3075535504139486, 2.7376237866413837], 
[1.2313517197639763, 0.61482894164230362]], [[3.0071200019309283, 3.3372642509752155], [1.4536186329108967, 
2.3261369015078981]]]))
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0**arg1
      s1=numpy.array([[[0.11905884754297856, 2.9655937822878449], [4.8164299388691392, 0.11760441861579077]], 
[[1.582157324334847, 0.030325011975619182], [4.5391626356410981, 1.9111707479641129]], [[0.7190645199566712, 
3.5013050040419027], [0.66612345301588516, 0.29826763521231692]], [[1.0625575970077761, 0.37471520184865481], 
[1.3721111625466029, 4.5780537168616933]], [[4.1144344916811306, 0.068918476905670281], [3.0734321788703389, 
3.313329058122747]], [[4.5376812316022965, 3.7619471411779135], [2.5908690232568672, 1.7370280792951358]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[1.1449424015897263, 0.23969524471003428], [0.80818762785169129, 0.88212200744957991]], 
[[0.5066795457759532, 1.0492635731164728], [556.0099092724073, 9.9528659257316239]], [[3.1805477938546356, 
61.814356202149881], [1.8805628834690302, 1.5243646399668422]], [[0.53595376484824486, 1.5502091674709366], 
[8.5456503797288583, 28.406933021495398]], [[697.51778518378865, 0.99686017608842836], [127.89784901263251, 
0.71594071910792723]], [[14.941695760514273, 205.13316123343245], [1.0394668819773236, 
2.184171780660682]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[0.84107358785010755, 0.01106847579967757], [22.399061924952701, 
1.0924968125194789]], [[9.9722857702455041, 1.0408382399504701], [75.575302842097884, 6.3003431904799188]], 
[[0.89995998176995784, 182.83909996465306], [1.9040904143082436, 1.2096091475212742]], [[0.65691136671141348, 
1.5485304228342174], [4.6930174204198805, 214.04627557918383]], [[31.200693233240457, 1.0718725122937658], 
[1.8957598445735309, 0.19955999885272607]], [[147.80724652790124, 93.103875137778942], [2.6356511037977408, 
4.3336692302015445]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank4_Symbol_rank0(self):
      arg0=Data(numpy.array([[[[1.0161807738806319, 4.9934000237806426, 2.6109415809074026, 4.6358751081809393], 
[2.690228614962479, 3.9734758220596045, 3.8644269031191252, 2.1200104831294735], [3.0581265900440857, 
4.8196737484743366, 4.1194641198838555, 4.3601245022543917]], [[3.0846016790282316, 0.083033574028380158, 
4.8542947691599627, 0.91181022091156883], [3.9538315481452044, 1.8372173546438304, 3.7895717843350893, 
2.9521122951264798], [1.7334115281981752, 4.7188295816002892, 0.38774872296983809, 1.6685062524449819]]], 
[[[4.2612952904770465, 3.4791331866207531, 3.2132713477425683, 2.6231184628904818], [4.5344963130876303, 
1.7846047536110858, 0.24123831077407637, 2.7265045996410424], [3.4868925345602588, 3.8635189868650235, 
4.5812825710850875, 0.70790463331960218]], [[0.13800174765339407, 2.6118574997827144, 0.4724488397042052, 
4.433727097544506], [3.6485903148717944, 3.5097703245554608, 3.0972709386215902, 4.130036401746584], 
[1.0238918867611038, 4.5648929366646316, 4.6636999116742821, 2.7421722603841707]]], [[[1.5642290642575911, 
2.1902062588734021, 4.8328914230627937, 1.0991490647036595], [2.0131570986907596, 0.10175041391839793, 
0.50588542413559301, 4.8424625932887162], [3.1469919244043076, 1.0449992449580208, 1.6328632813674036, 
0.62344241657922617]], [[1.1592508803749391, 1.6022592628733481, 3.2573585481745733, 2.7727321836126517], 
[4.9106187845963989, 0.88286476887169851, 1.5758678358380962, 1.128449841763649], [3.7272334190066601, 
1.0442933191287467, 4.992108681465159, 4.4741123797093829]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[1.0715397330571592, 1.0768365911926971, 4.9531592877285604, 
3.9377288511238513], [2.7979985614622249, 0.27900974364231701, 0.15340456187854898, 1.6553541822862385], 
[0.17871304385101094, 4.7563925687374784, 3.0319581306919559, 0.89977161775715553]], [[3.7475072064743817, 
4.921812267949961, 0.93843041843717445, 3.0480962153728468], [2.112553868222971, 2.9107527059850891, 
2.5888278909755469, 1.0213427041719825], [2.7151630167264771, 1.7290174681629371, 0.44078453397383605, 
1.0049916124899156]]], [[[2.928722430754068, 1.2862784661464339, 0.96554950847284715, 2.4411589692199045], 
[4.2159684572039389, 3.954043341849236, 1.7864056795194061, 1.5948812975102813], [1.8504183760842623, 
4.8393037360746911, 2.5231540100041894, 4.8398256214210464]], [[4.4102459363946664, 4.0532749948798754, 
1.9008805422320514, 4.5931628260640442], [2.3772077504645126, 3.9342126784828872, 2.3573332479337092, 
1.4847542467464161], [2.2749866252540474, 1.9398644535229925, 2.9228921218681685, 4.7698912237408768]]], 
[[[4.4486041340292095, 2.8365665513643714, 2.1882345320570828, 4.3845982587228596], [1.6689280953822758, 
3.3693710449803249, 1.0604492512871586, 0.093698717761487763], [0.24109994475275801, 1.9165440843671577, 
2.8894011025096118, 3.5720057186209999]], [[4.0711678089783412, 0.71450971028370791, 1.2088612434668287, 
3.0612681853557051], [1.4338050848387607, 2.1256586381987148, 0.48066092150280837, 3.3900573415439847], 
[0.65465564313408575, 1.194796849907505, 4.5959303761360601, 0.039505946135720484]]]]))
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(1.18853581822)
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[1.019260636100993, 6.7619053588062972, 3.128803921798561, 6.1904385616709945], 
[3.2420510122232695, 5.1538964581191156, 4.986222636585401, 2.4426678250543254], [3.7755584763686341, 
6.4832225657708413, 5.3797268283381943, 5.7552916601955291]], [[3.8144387134172777, 0.051939091610215274, 
6.5386109691352088, 0.89607633910535356], [5.1236265815472182, 2.0604603541844524, 4.8716390526214859, 
3.620510068230816], [1.9228402496131398, 6.3223159169065752, 0.32432272822061281, 1.8375729930427491]]], 
[[[5.6005770500488214, 4.4010645953810288, 4.0042865155515033, 3.1461547503073883], [6.0298748066688361, 
1.9905208491204809, 0.1845080345508498, 3.2940759824948844], [4.4127331061493136, 4.9848303295572638, 
6.1038917018347592, 0.66326888220124192]], [[0.094999653894413533, 3.1301084841355693, 0.4101653790917007, 
5.8709458246850685], [4.6569954844105226, 4.4471652048532633, 3.8330665745855819, 5.3961404905075847], 
[1.0284599096577698, 6.0779466435764995, 6.234624016311936, 3.3165861456075794]]], [[[1.7018956440886388, 
2.5390931668806114, 6.5043600257217173, 1.1189153539685697], [2.297044936973395, 0.066133454243125389, 
0.44489282270935487, 6.5196728703553051], [3.9063107443879259, 1.0537073828278143, 1.7910119855459075, 
0.57030621966199524]], [[1.1920025363228119, 1.7511859139949806, 4.0696690601381675, 3.3605621460872181], 
[6.6288799460568288, 0.86236933018890449, 1.7169567164519057, 1.154455097748581], [4.7765401205153974, 
1.052861426913057, 6.759827022628639, 5.9345588931638469]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[1.0855902021078083, 1.091971225107639, 6.6971880954172942, 
5.0988350560037299], [3.3969898046655262, 0.21933036653521729, 0.10773073617962339, 1.8203701736564144], 
[0.12916977968684343, 6.382176282797535, 3.7371908537007932, 0.88203249126235572]], [[4.8074356972288195, 
6.6468427743305494, 0.92725435035433856, 3.7608448151348717], [2.432459929097341, 3.5603027573745702, 
3.0973331785687268, 1.0254173123661972], [3.2777964174044669, 1.9170484180795122, 0.37770285811137821, 
1.00593549810439]]], [[[3.5864416733260889, 1.3488031222454062, 0.95918858977117283, 2.8884962124547875], 
[5.5298440877566515, 5.1239527836984635, 1.9929085180435138, 1.7416060452966906], [2.0780686381310232, 
6.5146184065759556, 3.0041703520395142, 6.5154534285280183]], [[5.8340094920389527, 5.2771481190665765, 
2.1455952616148499, 6.1227092854726894], [2.7987832300426101, 5.0934241367559823, 2.7709945782111136, 
1.5996228973765358], [2.6563307326933896, 2.1979944857452987, 3.5779575329649607, 6.4037095514786779]]], 
[[[5.8943668035496186, 3.452714525823485, 2.5363766289603102, 5.7937075366888973], [1.8381251851153453, 
4.2365339543792304, 1.0722489930020209, 0.059960964445216723], [0.18438226160148072, 2.1666248950577311, 
3.5292840728385744, 4.5410458746462217]], [[5.3048471523363308, 0.6706307139366382, 1.2528739123761419, 
3.7801687881922774], [1.5345961642531996, 2.4504044985696787, 0.41865283390595914, 4.2674659354765909], 
[0.60440042408744099, 1.2355683341877888, 6.1270942311992007, 0.0214824137731329]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_taggedData_rank4_Symbol_rank4(self):
      arg0=Data(numpy.array([[[[3.2006240311647018, 2.9720940029604108, 4.5216393012472444, 1.71646401225425], 
[3.2330440012385155, 0.77660993613082818, 1.7085292564813197, 1.5331521312224843], [4.9241559847502616, 
3.17219570419913, 2.0583605870435675, 2.1464010713673778]], [[0.4981802049220026, 4.9586953085058898, 
0.12124481907540632, 3.6085483770196225], [1.0717059492409589, 2.9793872463805875, 4.6512010825170478, 
0.052693503701464812], [3.1234157960547586, 3.1807078442878201, 3.0711218238380837, 0.62626186557961416]]], 
[[[2.3526669171435608, 3.9432736357042959, 3.6949884223785774, 3.0656467359602484], [1.526873373209588, 
0.85787489946733975, 1.9537576708349311, 1.8906968981140191], [0.43707711227754997, 0.52262346408345284, 
3.3802082611808948, 3.1125683718976549]], [[0.14872667689757699, 2.8609947855191695, 3.6072374886258589, 
3.9911256241130348], [3.1398999332694757, 3.9432148134422511, 2.6903306631856934, 3.2695430333250259], 
[4.9495533376432448, 3.7188694128981212, 3.9013866265184709, 4.8587096325735306]]], [[[4.858808778745539, 
3.5528425167048492, 2.8168883255415871, 4.0350306411536705], [4.9690989802364207, 2.0982513660266968, 
0.56240671856112046, 3.1201020435847555], [1.9749921424144206, 3.0778305008453368, 3.6504900651425003, 
1.3166995154715673]], [[4.5916550450781548, 1.982798339896618, 1.3059622958441095, 4.1050790128185772], 
[1.5250249977523065, 1.1023445584400378, 1.6605011229768303, 4.2013460626476054], [2.7651730472155274, 
1.6633914953920366, 4.2742826352063243, 2.7362199487174177]]]]),self.functionspace)
      arg0.setTaggedValue(1,numpy.array([[[[4.2187369865969551, 2.0109695073692389, 0.63063226866830679, 
1.5500082693809558], [2.0322212196523393, 2.3491913703216833, 1.6661798158715397, 1.4896631368285105], 
[3.5256181992808897, 3.500757460249964, 4.2250807526215386, 1.7802059830274755]], [[2.8007600022115553, 
1.5317620716312019, 2.7852687422102091, 3.2038146990293268], [2.2399767526908292, 1.6470361419992035, 
3.0990027198402399, 0.65178592985792294], [2.9520991528003182, 1.8366515606844738, 3.2343120802167209, 
4.4097538236857581]]], [[[1.8613918411920738, 3.1059757127453413, 1.9505570310412725, 2.3695430968411331], 
[2.5474242224802608, 2.8791743223540518, 2.2038859445335235, 4.7411594339816343], [4.5775610018479682, 
0.69890246562149327, 4.7501681399562008, 3.1549338941723319]], [[1.1531579286066411, 2.3997463172746878, 
1.655562992873266, 4.1015895297227942], [3.3690990722016285, 3.3755213296533642, 2.7423729009127613, 
4.1913880700500901], [2.3661626337011334, 3.0458959124517788, 4.3686016653898561, 0.77343820499198057]]], 
[[[3.3558376488016104, 2.5243708513229102, 3.8945010822777144, 4.536873547928872], [4.2745931873470466, 
3.6386522195910538, 1.467622343077492, 0.27452461358153885], [0.012677082028788042, 2.9753399056349612, 
3.5790583898753328, 3.2385470165834542]], [[2.8911064029083624, 2.0176114494904018, 4.8036903728072442, 
1.3965282836953588], [3.4226531620470593, 2.1786337952764891, 4.8262226395609291, 3.2621485511361521], 
[4.7203057737596366, 3.3001920777928646, 3.4621841562189295, 2.4495724419553846]]]]))
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0**arg1
      s1=numpy.array([[[[0.017172498798869864, 4.8556776900254048, 3.9466069595635616, 0.23575742101085581], 
[1.0191489167481367, 3.8028841410994203, 0.90220337520383964, 0.50509097047474505], [2.8485106527446868, 
1.9303998203800037, 2.3956518227796653, 4.3747683397737092]], [[2.3762269540323002, 3.4108535443664616, 
2.5476671993317543, 1.1541049197121374], [1.3600097254399781, 0.34594839225069485, 4.511120804473232, 
4.8487939829992284], [1.8813373200568329, 2.1123920348032881, 2.8111098516388942, 4.9534040909472239]]], 
[[[1.4745572007380618, 3.1891933153330916, 3.9822742378759908, 1.4744065172990313], [3.4472475606733348, 
2.6208678681011945, 4.3260996614460909, 0.30001555896426541], [3.8315325661604813, 0.91062276377386975, 
3.5556860746776584, 0.51974194816111396]], [[0.045392502466932827, 0.096783116069442385, 2.9493425214108968, 
1.6449205996923515], [0.52191698509503504, 4.3431530925237372, 3.7132809838646068, 1.8215259649891371], 
[0.91550956752726043, 1.9490728208853261, 3.8003165445192022, 1.3616725975850601]]], [[[1.0847835647619328, 
3.4174010452921721, 1.3538621870424268, 4.8171839377497063], [1.4937735542845021, 2.879772185099243, 
1.9077743622712586, 1.0705751702037787], [1.7149531535125937, 0.067795817626320781, 0.44706556768099398, 
4.6104994692308887]], [[4.2249770173725709, 3.9697072869886947, 3.9756262794629116, 3.601625321462107], 
[0.062733243685009446, 2.1073882964109689, 0.48369641959960019, 1.9217999779272898], [4.1532968906695649, 
0.29103029262001745, 3.4712427123270939, 4.410824754570883]]]])
      sub=res.substitute({arg1:s1})
      ref=Data(numpy.array([[[[1.0201784412141055, 198.17060181447593, 385.65184967493582, 1.135839250632015], 
[3.306512171642761, 0.3823442049193797, 1.621334904636043, 1.2409018572256385], [93.780925146117724, 
9.2859256311610938, 5.6375220195504463, 28.259094400919615]], [[0.19095083688725506, 235.39457032445074, 
0.0046289103007681638, 4.3976454961844729], [1.0987608205923123, 1.4588930574873655, 1026.7551287776535, 
6.3396225950855539e-07], [8.5224705021067297, 11.521982998133417, 23.433951219944998, 0.098457811334582729]]], 
[[[3.5309139235542459, 79.488241743458659, 182.13396967931382, 5.215925351259723], [4.3014464228538438, 
0.66913384719600588, 18.127404263650003, 1.2105727046882528], [0.041955270037669509, 0.55382990319731729, 
75.990078096150853, 1.8042411254701407]], [[0.91713371706844349, 1.1070905513170093, 43.98444740138892, 
9.7443797863419395], [1.8169743653644337, 387.14144494498265, 39.444748360321249, 8.6526977708750987], 
[4.3239615779662524, 12.935178836155854, 176.53032170402824, 8.6063055154383115]]], [[[5.5556675772442459, 
76.127023967807077, 4.0637416935175956, 828.84944685867265], [10.966830632323074, 8.4503815367565878, 
0.33354370788928039, 3.3809974636366951], [3.2127783678566253, 1.0791975446620139, 1.7840533834054269, 
3.5554391898809672]], [[626.33240639633607, 15.139405197515558, 2.8899936852128745, 161.79098497472154], 
[1.0268276587276473, 1.2279455644460764, 1.2779942497465298, 15.777126956037312], [68.328819515021465, 
1.1596210418305877, 154.83866262404433, 84.761078425138805]]]]),self.functionspace)
      ref.setTaggedValue(1,numpy.array([[[[1.0250285097988463, 29.733035293118729, 0.16210459115045778, 
1.1088496316535343], [2.0600050853513103, 25.737104674864426, 1.5850326719660581, 1.2229965211852374], 
[36.208082217303108, 11.231823531281924, 31.570596805403202, 12.466637957289198]], [[11.556572257398606, 
4.2821358559120686, 13.594815945615958, 3.8334801788173443], [2.9945707715838452, 1.1884149299585596, 
164.42263038201077, 0.12549700193153193], [7.664342128988376, 3.6118362340853811, 27.10522305515612, 
1556.125627773931]]], [[[2.4997195101266017, 37.129127873239369, 14.305114169334903, 3.5678611586714224], 
[25.114910006161519, 15.983822216881409, 30.526029171546885, 1.5950558124684695], [339.81396588255933, 
0.72164264450866578, 254.77963173651892, 1.816963369262611]], [[1.0064895891453465, 1.0884126623772876, 
4.4232948759932054, 10.191959599952058], [1.8850305920842412, 197.0902885255615, 42.35325893560389, 
13.603198175882387], [2.2000948397337123, 8.7658870979951917, 271.33386145526475, 0.7048102725668427]]], 
[[[3.7186064672489487, 23.676560814155742, 6.3007232512082227, 1457.8663257374251], [8.7581817156913662, 
41.245919274253097, 2.0790384352600753, 0.25058730637803195], [0.00055817061243076092, 1.0767225287803224, 
1.7683611114155342, 225.40801571796231]], [[88.712562184430894, 16.222425153892441, 512.49249281408129, 
3.3297743341338832], [1.0802451310345724, 5.1604253664135173, 2.1412058874671684, 9.7017809470738108], 
[629.79332804265857, 1.4155024330124149, 74.510131229520084, 52.024638590061201]]]]))
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank0_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(3.89201973813)+(1.-msk_arg0)*(2.54793799473)
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(1.19010675911)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*(5.03931257301)+(1.-msk_ref)*(3.04374385558)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank0_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(4.88725732399)+(1.-msk_arg0)*(0.446085930048)
      arg1=Symbol(shape=(2,))
      res=arg0**arg1
      s1=numpy.array([2.6436168652771852, 2.5263281686556338])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([66.316927996841926, 55.056023448709631])+(1.-msk_ref)*numpy.array([0.11835780466515407, 
0.13011163831115394])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank0_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.221444781589)+(1.-msk_arg0)*(3.92339295268)
      arg1=Symbol(shape=(4, 5))
      res=arg0**arg1
      s1=numpy.array([[4.6753994855762322, 0.74230895194578139, 2.1240057964580989, 2.0718366932114969, 
2.7712387841020072], [4.8863576750157955, 3.3414363161035081, 4.4581631371080581, 3.9608316372247754, 
1.8708117955421679], [0.79134090721377925, 2.3698715200137062, 0.74563755408533605, 4.505431587220242, 
0.4266953364877229], [3.3582640582818506, 0.67592276895656023, 2.452334295239794, 2.7354796748175381, 
3.2941264764122526]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[0.00086866920141233242, 0.32657631384691554, 0.040676166994520334, 
0.044004484321484766, 0.015331132004679567], [0.0006320228919676138, 0.006490022445223345, 0.0012052761606645113, 
0.0025509778179644465, 0.059581973194475848], [0.30330653494922644, 0.028077810453111935, 0.32494161295940505, 
0.0011223757616437124, 0.52556667655735056], [0.0063274466863977204, 0.36095246360643463, 0.024795450879728644, 
0.016180314072261656, 0.0069698203697718527]])+(1.-msk_ref)*numpy.array([[596.49404196608316, 2.758547548814811, 
18.236484447034929, 16.981275833658771, 44.175210333542509], [795.87329080398752, 96.312603680055275, 
443.24228674444652, 224.59209670569982, 12.901182307496537], [2.9497748837706252, 25.521265789520019, 
2.7711276915218876, 472.82742687229063, 1.7918956896122926], [98.553740344816347, 2.519239939249529, 
28.566511661869622, 42.067800778483488, 90.281149199634811]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank0_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.664360304691)+(1.-msk_arg0)*(1.92939831153)
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0**arg1
      s1=numpy.array([[[2.2563487326899501, 3.8457355666351378], [1.9663888270560579, 1.8151676332476596]], 
[[1.9217762174895128, 2.4048860435314841], [4.813058542284506, 4.3600788139918567]], [[3.6808048059364755, 
3.3840843094394528], [3.0170356512966436, 2.5508193338300997]], [[3.8450720577204835, 0.67106380009782396], 
[1.9102190488265116, 1.1868982505742023]], [[0.59290803321196273, 3.8932917892980128], [4.4531448032102059, 
4.4445121819332796]], [[0.41428618209261464, 0.051071515023622144], [2.4927355451451532, 4.7132104714638183]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[0.3974484050535696, 0.20749684554046716], [0.44748303191812167, 0.47602844004520567]], 
[[0.45572159368027115, 0.3740253845053354], [0.13970710514501208, 0.1681380586023484]], [[0.22197424916106345, 
0.25061038586978807], [0.29119610241028077, 0.35235777758344106]], [[0.20755315311788622, 0.76001390077105713], 
[0.4578804680988246, 0.61547615679232826]], [[0.78469649471659941, 0.20350059736582529], [0.1618593719639588, 
0.16243176848774246]], [[0.84415908115348237, 0.9793318673956567], [0.36082725088606604, 
0.14552953583002134]]])+(1.-msk_ref)*numpy.array([[[4.4056635605850065, 12.521522255051883], [3.6412493793131264, 
3.2967695028025417]], [[3.5360386886022686, 4.8574364755138051], [23.645706011695786, 17.557470752187442]], 
[[11.235238076367091, 9.2446772457911379], [7.2632004540196622, 5.3463783054067413]], [[12.516063266569189, 
1.5543052033699192], [3.5092825682082269, 2.1815579144301767]], [[1.4764846419069859, 12.919054099710024], 
[18.664872368805973, 18.559278355021636]], [[1.3129444351849795, 1.0341342656584545], [5.1461364913354837, 
22.143866788073627]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank0_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.749272636001)+(1.-msk_arg0)*(2.76941992383)
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0**arg1
      s1=numpy.array([[[[2.6380701938784497, 2.9629720449422932, 2.0446868934255567, 4.8175127309430978], 
[1.3635679921902024, 4.0348929972817613, 0.48578845331301845, 0.6568058052005824], [3.6259594179773944, 
4.8833845034250496, 0.17118845347763492, 2.8807344472514744]], [[3.7872819664046382, 2.0283029183322161, 
4.195326597362893, 0.8191596059342755], [4.662510379944262, 4.789692391303805, 3.4481515021939262, 4.9761891209019513], 
[2.2719212875559021, 0.82975847999621222, 4.439414472830145, 1.8852687223641864]]], [[[2.7020104975723047, 
4.5020705352915407, 1.5409389663367554, 0.97655922243535154], [4.4327920846767279, 1.2070229778784267, 
1.180070165465704, 4.1464377615296018], [2.3485800159004779, 0.92564409151457272, 1.3159639335830047, 
2.1511673414313619]], [[0.67131880064612526, 4.8955556418535782, 2.2506779725819133, 0.87252916617244536], 
[4.4692090605181223, 0.5599807925656004, 0.45977109117815407, 4.241776420974686], [0.10366241812321508, 
3.474014078895685, 3.7814907650865814, 0.42360844286683369]]], [[[4.6735648406283454, 4.8096752492979133, 
2.4020835023858504, 0.028018252523416498], [3.0625590290547757, 0.58899929292552688, 2.9657775813630796, 
2.8232788482355757], [3.5721497242605142, 0.1026911589702196, 0.96937822981287203, 1.4745359217544272]], 
[[4.4358232648059834, 4.8858574700575055, 0.12958247843545609, 2.3143102578684847], [4.4995406401536417, 
2.7438403794734514, 0.79036172567538532, 2.8291086208021698], [2.1403277048976621, 1.7854892302428567, 
3.2175511398028198, 2.2141578239271626]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[0.46697236820910309, 0.4251688569858818, 0.55421437937030293, 0.24892928289707733], 
[0.67462579387593102, 0.31202205865545657, 0.86916353458001161, 0.82729945889608358], [0.35111490324641437, 
0.24424084521249903, 0.95178705456471835, 0.4353822920175896]], [[0.33513967826607993, 0.55684161809685462, 
0.29790193303502599, 0.78942349360018449], [0.26031974043813144, 0.25093633422170408, 0.36960623321731489, 
0.23778491271896876], [0.5190290391620882, 0.7870120299198623, 0.27763514033863773, 0.58031323020447134]]], 
[[[0.4584327319058415, 0.2726590153429701, 0.64095522784194636, 0.75435958166730099], [0.27816636635874648, 
0.70580941159230892, 0.71132202006551259, 0.30213569162871079], [0.50767024397026028, 0.76552810261749937, 
0.68395982577298942, 0.53743925812492843]], [[0.82384097660306221, 0.24338427701318588, 0.5222214683526496, 
0.77735542213054876], [0.27525763916079377, 0.85074760711847108, 0.8757155010308979, 0.29393436593140532], 
[0.97052084128694038, 0.36685728768934739, 0.33570038099456212, 0.88490447516411319]]], [[[0.25949041102751735, 
0.24949307499742268, 0.49989006439112732, 0.99194508136056547], [0.41312093974828379, 0.84365128690300062, 
0.4248246841166638, 0.44266318304703822], [0.35661109732082952, 0.97079297101378215, 0.75592484780330382, 
0.65335916612578604]], [[0.2779230891091658, 0.24406656155341364, 0.96328661097812074, 0.5127170622032361], 
[0.27285820009128608, 0.45293076326734039, 0.79601298639062723, 0.44191890575185372], [0.53912347815665429, 
0.59727019317581997, 0.39504573523467901, 0.52775565076090036]]]])+(1.-msk_ref)*numpy.array([[[[14.691046718037269, 
20.454350037996647, 8.0268769664070003, 135.27381407927967], [4.0107650751284751, 60.952499669669109, 
1.6402399176594569, 1.952376358334921], [40.186842086843704, 144.66206840658916, 1.1905067348617921, 
18.810688455066671]], [[47.364305203048531, 7.8940253918321845, 71.773735613725677, 2.3034937866731684], 
[115.51607639444967, 131.49411855954435, 33.529235226876615, 159.00485859216263], [10.117484786482814, 
2.3284979957690046, 92.033787676254107, 6.8237293167676798]]], [[[15.679752919549529, 98.099238680439939, 
4.8050084874390588, 2.7040758813612364], [91.415034073834107, 3.419585458341726, 3.3269773094822024, 
68.286943858713769], [10.939199023944678, 2.5674062272419249, 3.8209180362455766, 8.9464834982803971]], 
[[1.981453691419933, 146.46675031929558, 9.900901847104036, 2.4321880005782641], [94.869822531040455, 
1.7690057222223956, 1.597340808107695, 75.251373799975866], [1.1113710866765436, 34.42428766459463, 47.085719191365783, 
1.5395706950189538]]], [[[116.82419551043782, 134.19814758177736, 11.551938486777251, 1.0289516346644614], 
[22.638197863126429, 1.822076880682197, 20.512888611573942, 17.741362775456043], [38.043378121935739, 
1.1102720828025088, 2.6843681900085201, 4.4907429327926405]], [[91.697730193603263, 145.02693984793967, 
1.1411056053730484, 10.563916657693198], [97.846757795918251, 16.362297569123484, 2.2369031706777256, 
17.847032010620911], [8.8482427982193652, 6.1642530670809563, 26.509932823363236, 9.5393470687936492]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank1_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([3.3709569318191632, 
1.2518416586949326])+(1.-msk_arg0)*numpy.array([2.6194474552657527, 3.9603311830279626])
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(0.699591439915)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([2.3399771894150048, 1.1701585901306912])+(1.-msk_ref)*numpy.array([1.961444030062393, 
2.6191950693986477])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank1_Symbol_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([1.6325984980871546, 
4.4634392931456688])+(1.-msk_arg0)*numpy.array([3.7974296853432312, 0.81724289350233981])
      arg1=Symbol(shape=(2,))
      res=arg0**arg1
      s1=numpy.array([0.19493460590675868, 0.37275617681790979])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([1.1002656649360483, 1.7464991087435378])+(1.-msk_ref)*numpy.array([1.2970675798548577, 
0.92753082328744252])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank2_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[2.1710667104193604, 1.4559516480232357, 2.5777414584158964, 2.9313298523853257, 
0.51769963017579546], [4.7332637011187346, 3.4799432615947694, 0.89287544846598499, 3.7033053793353461, 
4.3177558633309507], [3.1132844391398979, 0.71886157115995986, 3.4275557958866809, 1.6958438413819459, 
2.2437219123332213], [3.2601863086089589, 2.9153362371666889, 4.6536974587183266, 3.0769748692539687, 
4.679079382254673]])+(1.-msk_arg0)*numpy.array([[1.0917510285611751, 2.0349695647299555, 1.9174547529203791, 
4.1175984092201476, 4.8257318447594324], [4.6930015304755255, 4.9314833261953614, 0.18415505878378155, 
2.3917899551644628, 2.510760119188574], [2.4136062308358044, 1.5105356175464366, 2.4434080721948672, 
3.9705640603320216, 4.4142940851681312], [2.0234503038025746, 4.5858966120021867, 1.4239860863791243, 
4.5128095376739035, 3.050993811823203]])
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(3.78316573875)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[18.779775455751572, 4.1420177962146392, 35.957264419285686, 58.476737694947722, 
0.082853171282090368], [358.29836883297418, 111.90699977530605, 0.65137899230049978, 141.60168268705098, 
253.09677617643018], [73.439289997168572, 0.28685689187808994, 105.66597309931271, 7.375725087754101, 
21.270319186387162], [87.434158812368196, 57.278835913940632, 336.04006980086507, 70.251208719601834, 
343.026691645879]])+(1.-msk_ref)*numpy.array([[1.3938881764903226, 14.700284663667441, 11.73805133343139, 
211.49498553743527, 385.50749280191758], [346.90396824584991, 418.45522695975438, 0.001659846678596532, 
27.087693111746457, 32.548382942851433], [28.034349489398465, 4.760824245330129, 29.366567710791657, 
184.31350289398216, 275.18010771503333], [14.387946682774784, 317.89053486869193, 3.808357224353176, 
299.14481902814708, 68.033345855427385]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank2_Symbol_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[3.570171507891458, 3.1925694331560868, 4.5018416476512648, 0.25527130916380369, 
0.43235850469228304], [2.7614123626716611, 4.8629662424881381, 0.31566780592211785, 0.75223774883725547, 
2.8344524603130581], [4.1625009680053227, 3.4352746862750698, 3.8426450917716615, 4.8222819583175216, 
0.41107111273596691], [0.44077203312096136, 4.5647980946198539, 1.7844272172243647, 2.7039700473815622, 
3.8584303383699088]])+(1.-msk_arg0)*numpy.array([[2.8083669806591458, 3.3804236512827193, 0.30556511732708047, 
1.0531902353612865, 4.3417577083547734], [0.042647457536026633, 0.095513045600019331, 3.3701513532352094, 
3.3845187763656259, 2.1086823219998951], [3.8741326371193887, 1.4235160621623837, 3.4359394711377971, 
0.14327528084815525, 2.1907628823085319], [2.1636863884214863, 3.1668527190901887, 1.7832546977954595, 
3.7831163105644987, 0.85435592651569148]])
      arg1=Symbol(shape=(4, 5))
      res=arg0**arg1
      s1=numpy.array([[1.6740108349001657, 1.0983217973929327, 2.6623064498244773, 2.2798561052986765, 
3.6845747147837828], [4.3798182023690195, 4.2698739944277841, 4.2383864724604932, 4.9591926764314964, 
1.9824609539459526], [3.2387478783732324, 3.3259184012655032, 2.1941523398057003, 4.0625995557427093, 
4.8743264039948446], [4.8481310965503583, 1.7779251415271708, 1.5207108244957652, 2.8795914146785719, 
3.1141173303804011]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[8.417972599993977, 3.5785602759130959, 54.89401322776628, 0.04446823603392755, 
0.045523959981275147], [85.521388499072543, 857.00193926760005, 0.0075430058128210555, 0.24368075613780074, 
7.8886461687139846], [101.375399563874, 60.612767354257009, 19.176435549891746, 596.73527193917596, 
0.013125179067304385], [0.018840957902037998, 14.873123822489182, 2.412442697829198, 17.538356540586207, 
67.011850455779282]])+(1.-msk_ref)*numpy.array([[5.6326999533695528, 3.8104873485274715, 0.042578313772200164, 
1.1254140167308759, 223.62815838004201], [9.9811609597114664e-07, 4.4156660315269248e-05, 172.33841809891487, 
422.54920258499891, 4.3887361076615221], [80.343152159641292, 3.2364636570138612, 15.002564341252342, 
0.00037313137629075503, 45.727033232007521], [42.176056669762183, 7.7638809912833535, 2.410032511928951, 
46.12867892129659, 0.61251300718761692]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank3_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[1.5223177457356358, 1.9540813532193144], [1.7658770955044887, 
0.048393995361373808]], [[1.7714171837169259, 1.977357651103324], [2.5846107184639711, 3.7089400428773036]], 
[[2.8603413240541582, 1.6676809382305471], [0.44275419010954908, 0.15393526575638181]], [[4.8767216354949099, 
3.047753279851193], [4.6159021329428001, 4.3066836423126515]], [[0.80713323299916717, 0.56365109921046286], 
[2.4896083503558053, 1.1823857559116824]], [[0.090970997435440445, 1.4168781494147886], [3.3308485429356764, 
1.229332657395757]]])+(1.-msk_arg0)*numpy.array([[[4.8738266448708476, 0.37837406687006453], [2.4128940532238126, 
2.5191824976804345]], [[0.68731347762430706, 2.0800642347991998], [3.7341243472476169, 4.5911875734856711]], 
[[2.7325477494968777, 2.2906139421110936], [3.3644486451163687, 4.6411730478567916]], [[4.4733489963068944, 
3.0251105244011502], [4.1822689749563118, 1.1188887264887919]], [[0.50556067184312148, 2.9433837359996327], 
[4.758898551496948, 2.6624250527447364]], [[3.4745259416081473, 2.9478500986424039], [4.6344624693894829, 
1.533474985881514]]])
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(1.64119571575)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[1.9930901042958082, 3.002574660335485], [2.5427898612694202, 0.0069420127858769061]], 
[[2.5558956598031486, 3.0614967927086982], [4.7513950253624326, 8.5950885220880409]], [[5.6113975378576457, 
2.314892597573075], [0.26259333944973678, 0.046372745994703042]], [[13.469537098377412, 6.2273831362190117], 
[12.30764818435267, 10.983802793710373]], [[0.70352398635403779, 0.39026469662582264], [4.4681583148783117, 
1.3164733429091755]], [[0.019559420904627398, 1.7716023809068993], [7.2046823707751297, 
1.4033469855089122]]])+(1.-msk_ref)*numpy.array([[[13.456416622942902, 0.20290200992370636], [4.2444374915697596, 
4.555600020117053]], [[0.54042975712737396, 3.3267961785662878], [8.6910804035262679, 12.199682654465931]], 
[[5.2058678192157588, 3.8971894283303108], [7.3243458168738007, 12.418428073578472]], [[11.69003291392138, 
6.1516337953720388], [10.467877865690189, 1.2024550430674248]], [[0.32646300963953717, 5.8812480980889106], 
[12.939594832592748, 4.9884248325848493]], [[7.721745317334916, 5.8959018271931791], [12.38897315671394, 
2.017120291385428]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank3_Symbol_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[3.2694147608330524, 2.7527199769451434], [0.21545214987730152, 3.0500289674019014]], 
[[1.6058129332411193, 2.5434141647086048], [2.9555222556655023, 3.2877205097850331]], [[0.19933270634796588, 
4.4193317443959232], [4.979239238863074, 3.7764674254066466]], [[0.71102167655485993, 0.17006789794181962], 
[4.5674879813111398, 3.8021974340236682]], [[4.7323214482854405, 2.2322683541753774], [0.58703630490664815, 
2.6080011436019808]], [[3.9829219244464094, 1.7501266378889557], [2.6933171506458407, 
3.9085099265339287]]])+(1.-msk_arg0)*numpy.array([[[4.2492652605934849, 0.37104573464329665], [1.6255362467169652, 
0.7269444706737751]], [[2.2316021664286874, 3.8401440464240175], [2.1258690550647401, 4.8611415232683992]], 
[[4.6171911065369065, 3.7927024759450751], [4.592662337060478, 4.731635762073739]], [[2.9992587663105215, 
4.1201833347031194], [4.5446842597416097, 0.84038914646896157]], [[1.579130387005959, 3.0431933475441504], 
[3.7251322657496884, 2.5245276014750906]], [[3.8889833829004306, 3.0931784383397893], [2.0210894964713697, 
1.881130788791491]]])
      arg1=Symbol(shape=(6, 2, 2))
      res=arg0**arg1
      s1=numpy.array([[[3.3768399353556626, 3.4694128090700103], [2.6611077778706638, 4.7864233330931185]], 
[[4.9555409678460247, 0.010698537162774534], [2.8076143657320407, 3.7887318420629978]], [[2.1423378448144343, 
3.7966127742235574], [1.5741855830682956, 2.1033597558551511]], [[3.9825245309777801, 3.6349263894947268], 
[3.5799239365129902, 2.0164283244391386]], [[3.6226439393408643, 4.9183593732809756], [1.7503012770341879, 
1.0972210847818915]], [[3.6053129680740859, 0.62203119125832551], [2.0971628851516746, 3.4194100307384194]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[54.611349170405944, 33.551816341891154], [0.016825760933191474, 208.00979874129121]], 
[[10.455137735826503, 1.0100372010979246], [20.958463298115625, 90.860711273194042]], [[0.031583528579419061, 
281.94806992465448], [12.515953279940252, 16.361342175787566]], [[0.25711048110146106, 0.0015972351316030258], 
[229.9296414285823, 14.777410118323925]], [[278.96748872614285, 51.911088761839025], [0.39363525259421744, 
2.8627384505791156]], [[145.85298054850662, 1.4164354746260404], [7.9869868217120832, 
105.76186523993667]]])+(1.-msk_ref)*numpy.array([[[132.34735806170755, 0.032074955115904545], [3.643215388155344, 
0.21731329247521483]], [[53.405342835947991, 1.0144990942028391], [8.3099079183543108, 399.82240904268821]], 
[[26.504634051058805, 157.77722575526394], [11.020782121972983, 26.290062276398668]], [[79.381573877626664, 
171.86061546946161], [225.84646216243317, 0.70423922334287992]], [[5.2335663139405515, 238.3353173607245], 
[9.9924053355866782, 2.7623614919775887]], [[133.82698542544125, 2.0185856905716197], [4.3738376971351727, 
8.6766294923839542]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank4_Symbol_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[4.2210862097767281, 2.3165015484157108, 3.8535125144140969, 0.63614319013276677], 
[2.840224399132766, 3.577593007293467, 2.388141911082946, 2.2047375654390331], [4.492287134665486, 3.8362964646408351, 
3.6560184014703254, 3.9137581420700429]], [[2.9588924211140055, 2.118447288580581, 4.1807417953351003, 
3.7006448311228328], [0.39768675229342254, 1.8788074831961259, 3.9844711456080408, 0.07809347949972989], 
[3.8711342212325262, 3.7149718608500319, 0.70756729697067877, 0.43624819276581323]]], [[[0.26907513287332052, 
4.0761412850153427, 1.2533550328934382, 2.2265181546120738], [3.5088611765678013, 0.98888133846137038, 
3.5410386586259373, 2.062804512034432], [4.7564598094994617, 1.3663058149455849, 2.9201920356779034, 
3.5725625634666787]], [[4.39765531797, 4.3874629438483499, 4.4532722631579444, 1.6891663405590658], 
[2.5138171273008525, 2.1734638212049497, 1.5542357323765577, 3.8993557292847467], [2.2522464326652059, 
1.6456811593504299, 1.4851747097600771, 3.3255488686978874]]], [[[4.8011603734284289, 4.4461292944450879, 
3.6977890526234787, 3.2646764452028645], [1.9072173856457764, 4.9740583772465659, 4.7380532894139735, 
4.5177381053402677], [3.6833166806964885, 3.5930438285443245, 0.88567327002182072, 4.0797253917446934]], 
[[1.9863967367802593, 3.4923001330895049, 2.5587190996826781, 1.755230594863167], [4.6422830187755952, 
1.0924400407279413, 2.4750792013100065, 3.1078250320154659], [4.2998683200566656, 2.5012122344576246, 
2.6316418431181825, 4.8509967109453545]]]])+(1.-msk_arg0)*numpy.array([[[[0.039945792184537678, 3.067453434964015, 
1.2142824898811153, 4.5543167054890441], [1.0858861964487661, 1.3137755035429755, 2.2869710498750653, 
3.9662312006456935], [0.59180288029219408, 3.174489793603664, 1.4893332362854206, 3.6246094177922581]], 
[[2.1097587290344424, 3.9209692838253027, 3.4697145472191004, 4.4844331876574737], [2.0106328395394732, 
3.883339843241794, 2.3885153844238851, 4.6842180044033839], [0.47959403795834843, 4.4538052844369718, 
0.52115820818812497, 0.68146956506355039]]], [[[4.1513075793004823, 2.2106356196209669, 3.6829268472635159, 
3.4095578643849809], [0.92778168804554084, 2.0505550394214667, 2.3581962417791229, 2.7849110748581074], 
[2.6440850468954888, 3.9351533212204188, 4.4358685424416295, 0.070263685530420922]], [[1.2102020771763233, 
0.014641556700371726, 4.4059897217018236, 0.020490263177473482], [1.9599163687040377, 3.9105138277062905, 
2.6425992487052081, 0.12820085811092669], [2.503623701367871, 1.0030419818988972, 2.7915819960528307, 
0.41467088410060954]]], [[[4.8111503042329531, 1.0720773429998696, 0.28593419328032837, 2.7354043126489116], 
[2.5521811793728593, 4.1004627320632485, 0.25942279879033042, 2.6571020799477933], [0.69191778055117226, 
2.4713808160674722, 1.4508986449085861, 4.8211740464979327]], [[2.8475170637348244, 3.6313348667143273, 
0.11878419891550397, 1.3495535255999038], [2.2619648010145479, 2.1579477142953172, 2.1655884240795409, 
4.5607342407485305], [0.47588468473046325, 0.16540502165609189, 4.3161428655579117, 2.988108533562821]]]])
      arg1=Symbol(shape=())
      res=arg0**arg1
      s1=numpy.array(3.00721465635)
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[75.994976867439803, 12.50633121610999, 57.782599581306592, 0.2565945118887632], 
[23.084939750462468, 46.213277188139855, 13.705912822314609, 10.77824143665204], [91.645226942686008, 
57.009759686387333, 49.327264383384161, 60.542076750768885]], [[26.108778957812682, 9.5588378006294121, 
73.831580911631121, 51.160189377363373], [0.062479018571655304, 6.6622789152002282, 63.891565065975122, 
0.0004675789126183552], [58.58085952446153, 51.758134982919799, 0.35336162394001208, 0.082528076782454038]]], 
[[[0.019297783239073128, 68.414857246086015, 1.9721043855197204, 11.101629409398022], [43.594504470870952, 
0.9669355121908868, 44.80781776747363, 8.8235413438107848], [108.82730249961899, 2.5563580473041565, 
25.095278510208956, 46.018142912233621]], [[85.961538704520152, 85.363798865089876, 89.272509062018315, 
4.8379328339455574], [15.991504055444196, 10.32499108131171, 3.7664516474452041, 59.874566270699532], 
[11.491897023948152, 4.4729902471536089, 3.2852766859308606, 37.098390391152456]]], [[[111.93201389233471, 
88.842594287004516, 51.041555891817247, 35.093575417317567], [6.9698524133185407, 124.49707945056491, 
107.56575858008239, 93.215510790306027], [50.443173316528799, 46.816076667090435, 0.69412903294027151, 
68.595920262571042]], [[7.8767746910450036, 42.978677215453878, 16.865979797461918, 5.4295686447304004], 
[101.159128946537, 1.3045774124821155, 15.261840560843588, 30.263739220960041], [80.34070323696676, 15.751581054399395, 
18.353214527131929, 115.46250497279942]]]])+(1.-msk_ref)*numpy.array([[[[6.227635835295454e-05, 29.096843083285542, 
1.7929473444459771, 95.503652502858856], [1.2811828278654707, 2.2720535806236151, 12.033000738069527, 
63.016049222056161], [0.20648454901252414, 32.258280229865036, 3.3130174338816314, 48.063846177673497]], 
[[9.4414262588103739, 60.878150526760194, 42.148223517841103, 91.164240877455583], [8.1693359575756226, 
59.138065030149569, 13.712359560482147, 103.93210257196748], [0.10972837322621674, 89.304645640604846, 
0.14088565633753561, 0.31560056174522472]]], [[[72.279431633968372, 10.865183447156275, 50.427120149651856, 
39.988713234801118], [0.79818308851312636, 8.6669110137735164, 13.195561190971887, 21.7592117054641], 
[18.615419706790959, 61.542823591376361, 88.227450699602016, 0.00034030814057911448]], [[1.7748901256399299, 
3.0445779900074598e-06, 86.452387509998957, 8.3649048506225469e-06], [7.5652104456080727, 60.391280920939067, 
18.583980127991737, 0.0020760439153237353], [15.79729403077487, 1.0091758491813847, 21.916329618863852, 
0.070852061423832075]]], [[[112.63385954023728, 1.2328107836351652, 0.02316730022354687, 20.616623599453593], 
[16.736715422016417, 69.649817974873542, 0.017290067604402621, 18.892380231011206], [0.33037678392750758, 
15.193363842122821, 3.0625090532330823, 113.3410267877975]], [[23.263648335824051, 48.332536178107922, 
0.0016504437263980404, 2.4632563964751952], [11.641662938111672, 10.104916840913434, 10.212894104870339, 
95.908921590674993], [0.10719597296797677, 0.0044669199408652526, 81.258618143030674, 26.891741309440064]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_pow_overloaded_expandedData_rank4_Symbol_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[4.0046281121518037, 0.17912816470252543, 4.186604586753079, 3.2008984142999877], 
[2.1732610328222237, 3.5591327971124795, 2.8011927910922534, 1.7713395741512676], [4.1863951635846055, 
1.7802519024535213, 3.0889056467458453, 2.7945043142416166]], [[2.1432610878391651, 3.5462336809826689, 
3.1735982963533362, 3.4363695811348696], [4.0108402401592045, 1.643703160814016, 4.4085847603067556, 
4.9184177076856335], [2.489140510676517, 1.0412913848971008, 4.4197321887721381, 4.2153884306131344]]], 
[[[3.7803210848208, 3.6398280401229397, 2.0236101278183094, 4.5541693738175519], [2.4112503026807666, 
2.7306223829688165, 0.73398277455182248, 4.7643620654034891], [2.7131841217568446, 3.0697770135007665, 
2.1252111735885495, 1.6953356322623179]], [[4.1911775778338658, 2.2269668597708558, 2.2085703281208002, 
0.070094318164635808], [2.4510955762529427, 1.1153170165763491, 3.4522669564720454, 1.0273284510342966], 
[4.8278478952640977, 4.7680886890999439, 0.33129633264939778, 2.7523260117636221]]], [[[1.8469163631313446, 
1.5247892210589278, 2.3006871008987484, 2.0094374714933072], [2.4763446019046906, 3.4258456412111444, 
3.6639258261790899, 4.2326446105282853], [2.169300019785017, 1.4347568208958732, 4.4959458538843062, 
1.7505402636784861]], [[0.67861301396188733, 2.428555743521049, 0.5490734228134776, 0.35199542025678943], 
[0.86806268215157323, 0.32371527967411984, 1.4475946758251814, 3.9678330916825226], [1.5956036679577594, 
0.95739272849017321, 4.6257789651238275, 3.9717686731574866]]]])+(1.-msk_arg0)*numpy.array([[[[3.6614487613448614, 
0.61866988344511109, 4.9453521033138657, 3.336621741649878], [4.1553202712985087, 1.7563864069243749, 
4.353925898955235, 3.9344446948487821], [3.2545713494721182, 2.7177272190593791, 4.6419346212272901, 
3.4445746226359395]], [[3.3008791988733606, 4.9312642523535848, 0.44612891205221544, 0.89438722979256113], 
[2.0476127735158651, 1.7710269436265909, 1.6307607302774327, 0.46507867321761676], [1.5476163668318901, 
1.612061931386467, 4.005191059815866, 1.8674995301113766]]], [[[1.2666019100210055, 2.8765962400203318, 
1.2763256783527941, 1.5716251059218727], [3.6178595086991834, 2.7484057417965286, 0.50153720741266605, 
4.8305353430307942], [0.23187698156826694, 0.95091912820189362, 3.3778532045181602, 2.4372860829606759]], 
[[1.7131674578285643, 3.9490981454586445, 1.6780019090780387, 3.5310975289923521], [2.3456985557443382, 
2.3425524483331341, 2.5987002991946562, 3.2431145596162669], [0.51806948754948734, 1.038250515543244, 
3.1896828710199445, 3.9187194770163001]]], [[[3.0512124341007194, 1.6743433902492326, 4.6959997496647707, 
1.1630338915636007], [0.14594963932603233, 4.0138834378255543, 4.4975621790330402, 0.7699000327961929], 
[3.7639380111712728, 1.2401305549207602, 1.0883746033751507, 0.28561108691022136]], [[4.1672046847087092, 
3.8258654213899042, 2.1332239334009686, 4.8998062887628464], [4.8734188379199521, 2.9680974102736362, 
1.102099707426597, 1.9454194502593292], [4.9450113758968754, 1.4216082301320121, 0.81046281085592908, 
2.1231440730976723]]]])
      arg1=Symbol(shape=(3, 2, 3, 4))
      res=arg0**arg1
      s1=numpy.array([[[[2.0254726202872, 3.6919769974914662, 3.3813686691149161, 0.09373063287153327], 
[1.6162669611842946, 4.9709325713095014, 4.0088393152109481, 3.7424030376065787], [3.4901235779635638, 
2.0539830752368236, 3.414996106475833, 0.39653630873783718]], [[1.6602396752962345, 3.4524174334885664, 
2.3263906936168035, 4.0904786995602391], [3.5263053211774347, 2.6239447885040228, 0.19431283905255856, 
0.012985596939993016], [0.30336677692584912, 3.3814046047237762, 1.8139491480051104, 1.7068169308963574]]], 
[[[3.7796058755742918, 0.5048138651560039, 4.1243863664103912, 4.4543337043262161], [4.7113584144666136, 
1.2578628350113628, 2.4614775803231095, 4.9550787549975999], [3.9142217468685172, 1.223074455845053, 
0.91758631769232324, 0.63590924526144332]], [[1.5426261315812613, 0.37308477051015732, 4.0416295548706698, 
4.9972454781286366], [2.5622727884423768, 1.2723963365437676, 0.54675032085327935, 2.2747603647550592], 
[4.9319193527986416, 3.7113682336235283, 0.5774316017156943, 4.3210733276807245]]], [[[3.3395390932579105, 
4.8861871770114211, 1.7279377363039217, 0.72713220720108918], [0.57694916336954549, 4.9349658201681228, 
1.0306577039927605, 0.45152739831893446], [1.2720221945497059, 1.554592467497466, 0.29353486797752176, 
2.3664226641838439]], [[3.1847561474031609, 4.1310775736116501, 4.4788521685746385, 1.2722456227629599], 
[2.2057576454442178, 3.530894953720086, 1.073018364096219, 4.8520723881119912], [1.3176068423805134, 
1.2504792548799262, 4.876110890905319, 4.6619458105804341]]]])
      sub=res.substitute({arg1:s1})
      msk_ref=1.-whereZero(self.functionspace.getX()[0],1.e-8)
      ref=msk_ref*numpy.array([[[[16.613962329168, 0.0017486311486089434, 126.69063036727491, 1.1152171877115666], 
[3.5064109586864305, 550.42075885638508, 62.133555452100822, 8.4966065094058578], [148.01277927509608, 
3.269525105620906, 47.063145201168908, 1.5030590809306388]], [[3.5453897266571519, 79.072666513352345, 
14.682695727663457, 155.92116917194539], [134.02693074666809, 3.6839070193302299, 1.3341221841214275, 
1.0209013213436253], [1.318706764357056, 1.1466185738458463, 14.815470274042406, 11.654224865734468]]], 
[[[152.34591889853269, 1.9197355212658576, 18.305659634578621, 856.59235604444245], [63.223779415997093, 
3.5379957805086573, 0.46707731489374876, 2288.5846065945393], [49.74322894940898, 3.9424656742504154, 
1.9971913605873199, 1.3988972384028238]], [[9.1207777229361628, 1.3481142712254719, 24.590656887978088, 
1.7044869074222006e-06], [9.9459704275539202, 1.1489720373711589, 1.9688323547616904, 1.0632512140862307], 
[2356.22906167467, 329.2953355431639, 0.52839405082563173, 79.42832429292362]]], [[[7.7590926484431781, 
7.8559270157256638, 4.2195615756873579, 1.6610213995770469], [1.6873647533234348, 435.57281863570262, 
3.8127293996659732, 1.9183701053366202], [2.6779719353579701, 1.7527757710519118, 1.5546253576726097, 
3.7622528418241332]], [[0.29090937370999143, 39.075342794630501, 0.068209231168206222, 0.26490106217761356], 
[0.7319115303721988, 0.018639648195352566, 1.487226641525681, 802.09239984700025], [1.8508678118182738, 
0.94700788085623444, 1751.9238412049178, 620.05569094382963]]]])+(1.-msk_ref)*numpy.array([[[[13.856822049793017, 
0.1698522615234575, 222.50343712128958, 1.1195664976441784], [9.9960868244070049, 16.443333318172524, 
364.05869186298531, 168.38120432754178], [61.470491313669029, 7.7956349835148826, 189.13615479213124, 
1.6330285111956921]], [[7.26188705507567, 246.82050235190837, 0.15293534435498729, 0.63345600430807603], 
[12.518574538497633, 4.4805273946663906, 1.0996896850839557, 0.99010814263927649], [1.1416620062301488, 
5.0262201940346474, 12.39161493146015, 2.90396926889709]]], [[[2.4430889022943201, 1.7047018615804435, 
2.735433708206461, 7.4921037130989347], [427.62797298014726, 3.5670030379673809, 0.18293758482727099, 
2450.4757056595199], [0.0032770090774681578, 0.94030333258667465, 3.0554384486298329, 1.7621278935769478]], 
[[2.2943818903026822, 1.6693380489197964, 8.1007981718172797, 547.06551016331537], [8.8866562959414264, 
2.9538697449771694, 1.6856526337855318, 14.531711428757538], [0.039028757184659611, 1.1494849825630689, 
1.9537983088661117, 365.6111871732702]]], [[[41.487252198220233, 12.409271372600921, 14.477856649277644, 
1.1160774287382553], [0.32944857634125907, 951.85642064916613, 4.7097294212465224, 0.88863208972274543], 
[5.397964054706577, 1.3973452674961897, 1.0251696549037674, 0.051538737074028972]], [[94.200810882618157, 
255.44743759847563, 29.764980791325101, 7.5522628734351223], [32.899896048305123, 46.587640503793793, 
1.109950949907152, 25.253014478094745], [8.2156055132226768, 1.552559066995099, 0.35889915295693109, 
33.447260310417171]]]])
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_add_overloaded_constData_rank0_Symbol_rank0(self):
      arg0=Data(0.499161085524,self.functionspace)
      arg1=Symbol(shape=())
      res=arg0+arg1
      s1=numpy.array(0.377874196418)
      sub=res.substitute({arg1:s1})
      ref=Data(0.877035281942,self.functionspace)
      self.assertTrue(isinstance(res,Symbol),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(sub-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
