/*
 * numbers.c:handles all those strange numeric response dished out by that
 * wacky, nutty program we call ircd 
 *
 *
 * written by michael sandrof
 *
 * copyright(c) 1990 
 *
 * see the copyright file, or do a help ircii copyright 
 */

#if defined(POSIX) || defined(SOLARIS) || defined (HPUX70)
# include "string.h"
#else
# include "strings.h"
#endif /* POSIX */
#include <ctype.h>
#include "irc.h"
#include "input.h"
#include "ircaux.h"
#include "vars.h"
#include "lastlog.h"
#include "hook.h"
#include "server.h"
#include "whois.h"
#include "numbers.h"
#include "window.h"

/*
 * numeric_banner: This returns in a static string of either "xxx" where
 * xxx is the current numeric, or "***" if SHOW_NUMBERS is OFF 
 */
char	*numeric_banner()
{
	static	char	thing[4];

	if (get_int_var(SHOW_NUMERICS_VAR))
		sprintf(thing, "%03.3u", -current_numeric);
	else
		strcpy(thing, "***");
	return (thing);
}


/*
 * display_msg: handles the displaying of messages from the variety of
 * possible formats that the irc server spits out.  you'd think someone would
 * simplify this 
 */
void	display_msg(from, ArgList)
char	*from,
	**ArgList;
{
	char	*ptr;
	char	*rest;

	rest = PasteArgs(ArgList, 0);
	if (from && (strnicmp(get_server_itsname(from_server), from,
			strlen(get_server_itsname(from_server))) == 0))
		from = null(char *);
	if (ptr = index(rest, ':'))
	{
		*(ptr++) = null(char);
		if (strlen(rest))
		{
			if (from)
				put_it("%s %s: %s (from %s)", numeric_banner(),
					rest, ptr, from);
			else
				put_it("%s %s: %s", numeric_banner(), rest,
					ptr);
		}
		else
		{
			if (from)
				put_it("%s %s (from %s)", numeric_banner(),
					ptr, from);
			else
				put_it("%s %s", numeric_banner(), ptr);
		}
	}
	else
	{
		if (from)
			put_it("%s %s (from %s)", numeric_banner(), rest, from);
		else
			put_it("%s %s", numeric_banner(), rest);
	}
}

/*
 * password_sendline: called by send_line() in get_password() to handle
 * hitting of the return key, etc 
 */
static	void	password_sendline(data, line)
char	*data;
char	*line;
{
	int	new_server;

	new_server = atoi(line);
	set_server_password(new_server, line);
	connect_to_server(get_server_name(new_server),
			get_server_port(new_server), -1);
}

/*
 * get_password: when a host responds that the user needs to supply a
 * password, it gets handled here!  the user is prompted for a password and
 * then reconnection is attempted with that password.  but, the reality of
 * the situation is that no one really uses user passwords.  ah well 
 */
static	void	get_password()
{
	char	server_num[8];

	say("password required for connection to server %s",
			get_server_name(from_server));
	sprintf(server_num, "%d", from_server);
	close_server(from_server);
	add_wait_prompt("Server Password:", password_sendline,
			server_num, WAIT_PROMPT_LINE);
}

/*ARGSUSED*/
static	void	nickname_sendline(data, nick)
char	*data;
char	*nick;
{
	int	new_server, server;

	new_server = atoi(data);
	if (nick = check_nickname(nick))
	{

		server = from_server;
		from_server = new_server;
		send_to_server("NICK %s", nick);
		if (new_server == primary_server)
			strmcpy(nickname, nick, NICKNAME_LEN);
		set_server_nickname(new_server, nick);
		from_server = server;
		update_all_status();
	}
	else
	{
		say("illegal nickname, try again");
		add_wait_prompt("Nickname: ", nickname_sendline, data,
				WAIT_PROMPT_LINE);
	}
}

/*
 * reset_nickname: when the server reports that the selected nickname is not
 * a good one, it gets reset here. 
 */
void	reset_nickname()
{
	char	server_num[10];

	say("You have specified an illegal nickname");
	say("Please enter your nickname");
	sprintf(server_num, "%d", from_server);
	add_wait_prompt("Nickname: ", nickname_sendline, server_num,
			WAIT_PROMPT_LINE);
	update_all_status();
	return;
}

/*ARGSUSED*/
static	void	channel_topic(from, ArgList)
char	*from,
	**ArgList;
{
	char	*topic, *channel;

	if (ArgList[1] && is_channel(ArgList[0]))
	{
		topic = ArgList[1];
		channel = ArgList[0];
		message_from(channel, LOG_CRAP);
		put_it("%s Topic for %s: %s", numeric_banner(), channel,
				topic);
	}
	else
	{
		PasteArgs(ArgList, 0);

		message_from(null(char *), LOG_CURRENT);
		put_it("%s Topic: %s", numeric_banner(), ArgList[0]);
	}
}

void	nickname_in_use(from, ArgList)
char	*from, **ArgList;
{
	if (is_server_connected(from_server))
		display_msg(from, ArgList);
	else if (never_connected || (from_server != primary_server) ||
			!attempting_to_connect)
	{
		display_msg(from, ArgList);
		reset_nickname();
	}
	else
	{
		send_to_server("USER %s . . :%s", username, realname);
		send_to_server("NICK %s", get_server_nickname(from_server));
	}
}

static	not_valid_channel(from, ArgList)
char	*from,
	**ArgList;
{
	char	*channel;

	if (!(channel = ArgList[0]) || !ArgList[1])
		return;
	PasteArgs(ArgList, 1);
	if (strnicmp(get_server_name(from_server), from,
			strlen(get_server_name(from_server))) == 0)
	{
		remove_channel(channel, from_server);
		put_it("%s %s %s", numeric_banner(), channel, ArgList[1]);
	}
}

static	void	cannot_join_channel(from, ArgList)
char	*from,	
	**ArgList;
{
	if (ArgList[0])
		remove_channel(ArgList[0], from_server);
	display_msg(from, ArgList);
}


/*ARGSUSED*/
static	void	version(from, ArgList)
char	*from,
	**ArgList;
{
	PasteArgs(ArgList, 1);
	put_it("%s Server %s: %s", numeric_banner(), ArgList[1], ArgList[0]);
}


/*ARGSUSED*/
static	void	invite(from, ArgList)
char	*from,
	**ArgList;
{
	char	*who,
		*channel;

	if ((who = ArgList[0]) && (channel = ArgList[1]))
	{
		message_from(channel, LOG_CRAP);
		if (do_hook(current_numeric, "%s %s %s", from, who, channel))
			put_it("%s Inviting %s to channel %s",
					numeric_banner(), who, channel);
	}
}


/*
 * numbered_command: does (hopefully) the right thing with the numbered
 * responses from the server.  I wasn't real careful to be sure I got them
 * all, but the default case should handle any I missed (sorry) 
 */
void	numbered_command(from, comm, ArgList)
char	*from;
int	comm;
char	**ArgList;
{
	char	*user;
	char	none_of_these = 0;
	char	blah[BIG_BUFFER_SIZE];

	if (!*ArgList[0])
		user = null(char *);
	else
		user = ArgList[0];
	if (!ArgList[1])
		return;
	ArgList++;
	current_numeric = -comm;/* must be negative of numeric! */
	switch (comm)
	{
	case 002:		/* server version, 2.8 style.. - phone */
		sprintf(blah, "%s %s", numeric_banner(), ArgList[0]);
		if (got_initial_version(blah) == -1)
			display_msg(from, ArgList);
		break;
	case 366:		/* end of names list */
		if (!funny_is_ignore_channel())
		{
			if (get_int_var(SHOW_END_OF_MSGS_VAR))
				display_msg(from, ArgList);
			else
			{
				PasteArgs(ArgList, 0);
				do_hook(current_numeric, "%s %s", from,
						ArgList[0]);
			}
		}
		else
			funny_set_ignore_mode();
		break;
	case 341:		/* invite confirmation */
		invite(from, ArgList);
		break;
	case 324:		/* MODE reply */
		funny_mode(from, ArgList);
		break;
	case 301:		/* user is away */
		user_is_away(from, ArgList);
		break;
	case 302:		/* user@host info on a user */
		userhost_returned(from, ArgList);
		break;
	case 303:		/* list of users who are on */
		ison_returned(from, ArgList);
		break;
	case 311:		/* whois name info */
		whois_name(from, ArgList);
		break;
	case 312:		/* whois host/server info */
		whois_server(from, ArgList);
		break;
	case 314:		/* whowas name info */
		whowas_name(from, ArgList);
		break;
	case 313:		/* whois operator info */
		whois_oper(from, ArgList);
		break;
	case 317:		/* whois idle time */
		whois_lastcom(from, ArgList);
		break;
	case 316:		/* whois channel op - redundant */
		whois_chop(from, ArgList);
		break;
	case 318:		/* end of whois */
		end_of_whois(from, ArgList);
		break;
	case 319:		/* whois channel list */
		whois_channels(from, ArgList);
		break;
	case 321:		/* list header line */
		ArgList[0] = "Channel\0Users\0Topic";
		ArgList[1] = ArgList[0] + 8;
		ArgList[2] = ArgList[1] + 6;
		ArgList[3] = null(char *);
		funny_list(from, ArgList);
		break;
	case 322:		/* list entries */
		funny_list(from, ArgList);
		break;
	case 352:	/* new WHO reply, support for /on numeric missing */
		whoreply(null(char *), ArgList);
		break;
	case 353:	/* new NAMES reply, support for /on numeric missing */
		funny_namreply(from, ArgList);
		break;
	case 401:		/* no such nickname */
		no_such_nickname(from, ArgList);
		break;
	case 463:		/* connection not allowed for your host */
		display_msg(from, ArgList);
		close_server(from_server);
		window_check_servers();
		if (from_server == primary_server)
			get_connected(from_server + 1);
		break;
	case 464:		/* invalid password */
		if (oper_command)
			display_msg(from, ArgList);
		else
			get_password();
		break;
	case 465:		/* You're banned from the server */
		display_msg(from, ArgList);
		close_server(from_server);
		window_check_servers();
		break;
	case 421:
		if (!strcmp("ISON", ArgList[0]) || !strcmp("USERHOST",
				ArgList[0]))
		{
			set_server_2_6_2(from_server, 0);
			convert_to_whois(from_server);
		}
		else
			display_msg(from, ArgList);
		break;
	case 381: 		/* oper response ok */
		set_server_operator(from_server, 1);
		set_server_flag(from_server, USER_MODE_S, 1);
		set_server_flag(from_server, USER_MODE_W, 1);
		update_all_status();	/* fix the status line */
	default:
		{
			/*
			 * The following accumulates the remaining arguments
			 * in ArgSpace for hook detection. We can't use
			 * PasteArgs here because we still need the arguments
			 * separated for use elsewhere.
			 */
			char	*ArgSpace = null(char *);
			int	i, len;

			for (i = len = 0; ArgList[i]; len +=
					strlen(ArgList[i++]))
				;
			len += (i - 1);
			ArgSpace = new_malloc(len + 1);
			ArgSpace[0] = '\0';
			for (i = 0; ArgList[i]; i++)
			{
				if (i)
					strcat(ArgSpace, " ");
				strcat(ArgSpace, ArgList[i]);
			}
			if (!do_hook(current_numeric, "%s %s", from, ArgSpace))
			{
				new_free(&ArgSpace);
				return;
			}
			new_free(&ArgSpace);
			none_of_these = 1;
		}
	}
	/* the following do not hurt the ircII if intercepted by a hook */
	if (none_of_these)
	{
		switch (comm)
		{
		case 221:
			put_it("%s Your user mode is \"%s\"", numeric_banner(),
				ArgList[0]);
			break;
		case 332:		/* topic */
			channel_topic(from, ArgList);
			break;
		case 351:		/* version response */
			version(from, ArgList);
			break;
		case 364: /* Should be new links reply */
			put_it("%s %-20s %s", numeric_banner(), ArgList[0],
				ArgList[1]);
			break;
		case 372:		/* new MOTD reply */
			if (!get_int_var(SUPPRESS_SERVER_MOTD_VAR))
			{
				PasteArgs(ArgList, 0);
				put_it("%s", ArgList[0]);
			}
			break;
		case 374:		/* new MOTD start */
			if (!get_int_var(SUPPRESS_SERVER_MOTD_VAR) &&
					get_server_motd(from_server))
			{
				set_server_motd(from_server, 0);
				PasteArgs(ArgList, 0);
				put_it("%s %s", numeric_banner(), ArgList[0]);
			}
			break;
		case 375:		/* new MOTD stuff */
			if (!get_int_var(SUPPRESS_SERVER_MOTD_VAR) &&
					get_server_motd(from_server))
			{
				set_server_motd(from_server, 1);
				PasteArgs(ArgList, 0);
				put_it("%s %s", numeric_banner(), ArgList[0]);
			}
			break;
		case 451:		/* not registered as a user */
	/*
	 * Sometimes the server doesn't catch the USER line, so
	 * here we send a simplified version again  -lynx 
	 */
			send_to_server("USER %s . . :%s", username, realname);
			send_to_server("NICK %s",
				get_server_nickname(from_server));
			break;
		case 462:		/* Identity problems */
			send_to_server("NICK %s",
				get_server_nickname(from_server));
			break;
		case 433:		/* nickname in use */
			nickname_in_use(from, ArgList);
			break;
		case 432:		/* bad nickname */
			display_msg(from, ArgList);
			reset_nickname();
			break;
		case 473:  /* cannot join channel */
		case 471:  /* channel is full */
		case 474:  /* banned from channel */
			cannot_join_channel(from, ArgList);
			break;
		case 403:		/* not a valid channel: for 2.5/2.6
						     * compatability */
			not_valid_channel(from, ArgList);
			break;
		case 384:		/* personal server port number */
			PasteArgs(ArgList, 0);
			put_it("%s %s %s", numeric_banner(), ArgList[0], user);
			break;
		case 385:		/* not operator any more */
			set_server_operator(from_server, 0);
			display_msg(from, ArgList);
			update_all_status();
			break;
		case 323:               /* end of list */
			funny_print_widelist();
			if (get_int_var(SHOW_END_OF_MSGS_VAR))
				display_msg(from, ArgList);
			break;
		case 315:		/* end of who list */
			if (cannot_open != null(char *))
				yell("Cannot open: %s", cannot_open);
		case 365:		/* end of links list */
		case 219:		/* new: end of stats list */
		case 368:		/* new: end of ban list */
		case 369:		/* new: end of who was list */
		case 376:		/* new: end of motd */
			if (get_int_var(SUPPRESS_SERVER_MOTD_VAR))
				break;
			set_server_motd(from_server, 0);
			if (!get_int_var(SHOW_END_OF_MSGS_VAR))
				break;

		default:
			display_msg(from, ArgList);
		}
	}
}


