/*
 * tenex -		file name or command expansion for "reader"
 *
 * Dave Clemans, 4/84 (initial version); 1/89 (more generality)
 *
 * Modeled from, (or taken from), the equivalent routines from
 * Ken Greer's "Tenex" C-shell.
 *
 * Main entry point:
 *	ExpandAName	given a line, expands last word in it
 *
 * $Id: tenex.c,v 1.3 89/02/20 20:20:26 dclemans Exp $
 *
 * $Log:	tenex.c,v $
 * Revision 1.3  89/02/20  20:20:26  dclemans
 * Add RCS identifiers
 * 
 */
#ifndef GEMDOS
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <pwd.h>
#include <signal.h>
#else
#include <types.h>
#include <stat.h>
#include "ndir.h"
#include <strings.h>
#include <ctype.h>
#define index  strchr
#define rindex strrchr
#ifndef NULL
#define	NULL	((char *)0)
#endif  NULL
#endif  GEMDOS
#ifndef LOC
#define LOC(r)  __FILE__,r,__LINE__
#endif  LOC

typedef enum {LIST, RECOGNIZE} COMMAND;

#define	TRUE	1
#define	FALSE	0
#define	FILSIZ	512		/* Max reasonable file name length */

#ifndef DIR_SEPARATOR
#ifndef GEMDOS
#define DIR_SEPARATOR '/'
#else
#define DIR_SEPARATOR '\\'
#endif  GEMDOS
#endif  DIR_SEPARATOR

/*
 * Interesting function definitions
 */
extern char *index();
extern char *var_normal();
extern int _writechar();
static char filetype();

/*
 * max of a,b
 */
static
max (a, b)
{
    if (a > b)
	return (a);
    return (b);
};	/* end of max */

/*
 * For qsort()
 */
static
fcompare (file1, file2)
char  **file1, **file2;
{
    return (strcmp (*file1, *file2));
};	/* end of fcompare */

/*
 * Append characters to end of global place
 */
static char *CharPtr;
static
CharAppend (c)
char c;
{
    *CharPtr++ = c;
    *CharPtr   = 0;
};	/* end of CharAppend */

/*
 * Set up to search for names
 */
ExpandAName (inputline, inputline_size, num_read, doRecognize)
char   *inputline;		/* match string prefix */
int     inputline_size;		/* max size of string */
int	num_read;		/* # actually in inputline */
{
    static char 
	    delims[] = " '\"\t;&<>()|^%";
    char word [FILSIZ + 1];
    register char *str_end, *word_start, *cmd_start, *wp;
    int space_left;
    int is_a_cmd;		/* UNIX command rather than filename */
    int rc;

    CharPtr = str_end = &inputline[num_read];

   /*
    * Find LAST occurence of a delimiter in the inputline.
    * The word start is one character past it.
    */
    for (word_start = str_end; word_start > inputline; --word_start)
	if (index (delims, word_start[-1]))
	    break;

    space_left = inputline_size - (word_start - inputline) - 1;

    is_a_cmd = starting_a_command (word_start, inputline);

    for (cmd_start = word_start, wp = word; cmd_start < str_end;
    	 *wp++ = *cmd_start++);
    *wp = 0;

    if (doRecognize)
	return search (word, wp, space_left, is_a_cmd, RECOGNIZE, CharAppend);
    else
    {	/* do LIST command */
#ifdef  GEMDOS
        _writechar('\r');
#endif  GEMDOS
	_writechar('\n');
	rc = search (word, wp, space_left, is_a_cmd, LIST, _writechar);
	_doStandardRetype('\0');
	return rc;
    }
};	/* end of ExpandAName */

/*
 * Are we starting a command (i.e., still in the first word on the line?
 */
static starting_a_command (wordstart, inputline)
register char *wordstart, *inputline;
{
    static char
	    cmdstart[] = ";&(|`",
	    cmdalive[] = " \t'\"";
    while (--wordstart >= inputline)
    {
	if (index (cmdstart, *wordstart))
	    break;
	if (!index (cmdalive, *wordstart))
	    return (FALSE);
    }
    if (wordstart > inputline && *wordstart == '&')	/* Look for >& */
    {
	while (wordstart > inputline &&
			(*--wordstart == ' ' || *wordstart == '\t'));
	if (*wordstart == '>')
		return (FALSE);
    }
    return (TRUE);
};	/* end of starting_a_command */

/*
 * expand "old" file name with possible tilde usage
 *		~person/mumble
 * expands to
 *		home_directory_of_person/mumble
 * into string "new".
 */

static char *
tilde (new, old)
char *new, *old;
{
#ifndef GEMDOS
    extern struct passwd *getpwuid (), *getpwnam ();

    register char *o, *p;
    register struct passwd *pw;
    static char person[40] = {0};

    if (old[0] != '~')
    {
	strcpy (new, old);
	return (new);
    }

    for (p = person, o = &old[1]; *o && *o != DIR_SEPARATOR; *p++ = *o++);
    *p = '\0';

    if (person[0] == '\0')			/* then use current uid */
	pw = getpwuid (getuid ());
    else
	pw = getpwnam (person);

    if (pw == NULL)
	return (NULL);

    strcpy (new, pw -> pw_dir);
    (void) strcat (new, o);
    return (new);
#else
    strcpy (new, old);
    return (new);
#endif  GEMDOS
};	/* end of tilde */

/*
 * parse full path in file into 2 parts: directory and file names
 * Should leave final slash (/) at end of dir.
 */
static
extract_dir_and_name (path, dir, name)
char   *path, *dir, *name;
{
    extern char *rindex ();
    register char  *p;

    p = rindex (path, DIR_SEPARATOR);
    if (p == NULL)
    {
	copyn (name, path, MAXNAMLEN);
	dir[0] = '\0';
    }
    else
    {
	p++;
	copyn (name, p, MAXNAMLEN);
	copyn (dir, path, p - path);
    }
};	/* end of extract_dir_and_name */


/*
 * Get a directory entry
 */
static char *
getentry (dir_fd, looking_for_lognames)
DIR *dir_fd;
{
#ifndef GEMDOS
    if (looking_for_lognames)			/* Is it login names we want? */
    {
	extern struct passwd *getpwent ();
	register struct passwd *pw;
	if ((pw = getpwent ()) == NULL)
	    return (NULL);
	return (pw -> pw_name);
    }
    else					/* It's a dir entry we want */
    {
#endif  GEMDOS
	register struct direct *dirp;
        register char *p;

	if (dirp = readdir (dir_fd))
	{
#ifdef  GEMDOS
            for (p = dirp->d_name; *p; p++)
                if (isupper(*p))
                    *p = _tolower(*p);
#endif  GEMDOS
	    return (dirp -> d_name);
        }
	return (NULL);
#ifndef GEMDOS
    }
#endif  GEMDOS
};	/* end of getentry */

/*
 * Free a list of strings
 */
static
free_items (items)
register char **items;
{
    register int i;
    for (i = 0; items[i]; i++)
	free (items[i]);
    free (items);
};	/* end of free_items */

#define FREE_ITEMS(items)\
{\
    free_items (items);\
    items = NULL;\
}
#define FREE_DIR(fd)\
{\
    closedir (fd);\
    fd = NULL;\
}

/*
 * Strip next directory from path; return ptr to next unstripped directory.
 */
 
static char *extract_dir_from_path (path, dir)
char *path, dir[];
{
    register char *d = dir;
    char sep;

#ifndef GEMDOS
    sep = ':';
#else
    sep = ',';
#endif  GEMDOS
    while (*path && (*path == ' ' || *path == sep)) path++;
    while (*path && (*path != ' ' && *path != sep)) *(d++) = *(path++);
    while (*path && (*path == ' ' || *path == sep)) path++;

    *(d++) = DIR_SEPARATOR;
    *d = 0;

    return path;
};	/* end of extract_dir_from_path */

/*
 * Perform a RECOGNIZE or LIST command on string "word".
 */
static
search (word, wp, max_word_length, looking_for_command, command, routine)
char   *word,
       *wp;			/* original end-of-word */
int max_word_length,
	looking_for_command;
COMMAND command;
int (*routine) ();
{
#define	MAXITEMS	2048
    register numitems,
	    name_length,		/* Length of prefix (file name) */
	    looking_for_lognames;	/* True if looking for login names */
    struct stat
	    dot_statb,			/* Stat buffer for "." */
	    curdir_statb;		/* Stat buffer for current directory */
    int	    dot_scan,			/* True if scanning "." */
	    dot_got;			/* True if have scanned dot already */
    char    tilded_dir[FILSIZ + 1],	/* dir after ~ expansion */
	    dir[FILSIZ + 1],		/* /x/y/z/ part in /x/y/z/f */
            name[MAXNAMLEN + 1],	/* f part in /d/d/d/f */
            extended_name[MAXNAMLEN+1],	/* the recognized (extended) name */
            *entry,			/* single directory entry or logname */
	    *path,			/* hacked PATH environment variable */
            *savepath;
    static DIR 
	    *dir_fd = NULL;
    static char
            **items = NULL;		/* file names when doing a LIST */

    savepath = (char *)NULL;
    tilded_dir[0] = '\0';
    if (items != NULL)
	FREE_ITEMS (items);
    if (dir_fd != NULL)
	FREE_DIR (dir_fd);

#ifndef GEMDOS
    looking_for_lognames = (*word == '~') && (index (word, DIR_SEPARATOR) == NULL);
#else
    looking_for_lognames = 0;
#endif  GEMDOS
    looking_for_command &= (*word != '~') && (index (word, DIR_SEPARATOR) == NULL);

    if (looking_for_command)
    {
        copyn (name, word, MAXNAMLEN);
        if ((savepath = path = var_normal ("$PATH")) == NULL)
	    path = ".";
    	path = extract_dir_from_path (path, dir);
    }
    numitems = 0;

    dot_got = FALSE;
    stat (".", &dot_statb);

cmdloop:	/* One loop per directory in PATH, if looking_for_command */

    if (looking_for_lognames)			/* Looking for login names? */
    {
#ifndef GEMDOS
	setpwent ();				/* Open passwd file */
#endif  GEMDOS
	copyn (name, &word[1], MAXNAMLEN);	/* name sans ~ */
    }
    else
    {						/* Open directory */
        if (!looking_for_command)
	    extract_dir_and_name (word, dir, name);
	if ((tilde (tilded_dir, dir) == 0) ||	/* expand ~user/... stuff */
	    
	   ((dir_fd = opendir (*tilded_dir ? tilded_dir : ".")) == NULL))
	{
	    if (looking_for_command)
	        goto try_next_path;
	    else
            {
                if (savepath != (char *)NULL)
                    free(savepath);
		return (0);
            }
	}
	dot_scan = FALSE;
#ifndef GEMDOS
	if (looking_for_command)
	{
	    /*
	     * Are we searching "."?
	     */
	    fstat (dir_fd->dd_fd, &curdir_statb);
	    if (curdir_statb.st_dev == dot_statb.st_dev &&
	        curdir_statb.st_ino == dot_statb.st_ino)
	    {
	        if (dot_got)			/* Second time in PATH? */
			goto try_next_path;
		dot_scan = TRUE;
		dot_got = TRUE;
	    }
        }
#endif  GEMDOS
    }

    name_length = strlen (name);

    while (entry = getentry (dir_fd, looking_for_lognames))
    {
	if (!is_prefix (name, entry))
	    continue;

	/*
	 * Don't match . files on null prefix match
	 */
	if (name_length == 0 && entry[0] == '.' && !looking_for_lognames)
	    continue;

	/*
	 * Skip non-executables if looking for commands:
	 * Only done for directory "." for speed.
	 * (Benchmarked with and without:
	 * With filetype check, a full search took 10 seconds.
	 * Without filetype check, a full search took 1 second.)
	 *                                   -Ken Greer
         */
	if (looking_for_command && dot_scan && filetype (dir, entry) != '*')
	    continue;

	if (command == LIST)		/* LIST command */
	{
	    extern char *malloc ();
	    register int length;
	    if (numitems >= MAXITEMS)
	    {
#ifndef GEMDOS
                errmsg(0,LOC("search"),"\nYikes!! Too many %s !!",
                    looking_for_lognames?"names in password file":"files");
#else
                errmsg(0,LOC("search"),"\r\nYikes!! Too many files !!");
#endif  GEMDOS
		break;
	    }
	    if (items == NULL)
	    {
		items = (char **) malloc (sizeof (items[1]) * (MAXITEMS + 1));
		if (items == NULL)
		    break;
		for (length = 1; length < MAXITEMS + 1; length++)
		    items[length] = (char *)NULL;
	    }
	    length = strlen(entry) + 1;
	    if ((items[numitems] = malloc (length)) == NULL)
	    {
                errmsg(0,LOC("search"),"out of memory");
		break;
	    }
	    copyn (items[numitems], entry, MAXNAMLEN);
	    numitems++;
	}
	else					/* RECOGNIZE command */
	    if (recognize (extended_name, entry, name_length, ++numitems))
		break;
    }

    if (looking_for_lognames)
#ifndef GEMDOS
	endpwent ();
#else
        ;
#endif  GEMDOS
    else
	FREE_DIR (dir_fd);

try_next_path:
    if (looking_for_command && *path &&
    	(path = extract_dir_from_path (path, dir), dir)) 
    	goto cmdloop;
    
    if (command == RECOGNIZE && numitems > 0)
    {
	if (looking_for_lognames)
	    copyn (word, "~", 1);
	else if (looking_for_command)
	    word[0] = 0;
	else
	    copyn (word, dir, max_word_length);		/* put back dir part */
	catn (word, extended_name, max_word_length);	/* add extended name */
	while (*wp) (*routine) (*wp++);
        if (savepath != (char *)NULL)
            free(savepath);
	return (numitems);
    }

    if (command == LIST)
    {
	qsort (items, numitems, sizeof (items[1]), fcompare);
	print_by_column (looking_for_lognames ? NULL:tilded_dir, items,
			 numitems, looking_for_command);
	if (items != NULL)
	    FREE_ITEMS (items);
    }
    if (savepath != (char *)NULL)
        free(savepath);
    return (0);
};	/* end of search */

/*
 * Object: extend what user typed up to an ambiguity.
 * Algorithm:
 * On first match, copy full entry (assume it'll be the only match) 
 * On subsequent matches, shorten extended_name to the first
 * character mismatch between extended_name and entry.
 * If we shorten it back to the prefix length, stop searching.
 */
static recognize (extended_name, entry, name_length, numitems)
char *extended_name, *entry;
{
    if (numitems == 1)				/* 1st match */
	copyn (extended_name, entry, MAXNAMLEN);
    else					/* 2nd and subsequent matches */
    {
	register char *x, *ent;
	register int len = 0;
	for (x = extended_name, ent = entry; *x && *x == *ent++; x++, len++);
	*x = '\0';				/* Shorten at 1st char diff */
	if (len == name_length)			/* Ambiguous to prefix? */
	    return (-1);			/* So stop now and save time */
    }
    return (0);
};	/* end of recognize */

/*
 * return true if check items initial chars in template
 * This differs from PWB imatch in that if check is null
 * it items anything
 */
static
is_prefix (check, template)
char   *check,
       *template;
{
    register char  *check_char,
                   *template_char;

    check_char = check;
    template_char = template;
    do
	if (*check_char == 0)
	    return (TRUE);
    while (*check_char++ == *template_char++);
    return (FALSE);
};	/* end of is_prefix */

/*
 * like strncpy but always leave room for trailing \0
 * and always null terminate.
 */
static copyn (des, src, count)
register char *des, *src;
register count;
{
    while (--count >= 0)
	if ((*des++ = *src++) == 0)
	    return;
    *des = '\0';
};	/* end of copyn */

/*
 * Concatonate src onto tail of des.
 * Des is a string whose maximum length is count.
 * Always null terminate.
 */
static catn (des, src, count)
register char *des, *src;
register count;
{
    while (--count >= 0 && *des)
	des++;
    while (--count >= 0)
	if ((*des++ = *src++) == 0)
	    return;
    *des = '\0';
};	/* end of catn */

/*
 * Return a character denoting the type of a file:
 *	* if executable
 *	/ if directory
 *	space otherwise
 */
static char
filetype (dir, file)
char *dir, *file;
{
    if (dir)
    {
	char path[512];
	struct stat statb;
	strcpy (path, dir);
	catn (path, file, sizeof path);
#ifndef GEMDOS
	if (lstat (path, &statb) >= 0)
	{
	    if ((statb.st_mode & S_IFMT) == S_IFDIR)
		return (DIR_SEPARATOR);
	    if ((statb.st_mode & S_IFMT) == S_IFLNK)
		return ('@');
	    if (statb.st_mode & (S_IEXEC |
	       (S_IEXEC>>3) | (S_IEXEC>>6)))
		return ('*');
	}
#else
        if (stat (path, &statb) >= 0)
        {
            if (statb.st_mode & S_IJDIR)
                return (DIR_SEPARATOR);
        }
#endif  GEMDOS
    }
    return (' ');
};	/* end of filetype */

/*
 * Print sorted down columns
 */
static
print_by_column (dir, items, count, looking_for_command)
register char *dir, *items[];
{
    register int i, rows, r, c, maxwidth = 0, columns;
    for (i = 0; i < count; i++)
	maxwidth = max (maxwidth, strlen (items[i]));
    maxwidth += looking_for_command ? 1:2;	/* for the file tag and space */
    columns = 80 / maxwidth;
    rows = (count + (columns - 1)) / columns;
    for (r = 0; r < rows; r++)
    {
	for (c = 0; c < columns; c++)
	{
	    i = c * rows + r;
	    if (i < count)
	    {
		register int w;
		prstring (items[i]);
		w = strlen (items[i]);
		/* Print filename followed by '/' or '*' or ' ' */
		if (!looking_for_command)
			_writechar (filetype (dir, items[i])), w++;
		if (c < (columns - 1))			/* Not last column? */
		    for (; w < maxwidth; w++)
			_writechar (' ');
	    }
	}
#ifdef  GEMDOS
        _writechar ('\r');
#endif
	_writechar ('\n');
    }
};	/* end of print_by_column */

static prstring(s)
register char *s;
{
	while (*s)
	{	/* dump the string */
		_writechar(*s);
		s++;
	}
};	/* end of prstring */
