/*
 * reader	-	read line with editing
 *
 * Dave Clemans, 4/84 (first version); 1/89 more generality
 *
 * This module in a sense is a superset of 'fgets' in that it reads a
 * line at a time from a file stream.  However, it is also a lot more.
 *
 * Depending on the value of the "EDITOR" environment variable, reader
 * provides 'vi' or 'emacs' style input editing.
 *
 * Reader keeps a history of the lines that it has seen.  The history
 * list can be referenced and its size changed through global variables.
 * There is an alternate point to reader (editor) that also asks for
 * a line from the user; except that the buffer starts out with the
 * contents of some line from the history list instead of being empty.
 *
 * External entry points:
 *	Reader		- the 'reader' get a line routine
 *	Editor		- the 'editor' edit a line routine
 *
 * $Id: reader.c,v 1.7 89/03/07 19:37:36 dclemans Exp $
 *
 * $Log:	reader.c,v $
 * Revision 1.7  89/03/07  19:37:36  dclemans
 * make work on bsd systems
 * 
 * Revision 1.6  89/02/20  20:32:33  dclemans
 * More RCS ID work
 * 
 * Revision 1.5  89/02/20  20:20:23  dclemans
 * Add RCS identifiers
 * 
 */
char reader_version[] = "Reader Version 0.2 (dgc); $Date: 89/03/07 19:37:36 $";

#include <ctype.h>
#include <strings.h>
#ifdef	GEMDOS
#include <osbind.h>
#define	index	strchr
#define	rindex	strrchr
#else
#include <sys/types.h>
#include <sgtty.h>
#include <signal.h>
#endif	GEMDOS
#include <errno.h>
#ifndef LOC
#define LOC(r)  __FILE__,r,__LINE__
#endif  LOC

#include "shell.h"
#define DEFAULT_MAXSIZE DEFAULT_HISTORY
#include "history.h"
struct History History;
struct savedState _savedState;
struct newState _newState;

#ifndef DIR_SEPARATOR
#ifndef GEMDOS
#define DIR_SEPARATOR '/'
#else
#define DIR_SEPARATOR '\\'
#endif  GEMDOS
#endif  DIR_SEPARATOR

/*
 * Interesting function declarations
 */
extern char *var_normal();
extern int wild_match();

/*
 * For saving pending input during terminal mode changes
 */
static struct	pendingInput
{
	char buffer[BUFSIZ];
	int size;
	int next;
} pendingInput;

#ifndef GEMDOS
/*
 * For restoring things and exiting after catching a signal
 */
static catchSignal(sig)
int sig;
{
	_savechar(EOF);
	signal(sig,catchSignal);
};	/* end of catchSignal */

/*
 * Handle a suspend signal
 */
static catchSuspend()
{
	int stream;

	stream = _savedState.stream;
	if (stream >= 0)
		_terminate(stream);
	signal(SIGTSTP,SIG_DFL);
	killpg(getpgrp(getpid()),SIGTSTP);
	if (stream >= 0)
		initialize(stream);
};	/* end of catchSuspend */
#endif  GEMDOS

/*
 * Initialize the users environment and terminal attributes to match
 * our needs.
 *
 * Before we change anything, state is saved.
 *
 * We do control character munging so that we can see characters that
 * CBREAK mode would otherwise hide from us.
 */
static initialize(fp)
register int fp;
{
#ifndef GEMDOS
	register int rc;

	if (!isatty(fp))
		return;
#endif  GEMDOS
	_savedState.stream = fp;

#ifndef GEMDOS
	/*
	 * Get the initial state
	 */
	ioctl(fp, TIOCGETP, &_savedState.basicAttributes);
	ioctl(fp, TIOCGETC, &_savedState.basicCharacters);
	ioctl(fp, TIOCLGET, &_savedState.localAttributes);
	ioctl(fp, TIOCGLTC, &_savedState.localCharacters);
	_savedState.sigint = (int)signal(SIGINT,SIG_IGN);
	_savedState.sigquit = (int)signal(SIGQUIT,SIG_IGN);
	_savedState.sigtstp = (int)signal(SIGTSTP,SIG_IGN);
	signal(SIGINT,_savedState.sigint);
	signal(SIGQUIT,_savedState.sigquit);
	signal(SIGTSTP,_savedState.sigtstp);
	_newState.basicAttributes = _savedState.basicAttributes;
	_newState.basicCharacters = _savedState.basicCharacters;
	_newState.localAttributes = _savedState.localAttributes;
	_newState.localCharacters = _savedState.localCharacters;

	/*
	 * We want to turn off all control characters
	 */
	_newState.basicAttributes.sg_erase = '\377';
	_newState.basicAttributes.sg_kill = '\377';
	if (_savedState.sigint == (int)SIG_IGN)
		_newState.basicCharacters.t_intrc = '\377';
	signal(SIGINT,catchSignal);
	if (_savedState.sigquit == (int)SIG_IGN)
		_newState.basicCharacters.t_quitc = '\377';
	signal(SIGQUIT,catchSignal);
	_newState.basicCharacters.t_startc = '\377';
	_newState.basicCharacters.t_stopc = '\377';
	_newState.basicCharacters.t_eofc = '\377';
	_newState.basicCharacters.t_brkc = '\377';
	if (_savedState.sigtstp == (int)SIG_IGN)
	{	/* then can clobber suspension chars */
		_newState.localCharacters.t_suspc = '\377';
		_newState.localCharacters.t_dsuspc = '\377';
	}
	signal(SIGTSTP,catchSuspend);
	_newState.localCharacters.t_rprntc = '\377';
	_newState.localCharacters.t_flushc = '\377';
	_newState.localCharacters.t_werasc = '\377';
	_newState.localCharacters.t_lnextc = '\377';

	/*
	 * We want to use cbreak mode, no echo, etc.
	 */
	_newState.basicAttributes.sg_flags &= ~(RAW|ECHO);
	_newState.basicAttributes.sg_flags |= CBREAK;

	/*
	 * Do all the mode changes
	 */
	ioctl(fp, TIOCSETN, &_newState.basicAttributes);
	ioctl(fp, TIOCSETC, &_newState.basicCharacters);
	ioctl(fp, TIOCLSET, &_newState.localAttributes);
	ioctl(fp, TIOCSLTC, &_newState.localCharacters);
#endif  GEMDOS
};	/* end of initialize */

/*
 * Reset the users environment and terminal attributes to where they
 * were when we were entered.  Typeahead (which would otherwise be
 * lost) is handled by re-inputting it after everything is in the
 * right state.
 */
#ifndef GEMDOS
_terminate(fp)
register int fp;
{
	register int rc;
	long available;
	char c;

	if (!isatty(fp))
		return;

	signal(SIGINT,_savedState.sigint);
	signal(SIGQUIT,_savedState.sigquit);
	signal(SIGTSTP,_savedState.sigtstp);

	ioctl(fp, TIOCSETN, &_savedState.basicAttributes);
	ioctl(fp, TIOCSETC, &_savedState.basicCharacters);
	ioctl(fp, TIOCLSET, &_savedState.localAttributes);
	ioctl(fp, TIOCSLTC, &_savedState.localCharacters);
};	/* end of _terminate */
#else
_terminate()
{
};	/* end of _terminate */
#endif  GEMDOS

/*
 * Get a line to use for an input buffer from the history list
 *
 * Returns it by setting the global History.currentLine
 */
static getHistoryBuffer()
{
	register int counter;
	register struct historyLine *hp,*bottom;

	if (!History.listTop)
		History.maxSize = DEFAULT_MAXSIZE;
	if (History.maxSize <= 0)
		History.maxSize = 1;
	if (History.currentSize > History.maxSize)
	{	/* user has shrunk the stack; so free up the excess entries */
		bottom = (struct historyLine *)NULL;
		for (counter = 0,hp = History.listTop;
		    hp; hp = hp->next,counter++)
		{	/* go through what we will keep */
			if (counter == (History.maxSize-1))
			{	/* set new end of list */
				bottom = hp;
                                hp = hp->next;
                                break;
			}
		}
		if (bottom)
		{	/* set new end of list */
			History.listBottom = bottom;
			bottom->next = (struct historyLine *)NULL;
		}
		for (; hp; )
		{	/* free up excess entries */
			bottom = hp;
			hp = hp->next;
			free(bottom);
		}
	}

	if (History.currentLine && (!History.currentLine->size ||
	    ((History.currentLine->size == 1) &&
	    ((History.currentLine->contents[0] == '\n') ||
	    (History.currentLine->contents[0] == '\r')))))
	{	/* empty entry at top; so use it */
		/* History.currentLine->command = ++History.currentCommand; leave same */
		History.listPointer = History.currentLine;
		History.currentLine->cursor = 0;
		History.currentLine->size = 0;
		History.currentLine->ttycursor = 0;
		History.currentLine->contents[0] = '\0';
		return;
	}

	if (History.currentSize < History.maxSize)
	{	/* more room available; make new entry */
		while (History.currentSize < History.maxSize)
		{	/* make all new entries at once */
			History.currentSize++;
			hp = (struct historyLine *)malloc(sizeof(*hp));
			if (!hp)
			{	/* out of core */
				errmsg(-1,LOC("getHistoryBuffer"));
				exit(-1);
			}
			hp->contents[0] = '\0';
			hp->size = 0;
			hp->cursor = 0;
			hp->ttycursor = 0;
			hp->command = 0;
			hp->next = (struct historyLine *)NULL;
			if (!History.listTop)
				History.listTop = History.listBottom = hp;
			else
			{	/* push onto top */
				hp->next = History.listTop;
				History.listTop = hp;
			}
		}
		History.currentLine = hp;
		History.listPointer = History.currentLine;
		hp->command = ++History.currentCommand;
		return;
	}

	/* Take entry from bottom and put on top */
	hp = History.listBottom;
	for (bottom = History.listTop; bottom && (bottom->next != hp); bottom = bottom->next)
		continue;
	if (!bottom)
	{	/* list screwed up */
		errmsg(0,LOC("getHistoryBuffer"),"screwed up history list");
		exit(-1);
	}
	History.listBottom = bottom;
	bottom->next = (struct historyLine *)NULL;
	hp->next = History.listTop;
	History.listTop = hp;
	History.currentLine = hp;
	History.listPointer = hp;
	hp->size = 0;
	hp->cursor = 0;
	hp->ttycursor = 0;
	hp->contents[0] = '\0';
	hp->command = ++History.currentCommand;
};	/* end of getHistoryBuffer */

/*
 * Get the next user input character; if pending input take from that
 * else get it from the system.
 */
int _readchar(fp)
register int fp;
{
	extern errno;
	char c;
    int cc;
	register int rc;

	if (pendingInput.size > 0)
	{	/* take from already read in chars */
		if (pendingInput.next != pendingInput.size)
		{	/* if something was already there */
			c = pendingInput.buffer[pendingInput.next++];
			c &= 0x7f;
			if (pendingInput.next == pendingInput.size)
				pendingInput.next = pendingInput.size = 0;
			return(c);
		}
	}
	errno = 0;
	cc = EOF;
#ifndef GEMDOS
	rc = read(fp,&c,1);
	if ((rc <= 0) && (errno == EINTR))
		return(_readchar(fp));
    cc = c;
#else
        if (fp >= 0)
        {
            rc = 1;
            cc = Bconin(2);
        }
        else rc = 0;
#endif  GEMDOS
	cc &= 0x7f;
	if (rc <= 0)
		cc = EOF;
	return(cc);
};	/* end of _readchar */

/*
 * Simple output routine
 */
_writechar(c)
char c;
{
#ifndef GEMDOS
	write(TTY,&c,1);
#else
        Bconout(2,c);
#endif  GEMDOS
};	/* end of _writechar */

/*
 * Save a character in the pending input buffer to be read later
 */
_savechar(c)
char c;
{
	if (pendingInput.size >= sizeof(pendingInput.buffer)-1)
		return;		/* no space to save it */
	pendingInput.buffer[pendingInput.size++] = c;
	pendingInput.buffer[pendingInput.size] = '\0';
};	/* end of _savechar */

/*
 * The basic "get a line" with editing routine.
 * We return the number of characters read, or
 *		0 on end-of-file
 *	       <0 on I/O error
 *
 * If history structure not allocated yet, allocate it.
 */
static int localReader(hp,fp,buffer,size)
struct historyLine *hp;
register int fp;
char *buffer;
int size;
{
	char nextCharacter;
	register char *cp;
	register int rc;
	extern errno;

	/*
	 * Get an input buffer
	 */
	getHistoryBuffer();
	if (hp)
	{	/* start with a non-empty line */
		strncpy(History.currentLine->contents,hp->contents,hp->size);
		History.currentLine->size = hp->size;
		History.currentLine->cursor = hp->size;
		_doStandardRetype('\0');
	}

	/*
	 * Non-terminal mode dependant initialization
	 */
        if (_savedState.localEditor == (char *)NULL)
	{
		_savedState.localEditor = var_normal("$VISUAL");
		if (_savedState.localEditor == (char *)NULL)
			_savedState.localEditor = var_normal("$EDITOR");
		if (_savedState.localEditor == (char *)NULL)
			_savedState.localEditor = var_normal("$FCEDIT");
	}
	if (_savedState.localEditor != (char *)NULL)
	{
		cp = rindex(_savedState.localEditor,DIR_SEPARATOR);
		if (cp != (char *)NULL)
			cp++;
		else	cp = _savedState.localEditor;
		if (_savedState.isEmacs >= 0 && _savedState.isVi >= 0)
		{
			if (wild_match(cp,"*[eE][mM][aA][cC][sS]*"))
			{
				_savedState.isEmacs = 1;
				_savedState.isVi = 0;
			}
			else
			{	/* else check for other editors */
				if (wild_match(cp,"*[vV][iI]*"))
				{
					_savedState.isEmacs = 0;
					_savedState.isVi = 1;
				}
			}
		}
	}

	/*
	 * Set up terminal modes
	 */
	initialize(fp);

	/*
	 * Start reading the line
	 */
	for (;;)
	{	/* loop til eol */
		nextCharacter = _readchar(fp);
		if (nextCharacter == EOF)
		{	/* I/O error */
			History.currentLine->size = 0;
			History.currentLine->cursor = 0;
			size = -1;
			break;
		}
#ifndef GEMDOS
		if ((nextCharacter == _savedState.basicCharacters.t_eofc) &&
		    (History.currentLine->size == 0))
		{	/* end-of-file */
			break;
		}
		if ((nextCharacter == '\\') ||
		    (nextCharacter == _savedState.localCharacters.t_lnextc))
		{	/* quote the next character */
			if (nextCharacter == '\\')
			{	/* save this character */
				_doStandardCharacter(nextCharacter);
				if (History.currentLine->size >= sizeof(History.currentLine->contents)-1)
					break;
			}
			nextCharacter = _readchar(fp);
			if (nextCharacter == EOF)
			{	/* end of file */
				History.currentLine->cursor = 0;
				History.currentLine->size = 0;
				size = -1;
				break;
			}
			_doStandardCharacter(nextCharacter);
			if (History.currentLine->size >= sizeof(History.currentLine->contents)-1)
				break;
			continue;
		}
#else
		if ((nextCharacter == '\032') &&
		    (History.currentLine->size == 0))
		{	/* end-of-file */
			break;
		}
#endif  GEMDOS

		/*
		 * Check for special editors
		 */
		if (_savedState.isEmacs)
		{	/* emacs style editing */
			rc = _doEmacs(nextCharacter);
			if (rc < 0)
			{	/* end of file */
				break;
			}
			if (rc > 0)
				continue;
		}
		if (_savedState.isVi)
		{	/* vi style editing */
			rc = _doVi(nextCharacter);
			if (rc < 0)
			{	/* end of file */
				break;
			}
			if (rc > 0)
				continue;
		}

		/*
		 * The "basic" editor; it basically matches the UCB 4.2
		 * "newtty" driver except for the extension that the
		 * "escape" character does command or filename completion.
		 */
#ifndef  GEMDOS
		if (nextCharacter == _savedState.basicCharacters.t_eofc)
		{	/* an eof */
			break;
		}
		if (nextCharacter == _savedState.basicAttributes.sg_erase)
		{	/* a simple backspace */
			_doStandardErase(nextCharacter);
			continue;
		}
		if (nextCharacter == _savedState.basicAttributes.sg_kill)
		{	/* kill a line */
			_doStandardKill(nextCharacter);
			continue;
		}
		if (nextCharacter == _savedState.localCharacters.t_rprntc)
		{	/* retype a line */
			_doStandardRetype(nextCharacter);
			continue;
		}
		if (nextCharacter == _savedState.localCharacters.t_werasc)
		{	/* erase a word */
			_deletePrevStdWord();
			continue;
		}
		switch(nextCharacter)
		{	/* decide what to do with it, if not caught above */
		    case '\032': /* other end-of-file character */
			size = 0;
			break;

		    default:	/* a character with no special meaning */
			_doStandardCharacter(nextCharacter);
			if (History.currentLine->size >= sizeof(History.currentLine->contents)-1)
				break;
			continue;

		    case '\033':/* name expansion requested */
			_doStandardExpand(nextCharacter);
			continue;

		    case '\014':/* list expansion possibilities character */
			ExpandAName(History.currentLine->contents,
				sizeof(History.currentLine->contents)-1,
				History.currentLine->size, 0);
			continue;

		    case '\n':	/* eol */
		    case '\r':
			_doStandardEndLine(nextCharacter);
			_doStandardCharacter(nextCharacter);
			break;
		}
		break;
#else
		switch(nextCharacter)
		{	/* decide what to do with it, if not caught above */
                    case '\027':
			_deletePrevStdWord();
			continue;

                    case '\022':
			_doStandardRetype(nextCharacter);
			continue;

                    case '\030':
			_doStandardKill(nextCharacter);
			continue;

		    case '\010':
			_doStandardErase(nextCharacter);
                        continue;

		    case '\004': /* other end-of-file character */
                    case '\032':
			size = 0;
			break;

		    default:	/* a character with no special meaning */
			_doStandardCharacter(nextCharacter);
			if (History.currentLine->size >= sizeof(History.currentLine->contents)-1)
				break;
			continue;

		    case '\033':/* name expansion requested */
			_doStandardExpand(nextCharacter);
			continue;

		    case '\014':/* list expansion possibilities character */
			ExpandAName(History.currentLine->contents,
				sizeof(History.currentLine->contents)-1,
				History.currentLine->size, 0);
			continue;

		    case '\r':
		    	_writechar('\n');
			_doStandardEndLine(nextCharacter);
			_doStandardCharacter(nextCharacter);
			break;
		    case '\n':	/* eol */
			_writechar('\r');
			_doStandardEndLine(nextCharacter);
			_doStandardCharacter(nextCharacter);
			break;
		}
		break;
#endif  GEMDOS
	}

	/*
	 * Clean up and return
	 */
	_terminate(fp);
	_savedState.stream = -1;
	buffer[0] = '\0';
	size = (size < History.currentLine->size) ? size : History.currentLine->size;
	if (size >= 0)
	{	/* copy in the line to be returned */
		if (size > 0)
			strncpy(buffer,History.currentLine->contents,size);
		buffer[size] = '\0';
		History.currentLine->contents[History.currentLine->size] = '\0';
	}
	else	History.currentLine->contents[0] = '\0';
#ifdef  GEMDOS
        _writechar('\r');
#endif  GEMDOS
	if (_savedState.localEditor != (char *)NULL)
		free(_savedState.localEditor);
	_savedState.localEditor = (char *)NULL;
	return(size);
};	/* end of localReader */

/*
 * The main "Reader" entry point; get a completely new line
 */
int Reader(fp,buffer,size)
int fp;
char *buffer;
int size;
{
	return(localReader((struct historyLine *)NULL,fp,buffer,size));
};	/* end of Reader */

/*
 * The main "Editor" entry point; edit an old line
 */
int Editor(hp,fp,buffer,size)
struct historyLine *hp;
int fp;
char *buffer;
int size;
{
	struct historyLine line;
	register int rc;

	strncpy(line.contents,hp->contents,hp->size);
	line.size = hp->size;
	line.cursor = hp->cursor;
	if (line.contents[line.size-1] == '\n')
		line.size--;
	if (line.contents[line.size-1] == '\r')
		line.size--;
	rc = localReader(&line,fp,buffer,size);
	return(rc);
};	/* end of Editor */

/*
 *	Is s2 a substring of s1?
 */
char *substr(s1,s2)
char *s1;
register char *s2;
{
	register char *s;
	register int sl;

	sl = strlen(s2);
	for (s = s1; *s; s++)
	{
		if (strncmp(s,s2,sl) == 0)
			return(s);
	}
	return((char *)NULL);
};	/* end of substr */
