/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Main Routine
 *
 * $Id: main.c,v 1.11 89/03/07 19:06:03 dclemans Exp $
 *
 * $Log:	main.c,v $
 * Revision 1.11  89/03/07  19:06:03  dclemans
 * portability fixes
 * 
 * Revision 1.10  89/02/22  21:32:00  dclemans
 * Implement simple background job monitoring facility
 * 
 * Revision 1.9  89/02/22  15:39:30  dclemans
 * and also the SYSV default window size
 * 
 * Revision 1.8  89/02/22  15:06:52  dclemans
 * pick up BSD default window size.
 * 
 * Revision 1.7  89/02/21  19:39:25  dclemans
 * Implement RANDOM and SECONDS variables
 * 
 * Revision 1.6  89/02/21  08:36:48  dclemans
 * Implement pseudo-signals ERR, EXIT, DEBUG
 * 
 * Revision 1.5  89/02/20  22:30:31  dclemans
 * Add hooks to setup default signals
 * 
 * Revision 1.4  89/02/20  20:12:00  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#include "shell.h"
#include <time.h>
extern unsigned long time();
#ifndef GEMDOS
#ifdef  unix
#ifndef USG
#include <sgtty.h>
#else
#include <termio.h>
#endif  /* USG */
#include <sys/param.h>
#include <fcntl.h>
extern int errno;
#endif  /* unix */
#endif  /* GEMDOS */

#ifdef  GEMDOS
#ifdef  MWC
long    _stksize = DEFSTACK;
#endif  /* MWC */
#endif  /* GEMDOS */

int flag_echoinput = 0;
int flag_echoexec = 0;
int flag_keywords = 0;
int flag_cmdhash = 0;
int flag_varerr = 0;
int flag_noglob = 0;
int flag_allexport = 0;
int flag_noexec = 0;
int flag_interactive = 1;
#ifndef GEMDOS
int flag_monitor = 1;
#else
int flag_monitor = 0;
#endif  /* GEMDOS */
struct  envinfo base_env;

#ifndef GEMDOS
static int max_sysfd = NOFILE;
static char open_sysfds[NOFILE/8+1];
#endif  /* GEMDOS */

#ifdef  MWC
extern  char *_iovector;
#endif  /* MWC */

main(argc,argv)
int argc;
char **argv;
{
    register struct phrase *pp;
    register struct phrase *sp;
    register int i;
    int flag,rc;
    char *p;
    char *cmd;
    char buffer[16];
#ifndef GEMDOS
#ifndef USG
    struct winsize window;
#endif  /* USG */
#endif  /* GEMDOS */

    base_env.start_at = time(0L);

    cmd_forceexit = 0;
    cmd_count = 1;
    for (i = 0; i <= PROMPT_MAX; i++)
    {   /* set up prompts */
        base_env.prompts[i] = (char *)NULL;
        base_env.prompts_issued[i] = 0;
    }
    base_env.columns = 0;
    base_env.lines = 0;
    base_env.homedir = (char *)NULL;
    base_env.shellname = (char *)NULL;
    base_env.envfile = (char *)NULL;
    base_env.cd_path = (char *)NULL;
    base_env.exec_path = (char *)NULL;
    base_env.separators = (char *)NULL;
    base_env.var = (struct varstack *)NULL;
    base_env.alias_table = (struct aliases *)NULL;
    base_env.func_table = (struct function *)NULL;
    base_env.temp_count = 0;
    base_env.break_level = 0;
    base_env.continue_level = 0;
#ifdef  GEMDOS
    base_env.fd_con = base_env.fd_aux = base_env.fd_prn = -1;
    base_env.tmpdir = (char *)NULL;
    base_env.exec_suff = (char *)NULL;
#endif  /* GEMDOS */
    base_env.dir = (struct dirstack *)NULL;
    base_env.io = (struct iostack *)NULL;
    base_env.pending_heredocs = (struct herestack *)NULL;
    base_env.history_list = (struct hist_phrase *)NULL;
    base_env.history_size = 0;

    if (main_varpush() < 0)
        exit(1);
    if (main_iopush() < 0)
    {   /* enough io set up to run? */
        errmsg(0,LOC("main"),"unable to create I/O environment to run in");
        exit(1);
    }
    base_env.fd_input = base_env.io->input;
    rc = lchdir(".",&p);
    if (rc != 0)
    {   /* if we couldn't find it... */
        errmsg(0,LOC("main"),"couldn't get home directory");
        exit(2);
    }
    if (main_dirpush() < 0)
    {   /* enough memory? */
        /* message already printed */
        exit(3);
    }
    var_define0("HOME",p,0);
    var_define0("PWD",p,0);
    free(p);
    var_settype("RANDOM",TYPE_FUNCTION,~0);
    var_settype("SECONDS",TYPE_FUNCTION,~0);

    for (i = 0; var_init[i] != (char *)NULL; i += 2)
        var_define0(var_init[i],var_init[i+1],0);
    sprintf(buffer,"%d",DEFAULT_HISTORY);
    var_define0("HISTSIZE",buffer,0);
#ifndef GEMDOS
#ifndef USG
    ioctl(base_env.io->input,TIOCGWINSZ,&window);
#else
    ioctl(base_env.io->input,TCGETWINSZ,&window);
#endif  /* USG */
    sprintf(buffer,"%d",window.ws_row);
    var_define0("LINES",buffer,0);
    sprintf(buffer,"%d",window.ws_col);
    var_define0("COLUMNS",buffer,0);
#endif  /* GEMDOS */
    var_readenv();          /* defines environment variables */

    alias_init();           /* defines default aliases */

    signal_init();          /* sets up default signals */

    if (argc > 0 && argv != (char **)NULL && argv != (char **)-1)
    {   /* if there are any args */
        var_setarg0(argv[0]);
        for (i = 1; i < argc; i++)
        {   /* look for our own args */
            if (argv[i][0] != '-')
            {   /* a file to read... */
#ifdef  GEMDOS
                if (strlen(argv[i]) == 0)
                    break;
#endif  /* GEMDOS */
#ifdef  GEMDOS
                base_env.fd_input = Fopen(argv[i],0);
#else
                base_env.fd_input = open(argv[i],0);
#endif  /* GEMDOS */
                if (base_env.io->input < 0)
                {   /* is that a file? */
                    errmsg(0,LOC("main"),"can't open %s",argv[i]);
                    exit(4);
                }
                flag_interactive = 0;
                i++;
                break;
            }
            p = &argv[i][1];
            if (*p == '\0' || *p == '-')
            {   /* end of switches? */
                if (*p == '\0')
                    flag_echoinput = flag_echoexec = 0;
                i++;
                break;
            }
            for (; *p != '\0'; p++)
            {   /* for each arg character */
                switch (*p)
                {   /* which arg? */
                    case 'a':   /* export all? */
                        flag_allexport = 1;
                        break;
                    case 'f':   /* no file expansion? */
                        flag_noglob = 1;
                        break;
                    case 'm':   /* monitor jobs? */
                        flag_monitor = 1;
                        break;
                    case 'n':   /* no execution? */
                        flag_noexec = 1;
                        break;
                    case 'u':   /* unknown var errors? */
                        flag_varerr = 1;
                        break;
                    case 'v':   /* echo input? */
                        flag_echoinput = 1;
                        break;
                    case 'x':   /* echo execution? */
                        flag_echoexec = 1;
                        break;
                    case 'k':   /* all keywords? */
                        flag_keywords = 1;
                        break;
                    case 'h':   /* command hashing? */
                        flag_cmdhash = 1;
                        break;
                    case 'i':   /* force interactiveness? */
                        flag_interactive = 1;
                        break;
                    case 't':   /* only one command? */
                        cmd_forceexit = 1;
                        break;
                    case 'c':   /* exec one command? */
                        i++;
                        cmd = argv[i];
                        if (cmd == (char *)NULL)
                        {   /* if nothing there */
                            errmsg(0,LOC("main"),"missing command arg to -c switch");
                            exit(5);
                        }
                        io_savestring(cmd);
                        cmd_forceexit = 1;
                        flag_interactive = 0;
                        break;
                    default:
                        errmsg(0,LOC("main"),"bad argument: '%s'",argv[i]);
                        exit(6);
                }
            }
        }
        var_setargs(argc-i+1,&argv[i-1]);
    }

    if (base_env.envfile != (char *)NULL)
        io_pushfile(base_env.envfile,0,0,0);

    while ((sp = lex_sentence(0)) != (struct phrase *)NULL)
    {   /* get sentences until EOF */
        if (base_env.prompts_issued[0] > 0 && base_env.history_size > 0 && sp->group != (struct phrase *)NULL)
        {   /* stick onto history list */
            if (base_env.history_list[base_env.history_size-1].cmd != (struct phrase *)NULL)
                sentence_free(base_env.history_list[base_env.history_size-1].cmd);
            for (i = base_env.history_size-1; i > 0; i--)
            {   /* slide up the commands */
                base_env.history_list[i].cmd = base_env.history_list[i-1].cmd;
                base_env.history_list[i].number = base_env.history_list[i-1].number;
            }
            base_env.history_list[0].cmd = copy_group(sp->group,0,0,1);
            base_env.history_list[0].number = base_env.prompts_issued[0];
        }
        rc = exec_sentence(sp,0);
        flag = 1;
        for (pp = sp->group; pp != (struct phrase *)NULL; pp = pp->next)
        {   /* check for early termination */
            if (pp->type == (struct token *)NULL)
                flag = 0;
        }
        sentence_free(sp);
        do_deletes();
        if (!flag || rc == 1)
            break;
        cmd_count++;
    }
    force_signal("EXIT");

    exit(cmd_lastrc);
}   /* end of main */

#ifndef GEMDOS
int main_fdget()
{
    register int i;

    i = max_sysfd-1;
    while (i >= 0)
    {   /* look for unused fd */
        if (!(open_sysfds[i/8] & (1<<(i%8))))
        {   /* an unused fd? */
            open_sysfds[i/8] |= (1<<(i%8));
            return i;
        }
        i--;
    }
}   /* end of main_fdget */

int main_fdput(i)
int i;
{
    open_sysfds[i/8] &= ~(1<<(i%8));
    return 0;
}   /* end of main_fdput */
#endif  /* GEMDOS */

int main_iopush()
{
    long fd;
    int i;
    register struct iostack *iop;
#ifndef GEMDOS
    int topfd,rc;
#endif  /* GEMDOS */

    iop = (struct iostack *)malloc(sizeof(*iop));
    if (iop == (struct iostack *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("main_iopush"));
        return -1;
    }
    iop->next = base_env.io;
    iop->input = iop->output = iop->errout = -1;
    base_env.io = iop;

#ifdef  GEMDOS
#define FD_INP  0
#define FD_OUT  1
#define FD_AUX  2
#define FD_PRN  3

    if (base_env.fd_con < 0)
    {   /* if console, default stderr not there yet? */
        fd = Fopen("con:",2);
        if (fd < 0)
        {   /* all set up for console? */
            iop->errout = FD_OUT;
            errmsg(0,LOC("main_iopush"),"can't set up stderr");
            return -1;
        }
        iop->errout = Fdup(fd);
        if (iop->errout < 0)
        {   /* all set up for console? */
            iop->errout = FD_OUT;
            errmsg(0,LOC("main_iopush"),"can't set up stderr");
            return -1;
        }
        base_env.fd_con = iop->errout;
    }
    else
    {   /* just dup currently setup stderr */
        iop->errout = Fdup(2);
        if (iop->errout < 0)
        {   /* all set up for stderr? */
            iop->errout = base_env.fd_con;
            errmsg(0,LOC("main_iopush"),"can't set up stderr");
            return -1;
        }
    }
    iop->input = Fdup(FD_INP);
    if (iop->input < 0)
    {   /* can we get a new stdin? */
        errmsg(0,LOC("main_iopush"),"can't setup stdin for use");
        return -1;
    }
    iop->output = Fdup(FD_OUT);
    if (iop->output < 0)
    {   /* can we get a new stdout? */
        errmsg(0,LOC("main_iopush"),"can't setup stdout for use");
        return -1;
    }
    if (base_env.fd_aux < 0)
    {   /* got ptr to aux: */
        base_env.fd_aux = Fdup(FD_AUX);
        if (base_env.fd_aux < 0)
        {   /* can we save the aux: handle? */
            errmsg(0,LOC("main_iopush"),"can't save aux:");
            return -1;
        }
    }
    if (base_env.fd_prn < 0)
    {   /* get ptr to prn: */
        base_env.fd_prn = Fdup(FD_PRN);
        if (base_env.fd_prn < 0)
        {   /* can we save the prn: handle? */
            errmsg(0,LOC("main_iopush"),"can't save prn:");
            return -1;
        }
    }
    for (i = 0; i <= MAXUFD; i++)
        Fclose(i);
#else
    topfd = main_fdget();
    if ((rc = dup2(2,topfd)) < 0)
    {   /* set up standard error? */
        if (iop->next != (struct iostack *)NULL)
            iop->errout = iop->next->errout;
        else iop->errout = 2;
        errmsg(0,LOC("main_iopush"),"can't setup stderr rc=%d,errno=%d",rc,errno);
        return -1;
    }
    iop->errout = topfd;
    fcntl(iop->errout,F_SETFD,1);
    topfd = main_fdget();
    if ((rc = dup2(1,topfd)) < 0)
    {   /* set up standard output? */
        errmsg(0,LOC("main_iopush"),"can't setup stdout rc=%d,errno=%d",rc,errno);
        return -1;
    }
    iop->output = topfd;
    fcntl(iop->output,F_SETFD,1);
    topfd = main_fdget();
    if ((rc = dup2(0,topfd)) < 0)
    {   /* set up standard input? */
        errmsg(0,LOC("main_iopush"),"can't setup stdin rc=%d,errno=%d",rc,errno);
        return -1;
    }
    iop->input = topfd;
    fcntl(iop->input,F_SETFD,1);

    for (i = 0; i < MAXUFD; i++)
        close(i);
#endif  /* GEMDOS */
    return 0;
}   /* end of main_iopush */

int main_iopop()
{
    register struct iostack *iop;

    iop = base_env.io;
    base_env.io = iop->next;

#ifdef  GEMDOS
    Fclose(iop->input);
    Fclose(iop->output);
    Fclose(iop->errout);
#else
    if (iop->input != -1)
    {   /* close it down */
        close(iop->input);
        main_fdput(iop->input);
    }
    if (iop->output != -1)
    {   /* close it down */
        close(iop->output);
        main_fdput(iop->output);
    }
    if (iop->errout != -1)
    {   /* close it down */
        close(iop->errout);
        main_fdput(iop->errout);
    }
#endif  /* GEMDOS */

    free(iop);
    return 0;
}   /* end of main_iopop */

int main_dirpush()
{
    register struct dirstack *dp;

    dp = (struct dirstack *)malloc(sizeof(*dp));
    if (dp == (struct dirstack *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("main_dirpush"));
        return -1;
    }
    dp->next = base_env.dir;
    base_env.dir = dp;
    dp->current = (char *)NULL;
    return 0;
}   /* end of main_dirpush */

int main_dirpop()
{
    int rc;
    char *newdir;
    register struct dirstack *dp;

    dp = base_env.dir;
    base_env.dir = dp->next;

    newdir = (char *)NULL;
    if (base_env.dir != (struct dirstack *)NULL)
    {   /* if something to go back to */
        rc = lchdir(base_env.dir->current,&newdir);
        if (rc == 0)
        {   /* if we got back to old dir */
            var_define0("PWD",newdir,0);
            free(newdir);
        }
    }
    if (dp->current != (char *)NULL)
        free(dp->current);
    free(dp);
    return rc;
}   /* end of main_dirpop */

int main_varpush()
{
    register struct varstack *vs;

    vs = (struct varstack *)malloc(sizeof(struct varstack));
    if (vs == (struct varstack *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("main_varpush"));
        return -1;
    }
    vs->table = (struct variable *)NULL;
    vs->next = base_env.var;
    base_env.var = vs;
    return 0;
}   /* end of main_varpush */

int main_varpop()
{
    register struct varstack *vs;

    if (base_env.var == (struct varstack *)NULL)
        return 1;
    vs = base_env.var;
    base_env.var = vs->next;

    var_tablefree(vs->table);
    free(vs);
    return 0;
}   /* end of main_varpop */

struct argstack
{
    int     argc;
    char    **argv;
    struct  argstack *next;
};
static struct argstack *argtop = (struct argstack *)NULL;

int main_argspush(pp)
struct phrase *pp;
{
    register struct argstack *as;

    as = (struct argstack *)malloc(sizeof(*as));
    if (as == (struct argstack *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("main_argspush"));
        return -1;
    }
    as->next = argtop;
    argtop = as;
    as->argc = var_argc;
    as->argv = var_argv;
    var_argc = 0;
    var_argv = (char **)NULL;
    if (pp->body != (struct token *)NULL)
        var_resetargs(pp->body->next);
    return 0;
}   /* end of main_argspush */

int main_argspop()
{
    register int i;

    if (argtop == (struct argstack *)NULL)
        return 1;
    for (i = 1; i < var_argc; i++)
        if (var_argv[i] != (char *)NULL)
            free(var_argv[i]);
    if (var_argv != (char **)NULL)
        free(var_argv);
    var_argc = argtop->argc;
    var_argv = argtop->argv;
    argtop = argtop->next;
    return 0;
}   /* end of main_argspop */
