/*
 * Command Input Shell
 * Dave Clemans
 * 12/88-1/89
 *
 * "spiritually" based on Bourne, Korn shells
 *
 * Built-in Commands (part 2)
 *
 * $Id: cmd2.c,v 1.8 89/03/05 15:37:54 dclemans Exp $
 *
 * $Log:	cmd2.c,v $
 * Revision 1.8  89/03/05  15:37:54  dclemans
 * more fixes
 * 
 * Revision 1.7  89/02/25  17:39:55  dclemans
 * miscellaneous bug fixes/speedups
 * 
 * Revision 1.6  89/02/22  21:31:52  dclemans
 * Implement simple background job monitoring facility
 * 
 * Revision 1.5  89/02/21  20:29:10  dclemans
 * Fix bug with shell variable references in history lists.
 * 
 * Revision 1.4  89/02/20  20:14:18  dclemans
 * Add RCS identifiers
 * 
 */
#include <stdio.h>
#ifndef GEMDOS
#include <sys/types.h>
#endif  /* GEMDOS */
#include <time.h>
#include "shell.h"

static int run_phrase(pp)
struct phrase *pp;
{
    register struct phrase *cpp;

    cpp = copy_phrase(pp,1,0,1);
    if (cpp == (struct phrase *)NULL)
    {   /* enough memory? */
        /* message already printed */
        return 1;
    }
    if (flag_echoexec)
        phrase_dump(cpp,0,0);
    exec_phrase(cpp,0);
    if (cpp->body != (struct token *)NULL)
        var_define0("_",cpp->body_end->name,0);
    phrase_free(cpp);
    return 0;
}   /* end of run_phrase */

static int run_args(pp)
struct phrase *pp;
{
    register struct phrase *cpp;
    register struct token *tp;
    register struct iotoken *iop;

    tp = pp->body;
    pp->body = tp->next;
    iop = pp->io;
    pp->io = (struct iotoken *)NULL;

    cpp = copy_phrase(pp,1,1,0);
    if (cpp == (struct phrase *)NULL)
    {   /* enough memory? */
        /* message already printed */
        pp->body = tp;
        pp->io = iop;
        return 1;
    }
    if (flag_echoexec)
        phrase_dump(cpp,0,0);
    exec_phrase(cpp,0);
    if (cpp->body != (struct token *)NULL)
        var_define0("_",cpp->body_end->name,0);
    phrase_free(cpp);
    pp->body = tp;
    pp->io = iop;
    return 0;
}   /* end of run_args */

static int run_group(npp)
register struct phrase *npp;
{
    for (; npp != (struct phrase *)NULL; npp = npp->next)
    {   /* finally, do the execution */
        if (run_phrase(npp))
            return 1;
        if (npp->type != (struct token *)NULL) switch (npp->type->type)
        {   /* any special handling? */
            case SYM_ANDIF:
                if (cmd_lastrc != 0 && npp != (struct phrase *)NULL)
                    npp = npp->next;
                break;
            case SYM_ORIF:
                if (cmd_lastrc == 0 && npp != (struct phrase *)NULL)
                    npp = npp->next;
                break;
        }
        if (npp == (struct phrase *)NULL)
            break;
        if (cmd_returnexit)
            break;
        if (base_env.break_level > 0)
            break;
        if (base_env.continue_level > 0)
            break;
    }
    return 0;
}   /* end of run_group */

int cmd_for(pp)
register struct phrase *pp;
{
    register struct token *tp,*args;
    int ctr;

    if (pp->body->next == (struct token *)NULL)
    {   /* missing operands? */
        errmsg(0,LOC("cmd_for"),"no operands to iterate over");
        return 1;
    }
    tp = pp->body->next;
    if (tp->next == (struct token *)NULL)
    {   /* iterate over positional args? */
        for (ctr = 1; ctr < var_argc; ctr++)
        {   /* for each known arg */
            var_define0(tp->name,var_argv[ctr],0);
            if (run_group(pp->group))
                break;
            if (cmd_returnexit)
                break;
            if (base_env.break_level > 0)
            {   /* if breaking */
                base_env.break_level--;
                break;
            }
            if (base_env.continue_level > 0)
            {   /* if continuing */
                base_env.continue_level--;
                if (base_env.continue_level > 0)
                    break;
            }
        }
        return cmd_lastrc;
    }
    if (strcmp(tp->next->name,"in") != 0)
    {   /* good separator? */
        errmsg(0,LOC("cmd_for"),"bad for usage: for <var> in <args>");
        return 1;
    }
    args = lex_reparse_tokens(tp->next->next,1);
    while (args != (struct token *)NULL)
    {   /* while there are args to iterate over */
        var_define0(tp->name,args->name,0);
        if (run_group(pp->group))
            break;
        args = args->next;
        if (cmd_returnexit)
            break;
        if (base_env.break_level > 0)
        {   /* if breaking */
            base_env.break_level--;
            break;
        }
        if (base_env.continue_level > 0)
        {   /* if continuing */
            base_env.continue_level--;
            if (base_env.continue_level > 0)
                break;
        }
    }
    if (args != (struct token *)NULL)
        tokens_free(args);
    return cmd_lastrc;
}   /* end of cmd_for */

static int select_menu(atp)
struct token *atp;
{
    register int ctr,i;
    int options,maxlength,ncol;
    register struct token *tp;
    char **args;
    char *p,*savep;
    char buffer[BUFSIZ];

    maxlength = 0;
    if (atp == (struct token *)NULL)
    {   /* look at positional args */
        options = var_argc-1;
        for (ctr = 1; ctr <= options; ctr++)
            if (strlen(var_argv[ctr]) > maxlength)
                maxlength = strlen(var_argv[ctr]);
        args = var_argv;
    }
    else
    {   /* look at token args */
        options = 0;
        for (tp = atp; tp != (struct token *)NULL; tp = tp->next)
        {   /* count, check max length */
            options++;
            if (strlen(tp->name) > maxlength)
                maxlength = strlen(tp->name);
        }
        ctr = 1;
        args = new_argv(options+1);
        if (args == (char **)NULL)
        {   /* enough memory? */
            errmsg(SHERR_NOMEM,LOC("select_menu"));
            return -1;
        }
        for (tp = atp; tp != (struct token *)NULL; tp = tp->next)
        {   /* build a fake argv */
            args[ctr] = tp->name;
            ctr++;
        }
    }
    maxlength += 7;     /* space for menu #, column separation */

    for (;;)
    {   /* until we get eof or a good menu response */
        ncol = base_env.columns / maxlength;
        if ((options < (base_env.lines*2/3)) || (ncol < 1))
        {   /* just a single list? */
            for (ctr = 1; ctr <= options; ctr++)
            {   /* print the menu */
                sprintf(buffer,"%d) %s\n",ctr,args[ctr]);
                io_writestring(0,buffer);
            }
        }
        else
        {   /* a multi-column list? */
            for (ctr = 1; ctr <= options; )
            {   /* build lines... */
                p = buffer;
                *p = '\0';
                for (i = 0; i < ncol && (ctr+i) <= options; i++)
                {   /* for each column */
                    sprintf(p,"%d) %s",ctr+i,args[ctr+i]);
                    savep = p;
                    while (*p)
                        p++;
                    while ((int)(p - savep) < maxlength)
                        *p++ = ' ';
                    *p = '\0';
                }
                ctr += ncol;
                strcpy(p,"\n");
                io_writestring(0,buffer);
            }
        }
        io_prompt(2);
        ctr = read(base_env.io->input,buffer,sizeof buffer);
        if (ctr <= 0)
        {   /* eof yet? */
            if (atp != (struct token *)NULL)
                free(args);
            return ctr;
        }
        buffer[ctr] = '\0';
        for (p = buffer; *p; p++)
#ifdef  GEMDOS
            if (*p == '\n' || *p == '\r')
#else
            if (*p == '\n')
#endif  /* GEMDOS */
                *p = '\0';
        for (p = buffer; *p && isspace(*p); p++)
            /* do nothing */;
        ctr = atoi(p);
        if (ctr > 0 && ctr <= options)
        {   /* a selection made? */
            if (atp != (struct token *)NULL)
                free(args);
            return ctr;
        }
    }
}   /* end of select_menu */

int cmd_select(pp)
register struct phrase *pp;
{
    register struct token *tp,*args;
    struct token *atp;
    int ctr;

    if (pp->body->next == (struct token *)NULL)
    {   /* missing operands? */
        errmsg(0,LOC("cmd_select"),"no operands to iterate over");
        return 1;
    }
    tp = pp->body->next;
    if (tp->next == (struct token *)NULL)
    {   /* iterate over positional args? */
        while ((ctr = select_menu((struct token *)NULL)) > 0)
        {   /* for each known arg */
            var_define0(tp->name,var_argv[ctr],0);
            if (run_group(pp->group))
                break;
            if (cmd_returnexit)
                break;
        }
        return cmd_lastrc;
    }
    if (strcmp(tp->next->name,"in") != 0)
    {   /* good separator? */
        errmsg(0,LOC("cmd_select"),"bad select usage: select <var> in <args>");
        return 1;
    }
    args = lex_reparse_tokens(tp->next->next,1);
    while (args != (struct token *)NULL && (ctr = select_menu(args)) > 0)
    {   /* while there are args to iterate over */
        for (atp = args; ctr > 1; ctr--)
            atp = atp->next;
        var_define0(tp->name,atp->name,0);
        var_define0("REPLY",atp->name,0);
        if (run_group(pp->group))
            break;
        if (cmd_returnexit)
            break;
    }
    if (args != (struct token *)NULL)
        tokens_free(args);
    return cmd_lastrc;
}   /* end of cmd_select */

int cmd_while(pp)
register struct phrase *pp;
{
    register char *cmd;
    int until;

    until = 0;
    cmd = strcopy(pp->body->name);
    if (cmd != (char *)NULL)
    {   /* if we got the copy */
        stripquotes(cmd);
        if (strcmp(cmd,"until") == 0)
            until = 1;
        free(cmd);
    }
    if (pp->body->next == (struct token *)NULL)
    {   /* missing operands? */
        errmsg(0,LOC("cmd_while"),"no operands to loop over");
        return 1;
    }

    for (;;)
    {   /* while we have to loop */
        if (run_args(pp))
            return 1;
        if (!until && cmd_lastrc != 0)
            break;
        if (until && cmd_lastrc == 0)
            break;

        if (run_group(pp->group))
            break;
        if (cmd_returnexit)
            break;
        if (base_env.break_level > 0)
        {   /* if breaking */
            base_env.break_level--;
            break;
        }
        if (base_env.continue_level > 0)
        {   /* if continuing */
            base_env.continue_level--;
            if (base_env.continue_level > 0)
                break;
        }
    }
    return cmd_lastrc;
}   /* end of cmd_while */

int cmd_if(pp)
struct phrase *pp;
{
    register struct phrase *npp;
    int condition;

    if (pp->body->next == (struct token *)NULL)
    {   /* missing operands? */
        errmsg(0,LOC("cmd_if"),"no condition to test");
        return 1;
    }
    if (run_args(pp))
        return 1;
    condition = cmd_lastrc;

    for (npp = pp->group; npp != (struct phrase *)NULL; npp = npp->next)
    {   /* finally execute the sentence */
        if (npp->body != (struct token *)NULL)
        {   /* check for else, etc. */
            if (strcmp(npp->body->name,"else") == 0)
                condition = !condition;
            else if (strcmp(npp->body->name,"elif") == 0)
            {   /* an else-if test... */
                if (npp->body->next == (struct token *)NULL)
                {   /* missing operands? */
                    errmsg(0,LOC("cmd_if"),"no condition to test");
                    return 1;
                }
                if (run_args(npp))
                    return 1;
                condition = cmd_lastrc;
            }
        }
        if (condition)
            continue;
        if (run_phrase(npp))
            return 1;
    }
    return cmd_lastrc;
}   /* end of cmd_if */

int cmd_function(pp)
struct phrase *pp;
{
    register struct function *fp,*sfp;
    register struct token *tp;
    int rc;

    tp = pp->body->next;
    if (tp == (struct token *)NULL)
    {   /* if no name, just dump table */
        errmsg(0,LOC("cmd_function"),"unnamed functions are illegal");
        return 1;
    }

    fp = new_function();
    if (fp == (struct function *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("cmd_function"));
        return -1;
    }
    fp->code = copy_phrase(pp,0,0,1);
    if (fp->code == (struct phrase *)NULL)
    {   /* enough memory? */
        /* message already printed */
        free(fp);
        return -1;
    }

    for (sfp = base_env.func_table; sfp != (struct function *)NULL; )
    {   /* look for where to put the function */
        rc = strcmp(tp->name,sfp->name);
        if (rc == 0)
        {   /* replace existing value */
            if (sfp->code != (struct phrase *)NULL)
                phrase_free(sfp->code);
            sfp->code = fp->code;
            free(fp);
            return 0;
        }
        else if (rc < 0)
        {   /* go down the left side? */
            if (sfp->left == (struct function *)NULL)
                break;
            sfp = sfp->left;
        }
        else
        {   /* go down right side */
            if (sfp->right == (struct function *)NULL)
                break;
            sfp = sfp->right;
        }
    }
    fp->name = strcopy(tp->name);
    if (fp->name == (char *)NULL)
    {   /* enough memory? */
        /* message already printed */
        phrase_free(fp->code);
        free(fp);
        return -1;
    }
    if (base_env.func_table == (struct function *)NULL)
        base_env.func_table = fp;
    else if (rc < 0)
        sfp->left = fp;
    else
        sfp->right = fp;
    return 0;
}   /* end of cmd_function */

int cmd_case(pp)
struct phrase *pp;
{
    register struct token *tp;
    register struct phrase *npp,*cpp;
    char *exp;
    int match,state;

    tp = pp->body->next;
    if (tp == (struct token *)NULL)
    {   /* if no operand to switch on */
        errmsg(0,LOC("cmd_case"),"no operands for comparison");
        return -1;
    }
    exp = tp->name;

    match = state = 0;
    cpp = (struct phrase *)NULL;
    for (npp = pp->group; npp != (struct phrase *)NULL; )
    {   /* step through the case */
        if (cpp != (struct phrase *)NULL)
            phrase_free(cpp);
        if (npp->next == (struct phrase *)NULL)
            break;
        cpp = copy_phrase(npp,1,0,1);
        npp = npp->next;
        if (state == 0)
        {   /* if looking for case match */
            if (cpp->type == (struct token *)NULL)
                continue;
            if (cpp->type->type == SYM_EOL &&
                cpp->body == (struct token *)NULL &&
                cpp->io == (struct iotoken *)NULL)
                continue;
            if (cpp->type->type != SYM_PIPE && cpp->type->type != SYM_RPAREN)
            {   /* bad syntax... */
                errmsg(0,LOC("cmd_case"),"bad case argument");
                return -1;
            }
            if (!match)
            {   /* a case match? */
                if (cpp->body != (struct token *)NULL)
                {   /* try a wild-card match */
                    match = wild_match(exp,cpp->body->name);
                }
            }
            if (cpp->type->type == SYM_RPAREN)
            {   /* end of match stuff */
                if (match)
                    state = 1;
                else state = -1;
                continue;
            }
        }
        if (state == 0)
        {   /* if still looking */
            continue;
        }
        if (state < 0)
        {   /* if skipping some sentences */
            if (cpp->type == (struct token *)NULL)
                continue;
            if (cpp->type->type == SYM_DSEMI)
            {   /* end of group? */
                state = 0;
            }
            continue;
        }
        /* else actually exec some sentences */
        if (cpp->type == (struct token *)NULL)
            continue;
        if (flag_echoexec)
            phrase_dump(cpp,0,0);
        exec_phrase(cpp,0);
        if (cpp->body != (struct token *)NULL)
            var_define0("_",cpp->body_end->name,0);
        if (cmd_returnexit)
            break;
        if (base_env.break_level > 0)
        {   /* if breaking */
            base_env.break_level--;
            break;
        }
        if (base_env.continue_level > 0)
        {   /* if continuing */
            base_env.continue_level--;
            if (base_env.continue_level > 0)
                break;
        }
        if (cpp->type->type == SYM_DSEMI)
        {   /* end of group? */
            phrase_free(cpp);
            return cmd_lastrc;
        }
    }

    return cmd_lastrc;
}   /* end of cmd_case */

int cmd_break(pp)
struct phrase *pp;
{
    if (pp->body->next == (struct token *)NULL)
        base_env.break_level = 1;
    else    base_env.break_level = atoi(pp->body->next->name);
    return 0;
}   /* end of cmd_break */

int cmd_continue(pp)
struct phrase *pp;
{
    if (pp->body->next == (struct token *)NULL)
        base_env.continue_level = 1;
    else    base_env.continue_level = atoi(pp->body->next->name);
    return 0;
}   /* end of cmd_continue */

int cmd_return(pp)
struct phrase *pp;
{
    if (pp->body->next != (struct token *)NULL)
        cmd_lastrc = atoi(pp->body->next->name);
    cmd_returnexit = 1;
}   /* end of cmd_return */

static void cmd_fc_dump(curr,nonum)
struct hist_phrase *curr;
int nonum;
{
    char buffer[16];
    register struct phrase *pp;

    if (curr->cmd == (struct phrase *)NULL)
        return;
    if (!nonum)
    {   /* print out the command number? */
        sprintf(buffer,"%-5d",curr->number);
        io_writestring(0,buffer);
    }
    for (pp = curr->cmd; pp != (struct phrase *)NULL; pp = pp->next)
    {   /* dump out the lines */
        phrase_dump(pp,1,5);
        if (pp->next != (struct phrase *)NULL)
            io_writestring(0,"\n     ");
    }
}   /* end of cmd_fc_dump */

static struct phrase *cmd_fc_edit(first,last,edit)
struct hist_phrase *first;
struct hist_phrase *last;
char *edit;
{
    register struct hist_phrase *curr;
    int save_output;
    char *editor;
    register struct phrase *torun,*toend;
    struct phrase *pp;
    struct token *tp;

    editor = edit;
    if (editor == (char *)NULL)
        editor = var_normal("$FCEDIT");
    if (editor == (char *)NULL)
        editor = var_normal("$VISUAL");
    if (editor == (char *)NULL)
        editor = var_normal("$EDITOR");
    if (editor == (char *)NULL)
    {   /* any editor selected? */
        errmsg(0,LOC("cmd_fc_edit"),"no editor available to edit commands");
        return (struct phrase *)NULL;
    }
    torun = new_phrase();
    if (torun == (struct phrase *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("cmd_fc_edit"));
        return (struct phrase *)NULL;
    }
    tp = new_token(1);
    if (tp == (struct token *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("cmd_fc_edit"));
        free(torun);
        return (struct phrase *)NULL;
    }
    tp->type = SYM_EOL;
    strcpy(tp->name,"\n");
    torun->type = tp;
    tp = new_token(strlen(editor));
    if (tp == (struct token *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("cmd_fc_edit"));
        phrase_free(torun);
        return (struct phrase *)NULL;
    }
    tp->type = SYM_WORD;
    strcpy(tp->name,editor);
    torun->body = torun->body_end = tp;
    tp = new_token(16
#ifdef  GEMDOS
                     +strlen(base_env.tmpdir)
#endif  /* GEMDOS */
                                             );
    if (tp == (struct token *)NULL)
    {   /* enough memory? */
        errmsg(SHERR_NOMEM,LOC("cmd_fc_edit"));
        phrase_free(torun);
        return (struct phrase *)NULL;
    }
    tp->type = SYM_WORD;
#ifdef  GEMDOS
    sprintf(tp->name,"%stemp%d.tmp",base_env.tmpdir,base_env.temp_count++);
#else
    sprintf(tp->name,"/tmp/sh%d.temp",base_env.temp_count++);
#endif  /* GEMDOS */
    torun->body_end->next = tp;
    torun->body_end = tp;
    if (edit == (char *)NULL)
        free(editor);

    save_output = base_env.io->output;
    base_env.io->output = creat(tp->name,0666);
    if (base_env.io->output < 0)
    {   /* did we get a temporary file? */
        errmsg(0,LOC("cmd_fc_edit"),"unable to open tempfile");
        phrase_free(torun);
        return (struct phrase *)NULL;
    }
    if (first < last) for (curr = first; curr <= last; curr++)
        cmd_fc_dump(curr,1);
    else for (curr = last; curr >= first; curr--)
        cmd_fc_dump(curr,1);
    close(base_env.io->output);
    base_env.io->output = save_output;

    phrase_dump(torun,1,0);
    exec_phrase(torun,1);
    phrase_free(torun);

    if (io_pushfile(tp->name,1,0,1) < 0)
    {   /* if saving the file failed */
        errmsg(0,LOC("cmd_fc_edit"),"unable to put tempfile into input stream");
        delete_later(tp->name);
        return (struct phrase *)NULL;
    }
    torun = toend = (struct phrase *)NULL;
    while ((pp = lex_phrase(1,0)) != (struct phrase *)NULL)
    {   /* get back contents of file */
        if (pp->type != (struct token *)NULL && pp->type->type == SYM_MARKER)
        {   /* reached end of file */
            phrase_free(pp);
            break;
        }
        if (torun == (struct phrase *)NULL)
            torun = toend = pp;
        else
        {   /* tack onto end */
            toend->next = pp;
            toend = pp;
        }
    }
    return torun;
}   /* end of cmd_fc_edit */

static int cmd_fc_copy(curr,cpp,lcpp)
struct hist_phrase *curr;
struct phrase **cpp;
struct phrase **lcpp;
{
    register struct phrase *pp,*np;

    if (curr->cmd == (struct phrase *)NULL)
        return 0;
    for (pp = curr->cmd; pp != (struct phrase *)NULL; pp = pp->next)
    {   /* copy block of phrases */
        np = copy_phrase(pp,0,0,1);
        if (np == (struct phrase *)NULL)
        {   /* enough memory? */
            /* message already printed */
            sentence_free(*cpp);
            return 1;
        }
        if (*cpp == (struct phrase *)NULL)
            *cpp = *lcpp = np;
        else
        {   /* tack onto end */
            (*lcpp)->next = np;
            *lcpp = np;
        }
    }
    return 0;
}   /* end of cmd_fc_copy */

static char *cmd_fc_substr(s1,s2,l)
char *s1;
register char *s2;
register int l;
{
    register char *s;

    for (s = s1; *s; s++)
    {   /* scan the string */
        if (strncmp(s,s2,l) == 0)
            return(s);
    }
    return (char *)NULL;
}   /* end of cmd_fc_substr */

int cmd_fc_smatch(hp,num,pfx)
struct hist_phrase *hp;
int num;
char *pfx;
{
    struct phrase *pp;
    struct token *tp;
    register char *p,*q;
    register int rc;
    int len;

    pp = hp->cmd;
    if (pp == (struct phrase *)NULL)
        return 0;
    if (num < 0 || num >= base_env.history_size)
    {   /* if to do string selection */
        tp = pp->body;
        if (tp == (struct token *)NULL)
            return 0;
        p = strcopy(tp->name);
        if (p == (char *)NULL)
            return 0;
        stripquotes(p);
        q = strrchr(p,DIR_SEPARATOR);
        if (q == (char *)NULL)
            q = p;
        else q++;
        len = strlen(pfx);
        rc = strncmp(q,pfx,len);
        if (rc != 0)
            rc = strncmp(p,pfx,len);
        free(p);
        return (rc == 0);
    }
    else return (hp->number == num);
}   /* end of cmd_fc_smatch */

int cmd_fc(pp)
struct phrase *pp;
{
    register struct token *tp;
    register int ctr;
    register char *p;
    char *localEd;
    char *rStart,*rEnd,*q;
    int list,nonum,reverse,rKey;
    struct hist_phrase *first,*last,*curr;
    struct phrase *np,*cpp,*lcpp;
    struct token *ntp,*lntp;
    int iFirst,iLast;

    localEd = rStart = rEnd = (char *)NULL;
    iFirst = iLast = -1;
    rKey = list = nonum = reverse = 0;
    for (tp = pp->body->next; tp != (struct token *)NULL; tp = tp->next)
    {   /* look for options / range start & end */
        if (tp->name[0] == '-')
        {   /* if an option */
            stripquotes(tp->name);
            for (p = &tp->name[1]; *p; p++)
            {   /* for each option char */
                switch (*p)
                {   /* which option? */
                    case 'e':
                        if (tp->next != (struct token *)NULL &&
                            (tp->next->name[0] != '-' ||
                            strcmp(tp->next->name,"-") == 0))
                        {   /* get the arg if there */
                            localEd = tp->next->name;
                            stripquotes(localEd);
                            tp = tp->next;
                        }
                        else if (tp->next != (struct token *)NULL &&
                            tp->next->name[0] == '-')
                        {   /* use FCEDIT */
                            localEd = "";
                        }
                        else
                        {   /* no editor name... */
                            errmsg(0,LOC("cmd_fc"),"no editor name supplied for -e");
                            return -1;
                        }
                        break;
                    case 'l':
                        list = 1;
                        break;
                    case 'r':
                        reverse = 1;
                        break;
                    case 'n':
                        nonum = 1;
                        break;
                    default:
                        errmsg(0,LOC("cmd_fc"),"unknown option in arg: %s",p);
                        return -1;
                }
            }
        }
        else
        {   /* range start & end */
            rStart = tp->name;
            stripquotes(rStart);
            if (strchr(rStart,'=') != (char *)NULL)
                rKey = 1;
            else rKey = 0;
            if (tp->type == SYM_NUMBER)
                iFirst = atoi(rStart);
            if (tp->next != (struct token *)NULL)
            {   /* end of range marker */
                rEnd = tp->next->name;
                stripquotes(rEnd);
                if (tp->next->type == SYM_NUMBER)
                    iLast = atoi(rEnd);
            }
            break;
        }
    }

    first = last = curr = (struct hist_phrase *)NULL;
    if (rStart == (char *)NULL && rEnd == (char *)NULL)
    {   /* select whole range */
        if (list)
        {   /* if listing, default to full table */
            first = &base_env.history_list[1];
            last = &base_env.history_list[base_env.history_size-1];
        }
        else
        {   /* default to last used entry */
            first = last = &base_env.history_list[1];
        }
    }
    else if (rStart != (char *)NULL && rEnd == (char *)NULL)
    {   /* select just one */
        rKey = 0;
        for (ctr = 1; ctr < base_env.history_size; ctr++)
        {   /* look for a leading match */
            if (cmd_fc_smatch(&base_env.history_list[ctr],iFirst,rStart))
            {   /* found a match */
                first = last = &base_env.history_list[ctr];
                break;
            }
        }
    }
    else if (!rKey && rStart != (char *)NULL && rEnd != (char *)NULL)
    {   /* select a whole range */
        for (ctr = 1; ctr < base_env.history_size; ctr++)
        {   /* look for a leading match */
            if (first == (struct hist_phrase *)NULL)
            {   /* looking for 1st phrase? */
                if (cmd_fc_smatch(&base_env.history_list[ctr],iFirst,rStart))
                {   /* found a match */
                    first = &base_env.history_list[ctr];
                }
            }
            else
            {   /* find end of match */
                if (cmd_fc_smatch(&base_env.history_list[ctr],iLast,rEnd))
                {   /* found a match */
                    last = &base_env.history_list[ctr];
                    break;
                }
            }
        }
    }
    else if (rKey && rStart != (char *)NULL && rEnd != (char *)NULL)
    {   /* select just one, but do a string substitution */
        for (ctr = 1; ctr < base_env.history_size; ctr++)
        {   /* look for a leading match */
            if (cmd_fc_smatch(&base_env.history_list[ctr],iLast,rEnd))
            {   /* found a match */
                first = last = &base_env.history_list[ctr];
                break;
            }
        }
    }
    else
    {   /* illegal condition */
        errmsg(0,LOC("cmd_fc"),"reached illegal state in command");
        return 1;
    }
    if (first == (struct hist_phrase *)NULL || last == (struct hist_phrase *)NULL)
    {   /* anything selected? */
        errmsg(0,LOC("cmd_fc"),"nothing selected for fc to operate on");
        return 1;
    }

    if (list)
    {   /* if just want to dump out the lines */
        if (reverse) for (curr = first; curr <= last; curr++)
        {   /* step through the selected part of the array */
            cmd_fc_dump(curr,nonum);
        }
        else for (curr = last; curr >= first; curr--)
        {   /* reverse step through the selected part of the array */
            cmd_fc_dump(curr,nonum);
        }
        return 0;
    }

    if (localEd == (char *)NULL ||
        (localEd != (char *)NULL && strcmp(localEd,"-") != 0))
    {   /* edit the commands in a real editor */
        cpp = cmd_fc_edit(first,last,localEd);
        if (cpp == (struct phrase *)NULL)
        {   /* did it work? */
            /* error message already printed */
            return 1;
        }
    }
    else
    {   /* just make a copy of the sentences to execute */
        cpp = lcpp = (struct phrase *)NULL;
        if (!reverse) for (curr = first; curr <= last; curr++)
        {   /* step through the selected part of the array */
            if (cmd_fc_copy(curr,&cpp,&lcpp))
            {   /* if copy didn't work */
                /* message already printed */
                return 1;
            }
        }
        else for (curr = last; curr >= first; curr--)
        {   /* reverse step through the selected part of the array */
            if (cmd_fc_copy(curr,&cpp,&lcpp))
            {   /* if copy didn't work */
                /* message already printed */
                return 1;
            }
        }
        if (rKey)
        {   /* if need to do a string substitution */
            p = strchr(rStart,'=');
            /* ASSERT that p != NULL */
            for (np = cpp; np != (struct phrase *)NULL; np = np->next)
            {   /* for each selected sentence */
                for (lcpp = np; lcpp != (struct phrase *)NULL; lcpp = lcpp->next)
                {   /* for base contents of each sentence */
                    lntp = (struct token *)NULL;
                    for (ntp = lcpp->body; ntp != (struct token *)NULL; ntp = ntp->next)
                    {   /* look at words in phrase */
                        q = cmd_fc_substr(ntp->name,rStart,(int)(p-rStart));
                        if (q != (char *)NULL)
                        {   /* if need to do token replacement */
                            tp = new_token(strlen(ntp->name)-(int)(p-rStart)+strlen(&p[1]));
                            if (tp == (struct token *)NULL)
                            {   /* enough memory? */
                                errmsg(SHERR_NOMEM,LOC("cmd_fc"));
                                sentence_free(cpp);
                                return 1;
                            }
                            tp->next = ntp->next;
                            tp->type = ntp->type;
                            strcpy(tp->name,ntp->name);
                            strcpy(&tp->name[(int)(q-ntp->name)],&p[1]);
                            strcat(tp->name,&ntp->name[(int)(q-ntp->name)+(int)(p-rStart)]);
                            if (lntp != (struct token *)NULL)
                                lntp->next = tp;
                            else lcpp->body = tp;
                            free(ntp);
                            break;
                        }
                        lntp = ntp;
                    }
                    if (ntp != (struct token *)NULL)
                        break;
                }
                if (lcpp != (struct phrase *)NULL)
                    break;
            }
        }
    }

    if (base_env.history_size > 0)
    {   /* if saving history of executed lines */
        phrase_free(base_env.history_list[0].cmd);
        base_env.history_list[0].cmd = copy_group(cpp,0,0,1);
    }
    for (np = cpp; np != (struct phrase *)NULL; np = np->next)
    {   /* finally run the finished phrases */
        phrase_dump(np,1,0);
        lcpp = copy_phrase(np,1,0,1);
        if (lcpp == (struct phrase *)NULL)
        {   /* enough memory? */
            /* message already printed */
            continue;
        }
        exec_phrase(lcpp,0);
        phrase_free(lcpp);
        if (cmd_forceexit || cmd_returnexit)
            break;
    }
    sentence_free(cpp);
    return 0;
}   /* end of cmd_fc */

int cmd_eval(pp)
struct phrase *pp;
{
    register struct phrase *torun,*toend,*np;
    struct token *tp;
    char tempfile[32];
    int save_output;

    if (pp->body->next == (struct token *)NULL)
    {   /* anything to eval? */
        return 1;
    }
    np = copy_phrase(pp,1,1,1);
    if (np == (struct phrase *)NULL)
    {   /* enough memory? */
        /* message already printed */
        return 1;
    }
    tp = np->body;
    np->body = tp->next;
    free(tp);		/* free "eval" word */

#ifdef  GEMDOS
    sprintf(tempfile,"%stemp%d.tmp",base_env.tmpdir,base_env.temp_count++);
#else
    sprintf(tempfile,"/tmp/sh%d.temp",base_env.temp_count++);
#endif  /* GEMDOS */
    save_output = base_env.io->output;
    base_env.io->output = creat(tempfile,0666);
    if (base_env.io->output < 0)
    {   /* did we get a temporary file? */
        errmsg(0,LOC("cmd_eval"),"unable to open tempfile");
        phrase_free(np);
        return 1;
    }
    phrase_dump(np,1,0);
    close(base_env.io->output);
    base_env.io->output = save_output;
    phrase_free(np);

    if (io_pushfile(tempfile,1,0,1) < 0)
    {   /* if saving the file failed */
        errmsg(0,LOC("cmd_eval"),"unable to put tempfile into input stream");
        delete_later(tempfile);
        return 1;
    }
    torun = toend = (struct phrase *)NULL;
    while ((np = lex_phrase(1,0)) != (struct phrase *)NULL)
    {   /* get back contents of file */
        if (np->type != (struct token *)NULL && np->type->type == SYM_MARKER)
        {   /* reached end of file */
            phrase_free(np);
            break;
        }
        if (torun == (struct phrase *)NULL)
            torun = toend = np;
        else
        {   /* tack onto end */
            toend->next = np;
            toend = np;
        }
    }
    for (np = torun; np != (struct phrase *)NULL; np = np->next)
    {   /* finally run the finished phrases */
        exec_phrase(np,0);
        if (cmd_forceexit || cmd_returnexit)
            break;
    }
    return 0;
}   /* end of cmd_eval */

int cmd_exec(pp)
struct phrase *pp;
{
    if (pp->body->next == (struct token *)NULL)
        return 0;
    run_args(pp);
    cmd_forceexit = 1;
    return 0;
}   /* end of cmd_exec */

int cmd_time(pp)
struct phrase *pp;
{
    time_t tstart,tend;
    long min,sec;
    char buffer[256];

    if (pp->body->next == (struct token *)NULL)
        return 0;
    tstart = time(0L);
    run_args(pp);
    tend = time(0L);
    tend -= tstart;
    min = tend / 60L;
    sec = tend % 60L;
    if (min == 0)
    {   /* only seconds to worry about? */
        if (sec == 1)
            sprintf(buffer,"1 second");
        else sprintf(buffer,"%ld seconds",sec);
    }
    else
    {   /* both minutes and seconds */
        if (min == 1 && sec == 1)
            sprintf(buffer,"1 minute, 1 second");
        else if (sec == 1)
            sprintf(buffer,"%ld minutes, 1 second",min);
        else if (min == 1)
            sprintf(buffer,"1 minute, %ld seconds",sec);
        else sprintf(buffer,"%ld minutes, %ld seconds",min,sec);
    }
    strcat(buffer,"\n");
    io_writestring(1,buffer);
    return 0;
}   /* end of cmd_time */
