/***********************************************************
Copyright 1992 by Stichting Mathematisch Centrum, Amsterdam, The
Netherlands.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its 
documentation for any purpose and without fee is hereby granted, 
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in 
supporting documentation, and that the names of Stichting Mathematisch
Centrum or CWI not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior permission.

STICHTING MATHEMATISCH CENTRUM DISCLAIMS ALL WARRANTIES WITH REGARD TO
THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
FITNESS, IN NO EVENT SHALL STICHTING MATHEMATISCH CENTRUM BE LIABLE
FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT
OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.

******************************************************************/


#include <unistd.h>
#include <sys.s>
#include <stdio.h>
#include <sys/file.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/proc.h>
#include <sys/user.h>
#include <sys/ptrace.h>
#include <sys/fs/dbfcntl.h>
#include <sys/sgigsc.h>
#include <sys/syssgi.h>
#include <sys/sysmips.h>

#define BITSPERLONG	(sizeof(long) * 8)
#ifndef SYSMASKLEN
#define SYSMASKLEN	(sizeof(sysset_t)/sizeof(long))
#endif

#define last(table)	(sizeof(table)/sizeof(table[0]))

#define REG_SYSCALL	(GPR_BASE+2)
#define REG_ARG0	(GPR_BASE+4)
#define REG_ARG1	(GPR_BASE+5)
#define REG_ARG2	(GPR_BASE+6)
#define REG_ARG3	(GPR_BASE+7)
#define REG_ARG4	(GPR_BASE+8)
#define REG_SP		(GPR_BASE+29)
#define REG_RETADDR	(GPR_BASE+31)

#define MAX_ARGS	7		/* max # of args to a syscall */

#define BUFLEN		32

/*
 * sys_args codes the number and types of arguments.  Each character in
 * the string corresponds to one argument.  The following codes are used:
 *	i	the argument is an integer
 *	p	the argument is a pointer
 *	P	the argument is a pointer which points to a buffer of longs,
 *		the length of the buffer (in longs) is given by the previous
 *		argument
 *	s	the argument is a (null terminated) string
 *	b	the argument is a buffer, the length of which is given by
 *		the following argument
 *	G	sgigsc system call
 *	S	syssgi system call
 *	M	sysmips system call
 *	B	the argument is a pointer which points to a result buffer,
 *		the length of the filled-in buffer is returned by the call
 *	k	the argument is a signal number
 * The following codes can only be used in indirections:
 *	I	short (2 byte) integer
 *	C	very short (1 byte) integer
 *	X	a 1 byte pad
 */
struct syscalltab {
	const char *sys_name;	/* name of the system call */
	const char *sys_args;	/* args, see above */
	const struct syscalltab * const sys_indirect;
};

struct sgigsc_cmd {
	int sc_cmd;
	const char *sc_name;
	const char *sc_args;
};

const struct syscalltab msgsystab[] = {
	{	"msgget",	"ii",		},
	{	"msgctl",	"iiiii",	},
	{	"msgrcv",	"ipiii",	},
	{	"msgsnd",	"ip(4)ii",	},
};

const struct syscalltab procblktab[] = {
	{	"blockproc",	"i",		},
	{	"unblockproc",	"i",		},
	{	"setblockproccnt","ii",		},
	{	"blockprocall",	"i",		},
	{	"unblockprocall","i",		},
	{	"setblockproccntall","ii"	},
};

const struct syscalltab semsystab[] = {
	{	"semctl",	"iiiii",	},
	{	"semget",	"iii",		},
	{	"semop",	"ipi",		},
};

const struct syscalltab shmsystab[] = {
	{	"shmat",	"ipi",		},
	{	"shmctl",	"iiiii",	},
	{	"shmdt",	"p",		},
	{	"shmget",	"iii",		},
};

const struct syscalltab syscalltab[] = {
	{	"syscall",	"iiiii",	},	/* 0 */
	{	"exit",		"i",		},	/* 1 */
	{	"fork",		"",		},	/* 2 */
	{	"read",		"iBi",		},	/* 3 */
	{	"write",	"ibi",		},	/* 4 */
	{	"open",		"sii",		},	/* 5 */
	{	"close",	"i",		},	/* 6 */
	{	"wait",		"p",		},	/* 7 */
	{	"creat",	"si",		},	/* 8 */
	{	"link",		"ss",		},	/* 9 */
	{	"unlink",	"s",		},	/* 10 */
	{	"execv",	"sp",		},	/* 11 */
	{	"chdir",	"s",		},	/* 12 */
	{	"time",		"p",		},	/* 13 */
	{	"mknod",	"sii",		},	/* 14 */
	{	"chmod",	"si",		},	/* 15 */
	{	"chown",	"sii",		},	/* 16 */
	{	"brk",		"p",		},	/* 17 */
	{	"stat",		"sp",		},	/* 18 */
	{	"lseek",	"iii",		},	/* 19 */
	{	"getpid",	"",		},	/* 20 */
	{	"mount",	"ssiipi",	},	/* 21 */
	{	"umount",	"s",		},	/* 22 */
	{	"setuid",	"i",		},	/* 23 */
	{	"getuid",	"",		},	/* 24 */
	{	"stime",	"p(4i)",	},	/* 25 */
	{	"ptrace",	"iipi",		},	/* 26 */
	{	"alarm",	"i",		},	/* 27 */
	{	"fstat",	"ip",		},	/* 28 */
	{	"pause",	"",		},	/* 29 */
	{	"utime",	"sp(8ii)",	},	/* 30 */
	{	"stty",		"ip",		},	/* 31 */
	{	"gtty",		"ip",		},	/* 32 */
	{	"access",	"si",		},	/* 33 */
	{	"nice",		"i",		},	/* 34 */
	{	"statfs",	"spii",		},	/* 35 */
	{	"sync",		"",		},	/* 36 */
	{	"kill",		"ik",		},	/* 37 */
	{	"fstatfs",	"ipii",		},	/* 38 */
	{	"setpgrp",	"i",		},	/* 39 */
	{	"syssgi",	"Siiii",	},	/* 40 */
	{	"dup",		"i",		},	/* 41 */
	{	"pipe",		"p",		},	/* 42 */
	{	"times",	"p",		},	/* 43 */
	{	"profil",	"piii",		},	/* 44 */
	{	"plock",	"i",		},	/* 45 */
	{	"setgid",	"i",		},	/* 46 */
	{	"getgid",	"",		},	/* 47 */
	{	"signal",	"kp",		},	/* 48 */
	{	"msgsys",	0,	msgsystab},	/* 49 */
	{	"sysmips",	"iiii",		},	/* 50 */
	{	"acct",		"s",		},	/* 51 */
	{	"shmsys",	0,	shmsystab},	/* 52 */
	{	"semsys",	0,	semsystab},	/* 53 */
	{	"ioctl",	"iip",		},	/* 54 */
	{	"uadmin",	"iii",		},	/* 55 */
	{	"sysmp",	"iiiii",	},	/* 56 */
	{	"utssys",	"iiiii",	},	/* 57 */
	{	"#58",		"",		},	/* 58 */
	{	"execve",	"spp",		},	/* 59 */
	{	"umask",	"i",		},	/* 60 */
	{	"chroot",	"s",		},	/* 61 */
	{	"fcntl",	"iii",		},	/* 62 */
	{	"ulimit",	"ii",		},	/* 63 */
	{	"#64",		"",		},	/* 64 */
	{	"#65",		"",		},	/* 65 */
	{	"#66",		"",		},	/* 66 */
	{	"#67",		"",		},	/* 67 */
	{	"#68",		"",		},	/* 68 */
	{	"#69",		"",		},	/* 69 */
	{	"advfs",	"iiiii",	},	/* 70 */
	{	"unadvfs",	"iiiii",	},	/* 71 */
	{	"rmount",	"iiiii",	},	/* 72 */
	{	"rumount",	"iiiii",	},	/* 73 */
	{	"rfstart",	"iiiii",	},	/* 74 */
	{	"#75",		"",		},	/* 75 */
	{	"rdebug",	"iiiii",	},	/* 76 */
	{	"rfstop",	"iiiii",	},	/* 77 */
	{	"rfsys",	"iiiii",	},	/* 78 */
	{	"rmdir",	"s",		},	/* 79 */
	{	"mkdir",	"si",		},	/* 80 */
	{	"getdents",	"iBi",		},	/* 81 */
	{	"sginap",	"i",		},	/* 82 */
	{	"sgikopt",	"sbi",		},	/* 83 */
	{	"sysfs",	"isp",		},	/* 84 */
	{	"getmsg",	"ip(12iip)p(12iip)p",},	/* 85 */
	{	"putmsg",	"ip(12iip)p(12iip)i",},	/* 86 */
	{	"poll",		"pii",		},	/* 87 */
	{	"sigreturn",	"",		},	/* 88 */
	{	"accept",	"ip(16IIiii)p(4i)",},	/* 89 */
	{	"bind",		"ipi",		},	/* 90 */
	{	"connect",	"ip(16IIiii)i",	},	/* 91 */
	{	"gethostid",	"",		},	/* 92 */
	{	"getpeername",	"ipp(4i)",	},	/* 93 */
	{	"getsockname",	"ipp(4i)",	},	/* 94 */
	{	"getsockopt",	"iiipp(4i)",	},	/* 95 */
	{	"listen",	"ii",		},	/* 96 */
	{	"recv",		"ipii",		},	/* 97 */
	{	"recvfrom",	"ipiipp(4i)",	},	/* 98 */
	{	"recvmsg",	"ipi",		},	/* 99 */
	{	"select",	"ip(4i)p(4i)p(4i)p(8ii)",},/* 100 */
	{	"send",		"ibii",		},	/* 101 */
	{	"sendmsg",	"ipi",		},	/* 102 */
	{	"sendto",	"ibiipi",	},	/* 103 */
	{	"sethostid",	"i",		},	/* 104 */
	{	"setsockopt",	"iiipi",	},	/* 105 */
	{	"shutdown",	"ii",		},	/* 106 */
	{	"socket",	"iii",		},	/* 107 */
	{	"gethostname",	"pi",		},	/* 108 */
	{	"sethostname",	"bi",		},	/* 109 */
	{	"getdomainname","pi",		},	/* 110 */
	{	"setdomainname","bi",		},	/* 111 */
	{	"truncate",	"si",		},	/* 112 */
	{	"ftruncate",	"ii",		},	/* 113 */
	{	"rename",	"ss",		},	/* 114 */
	{	"symlink",	"ss",		},	/* 115 */
	{	"readlink",	"sBi",		},	/* 116 */
	{	"lstat",	"sp",		},	/* 117 */
	{	"nfsmount",	"iiiii",	},	/* 118 */
	{	"nfssvc",	"i",		},	/* 119 */
	{	"getfh",	"iiiii",	},	/* 120 */
	{	"async_daemon",	"",		},	/* 121 */
	{	"exportfs",	"sp",		},	/* 122 */
	{	"setregid",	"ii",		},	/* 123 */
	{	"setreuid",	"ii",		},	/* 124 */
	{	"getitimer",	"ip",		},	/* 125 */
	{	"setitimer",	"ip(16iiii)p",	},	/* 126 */
	{	"adjtime",	"p(8ii)p",	},	/* 127 */
	{	"BSD_getime",	"p",		},	/* 128 */
	{	"sproc",	"pip",		},	/* 129 */
	{	"prctl",	"ipp",		},	/* 130 */
	{	"procblk",	0,	procblktab},	/* 131 */
	{	"texturebind",	"pii",		},	/* 132 */
	{	"sgigsc",	"Gi",		},	/* 133 */
	{	"mmap",		"piiiii",	},	/* 134 */
	{	"munmap",	"pi",		},	/* 135 */
	{	"mprotect",	"",		},	/* 136 */
	{	"msync",	"pii",		},	/* 137 */
	{	"madvise",	"pii",		},	/* 138 */
	{	"pagelock",	"pii",		},	/* 139 */
	{	"getpagesize",	"",		},	/* 140 */
	{	"libattach",	"",		},	/* 141 */
	{	"libdetach",	"",		},	/* 142 */
	{	"BSDgetpgrp",	"i",		},	/* 143 */
	{	"BSDsetpgrp",	"ii",		},	/* 144 */
	{	"vhangup",	"",		},	/* 145 */
	{	"fsync",	"i",		},	/* 146 */
	{	"fchdir",	"i",		},	/* 147 */
	{	"getrlimit",	"ip",		},	/* 148 */
	{	"setrlimit",	"ip(8ii)",	},	/* 149 */
	{	"cacheflush",	"pii",		},	/* 150 */
	{	"cachectl",	"pii",		},	/* 151 */
	{	"fchown",	"iii",		},	/* 152 */
	{	"fchmod",	"ii",		},	/* 153 */
	{	"wait3",	"pip",		},	/* 154 */
	{	"socketpair",	"iiip",		},	/* 155 */
};

const char *signalname[] = {
	0,
	"SIGHUP",	/* 1 */
	"SIGINT",	/* 2 */
	"SIGQUIT",	/* 3 */
	"SIGILL",	/* 4 */
	"SIGTRAP",	/* 5 */
	"SIGIOT",	/* 6 */
	"SIGEMT",	/* 7 */
	"SIGFPE",	/* 8 */
	"SIGKILL",	/* 9 */
	"SIGBUS",	/* 10 */
	"SIGSEGV",	/* 11 */
	"SIGSYS",	/* 12 */
	"SIGPIPE",	/* 13 */
	"SIGALRM",	/* 14 */
	"SIGTERM",	/* 15 */
	"SIGUSR1",	/* 16 */
	"SIGUSR2",	/* 17 */
	"SIGCLD",	/* 18 */
	"SIGPWR",	/* 19 */
	"SIGSTOP",	/* 20 */
	"SIGTSTP",	/* 21 */
	"SIGPOLL",	/* 22 */
	"SIGIO",	/* 23 */
	"SIGURG",	/* 24 */
	"SIGWINCH",	/* 25 */
	"SIGVTALRM",	/* 26 */
	"SIGPROF",	/* 27 */
	"SIGCONT",	/* 28 */
	"SIGTTIN",	/* 29 */
	"SIGTTOU",	/* 30 */
	"SIGXCPU",	/* 31 */
	"SIGXFSZ",	/* 32 */
};

const char *sigextra[] = {
	"SIGDEFER",
	"SIGHOLD",
	"SIGRELSE",
	"SIGIGNORE",
	"SIGPAUSE",
};

const struct sgigsc_cmd sgigsc_cmd[] = {
	{ SGWM_LOCK, "SGWM_LOCK", "", },
	{ SGWM_UNLOCK, "SGWM_UNLOCK", "", },
	{ SGWM_SIO, "SGWM_SIO", "i", },
	{ SGWM_UNSIO, "SGWM_UNSIO", "i", },
	{ SGWM_WMRUNNING, "SGWM_WMRUNNING", "", },
	{ SGWM_MEWM, "SGWM_MEWM", "", },
	{ SGWM_LOSTGR, "SGWM_LOSTGR", "", },
	{ SGWM_TIEMOUSE, "SGWM_TIEMOUSE", },
	{ SGWM_UNTIEMOUSE, "SGWM_UNTIEMOUSE", },
	{ SGWM_CONSOLE, "SGWM_CONSOLE", },
	{ SGWM_QMEM, "SGWM_QMEM", "p(8pi)", },
	{ SGWM_UNQMEM, "SGWM_UNQMEM", "", },
	{ SGWM_WMMODELOCK, "SGWM_WMMODELOCK", "i", },
	{ SGWM_INIT, "SGWM_INIT", "", },
	{ SG4D80_TAKECX, "SG4D80_TAKECX", },
	{ SG4D80_OLDGIVECX, "SG4D80_OLDGIVECX", },
	{ SG4D80_GIVECX, "SG4D80_GIVECX", "p(12iii)", },
	{ SG4D80_RESET, "SG4D80_RESET", },
	{ SG4D80_MAPALL, "SG4D80_MAPALL", "p(8pp)", },
	{ SG4D80_GM_CMD, "SG4D80_GM_CMD", "p(20ibipi)", },
	{ SG4D80_INIT, "SG4D80_INIT", },
	{ SG4D80_WINADDR, "SG4D80_WINADDR", "p(8pp)", },
	{ SG4D80_GET_RVTYPE, "SG4D80_GET_RVTYPE", },
	{ SG4D80_ACT_CX, "SG4D80_ACT_CX", },
	{ SG4D80_DEACT_CX, "SG4D80_DEACT_CX", },
	{ SG4D80_STORE_CX, "SG4D80_STORE_CX", "p(12ibi)", },
	{ SG4D80_FEEDBACK, "SG4D80_FEEDBACK", "p(20iibii)", },
	{ SG4D80_PICK, "SG4D80_PICK", "p(8bi)", },
	{ SG4D80_ENDPICK, "SG4D80_ENDPICK", "p(12iii)", },
	{ SG4D80_PROBE, "SG4D80_PROBE", },
	{ SG4D80_RELEASE_CX, "SG4D80_RELEASE_CX", },
	{ SG4D100_PIXEL_DMA, "SG4D100_PIXEL_DMA", "p(16ibii)", },
	{ SG4D100_MAPALL, "SG4D100_MAPALL", "p(16pppp)", },
	{ SG4D100_WINADDR, "SG4D100_WINADDR", "p(12ppp)", },
	{ SG4D100_RECTWRITE, "SG4D100_RECTWRITE", "p(32piiiiiii)", },
	{ SG4D80_SWAPBUFFERS, "SG4D80_SWAPBUFFERS", "p(16iipi)", },
	{ SG4D80_CHANGE_MODE, "SG4D80_CHANGE_MODE", "p(20iiipi)", },
	{ SG4D80_DBL_BUFFER, "SG4D80_DBL_BUFFER", },
	{ SG4D80_MAP_MODE, "SG4D80_MAP_MODE", "p(12ipi)", },
	{ SG4D80_SETMAP, "SG4D80_SETMAP", "p(16ipii)", },
	{ SG4D80_WIDCHECK, "SG4D80_WIDCHECK", "p(12ipp)", },
	{ SG4D100_RECTREAD, "SG4D100_RECTREAD", },
	{ SG4DSP_SWAPBUFFERS, "SG4DSP_SWAPBUFFERS", "p(12ipi)", },
	{ SG4DSP_CHANGE_MODE, "SG4DSP_CHANGE_MODE", "p(12ipi)", },
	{ SG4DSP_INIT_CX, "SG4DSP_INIT_CX", "p(8ii)", },
	{ SG4DSP_COPY, "SG4DSP_COPY", "p(80biiiiiiiiiiiiiiiiiii)", },
	{ SG4DSP_TEXBIND, "SG4DSP_TEXBIND", "p(12ipi)", },
	{ SG4DSP_TEXSTATE, "SG4DSP_TEXSTATE", "p(16iipi)", },
	{ SG4DSP_ACCUMBUFFER, "SG4DSP_ACCUMBUFFER", },
	{ SG4DSP_MAPMEM, "SG4DSP_MAPMEM", "p(12bii)", },
	{ SGGR1_PROBE, "SGGR1_PROBE", "i", },
	{ SGGR1_DOWNLD, "SGGR1_DOWNLD", "p(24IIpipip)", },
	{ SGGR1_CHECKRE, "SGGR1_CHECKRE", "", },
	{ SGGR1_PROMIO, "SGGR1_PROMIO", },
	{ SGGR1_MAPALL, "SGGR1_MAPALL", "", },
	{ SGGR1_GINIT, "SGGR1_GINIT", "", },
	{ SGGR1_RELEASE_CX, "SGGR1_RELEASE_CX", "", },
	{ SGGR1_XMAPMODELO, "SGGR1_XMAPMODELO", "p(12ipi)", },
	{ SGGR1_XMAPMODEHI, "SGGR1_XMAPMODEHI", "p(12ipi)", },
	{ SGGR1_MAPCOLOR, "SGGR1_MAPCOLOR", "p(8ICCCXXX)", },
	{ SGGR1_GETMCOLOR, "SGGR1_GETMCOLOR", "p(8ICCCXXX)", },
	{ SGGR1_MAPAUXCOLOR, "SGGR1_MAPAUXCOLOR", "p(8ICCCXXX)", },
	{ SGGR1_SWAPBUFFERS, "SGGR1_SWAPBUFFERS", "p(12CCCXpi)", },
	{ SGGR1_SWAPINTERVAL, "SGGR1_SWAPINTERVAL", "p(4IXX)", },
	{ SGGR1_BLINK, "SGGR1_BLINK", "p(8IICCCX)", },
	{ SGGR1_CYCLEMAP, "SGGR1_CYCLEMAP", "p(8IIIXX)", },
	{ SGGR1_GAMMARAMP, "SGGR1_GAMMARAMP", "p(12ppp)", },
	{ SGGR1_SETCURSOR, "SGGR1_SETCURSOR", "p(8CCCXp)", },
	{ SGGR1_MAPCURSCOLOR, "SGGR1_MAPCURSCOLOR", "p(4CCCC)", },
	{ SGGR1_CURSDISPLAY, "SGGR1_CURSDISPLAY", "p(4CXXX)", },
	{ SGGR1_DISPLAYREG, "SGGR1_DISPLAYREG", },
	{ SGGR1_BLANKSCREEN, "SGGR1_BLANKSCREEN", "p(4CXXX)", },
	{ SGGR1_SETMONITOR, "SGGR1_SETMONITOR", "p(24CCXXiiiii)", },
	{ SGGR1_GETMONITOR, "SGGR1_GETMONITOR", "p", },
	{ SGGR1_DMA, "SGGR1_DMA", "p(32iiiiipICCi)", },
	{ SGGR1_GETHWCONFIG, "SGGR1_GETHWCONFIG", "p", },
	{ SGGR1_PICK, "SGGR1_PICK", "p(8pi)", },
	{ SGGR1_ENDPICK, "SGGR1_ENDPICK", "p(8ii)", },
	{ SGGR1_GSYNC, "SGGR1_GSYNC", "", },
	{ SGGR1_CREATECX, "SGGR1_CREATECX", "i", },
	{ SGGR1_INITCX, "SGGR1_INITCX", "p(8ii)", },
	{ SGGR1_DESTROYCX, "SGGR1_DESTROYCX", "i", },
	{ SGGR1_GIVECX, "SGGR1_GIVECX", "p(12iii)", },
	{ SGGR1_TAKECX, "SGGR1_TAKECX", "i", },
	{ SGGR1_ACT_CX, "SGGR1_ACT_CX", "i", },
	{ SGGR1_DEACT_CX, "SGGR1_DEACT_CX", "", },
	{ SGGR1_STORECX, "SGGR1_STORECX", "p(12ibi)", },
	{ SGGR1_LOADCX, "SGGR1_LOADCX", "p(16iibi)", },
	{ SGGR1_LOADMODE, "SGGR1_LOADMODE", "p(8ii)", },
	{ SGGR1_WRITECX, "SGGR1_WRITECX", "p(16iiiP)", },
	{ SGGR1_GETCURSOFF, "SGGR1_GETCURSOFF", "p", },
	{ SGGR1_CURSOFFHACK, "SGGR1_CURSOFFHACK", "p(8ii)", },
	{ SGGR1_POSCURSOR, "SGGR1_POSCURSOR", "p(8ii)", },
	{ SGGR1_MODERD, "SGGR1_MODERD", "p(8ip)", },
	{ SGGR1_PROTECT_HQMMSB, "SGGR1_PROTECT_HQMMSB", "i", },
	{ SG4DCG2_READ_REG, "SG4DCG2_READ_REG", "p(8ii)", },
	{ SG4DCG2_WRITE_REG, "SG4DCG2_WRITE_REG", "p(8ii)", },
	{ SG4DVMUX_READ_REG, "SG4DVMUX_READ_REG", },
	{ SG4DVMUX_WRITE_REG, "SG4DVMUX_WRITE_REG", },
	{ SGAUX_ALLOCSCRN, "SGAUX_ALLOCSCRN", },
	{ SGAUX_DEALLOCSCRN, "SGAUX_DEALLOCSCRN", },
	{ SGAUX_SETOUTSCRN, "SGAUX_SETOUTSCRN", },
	{ SGAUX_SETINSCRN, "SGAUX_SETINSCRN", },
	{ SGAUX_GETINSCRN, "SGAUX_GETINSCRN", "", },
	{ SGAUX_GETNSCRNS, "SGAUX_GETNSCRNS", "", },
	{ 0, 0, },					/* must be last */
};

const struct sgigsc_cmd syssgi_cmd[] = {
	{ SGI_SYSID, "SGI_SYSID", "p", },
	{ SGI_RDUBLK, "SGI_RDUBLK", "ipi", },
	{ SGI_TUNE, "SGI_TUNE", "ip", },
	{ SGI_IDBG, "SGI_IDBG", },
	{ SGI_INVENT, "SGI_INVENT", "ipi", },
	{ SGI_RDNAME, "SGI_RDNAME", "iBi", },
	{ SGI_SETLED, "SGI_SETLED", },
	{ SGI_SETNVRAM, "SGI_SETNVRAM", "ss", },
	{ SGI_GETNVRAM, "SGI_GETNVRAM", "sp", },
	{ SGI_ENABLE_FTIMER, "SGI_ENABLE_FTIMER", },
	{ SGI_DISABLE_FTIMER, "SGI_DISABLE_FTIMER", },
	{ SGI_QUERY_FTIMER, "SGI_QUERY_FTIMER", },
	{ SGI_QUERY_CYCLECNTR, "SGI_QUERY_CYCLECNTR", },
#ifdef SGI_PROCSZ
	{ SGI_PROCSZ, "SGI_PROCSZ", "ipp", },
#endif
	{ SGI_SIGACTION, "SGI_SIGACTION", "ip(12pii)" },
	{ SGI_SIGPENDING, "SGI_SIGPENDING", },
	{ SGI_SIGPROCMASK, "SGI_SIGPROCMASK", },
	{ SGI_SIGSUSPEND, "SGI_SIGSUSPEND", },
	{ SGI_SETSID, "SGI_SETSID", },
	{ SGI_SETPGID, "SGI_SETPGID", },
	{ SGI_SYSCONF, "SGI_SYSCONF", },
	{ SGI_WAIT4, "SGI_WAIT4", },
	{ SGI_PATHCONF, "SGI_PATHCONF", },
	{ SGI_READB, "SGI_READB", },
	{ SGI_WRITEB, "SGI_WRITEB", },
	{ SGI_SETGROUPS, "SGI_SETGROUPS", },
	{ SGI_GETGROUPS, "SGI_GETGROUPS", },
	{ SGI_ENABLE_FASTCLOCK, "SGI_ENABLE_FASTCLOCK", },
	{ SGI_DISABLE_FASTCLOCK, "SGI_DISABLE_FASTCLOCK", },
	{ SGI_SETTIMEOFDAY, "SGI_SETTIMEOFDAY", },
	{ SGI_SETTIMETRIM, "SGI_SETTIMETRIM", },
	{ SGI_GETTIMETRIM, "SGI_GETTIMETRIM", },
#ifdef SGI_SPROFIL
	{ SGI_SPROFIL, "SGI_SPROFIL", },
#endif
#ifdef SGI_RUSAGE
	{ SGI_RUSAGE, "SGI_RUSAGE", },
#endif
#ifdef SGI_SIGSTACK
	{ SGI_SIGSTACK, "SGI_SIGSTACK", },
#endif
#ifdef SGI_SIGSTATUS
	{ SGI_SIGSTATUS, "SGI_SIGSTATUS", },
#endif
#ifdef SGI_NETPROC
	{ SGI_NETPROC, "SGI_NETPROC", },
#endif
#ifdef SGI_SIGALTSTACK
	{ SGI_SIGALTSTACK, "SGI_SIGALTSTACK", },
#endif
#ifdef SGI_BDFLUSHCNT
	{ SGI_BDFLUSHCNT, "SGI_BDFLUSHCNT", "i", },
#endif
#ifdef SGI_SSYNC
	{ SGI_SSYNC, "SGI_SSYNC", },
#endif
	{ 0, 0, },					/* must be last */
};

const struct sgigsc_cmd sysmips_cmd[] = {
	{ SETNAME, "SETNAME", "s", },
	{ STIME, "STIME", "i", },
	{ FLUSH_CACHE, "FLUSH_CACHE", "", },
	{ SMIPSSWPI, "SMIPSSWPI", "p(16ppii)" },
	{ MIPS_FPSIGINTR, "MIPS_FPSIGINTR", "i", },
	{ MIPS_FPU, "MIPS_FPU", },
	{ MIPS_FIXADE, "MIPS_FIXADE", "i", },
	{ 0, 0, },					/* must be last */
};

const char *regnames[] = {
	"zero",		/* r0 */
	"at",		/* r1 */
	"v0",		/* r2 */
	"v1",		/* r3 */
	"a0",		/* r4 */
	"a1",		/* r5 */
	"a2",		/* r6 */
	"a3",		/* r7 */
	"t0",		/* r8 */
	"t1",		/* r9 */
	"t2",		/* r10 */
	"t3",		/* r11 */
	"t4",		/* r12 */
	"t5",		/* r13 */
	"t6",		/* r14 */
	"t7",		/* r15 */
	"s0",		/* r16 */
	"s1",		/* r17 */
	"s2",		/* r18 */
	"s3",		/* r19 */
	"s4",		/* r20 */
	"s5",		/* r21 */
	"s6",		/* r22 */
	"s7",		/* r23 */
	"t8",		/* r24 */
	"t9",		/* r25 */
	"k0",		/* r26 */
	"k1",		/* r27 */
	"gp",		/* r28 */
	"sp",		/* r29 */
	"s8",		/* r30 */
	"ra",		/* r31 */
};

extern int open(const char *, int, ...);
extern int fcntl(int, int, ...);
extern int setlinebuf(FILE *);
extern volatile void exit(int);
extern void *malloc(size_t);
extern void free(void *);
extern char *ctime (const time_t *);
extern long strtol(const char *, const char **, int);
extern int strlen(const char *);
extern void bzero(void *, int);
extern int getopt(int, const char * const *, const char *);
extern char *optarg;
extern int optind, opterr;
extern int atoi(const char *);
extern char *sys_siglist[];

int syscallcount[last(syscalltab)];
int signalcount[NSIG];
int Cflag;		/* trace the child after a fork() */
int cflag;		/* print just count of syscalls */
int rflag;		/* print address of caller */
int tflag;		/* print time each syscall happens */
int verbose;		/* print verbose info before and after each syscall */
int trace_fork;		/* trace forked process as well as parent */
int entered;		/* 1 when we are between syscall entry and exit */
int fd;			/* file descriptor of proc we're tracing */
pid_t pid;		/* PID of process being traced */
off_t bufptr;		/* arg that must be remembered till end of syscall */
size_t buflen = BUFLEN;	/* buffer length for printing strings */
pid_t child_pids[20];	/* PIDs of children */
int forked;		/* how often we forked */

int ignsyscall[sizeof(syscalltab)/sizeof(syscalltab[0])];
int ignsignal[sizeof(signalname)/sizeof(signalname[0])];

FILE *fout;

void printnum(unsigned long n)
{
	if (n > 9)
		fprintf(fout, "0x");
	fprintf(fout, "%lX", n);
}

void printstr(const char *s, int len)
{
	fprintf(fout, "\"");
	while (len > 0 && *s) {
		switch (*s) {
		case '\n': fprintf(fout, "\\n"); break;
		case '\t': fprintf(fout, "\\t"); break;
		case '\r': fprintf(fout, "\\r"); break;
		case '\b': fprintf(fout, "\\b"); break;
		case '\f': fprintf(fout, "\\f"); break;
		case '\\': fprintf(fout, "\\\\"); break;
		default:
			if (*s < ' ' || *s >= 0177)
				fprintf(fout, "\\%03o", *s & 0377);
			else
				fprintf(fout, "%c", *s);
			break;
		}
		len--;
		s++;
	}
	fprintf(fout, "\"");
	if (len == 0)
		fprintf(fout, "...");
}

void printbuf(const char *s, int maxlen, int len)
{
	fprintf(fout, "\"");
	while (len > 0) {
		if (*s < ' ' || *s >= 0177)
			fprintf(fout, "\\%03o", *s & 0377);
		else
			fprintf(fout, "%c", *s);
		len--;
		maxlen--;
		s++;
	}
	fprintf(fout, "\"");
	if (maxlen > 0)
		fprintf(fout, "...");
}

void setmasks(int fildes, int set)
{
	long mask[SYSMASKLEN];
	long signalmask;
	int i;

	if (set)
		for (i = 0; i < SYSMASKLEN; i++)
			mask[i] = ~0L;
	else
		for (i = 0; i < SYSMASKLEN; i++)
			mask[i] = 0L;
	if (fcntl(fildes, DFCSENTRYMASK, mask) < 0) {
		perror("DFCSENTRYMASK");
		exit(1);
		/*NOTREACHED*/
	}
	if (fcntl(fildes, DFCSEXITMASK, mask) < 0) {
		perror("DFCSEXITMASK");
		exit(1);
		/*NOTREACHED*/
	}
	if (set)
		signalmask = ~0L;
	else
		signalmask = 0L;
	if (fcntl(fildes, DFCSMASK, &signalmask) < 0) {
		perror("DFCSMASK");
		exit(1);
		/*NOTREACHED*/
	}
}

void do_syscall_exit(unsigned long syscall, unsigned long *regs)
{
	if (!entered)	/* can happen when we attach to a running process */
		return;
	entered = 0;
	if (!cflag && !ignsyscall[syscall]) {
		if (regs[REG_ARG3] != 0) {
			fprintf(fout, " = -1 (%s)",
				strerror((int) regs[REG_SYSCALL]));
		} else {
			fprintf(fout, " = ");
			printnum(regs[REG_SYSCALL]);
			if (bufptr != 0) {
				int n, len;
				char *buf;

				buf = malloc(buflen);
				n = len = regs[REG_SYSCALL];
				if (n > buflen || n < 0)
					n = buflen;
				if (lseek(fd, bufptr, L_SET) >= 0 &&
				    (n = read(fd, buf, n)) >= 0) {
					fprintf(fout, " ");
					printbuf(buf, len, n);
				}
				free(buf);
			}
		}
		if (rflag)
			fprintf(fout, " called from 0x%lx", regs[REG_RETADDR]);
		fprintf(fout, "\n");
	}
	if (syscall == 2 && regs[REG_ARG3] == 0) {		/* fork() */
		int cfd;
		char debugpid[16];

		/* set the kid on its way */
		sprintf(debugpid, "/debug/%05d", regs[REG_SYSCALL]);
		if ((cfd = open(debugpid, O_RDONLY)) >= 0) {
			fflush(fout);
			if (trace_fork && (child_pids[forked] = fork()) >= 0) {
				if (child_pids[forked] == 0) {
					/* we're the child */
					forked = 0;
					close(fd);
					fd = cfd;
					pid = regs[REG_SYSCALL];
					setmasks(fd, 1);
					return;
				} else {
					/* we're the parent */
					forked++;
					close(cfd);
				}
			} else {
				int runoption;

				if (trace_fork)
					perror("fork");

				/* If -C flag given, trace the child and set
				 * parent on it's merry way.
				 */
				(void) fcntl(cfd, DFCSTOP, 0);
				if (Cflag) {
					Cflag = cfd;	/* exch. cfd and fd */
					cfd = fd;
					fd = Cflag;
					Cflag = 1;
					pid = regs[REG_SYSCALL];
					setmasks(fd, 1);/* have child stop */
				}
				setmasks(cfd, 0);
				runoption = CLEARNOSIG;
				if (fcntl(cfd, DFCRUN, &runoption) < 0)
					perror("DFCRUN");
				(void) close(cfd);
			}
		} else {
			perror(debugpid);
		}
	}
}

void print_args(const char *argdescr, const unsigned long *args, char endch)
{
	const char *sep = "";

	while (*argdescr != endch) {
		if (*argdescr == '(') {
			unsigned long *malloc_args;
			int nargs;

			fprintf(fout, "{");
			nargs = strtol(argdescr+1, &argdescr, 10);
			if ((malloc_args = malloc(nargs)) == 0 ||
			    lseek(fd, args[-1], L_SET) < 0 ||
			    read(fd, (char *) malloc_args, nargs) != nargs)
				fprintf(fout, "?");
			else
				print_args(argdescr, malloc_args, ')');
			fprintf(fout, "}");
			if (malloc_args)
				free(malloc_args);
			while (*argdescr++ != ')')
				;
		}
		if (*argdescr == 0)
			break;
		fprintf(fout, "%s", sep);
		switch (*argdescr) {
		case 'S':
		case 'G':
		case 'M': {
			const struct sgigsc_cmd *scp;

			switch (*argdescr) {
			case 'S':	scp = syssgi_cmd; break;
			case 'G':	scp = sgigsc_cmd; break;
			case 'M':	scp = sysmips_cmd; break;
			}
			while (scp->sc_name) {
				if (*args == scp->sc_cmd) {
					fprintf(fout, "%s", scp->sc_name);
					if (scp->sc_args != 0) {
						if (*scp->sc_args != 0) {
							fprintf(fout, ",");
							print_args(scp->sc_args,
								args+1, '\0');
						}
						return;
					}
					scp = 0;
					break;
				}
				scp++;
			}
			if (scp != 0)
				printnum(*args);
			}
			break;
		case 'P': {
			int n, i;
			unsigned long *buf;

			buf = (unsigned long *) malloc(buflen);
			n = args[-1] * sizeof(long);
			if (n > buflen || n < 0)
				n = buflen;
			if (lseek(fd, *args, L_SET) < 0 ||
			    (n = read(fd, (char *) buf, n)) < 0)
				printnum(*args);
			else {
				const char *s;

				fprintf(fout, "{");
				s = "";
				n /= sizeof(long);
				for (i = 0; i < n; i++) {
					fprintf(fout, "%s", s);
					s = ",";
					printnum(buf[i]);
				}
				if (args[-1] > n)
					fprintf(fout, ",...");
				fprintf(fout, "}");
			}
			}
			break;
		case 'B':
			bufptr = *args;
			/* fall through */
		case 'p':
		case 'i':
			printnum(*args);
			break;
		case 's': {
			int n;
			char *buf;

			buf = malloc(buflen);
			if (lseek(fd, *args, L_SET) < 0 ||
			    (n = read(fd, buf, buflen)) < 0)
				printnum(*args);
			else
				printstr(buf, n);
			free(buf);
			}
			break;
		case 'b': {
			int n, len;
			char *buf;

			buf = malloc(buflen);
			n = len = args[1];
			if (n > buflen || n < 0)
				n = buflen;
			if (lseek(fd, *args, L_SET) < 0 ||
			    (n = read(fd, buf, n)) < 0)
				printnum(*args);
			else
				printbuf(buf, len, n);
			free(buf);
			}
			break;
		case 'k': {
			int i;
			unsigned long val = *args;

			for (i = 0; i < last(sigextra); i++)
				if (val & (1 << (i+8)))
					fprintf(fout, "%s|", sigextra[i]);
			val &= 0xFF;
			if (val > 0 && val < last(signalname))
				fprintf(fout, "%s", signalname[val]);
			else
				printnum(val);
			}
			break;
		case 'I':
		case 'C':
		case 'X': {
			int val_off;
			unsigned long val;

			sep = "";
			val_off = 0;
			do {
				val = *args;
				switch (*argdescr) {
				case 'I':
					if (val_off == 0)
						val >>= 16;
					else
						val &= 0xFFFF;
					val_off += 2;
					break;
				case 'C':
					switch (val_off) {
					case 0: val >>= 24; break;
					case 1: val = (val >> 16) & 0xFF; break;
					case 2: val = (val >>  8) & 0xFF; break;
					case 3: val &= 0xFF; break;
					}
					/* fall through */
				case 'X':
					val_off++;
					break;
				}
				if (*argdescr++ != 'X') {
					fprintf(fout, "%s", sep);
					sep = ",";
					printnum(val);
				}
			} while (val_off < 4 && (*argdescr == 'I' || *argdescr == 'C' || *argdescr == 'X'));
			argdescr--;	/* we've gone too far */
			}
			break;
		}
		sep = ",";
		args++;
		argdescr++;
	}
}

void print_syscall(unsigned long syscall, const unsigned long *args, const char *proc_name)
{
	const struct syscalltab *sp;

	/* the 1000 should be SYSVoffset, but that doesn't work in Cypress */
	syscall -= 1000;
	if (syscall == 0) {
		/* indirect system call */
		print_syscall(*args, args + 1, proc_name);
		return;
	}
	syscallcount[syscall]++;
	if (cflag)
		return;
	if (ignsyscall[syscall])
		return;
	if (trace_fork)
		fprintf(fout, "[%05d-%s] ", pid, proc_name);
	if (tflag) {
		struct timeval t;
		struct timezone z;
			(void) gettimeofday(&t, &z);
		fprintf(fout, "%.8s.%06ld ", ctime(&t.tv_sec) + 11,
			t.tv_usec);
	}
	if (syscall > last(syscalltab)) {
		fprintf(fout, "syscall %ld, args 0x%lx 0x%lx 0x%lx\n",
			syscall+1000, args[0], args[1], args[2]);
		return;
	}
	sp = &syscalltab[syscall];
	if (sp->sys_indirect) {
		sp = &sp->sys_indirect[*args]; /* this had better be in range */
		args++;
	}
	bufptr = 0;
	fprintf(fout, "%s(", sp->sys_name);
	print_args(sp->sys_args, args, '\0');
	fprintf(fout, ")");
}

void do_syscall_entry(unsigned long syscall, const unsigned long *regs, const char *proc_name)
{
	unsigned long args[MAX_ARGS];
	int i;

	entered = 1;
	for (i = 0; i < 4; i++)
		args[i] = regs[i + REG_ARG0];
	bzero(args + REG_ARG0, (MAX_ARGS - 4) * sizeof(unsigned long));
	if (lseek(fd, regs[REG_SP] + 16, L_SET) == regs[REG_SP] + 16)
		read(fd, (char *) &args[4], (MAX_ARGS-4)*sizeof(unsigned long));
	print_syscall(regs[REG_SYSCALL], args, proc_name);
}

void do_signal(const struct procinfo *pi, const char *proc_name)
{
	int sig = pi->pi_whatstop;

	if (0 < sig && sig < NSIG)
		signalcount[sig]++;
	if (cflag)
		return;
	if (ignsignal[sig])
		return;
	if (trace_fork)
		fprintf(fout, "[%05d-%s] ", pid, proc_name);
	if (tflag) {
		struct timeval t;
		struct timezone z;

		(void) gettimeofday(&t, &z);
		fprintf(fout, "%.8s.%06ld ", ctime(&t.tv_sec) + 11, t.tv_usec);
	}
	if (sig <= 0 || sig >= NSIG)
		fprintf(fout, "- Unknown signal %d\n", sig);
	else
		fprintf(fout, "- %s (%d)\n", sys_siglist[sig], sig);
}

void pr_summary(void)
{
	int i;

	if (cflag) {
		for (i = 0; i < last(syscalltab); i++)
			if (syscallcount[i] != 0)
				fprintf(fout, "%8d %s\n", syscallcount[i],
					syscalltab[i].sys_name);
		for (i = 1; i < NSIG; i++)
			if (signalcount[i] != 0)
				fprintf(fout, "%8d %s\n", signalcount[i],
					signalname[i]);
	}
}

void catchint()
{
	if (fd >= 0) {
		int runoption;

		(void) fcntl(fd, DFCSTOP, 0);
		if (fcntl(fd, DFCWSTOP, 0) < 0) {
			perror("DFCWSTOP");
			exit(1);
			/*NOTREACHED*/
		}
		setmasks(fd, 0);
		runoption = CLEARNOSIG;
		if (fcntl(fd, DFCRUN, &runoption) < 0)
			perror("DFCRUN");
		(void) close(fd);
	}
	pr_summary();
	exit(0);
	/*NOTREACHED*/
}

void pr_verbose_info(struct procinfo *pi, unsigned long *regs)
{
	int i;

	fprintf(fout, "stat=%d; ppid=%d; wchan=%x; whystop=%d; whatstop=%d\n",
		pi->pi_stat, pi->pi_ppid, (unsigned) pi->pi_wchan,
		pi->pi_whystop, pi->pi_whatstop);
	for (i = 0; i < 32; i++) {
		fprintf(fout, "r%d/%s", i, regnames[i]);
		fprintf(fout, "%*s", 5 - strlen(regnames[i]) - (i>=10), " ");
		fprintf(fout, "%08lx", regs[i]);
		if ((i & 3) == 3)
			fprintf(fout, "\n");
		else
			fprintf(fout, "  ");
	}
}

void attachproc(void)
{
	char debugpid[16];

	signal(SIGINT, catchint);
	sprintf(debugpid, "/debug/%05d", pid);
	fd = open(debugpid, O_RDONLY);
	if (fd < 0) {
		perror("open");
		exit(1);
		/*NOTREACHED*/
	}
	if (fcntl(fd, DFCSTOP, 0) < 0) {
		perror("DFCSTOP");
		exit(1);
		/*NOTREACHED*/
	}
	setmasks(fd, 1);
}

void startchild(const char * /* const */ *commandv)
{
	char debugpid[16];
	unsigned long regs[NPTRC_REGS];

	signal(SIGINT, SIG_IGN);
	if ((pid = fork()) < 0) {
		perror("fork");
		exit(1);
		/*NOTREACHED*/
	}
	if (pid == 0) {
		if (fout != 0)		/* close parent's output file */
			fclose(fout);
		signal(SIGINT, SIG_DFL);
		kill(getpid(), SIGSTOP); /* stop so that parent has a chance */
		execv(*commandv, commandv);
		perror("exec");
		exit(1);
		/*NOTREACHED*/
	}
	while (waitpid(pid, 0, WUNTRACED) != pid)	/* wait for child */
		;
	sprintf(debugpid, "/debug/%05d", pid);
	fd = open(debugpid, O_RDONLY);
	if (fd < 0) {
		perror("open");
		exit(1);
		/*NOTREACHED*/
	}

	/* shouldn't really be here, but we need it now */
	if (verbose && fout == 0)
		fout = stdout;

	if (fcntl(fd, DFCSEXEC, 0) < 0) {
		perror("DFCSEXEC");
		exit(1);
		/*NOTREACHED*/
	}
	kill(pid, SIGCONT);		/* continue child */
	if (fcntl(fd, DFCWSTOP, 0) < 0) {
		perror("DFCWSTOP");
		exit(1);
		/*NOTREACHED*/
	}
	if (verbose) {
		struct procinfo procinfo;

		if (fcntl(fd, DFCGETREGS, regs) < 0) {
			perror("DFCGETREGS");
			exit(1);
			/*NOTREACHED*/
		}
		if (fcntl(fd, DFCGETPRINFO, &procinfo) < 0) {
			perror("DFCGETPRINFO");
			exit(1);
			/*NOTREACHED*/
		}
		pr_verbose_info(&procinfo, regs);
	}
	setmasks(fd, 1);
	return;
}

volatile void usage(const char *command)
{
	fprintf(stderr, "Usage: %s [-Ccftr] [-s length] [-o file] program [args]\n",
		command);
	fprintf(stderr, "   or: %s [-Ccftr] [-s length] [-o file] -p pid\n",
		command);
	exit(1);
	/*NOTREACHED*/
}

void alrm_catch()
{
}

void trace(void)
{
	int runoption;
	int do_alarm;

	do_alarm = isatty(fileno(fout));
	if (do_alarm)
		sigset(SIGALRM, alrm_catch);
	runoption = CLEARNOSIG;
	for (;;) {
		struct procinfo procinfo;
		user_t u;
		unsigned long regs[NPTRC_REGS];

		if (fcntl(fd, DFCRUN, &runoption) < 0) {
			perror("DFCRUN");
			exit(1);
			/*NOTREACHED*/
		}
		if (do_alarm)
			alarm(1);
		while (fcntl(fd, DFCWSTOP, 0) < 0) {
			if (errno == ENOENT)
				goto loop_exit;		/* double break */
			if (errno == EINTR) {
				fflush(fout);
				continue;
			}
			perror("DFCWSTOP");
			exit(1);
			/*NOTREACHED*/
		}
		if (do_alarm)
			alarm(0);
		if (fcntl(fd, DFCGETPRINFO, &procinfo) < 0) {
			perror("DFCGETPRINFO");
			exit(1);
			/*NOTREACHED*/
		}
		if (fcntl(fd, DFCGETU, &u) < 0) {
			perror("DFCGETPR");
			exit(1);
			/*NOTREACHED*/
		}
		if (fcntl(fd, DFCGETREGS, regs) < 0) {
			perror("DFCGETREGS");
			exit(1);
			/*NOTREACHED*/
		}
		if (verbose)
			pr_verbose_info(&procinfo, regs);

		switch (procinfo.pi_whystop) {
		case SYSENTRY:
			do_syscall_entry(procinfo.pi_whatstop, regs, u.u_comm);
			break;
		case SYSEXIT:
			do_syscall_exit(procinfo.pi_whatstop, regs);
			break;
		case SIGNALLED:
			do_signal(&procinfo, u.u_comm);
			break;
		}
	}
loop_exit:
	if (entered && !cflag)
		fprintf(fout, " = ?\n");
}

void ignore(char *name)
{
	int i;

	for (i = 0; i < sizeof(syscalltab)/sizeof(syscalltab[0]); i++)
		if (strcmp(syscalltab[i].sys_name, name) == 0) {
			ignsyscall[i] = 1;
			return;
		}
	for (i = 1; i < sizeof(signalname)/sizeof(signalname[0]); i++)
		if (strcmp(signalname[i], name) == 0) {
			ignsignal[i] = 1;
			return;
		}
}

volatile int main(int argc, const char * /* const */ *argv)
{
	int i;
	int exit_status = 0;

	if (argc <= 1)
		usage(argv[0]);
	while ((i = getopt(argc, argv, "Ccfi:o:p:s:rtv")) != EOF)
		switch (i) {
		case 'C':
			Cflag = 1;
			break;
		case 'c':
			cflag = 1;
			break;
		case 'i':
			ignore(optarg);
			break;
		case 'f':
			trace_fork = 1;
			break;
		case 'r':
			rflag = 1;
			break;
		case 'o':
			/* need to open before we fork, since it may fail */
			if ((fout = fopen(optarg, "w")) == 0) {
				perror(optarg);
				exit(1);
				/*NOTREACHED*/
			}
			break;
		case 'p':
			/* don't allow two -p otions */
			if (pid != 0 || (pid = atoi(optarg)) <= 0)
				usage(argv[0]);
			break;
		case 's':
			if ((buflen = atoi(optarg)) <= 0)
				usage(argv[0]);
			break;
		case 't':
			tflag = 1;
			break;
		case 'v':
			verbose = 1;
			break;
		default:
			usage(argv[0]);
		}

	if (pid == 0) {
		/* no PID given, demand extra args and try to start child */
		if (optind >= argc)
			usage(argv[0]);
		startchild(argv + optind);
	} else {
		/* PID given, refuse extra args and try to attach to the proc */
		if (optind < argc)
			usage(argv[0]);
		attachproc();
	}

	/* if no output file given, use stdout */
	if (fout == 0)
		fout = stdout;

	(void) fcntl(fileno(fout), F_SETFL, fcntl(fileno(fout), F_GETFL, 0) | FAPPEND);
#if 0	/* temp hack */
	if (isatty(fileno(fout)))
		setvbuf(fout, (char *) 0, _IOLBF, BUFSIZ);
	else
		setvbuf(fout, (char *) 0, _IOFBF, BUFSIZ);
#else
	setvbuf(fout, (char *) 0, _IOLBF, BUFSIZ);
#endif

	trace();

	pr_summary();

	while (forked > 0) {
		int status;
		
		pid = wait(&status);
		if (pid < 0) {
			perror("wait");
			exit(1);
			/*NOTREACHED*/
		}
		for (i = 0; i < forked; i++) {
			if (child_pids[i] == pid) {
				forked--;
				if (i != forked)
					child_pids[i] = child_pids[forked];
				exit_status |= status;
			}
		}
	}

	exit(exit_status);
	/*NOTREACHED*/
}
