/*
 * common_test.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.33 $
 * $Date: 1992/02/10 23:37:25 $
 */

/*
 * This code implements a ping-pong test for another protocol.
 * The same protocol code is used for both the client and the server.  
 *
 * It expects the definitions of following macros which describe
 * the lower protocol:
 *
 *	HOST_TYPE (e.g., ETHhost)
 *	INIT_FUNC (e.g., ethtest_init)
 *	TRACE_VAR (e.g., ethtestp)
 *	PROT_STRING (e.g., "eth")
 *
 * It also needs declarations for CLIENT and SERVER, e.g.:
 *	HOST_TYPE SERVER = { 0xC00C, 0x4558, 0x04d2 };  
 *	HOST_TYPE CLIENT = { 0xC00C, 0x4558, 0x2694 };  
 *
 * It also needs definitions for the following macros controlling the test:
 *	TRIPS  (number of round trips to make)
 *	TIMES  (number of tests for each length)
 *	DELAY  (number of seconds to delay between each test
 *		(and timeout for declaring failure))
 *
 * Define the macro TIME to do timing calculations.
 *
 * *Finally*, define an array of integers 'lens[]' with the lengths for the
 * tests:
 *
 *	static int lens[] = { 
 *		  1, 200, 400, 600, 800, 1000, 1200
 *	};
 *
 */

#include "xkernel.h"
#ifndef XKMACHKERNEL
#include "x_stdio.h"
#else
#include "assert.h"
#endif ! XKMACHKERNEL

static	HOST_TYPE 	myHost;
static	XObj 		myProtl;
static	XObj	clientDownSes;
static 	int 	count;
static 	int	sentMsgLength;
static	int	serverParam, clientParam, testsSpecified, myTestSpecified;
static	char	*serverString;
static	int	trips = TRIPS;
#ifdef RPCTEST
static xkern_return_t	serverCallDemuxDefault();
static int		tryCallDefault();
static xkern_return_t	(* serverCallDemux)() = serverCallDemuxDefault;
static int		(* tryCall)() = tryCallDefault;
#endif
#ifdef TIME
#  ifndef RPCTEST
static	XTime 	starttime;
#  endif
static 	void	subtime(
#ifdef __STDC__
			XTime *t1, XTime *t2, XTime *result
#endif
			);
#endif
#ifdef STREAM_TEST
static	int	receivedLength = 0;
#endif

#define FAILURE_LIMIT 2

#ifdef __STDC__

static	void 	client( void * );
static 	void 	server( void * );
static 	int	isServer( void );
static 	int	isClient( void );
static 	int	runTest( int );
static  void	testInit( void );

#else

static	void 	client();
static 	void 	server();
static 	int	isServer();
static 	int	isClient();
static 	int	runTest();
static  void	testInit();

#endif __STDC__

#ifdef RPCTEST

#ifdef __STDC__
static 	xkern_return_t	testCallDemux( XObj, XObj, Msg *, Msg * );
#else
static 	xkern_return_t	testCallDemux();
#endif __STDC__

#else

static 	int 	gotone;
static 	xmsg_handle_t	clientPushResult = XMSG_NULL_HANDLE;
static 	xmsg_handle_t	serverPushResult = XMSG_NULL_HANDLE;

#ifdef __STDC__
static 	xkern_return_t	test_serverdemux( XObj, XObj, Msg * );
static 	xkern_return_t	test_clientdemux( XObj, XObj, Msg * );
#else
static 	xkern_return_t	test_serverdemux();
static 	xkern_return_t	test_clientdemux();
#endif __STDC__

#endif

#ifdef XKMACHKERNEL
static int
sscanf1(str, format, a1)
char *str, format;
int *a1;
{
  int n;

  while (*str >= '0' && *str <= '9')
    *a1 = 10*(*a1) + (*str++ - '0');
  return(1);
}
#else
#define sscanf1 sscanf
#endif XKMACHKERNEL

static void
processOptions()
{
    int		i;
    char 	*arg;

#define argPrefix(str) (! strncmp(arg, str, strlen(str)))
#define argEq(str) (! strcmp(arg, str) )

    for (i=1; i < globalArgc; i++) {
	arg = globalArgv[i];
	if ( argEq("-s") ) {
	    serverParam = 1;
	} else if ( argPrefix("-c") ) {
	    clientParam = 1;
	    serverString = arg + 2;
	} else if ( argPrefix("-test") ) {
	    testsSpecified = 1;
	    arg += strlen("-test");
	    if ( argEq(PROT_STRING) ) {
		myTestSpecified = 1;
	    }
	} else if ( argPrefix("-trips=") ) {
	    sscanf1(arg + strlen("-trips="), "%d", &trips);
	}
    }
#undef argPrefix
#undef argEq    
}


int
INIT_FUNC( self )
    XObj self;
{
    XObj	llp;

    processOptions();
    if ( testsSpecified && ! myTestSpecified ) {
	xTrace1(prottest, TR_SOFT_ERRORS, "Parameters indicate %s test should not run",
		PROT_STRING);
	return 0;
    }
    printf("%s timing test\n", PROT_STRING);
    myProtl = self;
    llp = xGetDown(self, 0);
    if ( ! xIsProtocol(llp) ) {
	Kabort("Test protocol has no lower protocol");
    }
    xControl(xGetDown(self, 0), GETMYHOST, (char *)&myHost, sizeof(HOST_TYPE));
    /* 
     * Call the per-test initialization function which gives the test
     * the opportunity to override the default functions
     */
    testInit();
    if (isServer()) {
	evDetach( evSchedule(server, 0, 0) );
    } else if (isClient()) {
#ifdef CONCURRENCY	
	int	i;

	for ( i=0; i < CONCURRENCY; i++ )
#endif
	    evDetach( evSchedule(client, 0, 0) );
    } else {
	printf("%stest: I am neither server nor client\n", PROT_STRING);
    }
    return 0;
}


static int
isServer()
{
    if ( serverParam ) {
	return TRUE;
    }
    return ! bcmp((char *)&myHost, (char *)&SERVER, sizeof(HOST_TYPE));
}


static int
isClient()
{
    
    if ( clientParam ) {
	str2ipHost(&SERVER, serverString);
	CLIENT = myHost;
	return TRUE;
    }
    return ! bcmp((char *)&myHost, (char *)&CLIENT, sizeof(HOST_TYPE));
}


#ifndef CUSTOM_ASSIGN

static void
clientSetPart( p )
    Part *p;
{
    partInit(p, 1);
    partPush(*p, &SERVER);
}

static void
serverSetPart( p )
    Part *p;
{
    partInit(p, 1);
    partPush(*p, ANY_HOST);
}

#endif ! CUSTOM_ASSIGN


#ifdef SAVE_SERVER_SESSN
#  ifdef __STDC__
static xkern_return_t	saveServerSessn( XObj, XObj, XObj );
#  endif

static xkern_return_t
saveServerSessn( self, s, hlpType )
    XObj self, s, hlpType;
{
    xTrace1(prottest, TR_MAJOR_EVENTS,
	    "%s test program duplicates lower server session",
	    PROT_STRING);
    xDuplicate(s);
    return XK_SUCCESS;
}
#endif SAVE_SERVER_SESSN


static void
server( foo )
    VOID *foo;
{
    Part p;
    
    printf("I am the  server\n");
#ifdef RPCTEST
    myProtl->calldemux = testCallDemux;
#else
    myProtl->demux = test_serverdemux;
#endif
#ifdef SAVE_SERVER_SESSN
    myProtl->opendone = saveServerSessn;
#endif 
#ifdef CUSTOM_OPENDONE
    myProtl->opendone = customOpenDone;
#endif 
    serverSetPart(&p);
    if ( xOpenEnable(myProtl, myProtl, xGetDown(myProtl, 0), &p)
		== XK_FAILURE ) {
	printf("%s test server can't openenable lower protocol\n",
	       PROT_STRING);
    } else {
	printf("%s test server done with xopenenable\n", PROT_STRING);
    }
    return;
}


static void
client( foo )
    VOID *foo;
{
    Part	p[2];
    int 	lenindex;
    
#ifndef RPCTEST
    myProtl->demux = test_clientdemux;
#endif

    printf("I am the client\n");
    clientSetPart(p);
    if ( clientDownSes == 0 ) {
    	clientDownSes = xOpen(myProtl, myProtl, xGetDown(myProtl, 0), p);
	if ( clientDownSes == ERR_XOBJ ) {
	    printf("%s test: open failed!\n", PROT_STRING);
	    Kabort("End of test");
	    return;
	}
    }
#ifdef USE_CHECKSUM
    xControl(clientDownSes, UDP_ENABLE_CHECKSUM, NULL, 0);
#endif
#ifdef INF_LOOP
    for (lenindex = 0; ; lenindex++) {
	if (lenindex >= sizeof(lens)/sizeof(long)) {
	    lenindex = 0;
	}
#else
    for (lenindex = 0; lenindex < sizeof(lens)/sizeof(long); lenindex++) {
#endif INF_LOOP	
	sentMsgLength = lens[lenindex];
	if ( runTest(lens[lenindex]) ) {
	    break;
	}
    }
    printf("End of test\n");
    xClose(clientDownSes);
#ifdef XK_TEST_ABORT
    Kabort("test client aborts at end of test");
#endif
    xTrace0(prottest, TR_FULL_TRACE, "test client returns");
}




#ifdef RPCTEST


static int
tryCallDefault( sessn, times, length )
  XObj sessn;
  int times;
  int length;
{
    xkern_return_t ret_val;
    int i;
    Msg	savedMsg, request, reply;
    char *buf;
    int count = 0;
    
    msgConstructAllocate(&savedMsg, length, &buf);
    msgConstructEmpty(&reply);
    msgConstructEmpty(&request);
    for (i=0; i<times; i++) {
	msgAssign(&request, &savedMsg);
	ret_val = xCall(sessn, &request, &reply);
	xIfTrace(prottest, TR_MAJOR_EVENTS) {
	    putchar('.');
	    if (! (++count % 50)) {
		putchar('\n');
	    }
	}
	if( ret_val == XK_FAILURE ) {
	    printf( "RPC call error %d\n" , ret_val );
	    goto abort;
	}
	if (msgLen(&reply) != length) {
	    printf("Bizarre reply length.  Expected %d, received %d\n",
		   length, msgLen(&reply));
	    goto abort;
	}
	msgTruncate(&reply, 0);
    }
    return times;

abort:
    msgDestroy(&savedMsg);
    msgDestroy(&reply);
    msgDestroy(&request);
    return i;
}


/* 
 * RPC runTest
 */
static int
runTest( len )
    int len;
{
    int 	test;
    static int	noResponseCount = 0;
    static int	total = 0;
#ifdef TIME    
    XTime 	startTime, endTime, netTime;
#endif

    for (test = 0; test < TIMES; test++) {
	printf("Sending (%d) ...\n", ++total);
	count = 0;
#ifdef PROFILE
	startProfile();
#endif
#ifdef TIME
	xGetTime(&startTime);
#endif
	if ( (count = tryCall(clientDownSes, trips, len)) == trips ) {
#ifdef TIME
	    xGetTime(&endTime);
	    subtime(&startTime, &endTime, &netTime);
	    printf("\nlen = %4d, %d trips: %6d.%06d\n", 
		   len, trips, netTime.sec, netTime.usec);
#else
	    printf("\nlen = %4d, %d trips\n", len, trips);
#endif
	}
	if ( count == 0 ) {
	    if ( noResponseCount++ == FAILURE_LIMIT ) {
		printf("Server looks dead.  I'm outta here.\n");
		return 1;
	    }
	} else {
	    noResponseCount = 0;
	}
	Delay(DELAY * 1000);
    }
    return 0;
}


static xkern_return_t
serverCallDemuxDefault(self, lls, dg, rMsg)
    XObj self, lls;
    Msg *dg, *rMsg;
{
    msgAssign(rMsg, dg);
    return XK_SUCCESS;
}


static xkern_return_t
testCallDemux( self, lls, dg, rMsg )
    XObj self, lls;
    Msg *dg, *rMsg;
{
    static int count = 0;
    
    xIfTrace(prottest, TR_MAJOR_EVENTS) {
	putchar('.');
	if (! (++count % 50)) {
	    putchar('\n');
	}
    }
    msgAssign(rMsg, dg);
    return serverCallDemux(self, lls, dg, rMsg);
}


#else	/* ! RPCTEST */



static int
runTest( len )
    int len;
{
    Msg		savedMsg, msg;
    int 	test;
    char	*buf;
    static int	total = 0;
    static int	noResponseCount = 0;
    
    msgConstructAllocate(&savedMsg, len, &buf);
    for (test = 0; test < TIMES; test++) {
	msgConstructCopy(&msg, &savedMsg);
	printf("Sending (%d) ...\n", ++total);
	printf("msg length: %d\n", msgLen(&msg));
	count = 0;
#ifdef PROFILE
	startProfile();
#endif
#ifdef TIME
	xGetTime(&starttime);
#endif
	clientPushResult = xPush(clientDownSes, &msg);
	do {
	    gotone = 0;
	    Delay(DELAY * 1000);
	} while (gotone);
	if (count < trips) {
	    printf("Test failed after receiving %d packets\n", count);
#ifdef STREAM_TEST
	    receivedLength = 0;
#endif
	} 
	if ( count == 0 ) {
	    if ( noResponseCount++ == FAILURE_LIMIT ) {
		printf("Server looks dead.  I'm outta here.\n");
		return 1;
	    }
	} else {
	    noResponseCount = 0;
	}
	msgDestroy(&msg);
    }
    msgDestroy(&savedMsg); 
    return 0;
}


static xkern_return_t
test_serverdemux( self, lls, dg )
    XObj self, lls;
    Msg *dg;
{
    static int count = 0;
    
    xIfTrace(prottest, TR_MAJOR_EVENTS) {
	putchar('.');
	if (! (count++ % 50)) {
	    putchar('\n');
	}
    }
#ifdef CUSTOM_SERVER_DEMUX
    customServerDemux(self, lls, dg);
#endif CUSTOM_SERVER_DEMUX
    serverPushResult = xPush(lls, dg);
    return XK_SUCCESS;
}


#  ifdef STREAM_TEST


static xkern_return_t
test_clientdemux( self, lls, dg )
    XObj self, lls;
    Msg *dg;
{
#ifdef TIME
    XTime 	now, total;
#endif
    char	*buf;
    Msg		msgToPush;
    
    gotone = 1;

    xTrace1(prottest, TR_EVENTS, "R %d", msgLen(dg));
    receivedLength += msgLen(dg);
    xTrace1(prottest, TR_DETAILED, "total length = %d", receivedLength);
    if (receivedLength == sentMsgLength) {
	/*
	 * Entire response has been received.
	 * Send another message
	 */
	if (++count < trips) {
	    xIfTrace(prottest, TR_MAJOR_EVENTS) { 
		putchar('.');
		if (! (count % 50)) {
		    putchar('\n');
		}
	    } 
	    msgConstructAllocate(&msgToPush, receivedLength, &buf);
	    receivedLength = 0;
	    xTrace0(prottest, TR_EVENTS, "S");
	    xPush(lls, &msgToPush);
	    msgDestroy(&msgToPush);
	} else {
#ifdef TIME
	    xGetTime(&now);
	    subtime(&starttime, &now, &total);
	    printf("\nlen = %4d, %d trips: %6d.%06d\n", 
		   receivedLength, trips, total.sec, total.usec);
#else
	    printf("\nlen = %4d, %d trips\n", receivedLength, trips);
#endif
	    receivedLength = 0;
#ifdef PROFILE
	    endProfile();
#endif
	}
    }
    return XK_SUCCESS;
}


#  else /* ! STREAM_TEST */


static xkern_return_t
test_clientdemux( self, lls, dg )
    XObj self, lls;
    Msg *dg;
{
#ifdef TIME
    XTime now, total;
#endif
    gotone = 1;
    if ( ++count < trips ) {
	xIfTrace(prottest, TR_MAJOR_EVENTS) {
	    putchar('.');
	    if (! (count % 50)) {
		putchar('\n');
	    }
	}
#ifdef CUSTOM_CLIENT_DEMUX
	customClientDemux(self, lls, dg);
#endif CUSTOM_CLIENT_DEMUX
	clientPushResult = xPush(clientDownSes, dg);
    } else {
#ifdef TIME
	xGetTime(&now);
	subtime(&starttime, &now, &total);
	printf("\nlen = %4d, %d trips: %6d.%06d\n", 
		msgLen(dg), trips, total.sec, total.usec);
#else
	printf("\nlen = %4d, %d trips\n", msgLen(dg), trips);
#endif
	
#ifdef PROFILE
	endProfile();
#endif
    }
    return XK_SUCCESS;
}


#  endif STREAM_TEST



#endif RPCTEST




#ifdef TIME
static void
subtime(t1, t2, t3)
    XTime *t1, *t2, *t3;
{
    t3->sec = t2->sec - t1->sec;
    t3->usec = t2->usec - t1->usec;
    if (t3->usec < 0) {
	t3->usec += 1000000;
	t3->sec -= 1;
    }
}
#endif


