/*     
 * ip_control.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.12 $
 * $Date: 1992/02/05 20:13:49 $
 */

#include "xkernel.h"
#include "ip.h"
#include "ip_i.h"
#include "route.h"


#define IPHOSTLEN	sizeof(IPhost)

static char	errBuf[200];

/*
 * ip_controlsessn
 */
int
ipControlSessn(s, opcode, buf, len)
    XObj s;
    int opcode;
    char *buf;
    int len;
{
    SSTATE        *sstate;
    PSTATE        *pstate;
    IPheader      *hdr;
    
    xAssert(xIsSession(s));
    
    sstate = (SSTATE *)s->state;
    pstate = (PSTATE *)s->myprotl->state;
    
    hdr = &(sstate->hdr);
    switch (opcode) {
	
      case GETMYHOST :
	checkLen(len, IPHOSTLEN);
	*(IPhost *)buf = sstate->hdr.source;
	return IPHOSTLEN;
	
      case GETPEERHOST :
	checkLen(len, IPHOSTLEN);
	*(IPhost *)buf = sstate->hdr.dest;  
	return IPHOSTLEN;
	
      case GETMYHOSTCOUNT:
      case GETPEERHOSTCOUNT:
	checkLen(len, sizeof(int));
	*(int *)buf = 1;
	return sizeof(int);

      case GETMYPROTO :
      case GETPEERPROTO :
	checkLen(len, sizeof(long));
	*(long *)buf = sstate->hdr.prot;
	return sizeof(long);
	
      case GETMAXPACKET :
	checkLen(len, sizeof(int));
	*(int *)buf = IPMAXPACKET;
	return sizeof(int);
	
      case GETOPTPACKET :
	checkLen(len, sizeof(int));
	*(int *)buf = sstate->mtu - IPHLEN;
	return sizeof(int);
	
      case IP_MYNET :
	{
	    IPinterfaceinfo	*ifInfo;

	    checkLen(len, IPHOSTLEN);
	    xAssert(sstate->llp);
	    ifInfo = ipLookupIfInfo(pstate, sstate->llp);
	    if ( ifInfo == 0 ) {
		xError(sprintf(errBuf,
			       "IP can't find info for interface %x!",
			       sstate->llp));
		return -1;
	    }
	    IP_AND(*(IPhost *)buf, sstate->hdr.source, ifInfo->netmask);
	    return IPHOSTLEN;
	}
	
      case GETPARTICIPANTS:
	/* 
	 * Since we completely rewrite the participant when we open
	 * the lower session, we'll just construct the participants
	 */
	{
	    Part	*p = (Part *)buf;
	    IPhost	*h;

	    checkLen(len, sizeof(Part));
	    partInit(p, 2);
	    /* 
	     * Remote host
	     */
	    h = X_NEW(IPhost);
	    *h = sstate->dest;
	    partPush(p[0], h);
	    /* 
	     * Local host
	     */
	    h = X_NEW(IPhost);
	    *h = sstate->hdr.source;
	    partPush(p[1], h);
	    return sizeof(Part) * 2;
	}

      case IP_REDIRECT:
	return ipControlProtl(s->myprotl, opcode, buf, len);
	
      default : 
	xTrace0(ipp,3,"Unrecognized opcode -- forwarding");
	return xControl(xGetDown(s, 0), opcode, buf, len);
    }
}



/*
 * ip_controlprotl
 */
int
ipControlProtl(self, opcode, buf, len)
    XObj self;
    int opcode;
    char *buf;
    int len;
{
    PSTATE	*pstate;
    IPhost 	net, mask, gw, dest;
    route 	*rt;
    XObj  	interface;
    
    xAssert(xIsProtocol(self));
    pstate = (PSTATE *) self->state;
    
    switch (opcode) {
	
      case GETMYHOST :
	{
	    int i, n;
	    
	    checkLen(len, IPHOSTLEN);
	    n = len / IPHOSTLEN;
	    if ( n > pstate->n_interfaces ) {
		n = pstate->n_interfaces;
	    }
	    for ( i=0; i < n; i++ ) {
		((IPhost *)buf)[i] = pstate->interface[i].myipaddr;
	    }
	    return n * IPHOSTLEN;
	}
	
      case IP_REDIRECT :
	checkLen(len, 2*IPHOSTLEN);
	net = *(IPhost *)buf;
	ipNetMask(&mask, &net);
	gw = *(IPhost *)(buf + IPHOSTLEN);
	xTrace3(ipp, 4, "IP_REDIRECT : net = %s, mask = %s, gw = %s",
		ipHostStr(&net), ipHostStr(&mask), ipHostStr(&gw));
	/* find the interface to send to this gateway on */
	xTrace0(ipp,5,"IP_REDIRECT : looking up gateway interface");
	interface = ipGetInterface( pstate, &gw );
	rt_add(pstate, &net, &mask, &gw, -1, interface, RTDEFAULTTTL);
	return 0;

	/* test control ops - remove later */
      case IP_GETRTINFO :
	/* get route info for a given dest address :
	   in : IP host address 
	   out : route structure for this address
	   */
	checkLen(len, sizeof(route));
	dest = *(IPhost *)buf;
	rt = rt_get(&dest);
	*(route *)buf = *rt;
	rt_free(rt);
	return (sizeof(route));

      default:
	xTrace0(ipp,3,"Unrecognized opcode");
	{
	    int	i, res;

	    /* 
	     * Try forwarding this to all interface protocols
	     */
	    for ( i=0; i < pstate->n_interfaces; i++ ) {
		xTrace1(ipp, 5, "forwarding control op to interface %d", i);
		if ( (res = xControl(pstate->interface[i].ethprotl,
				     opcode, buf, len)) != -1 ) {
		    return res;
		}
	    }
	    return -1;
	}
    }
}


