/* 
 * message.h
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "nmessage.h"
typedef enum {
  MSG_CREATE, MSG_PUSH, MSG_POP, MSG_TOP
}    MSG_STACK_OP;

typedef struct stack_stack {
  REF ref;
  int size;
  MSG_STACK_OP last_op;
  char *base;
  char stack[8];
}           STACK_HDR, *STACK;

typedef struct msg {
  int top;
  STACK stack;
  NMSG data;
}   Msg;

/* default stack size */
#define MSG_SSIZE 128

Msg msgf_overwrite();
Msg msg_truncateright();
Msg msg_truncateleft();
Msg msgf_make_exist();
Msg msgf_make_new();
Msg msgf_make_allstack();
char *msgf_push();
int msgf_pop();
char *msgf_top();
Msg msgf_clone_stack();
void msgf_break();
Msg msgf_join();
char *msg_top_underflow();
char *msg_push_overflow();
extern Msg NULL_MSG;


/* are two messages identical in the strictest sense */
/* note this is not the same as having the same data */
#define msg_equal(m,n) (((m).top == (n).top) && ((m).stack == (n).stack) && ((m).data == (n).data))

#define msg_isnull(m) ((m).stack == (STACK)0)
#define msg_clear(m) {(m).stack = (STACK)0;(m).top = (-1); (m).data =(NMSG)0;}

/* save a copy of a message */
#define msg_save(m1,m2) { \
  (m1) = (m2); \
  if ((m1).stack) ((m1).stack->ref.ref)++;  \
  if ((m1).data)  nmsg_save((m1).data,(m2).data); \
}

/* return maximum stack size */
#define msg_max_stack(m) ((m).stack->size)

/* return stack size */
#define msg_stack_len(m) ((m).top + 1)

/* return data size */
#define msg_data_len(m) (nmsg_len((m).data))

/* return length of message */
#define msg_len(m) (msg_stack_len(m) + msg_data_len(m))

/* copy message into buffer */
#define msg_externalize(m,b) { \
	msg_peek(m,0,msg_len(m),b); \
	msg_free(m); \
}


#ifdef MSGOPT
/* lots of fast macros */

#define msg_chopright(NEW,M,WHERE) { \
  int MTRslen; \
  Msg MTRtemp; \
  MTRslen = msg_stack_len(M); \
  if ((msg_data_len(M) == 0) && (msg_len(M) > WHERE)) { \
    if ((M).stack->ref.ref == 1) { \
      (NEW) = (M); \
    } else { \
      msg_clone_stack(MTRtemp,M); \
      msg_free(M); \
      (NEW) = (MTRtemp); \
    } \
    (NEW).stack->size = (NEW).stack->size - (MTRslen - WHERE);\
    (NEW).stack->base = (NEW).stack->base - (MTRslen - WHERE);\
    (NEW).top = (NEW).top - (MTRslen - WHERE);\
    } else { \
    NEW = msg_truncateright(M,WHERE); \
    } \
}


#define msg_overwrite(NEW,M) { \
  if ((M).stack->ref.ref == 1) { \
    (NEW) = (M); \
  } else {  \
    (NEW) = msgf_overwrite(M); \
  } \
}

#define msg_make_exist(M, S, D, L, R) { \
  (M).top = -1; \
  (M).stack = (STACK)malloc(sizeof(STACK_HDR)+(S)-8); \
  (M).stack->ref.ref = 1; \
  (M).stack->ref.ptr = (char *)(M).stack; \
  (M).stack->ref.xfree = free; \
  (M).stack->last_op = MSG_CREATE; \
  (M).stack->size = (S); \
  (M).stack->base = (M).stack->stack + (S) -1; \
  nmsg_makecontig((M).data,(L),(D),(R)); \
}
#define msg_make_new(M, S, D, L) { \
  REF *MMNr; \
  (M).top = -1; \
  (M).stack = (STACK)malloc(sizeof(STACK_HDR)+(S)-8); \
  (M).stack->ref.ref = 1; \
  (M).stack->ref.ptr = (char *)(M).stack; \
  (M).stack->ref.xfree = free; \
  (M).stack->last_op = MSG_CREATE; \
  (M).stack->size = (S); \
  (M).stack->base = (M).stack->stack + (S) -1; \
  nmsg_makeref(MMNr,(D)); \
  MMNr->ref--; \
  nmsg_makecontig((M).data,(L),(D),MMNr); \
}

#define msg_make_allstack(M, S, D, L) { \
  (M).top = -1; \
  (M).stack = (STACK)malloc(sizeof(STACK_HDR)+(S)-8+(((L)+3)&~0x3)); \
  (M).stack->ref.ref = 1; \
  (M).stack->ref.ptr = (char *)(M).stack; \
  (M).stack->ref.xfree = free; \
  (M).stack->last_op = MSG_CREATE; \
  (M).stack->size = (S)+(((L)+3)&~0x3); \
  (M).stack->base = (M).stack->stack + (S)+(L) -1; \
  if ((L) != 0) bcopy((D),msg_push((M),(L)),(L)); \
  (M).data = 0; \
 }

/* free a message */
#define msg_free(m) { \
  if ((m).data) nmsg_free((m).data); \
  if (((m).stack) && (--((m).stack->ref.ref) == 0)) { \
    (m).stack->ref.xfree((m).stack->ref.ptr); \
    msg_clear(m); \
  } \
}

/* make room for a header on a message */
#define msg_push(m,len) ((((m).top = (m).top + (len)) < (m).stack->size) ? (m).stack->base - (m).top  : msg_push_overflow(&(m),(len)))

/* pop header off stack */
#define msg_pop(m,len)   ((((m).top = (m).top - (len)) >= -1) ? 0 :  msg_pop_underflow(&(m),(len)))

/* get top of stack */
#define msg_top(m,len)  (((m).top - (len) >= -1) ?  (m).stack->base - (m).top : msg_top_underflow(&(m),(len)))

#define msg_clone_stack(m,new)  { \
  (new).top  = (m).top; \
  nmsg_save((new).data,(m).data); \
  (new).stack = (STACK) malloc(sizeof(STACK_HDR)+ (m).stack->size - 8); \
  bcopy((m).stack,(new).stack,sizeof(STACK_HDR)+ (m).stack->size-8);\
  (new).stack->ref.ref = 1; \
  (new).stack->ref.ptr = (char *)(new).stack; \
  (new).stack->ref.xfree = free; \
}

#define msg_break(M, R, WHERE, SIZE) { \
  NMSG HHHH; \
  if ((WHERE) < msg_stack_len(M)) { \
    msg_real_break(&(M),&(R),(WHERE),(SIZE)); \
  } else { \
    if ((SIZE) == 0) { \
      msg_make_allstack((R),(M).stack->size,0,0); \
    } else { \
      msg_make_allstack((R),(SIZE),0,0); \
    } \
    HHHH = (M).data; \
    nmsg_break(&((M).data),&HHHH,(WHERE)- msg_stack_len(M)); \
    (R).data = HHHH; \
  } \
}

#define msg_join(NEW,M1,M2) { \
  Msg TTTT; \
  if (msg_isnull(M1)) { \
    (NEW) = (M2); \
  } else if (msg_isnull(M2)) { \
    (NEW) = (M1); \
  } else if (msg_stack_len(M2) != 0) { \
    msg_real_join(&(NEW),(M1),(M2)); \
  } else { \
    msg_save(TTTT,(M1)); \
    nmsg_save((M2).data,(M2).data); \
    (TTTT).data = nmsg_join((TTTT).data,(M2).data); \
    msg_free(M1); \
    msg_free(M2); \
    (NEW) = TTTT; \
  } \
}

#define msg_peek(m,offset,len,b) { \
  if ((offset)+(len)  <= msg_stack_len(m)) { \
    bcopy(msg_top((m),msg_stack_len(m))+(offset),(b),(len)); \
  } else { \
    msgf_peek((m),(offset),(len),(b)); \
  } \
}

#else

/* lots of slow procedure calls for debugging */

#define msg_chopright(NEW,M,WHERE) (NEW) = msg_truncateright(M,WHERE)

#define msg_peek(m,offset,len,b) (msgf_peek(m,offset,len,b))

#define msg_make_exist(M, S, D, L, R) { \
  (M) = msgf_make_exist(S,D,L,R); \
}

#define msg_make_new(M, S, D, L) { \
  (M) = msgf_make_new(S,D,L); \
}

#define msg_make_allstack(M, S, D, L) { \
  (M) = msgf_make_allstack(S,D,L); \
}

#define msg_free(m) { msgf_free(m); msg_clear(m);}

#define msg_push(m,len)  (msgf_push(&(m),len))

#define msg_pop(m,len)  (msgf_pop(&(m),len))

#define msg_top(m,len)  (msgf_top(&(m),len))

#define msg_clone_stack(m,new)  { \
  (new) = msgf_clone_stack(m); \
}

#define msg_break(M, R, WHERE, SIZE) { \
  msgf_break(&(M),&(R),WHERE,SIZE); \
}

#define msg_join(NEW,M1,M2) { \
  (NEW) = msgf_join(M1,M2); \
}

#define msg_overwrite(NEW,M) { \
    NEW = msgf_overwrite(M); \
}

#endif
