/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 * $Header: vfind.c[2.12] Mon Mar  9 20:48:24 1992 axel@cs.tu-berlin.de accessed $
 */

/*
 * Credits for Steve Emmerson (steve@unidata.ucar.edu) who provided
 * the primary ``SinceName''. [15/Aug/90]      uli@coma
 */

#ifndef lint
static char *AtFSid = "$Header: vfind.c[2.12] Mon Mar  9 20:48:24 1992 axel@cs.tu-berlin.de accessed $";
static char *Objfile = "vfind.c[2.12] accessed";
#ifdef CFFLGS
  static char *Cflags = CFFLGS;
#endif
#endif

#include <atfs.h>
#include <atfsapp.h>
#include <stdio.h>
#include <ParseArgs.h>

#define BLKSIZE 512		/* used in size */
#define VF_FIRST 0		/* used in position */
#define VF_LAST 1		/* dito. */
#define MAXDIRS 100		/* # of directories per nesting depth */

typedef struct expr *Expr;
struct expr {
  int (*eval)();		/* evaluator */
  Expr left;			/* left op */
  Expr right;			/* right op */
};

extern char *getwd();
extern char *malloc();
extern char *re_comp();

static char *progname;		/* av[0] w/o leading path */ 
static char startdir[MAXPATHLEN+1]; /* dir at beginning of process */
static char dirprefix[MAXPATHLEN+1]; /* current path name prefix */
static char msg[MAXPATHLEN+1];	/* for error messages */
static int nac;			/* ac after option parsing */
static char **nav;		/* av after option parsing */
static int npaths;		/* pathname list */
static struct expr *exprl;	/* expression list */
static int idx;			/* used as array index */
static time_t now;		/* time in secs */
static int Bflag, Cflag, Fflag, Xflag, Sflag;
static int Uflag, Attrflag, Pruneflag, Hitsflag;
static int maxdepth = 0;	/* max traverse depth iff -cut given */
static dev_t this_dev;		/* device # of starting point */
static int truehits;		/* # of expression yielding true */

/* Options */
static int vf_cut_opt (arg, opt)
     /* ARGSUSED */
     char *arg, *opt;
{
  Cflag++;
  if ((*opt) && !(maxdepth = atoi(opt)) && (*opt != '0')) {
    (void) fprintf(stderr, "%s: -cut: integer expected\n", progname);
    return 1;
  }
  return 0;
}

static int vf_version_opt (arg, opt)
     /*ARGSUSED*/
     char *arg, *opt;
{
  char *vfversion();

  (void) printf("This is %s version %s.\nAtFS toolkit lib version %s.\n\
AtFS version %s.\n",
		progname, vfversion(), at_version(), af_version());
  return 1;
}

static char helpmsg[] =
  "path-list expression\n\
  primary expressions are: atime, ctime, mtime, stime, ltime, exec,\n\
  exit, vl, name, perm, print, prune, symbolic, state, type, last, first,\n\
  uda, user, locked, locker, eq, lt, le, gt, ge, newer, SinceName, size.";

OptDesc optdesc[] = {
  { "version", PSWITCH, vf_version_opt, NULL, NULL }, /* print version id */
  { "bpool", PSWITCH|PSET, NULL, &Bflag, NULL }, /* examine binary pool */
  { "h", PFAIL, NULL, NULL, helpmsg }, /* print usage */
  { "cut", POARG, vf_cut_opt, NULL, NULL }, /* cut unix tree at spec. level */
  { "force", PSWITCH|PSET, NULL, &Fflag, NULL }, /* force examination */
						 /* of AtFS, even if it */
						 /* is a symbolic link */
  { "xdev", PSWITCH|PSET, NULL, &Xflag, NULL },	/* do not cross devices */
  { "hits", PSWITCH|PSET, NULL, &Hitsflag, NULL}, /* return # of hits */
  { (char *) NULL, NULL, NULL, NULL, NULL },
};

static
void byebye(i) int i; {
  (void) chdir("/tmp");		/* in case of -pg. */
  (void) af_cleanup();
  exit(i);
}

static
Sfunc_t catchsig(sig) int sig; {
  (void) af_cleanup();
  (void) signal(sig, SIG_DFL);
  (void) kill(getpid(), sig);
}

static
void Usage() {
  (void) fprintf(stderr, "%s: usage: %s path-list expression.\n",
		 progname , progname);
  byebye(1);
}

/* primaries */
static int noop () {		/* for performance tests only */
  return 1;
}

static int print (attrs, exp)
     Af_attrs *attrs;
     /*ARGSUSED*/
     Expr exp;
{
  (void) printf("%s%s\n", dirprefix, at_getbndvers(attrs, 1));
  return 1;
}

static int prune ()
{
  Pruneflag++;
  return 0;
}

static int terminate (attr, exp)
     Af_attrs *attr;
     Expr exp;
{
  byebye((int) exp->left);
}

static int name (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  char *cp;

  if (cp = re_comp((char *) exp->left)) {
    (void) fprintf(stderr, "%s: %s\n", progname, cp);
    byebye(1);
  }

  return re_exec(at_getfilename(attrs));
}

static int state (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return attrs->af_state == (int) exp->left;
}

static int uda (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return at_matchuda(attrs, (char *) exp->left);
}

static int symname (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  char symnameuda[AF_UDANAMLEN+1];

  (void) sprintf(symnameuda, "%s=%s", SYMNAME, (char *) exp->left);
  return at_matchuda(attrs, symnameuda);
}

static time_t gettime (val)
     char *val;
{
  Af_set set;
  Af_attrs attrs;
  char thisvers[MAXPATHLEN + MAXNAMLEN + MAXTYPLEN + 1];
  char *cp1, *cp2;
  int hits;

  (void) af_initattrs(&attrs); (void) af_initset(&set);
  attrs.af_gen = AF_BUSYVERS; attrs.af_rev = AF_BUSYVERS;
  (void) strcpy(thisvers,val);

  if ((cp1 = rindex(thisvers, '[')) && (cp2 = rindex(thisvers,']')) &&
      (cp1 < cp2) && (*(cp2+1) == '\0')) {
    if ((cp2 = index(cp1, '.')) && cp2 > cp1) {
      *cp1 = '\0'; cp1++; *cp2 = '\0'; cp2++;
      attrs.af_gen = atoi(cp1); attrs.af_rev = atoi(cp2);
    }
  }

  (void) strcpy(attrs.af_syspath, af_afpath(thisvers));
  (void) strcpy(attrs.af_name, af_afname(thisvers));
  (void) strcpy(attrs.af_type, af_aftype(thisvers));

  if (af_find(&attrs, &set) <= 0) {
    (void) fprintf (stderr, "%s: cannot access < %s >\n", progname, val);
    byebye(1);
  }

  hits = af_nrofkeys(&set);
  if (hits > 1) {
    (void) fprintf(stderr, "%s: ambigious < %s >\n", progname, val);
    byebye(1);
  }
  (void) af_gattrs(&(set.af_klist[hits-1]), &attrs);
  return (attrs.af_state == AF_BUSY) ? attrs.af_mtime : attrs.af_stime;
}

static int newer (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  if (attrs->af_state == AF_BUSY)
    return (attrs->af_mtime > (time_t) exp->left);
  else
    return (attrs->af_stime > (time_t) exp->left);
}

static int SinceName (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  int status = 0;		/* return status = no match */
  Af_set set;
  Af_attrs sattrs;
  char symnameuda[AF_UDANAMLEN+1];

  (void) af_initattrs(&sattrs);
  (void) strcpy(sattrs.af_host, attrs->af_host);
  (void) strcpy(sattrs.af_syspath, attrs->af_syspath);
  (void) strcpy(sattrs.af_name, attrs->af_name);
  (void) strcpy(sattrs.af_type, attrs->af_type);

  (void) sprintf(symnameuda, "%s=%s", SYMNAME, (char *) exp->left);
  sattrs.af_udattrs[0] = symnameuda;
  sattrs.af_udattrs[1] = 0;

  (void)af_initset(&set);
  if (af_find(&sattrs, &set) == -1) {
    (void) sprintf(msg, "%s: af_find()", progname);
    (void) af_perror(msg);
  } else {
    int hits = af_nrofkeys(&set);

    if ((hits = af_nrofkeys(&set)) > 1) {
      (void)fprintf(stderr, "%s: ambigious < %s[%s] >\n", progname, 
	attrs->af_name, (char*)exp->left);
    } else if (hits == 1) {
      (void)af_gattrs(&(set.af_klist[0]), &sattrs);
      status = sattrs.af_stime < (attrs->af_state == AF_BUSY ? 
	attrs->af_mtime : attrs->af_stime);
    }
  }

  (void)af_dropset(&set);

  return status;
}

static int position (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  Af_set tset;
  Af_attrs tattrs;
  Af_key tkey;
  int hits, i;

  (void) af_initattrs(&tattrs);
  (void) strcpy(tattrs.af_host, attrs->af_host);
  (void) strcpy(tattrs.af_syspath, attrs->af_syspath);
  (void) strcpy(tattrs.af_name, attrs->af_name);
  (void) strcpy(tattrs.af_type, attrs->af_type);

  if (Sflag)
    tattrs.af_state = attrs->af_state;
  if (Uflag) {
    char *p = attrs->af_owner.af_userdomain;
    if (*p == '.') p++;
    (void) strcpy(tattrs.af_owner.af_username, attrs->af_owner.af_username);
    (void) strcpy(tattrs.af_owner.af_userdomain, p);
  }
  if (Attrflag)
    for (i = 0; i < AF_MAXUDAS; i++)
      tattrs.af_udattrs[i] = attrs->af_udattrs[i];

  if (af_find(&tattrs, &tset) == -1) {
    (void) sprintf(msg, "%s: af_find()", progname);
    (void) af_perror(msg);
    return 0;
  }

  if ((hits = af_nrofkeys(&tset)) > 1) {
    (void) af_sortset(&tset, AF_ATTHUMAN);
    if (((int) exp->left) == VF_FIRST)
      (void) af_setgkey(&tset, 0, &tkey);
    else
      (void) af_setgkey(&tset, hits - 1, &tkey);
    (void) af_gattrs(&tkey, &tattrs);
    hits = ((tattrs.af_gen == attrs->af_gen) &&
	    (tattrs.af_rev == attrs->af_rev)) ? 1 : 0;
  }

  (void) af_dropset(&tset);
  return hits;
}

static int size (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  int s;

  s = attrs->af_size / BLKSIZE;
  if (attrs->af_size % BLKSIZE) s++;

  return (s == (int) exp->left);
}

static int perm (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return (attrs->af_mode & (int) exp->right & 07777) == (int) exp->left;
}

static int user (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  char *p = attrs->af_owner.af_userdomain;

  if (*p == '.') p++;
  return (!strcmp((char *)exp->left, attrs->af_owner.af_username)
	  && (exp->right ? !strcmp((char *)exp->right, p) : 1));
}

static int locked (attrs, exp)
     Af_attrs *attrs;
     /*ARGSUSED*/
     Expr exp;
{
  return (attrs->af_locker.af_username[0] != '\0');
}

static int locker (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  char *p = attrs->af_locker.af_userdomain;

  if (*p == '.') p++;
  return (!strcmp((char *)exp->left, attrs->af_locker.af_username)
	  && (exp->right ? !strcmp((char *)exp->right,
				   p) : 1));
}

static int type (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return (attrs->af_mode&S_IFMT) == (int) exp->left;
}

static int vl (attrs, exp)
     Af_attrs *attrs;
     /*ARGSUSED*/
     Expr exp;
{
  (void) printf("%s %s %s %8d %s %s%s\n",
	       at_getmode(attrs),
	       at_getversstate(attrs, 0),
	       at_getuser(&(attrs->af_owner)),
	       attrs->af_size,
	       at_getdate(attrs),
	       dirprefix,
	       at_getbndvers(attrs, 1));
  return 1;
}

static int eq (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return (attrs->af_gen == (int) exp->left) &&
    (attrs->af_rev == (int) exp->right);
}

static int lt (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return (attrs->af_gen < (int) exp->left ||
	  (attrs->af_gen == (int) exp->left &&
	   attrs->af_rev < (int) exp->right));
}

static int le (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return lt(attrs, exp) || eq(attrs, exp);
}

static int gt (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return (attrs->af_gen > (int) exp->left ||
	  (attrs->af_gen == (int) exp->left &&
	   attrs->af_rev > (int) exp->right));
}

static int ge (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return gt(attrs, exp) || eq(attrs, exp);
}


#define SECSPERDAY 86400L
static int comptime(secs, days, sign)
     time_t secs;
     int days;
     char sign;
{
  int d;

  d = (int) ((now - secs) / SECSPERDAY);
  switch (sign) {
  case '+': return (d>days);
  case '-': return (d < (days * -1));
  default: return (d==days);
  }
}

static int mtime (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return comptime(attrs->af_mtime, (int) exp->left, *(char*)(exp->right));
}

static int atime (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return comptime(attrs->af_atime, (int) exp->left, *(char*)(exp->right));
}

static int chngtime (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return comptime(attrs->af_ctime, (int) exp->left, *(char*)(exp->right));
}

static int stime (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return comptime(attrs->af_stime, (int) exp->left, *(char*)(exp->right));
}

static int ltime (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return comptime(attrs->af_ltime, (int) exp->left, *(char*)(exp->right));
}

static int execute (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  char *pav[40], *arg;
  int i, j;
  int pid, wpid;
  union wait retcode;

  i = (int) exp->left;
  for (j = 0; strcmp((arg = nav[i++]), ";"); j++)
    if (!strcmp(arg, "{}"))
      pav[j] = at_getbndvers(attrs, 0);
    else
      pav[j] = arg;

  pav[j] = (char *) NULL;
  if (j == 0) return 1;

  (void) fflush(stdout);
  switch (pid = fork()) {
  case -1:
    (void) sprintf(msg, "%s: fork failed\n", progname);
    perror(msg);
    byebye(1);
    break;
  case 0:
    (void) chdir(startdir);
    /*VARARGS*/
    (void) execvp(pav[0], pav);
    (void) perror("vfind: exec failed");
    (void) kill (getpid(), SIGHUP);
    _exit(127);
    break;
  default:
    while ((wpid = wait(&retcode)) != -1 && wpid != pid);
    if (wretcode(&retcode)) {
      (void) fprintf(stderr, "%s: execution terminated\n", progname);
      byebye(1);
    }
    return (!wretcode(&retcode));
    break;
  }
  
  /*NOTREACHED*/
  return 0;
}

static int not (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return ! ((*exp->left->eval) (attrs, exp->left));
}

static int and (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return ((*exp->left->eval) (attrs, exp->left) &&
	  (*exp->right->eval) (attrs, exp->right)) ? 1 : 0;
}

static int or (attrs, exp)
     Af_attrs *attrs;
     Expr exp;
{
  return ((*exp->left->eval) (attrs, exp->left) ||
	  (*exp->right->eval) (attrs, exp->right)) ? 1 : 0;
}

/* constructors */
static Expr build(func, l, r)
     int (*func)();
     Expr l, r;
{
  Expr this;
  if ((this = (Expr) malloc((unsigned) sizeof (struct expr))) ==
      (Expr) NULL) {
    (void) fprintf(stderr, "%s: out of memory.\n", progname);
    byebye(1);
  }

  this->eval = func; this->left = l; this->right = r;
  return this;
}

static void skip() {
  if (nav[idx])
    idx++;
  else {
    (void) fprintf(stderr, "%s: parsing error\n", progname);
    byebye(1);
  }
}

static Expr primary () {
  char *prim, *val, *cp;
  char c;
  int i = 0;
  int mode = 0;
  int mask = 0;

  val = cp = (char *) NULL;

  if (!(prim = nav[idx])) {
    (void) fprintf(stderr, "%s: parsing error\n", progname);
    byebye(1);
  }

  if (*prim != '-') {
    (void) fprintf(stderr, "%s: %s is not a primary\n", progname, prim);
    byebye(1);
  }
  prim++;
  if (!strcmp(prim, "noop"))
    return build(noop, (Expr) NULL, (Expr) NULL);
  if (!strcmp(prim, "print"))
    return build(print, (Expr) NULL, (Expr) NULL);
  if (!strcmp(prim, "prune"))
    return build(prune, (Expr) NULL, (Expr) NULL);
  if (!strcmp(prim, "ls") || !strcmp(prim, "vl"))
    return build(vl, (Expr) NULL, (Expr) NULL);
  if (!strcmp(prim, "locked"))
    return build(locked, (Expr) NULL, (Expr) NULL);
  if(!strcmp(prim, "last"))
    return build(position, (Expr) VF_LAST, (Expr) NULL);
  if(!strcmp(prim, "first"))
    return build(position, (Expr) VF_FIRST, (Expr) NULL);

  skip();
  if (!(val = nav[idx])) {
    (void) fprintf(stderr, "%s: parsing error\n", progname);
    byebye(1);
  }

  if (!strcmp(prim, "name"))
    return build(name, (Expr) at_mkedpat(val), (Expr) NULL);
  if (!strcmp(prim, "names"))
    return build(name, (Expr) at_mkedpat(val), (Expr) NULL);
  if (!strcmp(prim, "type")) {
    c = *val;
    i = (c=='b' ? S_IFBLK : c=='c' ? S_IFCHR :
	 c=='d' ? S_IFDIR : c=='f' ? S_IFREG :
#ifdef S_IFLNK
	 c=='l' ? S_IFLNK :
#endif
#ifdef S_IFSOCK
	 c=='s' ? S_IFSOCK :
#endif
	 0);
      return build(type, (Expr) i, (Expr) NULL);
  }
  if (!strcmp(prim, "perm")) {
    while(c = *val++)
      if (c=='-') mask++;
      else {
	c -= '0'; mode <<= 3; mode += c;
      }
    return build(perm, (Expr) mode, (Expr) (mask ? mode : 07777));
  }
  if (!strcmp(prim, "atime"))
    return build(atime, (Expr) atoi(val), (Expr) val);
  if (!strcmp(prim, "ctime"))
    return build(chngtime, (Expr) atoi(val), (Expr) val);
  if (!strcmp(prim, "mtime"))
    return build(mtime, (Expr) atoi(val), (Expr) val);
  if (!strcmp(prim, "stime"))
    return build(stime, (Expr) atoi(val), (Expr) val);
  if (!strcmp(prim, "ltime"))
    return build(ltime, (Expr) atoi(val), (Expr) val);
  if (!strcmp(prim, "user")) {
    Uflag++;
    if (cp = rindex(val, '@')) *cp++ = '\0';
    return build(user, (Expr) val, (Expr) cp);
  }
  if (!strcmp(prim, "exit"))
    return build(terminate, (Expr) atoi(val), (Expr) NULL);
  if (!strcmp(prim, "eq")) {
    if (cp = rindex(val, '.')) *cp++ = '\0';
    return build(eq, (Expr) atoi(val), (Expr) atoi(cp));
  }
  if (!strcmp(prim, "le")) {
    if (cp = rindex(val, '.')) *cp++ = '\0';
    return build(le, (Expr) atoi(val), (Expr) atoi(cp));
  }
  if (!strcmp(prim, "lt")) {
    if (cp = rindex(val, '.')) *cp++ = '\0';
    return build(lt, (Expr) atoi(val), (Expr) atoi(cp));
  }
  if (!strcmp(prim, "ge")) {
    if (cp = rindex(val, '.')) *cp++ = '\0';
    return build(ge, (Expr) atoi(val), (Expr) atoi(cp));
  }
  if (!strcmp(prim, "gt")) {
    if (cp = rindex(val, '.')) *cp++ = '\0';
    return build(gt, (Expr) atoi(val), (Expr) atoi(cp));
  }
  if (!strcmp(prim, "locker")) {
    if (cp = rindex(val, '@')) *cp++ = '\0';
    return build(locker, (Expr) val, (Expr) cp);
  }
  if (!strcmp(prim, "exec") || !strcmp(prim, "ok")) {
    i = idx;
    while(nav[++idx] && strcmp(nav[idx], ";"));
    return build(execute, (Expr) i, (Expr) prim);
  }
  if (!strcmp(prim, "state")) {
    Sflag++;
    return build(state, (Expr) at_string2state(val), (Expr) NULL);
  }
  if (!strcmp(prim, "uda")) {
    Attrflag++;
    return build(uda, (Expr) val, (Expr) NULL);
  }
  if (!strcmp(prim, "symbolic")) {
    Attrflag++;
    return build(symname, (Expr) val, (Expr) NULL);
  }
  if (!strcmp(prim, "newer"))
    return build(newer, (Expr) gettime(val), (Expr) NULL);
  if (!strcmp(prim, "SinceName")) {
    return build(SinceName, (Expr) val, (Expr) NULL);
  }
  if (!strcmp(prim, "size"))
    return build(size, (Expr) atoi(val), (Expr) NULL);
  (void) fprintf(stderr,"%s: unknown primary `%s'.\n", progname, prim);
  byebye(1);

  /*NOTREACHED*/
  return 0;
}

static Expr alternation();

static Expr grouping () {
  Expr expr;

  if (!strcmp(nav[idx], "(")){
    skip();
    expr = alternation();
    if (nav[idx] && !strcmp(nav[idx], ")")) {
      skip();
      return expr;
    }
    (void) fprintf(stderr, "%s: missing closing ')'.\n", progname);
    byebye(1);
  }
  else {
    expr = primary();
    skip();		/* primary() does not skip the */
				/* processed token, so we do it here. */
    return expr;
  }

  /*NOTREACHED*/
  return 0;
}

static Expr negation() {
  if (nav[idx] && !strcmp(nav[idx], "!")) {
    skip();
    return build(not, grouping(), (Expr) NULL);
  }
  else
    return grouping();
}

static Expr concatenation() {
  Expr left;
  char *this;

  left = negation ();
  this = nav[idx];
  if (this && (!strcmp(this, "-a") || !strcmp(this, "!") ||
	       !strcmp(this, "(") || (*this == '-' && strcmp(this, "-o")))){
    if (!strcmp(this, "-a")) skip();
    return build(and, left, concatenation());
  }
  else
    return left;
}

static Expr alternation() {
  Expr left;

  left = concatenation();
  if (nav[idx] && !strcmp(nav[idx], "-o")) {
    skip();
    return build(or, left, concatenation());
  }
  else
    return left;
}

static Expr expression() {
  return alternation();
}

static void traverse (name)
     char *name;
{
  Af_set set, bset;
  Af_key key;
  Af_attrs attrs;
  struct stat buf, abuf;
  int hits, i, ndir;
  char *prefix, thisaso[MAXPATHLEN+1];
  Af_attrs *dirs[MAXDIRS];
  static int ncalls = 0;

  if (af_access(af_afpath(name), af_afname(name), af_aftype(name),
		AF_SOURCE) == -1) {
    af_perror(name);
    return;
  }
    
  if (lstat(name, &buf) == -1)
    bzero((char *) &buf, sizeof(buf));

  prefix = dirprefix + strlen(dirprefix);
  ndir = 0;
  
  (void) af_initset(&set); (void) af_initattrs(&attrs);
  if ((buf.st_mode&S_IFMT) == S_IFDIR) {
    (void) strcpy(attrs.af_syspath, name);
    i = strlen(name) - 1;
    if (name[i] == '/' && i > 0)
      name[i] = '\0';
    (void) sprintf(prefix, "%s/", name);
  }
  else {
    (void) strcpy(attrs.af_name, af_afpath(name));
    (void) strcpy(attrs.af_name, af_afname(name));
    (void) strcpy(attrs.af_type, af_aftype(name));
  }

  /* check only busy versions if AtFS is a symblic link */
  if (!Fflag) {
    (void) sprintf(thisaso, "%s/%s",
		   ((buf.st_mode&S_IFMT) == S_IFDIR) ? name : af_afpath(name),
		   AF_SUBDIR);
    /* pay for downward compatibility */
    if (stat (thisaso, &abuf) < 0)
      (void) sprintf(thisaso, "%s/%s",
		   ((buf.st_mode&S_IFMT) == S_IFDIR) ? name : af_afpath(name),
		     AF_OLDSUBDIR);

    if ((lstat(thisaso, &abuf) != -1) && ((abuf.st_mode&S_IFMT) != S_IFDIR))
      attrs.af_state = AF_BUSY;
  }
  
  if (af_find(&attrs, &set) == -1) {
    (void) sprintf(msg, "%s: af_find(%s)", progname, name);
    (void) af_perror(msg);
    return;
  }

  if (Bflag) {
    (void) af_initset(&bset);
    if (af_bpfind(&attrs, &bset) == -1) {
      (void) sprintf(msg, "%s: bp_find(%s)", progname, name);
      (void) af_perror(msg);
      (void) af_dropset(&set);
      return;
    }
    (void) af_union(&set, &bset, &set); (void) af_dropset(&bset);
  }

  hits = af_nrofkeys(&set);
  (void) af_sortset(&set, AF_ATTHUMAN);
  for (i = 0; i < hits; i++) {
    if (af_setgkey(&set, i, &key) == -1) {
      (void) sprintf(msg, "%s: af_setgkey()", progname);
      (void) af_perror(msg);
      continue;
    }
    if (af_gattrs(&key,&attrs) == -1) {
      (void) sprintf(msg, "%s: af_gattrs():", progname);
      (void) af_perror(msg);
      continue;
    }
    (void) af_dropkey(&key);
    
    if ((attrs.af_mode&S_IFMT) == S_IFDIR) {
      if (ndir == MAXDIRS) {
	(void) fprintf(stderr, "%s: too many directories.\n", progname);
	byebye(1);
      }
      dirs[ndir] = (Af_attrs *) malloc((unsigned) sizeof(Af_attrs));
      if (dirs[ndir] == (Af_attrs *) NULL) {
	(void) fprintf(stderr, "%s: out of memory\n", progname);
	byebye(1);
      }
      bcopy((char *) &attrs, (char *) dirs[ndir++], sizeof(Af_attrs));
      continue;
    }
    if ((*exprl->eval)(&attrs, exprl))
      truehits++;
  }
  (void) af_dropset(&set);

  if (((buf.st_mode&S_IFMT) != S_IFDIR) ||
      (Xflag && (buf.st_dev != this_dev)) ||
      (Cflag && ncalls > maxdepth) ||
      !strcmp(name, AF_SUBDIR) ||
      !strcmp(name, AF_OLDSUBDIR))
    return;

  if (chdir(name) == -1)
    return;

  (void) sprintf(prefix, "%s/", name);

  for (i = 0; i < ndir; i++) {
    (void) strcpy(thisaso, at_getfilename(dirs[i]));
    if (!strcmp(thisaso, ".") || !strcmp(thisaso, "..") ||
	!strcmp(thisaso, AF_SUBDIR) ||
	!strcmp(thisaso, AF_OLDSUBDIR))
      continue;

    if ((*exprl->eval)(dirs[i], exprl))
      truehits++;
    (void) free((char *) dirs[i]);

    if (Pruneflag) {
      Pruneflag = 0;
      continue;
    }
    
    ncalls++;
    traverse(thisaso);
    ncalls--;
  }

  (void) chdir("..");
  *prefix = '\0';

  return;
}

main (ac, av)
     int ac;
     char *av[];
{
  char *cp;
  struct stat buf;
  
  progname = (cp = rindex(av[0], '/')) ? ++cp : av[0];
  if (ParseArgs(ac, av, &nac, &nav, optdesc)) byebye(1);
  (void) getwd(startdir); (void) time(&now);

  if (nac < 2) Usage();
  for (idx = 0; idx < nac; idx++) /* find path list */
    if (*nav[idx] == '-' ||  *nav[idx] == '!' || *nav[idx] == '(') break;
  if (!(npaths = idx)){
    (void) fprintf(stderr, "%s: no path list.\n", progname); Usage();
  }
  if (idx == nac) {
    (void) fprintf(stderr, "%s: no expression.\n", progname); Usage();
  }

  if ((exprl = expression()) == (Expr) NULL) byebye(1);

  (void) signal(SIGINT, catchsig);
  (void) signal(SIGBUS, catchsig);
  (void) signal(SIGSEGV, catchsig);
  (void) signal(SIGFPE, catchsig);
  (void) signal(SIGTERM, catchsig);

  for (idx = 0 ; idx < npaths; idx++) {	/* for all directories */
    (void) chdir(startdir);	/* back to normal */
    cp = nav[idx];
    
    if (Xflag) {
      (void) lstat(cp, &buf);
      this_dev = buf.st_dev;
    }

    *dirprefix = '\0';
    traverse(cp);
  }

  byebye(Hitsflag ? truehits: 0);
}
