/*LINTLIBRARY*/
/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	afudattrs.c -- manage user defined attributes
 *
 *	Author: Andreas Lampen, TU-Berlin
 *
 *
 *	$Header: afudattrs.c[1.4] Fri Jan 31 18:04:59 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *      afInitUdas -- Initialize Uda part in revision list
 *      afEnterUda -- enter user defined attribute
 *      afReplUda  -- replace user defined attribute
 *      afDelUda   -- delete user defined attribute
 *      afLookupUda -- search user defined attribute
 *      afMatchUda -- see if two user defines attributes "match"
 *      afListUdas -- create list of user defined attributes
 *      afDropUdas -- Delete all user defined attributes
 *      afCopyUdas -- Copy list of user defined attributes
 */

#include "afsys.h"
#include "atfs.h"

LOCAL af_fhash (htabsiz, hstr) /* from Aho/Ullman */
     int  htabsiz; 
     char *hstr;
{
  register char *p;
  register unsigned long hval, g;

  hval = 0;
  for (p = hstr; (*p!=AF_UDANAMDEL) && (*p!='\0'); p++)
    {
      hval = (hval << 4) ^ (*p);
      if (g = hval & 0xf0000000)
	{
	  hval = hval ^ (g >> 24);
	  hval = hval ^ g;
	}
    }
  return (hval % htabsiz);
}

/*=====================================================================
 * af_vallookup -- look for single value
 *
 *=====================================================================*/ 

LOCAL char *af_vallookup (entry, value)
     char *entry, *value; 
{
  register char *valptr;

  if (valptr = index (entry, AF_UDANAMDEL))
    valptr += sizeof (char);

  /* search until value is found */
  while (af_valcmp (valptr, value))
    {
      if ((valptr = index (valptr, AF_UDAVALDEL)) == (char *)0) 
	return (char *)0;
      valptr += sizeof (char);
    }

  return (valptr);
}

/*=====================================================================
 * afInitUdas -- Initialize Uda part in revision list
 *
 *=====================================================================*/ 

EXPORT afInitUdas (key)
     Af_key *key;
{
  int i;

  VATTR(key).af_udanum = 0;
  VATTR(key).af_uhtab = (Af_hashent *)0;
  for (i=0; i<AF_UDANUM; i++)
    VATTR(key).af_udalist[i] = (char *)0;
}

/*=====================================================================
 * afEnterUda -- enter user defined attribute
 *
 *=====================================================================*/ 

EXPORT afEnterUda (key, symbol)
     Af_key *key;
     char   *symbol;
{
  int  totalSize, i, symindex;
  char *symptr;
  Af_hashent *new, *curptr;

  /* allocate memory */
  if ((symptr = af_malloc (key->af_ldes,
			   (unsigned) (strlen (symbol) + sizeof (char))))
      == (char *)0)
    return (ERROR);
  (void) strcpy (symptr, symbol);

  /* if there is a hashtable */
  if (VATTR(key).af_uhtab != (Af_hashent *)0)
    {
      symindex = af_fhash (AF_UDAHASHSIZE, symbol);
      if (!VATTR(key).af_uhtab[symindex].symbol)  /* entry is not used yet */
	{
	  VATTR(key).af_uhtab[symindex].symbol = symptr;
	  VATTR(key).af_uhtab[symindex].next = NULL;
	}
      else /* collision! */
	{
	  if ((new = (Af_hashent*)af_malloc (key->af_ldes,
					     (unsigned) sizeof (Af_hashent)))
	      == (Af_hashent*)0)
	    return (ERROR);
	  new->symbol = symptr;
	  new->next = NULL;
	  curptr = &(VATTR(key).af_uhtab[symindex]);
	  while (curptr->next)
	    curptr = curptr->next;
	  curptr->next = new;
	}
    }
  else
    {
      /* if linear list of user defined attributes is full */
      if (VATTR(key).af_udanum >= AF_UDANUM)
	{
	  /* initialize hashtable */
	  totalSize = AF_UDAHASHSIZE * sizeof(Af_hashent);
	  if ((VATTR(key).af_uhtab =
	       (Af_hashent *)af_malloc (key->af_ldes, (unsigned) totalSize))
	      == NULL)
	    return (ERROR);
	  bzero ((char *)VATTR(key).af_uhtab, totalSize);
	  for (i=0; i<AF_UDANUM; i++)
	    {
	      if (VATTR(key).af_udalist[i] == (char *)0)
		continue;
	      (void) afEnterUda (key, VATTR(key).af_udalist[i]);
	    }
	  (void) afEnterUda (key, symbol);
	}
      else /* linear list is not full */
	{
	  i=0;
	  while (VATTR(key).af_udalist[i]) i++;
	  VATTR(key).af_udalist[i] = symptr;
	}
    }
  VATTR(key).af_udanum += 1;
  return (AF_OK);
}

/*=====================================================================
 * afReplUda  -- replace user defined attribute
 *
 *=====================================================================*/ 

EXPORT afReplUda (key, symbol)
     Af_key *key;
     char   *symbol;
{
  int i, symindex;
  Af_hashent *entry;

  /* if there is a hashtable */
  if (VATTR(key).af_uhtab != (Af_hashent *)0)
    {
      symindex = af_fhash (AF_UDAHASHSIZE, symbol);
      if (VATTR(key).af_uhtab[symindex].symbol)  /* found something */
	{
	  if (!af_symcmp (VATTR(key).af_uhtab[symindex].symbol, symbol))
	    /* found it ? */
	    { 
	      if ((VATTR(key).af_uhtab[symindex].symbol =
		   af_realloc (key->af_ldes,
			       VATTR(key).af_uhtab[symindex].symbol,
			       (unsigned) (strlen (symbol) + sizeof (char))))
		  == (char *)0)
		return (ERROR);
	      (void) strcpy (VATTR(key).af_uhtab[symindex].symbol, symbol);
	      return (AF_OK);
	    }
	  else /* maybe it is somewhere down the gully */ 
	    { 
	      entry = &(VATTR(key).af_uhtab[symindex]);
	      while (entry->next)
		{
		  entry = entry->next;
		  if (!af_symcmp (entry->symbol, symbol))
		    {
		      if ((entry->symbol = 
			   af_realloc (key->af_ldes, entry->symbol,
				 (unsigned) (strlen (symbol) + sizeof (char))))
			  == (char *)0)
			return (ERROR);
		      (void) strcpy (entry->symbol, symbol);
		      return (AF_OK);
		    } /* if */
		} /* while */
	    } /* else */
	} /* if */
    }
  else
    {
      for (i=0; i<AF_UDANUM; i++)
	{
	  if (VATTR(key).af_udalist[i] == (char *)0)
	    continue;
	  if (!af_symcmp (VATTR(key).af_udalist[i], symbol))
	    {
	      if ((VATTR(key).af_udalist[i] =
		   af_realloc (key->af_ldes, VATTR(key).af_udalist[i],
			       (unsigned) (strlen (symbol) + sizeof (char))))
		  == (char *)0)
		return (ERROR);
	      (void) strcpy (VATTR(key).af_udalist[i], symbol);
	      return (AF_OK);
	    }
	}
    }
  FAIL ("ReplUda", "user defined attribute not found", AF_EINTERNAL, ERROR);
}

/*=====================================================================
 * afDelUda   -- delete user defined attribute
 *
 *=====================================================================*/ 

EXPORT afDelUda (key, symbol)
     Af_key *key;
     char   *symbol;
{
  int i, symindex;
  Af_hashent *del, *succ;

  /* if there is a hashtable */
  if (VATTR(key).af_uhtab != (Af_hashent *)0)
    {
      symindex = af_fhash (AF_UDAHASHSIZE, symbol);
      if (!VATTR(key).af_uhtab[symindex].symbol)
	FAIL ("DelUda", "symbol not found", AF_EINTERNAL, ERROR); 
      del = &(VATTR(key).af_uhtab[symindex]);
      if (!af_symcmp (del->symbol, symbol))  /* found it ? */
	{
	  (void) af_free (key->af_ldes, del->symbol);
	  if (del->next) /* if there are more entries */
	    {
	      del->symbol = del->next->symbol;
	      succ = del->next->next;
	      (void) af_free (key->af_ldes, (char *)del->next);
	      del->next = succ;
	    }
	  else
	    {
	      del->symbol = (char *)0;
	      del->next = (Af_hashent *)0;
	    }
	  VATTR(key).af_udanum -= 1;
	  return (AF_OK);
	}
      else /* search for entry in gully */
	{
	  while (del->next)
	    {
	      if (!af_symcmp (del->next->symbol, symbol))
		{	  
		  (void) af_free (key->af_ldes, del->next->symbol);
		  succ = del->next->next;
		  (void) af_free (key->af_ldes, (char *)del->next);
		  del->next = succ;
		  VATTR(key).af_udanum -= 1;
		  return (AF_OK);
		}
	      del = del->next;
	    }
	}
    }
  else
    {
      for (i=0; i<AF_UDANUM; i++)
	{
	  if (VATTR(key).af_udalist[i] == (char *)0)
	    continue;
	  if (!af_symcmp (VATTR(key).af_udalist[i], symbol))
	    {
	      af_free (key->af_ldes, VATTR(key).af_udalist[i]);
	      VATTR(key).af_udalist[i] = (char *)0;
	      VATTR(key).af_udanum -= 1;
	      return (AF_OK);
	    }
	}
    }
  FAIL ("DelUda", "user defined attribute not found", AF_EINTERNAL, ERROR);
}

/*=====================================================================
 * afLookupUda -- search user defined attribute
 *
 *=====================================================================*/ 

EXPORT char *afLookupUda (key, symbol)
     Af_key *key; 
     char   *symbol; 
{
  int i, symindex;
  Af_hashent *targ;

  /* if there is a hashtable */
  if (VATTR(key).af_uhtab != (Af_hashent *)0)
    {
      symindex = af_fhash (AF_UDAHASHSIZE, symbol);
      if (VATTR(key).af_uhtab[symindex].symbol)
	{
	  if (!af_symcmp (VATTR(key).af_uhtab[symindex].symbol, symbol))
	    return (VATTR(key).af_uhtab[symindex].symbol);
	  else
	    { 
	      targ = &(VATTR(key).af_uhtab[symindex]);
	      while (targ->next)
		{
		  targ = targ->next;
		  if (!af_symcmp (targ->symbol, symbol))
		    return (targ->symbol);
		}
	    }
	}
    }
  else
    {
      for (i=0; i<AF_UDANUM; i++)
	{
	  if (VATTR(key).af_udalist[i] == (char *)0)
	    continue;
	  if (!af_symcmp (VATTR(key).af_udalist[i], symbol))
	    return (VATTR(key).af_udalist[i]);
	}
    }
  return ((char *)0);
}

/*====================================================================
 *    afMatchUda
 *    returnes FALSE if entry matches, else TRUE
 *
 *====================================================================*/

EXPORT afMatchUda (key, entry)
     Af_key *key;
     char   *entry;
{
  char *udaptr, *valptr;

  /* if user defined attribute does not exist */
  if ((udaptr = afLookupUda (key, entry)) == (char*)0)
    return (ERROR);

  /* else compare values */
  if (valptr = index (entry, AF_UDANAMDEL))
    {
      do
	{
	  valptr = valptr + sizeof (char);
	  if ((af_vallookup (udaptr, valptr) == (char *)0) && (*valptr !='\0'))
	    return (ERROR);
	}
      while (valptr = index (valptr, AF_UDAVALDEL));
    }
  return (AF_OK);
}

/*=====================================================================
 * afListUdas -- create list of user defined attributes
 *               --> Pointerlist terminated by (char *)0
 *
 *=====================================================================*/ 

EXPORT afListUdas (key, symbollist)
     Af_key *key;
     char   **symbollist;
{
  int i, j=0;
  Af_hashent *h;

  /* if there is a hashtable */
  if (VATTR(key).af_uhtab != (Af_hashent *)0)
    {
      for (i = 0; i < AF_UDAHASHSIZE; i++)
	if (VATTR(key).af_uhtab[i].symbol != (char *)0)
	  {
	    h = &(VATTR(key).af_uhtab[i]);
	    while (h) 
	      {
		symbollist[j] = h->symbol;
		j++;
		h = h->next;
	      }
	  }
    }
  else
    {
      for (i=0; i<AF_UDANUM; i++)
	{
	  if (VATTR(key).af_udalist[i] == (char *)0)
	    continue;
	  symbollist[j] = VATTR(key).af_udalist[i];
	  j++;
	}
    }
  symbollist[j] = (char *)0;
  if (j==0)
    return (j);
  else
    return (j-1);
}

/*=====================================================================
 * afDropUdas -- Drop list of user defined attributes
 *
 *=====================================================================*/ 

EXPORT afDropUdas (key)
     Af_key *key;
{
  int i;
  Af_hashent *entry;

  /* if there is a hashtable */
  if (VATTR(key).af_uhtab != (Af_hashent *)0)
    {
      for (i=0; i<AF_UDAHASHSIZE; i++)
	{
	  if (VATTR(key).af_uhtab[i].next)
	    {
	      entry = VATTR(key).af_uhtab[i].next;
	      do
		{
		  (void) af_free (key->af_ldes, entry->symbol);
		  (void) af_free (key->af_ldes, (char *)entry); 
		}
	      while (entry = entry->next);
	    }
	  if (VATTR(key).af_uhtab[i].symbol)
	    (void) af_free (key->af_ldes, VATTR(key).af_uhtab[i].symbol);
	}
      (void) af_free (key->af_ldes, (char *)VATTR(key).af_uhtab);
    }
  else
    {
      for (i=0; i<AF_UDANUM; i++)
	{
	  if (VATTR(key).af_udalist[i] == (char *)0)
	    continue;
	  af_free (key->af_ldes, VATTR(key).af_udalist[i]);
	}
    }
  VATTR(key).af_udanum = 0;
  return (AF_OK);
}

/*=====================================================================
 * afCopyUdas -- Copy list of user defined attributes
 *
 *=====================================================================*/ 

EXPORT afCopyUdas (srckey, destkey)
     Af_key *srckey, *destkey;
{
  int i;
  char *udalist[AF_MAXUDAS+1];

  /* copy hashtable */
  (void) afListUdas (srckey, udalist);
  i=0;
  while (udalist[i] != (char *)0)
    {
      (void) afEnterUda (destkey, udalist[i]);
      i++;
    }
  VATTR(destkey).af_udanum = VATTR(srckey).af_udanum;
  return (AF_OK);
}

/*===============================================================
 * af_symcmp -- compare contents of hashtable entry
 *           Return values like strcmp.
 *
 *==============================================================*/

LOCAL af_symcmp (str1, str)
     char *str1, *str;
{
  while (*str1 == *str)
    {
      if ((*str1 == AF_UDANAMDEL) || (*str1 == '\0'))
	return (0);
      str1++;
      str++;
    }
  if (((*str1 == AF_UDANAMDEL) && (*str =='\0')) ||
      ((*str1 =='\0') && (*str == AF_UDANAMDEL)))
    return (0);
  else
    return (*str1 - *str);
}

/*===============================================================
 * af_valcmp -- compare single values in a string of the form
 *                          name=[value1 value2 ...]
 *           Return values like strcmp.
 *
 *==============================================================*/

LOCAL af_valcmp (str1, str2)
     char *str1, *str2;
{
  while (*str1 == *str2)
    {
      if ((*str1 == AF_UDAVALDEL) || (*str1 == '\0'))
	return (0);
      str1++;
      str2++;
    }
  if (((*str1 == AF_UDAVALDEL) && (*str2 =='\0')) ||
      ((*str1 =='\0') && (*str2 == AF_UDAVALDEL)))
    return (0);
  else
    return (*str1 - *str2);
}


/**** DEBUG **** DEBUG **** DEBUG **** DEBUG **** DEBUG **** DEBUG ****/

/*=====================================================================
 * af_dumphtb -- dump hashtable
 *
 *=====================================================================*/ 

EXPORT af_dumphtb (htab) /* for debug purposes */
     Af_hashent *htab;
{
  register int i;
  register Af_hashent *h;

  for (i = 0; i < AF_UDAHASHSIZE; i++)
    {
      if (htab[i].symbol[0])
	{
	  h = &htab[i];
	  while (h) 
	    {
	      fprintf (stderr, "\nsymbol: (%d) %s", i, h->symbol);
	      h = h->next;
	    }
	}
      else fprintf (stderr, ".");
    }
}


