/*
 *      $__copyright$
 *
 *	Shape/AtFS
 *
 *	afsys.h - Internal type and Constant-Definitions for the 
 *		  Attribute-Filesystem
 *
 *	Author:	Andreas Lampen (andy@coma.UUCP
 *				andy@db0tui62.BITNET)
 *
 *	$Header: afsys.h[1.12] Wed Apr 22 17:43:01 1992 andy@cs.tu-berlin.de accessed $
 */

#ifndef _ATFSYS_
#define _ATFSYS_

/*=========================================================================
 * Installation dependent constants 
 *=========================================================================*/

#include "config.h"


#define AF_SYMHASHSIZE 507       /* size of hashtable for symbols */
#define AF_UDAHASHSIZE  61       /* size of hashtable for user defined attrs */
#define AF_UDANUM        8       /* number of user defined attributes in */
                                 /* linear list */
#define AF_SEGLEN       32       /* size of segment for (re-)allocation */

/*=========================================================================
 * general constants 
 *=========================================================================*/

#ifndef TRUE
#define TRUE		1
#endif
#ifndef FALSE
#define FALSE		0
#endif

#define ERROR		-1
#define LOCAL		static
#define EXPORT
#define bool		short

/*=========================================================================
 * Hash stuff
 *=========================================================================*/

typedef struct Af_hshent Af_hashent;

/*=========================================================================
 * Internal Type Definitions
 *=========================================================================*/

/**** buffer for version-independent attributes ****/
typedef struct {
		char	*af_host;               /* hostname */  
		char	*af_syspath;    	/* system path */
		char	*af_globname;		/* global filename */
		char	*af_globtype;		/* global filename ext.(type)*/
		char	*af_ownname;		/* name of owner */
		char	*af_ownhost;		/* host of owner */
		char	*af_owndomain;		/* domain of owner */
	       } Af_cattrs;

/**** buffer for version-dependent attributes ****/
typedef struct {
		char	*af_name;	        /* filename */
		char	*af_type;	        /* filename extension (type) */
		int	af_gen;			/* generation number */
		int	af_rev;			/* revision number */
		short	af_state;		/* version state (see below) */
		short   af_class;               /* file class */
		char 	*af_auname;		/* name of author */
		char    *af_auhost;		/* host of author */
		char    *af_audomain;		/* domain of author */
		u_short af_mode;		/* protection (from inode) */
		char    *af_lckname;            /* name of locker */
		char    *af_lckhost;            /* host of locker */
		char    *af_lckdomain;          /* domain of locker */
		time_t	af_mtime;		/* date of last modification */
		time_t	af_atime;		/* date of last access */
		time_t	af_ctime;		/* date of last status change*/
		time_t	af_stime;		/* save date */
		time_t	af_ltime;		/* date of last lock change */
		off_t	af_notesize;		/* size of note */
		char	*af_note;		/* modification note */
		int     af_udanum;              /* number of uda entries */
		Af_hashent *af_uhtab;           /* hash table for udefattrs */
		char    *af_udalist[AF_UDANUM]; /* list of udattrs if # <= 4 */
		short	af_repr;		/* kind of representation */
		off_t	af_fsize;		/* size of file */
		off_t	af_dsize;		/* size of delta */
		char	*af_data;		/* ptr to chunk or delta */
		char    *af_hashname;           /* name of associated file */
		short   af_nrefs;               /* # of references to rev. */
		int	af_succgen,		/* physical 	  */
			af_succrev;		/*	successor */
		int	af_predgen,		/* physical 	    */
			af_predrev;		/*	predecessor */
	       } Af_vattrs;

/**** Descriptor for revision list ****/
typedef struct rvlist Af_revlist;

struct rvlist {
                char	  *af_arfilename;    /* filename of archive */
		Gid_t	  af_owngid;	     /* gid of archive's owner */
		time_t    af_lastmod;        /* last mod. of archive file */
		char	  *af_busyfilename;  /* filename of busy version */
		short	  af_nrevs;	     /* number of revs in list */
		short	  af_listlen;	     /* total length of list */
		off_t	  af_datasize;	     /* size of data-segment */
		short	  af_extent;	     /* extent of revision list */
		Af_cattrs af_cattrs;	     /* version-independent attrs */
		Af_vattrs *af_list;	     /* pointer to revision list */
		int       af_access;         /* access number */
		char      *af_mem;           /* list of ptrs to alloc. mem. */
		int       af_totalmem;       /* size of memory used */
		Af_revlist *af_next;         /* index of next freelist entry */
	      };

/* Constants for af_extent */

/**** extent and type of revision list ****/

#define AF_SEGMASK      0003   /* Mask for data flags */

#define AF_ATTRS	0001   /* Attributes are loaded */
#define AF_DATA		0002   /* Data are loaded */
#define AF_COMPLETE     0004   /* List is complete (binary pools only) */

#define AF_BPOOL        0010   /* Binary pool */
#define AF_ARCHIVE      0020   /* Line of development */

#define AF_UXWRITE      0100   /* Unix write permission on AtFS subdir */
#define AF_LISTBUSY     0200   /* List is currently in use */

/*=========================================================================
 * More Hash stuff
 *=========================================================================*/

struct Af_hshent { char          *symbol;
		   Af_hashent    *next;
		 };
 
/*=========================================================================
 * Internal Installation dependent constants
 *=========================================================================*/

/**** "syntactical sugar" for user defined attributes ****/
#define AF_UDANAMDEL    '='		/* Delimiter for UDA name in archive */
#define AF_UDAVALDEL	'\01'		/* Delimiter for UDA values in arch. */

/**** UNIX Environment ****/
#define AF_TMPDIR	"/tmp"		/* location of temporary files */
#define AF_SUBDIR	"AtFS"		/* subdirectory for archives */
#define AF_OLDSUBDIR	"AFS"		/* subdirectory for archives (old) */
#define AF_ARCHEXT	'A'		/* extension for archive names */
#define AF_DATAEXT	'D'		/* extension for datafile names */
#define AF_LCKEXT	'L'		/* extension for lock files */
#define AF_ARCHTMP	'T'		/* extension for temp archive names */
#define AF_DATATMP	'U'		/* extension for temp datafile names */
#define AF_ERRLOG	"/tmp/AtFSerrlog" /* Error Log file */

/**** file locking ****/
#define AF_LOCKTIMEOUT  (unsigned) 1    /* wait 1 sec when archive is locked */

/*=========================================================================
 * Internal Constant Definitions
 *=========================================================================*/

/**** general ****/

#define AF_READ         0
#define AF_WRITE        1
#define AF_RDWR         2

/**** representation types ****/

#define AF_CHUNK	0
#define AF_DELTA	1
#define AF_FILE         2 /* version resides in an own file (busy version) */

/**** Version numbering ****/

#define AF_INITGEN	1
#define AF_INITREV	0

/**** Modes for archive manipulation ****/

#define AF_CHANGE      01
#define AF_ALLVERS     02

/**** Environment interaction ****/

#define AF_ENVBPSIZE    "ATFSBPSIZ" /* name of environment variable defining */
                                   /* max. number of files in bin. pool */
#define AF_MAXBPSIZE    64         /* max # of files in binary pool if no */
                                   /* environment variable is present */

/**** Permissions for checkperm ****/

#define AF_OWNER        0001
#define AF_AUTHOR       0002
#define AF_LOCKHOLDER   0004
#define AF_WORLD        0010

#define AF_REMOTE       -2

/*=========================================================================
 * Useful macros
 *=========================================================================*/

#define CATTR(keyp)      keyp->af_ldes->af_cattrs
#define VATTR(keyp)      keyp->af_ldes->af_list[keyp->af_lpos]

/* compare filekeys -- returnes 0 if equal (like strcmp) */
#define af_keycmp(key1,key2) (((key1)->af_ldes != (key2)->af_ldes) || ((key1)->af_lpos != (key2)->af_lpos))

/* report error and return */
#define FAIL(msg1,msg2,errcd,retcd) { af_err (msg1, msg2, errcd); return (retcd); }
#define SFAIL(msg1,msg2,errcd,retcd) { af_serr (msg1, msg2, errcd); return (retcd); }

/* convert nil pointer to empty string */
#define NOTNIL(str) (str ? str : "")

/*=========================================================================
 * Declarations
 *=========================================================================*/

char	*af_ardir();
char	*af_asctime();
time_t  af_acttime ();
char	*af_bpfilename();
char	*af_enterhost();
char	*af_enterdomain();
char	*af_entersym();
void	af_err();
void    af_free();
void	af_frmemlist();
char	*af_garname();
char	*af_gbpname();
char	*af_gbusname();
Gid_t	af_getgid();
char 	*af_gethostname();
char 	*af_getdomain();
Uid_t	af_getuid();
char	*af_gtmpname();
char    *afLookupUda();
char	*af_malloc();
char    *af_realloc();
char	*af_rbphashname();
void	af_regtmpfile();
off_t	af_retfsize();
void	af_serr();
char	*af_uniqpath();
char	*af_unixname();
void	af_unregtmpfile();
char	*af_vallookup();
void	af_wng();

#endif
