/*  Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */

#ifdef STATISTICS

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"

#ifndef STATMAIN
EXPORT FILE *statfile = NULL;
#else
FILE *statfile;
#endif
bool _openfail = FALSE;

/****************************************************************************/

void _regkeyaccess (key)
     Af_key *key;
{
  register int i;
  static Af_key lastkey;

  if (_openfail) return;
  if (statfile == NULL)
    if ((statfile = fopen ("AtFS/statistics", "a")) == NULL)
      { _openfail = TRUE; return; }
  
  i = key->af_ldes->af_listlen-1;
  while (!(key->af_ldes->af_list[i].af_class & AF_VALID))
    {
      if (i-- == -1)
	{ fprintf (statfile, "*** key-error ***"); return; }
    }
  if (key->af_lpos == i)
    fprintf (statfile, "K %d   %d\n", AF_LASTVERS, AF_LASTVERS);
  else
    fprintf (statfile, "K %d   %d\n", VATTR(key).af_gen, VATTR(key).af_rev);
}

/****************************************************************************/

void _regfileaccess (name)
     char *name;
{
  if (_openfail) return;

  fprintf (statfile, "F %s\n", name);
}

/****************************************************************************/

void _regsearchattrs (attrbuf)
     Af_attrs *attrbuf;
{
  register int i = 0;

  if (_openfail) return;

  fprintf (statfile, "A");

  if (attrbuf->af_host[0]) fprintf (statfile, "h");
  if (attrbuf->af_syspath[0]) fprintf (statfile, "s");
  if (attrbuf->af_name[0] != '*') fprintf (statfile, "n");
  if (attrbuf->af_type[0] != '*') fprintf (statfile, "t");
  if (attrbuf->af_gen != AF_NOVNUM) fprintf (statfile, "g");
  if (attrbuf->af_rev != AF_NOVNUM) fprintf (statfile, "r");
  if (attrbuf->af_state != AF_NOSTATE) fprintf (statfile, "x");
  if (attrbuf->af_owner.af_username[0]) fprintf (statfile, "o");
  if (attrbuf->af_owner.af_userhost[0]) fprintf (statfile, "p");
  if (attrbuf->af_owner.af_userdomain[0]) fprintf (statfile, "q");
  if (attrbuf->af_author.af_username[0]) fprintf (statfile, "a");
  if (attrbuf->af_author.af_userhost[0]) fprintf (statfile, "b");
  if (attrbuf->af_author.af_userdomain[0]) fprintf (statfile, "c");
  if (attrbuf->af_size != AF_NOSIZE) fprintf (statfile, "i");
  if (attrbuf->af_mode != AF_NOMODE) fprintf (statfile, "m");
  if (attrbuf->af_locker.af_username[0]) fprintf (statfile, "l");
  if (attrbuf->af_locker.af_userhost[0]) fprintf (statfile, "k");
  if (attrbuf->af_locker.af_userdomain[0]) fprintf (statfile, "j");
  if (attrbuf->af_mtime != AF_NOTIME) fprintf (statfile, "1");
  if (attrbuf->af_atime != AF_NOTIME) fprintf (statfile, "2");
  if (attrbuf->af_ctime != AF_NOTIME) fprintf (statfile, "3");
  if (attrbuf->af_stime != AF_NOTIME) fprintf (statfile, "4");
  if (attrbuf->af_ltime != AF_NOTIME) fprintf (statfile, "5");
  while (attrbuf->af_udattrs[i++] != (char *)0) fprintf (statfile, "u");

  fprintf (statfile, "\n");
}

/****************************************************************************/

void _regcall (name)
     char *name;
{
  if (_openfail) return;

  fprintf (statfile, "C %s\n", name);
}

#ifdef STATMAIN

/****************************************************************************
 *
 *  MAIN PROGRAM
 *
 ****************************************************************************/

#include <signal.h>

extern char *optarg;
extern int optind;

main (ac, av)
     int ac;
     char **av;
{
  int cleanup (), getopt ();
  register short c;

  while ((c = getopt (ac, av, "v")) != EOF)
    {
      switch (c)
	{
	case 'v': /* print current version of this program */
	  (void) printf ("This is %s version %s.\n", av[0], af_version());
	  exit (0);
	default:
	  usage ();
	  exit (-1);
	}
    }  /* end of command line parsing */
  
  (void) signal (SIGINT, cleanup);

  if (optind == ac) /* no filename given */
    {
      if ((statfile = fopen ("AtFS/statistics", "r")) == NULL)
	{ usage (); exit (-1); }
    }
  else
    {
      if ((statfile = fopen (av[optind], "r")) == NULL)
	{ usage (); exit (-1); }
    }
  
  (void) statistics (statfile);

} /* end of main */

usage ()
{
  fprintf (stderr, "Usage: afstatistics [-v] [statfile]\n");
}

cleanup ()
{
  fprintf (stderr, "aborting...\n");
  (void) fclose (statfile);
}

statistics (infile)
     FILE *infile;
{
  char line[1024];
  int  gen, rev;
  
  while (fgets (line, 1024, infile) != NULL)
    {
      switch (line[0])
	{
	case 'A':
	  s_abufattrs (line);
	  break;
	case 'F':
	  s_fileaccess (&line[2]);
	  break;
	case 'K':
	  gen = atoi (&line[2]);
	  rev = atoi (&line[6]);
	  s_keyaccess (gen, rev);
	  break;
	default:
	  fprintf (stderr, "invalid input file\n");
	  return (-1);
	}
    }

  (void) printf ("AtFS Statistics:\n");

  (void) printf ("\n");
  p_keyaccess ();

  (void) printf ("\n");
  p_fileaccess ();

  (void) printf ("\n");
  p_abufattrs ();

}

/****************************************************************************
 *  A (Attributes in Attribute buffer)
 *
 ****************************************************************************/

int acnt=0, bcnt=0, ccnt=0, gcnt=0, hcnt=0, icnt=0, jcnt=0, kcnt=0, lcnt=0,
  mcnt=0, ncnt=0, ocnt=0, pcnt=0, qcnt=0, rcnt=0, scnt=0, tcnt=0, ucnt=0,
  vcnt=0, xcnt=0, _1cnt=0, _2cnt=0, _3cnt=0, _4cnt=0, _5cnt=0, zcnt=0;

s_abufattrs (line)
     char *line;
{
  register int i=1;

  while (line[i] != '\n')
    {
      switch (line[i++])
	{
	case 'a': acnt++; break;
	case 'b': bcnt++; break;
	case 'c': ccnt++; break;
	case 'g': gcnt++; break;
	case 'h': hcnt++; break;
	case 'i': icnt++; break;
	case 'j': jcnt++; break;
	case 'k': kcnt++; break;
	case 'l': lcnt++; break;
	case 'm': mcnt++; break;
	case 'n': ncnt++; break;
	case 'o': ocnt++; break;
	case 'p': pcnt++; break;
	case 'q': qcnt++; break;
	case 'r': rcnt++; break;
	case 's': scnt++; break;
	case 't': tcnt++; break;
	case 'u': ucnt++; break;
	case 'x': xcnt++; break;
	case '1': _1cnt++; break;
	case '2': _2cnt++; break;
	case '3': _3cnt++; break;
	case '4': _4cnt++; break;
	case '5': _5cnt++; break;
	default: zcnt++; break;
	}
    }
}

p_abufattrs ()
{
  register int all;

  all = acnt + bcnt + ccnt + gcnt + hcnt + icnt + jcnt + kcnt + lcnt + mcnt +
    ncnt + ocnt + pcnt + qcnt + rcnt + scnt + tcnt + ucnt +
      xcnt + _1cnt + _2cnt + _3cnt + _4cnt + _5cnt + zcnt;
  if (all == 0)
    all = 1;

  (void) printf ("   Attribute accesses during retrieve:\n");
  (void) printf ("\thost:  %.2f%% (%d)\toname: %.2f%% (%d)\tmtime: %.2f%% (%d)\n",
	  (float)(hcnt*100)/all, hcnt, (float)(ocnt*100)/all, ocnt,
	  (float)(_1cnt*100)/all, _1cnt);
  (void) printf ("\tpath:  %.2f%% (%d)\tohost: %.2f%% (%d)\tatime: %.2f%% (%d)\n",
	  (float)(scnt*100)/all, scnt, (float)(pcnt*100)/all, pcnt,
	  (float)(_2cnt*100)/all, _2cnt);
  (void) printf ("\tname:  %.2f%% (%d)\taname: %.2f%% (%d)\tctime: %.2f%% (%d)\n",
	  (float)(ncnt*100)/all, ncnt, (float)(acnt*100)/all, acnt,
	  (float)(_3cnt*100)/all, _3cnt);
  (void) printf ("\ttype:  %.2f%% (%d)\tahost: %.2f%% (%d)\tstime: %.2f%% (%d)\n",
	  (float)(tcnt*100)/all, tcnt, (float)(bcnt*100)/all, bcnt,
	  (float)(_4cnt*100)/all, _4cnt);
  (void) printf ("\tgen:   %.2f%% (%d)\tsize:  %.2f%% (%d)\tltime: %.2f%% (%d)\n",
	  (float)(gcnt*100)/all, gcnt, (float)(icnt*100)/all, icnt,
	  (float)(_5cnt*100)/all, _5cnt);
  (void) printf ("\trev:   %.2f%% (%d)\tmode:  %.2f%% (%d)\tuda:   %.2f%% (%d)\n",
	  (float)(rcnt*100)/all, rcnt, (float)(mcnt*100)/all, mcnt,
	  (float)(ucnt*100)/all, ucnt);
  (void) printf ("\tstate: %.2f%% (%d)\tlhost  %.2f%% (%d)\tother: %.2f%% (%d)\n",
	  (float)(xcnt*100)/all, xcnt, (float)(kcnt*100)/all, kcnt,
	  (float)(zcnt*100)/all, zcnt);
  (void) printf ("\todomn: %.2f%% (%d)\tadomn: %.2f%% (%d)\tldomn: %.2f%% (%d)\n",
	  (float)(qcnt*100)/all, qcnt, (float)(ccnt*100)/all, ccnt,
	  (float)(jcnt*100)/all, jcnt);
}

/****************************************************************************
 *  F (File accesses)
 *
 ****************************************************************************/

struct { char *fname; 
int count } fnamelist[1024];
int fnamecnt = 0;

s_fileaccess (name)
     char *name;
{
  register int hi = fnamecnt-1, lo=0, pos, res, i;
  char *malloc ();

  /* see if name exists in list */
  pos = (hi+lo)/2;
  while (hi >= lo)
    {
      if ((res = strcmp (name, fnamelist[pos].fname)) == 0)
	{ fnamelist[pos].count++; return; }
      else
	{	  
	  if (res < 0) hi = pos - 1;
	  else         lo = pos + 1;
	  pos = (hi+lo)/2;
	}
    }

  /* name not found: add it to the list */
  for (i=fnamecnt; i>pos; i--)
    {
      fnamelist[i].fname = fnamelist[i-1].fname;
      fnamelist[i].count = fnamelist[i-1].count;
    }
  fnamecnt++;

  if ((fnamelist[pos].fname = malloc (strlen (name) + 1)) == (char *)0)
    return;
  (void) strcpy (fnamelist[pos].fname, name);
  fnamelist[pos].count = 1;
}

p_fileaccess ()
{
  register int i, all=0;

  for (i=0; i<fnamecnt; i++)
    all = all + fnamelist[i].count;

  (void) printf ("   File accesses:\n");
  for (i=0; i<fnamecnt; i++)
    (void) printf ("\t%s:\t %.2f%% (%d)\n", fnamelist[i].fname,
	    (float)(fnamelist[i].count*100)/all, fnamelist[i].count);
}

/****************************************************************************
 *  K (Key accesses)
 *
 ****************************************************************************/

int busycnt=0, lastcnt=0, othercnt=0;

s_keyaccess (gen, rev)
     int gen, rev;
{
  if ((gen == AF_BUSYVERS) && (rev == AF_BUSYVERS))
    busycnt++;
  else
    {
      if ((gen == AF_LASTVERS) && (rev == AF_LASTVERS))
	lastcnt++;
      else
	othercnt++;
    }
}

p_keyaccess ()
{
  register int all;

  all = busycnt + lastcnt + othercnt;
  if (all == 0)
    all = 1;

  (void) printf ("   Key accesses: busy version:   %.2f%% (%d)\n",
	  (float)(busycnt*100)/all, busycnt);
  (void) printf ("                 last version:   %.2f%% (%d)\n",
	  (float)(lastcnt*100)/all, lastcnt);
  (void) printf ("                 other versions: %.2f%% (%d)\n",
	  (float)(othercnt*100)/all, othercnt);
}

#endif /* STATMAIN */

#endif /* STATISTICS */

afStatDummy ()
{
  /* this function definition was only introduced to suppress
   * warnings from ranlib.
   */
}
