/*LINTLIBRARY*/
/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	afsattrs.c - reading and writing simple attributes
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP
 *					   andy@db0tui62.BITNET)
 *
 *	$Header: afsattrs.c[1.17] Mon Feb 10 22:59:22 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *	af_rname -- get name
 *	af_rtype -- get type
 *	af_rsyspath -- get system pathname
 *	af_rgen -- return generation number
 *	af_rrev -- return revision number
 *      af_rdsize -- return delta size
 *      af_rowner -- return owner
 *      af_chowner -- change owner
 *      af_rauthor -- return author
 *      af_chauthor -- return author
 *	af_chmod -- change mode
 *	af_gattrs -- get attribute buffer
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"

char *malloc();

#ifdef MEMDEBUG
extern FILE *memprot;
#endif

/*====================================================================
 *    af_rname -- return name
 *
 *====================================================================*/

EXPORT char *af_rname (key)
     Af_key *key;
{
  static char name[MAXNAMLEN];

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rname", "", AF_EINVKEY, (char *)0);

  if (VATTR(key).af_name == (char *)0)
    name[0] = '\0';
  else
    (void) strcpy (name, VATTR(key).af_name);

  return (name);
}


/*====================================================================
 *    af_rtype -- return type
 *
 *====================================================================*/

EXPORT char *af_rtype (key)
     Af_key *key;
{
  static char type[MAXTYPLEN];

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rtype", "", AF_EINVKEY, (char *)0);

  if (VATTR(key).af_type == (char *)0)
    type[0] = '\0';
  else
    (void) strcpy (type, VATTR(key).af_type);

  return (type);
}


/*====================================================================
 *    af_rsyspath -- return system pathname
 *
 *====================================================================*/

EXPORT char *af_rsyspath (key)
     Af_key *key;
{
 static char syspath[MAXPATHLEN];

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rsyspath", "", AF_EINVKEY, (char *)0);

  if (CATTR(key).af_syspath == (char *)0)
    syspath[0] = '\0';
  else
    (void) strcpy (syspath, CATTR(key).af_syspath);

  return (syspath);
}


/*====================================================================
 *    af_rgen -- return generation number
 *
 *====================================================================*/

EXPORT af_rgen (key)
     Af_key *key;
{
  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rgen", "", AF_EINVKEY, ERROR);

  return (VATTR(key).af_gen);
}


/*====================================================================
 *    af_rrev -- return revision number
 *
 *====================================================================*/

EXPORT af_rrev (key)
     Af_key *key;
{
  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rrev", "", AF_EINVKEY, ERROR);

  return (VATTR(key).af_rev);
}

/*====================================================================
 *    af_rdsize -- return delta size
 *
 *====================================================================*/

EXPORT af_rdsize (key)
     Af_key *key;
{
  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rdsize", "", AF_EINVKEY, ERROR);

  if (VATTR(key).af_repr != AF_DELTA)
    return (ERROR);
  return (VATTR(key).af_dsize);
}

/*====================================================================
 *    af_rsize -- return file size
 *
 *====================================================================*/

EXPORT af_rsize (key)
     Af_key *key;
{
  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rdsize", "", AF_EINVKEY, ERROR);

  return (VATTR(key).af_fsize);
}

/*====================================================================
 *      af_rowner -- return owner
 *
 *====================================================================*/

EXPORT Af_user *af_rowner (key)
     Af_key *key;
{
  static Af_user owner;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rowner", "", AF_EINVKEY, (Af_user *)0);

  (void) strcpy (owner.af_username, CATTR(key).af_ownname);
  (void) strcpy (owner.af_userhost, CATTR(key).af_ownhost);
  (void) strcpy (owner.af_userdomain, CATTR(key).af_owndomain);
  return (&owner);
}

/*====================================================================
 *      af_chowner -- change owner
 *
 *====================================================================*/

EXPORT af_chowner (key, owner)
     /*ARGSUSED*/
     Af_key  *key;
     Af_user *owner;
{
  /* not yet implemented (chowner) */
  SFAIL ("chowner", "", AF_EACCES, ERROR);
}

/*====================================================================
 *      af_rauthor -- return author
 *
 *====================================================================*/

EXPORT Af_user *af_rauthor (key)
     Af_key *key;
{
  static Af_user author;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("rauthor", "", AF_EINVKEY, (Af_user *)0);

  (void) strcpy (author.af_username, VATTR(key).af_auname);
  (void) strcpy (author.af_userhost, VATTR(key).af_auhost);
  (void) strcpy (author.af_userdomain, VATTR(key).af_audomain);
  return (&author);
}

/*====================================================================
 *      af_chauthor -- return author
 *
 *====================================================================*/

EXPORT af_chauthor (key, author)
     Af_key  *key;
     Af_user *author;
{
  Uid_t uid;
  Gid_t gid;

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("chauthor", "", AF_EINVKEY, ERROR);
  if (af_checkperm (key, AF_OWNER) == ERROR)
    return (ERROR);

  if (VATTR(key).af_state == AF_BUSY)
    {
      if ((uid = af_getuid (author->af_username, author->af_userhost, author->af_userdomain))
	  == (Uid_t) -1)
	SFAIL ("chauthor", "", AF_EINVUSER, ERROR);
      if ((gid = af_getgid (author->af_username, author->af_userhost, author->af_userdomain))
	  == (Gid_t) -1)
	SFAIL ("chauthor", "", AF_EINVUSER, ERROR);
      if (chown (key->af_ldes->af_busyfilename, (int) uid, (int) gid) == ERROR)
	FAIL ("chauthor", "chown", AF_ESYSERR, ERROR);
    }
  else
    {
      VATTR(key).af_auname = af_entersym (author->af_username);
      VATTR(key).af_auhost = af_enterhost (author->af_userhost);
      VATTR(key).af_audomain = af_enterdomain (author->af_userdomain);
    }
  if (afUpdateAso (key, AF_CHANGE) == ERROR)
    return (ERROR);
  return (AF_OK);
}

/*====================================================================
 *    af_chmod -- change mode
 *
 *====================================================================*/

EXPORT af_chmod (key, mode)
     Af_key *key;
     int    mode;
{
  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("chmod", "", AF_EINVKEY, ERROR);

  if (af_checkperm (key, AF_OWNER | AF_AUTHOR) == ERROR)
    return (ERROR);

  if (VATTR(key).af_state == AF_BUSY)
    {
      if (chmod (key->af_ldes->af_busyfilename, mode) == ERROR)
	FAIL ("chmod", "chmod", AF_ESYSERR, ERROR);
    }
  else
    {
      VATTR(key).af_mode &= ~07777;
      VATTR(key).af_mode |= mode & 07777;
    }
 
  if (afUpdateAso (key, AF_CHANGE) == ERROR)
	return (ERROR);

  return (AF_OK);
}


/*====================================================================
 *    af_gattrs -- get attribute buffer
 *
 *====================================================================*/

EXPORT af_gattrs (key, attrbuf)
     Af_key   *key;
     Af_attrs *attrbuf;
{
  register int  i, size;
  register char *udattrs, *valptr;
  char          *udalist[AF_MAXUDAS+1];

  if (afAccessAso (key, AF_ATTRS))
    SFAIL ("gattrs", "", AF_EINVKEY, ERROR);

  (void) strcpy (attrbuf->af_host, CATTR(key).af_host);
  (void) strcpy (attrbuf->af_name, VATTR(key).af_name);
  (void) strcpy (attrbuf->af_type, NOTNIL(VATTR(key).af_type));
  (void) strcpy (attrbuf->af_syspath, CATTR(key).af_syspath);

  attrbuf->af_gen = VATTR(key).af_gen;
  attrbuf->af_rev = VATTR(key).af_rev;
  attrbuf->af_state = (int) VATTR(key).af_state;
  (void) strcpy (attrbuf->af_owner.af_username, CATTR(key).af_ownname);
  (void) strcpy (attrbuf->af_owner.af_userhost, CATTR(key).af_ownhost);
  (void) strcpy (attrbuf->af_owner.af_userdomain, CATTR(key).af_owndomain);
  (void) strcpy (attrbuf->af_author.af_username, VATTR(key).af_auname);
  (void) strcpy (attrbuf->af_author.af_userhost, VATTR(key).af_auhost);
  (void) strcpy (attrbuf->af_author.af_userdomain, VATTR(key).af_audomain);
  attrbuf->af_size = VATTR(key).af_fsize;
  attrbuf->af_mode = VATTR(key).af_mode;
  (void) strcpy (attrbuf->af_locker.af_username, NOTNIL(VATTR(key).af_lckname));
  (void) strcpy (attrbuf->af_locker.af_userhost, NOTNIL(VATTR(key).af_lckhost));
  (void) strcpy (attrbuf->af_locker.af_userdomain, NOTNIL(VATTR(key).af_lckdomain));
  attrbuf->af_mtime = VATTR(key).af_mtime;
  attrbuf->af_atime = VATTR(key).af_atime;
  attrbuf->af_ctime = VATTR(key).af_ctime;
  attrbuf->af_stime = VATTR(key).af_stime;
  attrbuf->af_ltime = VATTR(key).af_ltime;

  /* copy user defined attributes */
  if (VATTR(key).af_udanum > 0)
    {
      (void) afListUdas (key, udalist);
      size = 0;
      i=0;
      while (udalist[i])
	{
	  size += strlen (udalist[i]) + sizeof (char);
	  i++;
	}
      if ((udattrs = malloc ((unsigned) size)) == (char *)0)
	FAIL ("gattrs", "malloc", AF_ESYSERR, ERROR);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(udattrs-gattr)-AL %d bytes\n", udattrs, size);
#endif
      i=0;
      while (udalist[i] != (char *)0)
	{
	  (void) strcpy (udattrs, udalist[i]);
	  /* replace delimiters by '\n' */
	  if (valptr = index (udattrs, AF_UDANAMDEL))
	    {
	      while ((valptr = index (valptr, AF_UDAVALDEL)) != (char *)0)
		valptr[0] = '\n';
	    }
	  attrbuf->af_udattrs[i] = udattrs;
	  udattrs = udattrs + (strlen(udalist[i])+1);
	  i++;
	}
      attrbuf->af_udattrs[i] = (char *)0; /* finish list */
    }
  else
    attrbuf->af_udattrs[0] = (char *)0;

  return (AF_OK);
}
