/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 screen.c
 *	Date:	 8/22/90
 *	Version: 1.83
 *
 *	screen.c - Print the data on the screen.  This is a curses(3)
 *	based version of the output function.  An X version will
 *	probably appear in window.c if I get that ambitious.
 *
 *	The source for screen_format was merged from format.c V1.9.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)screen.c	1.83 (monitor) 8/22/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 *
 * Jan. 5, 1989 -- arr
 *
 *	Fix references to the mon_meter data structure.
 *
 * Feb. 15, 1989 -- arr
 *
 *	Check to make sure that mon_flag is correctly set using 
 *	MON$M_VALID.  Also removed the pgtok macros in favor of
 *	functions.
 *
 * March 11, 1989 -- arr
 *
 *	Changed needed for change to mon_fork structure in monitor.h.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 * Aug. 22, 1990 -- arr
 *
 *	Fix a problem for magnify_cpu() that was causeing the
 *	number of system calls and device interrupts to be too
 *	high.
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <curses.h>

#include <sys/types.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#include <net/if.h>
#include <netinet/in.h>

#include "include.h"
#include "monitor.h"
#include "screen.h"
#include "options.h"
#include "extern.h"

/*
 *	SPACE - Number of spaces between fields.
 */
#define	SPACE		(3)

/*
 *	Various functions used that don't return (int).
 */
double	calculate_disk(),
	delta_tty(),
	delta_cpu(),
	delta_fork(),
	delta_netif() ;

void	delta_syscall_intr() ;

char	*strchr(),
	*ctime(),
	*str_data() ;

/*
 *	Module name for error functions.  This currently isn't being
 *	used.  If it isn't being used after I'm done hacking the
 *	screen code then it needs to go away.
 */
#ifndef	lint
static	char	*module = "screen" ;
#endif

/*
 *	Assorted externs.
 */
extern int	changed_size ;		/* flag to say that window has changed size */
extern int	lines ;			/* length of display being used */
extern WINDOW	*wp ;			/* default display window */

/*
 *	<x,y> positions for the various fields.
 */
int	cpu_y = CPU_Y, cpu_x = CPU_X,
	tty_y = TTY_Y, tty_x = TTY_X,
	proc_y = PROC_Y, proc_x = PROC_X,
	free_y = FREE_Y, free_x = FREE_X,
	fork_y = FORK_Y, fork_x = FORK_X,
	disk_y = DISK_Y, disk_x = DISK_X,
	user_y = USER_Y, user_x = USER_X,
	netif_y = NETIF_Y, netif_x = NETIF_X,
	page_y = PAGE_Y, page_x = PAGE_X,
	page2_y = PAGE2_Y, page2_x = PAGE2_X,
	memory_y = MEMORY_Y, memory_x = MEMORY_X,
	loadave_y = LOADAVE_Y, loadave_x = LOADAVE_X ;

/*
 *	Line that the last ... will be on because of lack of space.
 */
int	last_disk, last_cpu, last_netif ;

/*
 *	Do whatever needs to be done to initialize the screen.  I suggest
 *	turning on curses and printing the headers.
 */
open_screen(op)
OPTION	*op ;
{
	/*
	 *	Setup the screen for use with curses.
	 */
	setup_curses(op, -1, -1) ;

	/*
	 *	Set the sleep time dependent on the terminal speed.
	 *	In REPLAY mode, the sleep time will be be used unless
	 *	set from the command line.
	 */
	if( !op->sleep_set && op->collect_mode != OPT_REPLAY )
		first.mon_sleep = op->opt_sleep = ttyspeed() ;

	screen_format(op, first.mon_cpu, first.mon_disk, first.mon_netif) ;

	/*
	 *	If the -sample option is turned on then turn on the
	 *	sample data option.
	 */
	if( op->opt_mag_sample )
		op->opt_sample = 1 ;
	else
		op->opt_sample = 0 ;

	/*
	 *	If PID is in the data, magnify it.
	 */
	if( op->opt_pid )
		op->opt_magnify |= MON$M_PID ;

	/*
	 *	If only SWAP, BUF or NAMEI are turned on, then
	 *	turn magnify it.
	 */
	if((op->opt_data & MON$M_ALL) == MON$M_SWAP )
		op->opt_magnify |= MON$M_SWAP ;
	else if((op->opt_data & MON$M_ALL) == MON$M_BUF )
		op->opt_magnify |= MON$M_BUF ;
	else if((op->opt_data & MON$M_ALL) == MON$M_NAMEI )
		op->opt_magnify |= MON$M_NAMEI ;

	if( op->opt_magnify )
		(void)select_function(op->opt_magnify, op) ;

	if( op->redraw_display )
		(*op->redraw_display)(op) ;
}

/*
 *	Do whatever needs to be done to clean up.
 *
 *	close_screen() might be extended someday to actually use the
 *	option structure.  Because of this the cleanup function which
 *	has been used in the error functions was moved out by itself.
 *
 *	ARGSUSED
 */
close_screen(op)
OPTION	*op ;
{
	/*
	 *	This will only need to be called in OPT_SCREEN mode.
	 */
	cleanup_screen() ;
}

cleanup_screen()
{
	wmove(wp, LINES - 1, 0) ;
	wprintw(wp, "\n");
	wrefresh(wp) ;

	/*
	 *	endwin() should return the state of the world to
	 *	normal.
	 */
	endwin() ;
}

/*
 *	Format the data for output to the screen.
 */
screen(op)
OPTION	*op ;
{
	register i ;
	double	 etime ;

	print_semi_static() ;

	if( op->opt_loadave ) {
		wmove(wp, loadave_y, loadave_x);
		for(i = 0 ; i < MON$N_LOADAVE; i++)
			wprintw(wp, LOADAVE_FORMAT, loadave.mon_loadave[i]);
	}

	if( op->opt_user ) {
		wmove(wp, user_y, user_x) ;
		wprintw(wp, "%3d user%s", mon_user.mon_user,
			mon_user.mon_user == 1 ? " " : "s");
	}

	if( op->opt_proc ) {
		proc_ticks = sample.mon_ticks ;

		wmove(wp, proc_y + 1, proc_x + PROC_OFF) ;
		wprintw(wp, PROC_FORMAT, mon_proc.mon_rq, mon_proc.mon_dw,
			mon_proc.mon_pw, mon_proc.mon_sl);
	}

	if( op->opt_memory ) {
		memory_ticks = sample.mon_ticks ;

		wmove(wp, memory_y + 1, memory_x + MEMORY_OFF) ;
		wprintw(wp, MEMORY_FORMAT, pgtok(memory.mon_rm),
			pgtok(memory.mon_vm), pgtok(memory.mon_avm),
			pgtok(memory.mon_free)) ;
	}
	else if( op->opt_free ) {
		memory_ticks = sample.mon_ticks ;

		wmove(wp, free_y + 1, free_x + FREE_OFF) ;
		wprintw(wp, FREE_FORMAT, pgtok(mon_free.mon_freemem));
	}

	if( op->opt_fork ) {
		etime = delta_fork(sample.mon_ticks);
		
		wmove(wp, fork_y + 1, fork_x + FORK_OFF) ;
		wprintw(wp, FORK_FORMAT, mon_fork.mon_fork / etime,
			mon_fork.mon_vfork / etime);
	}

	if( op->opt_page ) {
		paging_ticks = sample.mon_ticks ;

		wmove(wp, page_y + 1, page_x + PAGE1_OFF) ;
		wprintw(wp, PAGE1_FORMAT,
			mon_page.mon_pgrec - (mon_page.mon_xsfrec + mon_page.mon_xifrec),
			mon_page.mon_xsfrec + mon_page.mon_xifrec,
			pgtok((int)mon_page.mon_pgpgin), 
			pgtok((int)mon_page.mon_pgpgout),
			pgtok((int)mon_page.mon_deficit),
			mon_page.mon_swpin, mon_page.mon_swpout,
			mon_page.mon_faults, mon_page.mon_pgfrec);

		wmove(wp, page2_y + 1, page2_x + PAGE2_OFF) ;
		wprintw(wp, PAGE2_FORMAT,
			pgtok((int)mon_page.mon_pgin) + pgtok((int)mon_page.mon_pgout),
			pgtok((int)mon_page.mon_scan), 
			pgtok((int)mon_page.mon_dfree),
			mon_page.mon_nexfod, mon_page.mon_exfod,
			mon_page.mon_nzfod, mon_page.mon_zfod,
			mon_page.mon_nvrfod, mon_page.mon_vrfod);
	}

	if( op->opt_tty ) {
		etime = delta_tty(sample.mon_ticks);
		
		wmove(wp, tty_y + 1, tty_x + TTY_OFF) ;
		wprintw(wp, TTY_FORMAT, tty.mon_ttyin / etime, 
			tty.mon_ttyout / etime);
	}

	if( op->opt_cpu ) {
		double	total_time ;
		int	cursor = cpu_y ;

		/*
		 *	This makes sure that we keep proper track of
		 *	the number of system calls and device interrupts
		 *	even though we're not interested in the data.
		 */
		delta_syscall_intr((int *)NULL, (int *)NULL) ;

		for(i = 0; i < first.mon_cpu ; i++) {
			if( cursor++ == last_cpu )
				break ;

			if((cpu[i].mon_flag & MON$M_VALID)== 0 )
				continue ;

			etime = delta_cpu(i, sample.mon_ticks, &total_time);

			wmove(wp, cursor, cpu_x + CPU_OFF) ;
			wprintw(wp, CPU_FORMAT,
				100. * cpu[i].mon_time[0] / total_time,
				100. * cpu[i].mon_time[1] / total_time,
				100. * cpu[i].mon_time[2] / total_time,
				100. * cpu[i].mon_time[3] / total_time,
				cpu[i].mon_swtch / etime) ;

			/*
			 *	Before V4.0 only do the interrupts
			 *	and syscalls for the first CPU.
			 */
#ifndef V4_ULTRIX
			if( i != 0 )
				continue ;
#endif
			wmove(wp, cursor, cpu_x + CPU_OFF_1) ;
			wprintw(wp, CPU_FORMAT_1, cpu[i].mon_intr / etime,
				cpu[i].mon_syscall / etime);
		}
	}

	if( op->opt_netif ) {
		int	cursor = netif_y ;

		for(i = 0; i < first.mon_netif; i++) {
			if( cursor++ == last_netif )
				break ;

			if((netif[i].mon_flag & MON$M_VALID) == 0 )
				continue ;

			etime = delta_netif(i, sample.mon_ticks);

			wmove(wp, cursor, netif_x + NETIF_OFF) ;
			wprintw(wp, NETIF_FORMAT, netif[i].mon_ipackets / etime,
				netif[i].mon_ierrors / etime,
				netif[i].mon_opackets / etime, 
				netif[i].mon_oerrors / etime,
				netif[i].mon_collisions / etime);
		}
	}

	if( op->opt_disk ) {
		double	kbps, tps, tkbps = 0, ttps = 0 ;
		int	cursor = disk_y, msps ;

		for(i = 0; i < first.mon_disk ; i++) {
			if( cursor++ == last_disk )
				break ;

			if((disk[i].mon_flag & MON$M_VALID) == 0 )
				continue ;

			(void)calculate_disk(i, &kbps, &tps, &msps);

			wmove(wp, cursor, disk_x + DISK_OFF) ;
			wprintw(wp, DISK_FORMAT, kbps, tps, msps);

			if( op->opt_total ) {
				tkbps += kbps ;
				ttps  += tps ;
			}
		}

		/*
		 *	Print out the disk totals.
		 */
		if( op->opt_total ) {
			wmove(wp, last_disk + 2, disk_x + TOTAL_OFF) ;
			wprintw(wp, TOTAL_FORMAT, tkbps, ttps);
		}
	}
}

screen_prompt(op)
OPTION	*op ;
{
	wmove(wp, LINES - 1, 0) ;
	wprintw(wp, op->opt_prompt);
	wclrtoeol(wp) ;
	wrefresh(wp) ;
}

print_semi_static()
{
	char	*time_string, *newline ;

	wmove(wp, HOST_Y, HOST_X) ;
	wprintw(wp, HOST_FORMAT, HOST_LEN, HOST_LEN, first.mon_hostname);

	time_string = ctime((long *)&sample.mon_timestamp);
	if((newline = strchr(time_string, '\n')) != NULL )
		*newline = '\0' ;

	wmove(wp, DATE_Y, DATE_X) ;
	wprintw(wp, DATE_FORMAT, time_string);
}

/*
 *	Print the heading for the various fields.  Should only print
 *	headings for fields that are turned on.
 */
redraw_screen(op)
OPTION	*op ;
{
	register i ;

	if( changed_size ) {
		screen_format(op, first.mon_cpu, first.mon_disk, first.mon_netif) ;
		changed_size = 0 ;
	}

	wclear(wp) ;

	if( op->opt_proc ) {
		wmove(wp, proc_y, proc_x) ;
		wprintw(wp, PROC_HEADING);
	}

	if( op->opt_memory ) {
		wmove(wp, memory_y, memory_x) ;
		wprintw(wp, MEMORY_HEADING);
	}

	if( op->opt_free ) {
		wmove(wp, free_y, free_x) ;
		wprintw(wp, FREE_HEADING);
	}

	if( op->opt_fork ) {
		wmove(wp, fork_y, fork_x) ;
		wprintw(wp, FORK_HEADING);
	}

	if( op->opt_cpu ) {
		int	cursor = cpu_y ;

		wmove(wp, cpu_y, cpu_x) ;
		wprintw(wp, CPU_HEADING);

		for(i = 0; i < first.mon_cpu; i++) {
			if( ++cursor > last_cpu )
				break ;

			wmove(wp, cursor, cpu_x) ;
			wprintw(wp, "#%d", cpu[i].mon_cpu);
		}

		if((last_cpu-cpu_y) < first.mon_cpu ) {
			wmove(wp, LINES - 1, NETIF_MORE) ;
			wprintw(wp, "<cpu>") ;
		}
	}

	if( op->opt_tty ) {
		wmove(wp, tty_y, tty_x) ;
		wprintw(wp, TTY_HEADING);
	}

	if( op->opt_page ) {
		wmove(wp, page_y, page_x) ;
		wprintw(wp, PAGE1_HEADING);

		wmove(wp, page2_y, page2_x) ;
		wprintw(wp, PAGE2_HEADING);
	}

	if( op->opt_disk ) {
		int	cursor = disk_y ;

		wmove(wp, disk_y, disk_x) ;
		wprintw(wp, DISK_HEADING);

		for(i = 0; i < first.mon_disk; i++) {
			if( ++cursor > last_disk )
				break ;

			wmove(wp, cursor, disk_x) ;
			wprintw(wp, "%s%d", disk[i].mon_name, disk[i].mon_unit);
		}

		if( op->opt_total ) {
			wmove(wp, last_disk + 1, disk_x) ;
			wprintw(wp, TOTAL_LINE);
		}

		if((last_disk-disk_y) < first.mon_disk ) {
			wmove(wp, LINES - 1, NETIF_MORE) ;
			wprintw(wp, "<disk>");
		}
	}

	if( op->opt_netif ) {
		int	cursor = netif_y ;

		wmove(wp, netif_y, netif_x) ;
		wprintw(wp, NETIF_HEADING);

		for(i = 0; i < first.mon_netif; i++) {
			if( ++cursor > last_netif )
				break ;

			wmove(wp, cursor, netif_x) ;
			wprintw(wp, "%s%d", netif[i].mon_name, netif[i].mon_unit);
		}

		if((last_netif-netif_y) < first.mon_netif ) {
			wmove(wp, LINES - 1, NETIF_MORE) ;
			wprintw(wp, "<netif>");
		}
	}
}

extern	char	*sys_siglist[] ;

/*
 *	Use curses to print out info about the "last" record.
 */
screen_last(y)
int	y ;
{
	char	*date, *newline ;

	date = ctime((long *)&last.mon_edate);
	if((newline = strchr(date, '\n')) != NULL )
		*newline = '\0' ;

	wmove(wp, y++, 0) ;
	wprintw(wp, "This session ended at %s.", date);

	return y + 1 ;
}

/*
 *	This function is called when a "first" record has been
 *	found.
 */
screen_first(y)
{
	char	*date, *newline ;
	char	*word ;

	date = ctime((long *)&first.mon_sdate);
	if((newline = strchr(date, '\n')) != NULL )
		*newline = '\0' ;

	wmove(wp, y++, 0) ;
	wprintw(wp, "The next session was started on node %s at %s.",
		first.mon_hostname, date);

	word = first.mon_disk == 1 ? "is" : "are" ;

	wmove(wp, y++, 0) ;
	wprintw(wp, "The options are <%s>.", str_data(first.mon_options));

	wmove(wp, y++, 0) ;
	wprintw(wp, "There %s %d disk%s, %d cpu%s and %d netif%s.", word,
		first.mon_disk, first.mon_disk != 1 ? "\'s" : "",
		first.mon_cpu, first.mon_cpu != 1 ? "\'s" : "",
		first.mon_netif, first.mon_netif != 1 ? "\'s" : "");
}

/*
 *	Attempt to rationally format the screen.  The "screen" is divided
 *	into 3 areas.  Area 0 is the first two lines of the screen, one
 *	for the hostname, loadave, date and users information and the other
 *	is left blank.  Area1 is most of the left side of the screen below
 *	Area0.  Area2 is the remainder of the right side of the screen below
 *	Area0.
 *
 *	Screen_format() will attempt to compress all the fixed data into
 *	the top of Area1, leaving the bottom of Area1 and as much of Area2
 *	as possible for data that can "grow".  The data options that can
 *	"grow" are disk, netif and cpu.
 *
 *	I'll figure this out one of these days.
 *
 *	Preferred areas:
 *
 *	Area 0 - loadave and users.
 *	Area 1 - Proc, memory, paging, CPU and Netif.
 *	Area 2 - Fork, TTY, Disk.
 *
 *	1.  Figure out where to put the stuff on the top line.  The load
 *	    averages come first followed by the number of users.  If the
 *	    load average isn't being displayed, put the users where load
 *	    average would have been.
 *
 *	2.  If paging, cpu or netif are being display disk will have to
 *	    live in area2.  If not we'll put disk in area1 and move the
 *	    others over a comfortable distance.
 */
screen_format(op, cpus, disks, netifs)
OPTION	*op ;
int	cpus, disks, netifs ;
{
	int	area1_y = PROC_Y ;
	int	area1_x = PROC_X ;
	int	area2_y = FORK_Y ;
	int	area1_len = LINES - 4 ;
	int	area2_len = LINES - 4 ;
	int	area_width = DISK_X - PROC_X - 1 ;
	int	area_lines ;
	int	last_lines[MON$N_RECORDS], i ;

/*
 *	Initialize the array of "last_lines".  This describes
 *	where the line that a particular piece of data ends
 *	on.
 */
	for(i = 0; i < MON$N_RECORDS; i++)
		last_lines[i] = 0 ;

/*
 *	Initialize all the screen locations.
 */
	cpu_y = CPU_Y ; cpu_x = CPU_X ;
	tty_y = TTY_Y ; tty_x = TTY_X ;
	proc_y = PROC_Y ; proc_x = PROC_X ;
	free_y = FREE_Y ; free_x = FREE_X ;
	fork_y = FORK_Y ; fork_x = FORK_X ;
	disk_y = DISK_Y ; disk_x = DISK_X ;
	page_y = PAGE_Y ; page_x = PAGE_X ;
	user_y = USER_Y ; user_x = USER_X ;
	netif_y = NETIF_Y ; netif_x = NETIF_X ;
	page2_y = PAGE2_Y ; page2_x = PAGE2_X ;
	memory_y = MEMORY_Y ; memory_x = MEMORY_X ;
	loadave_y = LOADAVE_Y ; loadave_x = LOADAVE_X ;

	if( op->opt_proc ) {
		if( area_width >= PROC_LEN ) {
			proc_y = area1_y ;
			proc_x = area1_x ;
			area1_x += (PROC_LEN + SPACE) ;
			area_width -= (PROC_LEN + SPACE) ;
		}

		last_lines[MON$C_PROC] = proc_y + 1 ;
	}

	if( op->opt_memory ) {
		if( area_width >= MEMORY_LEN ) {
			memory_x = area1_x ;
			memory_y = area1_y ;
			area1_x += (MEMORY_LEN + SPACE) ;
			area_width -= (MEMORY_LEN + SPACE) ;
		}

		last_lines[MON$C_MEMORY] = memory_y + 1 ;
	}
	else if( op->opt_free ) {
		if( area_width >= FREE_LEN ) {
			free_x = area1_x ;
			free_y = area1_y ;
			area1_x += (FREE_LEN + SPACE) ;
			area_width -= (FREE_LEN + SPACE) ;
		}

		last_lines[MON$C_FREE] = free_y + 1 ;
	}

	if( op->opt_fork ) {
		if( area_width < FORK_LEN ) {
			fork_y = area2_y ;
			area2_y += 3 ;
			area2_len -= 3 ;
		}
		else {
			fork_x = area1_x ;
			fork_y = area1_y ;
			area1_x += (FORK_LEN + SPACE) ;
			area_width -= (FORK_LEN + SPACE) ;
		}

		last_lines[MON$C_FORK] = fork_y + 1 ;
	}

	if( op->opt_tty ) {
		if( area_width < TTY_LEN ) {
			tty_y = area2_y ;
			area2_y += 3 ;
			area2_len -= 3 ;
		}
		else {
			tty_x = area1_x ;
			tty_y = area1_y ;
			area1_x += (TTY_LEN + SPACE) ;
			area_width -= (TTY_LEN + SPACE) ;
		}

		last_lines[MON$C_TTY] = tty_y + 1 ;
	}

	if( area1_x != PROC_X )
		area1_y += 3 ;

	if( op->opt_page ) {
		page_y = area1_y ;
		page2_y = area1_y + 3 ;

		area1_x = PAGE1_LEN ;
		area1_y += 6 ;
		area1_len -= 6 ;
		last_lines[MON$C_PAGE] = page2_y + 1 ;
	}

	/*
 	 *	Figure out how many lines are available to display
	 *	CPU's and NETIF's.  If there are more CPU's and NETIF's
	 *	than lines, only display those that will fix and turn
	 *	on the approproiate "<more>" indicator.
	 */

	area_lines = area1_len - (op->opt_cpu ? 1 : 0 + op->opt_netif ? 1 : 0) ;

	if( op->opt_cpu && op->opt_netif )
		area_lines-- ;

	if((cpus + netifs) > area_lines )
		limit(area_lines, &cpus, &netifs);

	if( op->opt_cpu ) {
		cpu_y = area1_y ;
		area1_x = CPU_LEN ;
		area1_y += (2 + cpus) ;
		last_cpu = cpu_y + cpus ;
		last_lines[MON$C_CPU] = last_cpu ;
	}

	if( op->opt_netif ) {
		netif_y = area1_y ;
		area1_x = NETIF_LEN ;
		last_netif = netif_y + netifs ;
		last_lines[MON$C_NETIF] = last_netif ;
	}

	/*
 	 *	Do the same thing for disks, but also figure out if
	 *	we want to put them in area1.
	 *
	 *	This is seriously magical, but I'll try to explain
	 *	anyway.
	 *
	 *	If the area1 X coordinate hasn't changed then there isn't
	 *	anything in area1.  The number of lines left in that area
	 *	is (length - 1).  The -1 give one line blank line above the
 	 *	prompt.  The X coordinate of the disks is PROC_X.  "area2_y"
	 *	is used later.
	 *
	 *	If the area1 X coordinate hasn't changed we put the disk
	 *	data in the normal place.  Since there won't be anything
	 *	but error messages on the bottom line, we don't subtract
	 *	a line from those available.
	 */
	if( area1_x == PROC_X ) {
		area_lines = area1_len - 1 ;
		area2_y = area1_y ;
		disk_x = PROC_X ;
	}
	else {
		area_lines = area2_len ;
		disk_x = DISK_X ;
	}

	if( op->opt_disk ) {
		if( op->opt_total )
			area_lines -= 2 ;

		if( disks > area_lines )
			disks = area_lines ;

		disk_y = area2_y ;
		last_disk = disk_y + disks ;

		if( disk_x != PROC_X ) {
			area2_y += (2 + disks) ;

			if( op->opt_total )
				area2_y += 2 ;
		}

		last_lines[MON$C_DISK] = last_disk ;
	}

	/*
	 *	The number of lines to be printed on a screen dump is
	 *	the largest value in the array last_lines + 2.
	 */
	lines = max(last_lines, MON$N_RECORDS) ;

	/*
	 *	If disk totals are turned on one or two more lines
	 *	will be added to this.
	 */
	if( op->opt_total ) {
		int	diff = lines - last_lines[MON$C_DISK] ;

		if( diff == 0 )
			lines += 2 ;
		else if( diff == 1 )
			lines += 1 ;
		else
			;
	}

	/*
	 *	Two more for all cases.
	 */
	lines += 2 ;
}

/*
 *	Find the largest value in an array.
 */
max(array, n)
int	*array, n ;
{
	int	i, value = array[0] ;

	for(i = 1; i < n; i++)
		if( array[i] > value )
			value = array[i] ;

	return value ;
}

/*
 *	Change the value of a and/or b so that their total is <= total.
 */
limit(total, a, b)
int	 total, *a, *b ;
{
	if( *a > *b ) {
		*a = ((double)(*a)/(*a + *b)) * total ;
		*b = total - *a ;
	}
	else {
		*b = ((double)(*b)/(*a + *b)) * total ;
		*a = total - *b ;
	}
}

#include <sgtty.h>

static	int	Speeds[] = {
	 2,	/* 0 */
	15,	/* 50 */
	12,	/* 75 */
	12,	/* 110 */
	10,	/* 134.5 */
	10,	/* 150 */
	 8,	/* 200 */
	 8,	/* 300 */
	 5,	/* 600 */
	 5,	/* 1200 */
	 4,	/* 1800 */
	 3,	/* 2400 */
	 3,	/* 4800 */
	 2,	/* 9600 */
	 2,	/* EXTA */
	 2,	/* EXTB */
};

/*
 *	From the above return a sleep time, which is likely to
 *	provide enough time between updates to have the cursor
 *	sit at the prompt.  (i.e. Don't spend all the time redraw-
 *	ing the screen.)
 */
ttyspeed()
{
	struct sgttyb buf ;

	if( ioctl(fileno(stdout), (int)TIOCGETP, (char *)&buf) == -1 )
		return 2 ;
	else
		return Speeds[buf.sg_ospeed] ;
}

/*
 *	Simple help for screen display.
 */
help_screen()
{
	if( LINES < 15 )
		return ;

	wclear(wp) ;

	wmove(wp, 0, 0) ;

	wprintw(wp, "The commands recognized in screen mode are:\n\n") ;

	wprintw(wp, "\td  - Dump copy of the screen to a file.\n") ;
	wprintw(wp, "\tm  - \"Magnify\" a field.\n") ;
	wprintw(wp, "\tu  - \"Unmagnify\", return to the default screen.\n") ;
	wprintw(wp, "\tq  - Quit, which is the same as exit.\n") ;
	wprintw(wp, "\te  - Exit, which is the same as quit.\n") ;
	wprintw(wp, "\th  - This help screen.\n") ;
	wprintw(wp, "\t?  - This help screen.\n") ;
	wprintw(wp, "\tr  - Redraw the screen.\n") ;
	wprintw(wp, "\t^L - Redraw the screen.\n\n") ;

	wprintw(wp, "All commands cause the next sample to be taken and displayed.\n") ;

	continue_prompt() ;
}
