/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 replay.c
 *	Date:	 3/29/90
 *	Version: 1.45
 *
 *	replay.c - Functions to replay collected data.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)replay.c	1.45 (monitor) 3/29/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 *
 * Jan. 16, 1989 -- arr
 *
 *	Added skeleton of code needed to do floating point
 *	conversion between architectures.
 *
 * Feb. 10, 1989 -- arr
 *
 *	Added code to allocate space for BUF records.
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <curses.h>

#include <sys/types.h>
#include <sys/buf.h>
#include <sys/uio.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#include <net/if.h>
#include <netinet/in.h>

#include "include.h"
#include "options.h"
#include "monitor.h"
#include "extern.h"

/*
 *	Functions which don't return (int).
 */
char	*calloc() ;

void	free(),
	get_sample() ;

/*
 *	Module name for error functions.
 */
static	char	*module = "replay" ;

/*
 *	The record structure version number.
 */
extern	dev_t	record_version ;

/*
 *	Display information.
 */
extern	int	lines ;
extern	WINDOW	*wp ;

/*
 *	fd	    - This is the file descriptor of the data file.
 *	found_last  - We found a last record.  Nice to see, but of little
 *		      consequence.
 *	have_sample - Since the replay open function has to read the
 *		      first sample, to get device names, that record
 *		      was ordinarily lost.  By using this as a flag I
 *		      can now display that first record.  This makes the
 *		      "eatit" option obsolete.
 */
static	int	fd ;
static	int	found_last = 0 ;
static	int	have_sample = 0 ;

/*
 *	Open the data file and read the first record of the file.
 */
open_replay(op)
OPTION	*op ;
{
	if((fd = open(op->opt_file, O_RDONLY, 0)) == -1 )
		fatal("Can't open data file: %s (%s).\n", module, op->opt_file);

	if( read(fd, (char *)&first, MON$S_FIRST) == -1 )
		fatal("Can't read first record: %s.\n", module);

	if( first.mon_record != record_version )
		fatal("Record structure/data file version mismatch.\n", module) ;

	if( first.mon_type != MON$C_FIRST )
		panic("First record is not the right type!\n", module);

	/*
	 *	Do the standalone portion of replay startup.
	 */	
	first_record(op) ;
}

/*
 *	Given that a valid first record has been found, allocate
 *	the various structures that will be needed to store records
 *	and initialize any other data structures that have to setup.
 *
 *	This function will be written to be callable from almost
 *	ANYWHERE to setup a new replay session.
 */
first_record(op)
OPTION	*op ;
{
	n_cpu = first.mon_cpu ;
	n_disk = first.mon_disk ;
	n_netif = first.mon_netif ;
	n_buf = first.mon_buf ;

	/*
	 *	Turn off the sample record as part of the
	 *	the size calculations.  This no longer affects
	 *	magnification of the first/sample record.
	 */
	op->opt_data = first.mon_options & ~MON$M_SAMPLE ;

	/*
	 *	Disable magnify functions that don't have a correspond-
	 *	ing data option.  Starting a new sessions should reset
	 *	the magnify mask, from the original command line
	 *	options (I'll try to add this later).
	 */
	op->opt_magnify &= op->opt_data ;

	/*
	 *	If the sleep time hasn't been set from the command
	 *	line, set it to what was originally saved.
	 */
	if( !op->sleep_set )
		op->opt_sleep = first.mon_sleep ;

	/*
	 *	Initialize the tty and fork structures.  (paging may 
	 *	get included later)
	 */
	init_statics(op);

	/*
	 *	Allocates structure for netif's and cpu's.
	 */
	allocate_dynamics(op) ;

	/*
	 *	Setup an I/O vector for the readv(2) call, used in 
	 *	replay itself.
	 */
	init_iovec(op);

	/*
	 *	Get the next record and hope that it is a sample.  If
	 *	it is, then get the actual sample.
	 */
	if( get_record(fd) != MON$C_SAMPLE )
		fatal("Can't read first sample: %s.\n", module);

	get_sample(fd, iov, n_records) ;

	/*
	 *	If the architectures don't match do the floating
	 *	point conversions.
	 */
	if( architecture != first.mon_arch )
		convert_fp(op) ;

	/*
	 *	Set the sample flag so that replay() won't try to read one.
	 */
	have_sample = 1 ;
}

/*
 *	Read data samples to initialize the internal records.
 */
replay(op)
OPTION	*op ;
{
	register type ;

	/*
	 *	Clear the sample flag and return.
	 */
	if( have_sample ) {
		have_sample = 0 ;
		return MON_NORMAL ;
	}

/*
 *	Read the next record.
 */
	if((type = get_record(fd)) == -1 )
		return MON_EXIT ;

/*
 *	Depending on the type of record that we actually read:
 *
 *	Sample: Read the sample that occurs after it.
 *
 *	Last:   Make a note that we found the "last" record and
 *	return with MON_CONTINUE.
 *
 *	First:	We will give the user the option of replaying the 
 *	session that follows.
 *
 *	Anything else:  Panic!
 */
	switch( type ) {
	case MON$C_SAMPLE:
		get_sample(fd, iov, n_records) ;

		/*
		 *	If the architectures don't match do the floating
		 *	point conversions.
		 */
		if( architecture != first.mon_arch )
			convert_fp(op) ;

		break ;
	case MON$C_LAST:
		found_last = 1 ;
		return MON_CONTINUE ;
	case MON$C_FIRST:
		if( next_session(op) == 0 )
			return MON_EXIT ;

		found_last = 0 ;

		if( op->close_display )
			(*op->close_display)(op) ;

		free_totals(op);
		free_dynamics(op);

		restore_display(op);

		first_record(op) ;

		if( op->open_display )
			(*op->open_display)(op) ;

		return MON_CONTINUE ;
	default:
		panic("Unexpected data record!\n", module);
	}

	return MON_NORMAL ;
}

/*
 *	Free up virtual memory in reverse order from how it was
 *	allocated and generally clean up.
 */
close_replay(op)
OPTION	*op ;
{
	free_totals(op);

	free_dynamics(op) ;

	if( close(fd) == -1 )
		warning("Can't close data file: %s.\n", module);
}

/*
 *	Allocate space the dynamic records.
 */
allocate_dynamics(op)
OPTION	*op ;
{
	register char *p ;

	if( op->opt_cpu ) {
		if((p = calloc(n_cpu, MON$S_CPU)) == NULL )
			fatal("Can't allocate space for CPU structures: %s.\n", module);

		cpu = (struct mon_cpu *)p ;

		records[MON$C_CPU].addr = p ;
		records[MON$C_CPU].size = n_cpu * MON$S_CPU ;
	}

	if( op->opt_buf ) {
		if((p = calloc(n_buf, MON$S_BUF)) == NULL )
			fatal("Can't allocate space for BUF structures: %s.\n", module);

		mon_buf = (struct mon_buf *)p ;

		records[MON$C_BUF].addr = p ;
		records[MON$C_BUF].size = n_buf * MON$S_BUF ;
	}

	if( op->opt_netif ) {
		if((p = calloc(n_netif, MON$S_NETIF)) == NULL )
			fatal("Can't allocate space for netif structures: %s.\n", module);

		netif = (struct mon_netif *)p ;

		records[MON$C_NETIF].addr = p ;
		records[MON$C_NETIF].size = n_netif * MON$S_NETIF ;
	}

/*
 *	addr can probably be initialized statically.
 */
	if( op->opt_disk ) {
		records[MON$C_DISK].addr = (char *)disk ;
		records[MON$C_DISK].size = n_disk * MON$S_DISK ;
	}

	allocate_totals(op);
}

/*
 *	Free the space allocated for records.
 */
free_dynamics(op)
OPTION	*op ;
{
	if( op->opt_cpu )
		free((char *)cpu);

	if( op->opt_netif )
		free((char *)netif);
}

/*
 *	It has been determined that another session exists after
 *	the one we just finished.  Ask the user if he wants it.
 */
next_session(op)
OPTION	*op ;
{
	int	y = 0 ;

	if( op->opt_curses )  {
		wclear(wp) ;

		if( found_last )
			y = screen_last(y) ;

		screen_first(y) ;

		wmove(wp, LINES - 1, 0) ;
		wprintw(wp, "Do you want to continue? ");
		wrefresh(wp) ;

		if( wgetch(wp) == 'n' )
			return 0 ;
	}

	return 1 ;
}
