/*
 *	Author:  Alan Rollow, EIS/CXO, Digital Equipment Corp.
 *	File:	 pid.c
 *	Date:	 8/21/90
 *	Version: 1.30
 *
 *	pid.c - Read stuff in the proc table.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)pid.c	1.30 (monitor) 8/21/90" ;
#endif

/*
 * Modification History
 * 
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 *
 * 29-June-1988 -- arr
 *
 *	Removed code for close_pid() and changed open_pid() to
 *	setup_proc().  Some of the open_pid() was moved to open_live()
 *	in live.c.
 *
 *	Changed collect_pid() so that it could be called from live().
 *
 * 30-November-1988 -- arr
 *
 *	First pass at reading on the user structure on the PMAX.  It's
 *	ugly, but it works.
 * 
 *  1-December-1988 -- arr
 *
 *	After close examination of the VAX and MAX code I was
 *	able to reduce the architecture specific pieces to a
 *	mininium and isolate them as much as possible.
 *
 * Feb. 15, 1989 -- arr
 *
 *	Remove the pgtok() and btok() macros.  This use will the
 *	function version.
 *
 * November, 1989 -- arr
 *
 *	Changes for V4.0.
 *
 * Dec. 25, 1989 -- arr
 *
 *	Add a private namelist (pid_nm).
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>
#include <curses.h>
#include <pwd.h>
#include <grp.h>

#include <sys/types.h>
#include <sys/dk.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>
#include <sys/vmmac.h>

#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/param.h>
#include <sys/user.h>
#include <sys/proc.h>

#include <net/if.h>
#include <netinet/in.h>

#include <machine/pte.h>

#include "monitor.h"
#include "include.h"
#include "options.h"
#include "extern.h"

/*
 *	Module name for error functions.
 */
static	char	*module = "pid" ;

/*
 *	Declarations for functions that don't return (int).
 */
char	*str_flags(),
	*str_stats(),
	*str_timeval(),
	*strncpy(),
	*strcpy() ;

void	nlist() ;

/*
 *	Local copies of kernel stuff.
 */
static	int		nproc ;
static	struct proc 	proc ;
static	struct user	user ;
static	struct pte	*Usrptmap, *usrpt ;

#define	PID_NM_PROC	(0)
#define	PID_NM_NPROC	(1)
#define	PID_NM_USRPTMAP	(2)
#define	PID_NM_USRPT	(3)

struct nlist pid_nm[] = {
	{ "_proc" },		/* PID_NM_PROC */
	{ "_nproc" },		/* PID_NM_NPROC */
	{ "_Usrptmap" },	/* PID_NM_USRPTMAP */
	{ "_usrpt" },		/* PID_NM_USRPT */
	{ 0 }
} ;

/*
 *	Remember what the last {real,effective} uid and gid were so
 *	that I don't have to look them up at every sample.
 */
int	prev_uid = -1,
	prev_ruid = -1,
	prev_gid = -1,
	prev_rgid = -1 ;

/*
 *	File descriptors declared in data.c
 */
extern	int	mem, swap ;

/*
 *	Default display.
 */
extern	int	lines ;			/* length of the display being used. */
extern	WINDOW	*wp ;

/*
 *	Called from open_live().  This finishs the setup for getting
 *	information about a process.
 */
setup_proc(op)
OPTION	*op ;
{
	int	i ;
	struct proc *p ;

	/*
	 *	This is the address of the interesting proc 
	 *	structure.  If it's still 0 after walking
	 *	though the proc table then we didn't find 
	 *	the process we wanted.
	 */
	mon_pid.mon_proc = 0 ;

	/*
 	 *	Initialize and check the namelist.
	 */
	(void)nlist(op->opt_kernel, pid_nm) ;

	if( check_namelist_pid() == 0 ) {
		info("Disabling process-by-id data collection.\n", module) ;
		op->opt_pid = 0 ;
		return ;
	}

	/*
	 *	Get the address of the user page table map and
	 *	the user page table (I guess, this was taken
	 *	from ps(1)).
	 */
	Usrptmap = (struct pte *)pid_nm[PID_NM_USRPTMAP].n_value ;
	usrpt = (struct pte *)pid_nm[PID_NM_USRPT].n_value ;

	/*
	 *	Get the value of nproc and the address of the
	 *	proc table.
	 */
	readk((long)pid_nm[PID_NM_NPROC].n_value, (char *)&nproc, sizeof(int)) ;
	readk((long)pid_nm[PID_NM_PROC].n_value, (char *)&p, sizeof(struct proc *)) ;

	/*
	 *	While reading the entire proc table at one shot might
	 *	be more efficient this method (one at a time) will be
	 *	to change if Ultrix goes to a dynamic proc list.
	 */
	for(i = 0; i < nproc; i++) {
		readk((long)(p + i), (char *)&proc, sizeof(proc)) ;

		if( proc.p_stat == 0 )
			continue ;

		if( proc.p_pid == op->opt_proc_id ) {
			mon_pid.mon_proc = p + i ;
			break ;
		}
	}

	if( mon_pid.mon_proc == 0 )
		fatal("Process not in table.\n", module) ;
}

/*
 *	This is called from live() to collect information about
 *	a process.
 */
collect_pid(op)
OPTION	*op ;
{
	register struct mon_ucred *cp ;

	/*
	 *	Read the current contents of the proc structure.
	 */
	readk((long)mon_pid.mon_proc, (char *)&proc, sizeof(proc)) ;

	/*
	 *	If the pid of this proc entry is not the one we're
	 *	watching, we're done.
	 */
	if((mon_pid.mon_pid = proc.p_pid) != op->opt_proc_id )
		return MON_EXIT ;

	/*
	 *	Save the process status first.
	 */
	mon_pid.mon_stat = proc.p_stat ;

	/*
	 *	If the process is a zombie, ignore it, but keep running.
	 */
	if( proc.p_stat == SZOMB )
		return MON_NORMAL ;

	/*
	 *	Save the other interesting information from the
	 *	proc structure.
	 */
#ifdef	V4_ULTRIX
	mon_pid.mon_pflag = proc.p_type | proc.p_sched ;
#else
	mon_pid.mon_pflag = proc.p_flag ;
#endif
	mon_pid.mon_rssize = proc.p_rssize ;
	mon_pid.mon_pri = proc.p_pri ;
	mon_pid.mon_nice = proc.p_nice ;
	mon_pid.mon_pgrp = proc.p_pgrp ;
	mon_pid.mon_wchan = proc.p_wchan ;

	if( read_user() == -1 ) {
		info("User structure not available.\n", module) ;
		return MON_NORMAL ;
	}

	(void)strcpy(mon_pid.mon_comm, user.u_comm) ;

#ifdef	vax
	mon_pid.mon_cpundx = user.u_pcb.pcb_cpundx ;
#elif	mips
	mon_pid.mon_cpundx = 0 ;
#endif
	mon_pid.mon_self = user.u_ru ;
	mon_pid.mon_child = user.u_cru ;
	mon_pid.mon_ttyd = user.u_ttyd ;

	readk((long)user.u_cred, (char *)&mon_pid.mon_cred, sizeof(struct ucred)) ;
	cp = &mon_pid.mon_cred ;

	/*
	 *	Look up the names.
	 */
	if( prev_uid != cp->cr_uid )
		uid_string(prev_uid = cp->cr_uid, mon_pid.mon_username) ;

	if( prev_ruid != cp->cr_ruid )
		uid_string(prev_ruid = cp->cr_ruid, mon_pid.mon_realuser) ;

	if( prev_gid != cp->cr_gid )
		gid_string(prev_gid = cp->cr_gid, mon_pid.mon_groupname) ;

	if( prev_rgid != cp->cr_rgid )
		gid_string(prev_rgid = cp->cr_rgid, mon_pid.mon_realgroup) ;


	return MON_NORMAL ;
}

/*
 *	Read the user structure and put a copy into *user.  If
 *	user structure is swapped then read it directly into
 *	*user from the swap device.  Otherwise we have to perform
 *	ritual of arcane magick to get the physical address and
 *	read it from /dev/mem.  I will not attempt to document
 *	the ritual except to copy comments as were provided in
 *	the great grimores known as ps.c and pstat.c.
 *
 *	For some reason unknown to me at the moment, this function
 *	returns a 0 upon success, but doesn't return an error on
 *	failure.  Instead it causes the program to exit via readk(),
 *	readm() or reads().  I'll fix this in a future hack.
 */
read_user()
{
	union {
		struct user user;
		char upages[UPAGES+CLSIZE][NBPG];
	} un_user;

	register i, ncl, addr ;
	int ublkno;

#ifdef	vax
	struct pte wpte[UPAGES+CLSIZE];
#elif	mips
	struct pte wpte[UPAGES] ;
#endif

#ifdef	V4_ULTRIX
	struct dmap l_dmap;
#endif

	/*
	 *	If the user structure is on the swap device
	 *	we can read it easily based on the address
	 *	in the proc structure.  Only to simplify
	 *	nesting in the code below we'll return as
	 *	soon as it's read.
	 *
	 *	The ritual for getting the swap space address
	 *	became more complicated in V4.0.
	 */
#ifdef	V4_ULTRIX
	if((proc.p_sched & SLOAD) == 0 ) {
		readk(proc.p_smap, (char *)&l_dmap, sizeof(struct dmap)) ;

		readk(l_dmap.dm_ptdaddr, (char *)&ublkno, sizeof(int)) ;
#else
	if((proc.p_flag & SLOAD) == 0 ) {
		ublkno = proc.p_swaddr ;
#endif
		addr = dtob(ublkno) ;

		reads((long)addr, (char *)&user, sizeof(struct user)) ;

		return 0 ;
	}

	/*
	 *	Architecture specific versions of this ritual are
	 *	provided below.
	 */
	addr = get_uaddr(&proc) ;

#ifdef	vax
	readm((long)addr, (char *)wpte, sizeof(wpte)) ;
#elif	mips
	readk((long)addr, (char *)wpte, sizeof(wpte)) ;
#endif

	ncl = (sizeof(struct user) + NBPG*CLSIZE - 1) / (NBPG*CLSIZE);

	/*
	 *	Back fill the union un_user with the clicks of
	 *	of the user structure.
	 */
	while (--ncl >= 0) {
		i = ncl * CLSIZE;
#ifdef	vax
		addr = ctob(wpte[CLSIZE+i].pg_pfnum) ;
#elif	mips
		addr = ctob(wpte[i].pg_pfnum) ;
#endif
		readm((long)addr, un_user.upages[i], CLSIZE*NBPG) ;
	}

	user = un_user.user ;

	return 0 ;
}

/*
 *	Architecture specific rituals for getting the physical
 *	address of the user structure.  It is interesting to
 *	note that the VAX ritual is a complex ritual more suited 
 *	to the Cabbalist or Magick Square mystic, while the PMAX 
 *	ritual is a simple Word of Power.
 */
#ifdef	vax
get_uaddr(p)
struct proc *p ;
{
	struct pte *pteaddr, apte;

	pteaddr = &Usrptmap[btokmx(p->p_p0br) + p->p_szpt - 1];

	readk((long)pteaddr, (char *)&apte, sizeof(apte)) ;

	return ctob(apte.pg_pfnum+1) - (UPAGES+CLSIZE) * sizeof(struct pte) ;
}
#elif	mips
get_uaddr(procp)
struct proc *procp ;
{
	return (int)procp->p_addr ;
}
#endif
/*
 *	Data for delta function.
 */
int	pid_ticks[2],
	soft_faults[2],
	hard_faults[2],
	in_block[2],
	out_block[2],
	send_message[2],
	rcv_message[2] ;

/*
 *	Calculate the changes in process (by id) stats.
 *
 *	"Which" is one of {CHILD,SELF} depending on whether the
 *	child or parent stats are being used.
 */
double	delta_pid(ticks, which)
int	ticks ;
int	which ;
{
	double	 etime ;
	register tmp ;
	register struct rusage *rp ;

	rp = which == SELF ? &mon_pid.mon_self : &mon_pid.mon_child ;

	etime = (double)(ticks - pid_ticks[which])/first.mon_hz ;
	pid_ticks[which] = ticks ;

	tmp = rp->ru_minflt ;
	      rp->ru_minflt -= soft_faults[which] ;
	      soft_faults[which] = tmp ;

	tmp = rp->ru_majflt ;
	      rp->ru_majflt -= hard_faults[which] ;
	      hard_faults[which] = tmp ;

	tmp = rp->ru_inblock ;
	      rp->ru_inblock -= in_block[which] ;
	      in_block[which] = tmp ;

	tmp = rp->ru_oublock ;
	      rp->ru_oublock -= out_block[which] ;
	      out_block[which] = tmp ;

	tmp = rp->ru_msgsnd ;
	      rp->ru_msgsnd -= send_message[which] ;
	      send_message[which] = tmp ;

	tmp = rp->ru_msgrcv ;
	      rp->ru_msgrcv -= rcv_message[which] ;
	      rcv_message[which] = tmp ;

	return etime ;
}

/*
 *	Functions to turn {u,g}id's into names.
 */
gid_string(gid, buf)
char	*buf ;
{
	struct group *gp, *getgrgid() ;

	buf[MON$S_GROUPNAME] = '\0' ;

	if((gp = getgrgid(gid)) == NULL )
		(void)sprintf(buf, "#%d", gid) ;
	else
		(void)strncpy(buf, gp->gr_name, MON$S_GROUPNAME) ;
}

uid_string(uid, buf)
char	*buf ;
{
	struct passwd *up, *getpwuid() ;

	buf[MON$S_USERNAME] = '\0' ;

	if((up = getpwuid(uid)) == NULL )
		(void)sprintf(buf, "#%d", uid) ;
	else
		(void)strncpy(buf, up->pw_name, MON$S_USERNAME) ;
}

/*
 *	Constants for the placement of stuff on the screen.
 *
 *	Offsets within the line.
 */
#define	PID_OFFSET	(14)
#define	PID_PRI		(39)
#define	PID_NICE	(52)
#define	PID_TTYD	(68)
#define	PID_CPU_X	(10)
#define	PID_FAULT	(36)
#define	PID_BLOCK	(50)
#define	PID_MESSAGE	(69)
#define	PID_SWAP	(0)
#define	PID_SIG		(14)
#define	PID_MAXRSS	(29)
#define	PID_ID		(46)
#define	PID_CSW		(46)
/*
 *	Vertical offsets.  (what line is it on...)
 */
#define	PID_COMM	(3)
#define	PID_WCHAN	(PID_COMM+1)
#define	PID_FLAG	(PID_COMM+2)
#define	PID_STATE	(PID_COMM+3)
#define	PID_RSSIZE	(9)
#define	PID_CPU		(11)
#define	PID_USER	(PID_CPU+1)
#define	PID_SYSTEM	(PID_CPU+2)
#define	PID_SINGLE	(15)
/*
 *	Widths of fields for building headers.
 */
#define	STATE_WIDTH	(28)
#define	PRI_WIDTH	(16)
#define	NICE_WIDTH	(15)
#define	TTYD_WIDTH	(5)
#define	CPU_WIDTH	(28)
#define	FAULT_WIDTH	(16)
#define	BLOCK_WIDTH	(15)
#define	SWAP_WIDTH	(13)
#define	SIG_WIDTH	(14)
#define	ID_WIDTH	(28)
#define	RSSIZE_WIDTH	(13)
#define	MAXRSS_WIDTH	(16)
#define INDEX_WIDTH	(14)

#define	PID_LINES	(18)

/*
 *	Clear the screen and print the headers for process data.
 *
 *	This function will also look-up the names corresponding to
 *	the user and group id's.
 */
redraw_pid()
{
	lines = PID_LINES ;

	wclear(wp) ;

	wmove(wp, 0, 0);

	sample_header() ;

	wmove(wp, PID_COMM, 0) ;
	wprintw(wp, "Command:\nWChan:\nFlags:\n") ;
	wrefresh(wp) ;

	wprintw(wp, "%-*.*s %-*.*s %-*.*s %-*.*s",
		STATE_WIDTH, STATE_WIDTH, "State:",
		PRI_WIDTH,   PRI_WIDTH,   "Priority:",
		NICE_WIDTH,  NICE_WIDTH,  "Nice:",
		TTYD_WIDTH,  TTYD_WIDTH,  "ttyd:") ;

	wmove(wp, PID_RSSIZE - 1, 0) ;
	wprintw(wp, "%-*.*s %-*.*s Effective ID:",
		RSSIZE_WIDTH, RSSIZE_WIDTH, "RSSize (Kb):",
		INDEX_WIDTH, INDEX_WIDTH, "CPU Index:") ;

	wmove(wp, PID_RSSIZE, 0) ;
	wprintw(wp, "%*.*s Real ID:", ID_WIDTH, ID_WIDTH, "") ;

	wmove(wp, PID_CPU, 0) ;
	wprintw(wp, "%-*.*s %-*.*s %-*.*s Messages:",
		CPU_WIDTH,   CPU_WIDTH, "CPU time:",
		FAULT_WIDTH, FAULT_WIDTH, "Page Faults:",
		BLOCK_WIDTH, BLOCK_WIDTH, "Block I/O:");

	wmove(wp, PID_USER, 0) ;
	wprintw(wp, "%-*.*s %-*.*s %-*.*s (Send)",
		CPU_WIDTH,   CPU_WIDTH, "(User)",
		FAULT_WIDTH, FAULT_WIDTH, "(Hard)",
		BLOCK_WIDTH, BLOCK_WIDTH, "(I)") ;

	wmove(wp, PID_SYSTEM, 0) ;
	wprintw(wp, "%-*.*s %-*.*s %-*.*s (Rcv.)",
		CPU_WIDTH,   CPU_WIDTH, "(System)",
		FAULT_WIDTH, FAULT_WIDTH, "(Soft)",
		BLOCK_WIDTH, BLOCK_WIDTH, "(O)") ;

	wmove(wp, PID_SINGLE, 0) ;
	wprintw(wp, "%-*.*s %-*.*s %-*.*s Context Switches",
		SWAP_WIDTH, SWAP_WIDTH, "Swaps:",
		SIG_WIDTH, SIG_WIDTH, "Signals:",
		MAXRSS_WIDTH, MAXRSS_WIDTH, "MAXrss (Kb):");
}

/*
 *	Print the data for the process.
 */
magnify_pid()
{
	register struct rusage *rp = &mon_pid.mon_self ;
	double	 etime ;

	etime = delta_pid(sample.mon_ticks, SELF) ;

	sample_body(etime) ;

	wmove(wp, PID_COMM, PID_OFFSET) ;
	wprintw(wp, "%s", mon_pid.mon_comm) ;
	wclrtoeol(wp) ;

	wmove(wp, PID_WCHAN, PID_OFFSET) ;
	wprintw(wp, "%x", mon_pid.mon_wchan) ;
	wclrtoeol(wp) ;

	wmove(wp, PID_FLAG, PID_OFFSET) ;
	wprintw(wp, "<%s>", str_flags(mon_pid.mon_pflag)) ;
	wclrtoeol(wp) ;

	/*
	 *	Stuff that goes on the STATE line.
	 */
	wmove(wp, PID_STATE, PID_OFFSET) ;
	wprintw(wp, "%-9s", str_stats(mon_pid.mon_stat)) ;

	wmove(wp, PID_STATE, PID_PRI) ;
	wprintw(wp, "%3d", mon_pid.mon_pri) ;

	wmove(wp, PID_STATE, PID_NICE) ;
	wprintw(wp, "%3d", mon_pid.mon_nice) ;

	wmove(wp, PID_STATE, PID_TTYD) ;
	wprintw(wp, "<%d,%d>", major(mon_pid.mon_ttyd), minor(mon_pid.mon_ttyd)) ;
	wclrtoeol(wp) ;

	/*
	 *	Prin the user ID and resident set size.
	 */
	wmove(wp, PID_RSSIZE-1, PID_ID) ;
	wprintw(wp, "<%s,%s>", mon_pid.mon_username, mon_pid.mon_groupname) ;
	wclrtoeol(wp) ;

	wmove(wp, PID_RSSIZE, PID_ID) ;
	wprintw(wp, "<%s,%s>", mon_pid.mon_realuser, mon_pid.mon_realgroup) ;
	wclrtoeol(wp) ;

	wmove(wp, PID_RSSIZE, 0) ;
	wprintw(wp, "%-*d %-10d", RSSIZE_WIDTH, pgtok(mon_pid.mon_rssize),
		mon_pid.mon_cpundx) ;

	/*
	 *	Stuff that goes on the "user" line.
	 */
	wmove(wp, PID_USER, PID_CPU_X) ;
	wprintw(wp, "%s", str_timeval(&rp->ru_utime)) ;

	wmove(wp, PID_USER, PID_FAULT) ;
	wprintw(wp, "%-7.2f", rp->ru_majflt / etime) ;

	wmove(wp, PID_USER, PID_BLOCK) ;
	wprintw(wp, "%-7.2f", rp->ru_inblock / etime ) ;

	wmove(wp, PID_USER, PID_MESSAGE) ;
	wprintw(wp, "%-7.2f", rp->ru_msgsnd / etime ) ;

	/*
	 *	Stuff that goes on the "system" line.
	 */
	wmove(wp, PID_SYSTEM, PID_CPU_X) ;
	wprintw(wp, "%s", str_timeval(&rp->ru_stime)) ;

	wmove(wp, PID_SYSTEM, PID_FAULT) ;
	wprintw(wp, "%-7.2f", rp->ru_minflt / etime ) ;

	wmove(wp, PID_SYSTEM, PID_BLOCK) ;
	wprintw(wp, "%-7.2f", rp->ru_oublock / etime ) ;

	wmove(wp, PID_SYSTEM, PID_MESSAGE) ;
	wprintw(wp, "%-7.2f", rp->ru_msgrcv / etime ) ;

	/*
	 *	The leftovers.
	 */
	wmove(wp, PID_SINGLE + 1, PID_SWAP) ;
	wprintw(wp, "%-10d", rp->ru_nswap) ;

	wmove(wp, PID_SINGLE + 1, PID_SIG) ;
	wprintw(wp, "%-10d", rp->ru_nsignals) ;

	wmove(wp, PID_SINGLE + 1, PID_MAXRSS) ;
	wprintw(wp, "%-10d", pgtok((int)rp->ru_maxrss)) ;

	wmove(wp, PID_SINGLE + 1, PID_CSW) ;
	wprintw(wp, "%-10d", rp->ru_nvcsw + rp->ru_nivcsw) ;
}

/*
 *	Dump the contents of the PID structure.
 */
f_pid(p)
struct mon_pid *p ;
{
	if((p->mon_flag & MON$M_VALID) == 0 )
		return ;

	printf("%s.\n", records[MON$C_PID].string) ;

	printf("\tpid:       %d\n", p->mon_pid) ;
	printf("\tpgrp:      %d\n", p->mon_pgrp ) ;
	printf("\tpri:       %d\n", p->mon_pri ) ;
	printf("\tnice:      %d\n", p->mon_nice ) ;
	printf("\twchan:     %x\n", p->mon_wchan) ;
	printf("\tttyd:      <%d,%d>\n", major(p->mon_ttyd), minor(p->mon_ttyd)) ;
	printf("\tpflag:     <%s>\n", str_flags(p->mon_pflag)) ;
	printf("\tstat:      %s\n", str_stats(p->mon_stat)) ;
	printf("\tcomm:      %s\n", p->mon_comm) ;
	printf("\trssize:    %d Kb.\n", pgtok(p->mon_rssize)) ;

	printf("\tcpundx:    %x\n", p->mon_cpundx) ;

	printf("\tusername:  %s\n", mon_pid.mon_username) ;
	printf("\trealuser:  %s\n", mon_pid.mon_realuser) ;
	printf("\tgroupname: %s\n", mon_pid.mon_groupname) ;
	printf("\trealgroup: %s\n", mon_pid.mon_realgroup) ;

	printf("\tself:\n") ;
	print_rusage(&p->mon_self, "\t\t") ;

	printf("\tchild:\n") ;
	print_rusage(&p->mon_child, "\t\t") ;

	printf("\tcred:\n") ;
	print_cred(&p->mon_cred, "\t\t") ;
}

/*
 *	Check the namelist.
 */
check_namelist_pid()
{
	if( pid_nm[PID_NM_PROC].n_value == 0 ) {
		info("There is no process table.\n", module) ;
		return 0 ;
	}

	if( pid_nm[PID_NM_NPROC].n_value == 0 ) {
		info("Can't find out how big the process table is.\n", module) ;
		return 0 ;
	}

	if( pid_nm[PID_NM_USRPTMAP].n_value == 0 ) {
		info("There is no user page table map.\n", module) ;
		return 0 ;
	}

	if( pid_nm[PID_NM_USRPT].n_value == 0 ) {
		info("There is no user page table.\n", module) ;
		return 0 ;
	}

	return 1 ;
}
