/*
 *	Author:  Alan Rollow, CSC/CS, Digital Equipment Corp.
 *	File:	 delta.c
 *	Date:	 8/22/90
 *	Version: 1.27
 *
 *	delta.c - The functions here calculate the differences between
 *	the current copy of a data structure and the previous copy.
 *
 *	The delta_ functions for tty's and fork's have moved into
 *	os.c.  The others will stay here, because pointers to "total"
 *	structure and their allocation functions need to be in
 *	one place.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)delta.c	1.27 (monitor) 8/22/90" ;
#endif

/*
 * Modification History
 *
 * 27-June-1988 -- arr
 *
 *	Change include of monitor.h to include.h.
 *
 *	Change include of record.h to monitor.h.
 *
 * Feb. 17, 1989 -- arr
 *
 *	Added mon_soft and mon_tlbpid to delta_cpu().
 *
 * Mar. 26, 1990 -- arr
 *
 *	Added hack to work-around DECmumble include file problem.
 *
 * Aug. 22, 1990 -- arr
 *
 *	Added delta_syscall_intr() that helps fix a problem for
 *	magnify_cpu() where the number of system calls and device
 *	interrupts is too high. 
 */

#include <nlist.h>
#include <stdio.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/buf.h>
#include <sys/file.h>
#include <sys/socket.h>
#include <sys/dk.h>
#include <sys/param.h>
#include <sys/dir.h>

#if defined(V4_ULTRIX) && defined(mips)
#	include <mips/cpu.h>
#endif

#include <sys/user.h>
#include <sys/vmsystm.h>
#include <sys/vmmeter.h>

#include <net/if.h>
#include <netinet/in.h>

#include "monitor.h"
#include "include.h"
#include "options.h"
#include "extern.h"

/*
 *	Module name for error functions.
 */
static	char	*module = "delta" ;

/*
 *	Functions that don't return (int).
 */
void	free() ;

char	*calloc() ;

/*
 *	Define the structures and other convient names of the various
 *	total structures.
 */
typedef	struct	tot_netif {
	int	t_ticks ;		/* clock ticks since last sample */
	int	t_ipackets,
		t_opackets,
		t_ierrors,
		t_oerrors,
		t_collisions ;
} T_NETIF ;

typedef	struct	tot_cpu {
	int	t_ticks ;		/* clock ticks since last sample */
	int	t_states[CPUSTATES],
		t_swtch,
		t_trap,
		t_syscall,
		t_intr,
		t_pdma,
		t_tlbpid,
		t_soft ;
} T_CPU ;

#define	TOT$S_CPU	(sizeof(T_CPU))
#define	TOT$S_NETIF	(sizeof(T_NETIF))

static	struct tot_cpu *t_cpu = 0 ;
static	struct tot_netif *t_netif = 0 ;

/*
 *	How much did the network counters change...
 */
double	delta_netif(index, ticks)
register index, ticks ;
{
	register tmp ;
	double	 etime ;

	etime = (double)(ticks - t_netif[index].t_ticks)/first.mon_hz ;
	t_netif[index].t_ticks = ticks ;

	tmp = netif[index].mon_collisions ;
	      netif[index].mon_collisions -= t_netif[index].t_collisions ;
	      t_netif[index].t_collisions = tmp ;

	tmp = netif[index].mon_ipackets ;
	      netif[index].mon_ipackets -= t_netif[index].t_ipackets ;
	      t_netif[index].t_ipackets = tmp ;

	tmp = netif[index].mon_opackets ;
	      netif[index].mon_opackets -= t_netif[index].t_opackets ;
	      t_netif[index].t_opackets = tmp ;

	tmp = netif[index].mon_ierrors ;
	      netif[index].mon_ierrors -= t_netif[index].t_ierrors ;
	      t_netif[index].t_ierrors = tmp ;

	tmp = netif[index].mon_oerrors ;
	      netif[index].mon_oerrors -= t_netif[index].t_oerrors ;
	      t_netif[index].t_oerrors = tmp ;

	return etime ;
}

/*
 *	Calculate the total elapsed cpu time.
 *
 *	For V2.0 this does calculate the change in context switches.
 */
double	delta_cpu(index, ticks, dp)
int	index, ticks ;
double	*dp ;
{
	register i, tmp, total = 0 ;
	double	 etime ;

	etime = (double)(ticks - t_cpu[index].t_ticks) / first.mon_hz ;
	t_cpu[index].t_ticks = ticks ;

	for(i = 0; i < CPUSTATES; i++) {
		tmp = cpu[index].mon_time[i] ;
		      cpu[index].mon_time[i] -= t_cpu[index].t_states[i] ;
		      t_cpu[index].t_states[i] = tmp ;

		total += cpu[index].mon_time[i] ;
	}

	if( total == 0 )
		*dp = 1.0 ;
	else
		*dp = (double)total ;

	tmp = cpu[index].mon_swtch ;
	      cpu[index].mon_swtch -= t_cpu[index].t_swtch ;
	      t_cpu[index].t_swtch = tmp ;

	tmp = cpu[index].mon_trap ;
	      cpu[index].mon_trap -= t_cpu[index].t_trap ;
	      t_cpu[index].t_trap = tmp ;

	tmp = cpu[index].mon_syscall ;
	      cpu[index].mon_syscall -= t_cpu[index].t_syscall ;
	      t_cpu[index].t_syscall = tmp ;

	tmp = cpu[index].mon_intr ;
	      cpu[index].mon_intr -= t_cpu[index].t_intr ;
	      t_cpu[index].t_intr = tmp ;

	tmp = cpu[index].mon_pdma ;
	      cpu[index].mon_pdma -= t_cpu[index].t_pdma ;
	      t_cpu[index].t_pdma = tmp ;

	tmp = cpu[index].mon_tlbpid ;
	      cpu[index].mon_tlbpid -= t_cpu[index].t_tlbpid ;
	      t_cpu[index].t_tlbpid = tmp ;

	tmp = cpu[index].mon_soft ;
	      cpu[index].mon_soft -= t_cpu[index].t_soft ;
	      t_cpu[index].t_soft = tmp ;

	return etime ;
}

/*
 *	Local data for delta_syscall_intr().
 */
static	int	prev_syscall = 0,
		prev_intr = 0 ;

/*
 *	This keeps track of the change in the total number
 *	of system calls and device interrupts.  We don't
 *	what the values are in screen(), so we'll check for
 *	NULL pointers.
 */
void	delta_syscall_intr(sp, ip)
int	*sp, *ip ;
{
	register i, total_syscall, total_intr, tmp ;

	total_syscall = 0 ;
	total_intr = 0 ;

	for(i = 0; i < first.mon_cpu; i++) {
		if((cpu[i].mon_flag & MON$M_VALID) == 0 )
			continue ;

		total_syscall += cpu[i].mon_syscall ;
		total_intr    += cpu[i].mon_intr ;
	}

	tmp = total_syscall ;
	      total_syscall -= prev_syscall ;
	      prev_syscall = tmp ;

	tmp = total_intr ;
	      total_intr -= prev_intr ;
	      prev_intr = tmp ;

	if( sp )
		*sp = total_syscall ;

	if( ip )
		*ip = total_intr ;
}

/*
 *	Allocate space for the various total structures.
 */
allocate_totals(op)
OPTION	*op ;
{
	if( op->opt_cpu ) {
		prev_syscall = 0 ;
		prev_intr = 0 ;

		if((t_cpu = (T_CPU *)calloc(n_cpu, TOT$S_CPU)) == NULL )
			fatal("No space for cpu totals: %s.\n", module);
	}

	if( op->opt_netif )
		if((t_netif = (T_NETIF *)calloc(n_netif, TOT$S_NETIF)) == NULL )
			fatal("No space for netif totals: %s.\n", module);
}

/*
 *	Free the space allocated in allocate_totals().
 */
free_totals(op)
OPTION	*op ;
{
	if( op->opt_cpu )
		free((char *)t_cpu);

	if( op->opt_netif )
		free((char *)t_netif);
}
