/*
 *	Author:  Alan Rollow, EIS/CXO, Digital Equipment Corp.
 *	File:	 curses.c
 *	Date:	 5/23/90
 *	Version: 1.5
 *
 *	curses.c - Functions to handle curses style windows as opposed
 *	X style windows.
 */
#ifndef	lint
static	char	SccsId[] = "@(#)curses.c	1.5 (monitor) 5/23/90" ;
#endif

/*
 * Modification history.
 *
 * May 23, 1990 -- arr
 *
 *	Added curses function to "pause" screen.
 *
 * 28-December-1988 -- arr
 *
 *	Dust off V3.0 lint.
 */

#include <sys/types.h>
#include <sys/ioctl.h>

#include <stdio.h>
#include <signal.h>
#include <errno.h>
#include <curses.h>

#include "options.h"

/*
 *	Display data.
 *
 *	lines - Length of display actually being used.  This will be < LINES.
 *	cols  - Width of the display being used.  It will usually be == COLS.
 *	wp    - Address of a curses window where the drawing will take place.
 *	changed_size - A flag set when a window changes size.
 *
 *	Declared as extern in <curses.h>.
 *
 *	LINES - The actual length of the display area.  It will be set before
 *		initscr() is called.
 *	COLS  - Width of the display, like LINES.
 */
int	lines = 0 ;
int	cols = 0 ;
int	changed_size = 0 ;
WINDOW	*wp = (WINDOW *)NULL ;

/*
 *	Module name for error functions.
 */
static	char	*module = "curses" ;

/*
 *	Stuff defined elsewhere.
 */
extern	OPTION	option ;

/*
 *
 */
setup_curses(op, max_lines, max_cols)
OPTION	*op ;
int	max_lines, max_cols ;
{
	struct winsize size ;

	/*
	 *	If getwindow size doesn't fail set LINES and COLS.
	 */
	if( getwindowsize(&size) != -1 ) {
		LINES = size.ws_row ;
		COLS  = size.ws_col ;
	}
	else if( max_lines != -1 )
		LINES = max_lines ;
	else if( max_cols != -1 )
		COLS  = max_cols ;

	if( initscr() == NULL )
		fatal("Can't initialize screen: %s.\n", module);

	lines = LINES ;
	cols  = COLS ;

	if((wp = newwin(lines, cols, 0, 0)) == NULL )
		fatal("Can't create window.\n", module) ;

	op->opt_curses = 1 ;

	crmode();
	noecho();
}

/*
 *	Catch notice of the fact that the screen changed size, then
 *	close your eyes (you don't want to see this...).
 *
 *	1.  Turn off curses so that it looks like we're starting
 *	    over from scratch.
 *
 *	2.  Use setup_curses() to size the terminal windows and
 *	    create a display area.
 *
 *	3.  Set the changed_size flag.
 *
 *	4.  If there is a function to redraw the display, call it.
 *
 *	5.  Open your eyes.
 */
void	catch_sigwinch()
{
	endwin() ;

	setup_curses(&option, -1, -1) ;

	changed_size = 1 ;

	if( option.redraw_display )
		(*option.redraw_display)(&option);
}

/*
 *	Use the "get window size" ioctl() to:
 *
 *	a.  If we are on a sizeing terminal, get the window
 *	    size.
 *
 *	b.  If we aren't on a sizeing terminal, punt...
 */
getwindowsize(w)
struct winsize *w ;
{
	if( ioctl(fileno(stdin), (int)TIOCGWINSZ, (char *)w) == -1 ) {
		warning("Ioctl TIOCGWINSZ failed: %s.\n", module) ;
		return -1 ;
	}

	if( w->ws_xpixel == 0 || w->ws_ypixel == 0 ) {
		info("Window size is 0 x 0.\n", module) ;
		errno = EOPNOTSUPP ;
		return -1 ;
	}

	return 0 ;
}

curses_pause()
{
	/*
	 *	Move to the bottom of the screen and print
	 *	the message.
	 */
	wmove(wp, LINES - 1, 0) ;
	wprintw(wp, "Pausing; Press any key to continue...");
	wrefresh(wp) ;

	/*
	 *	Wait for a character to be typed.  We don't care what
	 *	it is.
	 */
	(void)wgetch(wp) ;
}
