/* Copyright (c) 1999, 2000 by Kevin Forchione.  All Rights Reserved. */
/*
 *  TADS ADV.T/STD.T LIBRARY EXTENSION
 *  SENSE.T				
 *  version 3.2
 *
 *	    sense.t provides an enhancement of the "visibility" and 
 *      "reachability" concepts established in ADV.T. The scope() 
 *      function will return a list of all objects which are valid 
 *      for a vantage for a particular filter.
 *
 *  MODULE SUMMARY
 *
 *       1. Setting Up Constants
 *       2. Defining Access Rules For PASS-ACROSS-CONTENTS Validation
 *       3. Path Determination and Information
 *       4. Path Validation and Scope Testing
 *       5. Command Storage and Validation
 *       6. Disambiguation and Command Phase Accessibility Error Handling
 *       7. Describing Objects
 *       8. Describing Locations
 *       9. Object Reactions
 *      10. Adding Sense-passing To ADV.T Classes
 *      11. Redefining Containment Classes and Adding Sense-passing
 *      12. Grammar
 *      13. Miscellaneous and Utility
 *
 *----------------------------------------------------------------------
 *  REQUIREMENTS
 *
 *      + HTML TADS 2.5.1 or later
 *      + Requires ADV.T and STD.T
 *      + Should be #included after ADV.T and STD.T.
 *
 *----------------------------------------------------------------------
 *  IMPORTANT LIBRARY INTERFACE AND MODIFICATION
 *
 *      This module makes replacements and extensive modifications to
 *      the following functions and classes:
 *
 *          + listcontgen()
 *          + listcontcont()
 *          + mainRestore()
 *          + die()
 *          + init()
 *          + thing
 *          + surface
 *          + qsurface
 *          + container
 *          + qcontainer
 *          + openable
 *          + transparentItem
 *          + lockable
 *          + keyedLockable
 *          + room
 *          + darkroom
 *          + nestedroom
 *          + vehicle
 *          + chairitem
 *          + beditem
 *          + movableActor
 *          + deepverb
 *
 *      The USE_CONTAINMENT #define's uses an alternative definition of
 *      the containment and room classes.
 *
 *----------------------------------------------------------------------
 *  COPYRIGHT NOTICE
 *
 *  	You may modify and use this file in any way you want, provided that
 *		if you redistribute modified copies of this file in source form, the
 *   	copies must include the original copyright notice (including this
 *   	paragraph), and must be clearly marked as modified from the original
 *   	version.
 *
 *------------------------------------------------------------------------------
  *		REVISION HISTORY: See sensever.txt
 */
 
#define __SENSE_MODULE_
 
#pragma C+


/*======================================================================
 *
 *  1.  S E T T I N G   U P   C O N S T A N T S
 *
/*====================================================================*/


/*----------------------------------------------------------------------
 *  D E F I N E S
 *--------------------------------------------------------------------*/

#define PTH_LCC             1
#define PTH_PATH            2
#define PTH_VANTAGE         3
#define PTH_TARGET          4
#define PTH_TLOC            5

#define PRX_REFLEXIVE       1
#define PRX_POSSESSIVE      2
#define PRX_CONTAINED       3
#define PRX_IMMEDIATE       4
#define PRX_LOCAL           5
#define PRX_REMOTE          6

#define GAF_ACCESS_RULE     1
#define GAF_FAILLOC         2
#define GAF_PASSTO_METHOD   3
 
#ifdef USE_CONTAINMENT

/*----------------------------------------------------------------------
 *  C O M P O U N D   W O R D S
 *--------------------------------------------------------------------*/

compoundWord 'get' 'in' 'getin';
compoundWord 'get' 'into' 'getinto';
compoundWord 'get' 'on' 'geton';
compoundWord 'get' 'onto' 'getonto';
compoundWord 'get' 'out' 'getout';
compoundWord 'getout' 'of' 'getoutof';
compoundWord 'get' 'off' 'getoff';
compoundWord 'getoff' 'of' 'getoffof';
compoundWord 'lie' 'in' 'liein';
compoundWord 'lie' 'on' 'lieon';

#endif

/*----------------------------------------------------------------------
 *  D E C L A R A T I O N S
 *--------------------------------------------------------------------*/

access_topic: function;
access_visible: function;
access_audible: function;
access_olfactory: function;
access_touchable: function;
access_reachable: function;

access_qVisible: function;
access_qAudible: function;
access_qOlfactory: function;

inBubbleScope: function;
bubbleScope: function;
scopeCeiling: function;
bubbleScopeList: function;
runRoutine: function;

buildClist: function;
path: function;
blocksPath: function;
getPlist: function;
inPlist: function;
getAccessfailed: function;
proximity: function;
testScope: function;
loopOverScope: function;
checkAccessibility: function;
getCmdPhrase: function;
scope: function;
locale: function;
objectReactions: function;


/*======================================================================
 *
 *  2.  D E F I N I N G   A C C E S S   R U L E S
 *
 *                  F O R 
 *
 *          P A S S - A C R O S S - C O N T E N T S 
 *
 *                  V A L I D A T I O N
 *
/*====================================================================*/


access_topic: function(o) { return true; }

access_visible: function(o) { return (o.contentsVisible); }

access_audible: function(o) { return (o.contentsAudible); }

access_olfactory: function(o) { return (o.contentsOlfactory); }

access_touchable: function(o) { return (o.contentsTouchable); }

access_reachable: function(o) { return (o.contentsReachable); }


/*======================================================================
 *
 *  3.  P A T H   D E T E R M I N A T I O N 
 *
 *                  A N D 
 *
 *          I N F O R M A T I O N
 *
/*====================================================================*/


/*
 *  buildClist(o)
 *
 *  returns a containment list for the object.
 */
buildClist: function(o) {
    local loc, tmpList = [];
    
    loc = o;
    while(loc)
    {
        tmpList += loc;
        loc = loc.location;
    }
    return tmpList;    
}

/*
 *  path(vantage, target)
 *
 *  Function builds the plist for the vantage and the target.
 *  The plist is the logical path along which action is to flow,
 *  not necessarily the valid path for the action, which is
 *  determined by the blocksPath() function.
 *
 *  The plist consists of two elements:
 *      a. lowest common containment (or nil, if none)
 *      b. the logical path 
 */
path: function(vantage, target) {
    local i, f, lcc, vlist = [], olist = [], nlist = [], path = [];

    //
    //  Build the containment lists for both vantage and target.
    //
    vlist = buildClist(vantage);
    olist = buildClist(target);
    
    //
    //  intersect the vlist and olist. The first element
    //  should be the lowest common containment (lcc).
    //
    nlist = intersect(vlist, olist);
    lcc = car(nlist);
     
    //
    //  First we build the path by walking up the vantage
    //  containment list (vlist) until we either reach the lowest common
    //  containment (lcc) or the end of the list.
    //
    if (lcc) 
        f = find(vlist, lcc);
    else
        f = length(vlist);
        
    for (i = 1; i <= f; ++i)
        path += vlist[i];
     
    //
    //  Continue to build the path by walking down the
    //  target list (olist). We start at the object just below the
    //  lowest common containment (lcc), if it exists; otherwise we
    //  start from the end of the olist.
    //
    if (lcc)
        f = find(olist, lcc) - 1;
    else
        f = length(olist);
            
    for (i = f; i > 0; --i)
        path += olist[i];
 
    return [lcc path ];
}

/*
 *  getPlist(plist, flag)
 *
 *  Function returns the object or list corresponding to the plist 
 *  and PTH_XXXX constant passed. It returns an object for the lcc, 
 *  vantage, or target. It returns a list for the tloc or path.
 */
getPlist: function(plist, flag) {
    local len, lcc, vantage, target, t1, t2, clist;
        
    switch(flag)
    {
        case PTH_LCC:
            return plist[1];
        case PTH_PATH:
            return plist[2];
        case PTH_VANTAGE:
            return plist[2][1];
        case PTH_TARGET:
            len = length(plist[2]);
            return plist[2][len];
        case PTH_TLOC:            
            lcc = getPlist(plist, PTH_LCC);
            
            if (lcc)
                return [lcc];
                
            vantage = getPlist(plist, PTH_VANTAGE);
            clist   = buildClist(vantage);
            len     = length(clist);
            t1      = clist[len];
            
            target  = getPlist(plist, PTH_TARGET);
            clist   = buildClist(target);
            len     = length(clist);
            t2      = clist[len];
            
            return [t1, t2];
        default:
            return nil;   
    }
}

/*
 *  inPlist(plist, o, flag)
 *
 *  Function determines if o is found in the specified part of the
 *  plist. If o is found then the function returns true; otherwise it
 *  returns nil.
 *
 *  The flags parameter should be a PTH_XXXX constant. If no flag 
 *  is provided the function defaults to PTH_PATH.
 */
inPlist: function(plist, o, ...) {
    local p, list, flag = PTH_PATH;
    
    if (argcount == 3) flag = getarg(3);
    
    if (datatype(plist) != DTY_LIST) return nil;
    if (datatype(o) != DTY_OBJECT) return nil;
    if (datatype(flag) != DTY_NUMBER) return nil;
           
    p = getPlist(plist, flag);
    
    if (p == nil) return nil;
    
    switch(flag)
    {
        case PTH_LCC:
        case PTH_VANTAGE:
        case PTH_TARGET:
            if (o == p) 
                return true;
            else
                return nil;
        case PTH_PATH:
        case PTH_TLOC:
            if (find(p, o))
                return true;
            else
                return nil;
        default:
            return nil;
    }
}


/*======================================================================
 *
 *  4.  P A T H   V A L I D A T I O N
 *
 *              A N D 
 *
 *      S C O P E   T E S T I N G
 *
/*====================================================================*/


/*
 *  blocksPath(plist, filter)
 *
 *  Checks every object along the path to determine if access is
 *  valid for the given action/access_filter. It returns the object that
 *  fails the checks; otherwise it returns nil. 
 */
blocksPath: function(plist, filter) {
    local i, o, len, lcc, path, vantage, target, tclist;
    
    if (filter == &access_topic) return nil;
    
    lcc     = plist[1];
    path    = plist[2];
    
    len     = length(path);
    vantage = path[1];
    target  = path[len];
    tclist  = buildClist(target);
    target.acessfailed = [];
            
    for (i = 1; i <= len; ++i)
    {
        o = path[i];
        if (!o.passToObject(plist, filter))
        {
            target.accessfailed = [ filter, o, &passToObject ];
            return o;
        }
        
        if (o == target) continue;
        
        if (o != vantage && o != lcc)
        {
            if (!o.passAcrossContents(plist, filter))
            {
                target.accessfailed = [ filter, o, 
                    &passAcrossContents ];
                return o;                                
            }
        }
        if (find(tclist, o))
        {
            if (!o.passToContents(plist, filter))
            {
                target.accessfailed = [ filter, o, &passToContents ];
                return o;
            }
        }
        else
        {
            if (!o.passToLocation(plist, filter))
            {
                target.accessfailed = [ filter, o, &passToLocation ];
                return o;
            }
        }       
    }
        
    return nil;
}

/*
 *  testScope(vantage, target, filter)
 *
 *  Function returns true if the target is in the scope of the vantage
 *  for the given access_filter or for at least one access_filter if no
 *  filter is passed. Otherwise it returns nil.
 */
testScope: function(vantage, target, ...) {
    local i, plist, filter, vsList = [];
    
    if (argcount > 2) filter = getarg(3);
    
    plist = path(vantage, target);
    
    if (filter)
    {
        if (blocksPath(plist, filter) == nil) return true;
    }
    else
    {
        for (i = 1; i <= length(global.accessRules); ++i)
        {
            if (blocksPath(plist, global.accessRules[i]) == nil)
                return true;
        }
    }
    return nil;
}

/*
 *  checkAccessibility(actor, verb, dobj, prep, iobj)
 *
 *  Function determines the accessibility of the action under
 *  consideration for the objects involved. The indirect object is
 *  checked first, followed by the direct object. The function will 
 *  check the command for all multiple direct objects before passing
 *  control.
 *
 *  If the command fails to pass testScope() for the specified
 *  action, objects, and filter then the verb.cantReach() method is
 *  called and the function returns nil.
 *
 *  If the command passes testScope() for the specified 
 *  action, objects, and filter then it returns true.
 */
checkAccessibility: function(actor, verb, dobj, prep, iobj) {
    if (iobj != nil)
    {
        /*
         *  Checking the indirect object accessibility
         */
        if (!testScope(actor, iobj, verb.iorequires))
        {
            verb.cantReach(actor, nil, [iobj], prep);
            return nil;
        }
    }
    
    if (dobj != nil)
    {
        /*
         *  Checking the direct object accessibility
         */
        if (!testScope(actor, dobj, verb.dorequires))
        {
            verb.cantReach(actor, [dobj], nil, prep);
            return nil;
        }
    }
    return true;
}


/*======================================================================
 *
 *  5.  C O M M A N D   S T O R A G E   
 *
 *                  A N D   
 *
 *          V A L I D A T I O N 
 *
/*====================================================================*/


/*
 *  command: object
 *
 *  The command object carries all the pertinent information connected
 *  with the command, including the plist and the word-string list that
 *  the user entered for the direct and indirect objects of the command.
 */
command: object
    actorPtr        = nil
    verbPtr         = nil
    dolistPtr       = []
    dobjPtr         = nil
    prepPtr         = nil
    iolistPtr       = []
    iobjPtr         = nil
    cantSenseWords  = []
    disambigPhase   = nil
    disambigObjType = 0
    statusPtr       = 0
;

/*
 *  preparseCmd(cmd)
 *
 *  Function stores the command word list for use by getCmdPhrase().
 */
preparseCmd: function(cmd) {
    command.disambigPhase   = true;
    command.disambigObjType = 0;
    global.cmdList          = cmd;
    return true;
}

/*
 *  preCommand(actor, verb, dolist, prep, iobj)
 *
 *  This function is used to store command elements that may have come
 *  about through execCommand() and to check the validity of the action.
 */
preCommand: function(actor, verb, dolist, prep, iobj) {
    local i, ret;
    
    command.disambigPhase   = nil;
    command.disambigObjType = 0;
    command.actorPtr        = actor;
    command.verbPtr         = verb;
    command.dolistPtr       = dolist;
    command.prepPtr         = prep;
    command.iobjPtr         = iobj;
    
    for (i = 1; i <= length(dolist); ++i)
    {
        local dobj = dolist[i];
        
        command.dobjPtr = dolist[i];
        ret = checkAccessibility(actor, verb, dobj, prep, iobj);
        if (ret == nil) abort;
    }
}

/*
 *  postAction(actor, verb, dobj, prep, iobj, status)
 *
 *  This function is used to store the status of the command and execute
 *  postAction object reactions.
 */
postAction: function(actor, verb, dobj, prep, iobj, status) {    
    command.statusPtr       = status;

    if (status != EC_ABORT)
        objectReactions(2);
}

/*
 *  endCommand(actor, verb, dolist, prep, iobj, status)
 *
 *  This function is used to execute endCommand object reactions and to 
 *  clear down the command elements.
 */
endCommand: function(actor, verb, dolist, prep, iobj, status) {
    if (status != EC_ABORT)
        objectReactions(3);
    
    command.actorPtr        = parserGetMe();
    command.verbPtr         = nil;
    command.dolistPtr       = [];
    command.dobjPtr         = nil;
    command.prepPtr         = nil;
    command.iolistPtr       = [];
    command.iobjPtr         = nil;
    command.cantSenseWords  = [];
    command.statusPtr       = 0;
}


/*======================================================================
 *
 *  6.  D I S A M B I G U A T I O N   A N D   C O M M A N D   P H A S E
 *
 *      A C C E S S I B I L I T Y   E R R O R   H A N D L I N G
 *
/*====================================================================*/


/*
 *  getAccessfailed(o, flag)
 *
 *  Function returns the object or list corresponding to the plist 
 *  and PTH_XXXX constant passed. It returns an object for the lcc, 
 *  vantage, or obj. It returns a list for the tloc or path.
 */
getAccessfailed: function(target, flag) {
    if (proptype(target, &accessfailed) == DTY_LIST
    && length(target.accessfailed) >= flag)
    {
        switch(flag)
        {
            case GAF_ACCESS_RULE:
                return target.accessfailed[1];
            case GAF_FAILLOC:
                return target.accessfailed[2];
            case GAF_PASSTO_METHOD:
                return target.accessfailed[3];
            default:
                return nil;   
        }
    }
    return nil;
}

/*
 *  getCmdPhrase(o, words)
 *
 *  Function returns list of the intersection of the object vocabulary
 *  words and the passed word list. If the intersection is an empty list
 *  the function returns nil.
 */
getCmdPhrase: function(o, words) {
    local voclist, newlist;
    
    voclist = getwords(o, &adjective);
    voclist += getwords(o, &noun);
    
    newlist = intersect(voclist, words);
    if (length(newlist)) return newlist;
    
    voclist = getwords(o, &adjective);
    voclist += getwords(o, &plural);
    
    newlist = intersect(voclist, words);
    if (length(newlist)) return newlist;    
    
    return nil;
}


/*======================================================================
 *
 *  7.  D E S C R I B I N G   O B J E C T S
 *
/*====================================================================*/


/*
 *  proximity(vantage, target)
 *
 *  Function returns a proximity flag PRX_XXXX indicating the relative
 *  proximity of the vantage to the target.
 */
proximity: function(vantage, target) {
    local plist, lcc, tloc;
    
    plist   = path(vantage, target); 
    lcc     = getPlist(plist, PTH_LCC);
    tloc    = getPlist(plist, PTH_TLOC)[1];
    
    if (lcc)
    {
        if (vantage == target)
            return PRX_REFLEXIVE;
        if (vantage == lcc)
            return PRX_POSSESSIVE;
        if (target == lcc && lcc != tloc)
            return PRX_CONTAINED;
        if (lcc != tloc)
            return PRX_IMMEDIATE;
        if (lcc == tloc)
            return PRX_LOCAL;
    }
    else return PRX_REMOTE;
}

/*
 *  proxdesc(vantage, target, descList)
 *
 *  Function returns the corresponding element of the descList for the
 *  proximity of the vantage to the target. This "works" because it is
 *  is assumed that the descList is ordered to correspond with the
 *  the proximity flags:
 *
 *          descPtr         prox_flag      val
 *      ----------------------------------------
 *      desc_remote         prx_remote      6
 *      desc_local          prx_local       5
 *      desc_immediate      prx_immediate   4
 *      desc_contained      prx_contained   3
 *      desc_possessive     prx_possessive  2
 *      desc_reflexive      prx_reflexive   1
 *      desc_default        n/a             
 *
 *  The function first checks to see if proptype(target, descList[p]) is
 *  not nil. If so then we return target.(descList[p]).
 *
 *  Otherwise we work our way up the list to descList[p+1] and check
 *  that it exists. If so then we return target.(descList[p+1]); 
 *  otherwise we continue until we reach the end of the list. 
 *
 *  If no match has been found we begin working our way down the list
 *  from descList[p-1]. If no match is found we return nil.
 */
proxdesc: function(vantage, target, descList) {
    local i, descPtr, p = proximity(vantage, target);
    
    for (i = p; i <= length(descList); ++i)
    {
        descPtr = descList[i];
        if (proptype(target, descPtr) != DTY_NIL)
        {
            target.(descPtr);
            return;
        }
    }
    for (i = p - 1; i > 0; --i)
    {
        descPtr = descList[i];
        if (proptype(target, descPtr) != DTY_NIL)
        {
            target.(descPtr);
            return;
        }
    }
    return nil;
}

/*
 *  MODIFIED TO USE ADESCR AND PLURALDESCR EXTENDED SENSE DESCRIPTION
 *  ROUTERS.
 *
 *  listcontgen: function(obj, flags, indent)
 *
 *  This is a general-purpose object lister routine; the other object lister
 *  routines call this routine to do their work.  This function can take an
 *  object, in which case it lists the contents of the object, or it can take
 *  a list, in which case it simply lists the items in the list.  The flags
 *  parameter specifies what to do.  LCG_TALL makes the function display a "tall"
 *  listing, with one object per line; if LCG_TALL isn't specified, the function
 *  displays a "wide" listing, showing the objects as a comma-separated list.
 *  When LCG_TALL is specified, the indent parameter indicates how many
 *  tab levels to indent the current level of listing, allowing recursive calls
 *  to display the contents of listed objects.  LCG_CHECKVIS makes the function
 *  check the visibility of the top-level object before listing its contents.
 *  LCG_RECURSE indicates that we should recursively list the contents of the
 *  objects we list; we'll use the same flags on recursive calls, and use one
 *  higher indent level.  To specify multiple flags, combine them with the
 *  bitwise-or (|) operator.
 */
replace listcontgen: function(obj, flags, indent) {
    local i, count, tot, list, cur, disptot, prefix_count;

    /*
     *   Get the list.  If the "obj" parameter is already a list, use it
     *   directly; otherwise, list the contents of the object. 
     */
    switch(datatype(obj))
    {
    case 2:
        /* it's an object - list its contents */
        list = obj.contents;

        /* 
         *   if the CHECKVIS flag is specified, check to make sure the
         *   contents of the object are visible; if they're not, don't
         *   list anything 
         */
        if ((flags & LCG_CHECKVIS) != 0)
        {
            local contvis;

            /* determine whether the contents are visible */
            contvis = (!isclass(obj, openable)
                        || (isclass(obj, openable) && obj.isopen)
                        || obj.contentsVisible);

            /* if they're not visible, don't list the contents */
            if (!contvis)
                return;
        }
        break;
        
    case 7:
        /* it's already a list */
        list = obj;
        break;

    default:
        /* ignore other types entirely */
        return;
    }

    /* count the items in the list */
    tot = length(list);

    /* we haven't displayed anything yet */
    count = 0;

    /* 
     *   Count how many items we're going to display -- this may be fewer
     *   than the number of items in the list, because some items might
     *   not be listed at all (isListed = nil), and we'll list each group
     *   of equivalent objects with a single display item (with a count of
     *   the number of equivalent objets) 
     */
    disptot = itemcnt(list);

    /* iterate through the list */
    for (i = 1 ; i <= tot ; ++i)
    {
        /* get the current object */
        cur = list[i];

        /* if this object is to be listed, figure out how to show it */
        if (cur.isListed)
        {
            /* presume there is only one such object (i.e., no equivalents) */
            prefix_count = 1;
            
            /*
             *   if this is one of more than one equivalent items, list it
             *   only if it's the first one, and show the number of such
             *   items along with the first one 
             */
            if (cur.isEquivalent)
            {
                local before, after;
                local j;
                local sc;

                /* get this object's superclass */
                sc = firstsc(cur);

                /* scan for other objects equivalent to this one */
                for (before = after = 0, j = 1 ; j <= tot ; ++j)
                {
                    if (isIndistinguishable(cur, list[j]))
                    {
                        if (j < i)
                        {
                            /*
                             *   note that objects precede this one, and
                             *   then look no further, since we're just
                             *   going to skip this item anyway
                             */
                            ++before;
                            break;
                        }
                        else
                            ++after;
                    }
                }
                
                /*
                 *   if there are multiple such objects, and this is the
                 *   first such object, list it with the count prefixed;
                 *   if there are multiple and this isn't the first one,
                 *   skip it; otherwise, go on as normal 
                 */
                if (before == 0)
                    prefix_count = after;
                else
                    continue;
            }

            /* display the appropriate separator before this item */
            if ((flags & LCG_TALL) != 0)
            {
                local j;
                
                /* wide listing - indent to the desired level */
                "\n";
                for (j = 1; j <= indent; ++j)
                    "\t";
            }
            else
            {
                /* 
                 *   "wide" (paragraph-style) listing - add a comma, and
                 *   possibly "and", if this isn't the first item
                 */
                if (count > 0)
                {
                    if (count+1 < disptot)
                        ", ";
                    else if (count == 1)
                        " and ";
                    else
                        ", and ";
                }
            }
            
            /* list the object, along with the number of such items */
            if (prefix_count == 1)
            {
                /* there's only one object - show the singular description */
                cur.adescR;
            }
            else
            {
                /* 
                 *   there are multiple equivalents for this object -
                 *   display the number of the items and the plural
                 *   description 
                 */
                sayPrefixCount(prefix_count); " ";
                cur.pluraldescR;
            }
            
            /* show any additional information about the item */
            if (cur.isworn)
                " (being worn)";
            if (cur.islamp && cur.islit)
                " (providing light)";        
                        
            /* increment the number of displayed items */
            ++count;

            /* 
             *   if this is a "tall" listing, and there's at least one
             *   item contained inside the current item, list the item's
             *   contents recursively, indented one level deeper 
             */
            if ((flags & LCG_RECURSE) != 0 && itemcnt(cur.contents) != 0)
            {
                /* 
                 *   if this is a "wide" listing, show the contents in
                 *   parentheses 
                 */
                if ((flags & LCG_TALL) == 0)
                {
                    if (cur.issurface)
                        " (upon which %you% see%s% ";
                    else
                        " (which contains ";
                }
                
                /* show the recursive listing, indented one level deeper */
                listcontgen(cur, flags, indent + 1);

                /* close the parenthetical, if we opened one */
                if ((flags & LCG_TALL) == 0)
                    ")";
            }
        }
    }
}

/*
 *  MODIFIED TO ACCEPT A PASSED LIST
 *
 *  listcontcont: function(obj)
 *
 *  This function lists the contents of the contents of an object.
 *  It displays full sentences, so no introductory or closing text
 *  is required.  Any item in the contents list of the object
 *  obj whose contentsVisible property is true has
 *  its contents listed.  An Object whose isqcontainer or
 *  isqsurface property is true will not have its
 *  contents listed.
 */
replace listcontcont: function(obj) {
    local list, i, tot;
    
    /*
     *   Get the list.  If the "obj" parameter is already a list, use it
     *   directly; otherwise, list the contents of the object. 
     */
    switch(datatype(obj))
    {
    case 2:
        /* it's an object - list its contents */
        list = obj.contents;
        break;
        
    case 7:
        /* it's already a list */
        list = obj;
        break;

    default:
        /* ignore other types entirely */
        return;
    }

    tot = length(list);
    i = 1;
    while (i <= tot)
    {
        showcontcont(list[i]);
        i = i + 1;
    }
}


/*======================================================================
 *
 *  8.  D E S C R I B I N G   L O C A T I O N S
 *
/*====================================================================*/


/*----------------------------------------------------------------------
 *  A C C E S S   R U L E S:  R O O M   L I S T I N G S
 *--------------------------------------------------------------------*/

access_qVisible: function(o) {
    return (access_visible(o)
        && !o.isqVisible);
}

access_qAudible: function(o) { 
    return (access_audible(o) 
        && !o.isqAudible); 
}

access_qOlfactory: function(o) { 
    return (access_olfactory(o) 
        && !o.isqOlfactory); 
}

/*----------------------------------------------------------------------
 *  B U B B L E   S C O P E   L I S T
 *
 *  These functions produce a "bubble-up/filter-down" style list of 
 *  all objects valid for the given access_filter limited to a top-level 
 *  location or a scope ceiling. 
 *  
 *  They are mainly used for extended location sense display.
 *
 *--------------------------------------------------------------------*/

/*
 *  inBubbleScope: function(vantage, target, limit, filter)
 *
 *  Function returns true if the target is within the scope list
 *  produced for the given access_filter for the vantage; otherwise
 *  it returns nil.
 */
inBubbleScope: function(vantage, target, limit, filter) {	
	local value = bubbleScope(vantage, limit, filter);
	
	if (find(value, target)) 
	    return true;
	return nil;
}
    
/*
 *	bubbleScope: function(vantage, limit, filter)
 *
 *	Function returns a list of all objects that are valid for a given vantage
 *	and access_filter. 
 *
 *	First it determines the scopeCeiling for a given vantage and access_filter. 
 *	This becomes the first entry in the list. 
 *
 *  limit can be useful in cases where the author wishes to limit the
 *  scope to within a select location, even though the access_filter might
 *  extend beyond the location.
 *
 *	Next it calls bubbleScopeList with the location, vantage, scope 
 *  ceiling, and access_filter and adds the return to the list. 
 *
 *	Next each object in global.floatingList is checked using 
 *  bubbleScopeList and the results are added to the list.
 */
bubbleScope: function(vantage, limit, filter) {
    local c, o, loc, ceiling, lst = [];
 
    ceiling = scopeCeiling(vantage, limit, filter);
        
    loc = ceiling;

    lst += ceiling;
    lst += bubbleScopeList(loc, vantage, ceiling, filter);

    /*
     *  Each floatingItem object is evaluated. If it has the same
     *  scope ceiling as the vantage or its scope ceiling is the
     *  vantage then it is added to scope along with any appropriate
     *  contents.
     */
	c = global.floatingList;
	o = car( c );
	while( o )
	{
		loc = o;
		ceiling = scopeCeiling(o, limit, filter);
		if (ceiling == lst[1]
		|| ceiling == vantage)
		{
			lst += o;
			if (runRoutine(loc, filter))
			    lst += bubbleScopeList(loc, o, ceiling, filter);
		}
		
		c = cdr( c );
		o = car( c );
	}
    
    return lst;
}

/*
 *  bubbleScopeList: function(loc, vantage, ceiling, filter)
 *
 *  Function returns a list of all valid objects for a given vantage, scope
 *	ceiling and access_filter. It recurses down each object in the location's
 *	contents.
 */
bubbleScopeList: function(loc, vantage, ceiling, filter) {
    local i, lst, len, ret = [];

    /* don't look in "nil" objects */
    if (loc == nil)
        return ret;

	/*
	 *	The location is added to the list if,
	 *		a. it is the vantage or the scope ceiling
	 *		b. it meets the requirements of a scope definition for the given
	 *		   access_filter.
	 */
	if (loc == vantage 
	||  loc == ceiling
	||  runRoutine(loc, filter))
    {
        lst = loc.contents;
        len = length(lst);
        ret += lst;
        for (i = 1 ; i <= len ; ++i)
        {
            /* recurse into this object */
      		ret += bubbleScopeList(lst[i], vantage, ceiling, filter);
        }
    }

    return ret;
}

/* 
 *	scopeCeiling: function(vantage, limit, filter)
 *
 *	Function returns the scopeCeiling for a given vantage and access_filter.
 *	The scopeCeiling is the highest level location that the vantage has access
 *	to for the given access_filter within a single top-level location.
 *
 *  limit acts to constrain the function a given scope ceiling "height",
 *  but the function will always return a valid scope ceiling for a
 *  valid location.
 */
scopeCeiling: function(vantage, limit, filter) {
    local loc;
    
    if (vantage == limit)
    	return vantage;
    	
    if (vantage.location)
    	loc = vantage.location;
    else
    	loc = vantage;
    	
    while( loc && loc.location ) 
   	{
   	    if (loc == limit)
   	        break;
   	        
   	    if (!runRoutine(loc, filter)) 
            break;
                        
        loc = loc.location;
    }
    return loc;
}

/*
 *  locale(actor, loc, access_rule)
 *
 *  Function acts as an accessor to the location's xtndXxAround()
 *  method. It will basicallly make the location listed and its 
 *  contents "unquiet" during the call, which will display the 
 *  location's ldesc_remote and the dispcont() or listcont() for 
 *  the location contents.
 */
locale: function(actor, loc, access_rule) {
    local lsave, qsave;

    switch(access_rule)
    {
        case &access_visible:
            lsave                   = loc.isListed;
            qsave                   = loc.isqVisible;
            loc.isListed            = true;
            loc.isqVisible          = nil;
            loc.xtndLkAround(actor, true);
            loc.isListed            = lsave;
            loc.isqVisible          = qsave;
            break;
        case &access_audible:
            lsave                   = loc.isListedAudible;
            qsave                   = loc.isqAudible;
            loc.isListedAudible     = true;
            loc.isqAudible          = nil;
            loc.xtndLnAround(actor, true);
            loc.isListedAudible     = lsave;
            loc.isqAudible          = qsave;
            break;
        case &access_olfactory:
            lsave                   = loc.isListedOlfactory;
            qsave                   = loc.isqolfactory;
            loc.isListedOlfactory   = true;
            loc.isqOlfactory        = nil;
            loc.xtndSmAround(actor, true);
            loc.isListedOlfactory   = lsave;
            loc.isqOlfactory        = qsave;
            break;
    }
}


/*======================================================================
 *
 *  9.  O B J E C T   R E A C T I O N S
 *
/*====================================================================*/


/*
 *  objectReaction(lvl)
 *
 *  Function produces object reactions for all objects within the scope
 *  ceiling of the actor's location. 
 */
objectReactions: function(lvl) {   
    local i;
	local actor = command.actorPtr;
	local dobj  = command.dobjPtr;
	local iobj  = command.iobjPtr;
	local loc   = scopeCeiling(actor, nil, &access_visible);
	local scope_list = bubbleScope(actor, loc, &access_visible) 
	    - [actor] - [loc] - [dobj] - [iobj];

    switch(lvl)
    {
        case 1:
            actor.actorPreAction;
            if (loc != actor) loc.roomPreAction;
            for (i = 1; i <= length(scope_list); ++i)
                runRoutine(scope_list[i], &scopePreAction);
            if (iobj) iobj.ioPreAction;
            if (dobj) dobj.doPreAction;
            break;
        case 2:
            actor.actorPostAction;
            if (loc != actor) loc.roomPostAction;
            for (i = 1; i <= length(scope_list); ++i)
                runRoutine(scope_list[i], &scopePostAction);
            if (iobj) iobj.ioPostAction;
            if (dobj) dobj.doPostAction;
            break;
        case 3:
            actor.actorEndCommand;
            if (loc != actor) loc.roomEndCommand;
            for (i = 1; i <= length(scope_list); ++i)
                runRoutine(scope_list[i], &scopeEndCommand);
            if (iobj) iobj.ioEndCommand;
            if (dobj) dobj.doEndCommand;
            break;
    }
}

/*
 *  scope(vantage, filter)
 *
 *  Function loops over every *thing* class object, testing its scope
 *  for a given filter. Returns a list of all objects that pass 
 *  testScope().
 */
scope: function(vantage, ...) {
    local o, filter, list = [];
    
    if (argcount == 2) filter = getarg(2);
    
    o = firstobj(thing);
    while(o)
    {
        if (testScope(vantage, o, filter))
            list += o;
        o = nextobj(o, thing);
    }   
    return list;
}

/*
 *  loopOverScope(prop, vantage, filter)
 *
 *  This function loops over every *thing* class object within the
 *  vantage's scope and executes either prop(o) or o.prop; 
 */
loopOverScope: function(prop, ...) {
    local o, vantage = parserGetMe(), filter;
    
    if (argcount > 1) vantage   = getarg(2);
    if (argcount > 2) filter    = getarg(3);
    
    o = firstobj(thing);
    while(o)
    {
        if (testScope(vantage, o, filter))
            runRoutine(o, prop);
        o = nextobj(o, thing);
    }
}


/*======================================================================
 *
 *  10.  A D D I N G   S E N S E - P A S S I N G 
 *
 *          T O   A D V . T   C L A S S E S
 *
/*====================================================================*/


/*
 *	MODIFICATIONS TO THING IMPLEMENTING NEW SCOPING MECHANISMS.
 *
 *  thing: object
 *
 *  The basic class for objects in a game.  The property contents
 *  is a list that specifies what is in the object; this property is
 *  automatically set up by the system after the game is compiled to
 *  contain a list of all objects that have this object as their
 *  location property.  The contents property is kept
 *  consistent with the location properties of referenced objects
 *  by the moveInto method; always use moveInto rather than
 *  directly setting a location property for this reason.  The
 *  adesc method displays the name of the object with an indefinite
 *  article; the default is to display "a" followed by the sdesc,
 *  but objects that need a different indefinite article (such as "an"
 *  or "some") should override this method.  Likewise, thedesc
 *  displays the name with a definite article; by default, thedesc
 *  displays "the" followed by the object's sdesc.  The sdesc
 *  simply displays the object's name ("short description") without
 *  any articles.  The ldesc is the long description, normally
 *  displayed when the object is examined by the player; by default,
 *  the ldesc displays "It looks like an ordinary sdesc."
 *  The isIn(object) method returns true if the
 *  object's location is the specified object or the object's
 *  location is an object whose contentsVisible property is
 *  true and that object's isIn(object) method is
 *  true.  Note that if isIn is true, it doesn't
 *  necessarily mean the object is touchable, because isIn is
 *  true if the object is merely visible within the location.
 *  The moveInto(object) method moves the object to be inside
 *  the specified object.  To make an object disappear, move it
 *  into nil.
 */
modify thing
    replace contentsVisible = (true)
    contentsAudible         = (true)
    contentsOlfactory       = (true)
    contentsTouchable       = (true)
    replace contentsReachable = (true)
    
    
    //
    //  The isqXxxx methods should return true or nil depending on
    //  whether the object is "quiet" about its contents or not.
    //
    isqVisible          = (nil)
    isqAudible          = (nil)
    isqOlfactory        = (nil)
    
    passToObject(plist, filter)     = (true)
    passToLocation(plist, filter)   = (true)
    passToContents(plist, filter)   = (true)
    passAcrossContents(plist, filter) = {
        return runRoutine(self, filter);
    }
    
    //
    //  Additional isListed attributes for the sense involved.
    //
    isListedAudible     = (nil)
    isListedOlfactory   = (nil)
    
    /*
     *  Modification to statusPrep and outOfPrep for thing class
     */
    statusPrep = { if (self.issurface) "on"; else "in"; }
    outOfPrep = { if (self.issurface) "off of"; else "out of"; }
	/*
 	 *	Modifying verifyRemove() to 'bubble up' verGrab() check, but only as
 	 *	high as thescope ceiling. This allows us to grab things when we're
 	 *	inside closed locations, such as enterables.
 	 */
    replace verifyRemove(actor) = {
        /*
         *   Check with each container to make sure that the container
         *   doesn't object to the object's removal.
         */
        local loc, ceiling;

		ceiling = scopeCeiling( actor, nil, &access_touchable );

        loc = self.location;
        while (loc)
        {
        	if (loc == ceiling)
      			break;
            if (loc != actor)
                loc.verGrab(self);
            loc = loc.location;
        }
    }
    
    /*
     *	isVisible() now returns true for all objects. This means that
     *  the cantReach() mechanism will handle the case where both the
     *  valid list produced by validXoList/validXo and the logical list
     *  produced by verXoVerb during Disambiguation are both empty.
     */
    isVisible(vantage) = (true)
        
    //
    //  Display contents for self, using a given actor,
    //  access_filter, isListedXxxx attribute and sense 
    //  description.
    //
    dispcont(actor, filter, listedAttr, descAttr) = {
        local i, plist, list, list_filter;
        local disptot = 0, stat, str;             
        
        switch(filter)
        {
            case &access_visible:
                list_filter = &sccope_qVisible;
                break;
            case &access_audible:
                list_filter = &access_qAudible;
                break;
            case &access_olfactory:
                list_filter = &access_qOlfactory;
                break;
            default:
                return nil;
        }
        
        list  = bubbleScope(self, self, list_filter) - self;
        
        for (i = 1; i <= length(list); ++i)
        { 
            plist = path(actor, list[i]);
            
            if (blocksPath(plist, filter) == nil
            && list[i].(listedAttr))
            {
                stat = outcapture(true);

                list[i].(descAttr);
                                
                str = outcapture(stat);
                if (length(str))
                {
                    say(str);
                    ++disptot;
                }   
            }
        }
        return disptot;
    }
    
    /*
     *  cantSense methods are at the object level. They are passed the
     *  actor and the word list. 
     */
    cantSee(vantage, target) = {
        local i;
        
        "%You% can't see any ";
        for (i = 1; i <= length(command.cantSenseWords); ++i)
        {
            say(command.cantSenseWords[i]);
            " ";
        }
        "here.";
    }        
    cantHear(vantage, target) = {        
        if (testScope(vantage, target, &access_visible))
            "%You% can't hear that here.";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }        
    cantSmell(vantage, target) = {
        if (testScope(vantage, target, &access_visible))
            "%You% can't smell that from here.";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }        
    cantTouch(vantage, target) = {
        if (testScope(vantage, target, &access_visible))
            "%You% can't touch that from here. ";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }
    replace cantReach(vantage, target) = {
        if (testScope(vantage, target, &access_visible))
            "%You% can't reach that from here. ";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }
        
    /*------------------------------------------------------------------
     *  Sense Descriptor Routers
     *
     *  Each Router must also define a list of description method points
     *  ordered by reflexive, possessive, contained, immediate, local,
     *  and remote. 
     *
     *  Default description methods should be defined at the local
     *  position in the list. Some examples of default description 
     *  methods are sdesc, adesc, thedesc, ldesc, readdesc, and
     *  actorDesc.
     *
     *  Messages built using a Descriptor Router will choose the 
     *  appropriate description message, based on the proximity of the
     *  vantage and target.
     *----------------------------------------------------------------*/

    sdescR = {
        return proxdesc(command.actorPtr, self, self.sdescList);
    }
    sdescList = [ &sdesc_reflexive, &sdesc_possessive,
        &sdesc_contained, &sdesc_immediate, &sdesc, &sdesc_remote
        ]

    adescR = {
        return proxdesc(command.actorPtr, self, self.adescList);
    }
    adescList = [ &adesc_reflexive, &adesc_possessive,
        &adesc_contained, &adesc_immediate, &adesc, &adesc_remote
        ]
    adesc = {
        "a "; self.sdescR;
    }
     
    thedescR = {
        return proxdesc(command.actorPtr, self, self.thedescList);
    }
    thedescList = [ &thedesc_reflexive,
        &thedesc_possessive, &thedesc_contained, &thedesc_immediate,
        &thedesc, &thedesc_remote ]
    thedesc = {
        "the "; self.sdescR;
    }
        
    pluraldescR = {
        return proxdesc(command.actorPtr, self, self.pluraldescList);
    }
    pluraldescList = [ &pluraldesc_reflexive,
        &pluraldesc_possessive, &pluraldesc_contained,
        &pluraldesc_immediate, &pluraldesc, &pluraldesc_remote ]
    replace pluraldesc = {
        self.sdescR;
        if (not self.isThem)
            "s";
    }

    ldescR = {
        return proxdesc(command.actorPtr, self, self.ldescList);
    }
    ldescList = [ &ldesc_reflexive, &ldesc_possessive,
        &ldesc_contained, &ldesc_immediate, &ldesc, &ldesc_remote
        ]
    ldesc = { "%You% see%s% nothing unusual about 
        <<self.thedescR>>. "; }

    listendescR = {
        return proxdesc(command.actorPtr, self, self.listendescList);
    }
    listendescList = [ &listendesc_reflexive,
        &listendesc_possessive, &listendesc_contained,
        &listendesc_immediate, &listendesc, &listendesc_remote ]
    listendesc = { "%You% hear%s% nothing unusual from 
        <<self.thedescR>>. "; }

    smelldescR = {
        return proxdesc(command.actorPtr, self, self.smelldescList);
    }
    smelldescList = [ &smelldesc_reflexive,
        &smelldesc_possessive, &smelldesc_contained,
        &smelldesc_immediate, &smelldesc, &smelldesc_remote ]
    smelldesc = { "%You% smell%s% nothing unusual about 
        <<self.thedescR>>. "; }

    touchdescR = {
        return proxdesc(command.actorPtr, self, self.touchdescList);
    }        
    touchdescList = [ &touchdesc_reflexive,
        &touchdesc_possessive, &touchdesc_contained,
        &touchdesc_immediate, &touchdesc, &touchdesc_remote ]
    touchdesc = { "%You% feel%s% nothing unusual about 
        <<self.thedescR>>. "; }
        
    tastedescR = {
        return proxdesc(command.actorPtr, self, self.tastedescList);
    }        
    tastedescList = [ &tastedesc_reflexive,
        &tastedesc_possessive, &tastedesc_contained,
        &tastedesc_immediate, &tastedesc, &tastedesc_remote ]
    tastedesc = { "%You% taste%s% nothing unusual about 
        <<self.thedescR>>. "; }
       
    /*------------------------------------------------------------------
     *  Sense Verification and Action Methods
     *
     *  Messages built using a Descriptor Router will choose the 
     *  appropriate description message, based on the proximity of the
     *  vantage and target.
     *----------------------------------------------------------------*/

    replace doInspect(actor) = {self.ldescR;}
    verDoListento(actor) = {}
    doListento(actor) = {self.listendescR;}
    verDoSmell(actor) = {}
    doSmell(actor) = {self.smelldescR;}
    verDoTouch(actor) = {}
    doTouch(actor) = { self.touchdescR; }
    verDoTaste(actor) = {}
    doTaste(actor) = { self.tastedesc; }
    replace doRead(actor) = {
        self.readdescR;
    }
    
    /*------------------------------------------------------------------
     *  Actor Validation uses testScope with a reachability test.
     *----------------------------------------------------------------*/
     
    validActor = {
        return testScope(parserGetMe(), self, &access_audible);
    }
;

modify fixeditem
    heredescR = {
        return proxdesc(command.actorPtr, self, self.heredescList);
    }
    heredescList = [ &heredesc_reflexive, &heredesc_possessive,
        &heredesc_contained, &heredesc_immediate, &heredesc, &heredesc_remote
        ]
;


/*======================================================================
 *
 *  11.  R E D E F I N I N G   C O N T A I N M E N T   C L A S S E S
 *
 *                          A N D 
 *
 *          A D D I N G   S E N S E - P A S S I N G   
 *
/*====================================================================*/


#ifdef USE_CONTAINMENT

////////////////////////////////////////////////////////////////////////
//
//  containment: thing
//
//  The containment class is POLYMORPHIC. It combines the behaviours 
//  of the following classes: surface, qsurface, container, qcontainer, 
//  openable, lockable, keyedLockable, transparentItem, room, darkroom, 
//  nestedroom, vehicle, chairitem, and beditem.
//
////////////////////////////////////////////////////////////////////////
containment: thing

    ////////////////////////////////////////////////////////////////////
    //
    //  IDENTITY ATTRIBUTES
    //
    ////////////////////////////////////////////////////////////////////

    isenterable = nil
    isvehicle = nil
    issurface = {       // simulates class inheritance
        if (self.isqsurface)
            return true;
        else
            return nil;
    }
    isqsurface = nil
    iscontainer = {     // simulates class inheritance
        if (self.isqcontainer 
        || self.isopenable 
        || self.islockable 
        || self.istransparent
        || self.ispermeable)
            return true;
        else 
            return nil;
    }
    isqcontainer = nil
    isopenable = {
        if (self.islockable
        || self.istransparent
        || self.ispermeable)
            return true;
        else
            return nil;
    }
    islockable = nil
    istransparent = nil
    ispermeable = nil

    ////////////////////////////////////////////////////////////////////
    //
    //  STATE ATTRIBUTES & METHODS
    //
    ////////////////////////////////////////////////////////////////////

    isopen = true  
    islocked = nil 
    islit = 
    {
        local i, o;

        if (self.lightsOn)
            return true;
        
        for (i = 1; i <= length(global.lamplist); ++i)
        {
            o = global.lamplist[i];
            if (o.isIn(self) && o.islit)
                return true;
        }

        if (self.location && self.contentsVisible)
            return self.location.islit;
        return nil;
    }
    contentsVisible = { 
        if(self.istransparent || self.ispermeable) 
            return true;
        else
            return self.isopen; 
    }
    contentsAudible = { 
        if (self.ispermeable)
            return true;
        else
            return self.isopen; 
    }
    contentsOlfactory = { 
        if (self.ispermeable)
            return true;
        else
            return self.isopen; 
    }
    contentsTouchable = { return self.isopen; }
    contentsReachable = { 
        if (self.ispermeable)
            return true;
        else
            return self.isopen; 
    }

    maxbulk = 10            // maximum bulk the container can contain
    isseen = nil            // room has not been seen yet
    mykey = nil     // set 'mykey' to the key which locks/unlocks me

    ////////////////////////////////////////////////////////////////////
    //
    //  LISTING ATTRIBUTES
    //
    ////////////////////////////////////////////////////////////////////

    isListed = true
    isListedAudible = nil
    isListedOlfactory = nil

    isqVisible = nil
    isqAudible = nil
    isqOlfactory = nil
   
    ////////////////////////////////////////////////////////////////////
    //
    //  DESCRIPTION METHODS
    //
    ////////////////////////////////////////////////////////////////////

    statusPrep = { if (self.issurface) "on"; else "in"; }
    outOfPrep = { if (self.issurface) "off of"; else "out of"; }
    ldesc =
    {	    
        if (self.isopenable)
        {
            "\^<<self.thedesc>> <<self.isdesc>>";
            if (self.isopen)
            {
                " open. ";
            }
            else
            {
                " closed. ";
            
                /* if it's transparent, list its contents anyway */
                if (!self.contentsVisible)
                    return; 
            }
        }
        if (self.contentsVisible && itemcnt(self.contents) != 0)
        {
            "\^<<self.statusPrep>> <<self.thedesc>> %you% see%s% "; 
            listcont(self); ". ";
        }
        else
        {
            "There's nothing <<self.statusPrep>> <<self.thedesc>>. ";
        }
    }

    ////////////////////////////////////////////////////////////////////
    //
    //  VEHICLE XOBJGEN METHODS
    //
    ////////////////////////////////////////////////////////////////////

    dobjGen(a, v, i, p) =
    {
        if (!self.isvehicle) return;
        
        if (a.isIn(self) 
        && v != inspectVerb 
        && v != getOutVerb
        && v != getOffVerb
        && v != disembarkVerb
        && v != outVerb)
        {
            "%You%'ll have to get <<self.outOfPrep>> <<
                self.thedesc>> first. ";
            exit;
        }
    }
    iobjGen(a, v, d, p) =
    {
        if (!self.isvehicle) return;
        
        if (a.isIn(self) && v != putVerb)
        {
            "%You%'ll have to get <<self.outOfPrep>> <<
                self.thedesc>> first. ";
            exit;
        }
    }
    
    ////////////////////////////////////////////////////////////////////
    //
    //  SURFACE / CONTAINER DIRECT OBJECT METHODS
    //
    ////////////////////////////////////////////////////////////////////

    verDoBoard(actor) =
    {
        if (!self.isenterable && (self.issurface || self.iscontainer))
        {
            "That's not something %you% can enter. ";
        }
        else if (actor.location == self)
        {
            "%You're% already <<self.statusPrep>> <<self.thedesc>>! ";
        }
        else if (actor.isCarrying(self))
        {
            "%You%'ll have to drop <<self.thedesc>> first!";
        }
        else if (self.isopenable && !self.isopen)
    	{
        	"%You%'ll have to open <<self.thedesc>> first!";
    	}    	    
    } 	
    doBoard(actor) =
    {
        if (find(buildClist(actor), self))
        {   
            local ret;
            "(getting <<self.outOfPrep>> <<self.thedesc>>)\n";
            ret = execCommand(actor, disembarkVerb, actor.location, EC_HIDE_SUCCESS);
            if (ret != EC_SUCCESS || actor.location != self)
                exitobj;
        }
        else if (actor.location != self.location)
        {
            local ret;
            "(getting <<self.location.issurface ? "onto" : "into">> 
            <<self.location.thedesc>>)\n";
            ret = execCommand(actor, boardVerb, self.location, EC_HIDE_SUCCESS);
            if (ret != EC_SUCCESS || !actor.isIn(self.location)) 
                exitobj;
        }
        if (addbulk(self.contents) + actor.bulk > self.maxbulk)
        {
            "%You% can't fit <<self.statusPrep>> <<self.thedesc>>. ";
        }
        else
        {
            "Okay, %you're% now <<self.statusPrep>> <<self.thedesc>>.\b";
            actor.travelTo(self);
        }
    }
    verDoClose(actor) =
    {
        if (!self.isopenable)
        {
            "I don't know how to close <<self.thedesc>>. ";
        }
        else if (!self.isopen)
        {
            caps(); self.thedesc; " <<self.isdesc>> already closed! ";
        }
    }
    doClose(actor) =
    {
        "Closed. ";
        self.isopen = nil;
    }
    verDoDrop(actor) =
    {
        if (actor.isIn(self))
        {
            "%You% can't drop <<self.thedesc>>. ";
        }
        else if (!actor.isCarrying(self))
        {
            "%You're% not carrying "; self.thedesc; "! ";
        }
    }
    verDoGetin(actor) = 
    {
        self.verDoBoard(actor);
        if (!self.iscontainer && !self.isvehicle)
            "That's not something %you% can get in. ";
    }
    doGetin(actor) = { self.doBoard(actor); }
    doSynonym('Getin') = 'Enter'
    verDoGeton(actor) =
    {
        self.verDoBoard(actor);
        if (!self.issurface && !self.isvehicle)
            "That's not something %you% can get on. ";
    }
    doGeton(actor) = { self.doBoard(actor); }
    verDoGetoff(actor) =
    {
        self.verDoUnboard(actor);
        if (!self.issurface && !self.isvehicle)
            "That's not something %you% can get off of. ";
    }
    doGetoff(actor) = { self.doUnboard(actor); }
    verDoGetout(actor) =
    {
        self.verDoUnboard(actor);
        if (!self.iscontainer && !self.isvehicle)
            "That's not something %you% can get out of. ";
    }
    doGetout(actor) = { self.doUnboard(actor); }
    doSynonym('Getout') = 'Exit'
    doInspect(actor) =
    {
	    if (actor.isIn( self )
	    && proptype(self, &insideDesc) != DTY_NIL)
	        self.insideDesc;
	    else
	        self.ldescR;
    }
    verDoLieon(actor) =
    {
        if (!self.issurface)
        {
            "I don't know how to lie <<self.statusPrep>> 
            <<self.thedesc>>. ";
        }
        else if (actor.location == self)
        {
            "%You're% already <<self.statusPrep>> 
            <<self.thedesc>>! ";
        }
    }
    doLieon(actor) =
    {
        "Okay, %you're% now lying <<self.statusPrep>> 
        <<self.thedesc>>.\n";
        actor.travelTo(self);
    }
    verDoLock(actor) =
    {
        if (!self.islockable)
        {
            "I don't know how to lock <<self.thedesc>>. ";
        }
        else if (self.islocked)
        {
            "\^<<self.itisdesc>> already locked! ";
        }
    }
    doLock(actor) =
    {
        if (self.mykey != nil)
        {
            askio(withPrep);
        }
        else if (self.isopen)
        {
            "%You%'ll have to close <<self.thedesc>> first. ";
        }
        else
        {
            "Locked. ";
            self.islocked = true;
        }
    }
    verDoLockWith(actor, io) =
    {
        if (!self.islockable)
        {
            "I don't know how to lock <<self.thedesc>>. ";
        }
        else if (self.islocked)
        {
            "\^<<self.itisdesc>> already locked. ";
        }
    }
    doLockWith(actor, io) =
    {
        if (self.mykey == nil)
        {
            "A key is not required to lock <<self.thedesc>>. ";
        }
        else if (self.isopen)
        {
            "%You% can't lock << self.thedesc >> when <<
            self.itisdesc>> open. ";
        }
        else if (io == self.mykey)
        {
            "Locked. ";
            self.islocked = true;
        }
        else
            "\^<<io.itnomdesc>> <<io.doesdesc>>n't fit the lock. ";
    }
    verDoLookin(actor) =
    {
        if (!self.iscontainer)
            "There's nothing in <<self.thedesc>>. ";
        /* we can look in it if either it's open or it's transparent */
        else if (!self.contentsVisible)
           "\^<<self.itisdesc>> closed. ";
    }
    doLookin(actor) =
    {
        self.doSearch(actor);
    }
    verDoMove(actor) =
    {
        if (actor.isIn(self))
        {
            "%You% can't move <<self.thedesc>>. ";
        }
    }
    verDoOpen(actor) =
    {
        if (!self.isopenable)
        {
            "I don't know how to open <<self.thedesc>>. ";
        }
        else if (self.islocked)
        {
            "\^<<self.itisdesc>> locked. ";
        }
        else if (self.isopen)
        {
            "\^<<self.thedesc>> <<self.isdesc>> already open! ";
        }
    }
    doOpen(actor) =
    {
        if (!self.contentsVisible
        && !actor.isIn(self)
        && itemcnt(self.contents))
        {
            "Opening <<self.thedesc>> reveals <<listcont(self)>>. ";
        }
        else
            "Opened. ";
        self.isopen = true;
    }
    verDoPutIn(actor, io) =
    {
        if (actor.isIn(self))
        {
            "%You% can't put <<self.thedesc>> anywhere. ";
        }
    }
    verDoPutOn(actor, io) =
    {
        if (actor.isIn(self))
        {
            "%You% can't put <<self.thedesc>> anywhere. ";
        }
    }
    verDoSearch(actor) = { self.verDoLookin(actor); }
    doSearch(actor) =
    {
        if (self.contentsVisible && itemcnt(self.contents) != 0)
            "\^<<self.statusPrep>> <<self.thedesc>> %you% see%s% 
            <<listcont(self)>>. ";
        else
            "There's nothing <<self.statusPrep>> <<self.thedesc>>. ";
    }
    verDoSiton(actor) =
    {
        if (!self.issurface)
        {
            "I don't know how to sit <<self.statusPrep>> 
            <<self.thedesc>>. ";
        }
        else if (actor.location == self)
        {
            "%You're% already <<self.statusPrep>> 
            <<self.thedesc>>! ";
        }
    }
    doSiton(actor) =
    {
        "Okay, %you're% now sitting <<self.statusPrep>> 
        <<self.thedesc>>.\n";
        actor.travelTo(self);
    }
    verDoTake(actor) =
    {
        if (actor.isIn(self))
        {
            "%You% can't have <<self.thedesc>>! ";
        }
        else if (actor.isCarrying(self))
        {
        	"%You% already %have% <<self.thedesc>>! ";
        }
    }
    verDoTakeOff(actor, io) =
    {
        self.verDoTake(actor);
    }
    verDoTakeOut(actor, io) =
    {
        self.verDoTake(actor);
    }
    verDoThrowAt(actor, iobj) =
    {
        if (actor.isIn(self))
        {
            "%You% can't throw <<self.thedesc>>.";
        }
    }
    verDoUnboard(actor) =
    {
        if (find(buildClist(actor), self) == nil)
        {
            "%You're% not <<self.statusPrep>> <<self.thedesc>>! ";
        }
        else if (self.location == nil)
        {
            "%You% can't leave <<self.thedesc>>! ";
        }
    	else if (!actor.location.isopen)
    	{
    		"%You%'ll have to open <<self.thedesc>> first!";
    	}
    }
    doUnboard(actor) =
    {
        if (actor.location != self)
        {
            local ret;
            "(getting <<actor.location.outOfPrep>> <<actor.location.thedesc>>)\n";
            self.leaveRoom(actor);
            actor.moveInto(actor.location.location);
            ret = execCommand(actor, disembarkVerb, 
                actor.location, EC_HIDE_SUCCESS);
            if (ret != EC_SUCCESS || actor.location == self)
                exitobj;
        }
        else
        {
            self.leaveRoom(actor);
            actor.moveInto(actor.location.location);
        }
        "Okay, %you're% no longer <<self.statusPrep>> <<self.thedesc>>. ";
    }
    verDoUnlock(actor) =
    {
        if (!self.islockable)
        {
            "I don't know how to unlock <<self.thedesc>>. ";
        }
        else if (!self.islocked)
        {
            "\^<<self.itisdesc>> not locked! ";
        }
    }
    doUnlock(actor) =
    {
        if (self.mykey != nil)
        {
            askio(withPrep);
        }
        else
        {
            "Unlocked. ";
            self.islocked = nil;
        }
    }
    verDoUnlockWith(actor, io) =
    {
        if (!self.islockable)
        {
            "I don't know how to unlock <<self.thedesc>>. ";
        }
        else if (!self.islocked)
        {
            "\^<<self.itisdesc>> not locked! ";
        }
        else if (self.mykey == nil)
        {
            "%You% %do%n't need anything to unlock << self.itobjdesc >>.";
        }
    }
    doUnlockWith(actor, io) =
    {
        if (self.mykey == nil)
        {
            "A key is not required to unlock <<self.thedesc>>. ";
        }
        else if (io == self.mykey)
        {
            "Unlocked. ";
            self.islocked = nil;
        }
        else
            "\^<<io.itnomdesc>> <<io.doesdesc>>n't fit the lock. ";
    }

    ////////////////////////////////////////////////////////////////////
    //
    //  SURFACE / CONTAINER INDIRECT OBJECT METHODS
    //
    ////////////////////////////////////////////////////////////////////

    verIoPutIn(actor) =
    {
        if (!self.iscontainer)
        {
            "%You% can't put anything into <<self.thedesc>>. ";
        }
        else if (!self.isopen)
        {
            "\^<<self.thedesc>> <<self.isdesc>> closed. ";
        }
    }
    ioPutIn(actor, dobj) =
    {
        if (addbulk(self.contents) + dobj.bulk > self.maxbulk)
        {
            "%You% can't fit <<dobj.thatdesc>> in "; self.thedesc; ". ";
        }
        else
        {
            dobj.doPutIn(actor, self);
        }
    }
    verIoPutOn(actor) = 
    {
        if (!self.issurface)
            "There's no good surface on <<self.thedesc>>. ";
    }
    ioPutOn(actor, dobj) =
    {
        dobj.doPutOn(actor, self);
    }

    ////////////////////////////////////////////////////////////////////
    //
    //  ACCESSIBILITY METHODS
    //
    ////////////////////////////////////////////////////////////////////

    passToContents(plist, filter) = (true)
    passToLocation(plist, filter) = {
        local i, ptr, list;
                
        switch(filter)
        {
	        case &access_visible:
                ptr = &seeInto;
                break;
            case &access_audible:
                ptr = &hearInto;
                break;
            case &access_olfactory:
                ptr = &smellInto;
                break;
            case &access_touchable:
                ptr = &touchInto;
                break;
            case &access_reachable:
                ptr = &reachInto;
                break;
            default:
                return nil;
        }
        
        list = self.retrieveLocs(self, ptr);
        
        if (list == true || list == nil)
            return list;
        
        for (i = 1; i <= length(list); ++i)
        {
            if (inPlist(plist, list[i], PTH_TLOC))
                return true;
        }
        return nil;
    }
    retrieveLocs(o, ptr) = {
        local list;
        
        switch(proptype(o, ptr))
        {
            case DTY_OBJECT:
                return [ self.(ptr) ];
                
            case DTY_LIST:
                return (self.(ptr));
                
            case DTY_CODE:
                list = self.(ptr);
                break;

            case DTY_TRUE:
                return true;
                
            case DTY_NIL:
                return nil;
                
            default:
                return nil;
        }
        switch(datatype(list))
        {
            case DTY_OBJECT:
                return [ list ];
                        
            case DTY_LIST:
                return (self.(ptr));
                
            case DTY_TRUE:
                return true;
                
            default:
                return nil;
        }
    }
    
    seeInto = {
        return self.location;
    }
    smellInto = {
        return self.location;
    }
    hearInto = {
        return self.location;
    }
    touchInto = {
        return self.location; 
    }
    reachInto = {
        return self.location;
    }
    
    cantSee(vantage, target) = {
        local i;
        
        "%You% can't see any ";
        for (i = 1; i <= length(command.cantSenseWords); ++i)
        {
            say(command.cantSenseWords[i]);
            " ";
        }
        if (self.location && contentsVisible)
            "from <<self.thedescR>>. ";
        else
            "here. ";
    }
    cantHear(vantage, target) = {
        local failloc, passto;
        
        passto = getAccessfailed(target, GAF_PASSTO_METHOD);
        if (!testScope(vantage, target, &access_visible))
            failloc = getAccessfailed(target, GAF_FAILLOC);
        
        if (failloc)
            failloc.cantSee(vantage, target);
        else if (contentsVisible && passto == &passAcrossContents)
            "%You%'ll have to open <<self.thedescR>> first. ";
        else if (self.location && contentsAudible)
            "%You% can't hear <<target.thedescR>> from <<self.thedesc>>. ";
        else
            "%You% can't hear <<target.thedescR>> from here. ";
    }    
    cantSmell(vantage, target) = {
        local failloc, passto;
        
        passto = getAccessfailed(target, GAF_PASSTO_METHOD);
        if (!testScope(vantage, target, &access_visible))
            failloc = getAccessfailed(target, GAF_FAILLOC);
        
        if (failloc)
            failloc.cantSee(vantage, target);
        else if (contentsVisible && passto == &passAcrossContents)
            "%You%'ll have to open <<self.thedescR>> first. ";
        else if (self.location && self.contentsOlfactory)
            "%You% can't smell that from <<self.thedescR>>. ";
        else
            "%You% can't smell that from here. ";
    }    
    cantTouch(vantage, target) = {
        local failloc, passto;
        
        passto = getAccessfailed(target, GAF_PASSTO_METHOD);
        if (!testScope(vantage, target, &access_visible))
            failloc = getAccessfailed(target, GAF_FAILLOC);
        
        if (failloc)
            failloc.cantSee(vantage, target);
        else if (self.contentsVisible && passto == &passAcrossContents)
            "%You%'ll have to open <<self.thedescR>> first. ";
        else if (self.location && self.contentsTouchable)
            "%You% can't touch that from <<self.thedescR>>. ";
        else
            "%You% can't touch that from here. ";
    }
    cantReach(vantage, target) = {
        local failloc, passto;
        
        passto = getAccessfailed(target, GAF_PASSTO_METHOD);
        if (!testScope(vantage, target, &access_visible))
            failloc = getAccessfailed(target, GAF_FAILLOC);
        
        if (failloc)
            failloc.cantSee(vantage, target);
        else if (self.contentsVisible && passto == &passAcrossContents)
            "%You%'ll have to open <<self.thedescR>> first. ";
        else if (self.location && self.contentsReachable)
            "%You% can't reach that from <<self.thedescR>>. ";
        else
            "%You% can't reach that from here. ";
    }
    
    verGrab(obj) = {
        if (self.contentsVisible
        && !(self.contentsTouchable
            || self.contentsReachable))
            "%You% will have to open << self.thedesc >> first. ";
    }

    ////////////////////////////////////////////////////////////////////
    //
    //  "ROOM & NESTEDROOM" RELATED METHODS
    //
    ////////////////////////////////////////////////////////////////////
        
    /*
     *   roomCheck is true if the verb is valid in the room.  This
     *   is a first pass; generally, its only function is to disallow
     *   certain commands in a dark room.
     */
    roomCheck(v) = {
        if (v == openVerb 
        || v == getOutVerb
        || v == getOffVerb
        || v == disembarkVerb)
            return true;
        else if (self.islit || v.isDarkVerb)
            return true;
        else
        {
            "It's pitch black.\n";
            return nil;
        }
    }
    roomAction(actor, v, dobj, prep, io) = {
        if (v == openVerb && find(buildClist(actor), dobj))
            return;
        if (v.isTravelVerb 
        || v == getOutVerb
        || v == getOffVerb
        || v == disembarkVerb)
            return;
            
        if (!self.islit && !v.isDarkVerb)
        {
            "%You% can't see a thing. ";
            exit;
        }
        else if (self.location && self.contentsVisible)
        {                
            self.location.roomAction(actor, v, dobj, prep, io);
        }
    }

    /*
     *   Whenever a normal object (i.e., one that does not override the
     *   default doDrop provided by 'thing') is dropped, the actor's
     *   location is sent roomDrop(object being dropped).  By default, 
     *   we move the object into this room.
     */
    roomDrop(obj) = {
        "Dropped. ";
        if (self.location && self.isopen && !self.isdroploc)
            obj.moveInto(self.location);
        else
            obj.moveInto(self);
    }
    
    // sent to room as actor is entering it    
    enterRoom(actor) = {
        if (self.issurface || self.iscontainer || self.isvehicle)
            return;
            
        self.senseAround(actor, (!self.isseen) || global.verbose);
        if (self.islit)
        {
            if (!self.isseen)
                self.firstseen;
            self.isseen = true;
        }
    }

    leaveRoom(actor) = {}

    /*
     *   dispParagraph - display the paragraph separator.  By default we
     *   display a \n\t sequence; games can change this (via 'modify') to
     *   customize the display format 
     */
    dispParagraph = "\n\t"

    /*
     *   dispBeginSdesc and dispEndSdesc - display begin and end of
     *   location short description sequence.  We'll call these just
     *   before and after showing a room's description.  By default these
     *   do nothing; games that want to customize the display format can
     *   change these (via 'modify').  
     */
    dispBeginSdesc = ""
    dispEndSdesc = ""

    /*
     *   dispBeginLdesc and dispEndLdesc - display begin and end of
     *   location long description sequence.  We'll call these just before
     *   and after showing a room's long description.  By default, we
     *   start a new paragraph before the ldesc; games that want to
     *   customize the display format can change these (via 'modify').  
     */
    dispBeginLdesc = { self.dispParagraph; }
    dispEndLdesc = ""

    senseAround(actor, verbosity) = {
        self.dispBeginSdesc;
        self.statusRoot;
        self.dispEndSdesc;
        
        self.lookAround(actor, verbosity, nil);
        self.listenAround(actor, verbosity, nil);
        self.smellAround(actor, verbosity, nil);
    }
    lookAround(actor, verbosity, sroot) = {
        if (self.islit)
        {
            if (sroot)
            {
                self.dispBeginSdesc;
                self.statusRoot;
                self.dispEndSdesc;
            }
            if (self.location && self.contentsVisible)
            {
                if (self.isopen)
                    self.location.nrmLkAround(actor, verbosity);
                else
                    self.location.xtndLkAround(actor, verbosity);
                if (self.isopen)
                    self.xtndLkAround(actor, verbosity);
                else
                    self.nrmLkAround(actor, verbosity);
            }
            else
            {
                self.nrmLkAround(actor, verbosity);
            }
        }
        else
        {
            self.dispParagraph;
            "It's pitch black. ";
        }
    }

    /*
     *  lookAround describes the room.  If verbosity is true, the full
     *  description is given, otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     *
     *  lookAround without location status     
     */
    nrmLkAround(actor, verbosity) = {
        local i, locList = [];
        local lookList  = self.contents;
        
        if (parserGetMe().location == self)
            lookList += parserGetMe(); 
        lookList -= [actor];
        lookList -= [actor.location];

        locList = self.retrieveLocs(self, &seeInto);
        if (locList == nil)
            locList = [];

        locList -= buildClist(actor);
        
        if (verbosity)
        {
            if (proptype(self, &insideDesc) != DTY_NIL)
            {
                self.dispBeginLdesc;
                self.insideDesc;
                self.dispEndLdesc;
            }
            else if (self.location == nil)
            {
                self.dispBeginLdesc;
                self.ldescR;
                self.dispEndLdesc;
            }
        
            for (i = 1; i <= length(lookList); ++i)
            {
                if (self.contents[i].isfixed)
                    self.contents[i].heredescR;
            }
        }

        //
        //  list all ACTORS in the room except the actor
        //
        for (i = 1; i <= length(lookList); ++i)
        {
            if (lookList[i].isactor)
            { 
                self.dispParagraph;
                lookList[i].actorDescR;
            }
        }

        self.dispParagraph;
        if (itemcnt(lookList))
        {
            "%You% see%s% "; listcont(lookList); " here. ";
        }
        listcontcont(lookList); "\n";

        "\n";
           
        //
        //  Display xtndLkAround() for each location which forms a valid 
        //
        for (i = 1; i <= length(locList); ++i)
        {
            if (testScope(actor, locList[i], &access_visible))
                locList[i].xtndLkAround(actor, verbosity);
        }
    }
    /*
     *   xtndLkAround describes the room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the room's ldesc) is displayed.
     */
    xtndLkAround(actor, verbosity) = {
        local i;
        local lookList = self.contents;
        
        if (parserGetMe().location == self)
            lookList += parserGetMe();
        lookList -= [actor];
        lookList -= [actor.location];
        
        if (verbosity 
        && self.isListed)
        {
            if (proptype(self, &insideDesc) != DTY_NIL)
            {
                self.dispBeginLdesc;
                self.insideDesc;
                self.dispEndLdesc;
            }
            else if (self.location == nil)
            {
                self.dispBeginLdesc;
                self.ldescR;
                self.dispEndLdesc;
            }
        }
        
        if (self.isqVisible) return;
        
        if (verbosity)
        {
            for (i = 1; i <= length(self.contents); ++i)
            {
                if (self.contents[i].isfixed)
                    self.contents[i].heredescR;
            }
        }
        
        self.dispParagraph;        
        if (itemcnt(lookList))
        {
            "\^<<self.statusPrep>> <<self.thedescR>> %you% can see "; 
            listcont(lookList); ". ";
        }
        listcontcont(lookList); "\n";

        //
        //  list all ACTORS in the room except the actor
        //
        for (i = 1; i <= length(lookList); ++i)
        {
            if (lookList[i].isactor)
            {   
                self.dispParagraph;
                lookList[i].actorDescR;
            }
        }

        "\n";
    }
    listenAround(actor, verbosity, sroot) = {        
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }        
        self.nrmLnAround(actor, verbosity);   
    }
    /*
     *  nrmLnAround describes the room.  
     *  If verbosity is true, the full description is given, 
     *  otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     */
    nrmLnAround(actor, verbosity) = {
        local i, locList = [], disptot = 0, ret;
        local verb          = command.verbPtr;

        locList = self.retrieveLocs(self, &hearInto);
        if (locList == nil)
            locList = [];
        
        if (verbose 
        && self.isListedAudible) 
        {
            self.dispBeginLdesc;
            self.listendescR;
            self.dispEndLdesc;
            ++disptot;
        }      

        if (self.isqAudible) return;

        self.dispParagraph;
        
        disptot += self.dispcont(actor, &access_audible, 
            &isListedAudible, &listendescR);
            
        "\n";
       
        for (i = 1; i <= length(locList); ++i)
        {
            if (testScope(actor, locList[i], &access_audible))
                disptot += locList[i].xtndLnAround(actor, verbosity);
        }
        
        if (disptot == 0
        && verb == listenVerb)
        {
            self.dispBeginLdesc;
            self.listendescR;
            self.dispEndLdesc;
        }
    }
    /*
     *  xtndLnAround describes the room.  
     *  If verbosity is true, the full description is given, 
     *  otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     */
    xtndLnAround(actor, verbosity) = {
        local disptot = 0;

        if (verbosity 
        && self.isListedAudible)
        {
            self.dispBeginLdesc;
            self.listendescR;
            self.dispEndLdesc;
            ++disptot;
        }
        
        if (self.isqAudible) return disptot;
        
        self.dispParagraph;
        
        disptot += self.dispcont(actor, &access_audible,
            &isListedAudible, &listendescR);
            
        "\n";
        return disptot;
    }
    smellAround(actor, verbosity, sroot) = {        
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }        
        self.nrmSmAround(actor, verbosity);   
    }
    /*
     *   nrmSmAround describes the room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the room's ldesc) is displayed.
     */
    nrmSmAround(actor, verbosity) = {
        local i, locList = [], disptot = 0, ret;
        local verb      = command.verbPtr;

        locList = self.retrieveLocs(self, &smellInto);
        if (locList == nil)
            locList = [];
        
        if (verbose 
        && self.isListedOlfactory)
        {
            self.dispBeginLdesc;
            self.smelldescR;
            self.dispEndLdesc;
            ++disptot;
        }      

        if (self.isqOlfactory) return;

        self.dispParagraph;
        
        disptot += self.dispcont(actor, &access_olfactory,
            &isListedOlfactory, &smelldescR);
            
        "\n";
       
        for (i = 1; i <= length(locList); ++i)
        {
            if (testScope(actor, locList[i], &access_olfactory))
                disptot += locList[i].xtndSmAround(actor, verbosity);
        }
        
        if (disptot == 0
        && verb == smellVerb)
        {
            self.dispBeginLdesc;
            self.smelldescR;
            self.dispEndLdesc;
        }
    }
    /*
     *  xtndSmAround describes objects outside of the room.  
     *  If verbosity is true, the full description is given, 
     *  otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     */
    xtndSmAround(actor, verbosity) = {
        local disptot = 0;

        if (verbosity 
        && self.isListedOlfactory)
        {
            self.dispBeginLdesc;
            self.smelldescR;
            self.dispEndLdesc;
            ++disptot;
        }
        
        if (self.isqOlfactory) return disptot;
        
        self.dispParagraph;
        
        disptot += self.dispcont(actor, &access_olfactory,
            &isListedOlfactory, &smelldescR);
            
        "\n";
        return disptot;
    }
    
#ifdef USE_HTML_STATUS

    /* 
     *   Generate HTML formatting that provides the traditional status
     *   line appearance, using the traditional status line computation
     *   mechanism.
     */
    statusLine = {
        /*
         *   Check the system to see if this is an HTML-enabled run-time.
         *   If so, generate an HTML-style banner; otherwise, generate an
         *   old-style banner. 
         */
        if (systemInfo(__SYSINFO_SYSINFO) == true
        && systemInfo(__SYSINFO_HTML) == 1)
        {
            "<banner id=StatusLine height=previous border>
            <body bgcolor=statusbg text=statustext><b>";

            self.statusRoot;
            
            "</b><tab align=right><i>";

            say(global.score); "/";
            say(global.turnsofar); " ";
        
            "</i></banner>";
        }
        else
        {
            self.statusRoot;
            self.dispParagraph;
        }
    }
    
#else /* USE_HTML_STATUS */

    /* use the standard HTML status line system */
    statusLine = {
        self.statusRoot;
        self.dispParagraph;
    }

#endif /* USE_HTML_STATUS */

    statusRoot = {
        caps();
        if (!self.islit)
            "In The Dark";
        else if (self.location && self.contentsVisible)
            "<<self.location.sdesc>>, <<self.statusPrep>> <<self.thedesc>>";  
        else
            self.sdesc;
    }
    
    /*
     *   Direction handlers.  The directions are all set up to
     *   the default, which is no travel allowed.  To make travel
     *   possible in a direction, just assign a room to the direction
     *   property.
     */
    north = { return self.noexit; }
    south = { return self.noexit; }
    east  = { return self.noexit; }
    west  = { return self.noexit; }
    up    = { return self.noexit; }
    down  = { return self.noexit; }
    ne    = { return self.noexit; }
    nw    = { return self.noexit; }
    se    = { return self.noexit; }
    sw    = { return self.noexit; }
    in    = { return self.noexit; }
    out   = { 
        if (self.isvehicle)
            return self.location;
        else
            return self.noexit;
    }
    
    /*
     *   noexit displays a message when the player attempts to travel
     *   in a direction in which travel is not possible.
     */
    noexit = {
        if (self.islit)
        {
            if (self.issurface || self.iscontainer)
                "%You're% not going anywhere until %you%
                get%s% <<outOfPrep>> <<thedesc>>. ";
            else
                "%You% can't go that way. "; 
        }
        else
        {
            darkTravel();
        }
    }
;

/*----------------------------------------------------------------------
 *	ADV.T CLASS REPLACEMENTS
/*--------------------------------------------------------------------*/

replace surface: containment
    issurface       = true
;
replace qsurface: containment
    isqsurface      = true
    isqVisible      = true
    isqAudible      = true
    isqOlfactory    = true
;
replace container: containment
    iscontainer     = true
;
replace qcontainer: containment
    isqcontainer    = true
    isqVisible      = true
    isqAudible      = true
    isqOlfactory    = true
;
replace openable: containment
    isopenable          = true
;
replace lockable: containment
    islockable      = true
;    
replace keyedLockable: containment
    islockable      = true
;
replace transparentItem: containment
    istransparent   = true
;
replace permeable: containment
    ispermeable = true
;
replace room: containment
    lightsOn        = true
;
replace darkroom: containment
;
replace nestedroom: containment
;
replace vehicle: containment
    isvehicle       = true
    iscontainer     = true
;
replace chairitem: fixeditem, containment
    issurface       = true
    isenterable     = true
;
replace beditem: fixeditem, containment
    issurface = true
    isenterable = true
;
modify movableActor
    passToLocation(plist, filter) = true
    passToContents(plist, filter) = true
    isListedAudible     = nil
    isListedOlfactory   = nil
    
    isqVisible          = true
    isqAudible          = nil
    isqOlfactory        = nil

    /*------------------------------------------------------------------
     *  Sense Descriptor Routers
     *
     *  Each Router must also define a list of description method points
     *  ordered by reflexive, possessive, contained, immediate, local,
     *  and remote. 
     *
     *  Default description methods should be defined at the local
     *  position in the list. Some examples of default description 
     *  methods are sdesc, adesc, thedesc, ldesc, readdesc, and
     *  actorDesc.
     *
     *  Messages built using a Descriptor Router will choose the 
     *  appropriate description message, based on the proximity of the
     *  vantage and target.
     *----------------------------------------------------------------*/

    actorDescR = {
        return proxdesc(command.actorPtr, self, self.actorDescList);
    }
    actorDescList = [ &actorDesc_reflexive, &actorDesc_possessive,
        &actorDesc_contained, &actorDesc_immediate, &actorDesc,
        &actorDesc_remote ]
    actorDesc_remote = { "\^<<self.adesc>> is there. "; }
;

modify basicMe
    isqVisible          = true
    isqAudible          = true
    isqOlfactory        = true
;

#else /* USE_CONTAINMENT */

modify qsurface
    isqVisible          = true
    isqAudible          = true
    isqOlfactory        = true
;

modify qcontainer
    isqVisible          = true
    isqAudible          = true
    isqOlfactory        = true
;

modify openable
    contentsVisible     = { return self.isopen; }
    contentsAudible     = { return self.isopen; }
    contentsOlfactory   = { return self.isopen; }
    contentsTouchable   = { return self.isopen; }
    replace contentsReachable   = { return self.isopen; }
;

modify transparentItem
    replace contentsVisible     = (true)
    contentsAudible             = { return self.isopen; }
    contentsOlfactory           = { return self.isopen; }
    contentsTouchable           = { return self.isopen; }
    replace contentsReachable   = { return self.isopen; }
    
    cantHear(vantage, target) = {        
        if (getAccessfailed(target, GAF_PASSTO_METHOD) !=
            &passAcrossContents)
            pass cantHear;

        "%You%'ll have to open <<self.thedesc>> first. ";
    }
    cantSmell(vantage, target) = {    
        if (getAccessfailed(target, GAF_PASSTO_METHOD) !=
            &passAcrossContents)
            pass cantSmell;

        "%You%'ll have to open <<self.thedesc>> first. ";
    }
    cantTouch(vantage, target) = {        
        if (getAccessfailed(target, GAF_PASSTO_METHOD) !=
            &passAcrossContents)
            pass cantTouch;
            
        "%You%'ll have to open <<self.thedesc>> first. ";
    }
    cantReach(vantage, target) = {        
        if (getAccessfailed(target, GAF_PASSTO_METHOD) !=
            &passAcrossContents)
            pass cantReach;
            
        "%You%'ll have to open <<self.thedesc>> first. ";
    }
;

/*
 *  permeable: openable
 *
 *  Permeable class objects allow for all access unconditionally except
 *  for access_touchable. This class is good for creating cages and such
 *  where you can reach an object only by projecting something at it.
 */
class permeable: openable
    contentsVisible     = (true)
    contentsAudible     = (true)
    contentsOlfactory   = (true)
    contentsTouchable   = {return self.isopen;}
    contentsReachable   = (true)
    cantHear(vantage, target) = {        
        if (getAccessfailed(target, GAF_PASSTO_METHOD) !=
            &passAcrossContents)
            pass cantHear;

        "%You%'ll have to open <<self.thedesc>> first. ";
    }
    cantSmell(vantage, target) = {    
        if (getAccessfailed(target, GAF_PASSTO_METHOD) !=
            &passAcrossContents)
            pass cantSmell;

        "%You%'ll have to open <<self.thedesc>> first. ";
    }
    cantTouch(vantage, target) = {        
        if (getAccessfailed(target, GAF_PASSTO_METHOD) !=
            &passAcrossContents)
            pass cantTouch;
            
        "%You%'ll have to open <<self.thedesc>> first. ";
    }
    cantReach(vantage, target) = {        
        if (getAccessfailed(target, GAF_PASSTO_METHOD) !=
            &passAcrossContents)
            pass cantReach;
            
        "%You%'ll have to open <<self.thedesc>> first. ";
    }
;

/*
 *  MODIFIED TO USE SENSE-PASSING METHODS AND ENHANCED ROOM DISPLAY
 *
 *  room: thing
 *
 *  A location in the game.  By default, the islit property is
 *  true, which means that the room is lit (no light source is
 *  needed while in the room).  You should create a darkroom
 *  object rather than a room with islit set to nil if you
 *  want a dark room, because other methods are affected as well.
 *  The isseen property records whether the player has entered
 *  the room before; initially it's nil, and is set to true
 *  the first time the player enters.  The roomAction(actor,
 *  verb, directObject, preposition, indirectObject) method is
 *  activated for each player command; by default, all it does is
 *  call the room's location's roomAction method if the room
 *  is inside another room.  The lookAround(verbosity)
 *  method displays the room's description for a given verbosity
 *  level; true means a full description, nil means only
 *  the short description (just the room name plus a list of the
 *  objects present).  roomDrop(object) is called when
 *  an object is dropped within the room; normally, it just moves
 *  the object to the room and displays "Dropped."  The firstseen
 *  method is called when isseen is about to be set true
 *  for the first time (i.e., when the player first sees the room);
 *  by default, this routine does nothing, but it's a convenient
 *  place to put any special code you want to execute when a room
 *  is first entered.  The firstseen method is called after
 *  the room's description is displayed.
 */
modify room
    passToContents(plist, filter) = (true)
    passToLocation(plist, filter) = {
        local i, ptr, list;
                
        switch(filter)
        {
	        case &access_visible:
                ptr = &seeInto;
                break;
            case &access_audible:
                ptr = &hearInto;
                break;
            case &access_olfactory:
                ptr = &smellInto;
                break;
            case &access_touchable:
                ptr = &touchInto;
                break;
            case &access_reachable:
                ptr = &reachInto;
                break;
            default:
                return nil;
        }
        
        list = self.retrieveLocs(self, ptr);
        
        if (list == true || list == nil)
            return list;
        
        for (i = 1; i <= length(list); ++i)
        {
            if (inPlist(plist, list[i], PTH_TLOC))
                return true;
        }
        return nil;
    }
    retrieveLocs(o, ptr) = {
        local list;
        
        switch(proptype(o, ptr))
        {
            case DTY_OBJECT:
                return [ self.(ptr) ];
                
            case DTY_LIST:
                return (self.(ptr));
                
            case DTY_CODE:
                list = self.(ptr);
                break;

            case DTY_TRUE:
                return true;
                
            case DTY_NIL:
                return nil;
                
            default:
                return nil;
        }
        switch(datatype(list))
        {
            case DTY_OBJECT:
                return [ list ];
                        
            case DTY_LIST:
                return (self.(ptr));
                
            case DTY_TRUE:
                return true;
                
            default:
                return nil;
        }
    }
        
    replace enterRoom(actor) = {
        self.senseAround(actor, (!self.isseen) || global.verbose);
        if (self.islit)
        {
            if (!self.isseen)
                self.firstseen;
            self.isseen = true;
        }
    }
    senseAround(actor, verbosity) = {
        self.dispBeginSdesc;
        self.statusRoot;
        self.dispEndSdesc;
        
        self.lookAround(actor, verbosity, nil);
        self.listenAround(actor, verbosity, nil);
        self.smellAround(actor, verbosity, nil);
    }
    replace lookAround(actor, verbosity, sroot) = {
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }            
        self.nrmLkAround(actor, verbosity);       
    }
    /*
     *   nrmLkAround describes the room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the room's ldesc) is displayed.
     */
    replace nrmLkAround(actor, verbosity) = {
        local i, plist, locList = [];
        local lookList  = self.contents + parserGetMe() - actor;
        
        locList = self.retrieveLocs(self, &seeInto);
        if (locList == true)
            locList = global.tlist - self;
        else if (locList == nil)
            locList = [];

        if (verbosity)
        {
            self.dispBeginLdesc;
            self.ldescR;
            self.dispEndLdesc;
            
            for (i = 1; i <= length(self.contents); ++i)
            {
                if (self.contents[i].isfixed)
                    self.contents[i].heredescR;
            }
        }

        self.dispParagraph;
        if (itemcnt(self.contents))
        {
            "%You% see%s% "; listcont(self); " here. ";
        }
        listcontcont(self); "\n";

        //
        //  list all ACTORS in the room except the actor
        //
        for (i = 1; i <= length(lookList); ++i)
        {
            if (lookList[i].isactor)
            { 
                self.dispParagraph;
                lookList[i].actorDescR;
            }
        }
        "\n";
           
        //
        //  Display xtndLkAround() for each location which forms a valid 
        //
        for (i = 1; i <= length(locList); ++i)
        {
            plist = path(actor, locList[i]);
            if (blocksPath(plist, &access_visible) == nil)
                locList[i].xtndLkAround(actor, verbosity);
        }
    }
    /*
     *   xtndLkAround describes the room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the room's ldesc) is displayed.
     */
    xtndLkAround(actor, verbosity) = {
        local i;
        local lookList = self.contents + parserGetMe() - actor;
        
        if (verbosity 
        && self.isListed)
        {
            self.dispBeginLdesc;
            self.ldescR;
            self.dispEndLdesc;
        }
        
        if (self.isqVisible) return;
        
        if (verbosity)
        {
            for (i = 1; i <= length(self.contents); ++i)
            {
                if (self.contents[i].isfixed)
                    self.contents[i].heredescR;
            }
        }
        
        self.dispParagraph;
        
        if (itemcnt(self.contents))
        {
            if (itemcnt(self.contents))
                "\^<<self.statusPrep>> <<self.thedesc>> %you% can see "; 
            listcont(self); ". ";
        }
        listcontcont(self); "\n";

        //
        //  list all ACTORS in the room except the actor
        //
        for (i = 1; i <= length(lookList); ++i)
        {
            if (lookList[i].isactor)
            {   
                self.dispParagraph;
                lookList[i].actorDescR;
            }
        }
        "\n";
    }
    listenAround(actor, verbosity, sroot) = {        
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }        
        self.nrmLnAround(actor, verbosity);   
    }
    /*
     *  nrmLnAround describes the room.  
     *  If verbosity is true, the full description is given, 
     *  otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     */
    nrmLnAround(actor, verbosity) = {
        local i, plist, locList = [], disptot = 0, ret;
        local verb          = command.verbPtr;

        locList = self.retrieveLocs(self, &hearInto);
        if (locList == true)
            locList = global.tlist - self;
        else if (locList == nil)
            locList = [];
        
        if (verbose 
        && self.isListedAudible) 
        {
            self.dispBeginLdesc;
            self.listendescR;
            self.dispEndLdesc;
            ++disptot;
        }      

        if (self.isqAudible) return;

        self.dispParagraph;
        
        disptot += self.dispcont(actor, &access_audible, 
            &isListedAudible, &listendescR);
            
        "\n";
       
        for (i = 1; i <= length(locList); ++i)
        {
            plist = path(actor, locList[i]);
            if (blocksPath(plist, &access_audible) == nil)
                disptot += locList[i].xtndLnAround(actor, verbosity);
        }
        
        if (disptot == 0
        && verb == listenVerb)
        {
            self.dispBeginLdesc;
            self.listendescR;
            self.dispEndLdesc;
        }
    }
    /*
     *  xtndLnAround describes the room.  
     *  If verbosity is true, the full description is given, 
     *  otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     */
    xtndLnAround(actor, verbosity) = {
        local disptot = 0;

        if (verbosity 
        && self.isListedAudible)
        {
            self.dispBeginLdesc;
            self.listendescR;
            self.dispEndLdesc;
            ++disptot;
        }
        
        if (self.isqAudible) return disptot;
        
        self.dispParagraph;
        
        disptot += self.dispcont(actor, &access_audible,
            &isListedAudible, &listendescR);
            
        "\n";
        return disptot;
    }
    smellAround(actor, verbosity, sroot) = {        
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }        
        self.nrmSmAround(actor, verbosity);   
    }
    /*
     *   nrmSmAround describes the room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the room's ldesc) is displayed.
     */
    nrmSmAround(actor, verbosity) = {
        local i, plist, locList = [], disptot = 0, ret;
        local verb      = command.verbPtr;

        locList = self.retrieveLocs(self, &smellInto);
        if (locList == true)
            locList = global.tlist - self;
        else if (locList == nil)
            locList = [];
        
        if (verbose 
        && self.isListedOlfactory)
        {
            self.dispBeginLdesc;
            self.smelldescR;
            self.dispEndLdesc;
            ++disptot;
        }      

        if (self.isqOlfactory) return;

        self.dispParagraph;
        
        disptot += self.dispcont(actor, &access_olfactory,
            &isListedOlfactory, &smelldescR);
            
        "\n";
       
        for (i = 1; i <= length(locList); ++i)
        {
            plist = path(actor, locList[i]);
            if (blocksPath(plist, &access_olfactory) == nil)
                disptot += locList[i].xtndSmAround(actor, verbosity);
        }
        
        if (disptot == 0
        && verb == smellVerb)
        {
            self.dispBeginLdesc;
            self.smelldescR;
            self.dispEndLdesc;
        }
    }
    /*
     *  xtndSmAround describes objects outside of the room.  
     *  If verbosity is true, the full description is given, 
     *  otherwise an abbreviated description (without
     *  the room's ldesc) is displayed.
     */
    xtndSmAround(actor, verbosity) = {
        local disptot = 0;

        if (verbosity 
        && self.isListedOlfactory)
        {
            self.dispBeginLdesc;
            self.smelldescR;
            self.dispEndLdesc;
            ++disptot;
        }
        
        if (self.isqOlfactory) return disptot;
        
        self.dispParagraph;
        
        disptot += self.dispcont(actor, &access_olfactory,
            &isListedOlfactory, &smelldescR);
            
        "\n";
        return disptot;
    }
    cantTouch(vantage, target) = {
        if (testScope(vantage, target, &access_visible))
            "%You% can't touch that from here. ";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }
    replace cantReach(vantage, target) = {
        if (testScope(vantage, target, &access_visible))
            "%You% can't reach that from here. ";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }
;

/*
 *  MODIFIED LOOKAROUND TO PASS EXTRA ACTOR AND SROOT PARAMETER
 *
 *  darkroom: room
 *
 *  A dark room.  The player must have some object that can act as a
 *  light source in order to move about and perform most operations
 *  while in this room.  Note that the room's lights can be turned
 *  on by setting the room's lightsOn property to true;
 *  do this instead of setting islit, because islit is
 *  a method which checks for the presence of a light source.
 */
modify darkroom
    replace lookAround(actor, verbosity, sroot) = {
        if (self.islit)
            pass lookAround;
        else
        {
            self.dispParagraph;
            "It's pitch black. ";
        }
    }
    replace statusRoot = {
        if (self.islit)
            pass statusRoot;
        else
            "In The Dark";
    }
;

/*
 *  MODIFIED TO ADD SCOPE-PASSING METHODS
 *
 *  nestedroom: room
 *
 *  A special kind of room that is inside another room; chairs and
 *  some types of vehicles, such as inflatable rafts, fall into this
 *  category.  Note that a room can be within another room without
 *  being a nestedroom, simply by setting its location property
 *  to another room.  The nestedroom is different from an ordinary
 *  room, though, in that it's an "open" room; that is, when inside it,
 *  the actor is still really inside the enclosing room for purposes of
 *  descriptions.  Hence, the player sees "Laboratory, in the chair."
 *  In addition, a nestedroom is an object in its own right,
 *  visible to the player; for example, a chair is an object in a
 *  room in addition to being a room itself.  The statusPrep
 *  property displays the preposition in the status description; by
 *  default, it will be "in," but some subclasses and instances
 *  will want to change this to a more appropriate preposition.
 *  outOfPrep is used to report what happens when the player
 *  gets out of the object:  it should be "out of" or "off of" as
 *  appropriate to this object.
 */
modify nestedroom
    passToContents(plist, filter)   = (true)
    passToLocation(path, filter)    = (true)

    lookAround(actor, verbosity, sroot) = {
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }
        self.location.nrmLkAround(actor, verbosity);
    }
    listenAround(actor, verbosity, sroot) = {
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }
        self.location.nrmLnAround(actor, verbosity);
    }
    smellAround(actor, verbosity, sroot) = {
        if (sroot)
        {
            self.dispBeginSdesc;
            self.statusRoot;
            self.dispEndSdesc;
        }
        self.location.nrmSmAround(actor, verbosity);
    }
    
    cantSee(vantage, target) = {
        local i;
        
        "%You% can't see any ";
        for (i = 1; i <= length(command.cantSenseWords); ++i)
        {
            say(command.cantSenseWords[i]);
            " ";
        }
        "from <<self.thedesc>>. ";
    }
    cantHear(vantage, target) = {
        if (testScope(vantage, target, &access_visible))
            "%You% can't hear that from <<self.thedesc>>. ";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }    
    cantSmell(vantage, target) = {
        if (testScope(vantage, target, &access_visible))
            "%You% can't smell that from <<self.thedesc>>. ";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }    
    cantTouch(vantage, target) = {
        if (testScope(vantage, target, &access_visible))
            "%You% can't touch that from <<self.thedesc>>. ";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }
    replace cantTouch(vantage, target) = {
        if (testScope(vantage, target, &access_visible))
            "%You% can't reach that from <<self.thedesc>>. ";
        else getAccessfailed(target, GAF_FAILLOC).cantSee(vantage,
                target);
    }
;

modify movableActor
    isListedAudible     = (nil)
    isListedOlfactory   = (nil)
    
    isqVisible          = (true)
    isqAudible          = (nil)
    isqOlfactory        = (nil)
    
    /*------------------------------------------------------------------
     *  Sense Descriptor Routers
     *
     *  Each Router must also define a list of description method points
     *  ordered by reflexive, possessive, contained, immediate, local,
     *  and remote. 
     *
     *  Default description methods should be defined at the local
     *  position in the list. Some examples of default description 
     *  methods are sdesc, adesc, thedesc, ldesc, readdesc, and
     *  actorDesc.
     *
     *  Messages built using a Descriptor Router will choose the 
     *  appropriate description message, based on the proximity of the
     *  vantage and target.
     *----------------------------------------------------------------*/

    actorDescR = {
        return proxdesc(command.actorPtr, self, self.actorDescList);
    }
    actorDescList = [ &actorDesc_reflexive, &actorDesc_possessive,
        &actorDesc_contained, &actorDesc_immediate, &actorDesc,
        &actorDesc_remote ]
    actorDesc_remote = { "\^<<self.adesc>> is there. "; }
;

/*
 *  MODIFIED TO INCORPORATE ISQXXXX ATTRIBUTES
 *
 *  basicMe: Actor
 *
 *  A default implementation of the Me object, which is the
 *  player character.  adv.t defines basicMe instead of
 *  Me to allow your game to override parts of the default
 *  implementation while still using the rest, and without changing
 *  adv.t itself.  To use basicMe unchanged as your player
 *  character, include this in your game:  "Me: basicMe;".
 *  
 *  The basicMe object defines all of the methods and properties
 *  required for an actor, with appropriate values for the player
 *  character.  The nouns "me" and "myself" are defined ("I"
 *  is not defined, because it conflicts with the "inventory"
 *  command's minimal abbreviation of "i" in certain circumstances,
 *  and is generally not compatible with the syntax of most player
 *  commands anyway).  The sdesc is "you"; the thedesc
 *  and adesc are "yourself," which is appropriate for most
 *  contexts.  The maxbulk and maxweight properties are
 *  set to 10 each; a more sophisticated Me might include the
 *  player's state of health in determining the maxweight and
 *  maxbulk properties.
 */
modify basicMe
    isqVisible          = (true)
    isqAudible          = (true)
    isqOlfactory        = (true)
;

#endif /* USE_CONTAINMENT */


/*======================================================================
 *
 *  12.  G R A M M A R 
 *
/*====================================================================*/


/*
 *	Modify to use the new ACCESSIBILITY algorithms.
 */
modify deepverb
    dorequires = &access_touchable
    iorequires = &access_touchable
    
    //
    //  Use all parsed objects
    //
    replace validDoList(actor, prep, iobj) = {
        command.disambigObjType = 1;
        command.actorPtr        = actor;
        command.verbPtr         = self;
        command.prepPtr         = prep;
        if (iobj)
            command.iobjPtr     = iobj;
        return nil;
    }
    
    //
    //  Build scope path, then check that path is valid
    //
    replace validDo(actor, obj, seqno) = {        
        command.dobjPtr     = obj;
        if (testScope(actor, obj))
            return true;
        else        
            return (testScope(actor, obj, self.dorequires));
    }
    
    //
    //  Use all parsed objects
    //
    replace validIoList(actor, prep, dobj) = {
        command.disambigObjType = 2;
        command.actorPtr        = actor;
        command.verbPtr         = self;
        command.prepPtr         = prep;
        if (dobj)
            command.dobjPtr     = dobj;
        return nil;
    }
    
    //
    //  Build scope path, then check that path is valid
    //
    replace validIo(actor, obj, seqno) = {        
        command.iobjPtr     = obj;
        if (testScope(actor, obj))
            return true;
        else        
            return (testScope(actor, obj, self.iorequires));
    }
    
	//
	//  This function is called when nothing matched by the
	//  do/io vocabulary is accessible.
	//
	cantReach(actor, dolist, iolist, prep) = {		
	    local	i, list, cmdWords;
	    
		if (dolist != nil) {
			list = dolist;
			cmdWords = getCmdPhrase(dolist[1], global.cmdList);
			if (cmdWords == nil)
			    command.cantSenseWords = objwords(1);
		    else
		        command.cantSenseWords = cmdWords;
		}
		else {
			list = iolist;
			cmdWords = getCmdPhrase(iolist[1], global.cmdList);
			if (cmdWords == nil)
			    command.cantSenseWords = objwords(2);
		    else
		        command.cantSenseWords = cmdWords;
		}

        if (self.cantSenseActor(actor)) return;
        
        for (i = 1; i <= length(list); ++i)
        {
            if (dolist)
                self.cantSenseDoRedirect(actor, list[i], prep);
            else
                self.cantSenseIoRedirect(actor, list[i], prep);
        }
	}
	cantSenseActor(actor) = {
	    if (actor != command.actorPtr) return nil;
	    
        if (!actor.isactor)
        {
            "\^<<actor.thedesc>> <<actor.doesdesc>> not respond.";
            return true;
        }
        if (!isclass(actor, movableActor))
        {
            "There's no point in talking to <<actor.sdesc>>.";
            return true;
        }
        return nil;
	}
	cantSenseDoRedirect(vantage, target, prep) = {
	    local filter, failloc;
	    
	    filter  = getAccessfailed(target, GAF_ACCESS_RULE);
	    failloc = getAccessfailed(target, GAF_FAILLOC);
                            
        switch(filter)
        {
            case &access_visible:
                failloc.cantSee(vantage, target);
                break;
            case &access_audible:
                failloc.cantHear(vantage, target);
                break;
            case &access_olfactory:
                failloc.cantSmell(vantage, target);
                break;
            case &access_touchable:
                failloc.cantTouch(vantage, target);
                break;
            case &access_reachable:
                failloc.cantReach(vantage, target);
                break;
            default:
                failloc.cantSee(vantage, target);
                break;
        }	    
    }
    cantSenseIoRedirect(vantage, target, prep) = {
        self.cantSenseDoRedirect(vantage, target, prep); 
    }
	//
	//  Store the elements of the command and execute preAction object
	//  reactions.
	//
	verbAction( actor, dobj, prep, iobj ) = {
	    command.actorPtr    = actor;
	    command.verbPtr     = self;
	    command.dobjPtr     = dobj;
	    command.prepPtr     = prep;
	    command.iobjPtr     = iobj;
	    
	    objectReactions(1);
	}
;

/*----------------------------------------------------------------------
 *  VERBS FOR TOPIC ACCESS
 *--------------------------------------------------------------------*/

modify askVerb
    dorequires = &access_audible
    iorequires = &access_topic
    replace validIo(actor, obj, seqno) = { 
        command.iobjPtr = obj;
        return (seqno == 1);
    }
;
modify tellVerb
    dorequires = &access_audible
    iorequires = &access_topic
    replace validIo(actor, obj, seqno) = { 
        command.iobjPtr = obj;
        return (seqno == 1); 
    }
;

/*----------------------------------------------------------------------
 *  VERBS FOR SENSE OF SIGHT
 *--------------------------------------------------------------------*/

class visibleVerb: deepverb
    dorequires = &access_visible
;
replace lookVerb: visibleVerb
    verb = 'look' 'l' 'look around' 'l around'
    sdesc = "look"
    action(actor) = {
        actor.location.lookAround(actor, true, true);
    }
;
replace inspectVerb: visibleVerb
    verb = 'inspect' 'examine' 'look at' 'l at' 'x'
    sdesc = "inspect"
    doAction = 'Inspect'
;
replace lookInVerb: visibleVerb
    verb = 'look in' 'look on' 'l in' 'l on'
    sdesc = "look in"
    doAction = 'Lookin'
;
replace lookThruVerb: visibleVerb
    verb = 'look through' 'look thru' 'l through' 'l thru'
    sdesc = "look through"
    doAction = 'Lookthru'
;
replace lookUnderVerb: visibleVerb
    verb = 'look under' 'look beneath' 'l under' 'l beneath'
    sdesc = "look under"
    doAction = 'Lookunder'
;
replace lookBehindVerb: visibleVerb
    verb = 'look behind' 'l behind'
    sdesc = "look behind"
    doAction = 'Lookbehind'
;
replace searchVerb: visibleVerb
    verb = 'search'
    sdesc = "search"
    doAction = 'Search'
;

/*----------------------------------------------------------------------
 *  VERBS FOR SENSE OF HEARING
 *--------------------------------------------------------------------*/

class audibleVerb: darkVerb
    dorequires = &access_audible
;
listenVerb: audibleVerb
    verb = 'listen'
    action(actor) = {
        actor.location.listenAround(actor, true, true);
    }
    sdesc = "listen"
;
listentoVerb: audibleVerb
    verb = 'listen to'
    doAction = 'Listento'
    sdesc = "listen to"
;

/*----------------------------------------------------------------------
 *  VERBS FOR THE SENSE OF SMELL
 *--------------------------------------------------------------------*/

class olfactoryVerb: darkVerb
    dorequires = &access_olfactory
;
smellVerb: olfactoryVerb
    verb = 'smell'
    doAction = 'Smell'
    action(actor) = {
        actor.location.smellAround(actor, true, true);
    }
    sdesc = "smell"
;

/*----------------------------------------------------------------------
 *  VERBS FOR THE SENSE OF TOUCH & TASTE
 *--------------------------------------------------------------------*/

class tactileVerb: deepverb
    dorequires = &access_touchable
;
replace touchVerb: tactileVerb
    verb = 'touch' 'feel'
    sdesc = "touch"
    doAction = 'Touch'
;
replace pokeVerb: tactileVerb
    verb = 'poke' 'jab'
    sdesc = "poke"
    doAction = 'Poke'
;
tasteVerb: tactileVerb
    verb = 'taste'
    doAction = 'Taste'
    sdesc = "taste"
;

/*----------------------------------------------------------------------
 *  VERBS FOR REACHABLE ACCESS
 *--------------------------------------------------------------------*/

class projectileVerb: deepverb
    dorequires = &access_touchable
    iorequires = &access_reachable
;

/*
 *  If the indirect object cannot be reached then we redirect the action 
 *  to the indirect object's location. This is reasonable for such
 *  actions as <<throw x at y>> because we've released the object from 
 *  our possession.
 */
replace throwVerb: deepverb
    dorequires = &access_touchable
    iorequires = &access_reachable
    verb = 'throw' 'toss'
    sdesc = "throw"
    prepDefault = atPrep
    ioAction(atPrep) = 'ThrowAt'
    ioAction(toPrep) = 'ThrowTo'
	cantSenseIoRedirect(vantage, target, prep) = {		
            local filter, failloc, ret;
            
            filter  = getAccessfailed(target, GAF_ACCESS_RULE);
            failloc = getAccessfailed(target, GAF_FAILLOC);
            
            if (!testScope(vantage, target))
                filter  = &access_visible;
                                
            switch(filter)
            {
                case &access_visible:
                    failloc.cantSee(vantage, target);
                    break;
                case &access_audible:
                    failloc.cantHear(vantage, target);
                    break;
                case &access_olfactory:
                    failloc.cantSmell(vantage, target);
                    break;
                case &access_touchable:
                    failloc.cantReach(vantage, target);
                    break;
                case &access_reachable:
                    ret = execCommand(vantage, self,
                        command.dobjPtr, prep, failloc);
                    if (ret == EC_ABORT) abort;
                    exit;
            }
	}
;

/*----------------------------------------------------------------------
 *  MISCELLANEOUS VERBS
 *--------------------------------------------------------------------*/

/*
 *  MODIFIED TO USE parserGetMe().location.senseAround()
 */
modify verboseVerb
    replace verboseMode(actor) =
    {
        "Okay, now in VERBOSE mode.\n";
        global.verbose = true;
        parserGetMe().location.senseAround(actor, true);
    }
;

/*
 *  MODIFED TO USE parserGetMe().location.senseAround()
 */
modify undoVerb
    replace undoMove(actor) = {
        /* do TWO undo's - one for this 'undo', one for previous command */
        if (undo() && undo())
        {
            "(Undoing one command)\b";
            parserGetMe().location.senseAround(actor, true);
            scoreStatus(global.score, global.turnsofar);
        }
        else
            "No more undo information is available. ";
    }
;

#ifdef USE_CONTAINMENT

/*----------------------------------------------------------------------
 *	BOARDING AND UNBOARDING VERBS
/*--------------------------------------------------------------------*/

disembarkVerb: deepverb
    verb = 'disembark'
    sdesc = "disembark"
    doAction = 'Unboard'
    action(actor) = { askdo; }
    doDefault(actor, prep, io) =
    {
        if (actor.location and actor.location.location)
            return ([] + actor.location);
        else
            return [];
    }
;

replace getOffVerb: deepverb
    verb = 'getoff' 'getoffof'
    sdesc = "get off of"
    doAction = 'Getoff'
    action(actor) = { askdo; }
    doDefault(actor, prep, io) =
    {
        if (actor.location and actor.location.location)
            return ([] + actor.location);
        else
            return [];
    }
;
replace getOutVerb: deepverb
    verb = 'getout' 'getoutof'
    sdesc = "get out of"
    doAction = 'Getout'
    action(actor) = { askdo; }
    doDefault(actor, prep, io) =
    {
        if (actor.location and actor.location.location)
            return ([] + actor.location);
        else
            return [];
    }
;
replace boardVerb: deepverb
    verb = 'board'
    sdesc = "board"
    doAction = 'Board'
;
getinVerb: deepverb
    verb = 'getin' 'getinto'
    sdesc = "get in"
    doAction = 'Getin'
;
getonVerb: deepverb
    verb = 'geton' 'getonto'
    sdesc = "get on"
    doAction = 'Geton'
;
modify outVerb
    doAction = 'Exit'
    sdesc = "exit"
;

#endif /* USE_CONTAINMENT */


/*======================================================================
 *
 *  13.  M I S C E L L A N E O U S / U T I L I T Y
 *
/*====================================================================*/


/*
 *  runRoutine: function(o, prop)
 *
 *  This utility function is used to evaluate properties (o.prop)
 *  or filters (prop(o)) and is primarily used by path and scope 
 *  functions in access determination. 
 */
runRoutine: function(o, prop) {
    switch( datatype(prop) )
    {
        case DTY_TRUE: 
            return true;
        case DTY_FUNCPTR:
            return (prop(o));
        case DTY_PROP:
            return (o.(prop));
        default:
            return nil;
    }
}

modify global
    accessRules = [ &access_visible, &access_audible, &access_olfactory,
        &access_touchable, &access_reachable ]
;

/*
 *  MODIFIED TO USE parserGetMe().location.senseAround()
 *
 *   The die() function is called when the player dies.  It tells the
 *   player how well he has done (with his score), and asks if he'd
 *   like to start over (the alternative being quitting the game).
 */
replace die: function
{
    "\b*** You have died ***\b";
    scoreRank();
    "\bYou may restore a saved game, start over, quit, or undo
    the current command.\n";
    while (true)
    {
        local resp;

        "\nPlease enter RESTORE, RESTART, QUIT, or UNDO: >";
        resp = upper(input());
        if (resp == 'RESTORE')
        {
            resp = askfile('File to restore',
                            ASKFILE_PROMPT_OPEN, FILE_TYPE_SAVE);
            if (resp == nil)
                "Restore failed. ";
            else if (restore(resp))
                "Restore failed. ";
            else
            {
                parserGetMe().location.senseAround(parserGetMe(), true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
        }
        else if (resp == 'RESTART')
        {
            scoreStatus(0, 0);
            restart();
        }
        else if (resp == 'QUIT')
        {
            terminate();
            quit();
            abort;
        }
        else if (resp == 'UNDO')
        {
            if (undo())
            {
                "(Undoing one command)\b";
                parserGetMe().location.senseAround(parserGetMe(), true);
                scoreStatus(global.score, global.turnsofar);
                abort;
            }
            else
                "Sorry, no undo information is available. ";
        }
    }
}
;

/*
 *  MODIFIED TO USE parserGetMe().location.senseAround()
 *
 *   The init() function is run at the very beginning of the game.
 *   It should display the introductory text for the game, start
 *   any needed daemons and fuses, and move the player's actor ("Me")
 *   to the initial room, which defaults here to "startroom".
 */
replace init: function
{
#ifdef USE_HTML_STATUS
    /* 
     *   We're using the adv.t HTML-style status line - make sure the
     *   run-time version is recent enough to support this code.  (The
     *   status line code uses systemInfo to detect whether the run-time
     *   is HTML-enabled or not, which doesn't work properly before
     *   version 2.2.4.)  
     */
    if (systemInfo(__SYSINFO_SYSINFO) != true
        || systemInfo(__SYSINFO_VERSION) < '2.2.4')
    {
        "\b\b\(WARNING! This game requires the TADS run-time version
        2.2.4 or higher.  You appear to be using an older version of the
        TADS run-time.  You can still attempt to run this game, but the
        game's screen display may not work properly.  If you experience
        any problems, you should try upgrading to the latest version of
        the TADS run-time.\)\b\b";
    }
#endif

    /* perform common initializations */
    commonInit();
    
    // put introductory text here
    
    version.sdesc;                // display the game's name and version number

    setdaemon(turncount, nil);                 // start the turn counter daemon
    setdaemon(sleepDaemon, nil);                      // start the sleep daemon
    setdaemon(eatDaemon, nil);                       // start the hunger daemon
    
    parserGetMe().location = startroom;     // move player to initial location
    command.actorPtr = parserGetMe();       // set command actor pointer
    parserGetMe().location.senseAround(parserGetMe(), true);                      // show player where he is
    parserGetMe().location.isseen = true;                  // note that we've seen the room
    
    scoreStatus(0, 0);                          // initialize the score display
}
;

/*
 *  MODIFIED TO CALL parserGetMe().location.senseAround(), INSTEAD
 *  OF parserGetMe().location.lookAround(true).
 *
 *  mainRestore: function
 *  
 *  Restore a saved game.  Returns true on success, nil on failure.  If
 *  the game is restored successfully, we'll update the status line and
 *  show the full description of the current location.  The restore command
 *  uses this function; this can also be used by initRestore (which must be
 *  separately defined by the game, since it's not defined in this file).
 */
replace mainRestore: function(fname) {
    /* try restoring the game */
    switch(restore(fname))
    {
    case RESTORE_SUCCESS:
        /* update the status line */
        scoreStatus(global.score, global.turnsofar);

        /* tell the user we succeeded, and show the location */
        "Restored.\b";
        parserGetMe().location.senseAround(parserGetMe(), true);
        
        /* success */
        return true;

    case RESTORE_FILE_NOT_FOUND:
        "The saved position file could not be opened. ";
        return nil;

    case RESTORE_NOT_SAVE_FILE:
        "This file does not contain a saved game position. ";
        return nil;

    case RESTORE_BAD_FMT_VSN:
        "This file was saved by another version of TADS that is
        not compatible with the current version. ";
        return nil;

    case RESTORE_BAD_GAME_VSN:
        "This file was saved by a different game, or a different
        version of this game, and cannot be restored with this game. ";
        return nil;

    case RESTORE_READ_ERROR:
        "An error occurred reading the saved position file; the file
        might be corrupted.  The game might have been partially restored,
        which could make it impossible to continue playing; if the game
        does not seem to be working properly, you should quit the game
        and start over. ";
        return nil;

    case RESTORE_NO_PARAM_FILE:
        /* 
         *   ignore the error, since the caller was only asking, and
         *   return nil 
         */
        return nil;

    default:
        "Restore failed. ";
        return nil;
    }
}
;

#pragma C-
