#charset "us-ascii"

 /* Version 1.01 of William Tell has been updated for use with TADS 3.0.9 */

/*
 *   This is one attempt to translate the "William Tell" game from the
 *   Inform Beginner's Guide into a TADS 3 equivalent for the benefit
 *   of authors familiar with Inform who may be interested in learning,
 *   or learning about, TADS 3.
 *
 *   TADS 3 is much less fussy than Inform about the order in which game
 *   elements appear, so I have tried to follow the order of the Inform
 *   source in the IBG as far as possible to aid comparison.
 *
 *   Although this TADS 3 version is effectively the same game as the Inform
 *   original, there are a few cosmetic differences. In one or two places
 *   (especially in relation to the handling of NPCs) Inform and TADS 3 are
 *   so different that to have produced an exact equivalent of the Inform
 *   original would have given a misleading impression of the way TADS 3 is
 *   designed to work; in such cases I have allowed small (mainly cosmetic)
 *   variations to the Inform original in order to present this version in
 *   a way more in keeping with the design of TADS 3.
 *
 *   NOTE: Unlike Inform TADS 3 CODE IS CASE-SENSITIVE
 *   (e.g. name, Name and NAME are three different identifiers).
 */


/* 
 *   Include the main header for the standard TADS 3 adventure library.
 *   Note that this does NOT include the entire source code for the
 *   library; this merely includes some definitions for our use here.  The
 *   main library must be "linked" into the finished program by including
 *   the file "adv3.tl" in the list of modules specified when compiling.
 *   In TADS Workbench, simply include adv3.tl in the "Source Files"
 *   section of the project.
 *   
 *   Also include the US English definitions, since this game is written
 *   in English.  
 */
#include <adv3.h>
#include <en_us.h>

 
 /*
  *  The versionInfo definition effectively takes the place of the 
  *  CONSTANT Story, CONSTANT Headline, Release and Serial statements
  *  at the start of the Inform Version, but does a bit more besides.
  */
versionInfo: GameID
    name = 'William Tell'
    byline = 'by Roger Firth and Sonja Kesserich (ported to TADS 3 by Eric Eve)'
    htmlByline = 'ported by <a href="eric.eve@hmc.ox.ac.uk">
                  ERIC EVE</a>'
    version = '1.1'
    authorEmail = 'YOUR NAME <eric.eve@hmc.ox.ac.uk>'
    desc = 'A simple tutorial game.'
    htmlDesc = 'A simple tutorial game.'

    showCredit()
    {
        /* show our credits */
        "Original William Tell Game from the <i>Inform Beginner's Guide</i>
         by Roger Firth and Sonja Kesserich; ported to TADS 3 by Eric
         Eve.\b
         TADS 3 version of <q>William Tell</q> distributed by kind permission 
         of Roger Firth and Sonja Kesserich.\b
         TADS 3 language and library by Michael J. Roberts. ";

        /* 
         *   The game credits are displayed first, but the library will
         *   display additional credits for library modules.  It's a good
         *   idea to show a blank line after the game credits to separate
         *   them visually from the (usually one-liner) library credits
         *   that follow.  
         */
        "\b";
    }
    showAbout()
    {
        "This is a TADS 3 version of the <q>William Tell</q> game from
        the <i>Inform Beginner's Guide</i>. ";
    }
;

/****************************************************************************
 * OBJECT CLASSES                                                           *
 
/* Room is already a standard TADS 3 class */

/* The TADS 3 Decoration class does what the Inform version Prop class does */

/* The Furniture class can be made by combining two existing TADS 3 classes */

class Furniture : Surface, Heavy
;

 /* 
  *  In the Inform game, the Arrows and several objects cannot be dropped or
  *  given away. Rather than repeat the code to prevent parting with several
  *  objects on all those objects, it will be easier to define it once on
  *  an Undroppable mix-in class.
  */

class Undroppable: object
   /* 
    *   As in Inform, we can have a class or object decide how it will react
    *   to a command, but the syntax is rather different. Instead of a
    *   before property, we use dObjFor or iObjFor routines: we'll start
    *   by interrupting that for Drop 
    *   dobjFor means 'direct object for', which is what we want here.
    */

   dobjFor(Drop)
   {
      /* There are two stages at which we can veto an action, verify() and check(). We
         verify() would be used to prevent an action that is plainly illogical, such
         as eating a table or opening a door that's already open, the point being that
         when the player issues an ambiguous command and the parser has to decide which
         object is meant, it will prefer objects for which the current command is logical.
         
         In this case, however, dropping an arrow is not obvioiusly illogical, so we
         rule it out at the check() stage instead.
      */
    
      check()
      {
        
        /*
         *   Here we could simply have displayed a string using         
         *
         *   "Your arrows are sharp, and you guard them carefully. ";
         *  
         *   And this would work just as well, although reportFailure is the preferred
         *    way of telling TADS 3 that an action has failed.  
         *  
         */
         reportFailure(cannotDropMsg);
         
         /* We now need explicitly to abort the command */
         exit;
      }
   }
   
   /* 
    *  We'll customize cannotDropMsg on each instance, but we should define a 
    *  default here just in case.
    *
    *  Note that the expressions in braces {} are examples of parameter substitution
    *  strings; these are used throughout TADS 3 to create messages that can be used
    *  with different actors and/or objects. 
    *
    *  When the string is actually displayed {You/he} will be replaced with the name
    *  of the actor (or 'You' if, as is probable, the actor is the player character).
    *
    *  Similarly the (es} at the end of 'do{es}' will be adjusted to agree with {You/he}
    *  so that the game will display, e.g. "You do not..." or "Bob does not..." as
    *  appropriate.
    *
    *  Finally (it dobj/him} will display the appropriate pronoun, in the objective case,
    *  for the direct object of the command (i.e. 'him', 'her', 'it' or 'them' as appropriate).
    *  
    */
    
    cannotDropMsg = '{You/he} do{es} not want to part with {it dobj/him}. '
   
   /* Unlike Inform, we cannot simply list several actions in the same part of before
    * routine, but we can achieve the same effect by telling our TADS 3 version to treat
    * the Give and ThrowAt actions the same way as the Drop action on this class */
    
    dobjFor(GiveTo) asDobjFor(Drop)
    dobjFor(ThrowAt) asDobjFor(Drop)
   
   /* We should also block PutOn, which is the other way an object could be
    * parted with.
    */
   
   dobjFor(PutOn) asDobjFor(Drop)  
   
;



/*    We do need to define an Arrow class 
 *    As in Inform, a class definition looks much like an object definition.
 *    Here we use the keyword "class", followed by the new class name, followed
 *    by a colon, followed by a list of classes from which our new class inherits.
 *  
 *    The first three properties we define on this class are vocabWords (which is
 *    roughly equivalent to Inform's name property), name (equivalent to Inform's
 *    short_name) and desc (equivalent to Inform's description property). 
 *    This could instead have been written:
 *   
 *    class Arrow
 *      vocabWords = 'sharp true arrow*arrows'
 *      name = 'arrow'
 *     desc = "Just like all your other arrows -- sharp and true. "
 *  ;
 *  
 *   But since these three properties are so common, we use the short form instead.
 *  
 *   Note how the vocabWords property works: writing '*arrows' is the TADS 3 equivalent
 *   of writing 'arrows//p' in Inform, to denote that this is the plural form of the
 *   name.
 *  
 *   Note that in a real TADS 3 game one would probably give the arrows the same
 *   adjectives that appear in its description, and so write the vocabWords
 *   property as 'sharp true arrow*arrows'. The order of words does matter here -
 *   TADS 3 would understand 'sharp' and 'true' as adjectives and 'arrow' as a noun.
 *   If you wanted alternative nouns you might write, for example,
 *   'sharp true arrow/bolt/quarrel*arrows bolts quarrels'
 */

class Arrow: Undroppable, Thing 'sharp true arrow*arrows' 'arrow'  
   "Just like all your other arrows -- sharp and true. "
   
   /* We have to explicitly tell TADS 3 to treat all Arrows as equivalent */
   isEquivalent = true
   
   /* 
    * The following code makes arrows a logical (i.e. possible) thing to attack something with;
    * this will allow ATTACK X WITH ARROW or SHOOT X WITH ARROW
    * 
    */
    iobjFor(AttackWith)
     { 
         verify() { } 
         /* You first have to hold an arrow before you can shoot anything with it */
         preCond = [objHeld]
     } 
 
    
    /* Having made Arrows Undroppable, all we need to do is to define their
     * cannotDropMsg
     */
    cannotDropMsg = 'Your arrows are sharp, and you guard them carefully. '
   
   /* As with Inform, we terminate our class or object definition with a semicolon */
;
 
/* 
 * TADS 3 already defines a Person class for human-like actors, so we don't need
 * to make an NPC class from scratch. However, since the William Tell uses only
 * TALK TO for conversation, we need to neutralise TADS 3's much more sophisticated
 * conversation system, and reduce it to something equivalent to that in the 
 * Inform example.
 *
 * TADS 3 allows us to modify an existing class, so we start by modifying
 * the Person class defined in the library.
 */
 
modify Person
  
  /* If no other response is provided, the standard library Person class
   * ends up calling noResponseFromMsg(self), so we can just redefine this
   * message to the one we want in response to ASK and TELL 
   */
  noResponseFromMsg(obj) { return 'Just use T[ALK TO] ' + obj.theName; }
  
  /* The TalkTo action already exists in the TADS 3 library, but here we
   * want to modify what it does. On the class we simply make it invoke
   * a talkResponse property, which we define for the purpose, and provide
   * a default no response message. We'll customize this on individual
   * NPCs as required.
   */
   
  dobjFor(TalkTo)
  {
    action() { talkResponse(); }
  }
  talkResponse = "{The dobj/he} do{es} not respond." 
; 
 
/* 
 * TADS 3 does not provide a direct equivalent to the Inform found_in
 * property, so we add a framework for supporting it to the extent
 * needed in this game.
 *
 * Note that the needs for this doesn't turn up in TADS 3 games, since
 * there are usually other ways of achieving other effects; but the
 * particular case of the soldiers in the William Tell perhaps provides
 * an example where some such mechanism actually is necessary to produce
 * \ reasonable equivalent to the Inform version.
 *
 * First we define a Floating mix-in class for objects that are
 * going to use our new foundIn property
 */

class Floating : object
  /* foundIn will contain a list of the locations into which an object of this class
   * will be moved if the Player Character travels there.
   */
  foundIn = []
  
  /* 
   * Note, in principle the isFoundIn(loc) method could be overridden to
   * provide some other method of determining in which locations this
   * Floating items appears. E.g., it could be defined as { return loc.ofKind(GardenRoom); }
   * to make the Floating item appear in every room of the (putatively defined) custom
   * GardenRoom class. But we shall have no occasion to do this in this game.
   */
  
  isFoundIn(loc) { return foundIn.indexOf(loc) != nil; }
;

  /*
   * Next we define a TADS 3 equivalent of the Inform's MoveFloatingObjects.
   * This simply loops over all objects of the Floating class in the game,
   * and moves them into loc (the argument supplied to the function) if
   * loc is in their foundIn list.
   */

function moveFloatingObjects(loc)
{
    for( local obj = firstObj(Floating) ; obj != nil ; obj = nextObj(obj, Floating) )
        if(obj.isFoundIn(loc))
          obj.moveIntoForTravel(loc);
}

/*
 *  Finally we modify BasicLocation so that moveFloatingObjects is called
 *  whenever the player character is in the act of entering a new location;
 *  we call this at a point before the new room description is displayed,
 *  so that the floating item is in place when the room is described. 
 */


modify BasicLocation
   travelerArriving(traveler, origin, connector, backConnector)
   {
      if(traveler == gPlayerChar)
         moveFloatingObjects(self);
      inherited  (traveler, origin, connector, backConnector);
   }
;



 
/***********************************************************************************
 * The Game Objects
 */

/*
 *   Our definition defines two strings.  The first string, which must be
 *   in single quotes, is the "name" of the room; the name is displayed on
 *   the status line and each time the player enters the room.  The second
 *   string, which must be in double quotes, is the "description" of the
 *   room, which is a full description of the room.  This is displayed
 *   when the player types "look around," when the player first enters the
 *   room, and any time the player enters the room when playing in VERBOSE
 *   mode.  
 *
 *   Since the street is plainly outdoors, we use the OutdoorRoom class to
 *   define it. A Room would have four walls, a floor and a ceiling by
 *   default. An OutdoorRoom simply has ground and sky, which is what
 *   we want here. Note that both Room and OutdoorRoom provide light by
 *   default.
 */

street: OutdoorRoom 'A Street in Altdorf'
    /* Here we copy the Inform code by making desc a method
     * (the TADS 3 equivalent of an Inform 'embedded routine')
     *  although we could have employed a different technique here. 
     */
    desc()
    {
       /* Note that, as in Inform, a double-quoted string will
        * be displayed, but, unlike Inform, this will not result
        * in a newline or return.
        */
       
       "The narrow street runs north towards the town square.
        Local folk are pouring in through the town gate to the
        south, shouting greetings, offering produce for sale,
        exchanging news, enquiring with exaggerated disbelief about
        the prices of the goods displayed by merchants whose stalls
        make progress even more difficult. ";
        
        /* The test for !seen effectively does the same here as the
         * test for self hasnt visited in the Inform code. In TADS 3 ! 
         * (rather than ~) means 'not'; seen is a property of the street
         * object that is set to true once the player character has seen
         * the object. We could have written if(!self.seen), but in TADS 3
         * we don't need to refer to self explicitly in this context, since
         * TADS 3 assumes that we are referring to a property of
         * the self object unless we specify otherwise.
         */
        
        if(!seen) 
        
         /* In the string that follows we use \b (rather than ^) to generate
          * a blank line at the start of the text, and smart quotes,
          * <q> and </q> instead of Inform's ~ character to generate the
          * quotation marks. 
          */
          
          "\b<q>Stay close to me, son,</q> you say,
          <q>or you'll get lost among all these people.</q>";
    }
    
    /* We define the connection to below_square in much the same way as
     * in Inform, except that the TADS 3 property is called north rather
     * than n_to
     */
    
    north = below_square
    
    /* 
     * The way we prevent movement south is a little different. A TADS 3
     * direction property always has to contain an object; it can't contain
     * a routine or a string. So here we make it point to a special kind of
     * object, a FakeConnector, that simply displays the string we define
     * when the player character tries to move that way.
     *
     * We could have achieved almost the same effect with a NoTravelMessage
     * instead of a FakeConnector. The only difference is that with a NoTravelMessage
     * south will not show up as a possible exit in the exit lister, while with
     * a FakeConnector it will. We use a FakeConnector here since the room description
     * mentions a gate to the south, so one might expect to be able to go that way;
     * the FakeConnector thus provides a 'soft boundary' to the map.
     *
     * At first sight this syntax is a little odd. What we are in fact doing is
     * defining a new object of class FakeConnector and then associating it with
     * street's south property. In TADS 3 we call this an "anonymous nested object",
     * that is an object with no internal name defined within the definition of
     * another object. Although this may seem a little strange at first, it is
     * a very common technique in TADS 3 coding to handle this kind of case.
     */
    
    south: FakeConnector { "The crowd, pressing north towards the square, 
             makes that impossible. " }
        
;

/* 
 * Since the TADS 3 Decoration class already does what the custom Prop class
 * in the Inform class does, we use it here. 
 * As in the Inform example, since we don't refer to this object from anywhere
 * else in our code, we don't need to give it an internal name, so we start
 * the object definition with the class name. In TADS 3, this is called an
 * 'anonymous object'. The FakeConnector we defined above was another anonymous
 * object, but a nested one, defined inside another object definition.
 *
 * The @street (note the use of the @ sign) is the equivalent in TADS 3
 * of placing street after "south gate" in the Inform code to set the
 * initial location of this object. An alternative method would have been
 * to place a + sign at the start of the object decoration (i.e. before
 * Decoration), which would have been the exact equivalent of using
 * Inform's -> notation, i.e.:
 *
 * + Decoration 'south southern large wooden gate' ...
 * ;
 *
 * Yet another possibility would have been to set the Decorations's
 * location property explicitly:
 *
 * Decoration 'south southern large wooden gate' 'south gate'
 *   "The large wooden gate in the town walls is wide open. "
 *   location = street
 * ;
 *
 *  Note that in TADS 3 location is a property every object has, not a
 *  global variable giving the location of the player character (if you
 *  want the player character's location, you need to look at
 *  gPlayerChar.location).
 * 
 *  Here we use the nearest TADS 3 equivalent to the syntax preferred in
 *  the Inform example.
 */

Decoration 'south southern large town wooden gate' 'gate' @street
  "The large wooden gate in the town walls is wide open. "
;

/* 
 *  Since the next item is found in more than one location, we have to
 *  make it a MultiLoc as well as a Decoration. Strictly speaking, we
 *  should probably have used a MultiInstance, but that is a little more
 *  complicated to set up, and no practical harm will come from using
 *  MultiLoc here (theoretically, using MultiLoc means we are representing
 *  the stalls as a single object present in multiple locations, whereas
 *  a MultiInstance would represent several similar objects each present
 *  separately in its own location).
 *
 *  Note that the order of classes is important: MultiLoc has to
 *  come first.
 */
 
MultiLoc, Decoration 'assorted stalls' 'stalls'
  "Food, clothing, mountain gear; the usual stuff. "

  /* 
   *  locationList does roughly what found_in does in the
   *  Inform code, but note the rather different syntax
   *  for specifying a list; note also that a MultiLoc
   *  behaves rather differently - it is considered to
   *  be present in all the locations in its locationList,
   *  but it is not moved around with the Player Character.
   */
   
  locationList = [street, below_square] 
  
  /* 
   *    And now the equivalent to giving the stalls the pluralname attribute 
   *   (TADS 3 does not use attributes as such, but there are often equivalent
   *   properties.
   */
   
  isPlural = true
;

MultiLoc, Decoration 'mountain goods/gear/food/produce/clothing/stuff' 'produce'
  "Nothing special catches your eye. "
  locationList = [street, below_square]
 
  /* 
   *  We could set isPlural = true again, but strictly speaking 'produce' is
   *  a 'mass noun', like 'snow' or 'water', which we wouldn't normally
   *  pluralize, so we'll use the more accurate isMassNoun here.
   */
  
  isMassNoun = true
;

MultiLoc, Decoration 'merchant/trader*merchants traders' 'merchants'
  "A few crooks, but mostly decent traders touting their wares
   with raucous overstatement. "
   locationList = [street, below_square]
   isPlural = true
   
   /*   We don't bother to do anything equivalent to Inform's animate
    *   attribute, since these merchants are effectively just scenery.
    *   Any attempt to interact with them, apart from EXAMINE, will
    *   result in the message "The merchants aren't important. ", which
    *   is all we need here. 
    */
;

MultiLoc, Decoration 'local mountain people/folk/crowd' 'local people'
  "Mountain folk, just like yourself. "
  locationList = [street, below_square]
  isPlural = true     
;

//-----------------------------------------------------------------------

below_square: OutdoorRoom 'Further Along the Street'
  "People are still pushing and shoving their way from the southern
   gate towards the town square, just a little further north.
   You recognise the owner of a fruit and vegetable stall. "   
   north = south_square
   south = street
;


Furniture 'fruit vegetable veg stall/table' 'fruit and vegetable stall' @below_square
  "It's really only a small table, with a big heap of potatoes,
   some carrots and turnips, and a few apples. "
   
   /* We want SEARCH to behave like EXAMINE on this stall */
   dobjFor(Search) asDobjFor(Examine)   
;

 /* Note that the following definition will make the potatoes object
  * answer to BIG HEAP OF POTATOES as well as SPUD or HEAP or BIG HEAP */

Decoration 'big heap/potato/potatoes/spuds' 'potatoes' @below_square
  "Must be a particularly early variety... by some 300 years! "
  isPlural = true
;

Decoration 'carrot/carrots/turnip/turnips/apples/fruit/vegetables' 'fruit and vegetables'
  @below_square
  "Finely grown local produce. "
  isPlural = true
;

/* 
 *  The way TADS 3 treats NPCs is rather different from Inform; instead of 
 *  putting a lot of complicated code on one object, normally TADS 3 represents
 *  NPCs of any complexity with a series of objects. But for Helga, we'll do something
 *  a bit closer to the Inform code, even though this isn't what a TADS 3 
 *  NPC would normally look like. 
 */


stallholder : Person 'stallholder/greengrocer/monger/shopkeeper/
      merchant/owner/helga/dress/headscarf'  'Helga' @below_square
   "Helga is a plump, cheerful woman,
   concealed beneath a shapeless dress and a spotted headscarf. "
      
   isHer = true // equivalent to giving the female attribute in Inform
   isProperName = true // equivalent to giving the proper attribute
   
   /* 
    *  TADS 3 will normally display a message describing the presence
    *  of an actor (by default, this would be 'Helga is standing here. '),
    *  but the Inform version doesn't do this - presumably because Helga's
    *  presence is implied in the room description. Although we probably
    *  wouldn't do this in a typical TADS 3 game, here we'll suppress the
    *  display of a separate message announcing Helga's presence by
    *  overriding her actorHereDesc property to do nothing.  
    */
    
    actorHereDesc = nil
    
   /* 
    *  Probably the best way to duplicate the effect of the initial routine
    *  in the Inform code is to use Helga's afterTravel method, which
    *  is triggered whenever an actor travels to her vicinity
    */
    
    afterTravel(traveler, connector)
     {
       "Helga pauses from sorting potatoes to give
        you a cheery wave. ";
        
        /* 
         *  Probably the simplest check to apply here is whether 
         *  or not the apple has previously been moved. Once the
         *  apple is moved programmatically, its moved propertuy
         *  will be set to true. We also have to ensure that the
         *  traveler is the player character, otherwise this code
         *  will be triggered by Walter entering the location
         *  and the player won't see the message spoken by
         *  Helga.
         */
        if(!apple.moved && traveler == gPlayerChar)
        {
              
           /* 
            *   TADS 3 has no move X to Y syntax, instead we must call the
            *   moveInto method on the item to be moved, with the new container
            *   as its argument.
            */ 
                        
           apple.moveInto(gPlayerChar);
           
           "\n<q>Hello, Wilhelm, it's a fine day for trade! Is this
           young Walter? My, how he's grown. Here's an apple for him
           -- tell him to mind that scabby part, but the rest's good
           enough. How's Frau Tell? Give her my best wishes.</q><.p>";          
         
           /* 
            *  The <.p> tag causes a blank line, but ensures that only
            *  one blank line will be produced even if the next string
            *  displayed starts with a <.p> tag.
            */
        }
     }
   
     
   /* 
    *  Now we modify Helga's talkResponse to give the required sequence
    *  of responses. We can use a StopEventList to do most of the work
    *  for us, without having to define our own property variable to keep
    *  track of the number of times Helga has been addressed. 
    *  First, we simply get talkResponse to call the doScript method of
    *  a responses property we'll define in a minute. This will cause
    *  each item in the responses list to be invoked in turn, until
    *  we reach the last, which will be repeated indefinitely.
    */
    
    talkResponse() { responses.doScript;  }
    
    /* 
     * Now we define responses as an anonymous nested StopEventList. The list of events
     * is given within the nested object definition. If a list item is a single-quoted
     * string the StopEventList will display it; if it's a function it will call it.
     * We define the first item as a function:
     */
     
    responses: StopEventList
    {
      [ 
        new function {
           libScore.totalScore++ ; // equivalent to score = score + 1 in the Inform example
           "You warmly thank Helga for the apple. ";                 
        },
        
        '<q>See you again soon.</q>',
        
        'You\'ve nothing more to say to Helga right now. '
      ]
    }
      
;


//-------------------------------------------------------------------------------------------

south_square: OutdoorRoom 'South side of the square'
   "The narrow street to the south has opened onto the town square,
   and resumes at the far side of this cobbled meeting place.
   To continue along the street towards your destination --
   Johansson's tannery -- you must walk north across the square,
   in the middle of which you see Gessler's hat set on that
   loathsome pole. If you go on, there's no way you can avoid
   passing it. "
   
   /*
    *  Note that the Inform version included a description of
    *  the imperial soldiers in the room description of this location,
    *  but the more normal approach in TADS 3 would be to make the
    *  soldiers describe themselves, so that's what we'll be doing here.     
    */
   
   south = below_square
   north = mid_square
   
   /* 
    *   Finally, we define the ActorState the soldiers are to change
    *   to when they enter this room; amongst other things this will
    *   allow the way the soldiers are described to change according
    *   to location.
    */
   soldierState = soldiersSouth
;

/* 
 *   Rather than set up a separate object to represent the hat on the pole
 *   (which we could do using the standard TADS 3 Distant class) we'll
 *   connect the south of the square to the middle and north of the square with
 *   a DistanceConnector; the hat-pole placed in mid_square will then
 *   respond appopriately to attempts to interact with it from the
 *   distance of south_square or north_square.
 */

DistanceConnector [south_square, mid_square, north_square]
;

/* 
 *   We'll make the soldiers a Person rather than a Decoration, so they can
 *   respond to TALK TO etc. But this makes it difficult to implement the 
 *   object representing them as a MultiLoc (or MultiInstance).
 *   Instead we need to make the soldiers follow the player character
 *   into any of the locations where they may be found (this is achieved
 *   in the definition of the Floating class above). We also use
 *   different ActorStates in each location to make the soldiers
 *   describe themselves appropriately in each place.
 */

soldiers: Floating, Person  'gessler\'s vogt\'s imperial soldiers/guards/guard/soldier' 
    'Gessler\'s soldiers'  @south_square // initial location for soldiers.
    "They're uncouth, violent men, not from around here."  
    
     isPlural = true  
     
     /* 
      *   Since these soldiers are called "Gessler's soldiers" we don't need to
      *   us an article with them. We could set isProperName = true, but "Gessler's
      *   soldiers" is not exactly a proper name, so we set isQualifiedName = true instead.
      */
     isQualifiedName = true
     
     /*
      *   Note that we can often customise a reponse to an action performed
      *   on an object in TADS 3 simply by defining the appropriate message
      *   property on an object; we use this technique here to provide a
      *   customized response to ATTACK SOLDIERS.
      */
     uselessToAttackMsg = 'You\'re outnumbered many times. '
     
     talkResponse = "Such scum are beneath your contempt. "          
     
     /*
      *  Note that foundIn is not a standard property defined in the TADS 3
      *  library, but a property of the custom Floating class we defined above
      *  in order to emulate the behaviour of Inform's found_in.
      */
     
     foundIn = [south_square, mid_square, north_square, marketplace]
     
     
     /* 
      *  The following code makes the soldiers change to the appropriate 
      *  ActorState according to which room they're moved into.
      */
      
     moveIntoForTravel(dest)
     {
       inherited(dest);
       if(dest.soldierState != nil)
         setCurState(dest.soldierState);
     } 
;

/*   
 *   We now add a set of ActorStates to the soldiers. The soldiers NPC will be in
 *   one of these at any one time. Note the use of the + notation to locate these
 *   ActorStates within the actor object to which they belong.
 */

+ soldiersSouth : ActorState
  /* 
   *  The specialDesc property contains the string that will be displayed to describe the
   *   soldiers in a room description when they're in this state 
   */ 
  specialDesc = "Imperial soldiers jostle rudely through the throng,
   pushing, kicking and swearing loudly. "
  isInitState = true // The soldiers start out in this state, so we need to flag this  
;

+ soldiersMid: ActorState
  specialDesc = " A group of soldiers stands nearby, watching everyone who passes. "
  
  /* 
   *   This is perhaps a needlessly complicated means of achieving the desired
   *   effect in this case, but it shows how a given
   *   actor in a given ActorState can be made to respond to a particular action.
   */
  afterAction()
  {
    if(gActionIs(Salute) && gActor == gPlayerChar && gDobj == pole)
      "<.p><q>Why, thank you, sir,</q> sneers the soldier. ";
  }
  
  /* 
   *   When the soldiers are in this state, they can trap attempts to move in
   *  the beforeTravel method.
   */
   
  warnings_count = 0
  
  beforeTravel(traveler, connector)
  {
    if(traveler == gPlayerChar) // it's only the PC's movements we want to check
    {
        if(connector == south_square)
        {
            warnings_count = 0;
            pole.has_been_saluted = nil;
        }
        if(connector == north_square)
        {
            if(pole.has_been_saluted)
               "<q>Be sure to have a nice day!</q> the soldier tells you. ";
            else
            {
                switch(++warnings_count)
                {
                    case 1: "A soldier bars your way.\b
                     <q>Oi, you, lofty; forgot yer manners, didn't you?
                     How's about a nice salute for the vogt's hat?</q>";
                     exit;
                    case 2: "<q>I know you, Tell, yer a troublemaker,
                     aint you? Well, we don't want no bovver here,
                     so just be a good boy and salute the friggin'
                     hat. Do it now: I aint going to ask you again... </q>";
                     exit;
                    default: "<q>OK, <i>Herr</i> Tell, now you're in real trouble. I asked you
                      nice, but you was too proud and too stupid. I think it's time the
                      vogt had a little word with you.</q>\b
                      And with that the soldiers seize you and Walter
                      and, while the sergeant hurries off to fetch
                      Gessler, the rest drag you roughly towards the 
                      old lime tree growing in the marketplace. ";
                      
                      /* Change the son to the sonTied state and move him
                       * into the marketplace. Once again, to move an actor
                       * we must must moveIntoForTravel and not moveInto.
                       */
                      son.setCurState(sonTied);
                      son.moveIntoForTravel(marketplace);
                      
                      /* The Inform version moves the apple to the son at this
                       * point, but in TADS 3 this would result in the son being
                       * described as holding the apple, which is hardly appropriate.
                       * We'll instead give the apple a placeOn method which will
                       * carry out the appropriate transformation.
                       */
                      apple.placeOn(son);
                      
                      /*
                       * Finally, move the player character into the marketplace.
                       * The following code more or less carries out everything
                       * the Inform PlayerTo function would, for example, it
                       * ensures that a room description is shown.
                       *
                       * replaceAction is one way of synthesing an action in
                       * TADS 3: it's a little like writing <<TravelVia marketplace>>
                       * in Inform, except that Inform has no TravelVia action.
                       */
                      replaceAction(TravelVia, marketplace);                      
                }
            }
         } 
    }
  }  
;

+ soldiersNorth : ActorState
  specialDesc = "A few soldiers stand around, eyeing you suspiciously. "
;

+ soldiersMarket: ActorState
  specialDesc = "The other soldiers stand around, watching eagerly. "
;

//---------------------------------------------------------------------------------------

mid_square: OutdoorRoom 'Mid Square'
   "There is less of a crush in the middle of the square; most
    people prefer to keep as far away as possible from the pole
    which towers here, topped with that absurd ceremonial hat. "
    south = south_square
    north = north_square
    soldierState = soldiersMid
    
    /*
     *   The code for checking that the player character has performed his
     *   salute more logically belongs on the soldiers, so we've placed it
     *   above on the soldierMid ActorState
     */
;

 /* 
  *   We shan't be putting anything on the pole, so the TADS 3 library Fixture class
  *   is the appropriate one to use here, rather than the custom Furniture class.
  */

pole: Fixture 'wooden pine black hat/pole/brim/feathers' 'hat on a pole' @mid_square
  "The pole, the trunk of a small pine some few inches in diameter,
   stands about nine or ten feet high. Set carefully on top is
   Gessler's ludicrous black and red leather hat, with a widely
   curving brim and a cluster of dyed goose feathers. "
   
   /* Note that TADS 3 uses nil rather than false to mean "not true" */
   has_been_saluted = nil
   
   uselessToAttackMsg = 'Tempting, but you\'re not looking for trouble. '
   
   dobjFor(Salute)
   {
    
     /* 
      *   We make the pole-hat the most likely object for the player to salute;
      *   This means the parser will choose it as the default object if the player
      *   types simply SALUTE.
      *
      *   We do this by setting it's logical rank to something more than the
      *   default 100. The second parameter of the logicalRank macro ('hat') is
      *   an arbitrary string the purpose of which is too rarefied to concern us here.
      */
     
     verify() { logicalRank(120, 'hat'); }
     
     action()
     {
        /* 
         *   note we don't need to stipulate self.has_been_saluted here (although
         *   it would not be incorrect to do so), since TADS 3 will automatically
         *   assume we mean to refer to the self object in this kind of context 
         */
         
        has_been_saluted = true; 
        
        "{You/he} salute{s} the hat on the pole. ";
        
        /* 
         *  We'll leave the soldier to respond to the salute, although we
         *  could just put the text in here -- see the definition of the
         *  soldierMid ActorState above.  
         */
     }
   }
;



//----------------------------------------------------------------------------------------

north_square : OutdoorRoom 'North side of the square'
  "A narrow street leads north from the cobbled square. In its
   centre, a little way south, you catch a last glimpse of the pole
   and hat. "
   
   /* Once again we use a FakeConnector to display the required message */
   south : FakeConnector { "You hardly feel like going through all
     that again. " }
      
   /* 
    *  We use the same technique for the northern exit, but we also employ a
    *  little cunning. TADS 3 allows a function call to be embedded in a double-quoted
    *  string using the << >> syntax. Here we embed a call to finishGameMsg, which
    *  ends the game and displays an appropriate message (the equivalent of setting
    *  deadflag in Inform).
    *
    *  Note that we could simply have called this method with a single-quoted string
    *  as its first argument:
    *
    *    finishGameMsg('YOU HAVE SCREWED UP A FAVOURITE FOLK STORY', []);
    *
    *   But (a) to make this resemble the Inform code a little more closely and (b)
    *   to allow us to use the same message elsewhere, we use a custom
    *   finish type object (ftScrewUp) instead (we'll define ftScrewUp below).  
    *   To allow the player to UNDO or see the SCORE after this message is displayed,
    *   we list the appropriate finish options in the second argument to the 
    *   finishGameMsg function.
    */
   
   north : FakeConnector { "With Walter at your side, you leave the square by the
         north street, heading for Johansson's tannery. 
         <<finishGameMsg(ftScrewUp, [finishOptionUndo, finishOptionScore])>>"     
   }
   soldierState = soldiersNorth
;



//-----------------------------------------------------------------------------------------

marketplace : OutdoorRoom 'Marketplace near the square'
  "Altdorf's marketplace, close by the town square, has been hastily
   cleared of stalls. A troop of soldiers has pushed back the crowd
   to leave a clear space in front of the lime tree, which has been
   growing here for as long as anybody can remember. Usually it
   provides shade for the old men of the town, who gather below to
   gossip, watch the girls, and play cards. Today, though, it
   stands alone... apart, that is, from Walter, who has been lashed
   to the trunk. About forty yards away, you are restrained by two
   of the vogt's men. "
   
   /* 
    *  This is the equivalent of the cant_go in the Inform version 
    *  In a more typical situation, we'd probably override cannotGoThatWayMsg
    *  (with a single-quoted string, i.e.
    *     cannotGoThatWayMsg = 'What? And leave your son tied up here? ')
    *  and the standard cannotGothatWay method would display this message
    *  followed by a list of possible exits (a bit like the Inform extension
    *  smartCantGo), but since there are no possible exits from this
    *  location, that would be rather pointless here, so we'll stick
    *  more closely to what the Inform version of William Tell does.  
    */
   cannotGoThatWay = "What? And leave your son tied up here? " 
   
   soldierState = soldiersMarket
;

/*
 *    Since the tree is forty yards away from the player character,
 *    we'll use the TADS 3 library Distant class for it. This allows
 *    the player to examine it, but any other action (except those we
 *    explicitly allow for) will be elicit the response "The lime
 *    tree is too far away. "
 */
 
tree: Distant 'large lime tree' 'lime tree'
  "It's just a large tree. "
  
  /*
   *   We'll eventually define FIRE AT to be equivalent to ATTACK WITH,
   *   so here we need only trap the AttackWith action.
   */
   
  dobjFor(AttackWith)
  {
    verify() { } 
    action()
    {
       /* 
        *  We don't need to check that the Player Character has
        *  specified the bow or an arrow as the weapon to attack
        *  with, since we can leave indirect object of the command
        *  (e.g. the bow or arrow) to check that it's an appropriate
        *  object to attack something with.
        */
        
       "Your hand shakes a little, and your arrow flies
       high, hitting the trunk a few inches above Walter's
       head. ";
       
       finishGameMsg(ftScrewUp, [finishOptionUndo, finishOptionScore]);
    }
  }
;

governor: Person 'hermann gessler/vogt/governor' 'governor' @marketplace
  "Short, stout but with a thin, mean face, Gessler relishes the
   power he holds over the local community."
   
  isHim = true // equivalent to male attribute
  
  /* 
   *  Note that we use actorHereDesc to generate a description from
   *  within an Actor object itself, when we're not using an ActorState
   *  to do the job.
   */
   
  actorHereDesc = "Gessler is watching from a safe distance,
     a sneer on his face. "  
     
  talkResponse = "You cannot bring yourself to speak to him. " 
  
  /* 
   *  Once again we use the afterTravel method to trigger the governor's
   *  remarks when the player character enters the marketplace.
   *  Although not strictly necessary here, since the PC will only
   *  ever come this way once, we test for gRevealed('archery') to
   *  ensure that the message hasn't been delivered before. The
   *  <.reveal archery> tag at the end of the message displayed then
   *  sets this flag, so that once the message has been displayed,
   *  gRevealed('archery') will return true.
   *  Note that we could have used any tag value we wanted, e.g.
   *  <.reveal bx6> and gRevealed('bx6') -- we've simply chosen
   *  to employ something a bit more meaningful.
   */
   
  afterTravel(traveler, connector)
  {
     if(!gRevealed('archery'))
     "<q>It appears that you need to be taught a lesson,
     fool.</q> Gessler declares, <q>Nobody shall pass through the square without paying
     homage to His Imperial Highness Albert; nobody, hear me?
     I could have you beheaded for treason, but I'm going to
     be lenient. If you should be so foolish again, you can
     expect no mercy, but this time, I'll let you go free...
     just as soon as you demonstrate your archery skills by
     hitting this apple from where you stand. That shouldn't
     prove too difficult; here, sergeant, catch. Balance it on
     the little bastard's head.</q><.reveal archery>";
  } 
  
  dobjFor(AttackWith)
  {
     verify() { } // he's a possible target.
     action()
     {
        /* Once again we can leave it to the indirect object (the 'second' in
         * Inform-speak) to decide whether it's a valid weapon for use in
         * an attack. If we reach this action routine that will already
         * have been settled, so we don't need to test for it here.
         */
        
        "Before the startled soldiers can react, you turn
         and fire at Gessler; your arrow pierces his heart,
         and he dies messily. A gasp, and then a cheer,
         goes up from the crowd. ";
         
         finishGameMsg(ftScrewUp, [finishOptionUndo, finishOptionScore]);
     }
  }
;

//------------------------------------------------------------------------
// The Player Character and his possessions

/*
 *   Define the player character.  The name of this object is not
 *   important, but note that it has to match up with the name we use in
 *   the gameDef object to initialize the game, below.
 *   
 *   Note that we aren't required to define any vocabulary or description
 *   for this object, because the class Person, defined in the library,
 *   automatically provides the appropriate definitions for an Actor when
 *   the Actor is serving as the player character.  Note also that we
 *   don't have to do anything special in this object definition to make
 *   the Actor the player character; any Actor can serve as the player
 *   character, and we'll establish this one as the PC in gameMain, below.  
 */
 
me: Person
    /* We set the player character's description here */
    desc = "You wear the traditional clothing of a Swiss mountaineer. "

    /* the initial location -- note this is a property of me, got a global varaiable */
    location = street
;

/* 
 *   Now we can simply use the + notation to locate the player character's
 *   possessions in the player. We don't need to move them there in
 *   the game's startup code.
 */
 
+ bow: Wearable 'trusty yew bow' 'bow'
  "Your trusty yew bow, strung with flax. "
  
   /* 
    *  Instead of giving the bow the worn attribute, we set its wornBy
    *  property to the Actor who's wearing it.
    */
   wornBy = gPlayerChar
   
   /* 
    *   We make it logical to use the bow as a weapon, but we also
    *   apply couple of preconditions 
    */
   
   iobjFor(AttackWith)
   {
      /* To deploy the bow as a weapon, it must be held but not worn */
      preCond = [objHeld, objNotWorn]
      
      /* We'll make the bow the most likely weapon to use */
      verify() { logicalRank(120, 'weapon'); }
   }
;

/* 
 *  By making the quiver a RestrictedContainer we can allow nothing
 *  but arrows to be put in it.
 */

+ quiver: Undroppable, Wearable, RestrictedContainer 'goatskin quiver' 'quiver'
  "Made of goatskin, it usually hangs over your left shoulder. "
  wornBy = gPlayerChar
  
  /* 
   *  Test for an object being an arrow before allowing it to be put in
   *  the quiver -- this method is defined on the RestrictedContainer
   *  class in the library.
   */
  canPutIn(obj) { return obj.ofKind(Arrow); }
  cannotPutInMsg(obj) { return 'Only arrows go in there. '; }
  cannotDropMsg = 'But it was a present from Hedwig, your wife. '
;

/* Put three arrows in the quiver */

++ Arrow;
++ Arrow;
++ Arrow;



son: Person 'your quiet blond son/boy/lad/walter' 'your son' @street
   /* 
    *  We won't define the desc property here -- instead we'll let
    *  the ActorStates take care of providing Walter's description.
    */
    
  isQualifiedName = true
  isHim = true
  
  /* 
   *  Let the current actor state handle a TalkTo command 
   *  This is a fairly commom coding pattern with ActorStates --
   *  it's the way the TADS 3 library handles a lot of
   *  NPC programming. Here it provides a neat way of
   *  ensuring we get the appropriate response depending
   *  on Walter's current state.
   */
  
  talkResponse { curState.talkResponse; }  
  
  /* We'll handle attacking Walter in the same way -- by passing it to the state */
  
  dobjFor(AttackWith) { action() { curState.attackResponse; } }
;

/* 
 *  By defining Walter's initial ActorState as an AccompanyingState, we automatically
 *  arrange for Walter to follow the Player character around. Note that we also
 *  leave the description of Walter entirely to two ActorStates, so we'll always
 *  get the appropriate one.
 */

+ AccompanyingState
  /*
   *  The current ActorState's stateDesc property is added to the end of the Actor's
   *  desc(ription) property to provide the full description of the actor in response
   *  to an EXAMINE command. Since in this case we have defined no desc property on
   *  Walter himself, the stateDesc can be used to provide the appropriate full 
   *  description according to the actor's state. 
   */ 
    
  stateDesc = "A quiet, blond lad of eight summers, he's fast
   learning the ways of mountain folk. "
  specialDesc = "Your son, Walter, is at your side. "
  
  
   /*  We want Walter to start out in this state */
  isInitState = true
  talkResponse = "You point out a few interesting sights. "
  attackResponse = "He's your son! You don't want to hurt him! "
;

  /*  
   *  This is an example of how TADS 3 handles many of the commands Inform
   *  uses the life property for: in TADS 3 GiveTo, ShowTo, AskAbout, TellAbout and AskFor,
   *  as well as orders directed to the actor, can be handled by various kinds of TopicEntry
   *  object. A GiveTopic handles GiveTo commands. A GiveTo topic in the current actor state
   *  will handle the command GIVE APPLE TO WALTER. If Walter's response could simply be
   *  given by displaying a string, we could simply define the appropriate doube-quoted
   *  string; but since the response is a little more elaborate here we need to write
   *  some code in the GiveTopic's topicResponse() method:
   */

++ GiveTopic @apple
  topicResponse()
  {
     "You hand the apple to Walter.\b
     <q>Thank you, Papa,</q> he says. ";
     apple.moveInto(son);
     libScore.totalScore++;
  }
;


+ sonTied : ActorState
  stateDesc = "He stares at you, trying to appear brave and
   remain still. His arms are pulled back and tied behind
   the trunk, and the apple nestles amid his blond hair. "
   
  specialDesc = "Walter is lashed to the trunk of the tree. "
  
  talkResponse = "<q>Stay calm, my son, and trust in God.</q> "
  
  attackResponse()
  {
     "Oops! Surely you didn't mean to do that";
     finishGameMsg(ftScrewUp, [finishOptionUndo, finishOptionScore] );
  }
;

apple: Undroppable, Thing 'blotchy green brown apple' 'apple'
  desc()
  {
    /* 
     *  In the placeOn method below, we change the apple's class to
     *  Distant when it's placed on Walter's head. We can therefore test
     *  for the apple's belonging to the Distant class to decide which
     *  description to use. 
     */
    if(ofKind(Distant))
      "At this distance you can hardly see it. ";
    else
      "The apple is blotchy green and brown. ";
  }
  
  /* Again, we just need to define the appropriate message property */
  cannotEatMsg = 'Helga intended it for Walter...'
  
  placeOn(obj)
  {
     /* 
      *  When the apple is notionally placed on Walter's head, we don't want to
      *  move it into Walter, since Walter is not actually carrying it. It will
      *  work better if we move it into Walter's location.
      */
     moveInto(obj.location);
     
     /*
      *  One of the neat things we can do in TADS 3 is change an object's
      *  class dynamically at run-time. Once the apple is notionally on
      *  Walter's head some forty yards away, it's too distant from the
      *  player character for him to interact in any way apart from Examine
      *  and Shoot At. At this point we want the apple to behave like a
      *  Distant object, so we change its class to Distant.
      */
     setSuperclassList([Distant]);
  }
  
  dobjFor(AttackWith)
  {
     verify() { }
     check()
     {
        if(!ofKind(Distant))
        {
           reportFailure('Helga intended it for Walter to eat, not for you to
             vent your frustrations on! ');
           exit;     
        }
     }
     action()
     {
        libScore.totalScore++;
        "Slowly and steadily, you place an arrow in
        the bow, draw back the string, and take aim with
        more care than ever in your life. Holding your
        breath, unblinking, fearful, you release the
        arrow. It flies across the square towards your
        son, and drives the apple against the trunk of
        the tree. The crowd erupts with joy;
        Gessler looks distinctly disappointed.\b";
        
        /* The equivalent to setting deadflag = 2 is to call
         * call finishGameMsg with ftVictory as the first argument.
         * Other ftXXX objects defined in the library include
         * ftDeath, ftFailure and ftGameOver.
         */
         
        finishGameMsg(ftVictory, [finishOptionUndo, finishOptionScore]);
     }
  }
  
  cannotDropMsg = 'An apple is worth quite a bit -- better hang on to it. '
  
   
  /* 
   *   If Walter already has the apple, it's illogical to try to give it to him.
   *   Unfortunately the standard TADS 3 library will try to make the player character
   *   take the apple back from Walter and then report that the Walter won't allow it.
   *   This isn't the most sensible behaviour, so we override it to make the attempt
   *   to give someone something they already have illogical (which will prevent
   *   the attempt to take it back).
   */
   
  
  dobjFor(GiveTo)
  {
    verify()
    {
        /* 
         *  gIobj is the indirect object (second) of the current action,
         *  i.e. the person we're trying to give the apple to.
         * 
         *  We test for gIobj != nil to ensure that the other test we
         *  want is not carried out until the indiect object has been resolved;
         *  otherwise we might a run-time error (although in this particular
         *  case the test is safe in any case), then we test for the object
         *  being held by the person we're trying to give it to.         *
         */
        if(gIobj != nil && self.isHeldBy(gIobj))        
            /* illogicalAlready is what we use to mark an action as illogical
             * because the state of affairs it would bring about already exists.
             */
            illogicalAlready('But {the iobj/he} already {has} {the dobj/him}! ');  
            
        /* make sure we invoke the behaviour inherited from our superclass */          
        inherited;
    }
    
  /* 
   *   We've made the apple an Undroppable, but we still want to be
   *   able to give it away, so we need to override GiveTo's check to allow this.
   */   
    
    check() { }   
  }
  
;
/*----------------------------------------------------------------------
 *   TADS 3 has no Entrypoint routines as such, but the gameMain object
 *   does much of the work of the Inform initialise routine.
 *
 *
 *   The "gameMain" object lets us set the initial player character and
 *   control the game's startup procedure.  Every game must define this
 *   object.  For convenience, we inherit from the library's GameMainDef
 *   class, which defines suitable defaults for most of this object's
 *   required methods and properties.  
 */
gameMain: GameMainDef
    /* the initial player character is 'me' */
    initialPlayerChar = me
    
    /* We set the maximum score here, rather than in an independent constant.
     * Note that it's possible to use a scoring scheme in TADS 3 that allows
     * the maximum score to be calculated automatically -- but that's not
     * the scoring scheme used in this game.
     */
    maxScore = 3

    /* 
     *  Prior to TADS 3.0.9 the equivalent to setting lookmode = 2
     *  in the Inform version would be to insert the line
     *
     *     verboseMode = true
     *
     *  However, this will cause a run-time error in TADS 3.0.9, and
     *  so shouldn't be done. TADS 3.0.9 employs a new mechanism that
     *  reads this kind of setting from an option file which saves
     *  the player's default preferences, so this option should not
     *  normally be set by the game author.
     */
   

    
    /* 
     *   Show our introductory message.  This is displayed just before the
     *   game starts.  Most games will want to show a prologue here,
     *   setting up the situation for the player, and show the title of the
     *   game.  
     */
    showIntro()
    {
        "The place: Altdorf, in the Swiss canton of Uri. The year is 1307,
        at which time Switzerland is under rule by the Emperor Albert of
        Habsburg. His local governor -- the vogt -- is the bullying
        Hermann Gessler, who has placed his hat atop a wooden pole in
        the centre of the town square; everybody who passes through the
        square must bow to this hated symbol of imperial might.\b
        
        You have come from your cottage high in the mountains,
        accompanied by your younger son, to purchase provisions. You are
        a proud and independent man, a hunter and guide, renowned both
        for your skill as an archer and, perhaps unwisely (for his soldiers
        are everywhere), for failing to hide your dislike of the vogt.\b
        
        It's market-day: the town is packed with people from the
        surrounding villages and settlements.\b";
    }

    /* 
     *   Show the "goodbye" message.  This is displayed on our way out,
     *   after the user quits the game.  You don't have to display anything
     *   here, but many games display something here to acknowledge that
     *   the player is ending the session.  
     */
    showGoodbye()
    {
        "<.p>Thanks for playing!\b";
    }
;

/* 
 *  Instead of defining a DeathMessage routine, we need to define a custom 
 *  FinishType object for those finishGameMsg(ftScrewUp...) calls we've
 *  made above. 
 */

ftScrewUp: FinishType
  finishMsg = 'YOU HAVE SCREWED UP A FAVOURITE FOLK STORY'
;

/*------------------------------------------------------------------
 * Custom grammar and actions
 * 
 *   Note that in TADS 3 this works rather differently from Inform; instead
 *   of starting with a verb word and defining what actions it can perfom with
 *   what grammars, we start with an action, and define the various grammars
 *   that invoke it.
 */

 /* 
  *   First we need to define a new action. This is the rough equivalent of
  *   defining a [saluteSub; ] routine stub in Inform, but in TADS 3 we're actually
  *   defining a new Action class. For Salute we want a TAction, that is
  *   a verb that takes a direct object (but not indirect object).
  */

DefineTAction(Salute)
;

  /* 
   *  Now we define the grammar that can be used for this action
   *  Note that in this grammar definition the vertical bar symbol |
   *  is used to separate alternatives (a little like / in the Inform definitions),
   *  brackets are used to group phrasings, and singleDobj is roughly
   *  the equivalent of the noun token in Inform.
   */

VerbRule(Salute)
  ('salute' | 'acknowledge') singleDobj |
  ('bow' | 'nod' | 'kowtow' | 'genuflect') ('at' | 'to' | 'towards') singleDobj |
  ('pay' | 'give') 'homage' 'to' singleDobj |
  'wave' 'at' singleDobj
  
  /* 
   *  The next line is the rough equivalent of ->Salute in the Inform grammar
   *  definition, in that it tells the compiler that the grammar just defined
   *  is to be associated with the SaluteAction.
   */
  : SaluteAction
  
   /*
    *  The following definition is needed to enable the parser to synthesise messages
    *  such as "What do you want to salute?" or "(first saluting the Colonel)".
    */
  verbPhrase = 'salute/saluting (what)'
;

/* 
 *  Note that with an action that takes a direct object, we define
 *  the default behaviour on the classes we're interested in, not
 *  in a standalone routine like [saluteSub; ] (or in the SaluteAction class)
 *
 *  Note that such default behaviour should always at least be defined
 *  on Thing, since this is the class from which all physical objects
 *  in the TADS 3 world model descend.
 */

modify Thing
  dobjFor(Salute)
  {
    /* 
     *  We begin by specifying the precondition that an object must be
     *  visible before it can be saluted.
     */     
    preCond = [objVisible]
    
    /* 
     *  It's possible to salute an inanimate object, but it's not the
     *  most likely target of a salute command, so we give it a
     *  logical rank of 50 (the default logical rank is 100)
     */
    verify() { logicalRank(50, 'inanimate'); }
    
    /* In the action routine we simply display a default response */
    action() { "{The dobj/he} take{s} no notice. "; }
  }
;

/* 
 *   Provide different default handling for the Person class --
 *  roughly equivalent to objects having the animate attribute in
 *  Inform (we could define this on the Actor class, from which
 *  Person inherits, but all the Actors in this game are Persons).
 */

modify Person
  dobjFor(Salute)
  {
     /* Note that Person simply inherits Thing's preconditions 
      *
      * We override verify to an empty routine (equivalent to 
      * a logical rank of 100), since a Person is a more
      * likely target of a salute command than an inanimate Thing.
      */
     verify() { }
     
     /* Then we provide another default response */
     action() { "{The dobj/he} acknowledge{s} {you/him}. "; }
  }
;


/* 
 *  The TalkTo action already exists in TADS 3, so we simply need to
 *  modify its syntax
 */

modify VerbRule(TalkTo)
  ('talk' | 'chat' | 't' | 'converse' | 'gossip') ('to' | 'with' |) singleDobj
  : 
;

/* 
 *  Rather than define a new ShootAt action, we'll simple make use of the
 *  existing AttackWith action and define some additional grammar for it
 */

VerbRule(FireAt)
  'shoot' singleDobj 'with' singleIobj 
  | ('fire' | 'shoot') singleIobj 'at' singleDobj
  : AttackWithAction
  verbPhrase = 'shoot/shooting (what) (with what)'
;

/* 
 *  Finally, we can define some fancy grammar to deal with the
 *  case where the player types SHOOT X without saying what
 *  should be used to shoot with.
 */

VerbRule(ShootWithWhat)

  /* 
   *  The badness indicator at the start of this grammar definition is
   *  an indication to the parser to give a relatively low priority
   *  to matching this phrasing if the player's input can be construed
   *  as matching something better.
   */
   
  [badness 500] 'shoot' ('at' | ) singleDobj
  : AttackWithAction
   verbPhrase = 'shoot/shooting (what) (with what)' 
   
  /* 
   *  The following code instructs the parser to ask for the
   *  missing indirect object (of the AttackWithAction) -- or
   *  else to choose a default indirect object if there is a
   *  single object in scope that constitutes the best possibility.
   */ 
  construct()
  {
    /* set up the empty indirect object phrase */
    iobjMatch = new EmptyNounPhraseProd();
    iobjMatch.responseProd = withSingleNoun;
  }

;

/*
 *  And finally, a word about strings in TADS
 *  -----------------------------------------
 *
 *  If you've read through the source code and comments this far you'll probably
 *  have observed that the distinction between single-quoted and double-quoted
 *  strings in TADS is not the same as in Inform.
 *
 *  In TADS a double-quoted string is always an instruction to display that string
 *  (without any implied newline or return), whereas a TADS single-quoted string is 
 *  not necessarily a dictionary word, it is simply a data item of string type, a
 *  data item that can be examined, compared or concatenated with other single-quoted
 *  strings, or manipulated in a number of different ways.
 *
 *  A single-quoted string can also be displayed, using the say function (or
 *  the tadsSay function), e.g. 
 *
 *    say('This is a string. ');
 *
 *  Assigning a double-quoted string to an object property is the same as writing
 *  a method that displays it. The following three definitions are functionally
 *  equivalent:
 *
 *    desc = "This is a string. "
 *
 *    desc { "This is a string. "; }
 *
 *    desc { say('This is a string. '); }
 *
 *  Note that an object property expressed as a double-quoted string cannot be
 *  changed programmatically at run-time. For example if you had defined an object
 *  myObj with a desc property thus:
 *
 *    desc = "This is a string. "
 *
 *  You could not subsequently change it with a statement such as:
 *
 *    myObj.desc = "This is another string. "; // this won't compile.
 *
 *  To achieve the desired effect you'd need to use single-quoted strings, e.g.:
 *
 *    desc = "<<myDesc>>"
 *    myDesc = 'This is a string. '
 *
 *  Then subsequently you could call:
 *
 *    myObj.myDesc = 'This is another string. '; // this is perfectly legal
 *
 *  Finally, note the use of the powerful <<>> notation in TADS double-quoted strings.
 *  Between the << and the >> you can place any expression that evaluates to a number,
 *  a single or double-quoted string, or nil. The appropriate string is then displayed
 *  in situ, and any side-effects of evaluating the expression are carried out (so that
 *  you could use this simply to trigger the side-effects without displaying anything
 *  at all, a powerful technique that's often useful in TADS).
 *  In other words the statement:
 *
 *  "Some text <<expression>> more text. ";
 *
 *  Is functionally equivalent to the statement:
 *
 *  say('Some text ' + expression + ' more text. ');
 *
 *  Where + is the string concatenation operator.
 *
 *  The <<>> syntax is not available in single-quoted strings. 
 */
 

