(* The environments of the static semantics of Modules, Definition p 31 *)

(*
$File: Common/ModuleEnvironments.sml $
$Date: 1993/03/05 14:38:46 $
$Revision: 1.9 $
$Locker: birkedal $
*)

(*$ModuleEnvironments:
	STRID SIGID FUNID TYCON STATOBJECT_PROP ENVIRONMENTS_PROP
	MODULE_STATOBJECT PRETTYPRINT REPORT FLAGS LIST_HACKS FINMAP CRASH
	MODULE_ENVIRONMENTS
 *)

functor ModuleEnvironments(
	  structure StrId  : STRID
	  structure SigId  : SIGID
	  structure FunId  : FUNID
	  structure TyCon  : TYCON

	  structure StatObject : STATOBJECT_PROP

	  structure C : ENVIRONMENTS_PROP
	    sharing type C.TyNameSet  = StatObject.TyNameSet
		and type C.TyName     = StatObject.TyName
		and type C.strid      = StrId.strid = TyCon.strid
		and type C.tycon      = TyCon.tycon

	  structure ModuleStatObject : MODULE_STATOBJECT
	    sharing type C.TypeFcn
	      		 = StatObject.TypeFcn
			 = ModuleStatObject.TypeFcn
		and type ModuleStatObject.longstrid = StrId.longstrid
		and type ModuleStatObject.NameSet   = C.NameSet
		and type ModuleStatObject.StrName   = C.StrName
		and type ModuleStatObject.Env       = C.Env
		and type ModuleStatObject.Str       = C.Str
		and type ModuleStatObject.TyStr     = C.TyStr
		and type ModuleStatObject.ConEnv    = C.ConEnv

	  structure PP : PRETTYPRINT
	    sharing type ModuleStatObject.StringTree = PP.StringTree
	        and type C.StringTree = PP.StringTree

	  structure Report: REPORT
	  structure Flags : FLAGS
	  structure ListHacks: LIST_HACKS

	  structure FinMap : FINMAP
	    sharing type FinMap.StringTree = PP.StringTree
	        and type FinMap.Report = Report.Report

	  structure Crash : CRASH

	  structure Tail_T:
	    sig
	      datatype Tail = TYCON of TyCon.tycon
	                    | STRID of StrId.strid
	    end
         ): MODULE_ENVIRONMENTS =
  struct
    (********
    Constructors
    ********)

    type con = C.con

    (********
    Constructor environments
    ********)

    type ConEnv = C.ConEnv

    val domCE    = C.domCE
    and ClosCE   = C.ClosCE
    and emptyCE  = C.emptyCE
    and singleCE = C.singleCE
    and CE_plus_CE   = C.CE_plus_CE
    and ClosCE_to_VE = C.ClosCE_to_VE

    (********
    Identifiers
    ********)

    type id = C.id

    (********
    TypeSchemes
    ********)

    type TypeScheme = C.TypeScheme

    (********
    Variable environments
    ********)

    type VarEnv = C.VarEnv

    val emptyVE = C.emptyVE
    and ClosVE  = C.ClosVE
    and singleVarVE = C.singleVarVE
    and VEdom = C.VEdom
    and VE_plus_VE  = C.VE_plus_VE

    (********
    Exception constructors
    ********)

    type excon = C.excon

    (****
    Types
    ****)

    type TyVar = C.TyVar
    type Type = C.Type

    (********
    Exception constructor environments
    ********)

    type ExConEnv = C.ExConEnv

    val emptyEE  = C.emptyEE
    and singleEE = C.singleEE
    and EEdom = C.EEdom
    and VE_of_EE = C.VE_of_EE
    and EE_plus_EE = C.EE_plus_EE

    (********
    Type functions
    ********)

    type TypeFcn = ModuleStatObject.TypeFcn

    (********
    Type structures
    ********)

    type TyStr = C.TyStr

    val mkTyStr = C.mkTyStr
    and unTyStr = C.unTyStr

    (********
    Name sets
    ********)

    type NameSet = C.NameSet

    val NameSetUnion = C.NameSetUnion
    and NameSetMinus = C.NameSetMinus

    (********
    Environments
    ********)

    type Env = C.Env

    val emptyE   = C.emptyE
    and E_plus_E = C.E_plus_E
    and namesE   = C.namesE
    and VE_in_E  = C.VE_in_E
    and TE_in_E  = C.TE_in_E
    and impTyVarsE = C.impTyVarsE
    and VE_and_EE_in_E = C.VE_and_EE_in_E
    and VE_and_TE_in_E = C.VE_and_TE_in_E

    val bogus_Env = C.bogus_Env

    infixr E_plus_E

    (********
    Structure identifiers
    ********)

    type strid = StrId.strid

    (********
    Structures
    ********)

    type Str = C.Str
    val bogus_Str = C.bogus_Str

    (********
    Structure environments
    ********)

    type StrEnv = C.StrEnv

    val emptySE    = C.emptySE
    and SE_plus_SE = C.SE_plus_SE
    and SEdom      = C.SEdom
    and singleSE   = C.singleSE
    and SE_in_E    = C.SE_in_E

    infixr SE_plus_SE

    (********
    Core level contexts
    ********)

    type Context = C.Context

    val C_cplus_TE = C.C_cplus_TE

    (********
    Realisations
    ********)

    type Realisation = ModuleStatObject.Realisation

    (********
    Signatures
    ********)

    type Sig = ModuleStatObject.Sig

    fun impTyVarsSig Sig =
      let
	val (_, Str) = ModuleStatObject.unSig Sig
	val (_, E) = C.unStr Str
      in
	C.impTyVarsE E
      end

    (********
    Signature identifiers
    ********)

    type sigid = SigId.sigid

    (********
    Signature environments
    ********)

    datatype SigEnv = SIGENV of (sigid, Sig) FinMap.map

    val emptyG  = SIGENV FinMap.empty
    and singleG = SIGENV o FinMap.singleton

    infixr G_plus_G
    fun (SIGENV G) G_plus_G (SIGENV G') =
      SIGENV(FinMap.plus(G, G'))

    fun namesG (SIGENV G) = 
      FinMap.fold
      (fn (Sig, res) => C.NameSetUnion(res, ModuleStatObject.namesSig Sig))
      C.emptyN G

    fun Gdom (SIGENV G) = FinMap.dom G

    structure TVResult = C.TVResult
    open TVResult

    fun impTyVarsG(SIGENV G) =
      FinMap.fold
	(fn (Sig, result) => C.tvUnion(impTyVarsSig Sig, result))
	IMP_OK G

    (********
    Functor signatures
    ********)

    type FunSig = ModuleStatObject.FunSig

    (********
    Functor identifiers
    ********)

    type funid = FunId.funid

    (********
    Functor environments
    ********)

    datatype FunEnv = FUNENV of (funid, FunSig) FinMap.map

    val emptyF  = FUNENV FinMap.empty
    and singleF = FUNENV o FinMap.singleton

    fun Fdom (FUNENV F) = FinMap.dom F

    infixr F_plus_F
    fun (FUNENV F) F_plus_F (FUNENV F') = FUNENV(FinMap.plus(F, F'))

    fun namesF (FUNENV F) =
      FinMap.fold
      (fn (FunSig,l) => C.NameSetUnion(l, ModuleStatObject.namesFunSig FunSig))
      C.emptyN F

    local
      fun impTyVarsFunSig FunSig =
	let
	  val (_, Str, Sig) = ModuleStatObject.unFunSig FunSig
	  val (_, E) = C.unStr Str
	in
	  C.tvUnion(C.impTyVarsE E, impTyVarsSig Sig)
	end
    in
      fun impTyVarsF(FUNENV F) =
	FinMap.fold
	  (fn (FunSig, result) => C.tvUnion(impTyVarsFunSig FunSig, result))
	  IMP_OK F
    end

    (********
    The static basis
    ********)

    datatype Basis = BASIS of {N : NameSet, F : FunEnv, G : SigEnv, E : Env}

    val initialB =
      BASIS {N = namesE C.initialE, F = emptyF, G = emptyG, E = C.initialE}

    val emptyB =
      BASIS {N = C.emptyN, F = emptyF, G = emptyG, E = emptyE}

    fun B_plus_E (BASIS {N, F, G, E}, E') =
      BASIS {N = N, F = F, G = G, E = E E_plus_E E'}

    fun B_cplus_E (BASIS {N, F, G, E}, E') =
      BASIS {N = C.NameSetUnion(N, namesE E'), F = F, G = G, E = E E_plus_E E'}

    fun B_plus_N (BASIS {N, F, G, E}, N') = 
      BASIS {N = C.NameSetUnion(N, N'), F = F, G = G, E = E}

    fun B_plus_B (BASIS {N, F, G, E}, BASIS {N = N', F = F', G = G', E = E'}) =
      BASIS {N = C.NameSetUnion(N, N'),
	     F = F F_plus_F F',
	     G = G G_plus_G G',
	     E = E E_plus_E E'
	     }

    val bogus_Basis = emptyB

    fun C_of_B (B as BASIS {N, F, G, E}) =
      let
	val T_of_B = C.T_of_N N
      in
	C.mkC(T_of_B, E)
      end

    fun N_of_B (BASIS {N, ...}) = N

    fun E_in_B (N, E) =
      BASIS {N = N, F = emptyF, G = emptyG, E = E}

    fun B_plus_G (BASIS {N, F, G, E}, G') =
      BASIS {N = N, F = F, G = G G_plus_G G', E = E}

    fun B_plus_F (BASIS {N, F, G, E}, F') =
      BASIS {N = N, F = F F_plus_F F', G = G, E = E}

    fun G_in_B (N, G) =
      BASIS {N = N, F = emptyF, G = G, E = emptyE}

    fun F_in_B (N, F) =
      BASIS {N = N, F = F, G = emptyG, E = emptyE}

   (* The following are needed for top-level printing. *)

    fun E_of_B(BASIS{E, ...}) = E
    fun F_of_B(BASIS{F, ...}) = F
    fun G_of_B(BASIS{G, ...}) = G

    (********
    It is assumed that rea does not affect F and G. This holds as an invariant
    throughout elaboration
    ********)

    fun onB (rea, BASIS {N, F, G, E}) : Basis =
      BASIS {N = N, F = F, G = G, E = ModuleStatObject.onE(rea, E)}

  (*
    fun namesB(BASIS {N, F, G, E}) =
      ListUtil.fold C.NameSetUnion C.emptyN [N, namesF F, namesG G, namesE E]
					  (* MEMO: ListUtil is now dead. *)
  *)

    (********
    Qualified structure identifiers
    ********)

    type longstrid = StrId.longstrid

    (********
    Qualified structure identifiers
    ********)

    type longtycon = TyCon.longtycon

    (********
    Lookup functions for the static basis
    ********)

    fun lookup_strid (BASIS {E, ...}, strid) = 
      C.lookupE_strid(E, strid)

    fun lookup_sigid (BASIS {G = SIGENV G, ...}, sigid) =
      FinMap.lookup G sigid

    fun lookup_funid (BASIS {F = FUNENV F, ...}, funid) =
      FinMap.lookup F funid

    local
      (********
      Function which traverses down structure environments
      ********)

      fun lookup E [] = Some(E)
	| lookup E (strid :: rest) =
	  (case C.lookupE_strid(E, strid) of
	     None => None
	   | Some(Str) =>
	       let
		 val (m', E') = C.unStr Str
	       in
		 lookup E' rest
	       end)
    in
      fun lookup_lstrid (BASIS {N, F, G, E}, lstrid) =
	let
	  val (strid_list, strid) = StrId.explode_longstrid lstrid
	in
	  case (lookup E strid_list) of
	      Some(E) => C.lookupE_strid(E, strid)
	    | None => None
	end

      fun lookup_ltycon (BASIS {N, F, G, E}, ltycon) =
	let
	  val (strid_list, tycon) = TyCon.explode_LongTyCon ltycon
	in
	  case (lookup E strid_list) of
	    Some(E) => C.lookupE_tycon(E, tycon)
	  | None => None
	end
    end

    (********
    Structure names
    ********)

    type StrName = C.StrName

    (********
    Type environments
    ********)

    type TyEnv = C.TyEnv

    val emptyTE    = C.emptyTE
    and singleTE   = C.singleTE
    and TE_plus_TE = C.TE_plus_TE
    and TEdom      = C.TEdom

    fun Lookup_tycon (C, tycon) =
      C.Lookup_tycon(C, tycon)

    (********
    Type constructors
    ********)

    type tycon = TyCon.tycon

    (********
    Assemblies
    ********)

    type offspring_Str = (strid, StrName) FinMap.map
     and offspring_Ty  = (tycon, TyStr  ) FinMap.map
    type offspring     = {structures : offspring_Str, types : offspring_Ty};

    type Assembly_Str = (StrName, offspring)FinMap.map
     and Assembly_Ty  = (TypeFcn, TyStr)FinMap.map
    type Assembly     = {m_arcs : Assembly_Str, theta_arcs : Assembly_Ty}

    val emptyA : Assembly = {m_arcs = FinMap.empty, theta_arcs = FinMap.empty}
    val emptyOffspring_Str = FinMap.empty
    and emptyOffspring_Ty  = FinMap.empty;
    val emptyOffspring : offspring = 
			     {structures = emptyOffspring_Str,
			      types = emptyOffspring_Ty}

    val bogus_OffspringStr = FinMap.empty
    val bogus_OffspringTy = FinMap.empty
    val bogus_Assembly = emptyA

    (********
    Make an assembly which contains the immediate substructures and types of S
    ********)

    fun singleA_Str S : Assembly =
      let 
	fun SEofS S =
	  case C.unStr S of (m, E) =>
	    case C.unEnv E of (SE, _, _, _) => SE

	val offspring_Str = 
	  C.SEFold
	  (fn ((strid': (*eqtype*) StrId.strid, S'), off) =>
	     let
	       val (strname, _) = ModuleStatObject.unStr S'
	     in
	       FinMap.plus(off, FinMap.singleton(strid', strname))
	     end)
	  FinMap.empty (SEofS S)

	fun TEofS S =
	  case C.unStr S of (m,E) =>
	    case C.unEnv E of (_,TE,_,_) => TE

	val offspring_Ty = 
	  C.TEFold               (* unspecified *)
	  (fn ((tycon, tystr), off) =>
	     FinMap.plus(off, FinMap.singleton(tycon, tystr)))
	  FinMap.empty (TEofS S)

	val (strname, _) = ModuleStatObject.unStr S
	val offspring = {structures = offspring_Str,
			 types = offspring_Ty}
	val theta_arcs: Assembly_Ty =
	  C.TEFold
	  (fn ((tycon, tystr), off) =>
	    let 
	      val (theta,CE) = ModuleStatObject.unTyStr tystr
	    in 
	      FinMap.plus(off, FinMap.singleton(theta, tystr))
	    end)
	  FinMap.empty (TEofS S)   

	val m_arcs: Assembly_Str = FinMap.singleton(strname, offspring)

      in
	{m_arcs = m_arcs, theta_arcs = theta_arcs}
      end

    (********
    Make an assembly from a type structure
    ********)

    fun singleA_TyStr (TyStr : TyStr) : Assembly =
      let
	val (theta, CE) = ModuleStatObject.unTyStr TyStr
      in
	{m_arcs = FinMap.empty, theta_arcs = FinMap.singleton(theta, TyStr)}
      end

    (********
    Find the structure offspring of a structure
    ********)

    fun Alookup_Str (A:Assembly, m) : offspring_Str = 
      if m = C.bogus_StrName then
	bogus_OffspringStr
      else
	case FinMap.lookup (#m_arcs A) m
	  of Some off => #structures off
	   | None => Crash.impossible "ModuleEnvironments.Alookup_Str"

    val Aoffspring_Str_Fold = FinMap.Fold

    (********
    Find the type offspring of a structure
    ********)

    fun Alookup_Ty (A:Assembly, m) =
      case (FinMap.lookup (#m_arcs A) m) of
	Some(off) => #types off
      | None => Crash.impossible "ModuleEnvironments.Alookup_Ty"

    val Aoffspring_Ty_Fold = FinMap.Fold

    fun Alookup_TypeFcn(A:Assembly, theta)=
      case (FinMap.lookup (#theta_arcs A) theta) of
	Some tystr => tystr
      | None => Crash.impossible "ModuleEnvironments.Alookup_TypeFcn"

    (********
    Assemblies
    ********)

    local
      fun merge_offsprings(off1 as {structures = off_Str1, types = off_Ty1},
			   off2 as {structures = off_Str2, types = off_Ty2})=
	  (* the offsprings are assumed consistent *)
	  {structures = FinMap.plus(off_Str1,off_Str2),
	   types = FinMap.plus(off_Ty1, off_Ty2)}
      fun insert_Ty((theta,tystr), A_Ty): Assembly_Ty =
	  (* The assembly (theta,tystr,A_ty) is assumed consistent *)
	  FinMap.plus(A_Ty,FinMap.singleton(theta,tystr))    
      fun insert_Str((m,off), A_Str): Assembly_Str =
	  (* The assembly ((m,off),A_Str) is assumed consistent,
	     but if m is in A_str, the offspring of m in A_str may have
	     a different domain from Dom off, so merging is necessary *)
	  case FinMap.lookup A_Str m of
	    Some off' => 
	      FinMap.plus(A_Str, 
			  FinMap.singleton(m,
			    merge_offsprings(off,off')))
	  | None =>
	      FinMap.plus(A_Str, FinMap.singleton(m,off))

      fun onOffspring_Str(rea, off_Str): offspring_Str =
	 FinMap.composemap (ModuleStatObject.onStrName rea) off_Str
      fun onOffspring_Ty(rea, off_Ty): offspring_Ty =
	 FinMap.composemap (ModuleStatObject.onTyStr rea) off_Ty  (* onTyStr unspecified*)
      fun onOffspring(rea, {structures, types}): offspring =
	 {structures = onOffspring_Str(rea, structures),
	  types = onOffspring_Ty(rea, types)}

      fun onA_Ty(rea, A_Ty) =
	let fun onPair(theta,tystr) = 
	      (ModuleStatObject.onTypeFcn rea theta,
	       ModuleStatObject.onTyStr rea tystr)
	 in FinMap.Fold
	     (fn((theta,tystr), A_Ty') =>
		 insert_Ty(onPair(theta,tystr), A_Ty'))
	     FinMap.empty
	     A_Ty
	end

      fun onA_Str(rea, A_Str) = 
	let fun onPair(m,off) =
		(ModuleStatObject.onStrName rea m, 
		 onOffspring(rea, off))
	 in
	   FinMap.Fold
	    (fn ((m,off), A_Str') =>
	       insert_Str(onPair(m,off), A_Str'))
	    FinMap.empty
	    A_Str
	end

    in
      fun onA(rea, A: Assembly as {m_arcs,theta_arcs}): Assembly =
	{m_arcs = onA_Str(rea, m_arcs),
	 theta_arcs = onA_Ty(rea, theta_arcs)}

      fun union(A1,A2): Assembly =
	let
	  val {m_arcs = A_Str1, theta_arcs = A_Ty1} = A1
	  val {m_arcs = A_Str2, theta_arcs = A_Ty2} = A2
	in
	  {m_arcs = FinMap.Fold insert_Str A_Str2 A_Str1,
	   theta_arcs = FinMap.Fold insert_Ty A_Ty2 A_Ty1}
	end
    end

    (********
    Functions for making an assembly.
    *********
    The NameSet argument, N, for each function contains all the
    bound names (which can be ignored).
    ********)

    local
      infix union

      fun mkAsmbS (N, S) : Assembly * StrName =
	let
	  val (m, E) = C.unStr S
	  val (A1, off1) = mkAsmbEnv(N, E)
	in
	  if C.isIn_StrName(m, N) then
	    (A1, m)
	  else
	    let
	      val m_arcs = FinMap.singleton(m, off1)
	      val theta_arcs = FinMap.empty
	    in
	      (A1 union {m_arcs = m_arcs, theta_arcs = theta_arcs}, m)
	    end
	end

      and mkAsmbEnv (N, E) : Assembly * offspring =
	let
	  val (SE, TE, _, _) = C.unEnv E
	  val (A1, off_Str)  = mkAsmbSE(N, SE)
	  val (A2, off_Ty)   = mkAsmbTE(N, TE)
	in    
	  (A1 union A2, {structures = off_Str, types= off_Ty})
	end

      and mkAsmbSE (N, SE) : Assembly * offspring_Str =
	let
	  fun f ((strid, S), (A1, off)) =
	    let
	      val (A2, m1) = mkAsmbS(N, S)
	    in
	      (A1 union A2, FinMap.plus(off, FinMap.singleton(strid, m1)))
	    end
	in
	  C.SEFold f (emptyA, emptyOffspring_Str) SE
	end

      and mkAsmbTE (N, TE) : Assembly * offspring_Ty =
	let
	  fun f ((tycon, tystr), (A, off)) =
	    let 
	      val (theta, CE) = ModuleStatObject.unTyStr tystr
	      val theta_arcs = FinMap.singleton(theta, tystr)
	    in
	      (A union {m_arcs = FinMap.empty, theta_arcs = theta_arcs},
	       FinMap.plus(off, FinMap.singleton(tycon, tystr)))
	    end
	in
	  C.TEFold f (emptyA, emptyOffspring_Ty) TE
	end

      fun mkAsmbSig (Sigma) : Assembly = 
	let
	  val (N, S) = ModuleStatObject.unSig Sigma
	in
	  #1(mkAsmbS(N, S))
	end

      fun mkAsmbFunSig (Phi) : Assembly =
	let
	  val (N1, S1, N'S') = ModuleStatObject.unFunSig Phi
	  val (N2, S2) = ModuleStatObject.unSig N'S'
	  val A1 = #1(mkAsmbS(N1, S1))
	  val A2 = #1(mkAsmbS(C.NameSetUnion(N1, N2), S2))
	in
	  A1 union A2
	end

      fun mkAsmbE (E) : Assembly =
	#1(mkAsmbEnv(C.emptyN, E))

      fun mkAsmbF (FUNENV F) : Assembly =
	FinMap.fold (fn (Phi, A) => A union mkAsmbFunSig(Phi)) emptyA F

      fun mkAsmbG (SIGENV G) : Assembly =
	FinMap.fold (fn(Sig, A) => A union mkAsmbSig(Sig)) emptyA G

      fun mkAssembly (BASIS {N, F, G, E}) : Assembly =
	mkAsmbF(F) union mkAsmbG(G) union mkAsmbE(E)
    in
      (********
      Make an assembly from a structure.
      ********)

      val mkAsmbS = fn (S : Str) => mkAsmbS(C.emptyN, S)

      (********
      Make an assembly from a basis.
      *********
      An assembly contains all the free structures and types in B.
      ********)

      val mkAssembly : Basis -> Assembly = mkAssembly
    end

    (********
    Check for a cyclic assembly
    ********)

    fun cyclic (A as {m_arcs = A_Str, theta_arcs}) = 
      let
	fun cyclic_m (_, Some lstrid) = Some lstrid
	  | cyclic_m ((m, off:offspring), None) =
	    (* depth-first search for a path to a structure 
	       with name m in off *)
	  let
	    fun check_off1 path (_, Some lstrid) = Some lstrid
	      | check_off1 path ((strid, m'), None) =
		check_m m' (strid :: path)

	    and check_off2 (off:offspring_Str) path =
	      FinMap.Fold (check_off1 path) None off

	    and check_m m' path =
	      if m' = m then
		case (rev path) of
		  (h :: t) => Some (StrId.implode_longstrid(rev t, h))
		| [] => Crash.impossible "ModuleEnvironments.check_m"
	      else
		check_off2 (Alookup_Str(A, m')) path
	  in 
	    check_off2 (#structures off) []
	  end
      in 
	FinMap.Fold cyclic_m None A_Str
      end


    (********
    Result from the `covers' function
    ********)

    open Tail_T
    datatype CoverResult = OK | NOT_OK of strid list * Tail

    (********
    The `covers' function
    *********

    The covers function detects whether A covers S, where NofB are the
    rigid names.  It is assumed that if a structure S0 with name m occurs
    in S and if m is in NofB then m is in the domain of A. Similarly, it
    is assumed that whenever theta occurs in S and theta is grounded in
    NofB then theta is in the domain of A. This is the case when A is the
    assembly obtained from a basis B which has been produced by the
    elaborator at top-level and NofB is N of B.

    ********)

    fun covers (NofB, A, S) : CoverResult =
      let 
	exception Fail of string

	fun coversStr (path : strid list, S) : CoverResult  =
	  let
	    val (m, E) = C.unStr S
	    val (SE, TE, _, _) = C.unEnv E
	    val domSE = C.SEdom SE
	    val domTE = C.TEdom TE    (* TEdom unspecified *)
	  in
	    (* Check if m is rigid *)
	    if C.isIn_StrName(m, NofB) then
	      let
		val offA_Str = Alookup_Str(A, m)
		val offA_Ty  = Alookup_Ty(A,m)
	      in
		case (ListHacks.minus(domSE: (*eqtype*) StrId.strid list,
				      EqSet.list(FinMap.dom offA_Str)
				     ),
		      ListHacks.minus(domTE, EqSet.list(FinMap.dom offA_Ty)))
		  of (nil,nil) => 
		       coversSE(path, SE)  (* check substructures *)
		   | (strid::rest, _) => NOT_OK(rev path, STRID strid)
		   | (_, tycon::rest) => NOT_OK(rev path, TYCON tycon)
	      end
	    else
	      coversSE(path, SE)
	  end

	and coversSE (path, SE) : CoverResult =
	  let
	    fun f((strid: (*eqtype*) StrId.strid, S), OK) =
		      coversStr(strid :: path, S)
	      | f((strid, S), failure) = failure
	  in
	    C.SEFold f OK SE
	  end
      in
	coversStr(nil, S)
      end

    (********
    Get an instance of a signature
    *********
    The assembly contains any new names which have been introduced
    ********)

    fun Sig_instance (Sig) : Str * Assembly =
      let
	val (N, S) = ModuleStatObject.unSig Sig
	val rea = ModuleStatObject.renaming N

	val S' = ModuleStatObject.onS(rea, S)
	val (A, _) = mkAsmbS S'
      in
	(S', A)
      end


    (********
    PrettyPrinting for the Basis
    ********)

    type StringTree = PP.StringTree

    fun layoutSigEnv(SIGENV m) =
      let
	val l = FinMap.Fold (op ::) nil m

	fun format_id sigid =
	  implode ["signature ", SigId.pr_SigId sigid, " : "]

	fun layoutPair(sigid, Sig) = 
	  PP.NODE{start=format_id sigid,
		  finish="",
		  indent=3,
		  children=[ModuleStatObject.layoutSig Sig],
		  childsep=PP.NONE
		  }
      in 
	case l of
	  nil => PP.LEAF ""		(* No signatures => no printout *)
	| _ =>
	    PP.NODE{start="", finish="", indent=0, 
		    children=map layoutPair l, childsep=PP.RIGHT " "
		    }
      end

    fun layoutFunEnv (FUNENV m) =
      let
	val l = FinMap.Fold op :: nil m
	fun format_id funid = implode ["functor ", FunId.pr_FunId funid, " : "]

	fun layoutPair(funid, FunSig) = 
	  PP.NODE{start=format_id funid, finish="", indent=3,
		  children=[ModuleStatObject.layoutFunSig FunSig],
		  childsep=PP.NONE
		  }
      in
	case l of
	  nil => PP.LEAF ""		(* No functors => no printout *)
	| _ =>
	    PP.NODE{start="", finish="", indent=0, 
		    children=map layoutPair l, childsep=PP.RIGHT " "
		    }
      end

    val layoutBasis =
      if Flags.DEBUG_ENVIRONMENTS then
	fn (BASIS {N, F, G, E}) =>
	  PP.NODE{start="", finish="", indent=0,
		  children=[C.layoutNameSet N, layoutFunEnv F,
			    layoutSigEnv G, C.layoutEnv E
			   ],
		  childsep=PP.RIGHT " "
		  }
      else
	fn (BASIS {N, F, G, E}) =>
	  PP.NODE{start="", finish="", indent = 0,
		  children=[layoutFunEnv F, layoutSigEnv G, C.layoutEnv E],
		  childsep = PP.RIGHT " "
		 }

    type Report = Report.Report

    fun reportFunEnv(report, FUNENV m) =
      FinMap.reportMapSORTED (FunId.<) report m

    fun reportSigEnv(report, SIGENV m) =
      FinMap.reportMapSORTED (SigId.<) report m
  end;
