(* exception Error of string
 * fun error s = raise (Error s)
 *
 * general error function
 *)
 

local
datatype InputDevice =
  NoDevice | Locator (*| Keyboard*)

and InputModeType =
  Inactive | Event (* Sample *)

and Button =
  LeftButton | MiddleButton | RightButton

and ButtonStatus = 
  Up | Down

and WriteModeType =
  WriteReplace | WriteXor | WriteOr | WriteAnd

and Color =
  White | Black
in (*local*)

datatype Point = Point of {x:int, y:int}
and      Line  = Line of Point * Point
and      Rect  = Rect of {left:int, bottom:int, right:int, top:int}

local
datatype LocatorMeasure = LocatorMeasure of {position:        Point, 
					last_transition: Button,
					status:          ButtonStatus}

datatype srgp =
    SrgpBeep
  | SrgpBegin of {width:int, height:int}
  | SrgpEnableSynchronous
  | SrgpEnd
  | SrgpFillRectangleCoord of Rectangle
  | SrgpGetLocator 
  | SrgpLineCoord of Line'
  | SrgpPointCoord of Point
  | SrgpRectangleCoord of Rectangle
  | SrgpRefresh
  | SrgpSetColor of Color
  | SrgpSetInputMode of InputDevice * InputModeType
  | SrgpSetLocatorButtonMask of Button list
  | SrgpSetLocatorMeasure of Point
  | SrgpSetWriteMode of WriteModeType
  | SrgpWaitEvent of int 

withtype
    Line'      = {x1:int, y1:int, x2:int, y2:int}
and Rectangle = {left:int, bottom:int, right:int, top:int}

datatype Result = 
  InputDevice of InputDevice | Measure of LocatorMeasure

datatype 'a Option = None | Some of 'a
    

nonfix srgp_call 236 1
    
fun srgp_call(x:srgp):Result Option = srgp_call(x)



local
    val doneIt = ref false
in
    val createWindow: {width:int, height:int} -> {width:int, height:int} =
	fn p  => if !doneIt then error "Window already started"
		 else (srgp_call (SrgpBegin p);
		       doneIt := true;
		       p)
    val destroyWindow: unit -> unit =
	fn () => if !doneIt then
	            (srgp_call SrgpEnd;
		     doneIt := false;
		     ())
		 else error "No active window"
end (*local*)
in


    val drawPoint: Point -> Point =
        fn p => (srgp_call (SrgpPointCoord p); p)

    val drawLine: Line -> Line =
	fn (l as (Line(Point {x,y}, Point {x=x',y=y'}))) =>
	    (srgp_call (SrgpLineCoord {x1=x,y1=y,x2=x',y2=y'});
	     l)

local
    fun check (r as {bottom,top,left,right}:Rectangle) =
	if bottom <= top andalso left <= right then r
	else
	    error "Rect valid only if (bottom <= top) andalso (left <= right)"
in
	
    val drawRect: Rect -> Rect =
	fn (rect as (Rect r)) => (srgp_call (SrgpRectangleCoord (check r));
				  rect)

    val fillRect: Rect -> Rect =
	fn (rect as (Rect r)) => (srgp_call (SrgpFillRectangleCoord (check r));
				  rect)
end(*local*)
				  
local
    exception GraphicsError of string

    val waitEvent: int -> InputDevice =
        fn n =>
	    (case srgp_call (SrgpWaitEvent n) of
		Some (InputDevice idev) => idev
	      | _ => raise (GraphicsError "Impossible"))

    val waitEventForever: unit -> InputDevice =
	fn () => waitEvent ~1

    and pollEvent: unit -> InputDevice =
	fn () => waitEvent 0

    val readLocator: unit -> LocatorMeasure =
        fn () => 
	    (case srgp_call SrgpGetLocator of
		Some (Measure lm) => lm
	      | _  => raise (GraphicsError "Impossible"))

    datatype GCFunction =
	EnableSynchronous
      | Refresh
      | InputMode of InputDevice * InputModeType
      | ButtonMask of Button list
      | Color of Color
(*      | LocatorPosition of Point*)
      | WriteMode of WriteModeType

    val setFunction: GCFunction -> unit =
      fn EnableSynchronous   => (srgp_call SrgpEnableSynchronous;())
       | Refresh             => (srgp_call SrgpRefresh;())
       | (InputMode x)       => (srgp_call (SrgpSetInputMode x);())
       | (ButtonMask l)      => (srgp_call (SrgpSetLocatorButtonMask l);())
       | (Color c)           => (srgp_call (SrgpSetColor c);())
(*       | (LocatorPosition x) => (srgp_call (SrgpSetLocatorMeasure x);())*)
       | (WriteMode w)       => (srgp_call (SrgpSetWriteMode w);())
     
(*    val GCxor = WriteMode WriteXor*)
in

    local
        val wSize = ref (Rect {top=0,bottom=0,left=0,right=0})
    in

    val startGraphics:{width:int, height:int} -> Rect =
	fn (p as {width, height})=> if width>0 andalso height>0 then
	    (createWindow p;
	     setFunction EnableSynchronous;
	     setFunction (InputMode (Locator,Event));
	     setFunction (ButtonMask [LeftButton,MiddleButton,RightButton]);
             wSize := (Rect {left=0,bottom=0,top=height-1,right=width-1});
	     !wSize)
	     else error "width and height must be greater than zero"

    val windowSize: unit -> Rect =
        fn () => !wSize

    end (*local*)

    val getPoint : unit -> Point =
	fn () =>
            let fun getDown (LocatorMeasure {status=Down,position,...}) =
                            position
		  | getDown _ = getPt ()

                and getPt   () = 
		    (case waitEventForever () of
			 Locator => getDown (readLocator ())
		       | NoDevice => error "No input from mouse")
	    in
		getPt ()
	    end
	
     val clearGraphics: unit -> unit =
         fn () => (setFunction (Color Black);
		   fillRect (windowSize ());
		   setFunction (Color White))
             
end (*local*)
end (*local*)
end (*local*)
