-- (C) Copyright International Business Machines Corporation 16 November, 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: ator.p
-- Author: William Silverman
-- SCCS Info: @(#)ator.p	1.2 12/6/90

-- This process accepts a string which represents an integer, optionally
-- followed by a proper fraction, optionally followed by an exponent -
-- and converts it into a real.  An initial sign is allowed in the integer.

ator: using (string)
	linking(normalize3)

process (Q: string2realQ)
  
declare
  args: string2real;

  scopy: charstring;
  c: char;

  sign: converted_real;
  low: integer; mid: integer; high: integer;
  fractional: boolean;
  exponent: integer;

  zcode: integer;
  zero: integer; one: integer; ten: integer;
  ip2: integer;
  pof2: integer;
  rpof2: converted_real;

  excess: integer;

begin
  receive args from Q;

  block begin

    scopy := args.string;
    low <- 0; mid <- 0; high <- 0;
    fractional <- 'false';
    exponent <- 0;

    zcode <- convert of char#'0';
    zero <- 0; one <- 1; ten <- 10;

-- the following values are chosen for machines with 32 bit integers
-- and 64 bit (double) reals with maximum real bounded by 2^1023.
-- Adjustment may be necessary for machines which difer substantially
-- from these assumptions.

    ip2 <- 24;
    pof2 <- 16777216;				-- 2^24

  -- handle first character of real-literal - may be sign or digit
    remove c from scopy[0];
    sign <- converted_real#(convert of one);
    select c
      where ('+')
        remove c from scopy[0];
      where ('-')
        sign <- -sign;
        remove c from scopy[0];
      otherwise
    end select;

    excess <- 0;
    block declare
      ic: integer;
    begin

      while 'true' repeat
	select c

	  where('.')
	    if fractional then
	      exit formaterror;
	    end if;
	    fractional <- 'true';		-- start fraction part

	  where('e')
	    block declare
	      esign: integer;
	      ecum: integer;
	    begin

	   -- handle first character of exponent - may be sign or digit
              remove c from scopy[0];
	      esign <- 1;
	      select c
		where ('+')
		  remove c from scopy[0];
		where ('-')
		  esign <- -esign;
		  remove c from scopy[0];
		otherwise
	      end select;

	      ecum <- 0;
	      while 'true' repeat
		ic <- (convert of c) - zcode;
		if ic < zero or ic >= ten then
		  exit formaterror;
		end if; 
		ecum <- ten*ecum + ic;
		if size of scopy = zero then
		  exponent <- exponent + esign*ecum;
		  exit scaling;
		end if;
		remove c from scopy[0];
	      end while;
	      
	    on (Depletion)
	      exit formaterr;

	    end block;

	    
	  otherwise				-- digit
	    ic <- (convert of c) - zcode;
	    if zero <= ic and ic <= ten then
	      if excess = zero  then
		low <- ten*low + ic;
		if low >= pof2 then		-- start extended cumulation
		  mid <- low / pof2;
		  low <- low mod pof2;
		  excess <- 1;
		end if;
	      else				-- continue extended cumulation
		if excess = 3 then
		  exponent <- exponent + one;
		else
		  low <- ten*low + ic;
		  ic <- low / pof2;
		  low <- low mod pof2;
		  mid <- ten*mid + ic;
		  if excess = one then
		    if mid >= pof2 then
		      high <- mid / pof2;
		      mid <- mid mod pof2;
		      excess <- 2;
		    end if;
		  else
		    ic <- mid / pof2;
		    mid <- mid mod pof2;
		    high <- ten*high + ic;
		    if high >= pof2 then
		      high <- high/10;
		      exponent <- exponent + one;
		      excess <- 3;
		    end if;
		  end if;
		end if;
	      end if;
	      if fractional then			-- in fraction-part
		exponent <- exponent - one;
	      end if;
	    else					-- not a digit
	      if c = 'E' then
		insert 'e' into scopy at 0;		-- 'E' equivalent 'e'
	      else
		exit formaterror;
	      end if;
	    end if;

	end select;

	if size of scopy = zero then
	  exit scaling;
	end if;
	remove c from scopy[0];

      end while;

    on (NotFound)
      exit formaterror;

    on exit (scaling)
    end block;

    rpof2 <- converted_real#(convert of pof2);

  -- scale by resultant exponent of ten
    if exponent = 0 then
      args.real <- rpof2*(rpof2*(convert of high) + (convert of mid)) +
			convert of low;
    else

      block declare

	expof2: integer;

	two: integer;
	hip10: integer;
	mip10: integer;
	lop10: integer;
	exp2ofp10: integer;
	value: converted_real;
	rp2scale: converted_real;
	rp16scale: converted_real;
	rhigh: converted_real;
	rmid: converted_real;
	rlow: converted_real;
	rtwo: converted_real;
	rhip10: converted_real;
	rmip10: converted_real;
	rlop10: converted_real;
	kluge: converted_real;
	normalize: normalizeFn;

      begin

	two <- 2;
	select excess
	  where(0)
	    expof2 <- -two*ip2;
	    high <- low;
	    low <- 0;
	  where(1)
	    expof2 <- -ip2;
	    high <- mid;
	    mid <- low;
	    low <- 0;
	  otherwise
	    expof2 <- 0;
	end select;

	if high = zero then
	  args.real <- converted_real#(convert of zero);
	else
	  normalize <- normalizeFn#(procedure of process normalize3);
	  if exponent > zero then
--	    hip10 <- ten*(pof2/16);
	    hip10 <- 10485760;
	    mip10 <- 0;
	    lop10 <- 0;
	    exp2ofp10 <- 4;
	  else
	    exponent <- -exponent;
--	    hip10 <- pof2/ten;
--	    lop10 <- pof2*(pof2 - ten*hip10);
--	    mip10 <- lop10/ten;
--	    lop10 <- (pof2*(lop10 - ten*mip10))/ten;
	    hip10 <- 13421772;
	    mip10 <- 13421772;
	    lop10 <- 13421772;
	    exp2ofp10 <- -3;
--	    call normalize(pof2, lop10, mip10, hip10, exp2ofp10);
	  end if;

	  rtwo <- converted_real#(convert of two);
	  while exponent <> zero repeat
	
	    rhip10 <- converted_real#(convert of hip10);
	    rmip10 <- converted_real#(convert of mip10);
	    rlop10 <- converted_real#(convert of lop10);

	    if (exponent mod two) <> zero then	-- multiply by power of 10
	      call normalize(pof2, low, mid, high, expof2);
	      rhigh <- converted_real#(convert of high);
	      rmid <- converted_real#(convert of mid);
	      rlow <- converted_real#(convert of low);
	      low <- integer#(convert of ((rhip10*rlow + rmip10*rmid + rlop10*rhigh)/rpof2));
	      value <- (rhip10*rmid + rmip10*rhigh + (convert of low))/rpof2;
	      mid <- integer#(convert of value);
	      low <- integer#(convert of ((value - (convert of mid))*rpof2));
	      value <- (rhip10*rhigh + (convert of mid))/rpof2;
	      high <- integer#(convert of value);
	      mid <- integer#(convert of ((value - (convert of high))*rpof2));
	      expof2 <- expof2 + exp2ofp10;
	    end if;

	    exponent <- exponent/two;
	    if exponent <> zero then		-- square power of 10
	      kluge := rmip10;
	      lop10 <- integer#(convert of ((rmip10*kluge + rtwo*(rhip10*rlop10))/rpof2));
	      value <- (rtwo*(rhip10*rmip10) + (convert of lop10))/rpof2;
	      mip10 <- integer#(convert of value);
	      lop10 <- integer#(convert of ((value - (convert of mip10))*rpof2));
	      kluge := rhip10;
	      value <- (kluge*rhip10 + (convert of mip10))/rpof2;
	      hip10 <- integer#(convert of value);
	      mip10 <- integer#(convert of ((value - (convert of hip10))*rpof2));
	      exp2ofp10 <- two*exp2ofp10;
	      call normalize(pof2, lop10, mip10, hip10, exp2ofp10);
	    end if;
	  end while;
	  rp2scale <- converted_real#(convert of two);
	  rp16scale <- converted_real#(convert of integer#16);
	  if expof2 < zero then
	    expof2 <- -expof2;
	    rp16scale <- (convert of one)/rp16scale;
	    rp2scale <-  (convert of one)/rp2scale;
	  end if;
	  select (expof2 mod 4)
	    where(0)
	      rp2scale <- converted_real#(convert of one);
	    where(2)
	      kluge := rp2scale;
	      rp2scale <- kluge*rp2scale;
	    where(3)
	      kluge := rp2scale;
	      rp2scale <- (kluge*rp2scale)*rp2scale;
	    otherwise
	  end select;
	  expof2 <- expof2/4;
	  value <- sign * (rpof2*(rpof2*(convert of high) + (convert of mid)) +
				convert of low);
	  if expof2 = zero then
	    args.real <- rp2scale*value;
	  else
	    while expof2 <> one repeat
	      if expof2 mod two = one then
		rp2scale <- rp2scale*rp16scale;
	      end if;
	      expof2 <- expof2/two;
	      kluge := rp16scale;
	      rp16scale <- kluge*rp16scale;
	    end while;
	    args.real <- (value*rp16scale)*rp2scale;
	  end if;

	end if;

      on (Depletion)
	exit formaterror;		-- might be due to storage exhaustion!?
      end block;

    end if;

    return args;

  on exit (formaterror)
    return args exception badFormat;
  end block;

end process
