-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: printdef.p
-- Author: Andy Lowry, from Rob Strom
-- SCCS Info: @(#)printdef.p	1.6 2/5/92

-- This program produces a printed representation of a compiled
-- definitions module.  It doesn't show everything about all the
-- definitions yet, but it's complete enough to be useful on occasion.

printDef: 
using (common, main, filedef, absFormat, inferredtype, interpform,
  loadProg, parse, objectIO, load)
process ( initport: mainQ )

declare
  init: main;
  filename: charstring; -- of old/new definition
  filename2: charstring; -- of imported definitions
  arbobj: polymorph;
  procMaps: executable_printmappings;
  defMaps: definitions_printmappings;
  definitions: definitions_modules; -- all the definitions modules
  filedef: filed_definition; -- definition being loaded
  toBePrinted: filed_definition;  -- the definition to be printed
  imports: module_printmap; -- the definitions needed by the module being loaded
  imported: module_printmap; -- the definitions we have already loaded
  neededModule: module_printrec; -- the description of a module we need to import
  formatters: formatters;
  pathReadObj: readobject_func;
  pathload: load_func;
begin 
  receive init from initport;

  -- Load the Services we need

  unwrap pathReadObj from Init.rm.get("pathReadObj", "") {init};
  unwrap pathLoad from init.rm.get("pathLoad", "") {init};
  formatters <- (getFormattersOutport#(create of pathload("getformatters")))();

  -- Main loop: process each definition
  -- 1. build an empty list of definitions and printmappings
  -- 2. read the definition to be printed
  -- 4. insert it and its printmap into the full definitions list
  --    and printmappings
  -- 5. insert its dependencies into the needed module list
  -- 6. drag in dependent modules until none needed
  -- 7. For each definition in the module to be fixed:
  --    a. print the typename and the family (components)
  --    b. for inports, the entry formal
  --    c. for callmessages, the EXIT, and each exception formal
  --    d. for tables, the element type and formal typestate
  --    e. eventually, others
  for string in init.argv where (position of string >= 2) inspect
    block begin
      -- 1.
      new definitions;
      new defMaps;
      -- 2.
      filename := string | ".do";
      BLOCK
      BEGIN
	arbobj <- pathReadObj(filename);
	unwrap toBePrinted from arbobj { 
	  init, init(definitions_module),
	  checkeddefinitions(definitions_module),
	  init(definitions_module.id), 
	  init(definitions_module.type_definitions),
	  init(definitions_module.attr_definitions),
	  init(defmap), init(defmap.id), init(defmap.name), 
	  init(defmap.types), init(defmap.attributes),
	  init(defmap.components), init(defmap.exceptions),
	  init(procmaps), init(direct_imports)
	};
      ON (ReadObject_Intf.File_Not_Found)
	call Init.terminal.putLine
	    ("Definitions module '" | string | "' not found.");
	EXIT PrematureTermination;
      END BLOCK;
      -- 4. 
      insert copy of toBePrinted.definitions_module into definitions;
      insert copy of toBePrinted.DefMap into defMaps;
      -- 5. 
      imports := toBePrinted.Direct_Imports;
      new imported;
      REMOVE NeededModule FROM M IN imports WHERE(M.Name = string);
      INSERT NeededModule INTO imported;
      -- 6. 
      while size of imports <> 0 repeat
	block begin
	  -- get the name of a module which needs to be imported
	  remove NeededModule from map in imports where ('true');
	  
	  -- read the definition from disk
	  filename2 <- NeededModule.name | ".do";
	  arbobj <- pathReadObj(filename2);
	  unwrap filedef from arbobj { 
	    init, init(definitions_module),
	    checkeddefinitions(definitions_module),
	    init(definitions_module.id), 
	    init(definitions_module.type_definitions),
	    init(definitions_module.attr_definitions),
	    init(defmap), init(defmap.id), init(defmap.name), 
	    init(defmap.types), init(defmap.attributes),
	    init(defmap.components), init(defmap.exceptions),
	    init(procmaps), init(direct_imports)
	  };

	  -- verify that the moduleid on disk matches the moduleid expected
	  if filedef.definitions_module.id <> NeededModule.id 
	  then
	    call Init.terminal.putLine
		("Definitions module '" | NeededModule.Name 
		  | "' is inconsistent with '" | string | "'.");
	    EXIT PrematureTermination;
	  end if;

	  -- append the definitions module and definitions printmap
	  insert filedef.definitions_module into definitions;
	  insert filedef.defmap into defMaps;
	  discard filedef.defmap ;

	  -- put the module name into the set of modules imported
	  insert copy of NeededModule into imported;
      
	  -- for each module imported by the definition:
	  -- if it hasn't been imported yet, add it to the list of
	  -- needed imports if it has been imported already, verify
	  -- that the expected moduleid matches the loaded moduleid
	  for map in filedef.direct_imports where ('true') inspect 
	    block begin
	      inspect ImportedModule IN Imported WHERE(ImportedModule.Name = Map.Name)
	      BEGIN
		IF ImportedModule.Id <> Map.Id
	        THEN
                  call Init.terminal.putString
		      ("Definitions module '" | ImportedModule.Name
			| "' is inconsistent with '" | NeededModule.Name 
			| "'.");
		EXIT PrematureTermination;
	      END IF;
	    END INSPECT;
	  on (NotFound)
	    BLOCK BEGIN
	      insert copy of map into imports;
	    ON (DuplicateKey)
	    END BLOCK;
	  end block;
	end for;

	discard filedef.direct_imports;
	
      on (readobject_intf.file_not_found)  
	call Init.terminal.putString
	    ("Definitions module '" | NeededModule.Name | "' not found.");
	EXIT PrematureTermination;
      end block;
    end while;

    -- 7.
    block 
    declare
      Type: Typename;  -- typename of definition
    begin
      for definition in toBePrinted.Definitions_Module.Type_Definitions[]
      inspect
	new Type;
	Type.ModuleId := toBePrinted.Definitions_Module.Id;
	Type.TypeId := Definition.Id;
	-- a.
	call Init.terminal.putLine("");
	call Init.terminal.putLine(Formatters.FormatType(Formatters, Definitions, defMaps, Type)
	      | ": " | EVALUATE Family: Charstring FROM
	    SELECT (CASE OF Definition.Specification)
	    WHERE('nominaltype') Family := "nominal";
	    WHERE('integertype') Family := "integer";
	    WHERE('booleantype') Family := "boolean";
	    WHERE('enumerationtype') Family := "enumeration";
	    WHERE('realtype') Family := "real";
	    WHERE('recordtype') Family := "record";
	    WHERE('varianttype') Family := "variant";
	    WHERE('tabletype') Family := "table";
	    WHERE('inporttype') Family := "inport";
	    WHERE('outporttype') Family := "outport";
	    WHERE('callmessagetype') Family := "callmessage";
	    WHERE('polymorphtype') Family := "polymorph";
	    OTHERWISE Family := "should not occur";
	    END SELECT;
	  END);
	SELECT (CASE OF Definition.Specification)
	WHERE('inporttype')
	  REVEAL Definition.Specification.Inport_info;
	  call Init.terminal.putLine("  " | Formatters.FormatFormalTypestate(Formatters, Definitions, defMaps, Definition.Specification.Inport_info.Message_typestate, Definition.Specification.Inport_info.Message_type));
	WHERE('callmessagetype')
	  REVEAL Definition.Specification.Callmessage_info;
	  for Component in Definition.Component_Declarations[]
	  inspect
	    call Init.terminal.putLine("  " | EVALUATE ComponentName: Charstring FROM
		inspect Map in toBePrinted.DefMap.Components WHERE(Map.Type = Type.TypeId AND Map.Component = Component.Id)
		begin
		  ComponentName := Map.Name;
		end inspect;
	      END | ":" | Formatters.FormatType(Formatters, Definitions, defMaps, Component.Type));
	  end for;
	  call Init.terminal.putLine("  EXIT " | Formatters.FormatFormalTypestate(Formatters, Definitions, defMaps, Definition.Specification.Callmessage_info.Normal, Type));
	  for Exception in Definition.Specification.Callmessage_info.Exception_Specifications[]
	  inspect
	    call Init.terminal.putLine ("  " | EVALUATE ExceptionName: Charstring FROM
		BLOCK BEGIN
		  inspect Map in toBePrinted.DefMap.Exceptions WHERE(Map.Type = Type.TypeId AND Map.Exception = Exception.ExceptionId)
		  begin
		    ExceptionName := Map.Name;
		  end inspect;
		ON (NotFound)
		  ExceptionName := "*unnamed*";
		END BLOCK;
	      END | " " | Formatters.FormatFormalTypestate(Formatters, Definitions, defMaps, Exception.Post_Typestate, Type));
	    end for;
	  WHERE('tabletype')
	    REVEAL Definition.Specification.Table_info;
	    call Init.terminal.putLine("  OF " | Formatters.FormatType(Formatters, Definitions, defMaps, Definition.Specification.Table_info.Element_type) | " " | Formatters.FormatFormalTypestate(Formatters, Definitions, defMaps, Definition.Specification.Table_info.Element_Typestate, Definition.Specification.Table_info.Element_Type)); 
	  OTHERWISE
	  end select;
	  
	end for;
      end block;
      
      ----------- Done
    on exit(PrematureTermination)
    end block;
    
  end for;
  
  return init;
  
end process
