-- (C) Copyright International Business Machines Corporation 16 September
-- 1991.  All Rights Reserved.
--
-- See the file USERAGREEMENT distributed with this software for full
-- terms and conditions of use.
-- SCCS Info: @(#)choperands.p	1.2 2/4/92

-- This process revises operands using type equalities derived from
-- ubiquitous operations.
--
-- Input operands and equivalents tables.
--
-- Modifies operands.
--
-- The principal internal structure is trees, an enary tree which maps
-- the entire set of operands.  The first level of trees includes an
-- entry for every base operand of the process (possibly excluding
-- operands with base ids 1 and 2).  Each element is the root of a
-- sub-tree, for each "field" of the base element, etc.  The tree is
-- accessed by the program literal processes identified by  search
-- and  founded .
--
-- First all operands are 'enter'ed into trees, and a containers list
-- of Id prefixes is compiled.
--
-- Next assignments are accumulated for every path through trees, using
-- the 'assign' action'.
--
-- Next well-founded operands are identified and removed from containers,
-- using the founded.
--
-- Next type inference is performed for each invariant operand whose
-- type is implied by equivalents.  Any Id in containers which is
-- proven to be invariant (its immediate ancestor is not in containers
-- and has known, non-'variant' type) is removed from containers.
--
-- Next the 'findbase' action is applied for every remaining element of
-- containers.  This causes all operands which have not been shown to
-- be invariant to be marked 'indeterminate'.
--
-- Finally type inference is performed for every operand which is not
-- 'indeterminate' and whose type is implied by equivalents.
--

choperands: using(chdescriptors, choperands, chphase1, chtransform, interpform)

  process(Q: operandsQ)
    
  declare
    
    cm: operandsCm;
    
    cons: opdict;
    containers: opdict;
    chop: ch_operand;
    continue: boolean;
    e: empty;
    
    eqv: equivalent;
    drops: equivalents;
    x: integer;
    
    search: searchFn;
    founded: foundedFn;
    id: operand;
    tree: optree;
    trees: optrees;
    action: search_action;
    summer: cumulator;
    family: data_family;
    
--rands: ch_operands;
    
  begin
    receive cm from Q;
    
    -- This process searches trees for a (partial) operand id.
    --
    -- Input a (partial) operand Id rest, trees, search_action action.
    --
    -- Modifies action
    --
    -- The process recurses on Id until it has traced a full path
    -- through trees.  Different actions imply different side-effects.
    -- 
    --  'enter'      If Id is in tree, set its index.
    --               If Id is not in trees, recursively add it.
    --  'found'      Verify that each sub-field (partial Id) is
    --               well-founded (Its immediate ancestor is well-
    --               founded, and its type is known.)
    --               Recursion stops when some partial Id is not well-
    --               founded, and the reply to the action is 'false'.
    --               If the Id and all of its prefixes are well-founded,
    --               the reply to the action is 'true'.
    --  'findbase'   If the leaf Id operand is a 0 component of a 1-field
    --               object, it may be a variant.  If it is also assigned
    --               more than once, all of its sub-field operands are
    --               marked 'indeterminate'.
    --  'assign'     References and assignments are accumulated along
    --               the traced path through trees.  This implements
    --               the conservative assumption that assignment to a
    --               compound object assigns each of its sub-fields.
    --               The literal value (if any) of a field which is
    --               assigned more than once is invalidated.
    --               
    
    search <- procedure of process(Q: searchQ)
      declare
	cm: search;
	comp: integer;
	tree: optree;
	trees: optreelist;
	op: ch_operand;
	e: empty;
	id: operand;
      begin
	receive cm from Q;
	remove comp from cm.rest[0];
	block begin
	    if case of cm.action = 'found' then
		inspect cmt in cm.trees[comp] begin
		    if cmt.index < 0 then
			exit unfounded;
		      else
			reveal cm.action.connect;
			inspect rand in cm.action.connect.operands[cmt.index]
			  begin
			    if 'variant' <= rand.Family then
				exit unfounded;
			      end if;
			  end inspect;
		      end if;
		  end inspect;
	      end if;
	    if size of cm.rest <> 0 then
		remove tree from cm.trees[comp];
		call cm.search(cm.rest, tree.content, cm.search, cm.action);
		insert tree into cm.trees;
	      else
		select case of cm.action
		  where('enter')
		    reveal cm.action.index;
		    remove tree from cm.trees[comp];
		    tree.index := cm.action.index;
		    insert tree into cm.trees;
		  where('found')
		    reveal cm.action.connect;
		    cm.action.connect.reply <- 'true';
		  where('findbase')
		    inspect cmt in cm.trees[comp] begin
		      if evaluate maybe: boolean from
			      block begin -- structural check
				  maybe <- size of cmt.content = 1;
				  if maybe then
				      inspect var in cmt.content[] begin
					  maybe <- var.component = 0 and
					     var.sum > 1;
					end inspect;
				    end if;
				end block;
			    end
			then
			  -- This variant or unfounded base has had its [0]
			  -- field assigned at least twice, and it has the
			  -- structure of a variant, therefore it "may be a
			  -- variant", so all of its subfields are
			  -- indeterminate.
			  reveal cm.action.operands;
			  new trees;
			  for tr in cmt.content[] inspect
			      insert copy of tr into trees;
			    end for;
			  while size of trees <> 0 repeat
			      remove tree from trees[];
			      if 0 <= tree.index then
				  remove op
				     from cm.action.operands[tree.index];
				  unite op.Literal.indeterminate from e;
				  op.Family <- 'unknown';
				  insert op
				     into cm.action.operands at tree.index;
				end if;
			      for tr in tree.content[] inspect
				  insert copy of tr into trees;
				end for;
			    end while;
			end if;
		    end inspect;
	          otherwise -- 'assign'
		    reveal cm.action.cumulate;
		    remove tree from cm.trees[comp];
		    if 0 <= tree.index then
			remove op from cm.action.cumulate.operands[tree.index];
			op.References <- op.References +
			   cm.action.cumulate.sum;
			op.Assignments <- op.Assignments +
			   cm.action.cumulate.sum;
			cm.action.cumulate.sum := op.Assignments;
			if op.Assignments > 1 then
			    if case of op.Literal < 'variable' then
				unite op.Literal.variable from e;
			      end if;
			  end if;
			insert op into cm.action.cumulate.operands
			   at tree.index;
		      end if;
		    tree.sum := cm.action.cumulate.sum;
		    for t in tree.content[] inspect
			new id;
			insert copy of t.component into id;
			reveal cm.action.cumulate;
			cm.action.cumulate.sum := tree.sum;
			call cm.search(id, tree.content, cm.search, cm.action);
		      end for;
		    insert tree into cm.trees;
		  end select;
	      end if;
	  on(NotFound) -- must be 'enter'
	    reveal cm.action.index;
	    new tree;
	    tree.component <- comp;
	    tree.sum <- 0;
	    new tree.content;
	    if size of cm.rest = 0 then
		tree.index := cm.action.index;
	      else
		tree.index := -1;
		call cm.search(cm.rest, tree.content, cm.search, cm.action);
	      end if;
	    insert tree into cm.trees;
	  on exit(unfounded)
	    reveal cm.action.connect;
	    cm.action.connect.reply <- 'false';
	  end block;
	return cm;
      end process;
    
    -- This process attempts to show that an id is well-founded.
    --
    -- Input an operand Id, trees, operands.
    --
    -- Output boolean reply.
    --
    -- The search process is called with 'found' action for Id.
    -- The reply it returns in action.connect.reply is relayed to the caller.
    --
    
    founded <- procedure of process(Q: foundedQ)
      declare
	cm: founded;
	id: operand;
	founder: connector;
	action: search_action;
      begin
	receive cm from Q;
	id := cm.id;
	new founder;
	founder.reply <- 'true';  -- arbitrary
	founder.operands <- cm.operands;
	unite action.connect from founder;
	call cm.search(id, cm.trees, cm.search, action);
	reveal action.connect;
	dissolve action.connect into founder;
	cm.operands <- founder.operands;
	cm.reply <- founder.reply;
	return cm;
      end process;
    
    new trees;
    new containers;
    -- trees and initial containers
    for op in cm.operands[] inspect -- enter operand Id s
	unite action.index from position of op;
	call search(copy of op.Id, trees, search, action);
	block
	  declare
	    i: integer;
	  begin
	    id := op.Id;
	    x <- size of id;
	    while x > 1 repeat
		x <- x - 1;
		remove i from id[x];
		insert copy of id into containers;
	      end while;
	  on (DuplicateKey, NotFound)
	  end block;
      end for;
    
    -- cumulate assignments
    new summer;
    summer.operands <- cm.operands;
    summer.sum <- 0;
    unite action.cumulate from summer;
    for t in trees[] inspect
	new id;
	insert copy of t.component into id;
	action.cumulate.sum <- 0;
	call search(id, trees, search, action);
	reveal action.cumulate;
      end for;
    dissolve action.cumulate into summer;
    cm.operands <- summer.operands;
    
    -- remove well-founded operands from containers
    cons := copy of containers;
    while size of cons > 0 repeat
      remove id from cons[];
      if founded(id, trees, search, cm.operands)
      then
	block begin
	  while 'true' repeat
	    remove id from containers[id];
	    remove x from id[size of id - 1];
	    remove id from cons[id];
	  end while;
	on (NotFound)
	end block;
      end if;
    end while;

    -- find implicit types - use strong criteria
    continue <- 'true';
    while continue repeat
	new drops;
	continue <- 'false';
	for eq in cm.equivalents[] inspect
	    inspect op1 in cm.operands[eq.x1] begin
		if op1.Family < 'table' then
		    insert copy of eq into drops;
		  else
		    inspect op2 in cm.operands[eq.x2] begin
			if op2.Family <> 'unknown' then
			    id := op2.Id;
			    remove x from id[size of id - 1];
			    if not exists of containers[id]
			      then
				id := op1.Id;
				remove x from id[size of id - 1];
				if not exists of containers[id]
				  then
				    family := op2.Family;
				    if family < 'variant' then
					block begin
					    remove id from containers[op1.Id];
					  on (NotFound)
					  end block;
				      end if;
				    -- keep  table,table  but dont set 
				    -- continue; permits later  table,string .
				    if family <> op1.Family then
					continue <- 'true';
					if family <> 'table' then
					    insert copy of eq into drops;
					  end if;
					remove chop from cm.operands[eq.x1];
					chop.Family <- family;
					insert chop into cm.operands at eq.x1;
				      end if;
				  end if;
			      end if;
			  end if;
		      end inspect;
		  end if;
	      end inspect;
	  end for;
	
	for drop in drops[] inspect
	    remove eqv from cm.equivalents[drop];
	  end for;
	   
      end while;
    
    -- mark indeterminate operands
    while size of containers <> 0 repeat
	remove id from containers[]; -- container which may be variant
	                             -- (if assigned more than once)
	unite action.operands from cm.operands;
	call search(id, trees, search, action);
	reveal action.operands;
	dissolve action.operands into cm.operands;
      end while;
    
    -- find residual implicit types - use weak criteria
    continue <- 'true';
    while continue repeat
	new drops;
	continue <- 'false';
	for eq in cm.equivalents[] inspect
	    inspect op1 in cm.operands[eq.x1] begin
		if op1.Family < 'table' or
		       case of op1.Literal = 'indeterminate' then
		    insert copy of eq into drops;
		  else
		    inspect op2 in cm.operands[eq.x2] begin
			if op2.Family <> 'unknown' then
			    -- keep  table,table  but dont set continue;
			    -- permits later  table,string .
			    if op1.Family <> op2.Family then
				continue <- 'true';
				remove chop from cm.operands[eq.x1];
				chop.Family := op2.Family;
				insert chop into cm.operands at eq.x1;
			      end if;
			  end if;
		      end inspect;
		  end if;
	      end inspect;
	  end for;
	
	for drop in drops[] inspect
	    remove eqv from cm.equivalents[drop];
	  end for;
	   
      end while;
        
    return cm;
    
  end process
