-- (C) Copyright International Business Machines Corporation 16 September
-- 1991.  All Rights Reserved.
--
-- See the file USERAGREEMENT distributed with this software for full
-- terms and conditions of use.
-- SCCS Info: @(#)chdictionary.p	1.3 3/13/92

-- This server serves two capabilities - one to reference operands and
-- another simply to look them up.
--
-- Input is a lookup request, operands and an interpform!operand ID.
--
-- Output for either is the index of the operand with that ID in operands.
--
-- Modifies operands.
--
-- A lookup request simply finds the operand associatively or adds it
-- to the dictionary and to operands, if it is new.
--
-- A reference request is the same for a new operand.
--
-- For an operand which is already in the dictionary, operand.Family
-- is recomputed, using the request.family.
--
--   If the two agree, or if operand.Family is already 'indeterminate',
--   no change is made.
--   If the operand is 'unknown', its Family is reset to request.family.
--   Otherwise, if request.family is not unknown, operand.Family becomes
--   either operand.Family becomes 'string' or 'indeterminate' depending
--   on its compatibility with request.family ('table' is compatible with
--   'string').


chdictionary: using(chdescriptors, chdictionary, chphase1, chtransform)

  process(I: dictionaryI)
    
  declare
    
    cm: dictionary_init;
    rQ: lookupQ;
    lQ: lookupQ;
    incorporate : incorporateFn;
    request : lookup;
    operand: ch_operand;
    chop: ch_operand;
    dictionary: dictionary;
    refs: counter;
    e: empty;
    
  begin
    receive cm from I;
    discard I;
    
    new rQ;
    connect cm.reference to rQ;
    new lQ;
    connect cm.lookup to lQ;
    return cm;
    
    -- This process adds a new operand to operands and to the dictionary.
    --
    -- Input request is a operands, dictionary, a lookup callmessage
    -- (see chphase1.d) and references.
    --
    -- Output request.operand_index.
    --
    -- Modifies operands, dictionary.
    --
    
    incorporate <- procedure of process (Q : incorporateQ)
      declare
	chop: ch_operand;
	cm: incorporate;
	association: association;
	e: empty;
      begin
	receive cm from Q;
	new chop;
	chop.ID := cm.request.ID;
	chop.Family := cm.request.family;
	chop.Assignments <- 0;
	chop.References <- cm.references;
	unite chop.Literal.none from e;
	new association;
	association.ID := chop.ID;
	association.index <- size of cm.request.operands;
	cm.request.operand_index := association.index;
	insert association into cm.dictionary;
	insert chop into cm.request.operands;
	return cm;
      end process;
    
    new dictionary;
    while 'true' repeat
	
	block begin
	
	    select
	    
	      event rQ
	    
		receive request from rQ;
		if size of request.ID = 0 then
		    request.operand_index <- -1;
		  else
	    
		    refs <- 1;
		    -- update operands
		    inspect association in dictionary[request.ID] begin
			remove chop from request.operands[association.index];
			chop.References <- chop.References + refs;
			if chop.Family <> request.family then
			    if case of chop.Literal <> 'indeterminate' then
				if chop.Family = 'unknown' then
				    chop.Family := request.family;
				  else
				    if request.family <> 'unknown' then
					-- if not compatible families
					if request.family = 'string' and
					      chop.Family  = 'table'
					  then
					    chop.Family <- 'string';
					  else
					    if chop.Family <> 'string' or
						   request.family <> 'table'
					      then
						-- must be a variant component
						unite
						   chop.Literal.indeterminate
						   from e;
						chop.Family <- 'unknown';
					      end if;
					  end if;
				      end if;
				  end if;
			      end if;
			  end if;
			request.operand_index := association.index;
			insert chop into request.operands at association.index;
		      end inspect;
		
		  end if;
	    
	      event lQ
	    
		receive request from lQ;
	    
		refs <- 0;
		-- lookup operand
		inspect association in dictionary[request.ID] begin
		    request.operand_index := association.index;
		  end inspect;
		
	      otherwise
		exit cantHappen;
		
	      end select;
	  
	  on (NotFound)
	    call incorporate(request, refs, dictionary);
	  end block;
	
	return request;
	
      end while;
    
  on (disconnected)
    
  on exit(cantHappen)
    
  end process
