-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- SCCS info: @(#)class.p	1.4 3/13/90 
-- This file contains Class,  the 'master' class function that calls
-- type class functions:
-- 'boolean', 'numeric', 'variant', 'variantcomponent',
-- 'string', 'inport', 'outport', 'orderedscalar', 
-- 'table', 'integer', 'enumerationorboolean', 'nominal', 'enumeration',
-- 'newable', 'polymorph', 'copyable', 'orderedtable', 'callmessage'

-- The type class functions have signature:
--    typename X definitions -> boolean,  
-- except copyable, which takes an additional argument that is
-- maintained through recursive calls, and should be empty for the
-- top-level call.  Also Variantcomponent requires the argument object
-- itself (not just its type), declarations to resolve the parent
-- type, and a capability to the variant class checker.

class: USING(type, errors)

PROCESS(SetupQ: SetupClassQueue) 

DECLARE
  SetupM: SetupClassMessage;
  ClassQ: ClassQueue;
  ClassC: ClassCapa;
  BooleanPort: ClassRuleCapa;
  NumericPort: ClassRuleCapa;
  VariantPort: ClassRuleCapa;
  VariantComponentPort: VariantComponentCapa;
  StringPort: ClassRuleCapa;
  InportPort: ClassRuleCapa;
  OutportPort: ClassRuleCapa;
  OrderedScalarPort: ClassRuleCapa;
  TablePort: ClassRuleCapa;
  IntegerPort: ClassRuleCapa;
  EnumerationOrBooleanPort: ClassRuleCapa;
  NominalPort : ClassRuleCapa;
  EnumerationPort: ClassRuleCapa;
  NewablePort: ClassRuleCapa;
  PolymorphPort: ClassRuleCapa;
  CallMessPort: ClassRuleCapa;
  OrderedTablePort: ClassRuleCapa;
  CopyablePort: CopyableCapa;
  findType: findTypeCapa;
  program: proc;
  defs: definitions_modules;
  encounteredTypes: SetOfTypename;
  blankError: error;
  error: error;
  errObj: errorObject;
  bp: backpatchRecord;
  bpClass: backpatchClass;

BEGIN
  NEW ClassQ;
  CONNECT ClassC TO ClassQ;
  
  RECEIVE SetupM FROM SetupQ;
  
  -- load the individual class rule processes
  BooleanPort <- PROCEDURE OF SetupM.load("boolean");
  NumericPort <- PROCEDURE OF SetupM.load("numeric");
  VariantPort <- PROCEDURE OF SetupM.load("variant");
  VariantComponentPort <- PROCEDURE OF SetupM.load("variantcomponent");
  StringPort <- PROCEDURE OF SetupM.load("string");
  InportPort <- PROCEDURE OF SetupM.load("inport");
  OutportPort <- PROCEDURE OF SetupM.load("outport");
  OrderedScalarPort <- PROCEDURE OF SetupM.load("orderedscalar");
  TablePort <- PROCEDURE OF SetupM.load("table");
  IntegerPort <- PROCEDURE OF SetupM.load("integer");
  EnumerationOrBooleanPort <- 
      PROCEDURE OF SetupM.load("enumerationorboolean");
  NominalPort <- PROCEDURE OF SetupM.load("nominal");
  EnumerationPort <- PROCEDURE OF SetupM.load("enumeration");
  NewablePort <- PROCEDURE OF SetupM.load("newable");
  PolymorphPort <- PROCEDURE OF SetupM.load("polymorph");
  CallMessPort <- PROCEDURE OF SetupM.load("callmess");
  OrderedTablePort <- PROCEDURE OF SetupM.load("orderedtable");
  CopyablePort <- PROCEDURE OF SetupM.load("copyable");
  
  -- keep a copy of findtype and teh absprog
  findType := SetupM.findType;
  program := SetupM.program;
  defs := SetupM.definitions;
  
  -- provide an output port capacity to calling environment
  SetupM.class_func <- ClassC;  
  RETURN SetupM;
  discard SetupQ;
  
  -- make a blank error object template so we can build error
  -- objects more easily when needed
  new blankError;
  blankError.code <- 'class rule violation';
  new blankError.objects;
  
  -- now loop, handling requests
  while 'true' repeat
    block declare
      ClassM: classMessage;
      argType: typename;
      classFunc: classRuleCapa;
      inClass: boolean;
    begin
      receive ClassM from ClassQ;
      
      -- get the type of the argument operand
      block begin
	argType <- findType(ClassM.object, program.executable_part.scopes,
	  defs, ClassM.inferred);
      on (findTypeMessage.unknown_type)
	-- object type not yet known... add a backpatch entry so
	-- this rule will get checked later
	new bp;
	bp.triggerObj := ClassM.object.root;
	new bpClass;
	bpClass.function := ClassM.function;
	bpClass.resultObj := ClassM.object;
	bpClass.position := ClassM.position;
	unite bp.info.class from bpClass;
	insert bp into classM.backpatch;
	exit backpatched;
      end block;
      
      -- Check whether the type is in the required class
      block begin
	select ClassM.function
	where ('boolean')
	  classFunc := BooleanPort;
	where ('numeric')
	  classFunc := NumericPort;
	where ('variant')
	  classFunc := VariantPort;
	where ('variantcomponent')
	  inClass <- variantComponentPort(ClassM.object,
	    program.executable_part.scopes, defs, VariantPort);
	  exit classFuncCalled;
	where ('string')
	  classFunc := StringPort;
	where ('inport')
	  classFunc := InportPort;
	where ('outport')
	  classFunc := OutportPort;
	where ('orderedscalar')
	  classFunc := OrderedScalarPort;
	where ('table')
	  classFunc := TablePort;
	where ('integer')
	  classFunc := IntegerPort;
	where ('enumerationorboolean')
	  classFunc := EnumerationOrBooleanPort;
	where ('nominal')
	  classFunc := NominalPort;
	where ('enumeration')
	  classFunc := EnumerationPort;
	where ('newable')
	  classFunc := NewablePort;
	where ('polymorph')
	  classFunc := PolymorphPort;
	where ('callmessage')
	  classFunc := CallMessPort;
	where ('orderedtable')
	  classFunc := OrderedTablePort;
	where ('copyable')
	  new encounteredTypes;
	  inClass <- CopyablePort(argType, defs, encounteredTypes);
	  exit classFuncCalled;
	otherwise
	  exit cantHappen;
	end select;  
	-- most of the class functions have a normal interface
	inClass <- classFunc(argType, defs);
      on exit(classFuncCalled)
	-- in some case a special interface is needed and the call is
	-- made in the select clause itself
      end block;
      
      if not inClass then
	-- type not in class.. issue an error
	error := blankError;
	-- variantcomponent violation gets its own error code (since
	-- the type of the object is really not of interest in this
	-- case)
	if (ClassM.function = 'variantcomponent') then
	  error.code <- 'variantcomponent';
	end if;
	unite error.position.apos from copy of ClassM.position;
	-- error objects are the object, its type, and the class
	-- rule function
	unite errObj.objectname from copy of ClassM.object;
	insert errObj into error.objects;
	-- variantcomponent violation doesn't use the rest of the
	-- objects
	if (ClassM.function <> 'variantcomponent') then
	  unite errObj.typename from argType;
	  insert errObj into error.objects;
	  unite errObj.type_class_function from copy of ClassM.function;
	  insert errObj into error.objects;
	end if;
	insert error into ClassM.errors;
      end if;
      
      return ClassM;
      
    on exit(backpatched)
      -- Here when we added a backpatch entry... just return without
      -- actually invoking the rule
      return ClassM;
      
    on exit(cantHappen)
      discard ClassM;
      
    end block;
  end while;
end process

