-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: getdep.p
-- Author: Daniel Yellin
-- SCCS Info: "%w% 5/16/90"

-- getdep is given the name of a file.  It returns 
-- (1) the depdencies of that file: if ".po" file its dependencies are 
--     a ".p" file and some ".do" files (the uses list).  If it is a 
--     ".do" file, its dependencies are a ".d" file and some ".do" files.  
--     If it is a ".p" or a ".d" file, it has no dependencies.  
-- (2) a function to make the node: if it is a ".po" file, then the hermes
--     compiler needs to be called.  If it is a
--     ".do" file the Hermes definitions resolver needs to 
--     be called.  ".d" and ".p" files only have the null action associated
--     with them.   
--
-- To locate files, getdep currently uses the path variable HPATH.  
-- If HPATH is not defined, then the path
-- ".:/homes/lusitania/hermes/bin" is used instead.  
-- One addtional path variable, MKSCOPE, is important.  It delimits the
-- scope to be used in checking for consistency.  I.e., any file not in
-- MKSCOPE will assumed to be a "system" file and to be up-to-date.
-- Therefore, no dependencies will bve returned for such a file.  

getdep: using (root,stdenv,common,unix,unixstat,CLoad,timestamp,getuses,
               initloadfuncs,checkscope,findfile,make,getdep,mustmake)
process(setupQ: setupGetDepQ)

declare
  setupM: setupGetDepIntf;
  initloadfuncs: initloadfuncs!initloadfuncsFn; -- func to initiallize
                                        -- the findfile funcs
  findfilePO: findfile!findfile_func;  -- findfile relative to SYSLIBDD path
  findfileDO: findfile!findfile_func;  -- findfile relative to DEFLIBDD path
  findfilePsrc: findfile!findfile_func; -- findfile relative to PSRCLIB path
  findfileDsrc: findfile!findfile_func; -- findfile relative to DSRCLIB path
  initcheckscope: checkscope!initcheckscopeFn;  -- func to initiallize checkscope
  checkscope: checkscope!checkscopeFn;  -- file to check whether a path 
                                         -- is within the make "scope" or not
  capability: common!polymorph;
  stat: unixstat!statMtimeFunc;
  timestampInit: timestamp!timestampInitFn;
  timestamp: timestamp!tsrequestFn;
  mustMakeInit: mustMakeInitFn;
  mustMake: mustMakeFn;
  setupGetuses: getuses!setupGetusesService;
  getuses: getusesFn;
  RightString: SubstringFunc;
  LeftString: SubstringFunc;
  RightmostChar: CharPosFunc;   -- returns the position of the rightmost 
                                -- occurence of char in string

  compileProcess: getdep!ActionInitFn;
  resolveDefinition: getdep!ActionInitFn;
  nullAction: ActionFunc;
  cwd: charstring;

  environ: root!environ;
  std: stdenv!stdenv;
  unix: unix!unix;
  CLoader: CLoad!CLoadFn;

  getdepQ: GetDependenciesQ;
  getdepM: GetDependenciesIntf;

begin
   receive setupM from setupQ;

-- initialize the findfile functions
   initloadfuncs <- procedure of setupM.std.pathload("initloadfuncs");
   call initloadfuncs(setupM.unix.access, setupM.std.getCwd, 
               setupM.std.pathload, setupM.environ, findfilePO, 
               findfileDO, findfilePsrc, findfileDsrc);
-- initialize the checkScope function, which checks if a path is in 
-- MKSCOPE
   initcheckscope <- procedure of setupM.std.pathload("initcheckscope");
   cwd <- setupM.std.getCwd();
   checkscope <- initcheckscope(cwd, setupM.environ);   
   
-- get file stat function
   call setupM.CLoader(charstring#"File Modification Time", capability);
   unwrap stat from capability {init};
-- initialize the timestamp function
   timestampInit <- create of setupM.std.pathload("timestamp");
   timestamp <- timestampInit(stat, setupM.unix.access);
-- initialize the mustMake function
   mustMakeInit <- create of setupM.std.pathload("mustmake");
   mustMake <- mustMakeInit(timestamp);
-- load other functions 
   RightString <- procedure of setupM.std.pathload("rightstring");
   LeftString <- procedure of setupM.std.pathload("leftstring");
   RightmostChar <- procedure of setupM.std.pathload("rightmostchar");
   setupGetuses <- create of setupM.std.pathload("getuses");
   getuses <- setupGetuses(setupM.std.pathload, setupM.unix, 
                                      findfileDO, findfilePO);
-- load appropriate processes to serve as the "make" actions 
   if setupM.actionType = 'print'
   then  -- the make functions are just print routines
      compileProcess <- procedure of setupM.std.pathload("pprintshell");
      resolveDefinition <- procedure of setupM.std.pathload("dprintshell");
   else  -- setupM.actionType = 'compile'
      compileProcess <- procedure of setupM.std.pathload("pcomshell");
      resolveDefinition <- procedure of setupM.std.pathload("dcomshell");
   end if;
-- for .p and .d files, action is to do nothing.  
-- This is what null action does (it always returns true);
   nullAction <- create of setupM.std.pathload("nullaction");

-- copy over some parms to pass to pcom and dcom
   environ := setupM.environ;  
   std := setupM.std;
   unix := setupM.unix;
   CLoader := setupM.CLoader;

-- return the getdep function
   new getdepQ;
   connect setupM.getdepFunc to getdepQ;
   return setupM;

---------------------------------------------------------------------
----------------------- main processing loop ------------------------
---------------------------------------------------------------------

-- now wait and process requests
   while 'true' repeat
     block 
     declare
       absname: charstring;   -- name of file with path prefix
       name: charstring;      -- name of file with path prefix stripped off
       extension: charstring; -- file extension (empty, .d, .p, etc)
       fname: charstring;     -- name of file with extension but without prefix
       fullname: charstring;        
       prefix: charstring;
       dotpos: integer;
       slashpos: integer;
       uses: charstringSet;
     begin
       receive getdepM from getdepQ;
       if size of getdepM.nodename = 0 then
         exit dependencyFailure;
       end if;
-- Let getdepM.nodename be of the form "path/f.ext".  
-- Set the var fname to be "f.ext"
-- Set the var prefix to be "path"
-- Set the var name to be "f".
-- Set the var ext to be "ext".
-- Set the var absname to be "path/f".
       slashpos <- rightmostchar('/', getdepM.nodename);
       if slashpos = -1    -- then file has no path prefix = current dir
       then 
         fname := getdepM.nodename;
         prefix := cwd;
       else             -- strip off prefix
         fname <- Rightstring(getdepM.nodename, slashpos + 1);
         prefix <- LeftString(getdepM.nodename, slashpos);
       end if;
       if not exists of char in fname where (char = '.') then
         name := fname;
         new extension;
       else
         dotpos := position of char in fname where (char = '.');
         name := LeftString(fname, dotpos);
         extension := RightString(fname, dotpos + 1);
       end if;
       absname := prefix | "/" | name;

---------------------------------------------------------------------
------------------ get dependencies for this object -----------------
---------------------------------------------------------------------
-- now find the dependencies
       select 
         where (extension = "p" or extension = "d")
	   getdepM.action := nullAction;
           new getdepM.uses;
           getdepM.mustMake <- 'false';

         where (extension = "do")
           new getdepM.uses;
           if timestamp(getdepM.nodename) = -1 or checkscope(prefix)
           then  
               -- the file doesn't exist, or it exists and is in MKSCOPE.
               -- Hence insert its dependencies.
             block begin
-- Set the var fullname to be the full path name of "name.d" 
               fullname <- findfileDsrc(name | ".d", access#'read');
             on (findfile_Intf.file_Not_Found)
               call std.terminal.putString("Could not find file: "); 
               call std.terminal.putLine(name | ".d");
               exit dependencyFailure;
             end block;
-- action is to call dcom with full path name (absname)
             getdepM.action <- resolveDefinition(environ,std,unix,
                                               CLoader,absname);
             insert copy of fullname into getdepM.uses;  
             merge getuses(fullname) into getdepM.uses;
-- if this .do file doesn't exist or if its timestamp is earlier then its 
-- predecessors, then set mustMake to true.  
             getdepM.mustMake := mustMake(getdepM.nodename, getdepM.uses);
           else
              -- the file is not in MKSCOPE so don't try to remake it
	     getdepM.action := nullAction;
             getdepM.mustMake <- 'false';
           end if;

         where (extension = "po")
           new getdepM.uses;
           if timestamp(getdepM.nodename) = -1 or checkscope(prefix)
           then  
               -- the file doesn't exist, or it exists and is in MKSCOPE.
               -- Hence insert its dependencies.
             block begin
-- find full path name of name.p and insert it into getdep.uses
               fullname <- findfilePsrc(name | ".p", access#'read');
             on (findfile_Intf.file_Not_Found)


               call std.terminal.putString("Could not find file: "); 
               call std.terminal.putLine(name | ".p");
               exit dependencyFailure;
             end block;
-- action is to call pcom with full path name (absname)
             getdepM.action <- compileProcess(environ,std,unix,
                                                 CLoader,absname);
             insert copy of fullname into getdepM.uses;  
             merge getuses(fullname) into getdepM.uses;
-- if this .po file doesn't exist or if its timestamp is earlier then its 
-- predecessors, then set mustMake to true.  
             getdepM.mustMake := mustMake(getdepM.nodename, getdepM.uses);
           else
              -- the file is not in MKSCOPE so don't try to remake it
	     getdepM.action := nullAction;
             getdepM.mustMake <- 'false';
           end if;

         where (extension = "")		-- directory
           call std.terminal.putString("Can't make file without extension. File: "); 
           call std.terminal.putLine(getdepM.nodename);
           exit dependencyFailure;	
	-- for foo.X (X in {"p","po","lo"}), put foo.po into uses
        -- new getdepM.uses;
        -- getdepM.action := nullAction;
        -- getdepM.mustMake <- 'false';

         otherwise
           call std.terminal.putString("Invalid extension for file: "); 
           call std.terminal.putLine(getdepM.nodename);
           exit dependencyFailure;	
       end select;

       return getdepM;
     on exit (dependencyFailure)
       return getdepM exception dependencyFailure;
     on (getusesIntf.getusesfailure)
       return getdepM exception dependencyFailure;
     end block;
   end while;
end process
