;; Eulisp Module
;; Author: pab
;; File: link.em
;; Date: Wed Jan 29 21:49:20 1992
;;
;; Project:
;; Description: 
;;
;; Takes a compile-unit and links it to the existing system
;; I assume that the statics have been compressed.

(defmodule link
  (standard0
   list-fns
   stream

   comp-defn
   module-operators
   bci
   stop
   )
  ()

  (defun load-compiled-module (name)
    (let ((file-name (bytecode-file-name name)))
      (link-compile-unit (read-object compile-unit file-name))
      (list-length v1)))

  ((setter setter) module-value module-value-setter)

  (defun link-compile-unit (unit)
    (let ((msize (+ (list-length (compile-unit-statics unit)) 
		    (compile-unit-local-count unit)))
	  (code-vector (compile-unit-byte-codes unit)))
      (let ((module (make-module (compile-unit-name unit) msize)))
	(init-compiled-bindings module)
	(install-statics module (compile-unit-statics unit))
	(install-module-interface module unit)
	(install-code-vector code-vector
			     (mk-local-id-mker module
					       (list-length (compile-unit-statics unit)))
			     module)
	module)))

  (defun install-module-interface (module unit)
    (let ((imports (compile-unit-imports unit))
	  (exports (compile-unit-exports unit)))
      (format t "Imports")
      (mapc (lambda (x) 
	      (let ((xx (get-module (car x))))
		(if (not (null xx))
		    ;; XXX Whups, forget what name to import it as!
		    (add-module-import module (cdr x) (get-module (car x)) (cdr x) )
		  (error "No such module" clock-tick))))
	    imports)
      (format t "Exports")
      ;; Note that we forget where the hell it came from
      (mapc (lambda (x) (add-module-export module (cdr x)))
	    exports)))
      
  (defun install-statics (module statics)
    (fold (lambda (static count)
	    ((setter module-value) module count static)
	    (+ count 1))
	  statics 
	  0))

  (defun install-code-vector (code initstate module)
    ;; make it
    (let ((vect (link-vector code initstate)))
      ;; install it
      (setq v1 vect)
      (let ((initfn (add-code-vector vect 
				     (list-length vect)
				     (get-module-id (module-name module)))))
	;; return result of calling initcode
	(apply-bytefunction initfn))
      ))
  
  ;; Takes the format of the instructions, and munges them into a 
  ;; form, suitable for add-bytecodes
  ;; I should have done this before I wrote the wossnames.
  

  ;; Initialisation:
  ;; For each module, install it so that bytecodes can use it.

  (defun initialise-byteworld ()
    (format t "Starting world\n")
    (initialise-statics)
    (format t "Ret vect")
    (initialise-return-vector)
    (format t "Done"))

  ;; hand compiled code 

  (defconstant *exit-bytecodes* '(57))

  (defun initialise-return-vector ()
    (let ((ret (add-code-vector *exit-bytecodes* 
				(list-length *exit-bytecodes* )
				0)))
      (set-return-func ret)))

    
  (defconstant *all-modules* 
    '(formatted-io vectors characters semaphores module-operators
      errors others lists bit-vectors calls symbols strings 
      class-names ccc tables root classes list-operators
      arith bci sockets generics streams threads))

  (defun initialise-statics () 
    (mapc init-compiled-bindings *all-modules*))

  (defun init-compiled-bindings (x)
    (let/cc out 
	    (with-handler 
	     (lambda (c1 c2) (out ()))
	     (progn ((setter get-module-id) x
		     (install-module-statics (get-module x)))))))
  

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;; Linking
  ;;
  
  ;; module table
  (defconstant mod-count (mk-counter 0))
  (defconstant get-module-id (mk-finder))
    
  ;; Linking a code vector:
  ;; returns a list of bytecodes
  
  (defun link-vector (lst istate)
    (let ((out (make-simple-stream)))
      (convert (fold (lambda (o stream)
		       (fold (lambda (b stream)
			       (write-stream stream b))
			     (cons (car o)
				   (link-object (cdr o) istate))
			     stream))
		     lst 
		     out)
	       pair)))

  (defun link-object (obj local-mod-state)
    (cond ((null obj) nil)
	  ((eq (car obj) (the-link-handle))
	   (let ((val (if (eq (caadr obj) (the-local-handle))
			  (get-id local-mod-state (cadr obj))
			(get-non-local-id (cadr obj)))))
	     (append (int2bytes (car val))
		     (int2bytes (cadr val)))))
	  ((eq (car obj) (the-long-handle))
	   (int2bytes (cadr obj)))
	  (t obj)))

  (defun get-non-local-id (binding)
    (let ((r (list (get-module-id (car binding))
		   (module-binding-location (get-module (car binding))
					    (cdr binding)))))
      (if (or (null (car r))
	      (null (cadr r)))
	  (stop (list r binding))
	r)))
  
  (defun mk-local-id-mker (mod start)
    (let ((tab (make-table eq))
	  (count (mk-counter start))
	  (mod-id (install-module-statics mod)))
      ((setter get-module-id) (module-name mod) mod-id)
      (lambda (name)
	(let ((xx (table-ref tab name)))
	  (if (null xx)
	      (let ((c (count)))
		((setter table-ref) tab name c)
		(add-module-binding mod name c)
		(list mod-id c))
	    (list mod-id xx))))))

  (defun get-id (state id)
    (state (cdr id)))

  
  ;; making 4 bytes from integers.

  (defun int2bytes (x)
    (let ((sign (< x 0))
	  (val (abs x)))
      (let* ((v1 (/ val 256))
	     (v2 (/ v1 256))
	     (v3 (/ v2 256)))
	(list (modulo v2 256)
	      (modulo v1 256)
	      (modulo val 256)
	      (if sign 1 0)))))
	  

  ;; end module

  ;; start up
  (initialise-byteworld)
  
  
  )

