/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/


/******************************************************************
*
*  File:  torvol.c
*
*  Purpose: find volumes of bodies in toroidal domain
*           and volume gradients.
*
*/

#include "include.h"


void torvol()
{
  facet_id f_id;   /* main facet iterator */
  body_id b_id;
  body_id b0_id,b1_id;   /* facet adjacent bodies */


  /* adjust body volumes to the invariant constant for each */
  FOR_ALL_BODIES(b_id)
      set_body_volume(b_id,get_body_volconst(b_id));

  FOR_ALL_FACETS(f_id)
    {
       REAL *v[FACET_VERTS];  /* pointers to three vertex coordinates */
       facetedge_id fe;
       REAL t;    /* accumulator for this facet */
       int i;
       REAL adj[FACET_EDGES][MAXCOORD];  /* torus wrap adjustments for edge */


       /* find adjacent bodies */
       b0_id = get_facet_body(f_id);
       b1_id = get_facet_body(facet_inverse(f_id));
       if ( !valid_id(b0_id) && !valid_id(b1_id) ) continue;

       /* get basic info */
       fe = get_facet_fe(f_id);
       for ( i = 0 ; i < FACET_EDGES ; i ++ )
         {
           v[i] = get_coord(get_fe_tailv(fe));
           get_edge_adjust(get_fe_edge(fe),adj[i]); 
           fe = get_next_edge(fe);
         }

       /* basic tetrahedron */
       t = triple_prod(v[0],v[1],v[2]);

       /* torus wrap corrections */
       for ( i = 0 ; i < FACET_EDGES ; i++ )
         {
           /* two-vertex term */
           t += triple_prod(adj[(i+1)%FACET_EDGES],v[i],v[(i+1)%FACET_EDGES])/2;
           t -= triple_prod(adj[(i+2)%FACET_EDGES],v[i],v[(i+1)%FACET_EDGES])/2;

           /* one-vertex term */
           t += triple_prod(v[i],adj[(i+2)%FACET_EDGES],adj[i]);
         }

       if ( valid_id(b0_id) )
          set_body_volume(b0_id,get_body_volume(b0_id)+t/6);
       if ( valid_id(b1_id) )
          set_body_volume(b1_id,get_body_volume(b1_id)-t/6);

      }
}


/*************************************************************************
*
*  Function:  torvol_project()
*
*  Purpose:   Keeps constant volume of bodies by projecting force
*             perpendicular to gradients of body volumes.  
*             Adapted to toroidal domain.
*
*  Operation: Fills in matrix of dot products of body volume
*             gradients.  For each body i, it makes one pass
*             through all facets, storing the gradient of
*             the volume of that body in each vertex structure.
*             Then for each body j, it makes a second such pass,
*             this time dotting the gradient of body j volume
*             with the gradient of i stored there, and accumulating
*             the total in leftside[][].  Inverts matrix and solves
*             for pressures. In the case of a solidly filled torus,
*             to avoid a singular matrix, one body is omitted.
*
*
*  Input:    arrays must be allocated.
*            Number of bodies in bodycount.
*            
*/

void torvol_project(set_pressure_flag)
int set_pressure_flag;
{
  body_id bi_id;  /* identifier for body i */
  body_id bj_id;  /* identifier for body j */
  body_id b_id;
  vertex_id v_id;
  facet_id f_id;
  int i,k,n;
  REAL g[MAXCOORD],*f;
  int bodies = web.full_flag ? web.bodycount-1 : web.bodycount;
  int qfixed;
  REAL ***vg = NULL;  /* for approx curvature */
  REAL **vge = NULL;  /* for approx curvature */

  /* allocate space to hold vertex body volume gradients */
  for ( i = 0, qfixed = 0 ; i < web.quantity_count ; i++ )
      if ( get_quant(i)->attr & QFIXED ) { qfixed++; }
  if ( (web.bodycount == 0) && (qfixed == 0) )  return;
  vgrad_init(qfixed);

  leftside = dmatrix(0,web.bodycount,0,web.bodycount);
  for ( k = 0 ; k < web.bodycount ; k++ )
     leftside[k][k] = 1.0; /* for invertibility */
  rightside = vector(0,web.bodycount);
  vol_deficit = vector(0,web.bodycount);
  pressures = vector(0,web.bodycount);
  vol_restore = vector(0,web.bodycount);

  FOR_ALL_BODIES(bi_id)
    {
      i = ordinal(bi_id);

      /* first, install gradients of body i at vertices */
      if ( (get_battr(bi_id) & FIXEDVOL) == 0 ) continue;
      vol_deficit[i] = get_body_fixvol(bi_id) - get_body_volume(bi_id);

      leftside[i][i] = 0.0;

    }


  FOR_ALL_FACETS(f_id)
    { 
      REAL *v[FACET_VERTS];  /* pointers to three vertex coordinates */
      facetedge_id fe;
      int j;
      REAL adj[FACET_EDGES][MAXCOORD];  /* torus wrap adjustments for edge */
      vertex_id v_ids[FACET_VERTS];
      volgrad *vgptr;

      bi_id = get_facet_body(f_id);
      bj_id = get_facet_body(facet_inverse(f_id));
           

      /* get basic info */
      fe = get_facet_fe(f_id);
      for ( i = 0 ; i < FACET_EDGES ; i ++ )
        {
          v_ids[i] = get_fe_tailv(fe);
          v[i] = get_coord(get_fe_tailv(fe));
          get_edge_adjust(get_fe_edge(fe),adj[i]); 
          fe = get_next_edge(fe);
        }

      for ( i = 0 ; i < FACET_EDGES ; i++ )
        {
          int m;
          REAL ga[MAXCOORD],gb[MAXCOORD],gc[MAXCOORD],gd[MAXCOORD],
               ge[MAXCOORD],gf[MAXCOORD]; /* gradient parts */
          j = (i+1)%FACET_EDGES;
          k = (i+2)%FACET_EDGES;

          /* basic tetrahedron */
           cross_prod(v[j],v[k],ga);

          /* torus wrap corrections */
          /* two-vertex term */
          cross_prod(adj[j],v[j],gb);  /* - */
          cross_prod(adj[i],v[k],gc);  /* + */
          cross_prod(adj[k],v[j],gd);  /* + */
          cross_prod(adj[j],v[k],ge);  /* - */

          /* one-vertex term */
          cross_prod(adj[k],adj[i],gf);

          /* add parts to existing gradient */
          for ( m = 0 ; m < web.sdim ; m++ )
            g[m] = (ga[m] + (-gb[m]+gc[m]+gd[m]-ge[m])/2 + gf[m])/6;

          if ( valid_id(bi_id) && (get_battr(bi_id) & FIXEDVOL) )
           { 
             vgptr = get_bv_new_vgrad(bi_id,v_ids[i]);
             for ( n = 0 ; n < web.sdim ; n++ )
               vgptr->grad[n] += g[n];
           }

          if ( valid_id(bj_id) && (get_battr(bj_id) & FIXEDVOL) )
           { 
             vgptr = get_bv_new_vgrad(bj_id,v_ids[i]);
             for ( n = 0 ; n < web.sdim ; n++ )
               vgptr->grad[n] -= g[n];
           }
       }


    }  

  /* set up body normals */
  FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptri;
      for ( vgptri = get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
        { 
          for ( i = 0 ; i < web.sdim ; i++ )
            vgptri->normal[i] = vgptri->grad[i];   
        }
    }

  /* do area normalization if wanted */
  if ( web.area_norm_flag )
    {
      FOR_ALL_VERTICES(v_id)
        {
          REAL area = get_vertex_star(v_id)/star_fraction;
          volgrad *vgptri;
          for ( vgptri = get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
            { 
              for ( i = 0 ; i < web.sdim ; i++ )
                vgptri->normal[i] /= area;   
            }
        }
    }

  /* generate sides of matrix equation term */
  if ( !approx_curve_flag )
  FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptri,*vgptrj;

      if ( get_vattr(v_id) & FIXED ) continue;
      f = get_force(v_id);

      /* project gradients tangent to boundary */
      if ( get_vattr(v_id) & BOUNDARY )
        {
          REAL **a;
          struct boundary *bdry = get_boundary(v_id);
          a = dmatrix(0,2,0,2);
          b_proj(bdry,get_param(v_id),a,TANGPROJ);
          vgptri = get_vertex_vgrad(v_id);
          while ( vgptri )
            { REAL tmp[MAXCOORD];
              int m;
              matvec_mul(a,vgptri->grad,tmp,web.sdim,web.sdim);
              for ( m  = 0 ; m < web.sdim ; m++ ) vgptri->grad[m] = tmp[m];
              vgptri = vgptri->chain;
            }
          free_matrix(a);
        }
      vgptri = get_vertex_vgrad(v_id);

      while ( vgptri )
        { 
          int bi,bj;
          bi = ordinal(vgptri->b_id);
          rightside[bi] += dot(f,vgptri->grad,web.dimension);
          leftside[bi][bi] += dot(vgptri->grad,vgptri->normal,web.dimension);
          for ( vgptrj = vgptri->chain ; vgptrj ; vgptrj = vgptrj->chain )
            { REAL tmp = dot(vgptri->grad,vgptrj->normal,web.dimension);
              bj = ordinal(vgptrj->b_id);
              leftside[bi][bj] += tmp;
              leftside[bj][bi] += tmp;
            }
          vgptri = vgptri->chain;
         }
     }

  if ( approx_curve_flag )
    { int N = web.sdim*web.skel[VERTEX].max_ord;
      REAL *tempg = (REAL *)temp_calloc(N,sizeof(REAL));
      int j;
      double tmp;
      int bi,bj;

      vge = dmatrix(0,bodies+1,0,N);
      /* load volume gradients and calculate right side */
      FOR_ALL_VERTICES(v_id)
       {
         volgrad *vgptri,*vgptrj;
         ATTR attr = get_vattr(v_id);
	 int ord = ordinal(v_id);

         if ( attr & FIXED ) continue;

         f = get_force(v_id);
         for ( vgptri=get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
           {
             if (valid_id(vgptri->b_id))
               { if ( !(get_battr(vgptri->b_id)&FIXEDVOL) ) continue;
                 bi = ordinal(vgptri->b_id);
               }
             else bi = web.bodycount + vgptri->b_id; /* for quantities */
	     for ( j = 0 ; j < web.sdim ; j++ )
	       vge[bi][web.sdim*ord+j] = vgptri->grad[j];
             rightside[bi] += dot(f,vgptri->grad,web.sdim);
           }
        }

     /* convert gradients to vectors  and get dot products */
     for ( bi = 0 ; bi <= bodies ; bi++ )
       { memcpy((char*)tempg,(char*)vge[bi],N*sizeof(REAL));
         mobility_mult(vge[bi]);
	 leftside[bi][bi] += dot(tempg,vge[bi],N);  /* self product */ 
	 for ( bj = bi+1 ; bj <= bodies ; bj++ ) /* other products */
	   { tmp = dot(vge[bi],vge[bj],N);
	     leftside[bi][bj] += tmp;
	     leftside[bj][bi] += tmp;
           }
        }

    }  /* end approx_curve_flag */

  /* solve for coefficients */
  mat_inv(leftside,bodies);
  matvec_mul(leftside,rightside,pressures,bodies,bodies);

  /* install pressures into body structures */
  if ( set_pressure_flag == SET_PRESSURE )
    FOR_ALL_BODIES(b_id)
      set_body_pressure(b_id,-pressures[ordinal(b_id)]);
  
  /* solve for volume restoration coefficients */
  matvec_mul(leftside,vol_deficit,vol_restore,bodies,bodies);
  
  /* subtract multiples of volume gradients from force */
  /* and combine multiples of gradients for restoring motion */

  /* fix up forces and set restoring forces */
  FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptr;
      REAL *r;
      int bi;

      if ( get_vattr(v_id) & FIXED ) continue;
      r = get_restore(v_id);
      r[0] = r[1] = r[2] = 0.0;
      f = get_force(v_id);
      for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
        { int ord = ordinal(v_id);
          bi = ordinal(vgptr->b_id);
          for ( k = 0 ; k < web.dimension ; k++ )
            { 
               if ( approx_curve_flag )
		 {
                   r[k] += vol_restore[bi]*vge[bi][web.sdim*ord+k];
                   f[k] -= pressures[bi]*vge[bi][web.sdim*ord +k];
		 }
	       else
		{
                  r[k] += vol_restore[bi]*vgptr->normal[k];
                  f[k] -= pressures[bi]*vgptr->normal[k];
		}
            }
        }

    }

  vgrad_end();
  free_matrix(leftside);
  free((char *)rightside);
  free((char *)vol_deficit);
  free((char *)pressures);

}

