/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/


/********************************************************************
*
*  File: storage.h
*
*  Purpose:  Header file defining details of storage implementation.
*            All machine-dependent gory details should be here
*            (for inclusion in other source files that need to know)
*            or in storage.c (purely private details).
*
*     This version has element ids typed as longs.
*     Also implements id as offset from element base.
*     All elements of same type in one memory block, extended 
*      as needed.  To be used on 32 bit machines.
*/

#define NUMELEMENTS 5

/* these values used for identification and as index into skeleton info */
#define  VERTEX  0
#define  EDGE    1
#define  FACET   2
#define  BODY    3
#define  FACETEDGE    4 

/*****************************************************************
*
*  Universal element identifier.  Don't want to use straight
*  pointer since base changes when realloced.
*  (not actually used; just for documentation)
*/

typedef struct xelement_id {
    unsigned int  type : 6;   /* see enum below */
    unsigned int  valid: 1;   /* valid id bit */
    unsigned int  sign : 1;   /* set for reverse orientation */
    unsigned int  offset: 24;   /* offset from block start */
    } xelement_id;


/* masks for fields */
#define USERFLAG_1 0x80000000
#define USERFLAG_2 0x40000000
#define USERFLAG_3 0x20000000
#define TYPEMASK   0x1C000000
#define VALIDMASK  0x02000000
#define SIGNMASK   0x01000000
#define OFFSETMASK 0x00FFFFFF

/* shifts for fields */
#define TYPESHIFT  26
#define VALIDSHIFT 25
#define SIGNSHIFT  24

#define NULLID 0L 

/* to get type of an element */
#define id_type(id)  ((int)(((id)&TYPEMASK)>>TYPESHIFT))

/* to give switched orientation of first if that of second is inverted */
#define same_sign(id1,id2)    ((id1) ^ ((id2) & SIGNMASK))

/* number of elements to allocate memory for at one time */
#define BATCHSIZE 100

/* outside storage.*, element_id structure is not visible; acts like long */    
typedef long
     element_id, vertex_id, edge_id, facet_id, body_id, facetedge_id; 

/* macros for getting structure pointer from id */
#define vptr(v_id) ((struct vertex *)(vbase + ((v_id)&OFFSETMASK)))
#define eptr(e_id) ((struct edge   *)(ebase + ((e_id)&OFFSETMASK)))
#define fptr(f_id) ((struct facet  *)(fbase + ((f_id)&OFFSETMASK)))
#define bptr(b_id) ((struct body   *)(bbase + ((b_id)&OFFSETMASK)))
#define feptr(fe_id) ((struct facetedge *)(febase + ((fe_id)&OFFSETMASK)))

/* id attr bits */
#define VALID_BIT   0x0001
#define INVERSE_BIT 0x0001

#define edge_inverse(id)  inverse_id(id)
#define facet_inverse(id)  inverse_id(id)
#define fe_inverse(id)  inverse_id(id)
#define invert(id)     ((id) ^= SIGNMASK)
#define equal_id(a,b)  ((a)==(b))
#define equal_element(a,b)  (((a)|SIGNMASK) == ((b)|SIGNMASK))
#define valid_id(id)   ((id)&VALIDMASK)
#define inverted(id)   ((id)&SIGNMASK)
#define inverse_id(id) ((id) ^ SIGNMASK)
typedef int ORDTYPE;                /* element numbering type */

/* base structure for a skeleton of a particular dimension */
struct skeleton {
    int             type;   /* type of element, see defines above */
    char           *base;   /* to list of structures     */
    long             maxcount; /* elements allocated          */
    element_id      free;   /* start of free list        */
    element_id      used;   /* start of in-use elements  */
    element_id      last;   /* end of in-use chain, for adding on */
    element_id      discard; /* start of discard list */
    long            count;  /* number active             */
    element_id      current; /* current element in generation */
    ORDTYPE         max_ord; /* highest ordinal */
  } ;

/* individual dimension block pointer arrays, handy for fast addressing */
extern char *vbase;
extern char *ebase;
extern char *fbase;
extern char *bbase;
extern char *febase;

/* individual dimension block pointer arrays, handy for fast addressing */
extern struct vertex *vvbase;
extern struct edge *eebase;
extern struct facet *ffbase;
extern struct body *bbbase;
extern struct facetedge *fefebase;

/* unified block references, indexed by element type */
extern char *base[NUMELEMENTS];
