/* 
 *	FIG : Facility for Interactive Generation of figures
 *
 *	Copyright (c) 1988 by Supoj Sutanthavibul (supoj@sally.UTEXAS.EDU)
 *	March 1988.
 *
 *	%W%	%G%
 *
 *	Written by baron@cs.ucla.edu
*/
#include <suntool/sunview.h>
#include <suntool/panel.h>
#include <stdio.h>
#include "object.h"
#include "paintop.h"
#include "func.h"

#define			TOLERANCE	7
#define			LEFT		1
#define			RIGHT		2
#define			TOP		3
#define			BOTTOM		4
#define			LR_CENTER	5
#define			TB_CENTER	6

extern F_compound	objects, saved_objects, object_tails;

extern struct pixwin	*canvas_pixwin;
extern int		canvas_swfd;
extern struct cursor	null_cursor;
extern struct cursor	pick15_cursor, wait_cursor, crosshair_cursor;
extern int		foreground_color, background_color;

int			align_selected();
int			init_align();
int			align_done();
int			align_cancel();

extern			(*canvas_kbd_proc)();
extern			(*canvas_locmove_proc)();
extern			(*canvas_leftbut_proc)();
extern			(*canvas_middlebut_proc)();
extern			(*canvas_rightbut_proc)();
extern F_compound	*compound_search();
extern			null_proc();
extern			set_popupmenu();
extern Frame		base_frame;

static	F_compound	*cc;
static	int		llx, lly, urx, ury;
static	int		xcmin, ycmin, xcmax, ycmax;
static	int		dx, dy;

static	Frame		align_frame;
static	Panel		align_panel;
static	Panel_item	left_align_panel;
static	Panel_item	right_align_panel;
static	Panel_item	top_align_panel;
static	Panel_item	bottom_align_panel;
static	Panel_item	tb_center_align_panel;
static	Panel_item	lr_center_align_panel;

align_selected()
{
	canvas_kbd_proc = null_proc;
	canvas_locmove_proc = null_proc;
	canvas_leftbut_proc = init_align;
	canvas_middlebut_proc = null_proc;
	canvas_rightbut_proc = set_popupmenu;
	set_cursor(&pick15_cursor);
	reset_action_on();
}

init_align(x, y, xm, ym)
int	x, y, xm, ym;
{
	int		px, py;

	win_setcursor(canvas_swfd, &wait_cursor);
	if ((cc = compound_search(xm, ym, TOLERANCE, &px, &py)) != NULL)
	    align();
	else {
		reset_cursor();
		return;
	}
	set_action_on();
	property_selected();
	erase_pointmarker();
	(void)window_fit(align_panel);
	(void)window_fit(align_frame);
	(void)window_set(align_frame, WIN_SHOW, TRUE, 0);
	reset_cursor();
}

align()
{
	/* 
	 * Set up the align popup panel.
	 * Depending on what is selected in the panel, call other 
	 * support routines.
	 */

	align_panel = window_create(align_frame, PANEL, 0);

	(void)panel_create_item(align_panel, PANEL_BUTTON,
		PANEL_LABEL_X,	ATTR_COL(30),
		PANEL_LABEL_Y,	ATTR_ROW(0),
		PANEL_LABEL_IMAGE,	panel_button_image(align_panel,
					   "Done", 6, (Pixfont *)0),
		PANEL_NOTIFY_PROC,	align_done,
		0);
	(void)panel_create_item(align_panel, PANEL_BUTTON, 
		PANEL_LABEL_X,		ATTR_COL(50),
		PANEL_LABEL_Y,		ATTR_ROW(0),
		PANEL_LABEL_IMAGE,	panel_button_image(align_panel,
					  "Cancel", 6, (Pixfont *)0),
		PANEL_NOTIFY_PROC,	align_cancel,
		0);

	left_align_panel = panel_create_item(align_panel, PANEL_TOGGLE, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(2),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(2),
	      PANEL_LABEL_STRING,	"Left side:",
	      PANEL_VALUE,		0,
	      PANEL_FEEDBACK,		PANEL_MARKED,
	      0);
	right_align_panel = panel_create_item(align_panel, PANEL_TOGGLE, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(3),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(3),
	      PANEL_LABEL_STRING,	"Right side:",
	      PANEL_VALUE,		0,
	      PANEL_FEEDBACK,		PANEL_MARKED,
	      0);
	top_align_panel = panel_create_item(align_panel, PANEL_TOGGLE, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(4),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(4),
	      PANEL_LABEL_STRING,	"Top:",
	      PANEL_VALUE,		0,
	      PANEL_FEEDBACK,		PANEL_MARKED,
	      0);
	bottom_align_panel = panel_create_item(align_panel, PANEL_TOGGLE, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(5),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(5),
	      PANEL_LABEL_STRING,	"Bottom:",
	      PANEL_VALUE,		0,
	      PANEL_FEEDBACK,		PANEL_MARKED,
	      0);
	tb_center_align_panel = panel_create_item(align_panel, PANEL_TOGGLE, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(6),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(6),
	      PANEL_LABEL_STRING,	"Top/bottom center:",
	      PANEL_VALUE,		0,
	      PANEL_FEEDBACK,		PANEL_MARKED,
	      0);
	lr_center_align_panel = panel_create_item(align_panel, PANEL_TOGGLE, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(7),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(7),
	      PANEL_LABEL_STRING,	"Left/right center:",
	      PANEL_VALUE,		0,
	      PANEL_FEEDBACK,		PANEL_MARKED,
	      0);
}

make_align_frame()
{
	align_frame = window_create(base_frame, FRAME,
				    WIN_SHOW,		FALSE,
				    FRAME_LABEL,	"Align",
				    FRAME_SHOW_LABEL,	TRUE,
				    FRAME_DONE_PROC,	align_done,
				    0);
}

align_done(panel_local, item, event)
Panel		panel_local;
Panel_item	item;
Event		*event;
{
	erase_compound(cc);
	clean_up();
	set_latestcompound(copy_compound(cc));

	/*
	 * Get the align values and perform the necessary alignment.
	 */

	if ((int)panel_get_value(left_align_panel)) {
		do_align(LEFT);
	}

	if ((int)panel_get_value(right_align_panel)) {
		do_align(RIGHT);
	}

	if ((int)panel_get_value(top_align_panel)) {
		do_align(TOP);
	}

	if ((int)panel_get_value(bottom_align_panel)) {
		do_align(BOTTOM);
	}

	if ((int)panel_get_value(tb_center_align_panel)) {
		do_align(TB_CENTER);
	}

	if ((int)panel_get_value(lr_center_align_panel)) {
		do_align(LR_CENTER);
	}

	/*
	 * Get rid of the popup window.
	 */
	(void)window_set(align_frame, WIN_SHOW, FALSE, 0);
	(void)window_destroy(align_panel);
	draw_compoundbox(cc, ERASE);
	/* 
	 * recompute the compound's bounding box
	 */
	compound_bound(cc, &cc->nwcorner.x, &cc->nwcorner.y, &cc->secorner.x,
		       &cc->secorner.y);
	draw_compound(cc);
	draw_compoundbox(cc, PAINT);
	saved_objects.compounds->next = cc; /* so we can undo later */
	set_action_object(F_CHANGE, O_COMPOUND);
	align_selected();
}

do_align(where)
int	where;
{
	compound_bound(cc, &xcmin, &ycmin, &xcmax, &ycmax);

	align_ellipse(where);
	align_arc(where);
	align_line(where);
	align_spline(where);
	align_compound(where);
	align_text(where);
	set_modifiedflag();
}

align_ellipse(where)
int	where;
{
	F_ellipse	*e;

	for (e = cc->ellipses; e != NULL; e = e->next) {
		ellipse_bound(e, &llx, &lly, &urx, &ury);
		get_dx_dy(where);
		translate_ellipse(e, dx, dy);
	}
}

align_arc(where)
int	where;
{
	F_arc	*a;

	for (a = cc->arcs; a != NULL; a = a->next) {
		arc_bound(a, &llx, &lly, &urx, &ury);
		get_dx_dy(where);
		translate_arc(a, dx, dy);
	}
}

align_line(where)
int	where;
{
	F_line	*l;

	for (l = cc->lines; l != NULL; l = l->next) {
		line_bound(l, &llx, &lly, &urx, &ury);
		get_dx_dy(where);
		translate_line(l, dx, dy);
	}
}

align_spline(where)
int	where;
{
	F_spline	*s;

	for (s = cc->splines; s != NULL; s = s->next) {
		spline_bound(s, &llx, &lly, &urx, &ury);
		get_dx_dy(where);
		translate_spline(s, dx, dy);
	}
}

align_compound(where)
int	where;
{
	F_compound	*c;

	for (c = cc->compounds; c != NULL; c = c->next) {
		compound_bound(c, &llx, &lly, &urx, &ury);
		get_dx_dy(where);
		translate_compound(c, dx, dy);
	}
}

align_text(where)
int	where;
{
	F_text	*t;

	for (t = cc->texts; t != NULL; t = t->next) {
		text_bound(t, &llx, &lly, &urx, &ury);
		get_dx_dy(where);
		translate_text(t, dx, dy);
	}
}

get_dx_dy(where)
int	where;
{
	switch(where) {
	case LEFT:
		dx = xcmin - llx;
		dy = 0;
		break;
	case RIGHT:
		dx = xcmax - urx;
		dy = 0;
		break;
	case TOP:
		dy = ycmin - lly;
		dx = 0;
		break;
	case BOTTOM:
		dy = ycmax - ury;
		dx = 0;
		break;
	case LR_CENTER:
		dx = (xcmin-llx) + (abs(xcmin-llx)+abs(xcmax-urx))/2;
		dy = 0;
		break;
	case TB_CENTER:
		dy = (ycmin-lly) + (abs(ycmin-lly)+abs(ycmax-ury))/2;
		dx = 0;
		break;
	}
}

static property_proc()
{
	put_msg("Close the ALIGN window first!");
	}

static property_selected()
{
	canvas_kbd_proc = null_proc;
	canvas_locmove_proc = null_proc;
	canvas_leftbut_proc = property_proc;
	canvas_middlebut_proc = property_proc;
	canvas_rightbut_proc = property_proc;
	}

align_cancel(panel_local, item, event)
Panel		panel_local;
Panel_item	item;
Event		*event;
{
	(void)window_set(align_frame, WIN_SHOW, FALSE, 0);
	(void)window_destroy(align_panel);
	align_selected();
}

