/****************************************************************************
 *
 * Copyright (c) 1996,1999 John Forkosh Associates, Inc.  All rights reserved.
 * --------------------------------------------------------------------------
 *
 * Function:	texlist ( infp, outfp )
 *
 * Purpose:   o	Produces program listing of all source files names
 *		in infp, on the device named by outfp.  Output file is
 *		ready for LaTeX2e processing to produce .dvi file.
 *
 * Arguments:	infp (I)	pointer to FILE device from which
 *				names of files to be listed will be read.
 *				Defaults to stdin if passed as NULL.
 *		outfp (I)	pointer to FILE device on which
 *				listing (of all files) will be produced.
 *				Defaults to stdout if passed as NULL.
 *
 * Returns:	(int)		number of files listed,
 *				or 0 for any error.
 *
 * Source:	texlist.c
 *
 * --------------------------------------------------------------------------
 * Notes:     o	To build the executable from source, under Unix, just enter
 *			cc -DTESTDRIVE texlist.c -o texlist
 *		Note that the TESTDRIVE symbol causes a one-line main()
 *		to be compiled along with the "main" function texlist().
 *		(This permits texlist() to be incorporated within another
 *		program.)
 *	      o	The one-line main() calls texlist(NULL,NULL), which
 *		then runs as a Unix filter, i.e., filenames are read
 *		from stdin until an empty line (or eof) is found, and
 *		the output LaTeX file is written to stdout.  As usual,
 *		the user can pipe either or both as desired.  For example,
 *		under (most versions of) Unix,
 *			ls *.c > tocfile.txt
 *		will generate toclist.txt containing the names of all
 *		C source files in the directory, one filename per line.
 *		Then issue
 *			texlist <tocfile.txt >listing.tex
 *			latex texlist
 *		to generate the .dvi file.
 *		     If running, e.g., under VMS, you may assign sys$input
 *		and sys$output, as desired, to achieve the same effect.
 *		(And use   dir/col=1/out=tocfile.txt *.c   rather than ls.)
 *	      o	A Table of Contents is generated as the last page (or pages)
 *		of the LaTeX file (once all the page numbers are known).
 *		After printing, you can place these page(s) on top.
 *		(The toc is suppressed if you only print one file.)
 *	      o	A built-in version of banner.c, separately copyrighted,
 *		is included below.
 *	      o	Filenames read from stdin (or piped file) can be
 *		automatically lowercased by compiling with -DLOWER switch.
 *		Similarly, banners can be suppressed by compiling
 *		with -DNOBANNER switch.
 *
 * --------------------------------------------------------------------------
 * Revision History:
 * 01/06/92	J.Forkosh	Installation (produced ascii listng only).
 * 04/26/96	J.Forkosh	LaTeX enahancement.
 * 01/29/99	J.Forkosh	Various enhancements prior to copyright
 *				registration, e.g., embedded a copy of
 *				banner(), see separate copyright notice below.
 *
 ***************************************************************************/

/* --------------------------------------------------------------------------
Headers and parameters
-------------------------------------------------------------------------- */
/* --- standard headers --- */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/* --- lowercase filenames (cc with -DNOLOWER if lowercase not wanted) --- */
#ifdef	LOWER
#define	slower(p) {char *q=p; while(*q!='\000'){*q=tolower(*q); q++;}}
#else
#define	slower(p) /* (p) */
#endif
/* --- banner info  (cc with -DNOBANNER to suppress banners) --- */
#ifdef	NOBANNER
#define dobanner(fp,string) /* stub out banner() calls */
#define BANRLNS 0
#else
#define dobanner(fp,string) banner(fp,string)
#define	BANRLNS 10			/* height of banner */
#endif
#define	MAXBANR 8			/* max chars for banner */
/* --- additional adjustable parameters --- */
#define	NUMCOLS 90			/* display width of page */
#define	MAXCOLS 255			/* max width of page */
#define	MAXROWS 57			/* max length of page */
#define	TABSZ 8				/* #cols per tab */
#define	MAXFILES 512			/* max files for table of contents */

/* --------------------------------------------------------------------------
Strings for LaTeX header, preamble, etc.
-------------------------------------------------------------------------- */
/* --- stars, stripes, lonestar are three frequently-used strings --- */
static	char	stars[]          =	/* comment line with asterisks... */
"%**************************************************************************";
static	char	stripes[]        =	/* comment line with dashes... */
"%* ------------------------------------------------------------------------";
static	char	lonestar[]       = "%*"; /* comment line for layout */
/* --- header info strings --- */
static	char	filenamehdr[128] = "%* File:      noname";
static	char	purposehdr[128]  =
	      "%* Purpose:   LaTeX version of program listing files";
static	char	creationhdr[128] = "%* mm/dd/yy   J.Forkosh      Creation.";
static	char	ltx_title[128]   = "\\title{your_title_here}";
static	char	ltx_author[128]  = "\\author{your_name_here}";
static	char	clearpage[128]   = "\\clearpage";
static	char	begverbatim[128] = "\\begin{verbatim}";
static	char	endverbatim[128] = "\\end{verbatim}";
static	char	copyright[]      =	/* copyright string... */
"%* Copyright (c) 1999, John Forkosh Associates, Inc.  All rights reserved.";

/* --------------------------------------------------------------------------
List of document header, preamble, and trailer strings
-------------------------------------------------------------------------- */
static	char	*ltx_header[] = {
  /* --- same header for everyone --- */
  stars, lonestar, copyright, stripes, lonestar, filenamehdr, lonestar,
  purposehdr, lonestar, stripes, "%* Revision History:", creationhdr,
  lonestar, stars, "", stripes, "% Preamble Settings", stripes,
  NULL } ; /* --- end-of-tex_header[] --- */
static	char	*ltx_preamble[] = {	/* LaTeX Preamble */
  "% --- page formatting ---",
  "\\documentclass[10pt]{article}          %type=10pt, LaTeX2e article style",
  "\\setlength{\\oddsidemargin}{0.0in}      %left margin relative to 1inch",
  "\\setlength{\\textwidth}{7.0in}          %right margin = left + textwidth",
  "\\setlength{\\topmargin}{-0.75in}        %top margin relative to 1inch",
  "\\setlength{\\textheight}{10.0in}        %top margin + textheight",
  "\\setlength{\\parskip}{0.0in}            %vert space between paragraphs",
  "", stripes, "%* Document settings", stripes,
  "\\begin{document}",
  "\\pagestyle{empty}                      %no page headers or footers",
  "\\setcounter{page}{1}                   %first page number of document",
  "\\tolerance=10000                       %prevent overfull hboxes",
  "\\newcommand{\\ulsection}[2]{\\noindent{%",
  "\\bf\\it\\vspace*{-0.04in}\\hrulefill\\\\%",
  "\\underline{\\hbox to\\hsize{#1\\hfill #2}}\\vspace{0.04in}}\\\\}",
  "% --- title ---",
  ltx_title, ltx_author,
  "\\date{ }",
  "%%%\\maketitle                          %no title page for listing",
  "", stripes, "%* Text of Document", stripes,
  NULL } ; /* --- end-of-ltx_preamble[] --- */
/* --- LaTeX document trailer commands --- */
static	char	endoffile[] =		/* comment line at end-of-file */
"%* ---------------------------- END-OF-FILE -------------------------------";
static	char	*ltx_trailer[] = { "\\end{document}", endoffile, NULL };
/* --- end-of-global data --- */


/* --- entry point --- */
int	texlist ( infp, outfp )
FILE	*infp, *outfp;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
FILE	*listfp, *fopen();		/* pointer to filename being listed */
int	ifile=0, nfiles=0;		/* #files listed returned to caller */
int	nlines=0, pglines=0;		/* #lines in current file, page */
int	npages=0, totpages=0;		/* #pages in current file, all files*/
static	char inbuff[256], outbuff[256];	/* ascii buffer for in,out files */
static	char listbf[256], banstr[MAXBANR+1]; /* buffer for listfile, banner */
static	char *fnames[MAXFILES]; /*char *malloc();*/ /* table of cont. names */
static	int   fpages[MAXFILES];		/* 1st page of corresponding name */
char	*buffptr;			/* ptr to delimiter char in buffer */
char	*detab();			/* det-tab output line */
int	fputps();			/* print array of lines */

/* --------------------------------------------------------------------------
Initialization
-------------------------------------------------------------------------- */
if ( infp == (FILE *)NULL )		/* user passed null input file ptr */
  infp = stdin;				/* default it to stdin */
if ( outfp == (FILE *)NULL )		/* user passed null output file ptr */
  outfp = stdout;			/* default it to stdout */
fputps(outfp,ltx_header);		/* print LaTeX file header */
fputps(outfp,ltx_preamble);		/* and print LaTeX preamble */

/* --------------------------------------------------------------------------
Read filenames and list corresponding files
-------------------------------------------------------------------------- */
/* --- read lines from infp containing filenames to be listed --- */
while ( fgets(inbuff,255,infp) != (char *)NULL ) /* still got files to list */
  {
  if ( (buffptr=strchr(inbuff,'\n'))	/* look for terminating newline */
  !=   (char *)NULL ) *buffptr = '\000'; /* and replace it with null */
  if ( strlen(inbuff) < 1 ) break;	/* blank line terminates program too*/
  if ( /* 1 || */ nfiles>0 )		/* page eject between listings */
    fprintf(outfp,"%s\n",clearpage);
  nlines = npages = pglines = 0;  totpages++; /* reset line, page counts */
  fprintf(outfp,"%s\n",begverbatim);	/* flip LaTeX into verbatim mode */
  memcpy(banstr,inbuff,MAXBANR);	/* copy filename to banner string */
  banstr[MAXBANR] = '\000';		/* and null-terminate it */
  dobanner(outfp,banstr); pglines=BANRLNS; /* print banner; set line count*/
  fprintf(outfp,"\n");   pglines++;	/* add a blank line after banner */  
  slower(inbuff);			/* lowercase filename */
  if ( (listfp = fopen(inbuff,"r"))	/* open file to be listed for read */
  ==   (FILE *)NULL )			/* failed to open list file */
    { fprintf(outfp,"Can't open %s for read\n\n\n",inbuff); /* display err */
      continue; }			/* and try next file */
  npages = 1;				/* set first page */
  if ( (fnames[nfiles] = malloc(strlen(inbuff)+1)) /*alloc for filename+null*/
  !=   (char *)NULL ) strcpy(fnames[nfiles],inbuff); /* and copy filename */
  while ( (buffptr=strchr(inbuff,'_')) != (char *)NULL )
    *buffptr = '-';
  while ( (buffptr=strchr(inbuff,'$')) != (char *)NULL )
    *buffptr = '-';
  fpages[nfiles] = totpages;		/* starts on this page */
  nfiles++;				/* count another file listed */
  /* --- list the file --- */
  while ( fgets(listbf,255,listfp) != (char *)NULL ) /* got lines to list */
    {
    if ( pglines >= MAXROWS )		/* need a page break */
      {	npages++; totpages++; pglines=0; /* reset page counters */
        fprintf(outfp,"%s\n",endverbatim); /* out of verbatim for pagebreak */
        fprintf(outfp,"%s\n",clearpage); /* and issue pagebreak */
	fprintf(outfp,"\\ulsection{%s(%d)}{Page\\#%d(%d)}\n",
	inbuff,nfiles,npages,totpages);
        fprintf(outfp,"%s\n",begverbatim); /* re-enter verbatim mode */
	pglines=3; }
    if ( (buffptr=strchr(listbf,'\n'))	/* look for terminating newline */
    !=   (char *)NULL ) *buffptr = '\000'; /* and replace it with null */
    nlines++;  pglines++;		/* bump line counts */
    sprintf(outbuff,"%3d: %s",nlines,detab(TABSZ,listbf)); /* format output */
    outbuff[MAXCOLS] = '\000';		/* and null-terminate it */
    fprintf(outfp,"%s\n",outbuff);	/* print output line */
    } /* --- end-of-while(fgets(listfp)!=NULL) --- */
  fclose(listfp);			/* done listing this file */
  fprintf(outfp,"%s\n",endverbatim);	/* flip LaTeX out of verbatim mode */
  } /* --- end-of-while(fgets(infp)!=NULL) --- */

/* --------------------------------------------------------------------------
Print table of contents at end of listings
-------------------------------------------------------------------------- */
if ( nfiles > 1 )			/* unnecessary for 1 file */
  {
  fprintf(outfp,"%s\n",clearpage);	/* page eject for toc */
  fprintf(outfp,"%s\n",begverbatim);	/* flip LaTeX into verbatim mode */
  npages = 1;  pglines = 0;		/* reset line, page counts */
  dobanner(outfp,">CONTENTS");pglines=BANRLNS;/*print banner; set line count*/
  fprintf(outfp,"\n");   pglines++;	/* add a blank line after banner */  
  /* --- list the table of contents --- */
  for ( ifile=0; ifile<nfiles; ifile++ )	/* for each file listed */
    {
    if ( pglines >= MAXROWS )		/* need a page break */
      {	npages++; pglines=0;		/* reset page counters */
        fprintf(outfp,"%s\n",endverbatim); /* out of verbatim for pagebreak */
        fprintf(outfp,"%s\n",clearpage); /* and issue pagebreak */
	fprintf(outfp,"\\ulsection{Table of Contents}{Page\\#%d}\n",npages);
        fprintf(outfp,"%s\n",begverbatim); /* re-enter verbatim mode */
	pglines=3; }
    if ( fnames[ifile] == (char *)NULL ) continue; /* no name to print */
    pglines++;				/* bump line count */
    sprintf(outbuff,"%4d: %s ",ifile+1,fnames[ifile]); /* format output */
    memset(outbuff+strlen(outbuff),'.',NUMCOLS-strlen(outbuff)); /* ... */
    sprintf(outbuff+NUMCOLS-12,"%4d",fpages[ifile]); /* page number */
    outbuff[NUMCOLS] = '\000';		/* and null-terminate it */
    fprintf(outfp,"%s\n",outbuff);	/* print output line */
    } /* --- end-of-for(ifile) --- */
  fprintf(outfp,"%s\n",endverbatim);	/* flip LaTeX out of verbatim mode */
  } /* --- end-of-if(nfiles>3) --- */

/* --------------------------------------------------------------------------
End-of-Job
-------------------------------------------------------------------------- */
end_of_job:
  fputps(outfp,ltx_trailer);		/* print LaTeX trailer */
  return ( nfiles );			/* back to caller with total #lines */
} /* --- end-of-function texlist() --- */


/****************************************************************************
 * Function:	detab ( tabsz, line )
 * Purpose:   o	Changes tabs to spaces in line.
 * --------------------------------------------------------------------------
 * Arguments:	tabsz (I)	int containing number of spaces per tab.
 *		line (I)	address of char string containing line
 *				to be de-tabbed.
 * --------------------------------------------------------------------------
 * Returns:	(char *)	pointer to de-tabbed version of line,
 *				or NULL for any error.
 * --------------------------------------------------------------------------
 * Notes:     o	Successive calls to detab() re-use returned buffer.
 ***************************************************************************/
/* --- entry point --- */
char	*detab ( tabsz, line )
int	tabsz;
char	*line;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
static	char outbuff[256];		/* buffer returned to caller */
char	*outptr = (char *)NULL;		/* ptr to outbuff, or null, returned*/
int	index=0, outdex=0;		/* index to line and to outbuff */
int	tabcol = 0;			/* column for current tab */
char	linech = ' ';			/* current char from line */
/* --------------------------------------------------------------------------
Initialization
-------------------------------------------------------------------------- */
memset(outbuff,' ',256);		/* blank-fill output buffer */
outbuff[255] = '\000';			/* make sure it's null-terminated */
/* --------------------------------------------------------------------------
Remove tabs as required
-------------------------------------------------------------------------- */
while ( (linech = line[index]) != '\000' ) /* still got input chars */
  {
  if ( linech == '\t' )			/* got an input tab char */
    { tabcol = ((outdex/tabsz)+1)*tabsz; /* col where we want current tab */
      outdex = tabcol-1; }		/* set it (bumped by 1 below) */
  else					/* not a tab char */
    if ( linech != '\f' )		/* turn formfeeds into blanks */
     if ( isprint(linech) )		/* also make sure char is printable*/
      outbuff[outdex] = linech;		/* copy printable char as is */
  index++;				/* ready for next input char */
  outdex++;				/* and output char */
  } /* --- end-of-while(linech!='\000') --- */
outbuff[outdex] = '\000';		/* null-terminate output buffer */
outptr = outbuff;			/* and return it to caller */
/* --------------------------------------------------------------------------
End-of-Job
-------------------------------------------------------------------------- */
end_of_job:
  return ( outptr );			/* ptr to outbuff, or null, returned*/
} /* --- end-of-function detab() --- */


/****************************************************************************
 * Function:	fputps ( fp, ps )
 * Purpose:   o	Prints an array of strings
 * --------------------------------------------------------------------------
 * Arguments:	fp (I)		FILE pointer to print device/file.
 *		ps (I)		address of char pointer containing array of
 *				lines to be printed, and terminated by
 *				a NULL pointer.
 * --------------------------------------------------------------------------
 * Returns:	(int)		number of lines printed,
 *				or -1 for any error.
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	fputps ( fp, ps )
FILE	*fp;
char	**ps;
{
int	nlines = 0;			/* #lines printed back to caller */
while ( *ps != (char *)NULL )
  {
  fprintf(fp,"%s\n",*ps);		/* print current line */
  ps++;					/* bump ptr to next line */
  nlines++;				/* and increment counter */
  } /* --- end-of-while(*ps!=NULL) --- */
return ( nlines );			/* back to caller with #lines */
} /* --- end-of-function fputps() --- */


/****************************************************************************
 *
 * Copyright (c) 1978,1992 John Forkosh Associates, Inc.  All rights reserved.
 * --------------------------------------------------------------------------
 *
 * Function:	banner ( fp, msgstr )
 *
 * Purpose:   o	Displays a banner consisting of the characters in
 *		msgstr (message string).  Each character is displayed
 *		eight rows high by ten columns wide.
 *
 * Arguments:	fp (I)		pointer to FILE device on which banner
 *				will be displayed.
 *		msgstr (I)	address of char string containing the
 *				message to be displayed on fp.
 *
 * Returns:	(int)		number of lines displayed,
 *				or 0 for any error.
 *
 * Notes:     o
 *
 * --------------------------------------------------------------------------
 * Revision History:
 * 10/??/74	J.Forkosh	Installation at University of Cincinnati,PL/1.
 * 09/01/78	J.Forkosh	Conversion to Univac Fieldata Fortran for NBC.
 * 07/01/82	J.Forkosh	Conversion to DG/F77 for Statistical Research.
 * 01/03/92	J.Forkosh	Conversion to "C" for John Forkosh Associates.
 *
 ***************************************************************************/

/* --------------------------------------------------------------------------
Headers and parameters
-------------------------------------------------------------------------- */
#if 0
/* --- standard headers --- */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
/* --- adjustable parameters --- */
#define	MAXCOLS 192			/* max width of banner */
#endif
#define	MAXCHARS 24			/* max chars across banner */
#define	COLSPACE 1			/* #cols between banner characters */
#define	MSGDELIM '/'			/* multiple-line banner separator */
#define	CENTER '>'			/* center banner if 1st char is this*/
#define	CENTERSZ 80			/* center on 80 columns */

/* --------------------------------------------------------------------------
Symbol Data
-------------------------------------------------------------------------- */
/* --- symbols recognized by banner() --- */
static	char symstr[] = " ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789.,-*#";
/* --- symbol height and width is inherent in symdat --- */
#define	SYMROWS	8			/* symbol height */
#define	SYMCOLS	10			/* symbol width */
/* --- each symbol parameterized by 8 numbers, each from 0 to (2**10)-1 --- */
static	int symdat[][8] =
  {
	0,	0,	0,	0,	0,	0,	0,	0,     /*spc*/
	24,	60,	102,	195,	511,	1023,	771,	771,   /* A */
	1020,	1022,	387,	510,	510,	387,	1022,	1020,  /* B */
	254,	511,	771,	768,	768,	771,	511,	254,   /* C */
	1020,	1022,	387,	387,	387,	387,	1022,	1020,  /* D */
	1023,	1023,	387,	504,	504,	387,	1023,	1023,  /* E */
	1023,	1023,	387,	504,	504,	384,	960,	960,   /* F */
	252,	510,	768,	782,	799,	771,	510,	252,   /* G */
	975,	975,	390,	510,	510,	390,	975,	975,   /* H */
	252,	252,	48,	48,	48,	48,	252,	252,   /* I */
	31,	31,	6,	6,	518,	774,	508,	248,   /* J */
	963,	966,	396,	504,	496,	412,	966,	963,   /* K */
	992,	992,	384,	384,	384,	387,	1023,	1023,  /* L */
	771,	903,	975,	891,	819,	771,	771,	771,   /* M */
	771,	899,	963,	867,	795,	783,	775,	771,   /* N */
	252,	510,	771,	771,	771,	771,	510,	252,   /* O */
	1020,	1022,	387,	510,	508,	384,	960,	960,   /* P */
	252,	510,	771,	771,	779,	775,	510,	253,   /* Q */
	1020,	1022,	387,	510,	508,	396,	390,	387,   /* R */
	254,	511,	768,	508,	254,	3,	1022,	508,   /* S */
	1023,	1023,	819,	48,	48,	48,	120,	120,   /* T */
	975,	975,	390,	390,	390,	390,	252,	120,   /* U */
	771,	771,	771,	771,	390,	204,	120,	48,    /* V */
	771,	771,	771,	819,	891,	975,	903,	771,   /* W */
	771,	390,	204,	48,	48,	204,	390,	771,   /* X */
	771,	390,	204,	120,	48,	48,	48,	48,    /* Y */
	1023,	1022,	780,	24,	96,	195,	511,	1023,  /* Z */
	252,	510,	771,	771,	771,	771,	510,	252,   /* 0 */
	16,	48,	112,	48,	48,	48,	48,	120,   /* 1 */
	252,	510,	903,	14,	56,	224,	511,	1023,  /* 2 */
	1023,	1022,	28,	56,	126,	3,	1022,	508,   /* 3 */
	24,	56,	120,	216,	511,	1023,	24,	24,    /* 4 */
	1023,	1023,	768,	1020,	1022,	3,	1022,	508,   /* 5 */
	252,	510,	768,	764,	1022,	771,	510,	252,   /* 6 */
	1023,	1023,	774,	12,	12,	12,	12,	12,    /* 7 */
	252,	510,	771,	510,	510,	771,	510,	252,   /* 8 */
	252,	510,	771,	511,	253,	3,	510,	252,   /* 9 */
	0,	0,	0,	128,	448,	992,	448,	128,   /* . */
	0,	0,	0,	192,	288,	32,	64,	128,   /* , */
	0,	0,	0,	510,	510,	0,	0,	0,     /* - */
	48,	48,	120,	1023,	252,	120,	204,	390,  /*star*/
	/* --- special typographical symbols --- */
	255,	255,	48,	62,	830,	816,	1008,	480,  /*JF=#*/
  } ;  /* --- end-of-array symdat[][8] --- */

/* --- entry point --- */
int	banner ( fp, msgstr )
FILE	*fp;
char	*msgstr;
{
/* --------------------------------------------------------------------------
Allocations and Declarations
-------------------------------------------------------------------------- */
static	char scanline[MAXCOLS];		/* buffer to construct symbol scan */
static	char msgline[MAXCHARS+1];	/* current line from msgstr */
int	msgstrlen = 0,			/* #chars from msgstr already used */
	msglinelen = 0;			/* #chars in current line */
char	*msglineptr = (char *)NULL;	/* ptr to char in msgline */
int	nlines = 0;			/* total #lines processed */
int	irow = 0;			/* row index of current scan */
int	col0=0, col1=0;			/* col0 for centering, end at col1 */
int	ichar = 0;  char msgchar = ' ';	/* index, character in msgline */
int	isymbol = 0,  idata = 0;	/* symbol index, data word for char */
int	icol = 0;			/* col driven by idata to place char*/

/* --------------------------------------------------------------------------
Initialization
-------------------------------------------------------------------------- */
if ( fp == (FILE *)NULL )		/* user passed null file ptr */
  fp = stdout;				/* default it to stdout */

/* --------------------------------------------------------------------------
Get next delimiter-separated line from msgstr
-------------------------------------------------------------------------- */
next_line:				/* target label for banner line loop*/
  strncpy( msgline, msgstr+msgstrlen, MAXCHARS ); /* extract next chars */
  msgline[MAXCHARS] = '\000';		/* make sure it's null-terminated */
  if ( (msglineptr=strchr(msgline,MSGDELIM)) /*look for delimiter in msgline*/
  !=   (char *)NULL )			/* found it */
    { *msglineptr = '\000';		/* null-terminate line at delim */
      msgstrlen++; }			/* but don't forget to count delim */
  if ( (msglinelen = strlen(msgline))	/* #chars for this line of banner */
  < 1 ) goto end_of_job;		/* all done when no chars in line */
  msgstrlen += msglinelen;		/* always count chars in line */
  if ( ++nlines > 1 )			/* bump line counter index */
    fprintf(fp,"\n");			/* vertical separation between lines*/
  col0 = 0;				/* default is left-justified */
  if ( *msgline == CENTER )		/* 1st char of msg toggles centering*/
    { strcpy(msgline,msgline+1);	/* shift out cenetring char */
      msglinelen--;			/* decrement length accordingly */
      col0 = (CENTERSZ - (msglinelen*(SYMCOLS+COLSPACE)-COLSPACE))/2;
      if ( col0 < 0 ) col0 = 0; }	/* msgline was too long */

/* --------------------------------------------------------------------------
Process the line
-------------------------------------------------------------------------- */
for ( irow=0; irow<SYMROWS; irow++ )	/* need SYMROWS scans for line */
  {
  memset( scanline, ' ', MAXCOLS-1 );	/* init scanline to blanks */
  for ( ichar=0; ichar<msglinelen; ichar++ ) /* each char in current line */
    {
    msgchar = (char)(toupper(msgline[ichar])); /* uppercase current char */
    if ( (msglineptr=strchr(symstr,msgchar)) /* look for char in symbols */
    ==   (char *)NULL ) continue;	/* caller wants char we don't have */
    isymbol = (int)(msglineptr-symstr);	/* offset into symstr[] for char */
    idata = symdat[isymbol][irow];	/* extract data word for this scan*/
    icol = col1 = col0+ichar*(SYMCOLS+COLSPACE)+SYMCOLS-1; /*last char col*/
    while ( idata > 0 )			/* more powers of two left in idata */
      {	if ( (idata%2) > 0 )		/* we want a char in this position */
	  scanline[icol] = msgchar;	/* so put it there */
	idata /= 2;  icol--; }		/* get ready for next digit */
    } /* --- end-of-for(ichar) --- */
  scanline[col1+1] = '\000';		/* null-terminate after last char */
  fprintf(fp,"%s\n",scanline);		/* and print scanline */
  } /* --- end-of-for(irow) --- */
goto next_line;

/* --------------------------------------------------------------------------
End-of-Job
-------------------------------------------------------------------------- */
end_of_job:
  return ( nlines );			/* back to caller with total #lines */
} /* --- end-of-function banner() --- */


#ifdef TESTDRIVE
/****************************************************************************
 * Function:	main ( argc, argv )
 * Purpose:   o	calls texlist(NULL,NULL), i.e., as a Unix "filter".
 * --------------------------------------------------------------------------
 * Arguments:	none
 * --------------------------------------------------------------------------
 * Returns:	none
 * --------------------------------------------------------------------------
 * Notes:     o
 ***************************************************************************/
/* --- entry point --- */
int	main ( argc, argv )
int	argc;
char	*argv[];
{
  /* --- input filesnames from stdin, output to stdout --- */
  texlist ( NULL, NULL );		/* call texlist() */
}
#endif
/* ------------------------ END-OF-FILE TEXLIST.C ------------------------ */
