#############################################################################
##
#A  Matrix Group and G-module library                   Derek Holt
#A                                                      Charles Leedham-Green
#A                                                      Eamonn O'Brien
#A                                                      Sarah Rees 
##
#A  @(#)$Id$
##
#Y  Copyright 1994 -- School of Mathematical Sciences, ANU   
##
#H  $Log$
##
#module.g
#
############################################################################
##
## This file contains the basic information about the module record.
#############################################################################
## The function GModule creates a module as a record, out of a set of
## matrices, or a group (generated by a set of matrices).
## The compulsory flags .field, .dim, .matrices and .isGmodule
## are set at creation time. Other optional flags appear below.
## The field of the module may be specified as an argument. If not the
## field is set to be the smallest field over which the matrices are defined.
## Outside this file,
## flags should always be accessed or set through functions, which are
## also listed below. 
## In general, a flag .****** is accessed via the function ******Flag
## and set with the function Set******Flag, with the first letter of the
## flag capitalised in the function name.
## The function IsGModule is an exception to this convention. .isGModule 
## is never set outside
## of the function .GModule, so no "set" function is necessary. 
## Accessing the flag with the function IsGModule
## is in line with other gap functions such as IsGroup.
## So the flag .reducible is accessed via ReducibleFlag, set with 
## SetReducibleFlag. If a flag is unset, the access function returns
## the string "unknown", otherwise it returns the value of the flag.
##
#############################################################################
##
#F  GModule( matrices [,F] ) . . . . . make a module from a list of matrices
##  or from a matrix group. The underlying field may be specified as an
##  an optional argument.
##
GModule := function ( arg )
    local matrices, F,r,dim,ngens,i,j;

    if Number(arg) = 1  then
        if IsGroup(arg[1]) then matrices := arg[1].generators;
        else matrices := arg[1];
        fi;
        F:=Field(Flat(matrices));
    elif Number(arg) = 2  then
        if IsGroup(arg[1]) then matrices := arg[1].generators;
        else matrices := arg[1];
        fi;
        F := arg[2];
    else
        return Error("usage: GModule ( <matrices> [, <F>] )");
    fi;
    dim := Length(matrices[1]);
    ngens := Length(matrices);
    for i in [1..ngens] do
        if Length(matrices[i]) <> dim then
          return Error("Matrices must all be square of the same size.\n");
        fi;
        for j in [1..dim] do
          if Length(matrices[i][j]) <> dim then
            return Error("Matrices must all be square of the same size.\n");
          fi;
        od;
    od;
          
    r := rec( field:=F,
             dim:=Length(matrices[1]),
             matrices:=matrices,
             isGModule:=true);
    return r;
end;

#############################################################################
##
#F  IsGModule( module ) . . . . . check if module really is a module
##
IsGModule := function ( module )
    if IsRec(module)=false or IsBound(module.isGModule)=false then
      return false;
    fi;
    return module.isGModule;
end;
############################################################################
##
#F  SetMatricesFlag( module,dim ) . . . . . set the .matrices flag to 
##                                           be matrices
##
SetMatricesFlag := function ( module,matrices )
    module.matrices := matrices;
end;
############################################################################
##
#F  MatricesFlag( module ) . . . . . access the .matrices flag
##
MatricesFlag := function ( module )
    if IsBound(module.matrices)=false then return "unknown"; fi;
    return module.matrices;
end;
############################################################################
##
#F  SetFieldFlag( module,F ) . . . . . set the .field flag to be F
##
SetFieldFlag := function ( module,F )
    module.field := F;
end;
############################################################################
##
#F  FieldFlag( module) . . . . . access the .field flag
##
FieldFlag := function ( module )
    if IsBound(module.field)=false then return "unknown"; fi;
    return module.field;
end;
############################################################################
##
#F  SetDimFlag( module,dim ) . . . . . set the .dim flag to be dim
##
SetDimFlag := function ( module,dim )
    module.dim := dim;
end;
############################################################################
##
#F  DimFlag( module ) . . . . . access the .dim flag
##
DimFlag := function ( module )
    if IsBound(module.dim)=false then return "unknown"; fi;
    return module.dim;
end;
############################################################################
## Functions dealing with optional flags.
############################################################################
##
#F  SetReducibleFlag( module,reducible ) . . . . . set the .reducible flag to be reducible
##
SetReducibleFlag := function ( module,reducible )
    module.reducible := reducible;
end;
############################################################################
##
#F  ReducibleFlag( module ) . . . . . access the .reducible flag
##
ReducibleFlag := function ( module )
    if IsBound(module.reducible)=false then return "unknown"; fi;
    return module.reducible;
end;
############################################################################
##
#F  UndoReducibleFlag( module ) . . . . . undo the .Reducible flag
##
UndoReducibleFlag := function ( module )
    Unbind(module.Reducible);
end;
############################################################################
##
#F  SetAbsReducibleFlag( module,absReducible ) . . . . . set the .absReducible flag to be absReducible
##
SetAbsReducibleFlag := function ( module,absReducible )
    module.absReducible := absReducible;
end;
############################################################################
##
#F  AbsReducibleFlag( module ) . . . . . access the .absReducible flag
##
AbsReducibleFlag := function ( module )
    if IsBound(module.absReducible)=false then return "unknown"; fi;
    return module.absReducible;
end;
############################################################################
##
#F  UndoAbsReducibleFlag( module ) . . . . . undo the .absReducible flag
##
UndoAbsReducibleFlag := function ( module )
    Unbind(module.absReducible);
end;
############################################################################
##
#F  SetSemiLinearFlag( module,semiLinear ) . . . . . set the .semiLinear flag to be semiLinear
##
SetSemiLinearFlag := function ( module,semiLinear )
    module.semiLinear := semiLinear;
end;
############################################################################
##
#F  SemiLinearFlag( module) . . . . . access the .SemiLinear flag
##
SemiLinearFlag := function ( module )
    if IsBound(module.semiLinear)=false then return "unknown"; fi;
    return module.semiLinear;
end;
############################################################################
##
#F  SetTensorProdFlag( module,tensorProd ) . . . . . set the .tensorProd 
## flag to be tensorProd
##
SetTensorProdFlag := function ( module,tensorProd )
    module.tensorProd := tensorProd;
end;
############################################################################
##
#F  TensorProdFlag( module ) . . . . . access the .tensorProd flag
##
TensorProdFlag := function ( module )
    if IsBound(module.tensorProd)=false then return "unknown"; fi;
    return module.tensorProd;
end;
############################################################################
##
#F  UndoTensorProdFlag( module ) . . . . . undo the .tensorProd flag
##
UndoTensorProdFlag := function ( module )
    Unbind(module.tensorProd);
end;
############################################################################
## The next collection of flags has to do with imprimitivity
## The last two of these functions were added by EOB 
############################################################################
##
#F  SetImprimitiveFlag( module,imprimitive ) . . . . . set the .imprimitive flag to be imprimitive
##
SetImprimitiveFlag := function ( module,imprimitive )
    module.imprimitive := imprimitive;
end;
############################################################################
##
#F  ImprimitiveFlag( module) . . . . . access the .imprimitive flag
##
ImprimitiveFlag := function ( module )
    if IsBound(module.imprimitive)=false then return "unknown"; fi;
    return module.imprimitive;
end;

#############################################################################
##
#F  SetBlockSystemFlag (module, SystemOfBlocks)  . set block system field to SystemOfBlocks
##
SetBlockSystemFlag := function ( module,SystemOfBlocks )

    module.blockSystem := SystemOfBlocks;

end; #SetBlockSystemFlag

#############################################################################
##
#F  BlockSystemFlag (module)  . . . .  . return block system field of module
##
BlockSystemFlag := function (module)

    if IsBound (module.blockSystem) = false then return "unknown"; fi;
    return module.blockSystem;

end; #BlockSystemFlag

############################################################################
## The next collection of flags has to do with reducibility/irreducibility
##########################################################################
##
#F  SetSubbasisFlag( module,subbasis) . . . . . set the .subbasis flag to 
## be subbasis
##
SetSubbasisFlag := function ( module,subbasis )
    module.subbasis := subbasis;
end;
############################################################################
##
#F  SubbasisFlag( module ) . . . . . access the .subbasis flag
##
SubbasisFlag := function ( module )
    if IsBound(module.subbasis)=false then return "unknown"; fi;
    return module.subbasis;
end;
############################################################################
##
#F  SetAlgElFlag( module,algel ) . . . . . set the .algEl flag to be algel
##
SetAlgElFlag := function ( module,algel )
    module.algEl := algel;
end;
############################################################################
##
#F  AlgElFlag( module ) . . . . . access the .algel flag
##
AlgElFlag := function ( module )
    if IsBound(module.algEl)=false then return "unknown"; fi;
    return module.algEl;
end;
############################################################################
##
#F  SetAlgElMatFlag(module,M) . . . . . set the .algElMat flag to be M
##
SetAlgElMatFlag := function ( module,M )
    module.algElMat := M;
end;
############################################################################
##
#F  AlgElMatFlag( module ) . . . . . access the .algElMat flag
##
AlgElMatFlag := function ( module )
    if IsBound(module.algElMat)=false then return "unknown"; fi;
    return module.algElMat;
end;
############################################################################
##
#F  SetAlgElCharPolFlag(module,p) . . . . . set the .algElCharPol flag to be p
##
SetAlgElCharPolFlag := function ( module,p )
    module.algElCharPol := p;
end;
############################################################################
##
#F  AlgElCharPolFlag( module ) . . . . . access the .algElCharPol flag
##
AlgElCharPolFlag := function ( module )
    if IsBound(module.algElCharPol)=false then return "unknown"; fi;
    return module.algElCharPol;
end;
############################################################################
##
#F  SetAlgElCharPolFacFlag(module,p) . . . . . set the .algElCharPolFac 
## flag to be p
##
SetAlgElCharPolFacFlag := function ( module,p )
    module.algElCharPolFac := p;
end;
############################################################################
##
#F  AlgElCharPolFacFlag( module ) . . . . . access the .algElCharPolFac flag
##
AlgElCharPolFacFlag := function ( module )
    if IsBound(module.algElCharPolFac)=false then return "unknown"; fi;
    return module.algElCharPolFac;
end;
############################################################################
##
#F  SetAlgElNullspaceVecFlag(module,v) . . . . . set the .algElNullspaceVec 
## flag to be v
##
SetAlgElNullspaceVecFlag := function ( module,v )
    module.algElNullspaceVec := v;
end;
############################################################################
##
#F  AlgElNullspaceVecFlag( module ) . . . . . access the .algElNullspaceVec flag
##
AlgElNullspaceVecFlag := function ( module )
    if IsBound(module.algElNullspaceVec)=false then return "unknown"; fi;
    return module.algElNullspaceVec;
end;
############################################################################
##
#F  SetAlgElNullspaceDimFlag(module,ndim) . . . . . set the .algElNullspaceDim 
## flag to be ndim
##
SetAlgElNullspaceDimFlag := function ( module,ndim )
    module.algElNullspaceDim := ndim;
end;
############################################################################
##
#F  AlgElNullspaceDimFlag( module ) . . . . . access the .algElNullspaceDim flag
##
AlgElNullspaceDimFlag := function ( module )
    if IsBound(module.algElNullspaceDim)=false then return "unknown"; fi;
    return module.algElNullspaceDim;
end;
############################################################################
## The next collection of flags is associated with absolute irreducibility,
## and some also with semilinearity.
############################################################################
##
#F  SetFieldExtDegFlag(module,deg) . . . . . set the .fieldExtDeg 
## flag to be deg
##
SetFieldExtDegFlag := function ( module,deg )
    module.fieldExtDeg := deg;
end;
############################################################################
##
#F  FieldExtDegFlag( module ) . . . . . access the .fieldExtDeg flag
##
FieldExtDegFlag := function ( module )
    if IsBound(module.fieldExtDeg)=false then return "unknown"; fi;
    return module.fieldExtDeg;
end;
############################################################################
##
#F  UndoFieldExtDegFlag( module ) . . . . . undo the .fieldExtDeh flag
##
UndoFieldExtDegFlag := function ( module )
    Unbind(module.fieldExtDeg);
end;
############################################################################
##
#F  SetCentMatFlag(module,mat) . . . . . set the .centMat 
## flag to be mat
##
SetCentMatFlag := function ( module,mat )
    module.centMat := mat;
end;
############################################################################
##
#F  CentMatFlag( module ) . . . . . access the .centMat flag
##
CentMatFlag := function ( module )
    if IsBound(module.centMat)=false then return "unknown"; fi;
    return module.centMat;
end;
############################################################################
##
#F  UndoCentMatFlag( module ) . . . . . undo the .centMat flag
##
UndoCentMatFlag := function ( module )
    Unbind(module.centMat);
end;
## The next two added by dfh 2/12/92.
############################################################################
##
#F  SetCentMatMinPolyFlag(module,p) . . . . . set the .centMatMinPoly 
## flag to be p
##
SetCentMatMinPolyFlag := function ( module,p )
    module.centMatMinPoly := p;
end;
############################################################################
##
#F  CentMatMinPolyFlag( module ) . . . . . access the .centMatMinPoly flag
##
CentMatMinPolyFlag := function ( module )
    if IsBound(module.centMatMinPoly)=false then return "unknown"; fi;
    return module.centMatMinPoly;
end;
############################################################################
## The next collection of flags is associated with semilinearity.
############################################################################
##
#F  SetLinearPartFlag(module,linearPart) . . . . . set the 
## .linearPartFlag flag to be linearPart
##
SetLinearPartFlag := function ( module,linearPart )
    module.linearPart := linearPart;
end;
############################################################################
##
#F  LinearPartFlag( module ) . . . . . access the .linearPart flag
##
LinearPartFlag := function ( module )
    if IsBound(module.linearPart)=false then return "unknown"; fi;
    return module.linearPart;
end;
############################################################################
##
#F  SetFrobeniusAutosFlag(module,frobeniusAutos) . . . . . set the .frobeniusAutos 
## flag to be frobeniusAutos
##
SetFrobeniusAutosFlag := function ( module,frobeniusAutos )
    module.frobeniusAutos := frobeniusAutos;
end;
############################################################################
##
#F  FrobeniusAutosFlag( module ) . . . . . access the .frobeniusAutos flag
##
FrobeniusAutosFlag := function ( module )
    if IsBound(module.frobeniusAutos)=false then return "unknown"; fi;
    return module.frobeniusAutos;
end;
############################################################################
## The next collection of flags is associated with tensorproducts.
############################################################################
##
#F  SetTensorBasisFlag(module,basis) . . . . . set the .tensorBasis 
## flag to be basis
##
SetTensorBasisFlag := function ( module,basis )
    module.tensorBasis := basis;
end;
############################################################################
##
#F  TensorBasisFlag( module ) . . . . . access the .tensorBasis flag
##
TensorBasisFlag := function ( module )
    if IsBound(module.tensorBasis)=false then return "unknown"; fi;
    return module.tensorBasis;
end;
############################################################################
##
#F  UndoTensorBasisFlag( module ) . . . . . undo the .tensorBasis flag
##
UndoTensorBasisFlag := function ( module )
    Unbind(module.TensorBasis);
end;
############################################################################
##
#F  SetTensorFactorsFlag(module,factors) . . . . . set the .tensorFactors
## flag to be factors.
##
SetTensorFactorsFlag := function ( module,factors )
    module.tensorFactors := factors;
end;
############################################################################
##
#F  TensorFactorsFlag( module ) . . . . . access the .tensorFactors flag
##
TensorFactorsFlag := function ( module )
    if IsBound(module.tensorFactors)=false then return "unknown"; fi;
    return module.tensorFactors;
end;
############################################################################
##
#F  UndoTensorFactorsFlag( module ) . . . . . undo the .tensorFactors flag
##
UndoTensorFactorsFlag := function ( module )
    Unbind(module.tensorFactors);
end;
#############################################################################
##
#F  SetSymTensorProdFlag(module,symtensorprod) . . . . . set the .symTensorProd 
## flag to be symtensorprod.
##
SetSymTensorProdFlag := function ( module,symtensorprod )
    module.symTensorProd := symtensorprod;
end;
############################################################################
##
#F  SymTensorProdFlag( module ) . . . . . access the .symTensorProd flag
##
SymTensorProdFlag := function ( module )
    if IsBound(module.symTensorProd)=false then return "unknown"; fi;
    return module.symTensorProd;
end;
############################################################################
##
#F  SetSymTensorBasisFlag(module,symtensorbasis) . . . . . set the .symTensorBasis 
## flag to be symtensorbasis.
##
SetSymTensorBasisFlag := function ( module,symtensorbasis )
    module.symTensorBasis := symtensorbasis;
end;
############################################################################
##
#F  SymTensorBasisFlag( module ) . . . . . access the .symTensorBasis flag
##
SymTensorBasisFlag := function ( module )
    if IsBound(module.symTensorBasis)=false then return "unknown"; fi;
    return module.symTensorBasis;
end;
############################################################################
##
#F  SetSymTensorFactorsFlag(module,symtensorfactors) . . . . . set the .symTensorFactors 
## flag to be symtensorfactors.
##
SetSymTensorFactorsFlag := function ( module,symtensorfactors )
    module.symTensorFactors := symtensorfactors;
end;
############################################################################
##
#F  SymTensorFactorsFlag( module ) . . . . . access the .symTensorFactors flag
##
SymTensorFactorsFlag := function ( module )
    if IsBound(module.symTensorFactors)=false then return "unknown"; fi;
    return module.symTensorFactors;
end;
############################################################################
##
#F  SetSymTensorPermFlag(module,symtensorperm) . . . . . set the .symTensorPerm 
## flag to be symtensorperm.
##
SetSymTensorPermFlag := function ( module,symtensorperm )
    module.symTensorPerm := symtensorperm;
end;
############################################################################
##
#F  SymTensorPermFlag( module ) . . . . . access the .symTensorPerm flag
##
SymTensorPermFlag := function ( module )
    if IsBound(module.symTensorPerm)=false then return "unknown"; fi;
    return module.symTensorPerm;
end;
############################################################################
##
#F  SetExtraSpecialFlag(module,extraspecial) . . . . . set the .extraSpecial 
## flag to be extraspecial.
##
SetExtraSpecialFlag := function ( module,extraspecial)
    module.extraSpecial := extraspecial;
end;
############################################################################
##
#F  ExtraSpecialFlag( module ) . . . . . access the .extraSpecial flag
##
ExtraSpecialFlag := function ( module )
    if IsBound(module.extraSpecial)=false then return "unknown"; fi;
    return module.extraSpecial;
end;
############################################################################
##
#F  SetExtraSpecialPartFlag(module,extraspecialpart) . . . . . set 
## the .extraSpecialPart flag to be extraspecialpart.
##
SetExtraSpecialPartFlag := function ( module,extraspecialpart)
    module.extraSpecialPart := extraspecialpart;
end;
############################################################################
##
#F  ExtraSpecialPartFlag( module ) . . . . access the .extraSpecialPart flag
##
ExtraSpecialPartFlag := function ( module )
    if IsBound(module.extraSpecialPart)=false then return "unknown"; fi;
    return module.extraSpecialPart;
end;
############################################################################
##
#F  SetExtraSpecialPrimeFlag(module,extraspecialprime) . . . . . set 
## the .extraSpecialPrime flag to be extraspecialprime.
SetExtraSpecialPrimeFlag := function ( module,extraspecialprime)
    module.extraSpecialPrime := extraspecialprime;
end;
############################################################################
##
#F  ExtraSpecialPrimeFlag( module ) . . . access the .extraSpecialPrime flag
##
ExtraSpecialPrimeFlag := function ( module )
    if IsBound(module.extraSpecialPrime)=false then return "unknown"; fi;
    return module.extraSpecialPrime;
end;
############################################################################
