 /*
  * Khoros: $Id: graphics_2D.c,v 1.1 1991/05/10 15:56:56 khoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: graphics_2D.c,v 1.1 1991/05/10 15:56:56 khoros Exp $";
#endif

 /*
  * $Log: graphics_2D.c,v $
 * Revision 1.1  1991/05/10  15:56:56  khoros
 * Initial revision
 *
  */ 


/*
 *----------------------------------------------------------------------
 *
 * Copyright 1990, University of New Mexico.  All rights reserved.
 *
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as too the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"	 /* Copyright 1990 by UNM */
#include "X3D.h"


/* >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
   >>>>                                                       <<<<
   >>>>	    file name: graphics_2D.c 
   >>>>               
   >>>>   description: 2D Viewing System Control Utilities
   >>>>              
   >>>>      routines: 
   >>>>               _X2D_update_graphics()	      	      <<<<              
   >>>>               X2D_set_window()		      	      <<<<              
   >>>>               X2D_set_viewport()	      	      <<<<              
   >>>>               X2D_set_wc_min_max()	      	      <<<<              
   >>>>               X2D_set_wc_scale()	      	      <<<<              
   >>>>               X2D_set_clipping()	      	      <<<<              
   >>>>              
   >>>> modifications:
   >>>>                                                       <<<<
   >>>>>>>>>>>>>>>>>>>>>>>>>>>>> <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<< */


/************************************************************
*
*  MODULE NAME: _X2D_update_graphics
*
*      PURPOSE: Recomputes the X3D Graphics Matrix, used to
*		transform 2D world coordinates into device
*		coordinates.
*
*        INPUT: graphics - X3D graphics structure
*
*       OUTPUT: none
*
*    CALLED BY: X3D_set_window(),
*		X3D_set_viewport(), 
*		X3D_set_wc_min_max()
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/

_X2D_update_graphics (graphics)

X3DGraphics	*graphics;
{
	Matrix_2D	matrix1, matrix2;

	/*
	 * Convert to unit cube.
	*/
	_X2D_matrix_set_ndc(graphics->xv_min, graphics->xv_max,
			    graphics->yv_min, graphics->yv_max,
			    graphics->wc_min, graphics->wc_max, matrix1);
	/*
	 *  Set the device coordinate transformation matrix
	 */
	_X2D_matrix_set_dc(graphics, matrix2);
	_X2D_matrix_mult(matrix1, matrix2, graphics->matrix2D);
}



/************************************************************
*
*  MODULE NAME: X2D_set_window
*
*      PURPOSE: Sets the minimum and maximum device coordinate
*		for a given X11 Window.
*
*        INPUT: id        - X3D graphics structure ID
*		x, y,
*		width,
*		height	  - the offset and size of the window (in device
*			    coordinates).
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
************************************************************/

X2D_set_window (id, x, y, width, height)

int	id;
short	x, y, width, height;
{
	X3DGraphics 	*graphics;


	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X2D_set_window:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}


	if (graphics->device == X11)
	{
	   graphics->X11_xmin = x;
	   graphics->X11_ymin = y;
	   graphics->X11_xmax = width  +  x;
	   graphics->X11_ymax = height + y;
	}
	else if (graphics->device == POSTSCR)
	{
	   graphics->POS_xmin = x;
	   graphics->POS_ymin = y;
	   graphics->POS_xmax = width  +  x;
	   graphics->POS_ymax = height + y;
	}
	else if (graphics->device == IMPRESS)
	{
	   graphics->IMP_xmin = x;
	   graphics->IMP_ymin = y;
	   graphics->IMP_xmax = width  +  x;
	   graphics->IMP_ymax = height + y;
	}
	else if (graphics->device == HPGL)
	{
	   graphics->HPGL_xmin = x;
	   graphics->HPGL_ymin = y;
	   graphics->HPGL_xmax = width  +  x;
	   graphics->HPGL_ymax = height + y;
	}
	_X2D_update_graphics(graphics);
}




/************************************************************
*
*  MODULE NAME: X2D_set_viewport
*
*      PURPOSE: Sets the minimum and maximum viewport values
*		for a given X3D graphics structure.
*
*        INPUT: id     - X3D graphics structure ID
*		xmin   - A float containing the x minimum viewport 
*			 normalized device coordinate value.
*		xmax   - A float containing the x maximum world 
*			 normalized device coordinate value.
*		ymin   - A float containing the y minimum viewport 
*			 normalized device coordinate value.
*		ymax   - A float containing the y maximum world 
*			 normalized device coordinate value.
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
************************************************************/

X2D_set_viewport (id, xmin, xmax, ymin, ymax)

int	id;
Real	xmin, xmax, ymin, ymax;
{
	X3DGraphics *graphics;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X2D_set_viewport:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}
	graphics->xv_min = xmin;
	graphics->xv_max = xmax;
	graphics->yv_min = ymin;
	graphics->yv_max = ymax;
	if (graphics->xv_min == graphics->xv_max)
	{
	   graphics->xv_min -= 0.5;
	   graphics->xv_max += 0.5;
	}
	if (graphics->yv_min == graphics->yv_max)
	{
	   graphics->yv_min -= 0.5;
	   graphics->yv_max += 0.5;
	}
	_X2D_update_graphics(graphics);
}



/************************************************************
*
*  MODULE NAME: X2D_set_wc_min_max
*
*      PURPOSE: Sets the minimum and maximum world coordinate
*		points for a given X3D graphics structure.
*
*        INPUT: id     - X3D graphics structure ID
*		wc_min - A coord containing the minimum world 
*			 coordinate values
*		wc_max - A coord containing the maximum world 
*			 coordinate values
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young
*
*
*************************************************************/

X2D_set_wc_min_max (id, wc_min, wc_max)

int	id;
Coord	wc_min, wc_max;
{
	X3DGraphics *graphics;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X2D_set_wc_min_max:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}

	if (wc_min.x == wc_max.x)
	{
	   wc_min.x -= 0.5;
	   wc_max.x += 0.5;
	}
	if (wc_min.y == wc_max.y)
	{
	   wc_min.y -= 0.5;
	   wc_max.y += 0.5;
	}

	graphics->smin = wc_min;
	graphics->smax = wc_max;

	if (!(_X2D_scale_wc_coord(graphics->scale, wc_min, &graphics->wc_min)))
	   graphics->wc_min = wc_min;

	if (!(_X2D_scale_wc_coord(graphics->scale, wc_max, &graphics->wc_max)))
	   graphics->wc_max = wc_max;

	_X2D_update_graphics(graphics);
}



/************************************************************
*
*  MODULE NAME: X2D_set_wc_scale
*
*      PURPOSE: Sets the minimum and maximum world coordinate
*		points for a given X3D graphics structure.
*
*        INPUT: id     - X3D graphics structure ID
*		scale - A coord containing the maximum world 
*			 coordinate scale types for the x, y,
*			 & z axes.
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mark Young & Mike Lang
*
*
*************************************************************/

X2D_set_wc_scale(id, scale)

int	id;
Coord	scale;
{
	X3DGraphics *graphics;
	Coord smin, smax, wc_min, wc_max;

	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_wc_scale:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}

        wc_min = graphics->wc_min;
        wc_max = graphics->wc_max;
        smin   = graphics->smin;
        smax   = graphics->smax;

	graphics->scale = scale;
	if (!(_X2D_scale_wc_coord(graphics->scale, smin, &graphics->wc_min)))
	   graphics->wc_min = wc_min;

	if (!(_X2D_scale_wc_coord(graphics->scale, smax, &graphics->wc_max)))
	   graphics->wc_max = wc_max;

	_X2D_update_graphics(graphics);
}

/************************************************************
*
*  MODULE NAME: X2D_set_clipping
*
*      PURPOSE: Turns clipping to the viewwport on for 2D
*
*        INPUT: id     - X3D graphics structure ID
*		on_off - 1 for on; 0 for off
*
*       OUTPUT: none
*
*    CALLED BY: application program
*
*   WRITTEN BY: Mike Lang
*
*
*************************************************************/

X2D_set_clipping(id, on_off)

int	id;
int	on_off;
{
	X3DGraphics *graphics;
	if (!(graphics = _X3D_get_graphics(id)))
	{
	   (void) fprintf (stderr,"X3D_set_clipping:");
	   (void) fprintf (stderr,"\t unknown graphics id %d\n",id);
	   return;
	}
	if( on_off == 1 )
		graphics->clipping_2D = 1;
	else
		graphics->clipping_2D = 0;
}


