IMPLEMENTATION MODULE Screen;

	(********************************************************)
	(*							*)
	(*		Low-level screen functions		*)
	(*							*)
	(*  Programmer:		P. Moylan			*)
	(*  Last edited:	30 January 1994			*)
	(*  Status:						*)
	(*	Working for Hercules, CGA, EGA, VGA; and	*)
	(*	supports SVGA for adaptors with VESA drivers.	*)
	(*							*)
	(********************************************************)

(************************************************************************)
(*									*)
(*  A particular problem in supporting screen operations on PC or AT	*)
(*  compatibles is that there are enormous differences between models	*)
(*  in what sort of graphics interface is provided.  Because it is hard	*)
(*  to predict what hardware will be present, this module uses ROM BIOS	*)
(*  routines to do some of the low-level things.  Note however that the	*)
(*  Hercules graphics cards are not supported by the BIOS, so we have	*)
(*  to treat them as a special case.					*)
(*									*)
(*  For SuperVGA modes, we rely on the presence of a VESA BIOS.		*)
(*									*)
(************************************************************************)

FROM SYSTEM IMPORT
    (* type *)	BYTE,
    (* proc *)	ADR;

FROM Storage IMPORT
    (* proc *)	ALLOCATE, DEALLOCATE;

FROM LowLevel IMPORT
    (* proc *)	SEGMENT, OFFSET, Virtual, MakePointer, AddOffset,
		InByte, OutByte, IAND, IANDB, Div;

FROM MiscPMOS IMPORT
    (* type *)	RegisterPacket,
    (* proc *)	BIOS,
    (* proc *)	EnterCriticalSection, LeaveCriticalSection;

FROM TerminationControl IMPORT
    (* proc *)  SetTerminationProcedure;

(************************************************************************)

CONST
    VideoInt = 16;		(* Interrupt number for BIOS calls	*)

TYPE
    ModeSet = SET OF [0..511];

    (* Portability warning: the following declaration defines a type of	*)
    (* procedure which expects its parameters to be in the BX and DX	*)
    (* registers.  Because of the way the VESA BIOS works, it's		*)
    (* important to use precisely those registers.  When porting to a	*)
    (* compiler which doesn't let you specify which registers to use	*)
    (* for parameter passing, you might have to write an assembly	*)
    (* language routine to do the bank switching.			*)

    (*# save, call(reg_param => (bx, dx)) *)
    BankSwitchProc = PROCEDURE (CARDINAL,CARDINAL);
    (*# restore *)

CONST
    (* 'Monochrome' display modes supported by this module.  Note that	*)
    (* this means modes supported by a monochrome adaptor.  There are	*)
    (* some other single-colour modes which we don't classify as	*)
    (* monochrome, simply because they are not MDA modes.		*)

    MonochromeModes = ModeSet {0, 2, 7};

VAR
    (* Our opinion as to what type of adaptor is installed.	*)

    AdaptorType: VideoAdaptorType;

    (* ScreenSeg is a segment selector for the hardware video buffer,	*)
    (* and CRTCport is the port number to use when addressing the CRT	*)
    (* controller chip.  They are variables, because the addresses	*)
    (* depend on whether a colour or monochrome mode is in effect.	*)

    ScreenSeg: CARDINAL;
    CRTCport: CARDINAL;

    (* OriginalMode and OriginalPage are the video mode and text page	*)
    (* number in effect when the program first runs.  Procedure		*)
    (* RestoreOriginalMode switches back to them.			*)

    OriginalMode, CurrentMode: CARDINAL;
    OriginalPage: SHORTCARD;

    (* For the modes which use bank-switching, the hardware may allow	*)
    (* two separate windows into the graphics memory.  The following	*)
    (* two variables each have value 0 or 1.  Where two separate	*)
    (* windows are not supported, we have ReadWindow = WriteWindow.	*)

    ReadWindow, WriteWindow: CARDINAL;

    (* For the modes which use bank-switching, CurrentReadBank and	*)
    (* CurrentWriteBank are the currently selected banks.		*)
    (* (SetVideoMode initialises them to "impossible" values in order	*)
    (* to force a bank switch operation on the first read or write.)	*)
    (* For all other modes, the two values are left at zero.		*)

    CurrentReadBank, CurrentWriteBank: CARDINAL;

    (* Set of all modes which we believe the hardware will support.	*)

    SupportedModes: ModeSet;

    (* Modes supported by the VESA driver, if present.  The first set	*)
    (* is the set officially supported, and the second is a set of	*)
    (* modes we've found by trial and error to be supported even though	*)
    (* the VESA driver doesn't have them in its list of supported modes.*)

    VesaModes, ExtraVesaModes: ModeSet;

    (* The procedure which does the bank switching when a VESA driver	*)
    (* is being used.  See the definition of BankSwitchProc above for	*)
    (* a warning about portability.					*)

    BankSwitch: BankSwitchProc;

    (* Information about the current mode, as obtained from the VESA	*)
    (* driver.  If the current mode is not a VESA mode, these data	*)
    (* are obsolete.							*)

    VESA_01_Information:
		    RECORD
			mode_attributes: WORD;
			window_a_attrib: BYTE;
			window_b_attrib: BYTE;
			window_granularity: WORD;
			window_size: WORD;
			window_a_segment_adr: WORD;
			window_b_segment_adr: WORD;
			bank_switch_function: BankSwitchProc;
			bytes_per_row: CARDINAL;
	                x_res, y_res: CARDINAL;
			x_character_size,
			y_character_size: SHORTCARD;
			number_of_planes: BYTE;
			bits_per_pixel: BYTE;
			number_of_banks: BYTE;
			memory_model: BYTE;
			bank_size: BYTE;
			number_of_image_pages: BYTE;
			BIOS_reserved: BYTE;
			red_mask_size: SHORTCARD;
			red_field_position: SHORTCARD;
			green_mask_size: SHORTCARD;
			green_field_position: SHORTCARD;
			blue_mask_size: SHORTCARD;
			blue_field_position: SHORTCARD;
			reserved_mask_size: SHORTCARD;
			reserved_field_position: SHORTCARD;
			direct_col_mode_info: BYTE;
			unused: ARRAY [40..255] OF BYTE;
		    END (* VESA_01_Information RECORD *);

(************************************************************************)
(*		      OPERATIONS ON THE VIDEO MODE			*)
(************************************************************************)

PROCEDURE Supported (mode: CARDINAL): BOOLEAN;

    (* Returns TRUE iff the specified mode is a mode supported	*)
    (* by the hardware and by this module.			*)

    BEGIN
	RETURN (mode IN SupportedModes);
    END Supported;

(************************************************************************)

PROCEDURE BugFixes (mode: CARDINAL);

    (* Called when a new mode is set: we compensate, if possible, for	*)
    (* errors introduced by bugs in the BIOS.  (Obviously we're limited	*)
    (* to the small number of cases where we know the bug and have	*)
    (* worked out how to repair it - but every little bit helps.)	*)

    BEGIN
	IF AdaptorType = Trident THEN
	    (* Fixes for known bugs in the Trident VESA BIOS.	*)
	    IF (mode = 272) OR (mode = 273) THEN
		VESA_01_Information.bytes_per_row := 1280;
	    END (*IF*);
	    (* There are also bugs in modes 106,258,260 - being		*)
	    (* problems with screen synch - but we don't yet know how	*)
	    (* to fix those.						*)
	END (*IF*);
    END BugFixes;

(************************************************************************)

PROCEDURE DecodeWindows (attra, attrb: BYTE);

    (* Works out which windows are to be used for reading and writing.	*)
    (* This is for VESA modes, where there is a provision with some	*)
    (* adaptors and some modes to allow different banks to be selected	*)
    (* for reading and writing.						*)

    BEGIN
	ReadWindow := 0;  WriteWindow := 0;
	IF IANDB (attra, 2) = 0 THEN
	    ReadWindow := 1;
	END (*IF*);
	IF IANDB (attra, 4) = 0 THEN
	    WriteWindow := 1;
	END (*IF*);
    END DecodeWindows;

(************************************************************************)

PROCEDURE SetHerculesMode (newmode: CARDINAL;  ClearScreen: BOOLEAN);

    (* A version of procedure SetVideoMode (see below) to be used for	*)
    (* Hercules adaptors only.						*)

    TYPE ControlTable = ARRAY [0..11] OF BYTE;

    CONST
	(* Tables of values to put in the 6845 video controller.  These	*)
	(* are details like character positions per row, sync		*)
	(* positions, etc., and can be fiddled with only at great risk.	*)
	(* HGtable gives values suitable for Hercules graphics, and	*)
	(* MDAtable gives values suitable for MDA or Hercules text mode.*)

	HGtable = ControlTable (53,45,46,07,91,02,87,87,02,03,00,00);
	MDAtable= ControlTable (97,80,82,15,25,06,25,25,02,13,11,12);

    VAR tableptr: POINTER TO ControlTable;
	controlcode: SHORTCARD;
	j, buffersize, fillcode, savedPS: CARDINAL;
	p: POINTER TO CARDINAL;

    BEGIN
	IF newmode = HercGraphics THEN
	    tableptr := ADR(HGtable);  controlcode := 2;
	    buffersize := 32768;	(* words in video page 0 *)
	    fillcode := 0;

	    (* For maximal compatibility with any other hardware	*)
	    (* which might be present, we enable page 0 graphics only.	*)

	    OutByte (CRTCport+11, 1);

	ELSE
	    newmode := 7;
	    tableptr := ADR(MDAtable);  controlcode := 20H;
	    buffersize := 2000;	(* words for text mode *)
	    fillcode := 0720H;	(* space, attribute code 7 *)
	END (*IF*);

	(* Reprogram the 6845 video controller.  Warning: this part	*)
	(* is quite critical; wrong values could damage the		*)
	(* hardware.  We disable interrupts here to minimise the	*)
	(* time for which the 6845 register values are changing.	*)
	(* We also blank the screen during the operation, to avoid	*)
	(* or reduce the disconcerting effects while the monitor	*)
	(* regains synchronism.						*)
	
	savedPS := EnterCriticalSection();
	OutByte (CRTCport+4, controlcode);
	FOR j := 0 TO 11 DO
	    OutByte (CRTCport, BYTE(j));
	    OutByte (CRTCport+1, tableptr^[j]);
	END (*FOR*);
	LeaveCriticalSection (savedPS);

	(* Finished reprogramming, clear the screen buffer and	*)
	(* then enable video.					*)

	IF ClearScreen THEN
	    FOR j := 0 TO 2*(buffersize-1) BY 2 DO
		p := MakePointer (ScreenSeg, j);
		p^ := fillcode;
	    END (*FOR*);
	END (*IF*);

	OutByte (CRTCport+4, controlcode+8);
	CurrentReadBank := 0;  CurrentWriteBank := 0;
	CurrentMode := newmode;

    END SetHerculesMode;

(************************************************************************)

PROCEDURE SetVideoMode (newmode: CARDINAL;  ClearScreen: BOOLEAN): BOOLEAN;

    (* Sets the video mode.  The mode numbers are as defined in the	*)
    (* BIOS, plus HercGraphics to denote the Hercules graphics mode,	*)
    (* plus whatever the VESA BIOS (if present) will support.		*)
    (* Returns TRUE iff the mode change was successful.			*)

    VAR Registers: RegisterPacket;	

    BEGIN
	IF AdaptorType = Hercules THEN
	    SetHerculesMode (newmode, ClearScreen);
	    RETURN TRUE;

	ELSIF newmode IN VesaModes+ExtraVesaModes THEN

	    WITH Registers DO

		(* Set the mode *);

		AX := 4F02H;
		BX := newmode;
		IF NOT ClearScreen THEN
		    INC (BX, 8000H);
		END (*IF*);
		BIOS(VideoInt, Registers);
		IF AX <> 004FH THEN RETURN FALSE END(*IF*);

		(* Obtain the attributes of that mode. *)

		AX := 4F01H;
		CX := newmode;
		DI := OFFSET(ADR(VESA_01_Information));
		ES := SEGMENT(ADR(VESA_01_Information));
		BIOS(VideoInt, Registers);
		IF AX <> 004FH THEN RETURN FALSE END(*IF*);

	    END (*WITH*);
	    WITH VESA_01_Information DO
		DecodeWindows (window_a_attrib, window_b_attrib);
		BankSwitch := bank_switch_function;
	    END (*WITH*);
	    CurrentReadBank := MAX(CARDINAL);
	    CurrentWriteBank := MAX(CARDINAL);
	    CurrentMode := newmode;
	    BugFixes (newmode);

	ELSIF newmode IN SupportedModes THEN

	    (* For all the "standard" modes, let the BIOS do the job.	*)

	    Registers.AX := newmode;
	    IF NOT ClearScreen THEN
		INC (Registers.AX, 80H);
	    END (*IF*);
	    BIOS (VideoInt, Registers);
	    CurrentReadBank := 0;  CurrentWriteBank := 0;
	    CurrentMode := newmode;

	ELSE
	    (* Not a supported mode. *)

	    RETURN FALSE;

	END (*IF*);

	(* ScreenSeg and CRTCport depend on whether we are using	*)
	(* a monochrome or colour mode.					*)

	IF newmode IN MonochromeModes THEN
	    ScreenSeg := SEGMENT(Virtual(0B0000H));
	    CRTCport := 03B4H;
	ELSE
	    ScreenSeg := SEGMENT(Virtual(0B8000H));
	    CRTCport := 03D4H;
	END (*IF*);

	(* For an EGA or better adaptor, ScreenSeg depends on	*)
	(* whether we are emulating a CGA mode.			*)

	IF newmode >= 13 THEN
	    ScreenSeg:= SEGMENT (Virtual(0A0000H));
	END (*IF*);

	RETURN TRUE;

    END SetVideoMode;

(************************************************************************)

PROCEDURE RestoreOriginalMode;

    (* Sets the video mode back to what it was before this program ran.	*)
    (* Also restores the original text page.				*)

    VAR Registers: RegisterPacket;  dummy: BOOLEAN;

    BEGIN
	dummy := SetVideoMode (OriginalMode, FALSE);
	Registers.AH := 5;  Registers.AL := OriginalPage;
	BIOS (VideoInt, Registers);
    END RestoreOriginalMode;

(************************************************************************)
(*			MISCELLANEOUS OPERATIONS			*)
(************************************************************************)

PROCEDURE SelectReadBank (bank: CARDINAL);

    (* Switches to a new bank of screen memory for reading.  Should be	*)
    (* used only with the adaptors which support the high-resolution	*)
    (* modes using bank switching.					*)

    BEGIN
	IF bank <> CurrentReadBank THEN
	    BankSwitch (ReadWindow, bank);
	    CurrentReadBank := bank;
	    IF ReadWindow = WriteWindow THEN
		CurrentWriteBank := bank;
	    END (*IF*);
	END (*IF*);
    END SelectReadBank;

(************************************************************************)

PROCEDURE SelectWriteBank (bank: CARDINAL);

    (* Switches to a new bank of screen memory for writing.  Should be	*)
    (* used only with the adaptors which support the high-resolution	*)
    (* modes using bank switching.					*)

    BEGIN
	IF bank <> CurrentWriteBank THEN
	    BankSwitch (WriteWindow, bank);
	    CurrentWriteBank := bank;
	    IF ReadWindow = WriteWindow THEN
		CurrentReadBank := bank;
	    END (*IF*);
	END (*IF*);
    END SelectWriteBank;

(************************************************************************)

PROCEDURE WaitForVerticalRetrace;

    (* Busy wait until we reach the vertical retrace period.		*)
    (* Warning: I wrote this quickly for one specific application, and	*)
    (* haven't gotten around to getting it right for the general case.	*)

    CONST CGAStatusReg = 3DAH;

    BEGIN
	WHILE IANDB(InByte(CGAStatusReg),8) = 0 DO
	    (* nothing *)
	END (*WHILE*);
    END WaitForVerticalRetrace;

(************************************************************************)
(*		    INFORMATION ABOUT CURRENT MODE			*)
(************************************************************************)

PROCEDURE GetModeData (VAR (*OUT*) ScreenSegment, IObase: CARDINAL);

    (* Returns the segment of the screen memory and the port number	*)
    (* of the video controller.						*)

    BEGIN
	ScreenSegment := ScreenSeg;  IObase := CRTCport;
    END GetModeData;

(************************************************************************)

PROCEDURE GetExtendedModeInformation (
			VAR (*INOUT*) BytesPerRow,
				MaxX, MaxY, MaxColour: CARDINAL;
			VAR (*INOUT*) BitsPerPixel: SHORTCARD;
			VAR (*INOUT*) MultiBank: BOOLEAN);

    (* Returns information supplied by a VESA driver (if present) for	*)
    (* the current mode.  If the information is not available, the	*)
    (* parameter values are left unchanged.				*)

    BEGIN
	IF NOT (CurrentMode IN (VesaModes+ExtraVesaModes)) THEN
	    RETURN;
	END (*IF*);
	WITH VESA_01_Information DO
	    BytesPerRow := bytes_per_row;
	    MaxX := x_res - 1;
	    MaxY := y_res - 1;
	    BitsPerPixel := bits_per_pixel DIV number_of_planes;
	    CASE memory_model OF
		0, 2:	MaxColour := 1;
	      |
		1:	MaxColour := 3;
	      |
		3:	MaxColour := 15;
	      |
		4, 5:	MaxColour := 255;
	      |
		6:	IF reserved_mask_size > 0 THEN
			    MaxColour := 32767;
			ELSE
			    MaxColour := 65535;
			END (*IF*);
	      |
		ELSE	MaxColour := 1;
	    END (*CASE*);
	    MultiBank := y_res > Div(10000H, bytes_per_row);
	END (*WITH*);
    END GetExtendedModeInformation;

(************************************************************************)
(*			EQUIPMENT KIND DETECTION			*)
(************************************************************************)

PROCEDURE VideoKind (): VideoAdaptorType;

    (* Returns the display adaptor type.  This is a best guess, and it	*)
    (* is possible that some adaptor types will be misclassified.	*)
    (* In the present version, SVGA adaptors will be reported as VGA,	*)
    (* and no distinction is drawn between the "ordinary" Hercules	*)
    (* adaptor and the Hercules Plus or Hercules InColor.		*)

    BEGIN
	RETURN AdaptorType;
    END VideoKind;

(************************************************************************)
(*		 SPECIAL CASE: HERCULES GRAPHICS ADAPTOR		*)
(************************************************************************)

PROCEDURE CheckForHercules;

    (* The Hercules graphics interface is a special case because the	*)
    (* BIOS equipment information does not distinguish between it and	*)
    (* the Monochrome Display Adaptor.  The HGA has the same text mode	*)
    (* as the MDA, but it has a graphics mode as well.  One way to work	*)
    (* out the difference is to look at the "vertical retrace" bit in	*)
    (* the display status port.  The MDA does not implement this bit,	*)
    (* so reading it will return a constant.				*)

    VAR i, j, changes: CARDINAL;
	oldvalue, newvalue: BYTE;

    BEGIN
	changes := 0;
	FOR i := 0 TO 24 DO
	    oldvalue := IANDB (InByte (CRTCport+6), 80H);
	    j := 32767;
	    LOOP
		newvalue := IANDB (InByte (CRTCport+6), 80H);
		IF newvalue <> oldvalue THEN
		    INC (changes);  EXIT(*LOOP*);
		END (*IF*);
		DEC (j);
		IF j = 0 THEN EXIT(*LOOP*) END(*IF*);
	    END (*LOOP*);
	END (*FOR*);
	IF changes > 20 THEN
	    AdaptorType := Hercules;
	    INCL (SupportedModes, HercGraphics);
	END (*IF*);
    END CheckForHercules;

(************************************************************************)
(*		INITIALISATION FOR VESA SVGA ADAPTORS			*)
(************************************************************************)

PROCEDURE Match (StringPtr: ADDRESS;  location: CARDINAL;
				OtherString: ARRAY OF CHAR): BOOLEAN;

    (* Compares two strings, where the first is a substring specified	*)
    (* in terms of an offset from a memory address.			*)

    VAR p: POINTER TO CHAR;  k: CARDINAL;

    BEGIN
	p := AddOffset (StringPtr, location);
	k := 0;
	LOOP
	    IF k = HIGH(OtherString) THEN RETURN TRUE END(*IF*);
	    IF p^ <> OtherString[k] THEN RETURN FALSE END(*IF*);
	    INC (k);
	    p := AddOffset (p, 1);
	END (*LOOP*);
    END Match;

(************************************************************************)

PROCEDURE SpecificDriverChecks (OEMStringPtr: ADDRESS);

    (* Performs checks for specific VESA drivers which we need to know	*)
    (* about because of fixable bugs in the drivers, etc.		*)
    (* The present version of this procedure can handle the only	*)
    (* Trident driver I've been able to test in sufficient detail.	*)
    (* Obviously this procedure can be expanded as we learn about more	*)
    (* drivers.								*)

    BEGIN
	IF Match (OEMStringPtr, 21, "TRIDENT") THEN
	    AdaptorType := Trident;
	    ExtraVesaModes := ModeSet {272, 273, 275, 276};
	END (*IF*);
    END SpecificDriverChecks;

(************************************************************************)

PROCEDURE CheckVESADriver;

    (* Compiles a list of supported VESA modes, if any.	*)

    TYPE PointerToCardinal = POINTER TO CARDINAL;

    VAR Registers: RegisterPacket;
	ListPtr: PointerToCardinal;
	SysInfoPtr: POINTER TO
			    RECORD
				VesaSignature: ARRAY [0..3] OF CHAR;
				VesaVersion: WORD;
				OEM_OffsetString: ADDRESS;
				capabilities: LONGWORD;
				ModeListPointer: PointerToCardinal;
				NumberOf64KBlocks: CARDINAL;
				reserved: ARRAY [20..255] OF BYTE;
			    END (*RECORD*);

    (********************************************************************)

    PROCEDURE VesaSignatureAbsent(): BOOLEAN;

	BEGIN
	    WITH SysInfoPtr^ DO
		RETURN (VesaSignature[0] <> "V")
			OR (VesaSignature[1] <> "E")
			OR (VesaSignature[2] <> "S")
			OR (VesaSignature[3] <> "A");
	    END (*WITH*);
	END VesaSignatureAbsent;

    (********************************************************************)

    BEGIN
	(* Call the VESA BIOS subfunction 0, which is supposed to	*)
	(* return system information.  If this call fails, we presume	*)
	(* that there is no VESA driver present.			*)

	NEW(SysInfoPtr);
	WITH Registers DO
	    AX := 4F00H;
	    DI := OFFSET(SysInfoPtr);
	    ES := SEGMENT(SysInfoPtr);
	    BIOS(VideoInt,Registers);
	    IF (Registers.AX <> 004FH) OR VesaSignatureAbsent() THEN
		RETURN;
	    END (*IF*);
	END (*WITH*);

	(* We've found a VESA driver, now work out what modes it	*)
	(* supports.							*)

	AdaptorType := VESA;
	ListPtr := SysInfoPtr^.ModeListPointer;
	WHILE ListPtr^ <> 0FFFFH DO
	    VesaModes := VesaModes + ModeSet {ListPtr^};
	    ListPtr := AddOffset (ListPtr, 2);
	END (*WHILE*);
	ExtraVesaModes := ModeSet {};
	SpecificDriverChecks (SysInfoPtr^.OEM_OffsetString);
	SupportedModes := SupportedModes + VesaModes + ExtraVesaModes;
	SpecificDriverChecks (SysInfoPtr^.OEM_OffsetString);
	DISPOSE (SysInfoPtr);

    END CheckVESADriver;

(************************************************************************)
(*			    INITIALISATION				*)
(************************************************************************)

VAR Registers: RegisterPacket;
    p: POINTER TO CARDINAL;  dummy: BOOLEAN;

BEGIN
    VesaModes := ModeSet {};  ExtraVesaModes := ModeSet {};
    CurrentReadBank := 0;  CurrentWriteBank := 0;

    (* Read the low byte of the system equipment word at 0040:0010.	*)
    (* Bits 4 and 5 of this byte give the initial video mode: 00 is	*)
    (* unused, 01 or 10 implies a colour adaptor, 11 implies a		*)
    (* monochrome adaptor.						*)

    p := MakePointer (0040H, 0010H);
    IF IAND (p^, 30H) = 30H THEN

	(* Must be an MDA or Hercules *)

	AdaptorType := MDA;  ScreenSeg:= SEGMENT (Virtual(0B0000H));
	CRTCport := 03B4H;
	SupportedModes := ModeSet {0, 2, 7};
	CheckForHercules;

    ELSE

	(* Colour display present.  If there is an expansion ROM at	*)
	(* C000:0000, we assume that it's an EGA or better.		*)

	CRTCport := 03D4H;
	SupportedModes := ModeSet {0..7};
	p := MakePointer (0C000H, 0);
	IF p^ <> 0AA55H THEN
	    AdaptorType := CGA;
	    ScreenSeg:= SEGMENT (Virtual(0B8000H));
	ELSE
	    AdaptorType := EGA;
	    ScreenSeg:= SEGMENT (Virtual(0A0000H));
	    SupportedModes := SupportedModes + ModeSet {13..16};

	    (* It's at least EGA, test now for VGA.  If it passes that	*)
	    (* test, check for the presence of a VESA driver.		*)

	    Registers.AX := 1A00H;
	    BIOS (VideoInt, Registers);
	    IF Registers.AL = 26 THEN
		AdaptorType := VGA;
		SupportedModes := SupportedModes + ModeSet {17..19};
		CheckVESADriver;
	    END (*IF*);

	END (*IF*);

    END (*IF*);

    (* Find out what video mode is currently in effect, so that it can	*)
    (* be restored when we are done.					*)

    Registers.AH := 15;
    BIOS (VideoInt, Registers);
    OriginalMode := CARDINAL(Registers.AL);
    OriginalPage := Registers.BH;
    SetTerminationProcedure (RestoreOriginalMode);

    (* By setting that mode, we ensure that our global variables are	*)
    (* set to meaningful initial values.				*)

    dummy := SetVideoMode (OriginalMode, FALSE);

END Screen.
