IMPLEMENTATION MODULE Calculator;

	(********************************************************)
	(*							*)
	(*	Simple calculator (Algebraic notation version)	*)
	(*							*)
	(*  Programmer:		P. Moylan			*)
	(*  Last edited:	4 September 1993		*)
	(*  Status:		Working				*)
	(*							*)
	(********************************************************)

FROM Keyboard IMPORT
    (* proc *)	InKey, PutBack, LockStatus, SetLocks;

FROM KBdriver IMPORT
    (* const*)	NumLockLED;

FROM SoundEffects IMPORT
    (* proc *)	Beep;

FROM Windows IMPORT
    (* type *)	Window, Colour, FrameType, DividerType,
    (* proc *)	OpenWindow, CloseWindow, ChangeScrollingRegion,
		WriteChar, WriteString, SetCursor, ShiftWindow;

FROM Menus IMPORT
    (* type *)	Menu, ItemText,
    (* proc *)	CreateMenu, PositionMenu, SelectFromMenu, DestroyMenu;

FROM RealIO IMPORT
    (* proc *)	WriteLongReal;

FROM Str IMPORT
    (* proc *)	Copy;

FROM LowLevel IMPORT
    (* proc *)	IAND;

FROM MATHLIB IMPORT
    (* proc *)	Pow, Sqrt, Exp, Log, Log10, Sin, Cos,
		Tan, ASin, ACos, ATan, SinH, CosH, TanH;

(************************************************************************)
(*		    MISCELLANEOUS GLOBAL DEFINITIONS			*)
(************************************************************************)

TYPE CharSet = SET OF CHAR;

CONST
    EndMarker = '?';  Enter = CHR(0DH);
    UnaryOperatorSet = CharSet {'%', ')', 's', 'S'};
    UnknownOperatorPriority = 255;
    numberstart = 6;	(* the x cursor position for displaying numbers	*)
    numberwidth = 9;	(* The field size for displaying numbers	*)

VAR calc: Window;
    baserow, basecol: CARDINAL;

(************************************************************************)
(*			THE CALCULATOR STATE				*)
(************************************************************************)

CONST MaxRegisterNumber = 6;
      MaxMemoryNumber = 3;
      DisplayedRegisters = 3;

TYPE RegisterNumber = [0..MaxRegisterNumber];
     MemoryNumber = [0..MaxMemoryNumber];

VAR
    (* Array "Register" is a combined operand and operator stack.  It	*)
    (* would be more conventional to have separate stacks for the	*)
    (* operators and operands, but we adopt this slightly unusual stack	*)
    (* format because it makes it easier to maintain a user-friendly	*)
    (* screen display.							*)
    (* Note: Register[0].operator is never meaningful, except for	*)
    (* telling whether the stack is empty just after a PopStack.	*)

    Register: ARRAY RegisterNumber OF
		RECORD
		    ParenCount: CARDINAL;
		    value: LONGREAL;
		    operator: CHAR;
		END (*RECORD*);

    (* In addition to the stack, there is a set of "memory" registers	*)
    (* in which the user can save calculation results.			*)

    MemoryValue: ARRAY MemoryNumber OF LONGREAL;

(************************************************************************)
(*			THE PREFIX UNARY FUNCTIONS			*)
(************************************************************************)

(* To simplify the calculator logic, and to keep the display readable,	*)
(* all prefix unary functions are stored in the calculator stack in	*)
(* terms of the all-purpose binary function "f".  The first argument	*)
(* of f is the function number, and the second argument of f is the	*)
(* true argument of the original unary function.			*)

CONST
    MaxFunctionNumber = 13;	(* number of built-in functions		*)
    functionnamewidth = 5;	(* # of characters in a function name	*)

TYPE
    FunctionType = [0..MaxFunctionNumber];
    NameText = ARRAY [0..functionnamewidth-1] OF CHAR;
    NameArray = ARRAY FunctionType OF NameText;
    MathProc = PROCEDURE (LONGREAL): LONGREAL;
    FunctionArray = ARRAY FunctionType OF MathProc;

PROCEDURE Negative (x: LONGREAL): LONGREAL;

    BEGIN RETURN -x; END Negative;

CONST
    (* Array FunctionName gives the function names as displayed.	*)

    FunctionName = NameArray ('     ', 'SQRT ', 'EXP  ', 'LN   ', 'LOG10',
				'SIN  ', 'COS  ', 'TAN  ', 'ASIN ', 'ACOS ',
				'ATAN ', 'SINH ', 'COSH ', 'TANH ');

    (* Array Function is the set of built-in functions.			*)

    Function = FunctionArray (Negative, Sqrt, Exp, Log, Log10, Sin, Cos,
				Tan, ASin, ACos, ATan, SinH, CosH, TanH);

(************************************************************************)
(*			    DISPLAY ROUTINES				*)
(************************************************************************)

PROCEDURE LiteralAccumulatorDisplay (VAR (*IN*) Buffer: ARRAY OF CHAR);

    (* Displays "Buffer" as a text string at the screen location	*)
    (* reserved for the accumulator.					*)

    BEGIN
	SetCursor (calc, DisplayedRegisters+4, numberstart);
	WriteString (calc, Buffer);
    END LiteralAccumulatorDisplay;

(************************************************************************)

PROCEDURE WriteSpaces (N: CARDINAL);

    (* Writes a string of N spaces at the current cursor location.	*)

    VAR k: CARDINAL;

    BEGIN
	FOR k := 1 TO N DO
	    WriteChar (calc, " ");
	END (*FOR*);
    END WriteSpaces;

(************************************************************************)

PROCEDURE WriteParentheses (count: CARDINAL);

    (* Writes "count" left parentheses on the screen, with an	*)
    (* abbreviated display if count > 4.			*)

    VAR j: CARDINAL;

    BEGIN
	IF count > 4 THEN
	    WriteString (calc, "(..(");
	ELSE
	    FOR j := 1 TO count DO
		WriteChar (calc, "(");
	    END (*FOR*);
	    WriteSpaces (4-count);
	END (*IF*);
    END WriteParentheses;

(************************************************************************)

PROCEDURE DisplayAccumulator;

    (* Refreshes the display of the accumulator, i.e. Register[0].	*)
    (* This cannot be done by procedure DisplayRegister, because the	*)
    (* screen location would be wrong.					*)

    BEGIN
	SetCursor (calc, DisplayedRegisters+4, 1);
	WriteParentheses (Register[0].ParenCount);
	SetCursor (calc, DisplayedRegisters+4, numberstart);
	WriteLongReal (calc, Register[0].value, numberwidth);
    END DisplayAccumulator;

(************************************************************************)

PROCEDURE DisplayRegister (j: RegisterNumber);

    (* Refreshes the display of the left parentheses, the value, and	*)
    (* the trailing operator for register j.				*)

    BEGIN
	SetCursor (calc, DisplayedRegisters+3-j, 1);
	IF Register[j].operator = EndMarker THEN
	    WriteSpaces (numberstart+numberwidth+2);
	ELSE
	    WriteParentheses (Register[j].ParenCount);
	    SetCursor (calc, DisplayedRegisters+3-j, numberstart);
	    IF Register[j].operator = "f" THEN
		WriteSpaces (numberwidth - functionnamewidth);
		WriteString (calc,
				FunctionName[FunctionType(Register[j].value)]);
		IF Register[j].value = 0.0 THEN
		    WriteString (calc, " -");
		ELSE
		    WriteSpaces (2);
		END (*IF*);
	    ELSE
		WriteLongReal (calc, Register[j].value, numberwidth);
		WriteChar (calc, " ");  WriteChar (calc, Register[j].operator);
	    END (*IF*);
	END (*IF*);
    END DisplayRegister;

(************************************************************************)

PROCEDURE DisplayStack;

    (* Refreshes the display of the stack of registers.	*)

    VAR j: RegisterNumber;

    BEGIN
	DisplayAccumulator;
	FOR j := 1 TO DisplayedRegisters DO
	    DisplayRegister (j);
	END (*FOR*);
    END DisplayStack;

(************************************************************************)

PROCEDURE DisplayMemory (j: MemoryNumber);

    (* Refreshes the display of "memory" register j.	*)

    BEGIN
	SetCursor (calc, DisplayedRegisters+j+6, numberstart);
	WriteLongReal (calc, MemoryValue[j], numberwidth);
    END DisplayMemory;

(************************************************************************)

PROCEDURE InitialDisplay;

    (* Assumption: the calculator window calc is already open.  This	*)
    (* procedure puts the initial picture of the calculator onto the	*)
    (* screen.								*)

    CONST Corner1 = '';  Corner2 = '';  Corner3 = '';  Corner4 = '';
	horizontal = '';  vertical = '';

    VAR j: [1..numberwidth];  mem: MemoryNumber;

    BEGIN
	(* Draw a box for the accumulator.	*)

	SetCursor (calc, DisplayedRegisters+3, numberstart-1);
	WriteChar (calc, Corner1);
	FOR j := 1 TO numberwidth DO  WriteChar(calc, horizontal)  END(*FOR*);
	WriteChar (calc, Corner2);

	SetCursor (calc, DisplayedRegisters+4, numberstart-1);
	WriteChar (calc, vertical);
	SetCursor (calc, DisplayedRegisters+4, numberstart+numberwidth);
	WriteChar (calc, vertical);

	SetCursor (calc, DisplayedRegisters+5, numberstart-1);
	WriteChar (calc, Corner3);
	FOR j := 1 TO numberwidth DO  WriteChar(calc, horizontal)  END(*FOR*);
	WriteChar (calc, Corner4);

	(* Display the register contents.	*)

	DisplayStack;

	(* Display the memory values.	*)

	FOR mem := 0 TO MAX(MemoryNumber) DO
	    SetCursor (calc, DisplayedRegisters+mem+6, 2);
	    WriteChar (calc, "M");  WriteChar (calc, CHR(ORD("0")+mem));
	    DisplayMemory (mem);
	END (*FOR*);

    END InitialDisplay;

(************************************************************************)
(*			    NUMERIC INPUT				*)
(************************************************************************)

(************************************************************************)

PROCEDURE AcceptNumber (nextchar: CHAR);

    (* Reads a number from the keyboard.  On entry, nextchar holds the	*)
    (* first digit or the decimal point.  On exit, the input value is	*)
    (* in Register[0].value.						*)

    TYPE BufferSubscript = [1..numberwidth];

    VAR placevalue: LONGREAL;
	j: BufferSubscript;
	Buffer: ARRAY BufferSubscript OF CHAR;
	BufferFull: BOOLEAN;

    (********************************************************************)

    PROCEDURE GetNextChar;

	(* Displays the input so far (as a text string if it will fit,	*)
	(* otherwise by a call to WriteReal), and then reads nextchar.	*)

	BEGIN
	    IF BufferFull THEN
		DisplayAccumulator;
	    ELSE
		LiteralAccumulatorDisplay (Buffer);
	    END (*IF*);
	    nextchar := InKey();
	    IF NOT BufferFull THEN
		IF Buffer[1] <> " " THEN
		    BufferFull := TRUE;
		ELSE
		    FOR j := 1 TO numberwidth-1 DO
			Buffer[j] := Buffer[j+1];
		    END (*FOR*);
		    Buffer[numberwidth] := nextchar;
		END (*IF*);
	    END (*IF*);
	END GetNextChar;

    (********************************************************************)

    BEGIN
	Register[0].value := 0.0;  BufferFull := FALSE;
	FOR j := 1 TO numberwidth-1 DO
	    Buffer[j] := " ";
	END (*FOR*);
	Buffer[numberwidth] := nextchar;

	(* Read the part before the decimal point.	*)

	WITH Register[0] DO
	    WHILE nextchar IN CharSet {"0".."9"} DO
		value := 10.0*value + LONGREAL(ORD(nextchar) - ORD("0"));
		GetNextChar;
	    END (*WHILE*);
	END (*WITH*);

	(* Now the part after the decimal point, if any.	*)

	IF nextchar = "." THEN
	    GetNextChar;  placevalue := 0.1;
	    WHILE nextchar IN CharSet {"0".."9"} DO
		Register[0].value := Register[0].value
			+ placevalue*(LONGREAL(ORD(nextchar) - ORD("0")));
		placevalue := 0.1*placevalue;
		GetNextChar;
	    END (*WHILE*);
	END (*IF*);

	(* Correct for overshoot in input.	*)

	PutBack (nextchar);

    END AcceptNumber;

(************************************************************************)

PROCEDURE priority (operator: CHAR): CARDINAL;

    (* Returns the priority of an operator.	*)

    BEGIN
	CASE operator OF
		EndMarker:	RETURN 0;
	    |
		Enter,"=":	RETURN 1;
	    |
		"+","-":	RETURN 2;
	    |
		"*","/":	RETURN 3;
	    |
		"f":		IF Register[1].value = 0.0 THEN RETURN 7
				ELSE RETURN 4;
				END (*IF*);
	    |
		"x":		RETURN 5;
	    |
		"^":		RETURN 6;
	    |
		ELSE
				RETURN UnknownOperatorPriority;
	END (*CASE*);
    END priority;

(************************************************************************)

PROCEDURE TopOperatorPriority(): CARDINAL;

    (* TopOperatorPriority is normally the priority of the operator in	*)
    (* Register[1].  However any left parenthesis in Register[0]	*)
    (* overrides this; in that case we return an answer of 0.		*)

    BEGIN
	IF Register[0].ParenCount > 0 THEN RETURN 0
	ELSE RETURN priority (Register[1].operator)
	END (*IF*);
    END TopOperatorPriority;

(************************************************************************)
(*			    STACK MANIPULATION				*)
(************************************************************************)

PROCEDURE PushStack (LatestOperator: CHAR);

    (* Pushes the register stack, clearing the top one.  The argument	*)
    (* ends up as the operator in Register[1].  If the stack overflows	*)
    (* we give an audible alarm, but perform the push anyway.		*)

    VAR j: RegisterNumber;

    BEGIN
	Register[0].operator := LatestOperator;
	IF Register[MaxRegisterNumber].operator <> EndMarker THEN
	    Beep;
	END (*IF*);
	FOR j := MaxRegisterNumber TO 1 BY -1 DO
	    Register[j] := Register[j-1];
	END (*FOR*);
	WITH Register[0] DO
	    value := 0.0;  ParenCount := 0;
	END (*WITH*);
	DisplayStack;
    END PushStack;

(************************************************************************)

PROCEDURE PopStack;

    (* Pops the register stack, clearing the bottom register.	*)

    VAR j: RegisterNumber;

    BEGIN
	FOR j := 0 TO MaxRegisterNumber-1 DO
	    Register[j] := Register[j+1];
	END (*FOR*);
	WITH Register[MaxRegisterNumber] DO
	    ParenCount := 0;  value := 0.0;  operator := EndMarker;
	END (*WITH*);
	DisplayStack;
    END PopStack;

(************************************************************************)
(*			OPERATIONS ON THE MEMORIES			*)
(************************************************************************)

PROCEDURE GetMemoryNumber (): MemoryNumber;

    (* Returns the value of a one-digit memory number typed from the	*)
    (* keyboard.  Assumes memory number 0 (and does not consume the	*)
    (* typed key) if no valid memory number is specified.		*)

    VAR ch: CHAR;

    BEGIN
	SetCursor (calc, DisplayedRegisters+4, numberstart+numberwidth+1);
	WriteString (calc, "M#");
	ch := InKey();
	SetCursor (calc, DisplayedRegisters+4, numberstart+numberwidth+1);
	WriteString (calc, "  ");
	IF ch IN CharSet{"0"..CHR(ORD("0")+MaxMemoryNumber)} THEN
	    RETURN ORD(ch) - ORD("0");
	ELSE
	    PutBack(ch);  RETURN 0;
	END (*IF*);
    END GetMemoryNumber;

(************************************************************************)

PROCEDURE StoreToMemory;

    (* Gets a memory number from the keyboard, stores the accumulator	*)
    (* value in that memory register.					*)

    VAR mem: MemoryNumber;

    BEGIN
	mem := GetMemoryNumber();
	MemoryValue[mem] := Register[0].value;  DisplayMemory(mem);
    END StoreToMemory;

(************************************************************************)
(*				OPERATIONS				*)
(************************************************************************)

PROCEDURE Divide0 (first, second: LONGREAL): LONGREAL;

    (* Computes first/second, except that division by zero gives 0.0.	*)

    BEGIN
	IF second = 0.0 THEN RETURN 0.0
	ELSE RETURN first/second;
	END (*IF*);
    END Divide0;

(************************************************************************)

PROCEDURE BinaryOperation;

    (* Performs the binary operation requested by Register[1].operator.	*)

    VAR x, y, result: LONGREAL;
	command: CHAR;

    BEGIN
	command := Register[1].operator;
	x := Register[1].value;  y := Register[0].value;
	result := x;
	IF command = "+" THEN result := result + y
	ELSIF command = "-" THEN result := result - y
	ELSIF (command = "*") OR (command = "x") THEN result := result * y
	ELSIF command = "/" THEN result := Divide0 (x, y)
	ELSIF command = "^" THEN result := Pow (x, y)
	ELSIF command = "f" THEN
	    result := Function[VAL(FunctionType,x)] (y);
	ELSE Beep;
	END (*IF*);
	Register[1].value := result;
	PopStack;
    END BinaryOperation;

(************************************************************************)

PROCEDURE PostfixUnaryOperation (code: CHAR);

    (* Performs the unary operation requested by code.	*)

    BEGIN
	IF code = "%" THEN
	    Register[0].value := 0.01*Register[0].value*Register[1].value;
	ELSIF code = ")" THEN
	    IF Register[0].ParenCount > 0 THEN
		DEC (Register[0].ParenCount);
	    ELSIF Register[1].operator <> EndMarker THEN
		BinaryOperation;  PutBack (")");
	    ELSE
		Beep;
	    END (*IF*);
	ELSIF (code="s") OR (code="S") THEN
	    StoreToMemory;
	ELSE
	    Beep;
	END (*IF*);
	DisplayAccumulator;
    END PostfixUnaryOperation;

(************************************************************************)
(*		GETTING A FUNCTION NAME BY MENU SELECTION		*)
(************************************************************************)

PROCEDURE ReadBuiltinFunctionName;

    (* Allows the user to select a function name from a menu.  We then	*)
    (* load the stack with the function number, and the special		*)
    (* "binary operator" f.						*)

    VAR funcmenu: Menu;  menutext: ARRAY FunctionType OF ItemText;
	function: FunctionType;

    BEGIN
	menutext[0] := "    Function";
	FOR function := 1 TO MaxFunctionNumber DO
	    Copy (menutext[function], FunctionName[function]);
	END (*FOR*);
	CreateMenu (funcmenu, 3, menutext, MaxFunctionNumber);
	PositionMenu (funcmenu, white, blue, 14, 22, 60, 78);
	function := SelectFromMenu (funcmenu);
	DestroyMenu (funcmenu);
	IF function <> 0 THEN
	    Register[0].value := VAL(LONGREAL, function);
	    PushStack ("f");
	END (*IF*);
    END ReadBuiltinFunctionName;

(************************************************************************)
(*			THE CALCULATOR CONTROL LOGIC			*)
(************************************************************************)

PROCEDURE HandleFunctionKey;

    (* Looks after the cases where the keyboard input code was CHR(0).	*)
    (* In the present version, the arrow keys are looked after and all	*)
    (* other function keys are ignored.					*)

    VAR code: CHAR;

    BEGIN
	code := InKey();
	IF code = "H" THEN	(* cursor up *)
	    IF baserow > 0 THEN
		DEC(baserow);  ShiftWindow (calc, -1, 0);
	    END (*IF*);
	ELSIF code = "P" THEN	(* cursor down *)
	    IF baserow+DisplayedRegisters+MaxMemoryNumber+7 < 24 THEN
		INC(baserow);  ShiftWindow (calc, 1, 0);
	    END (*IF*);
	ELSIF code = "M" THEN	(* cursor right *)
	    IF basecol+numberstart+numberwidth+3 < 79 THEN
		INC(basecol);  ShiftWindow (calc, 0, 1);
	    END (*IF*);
	ELSIF code = "K" THEN	(* cursor left *)
	    IF basecol > 0 THEN
		DEC (basecol);  ShiftWindow (calc, 0, -1)
	    END (*IF*);
	END (*IF*);
    END HandleFunctionKey;

(************************************************************************)

PROCEDURE LoadAccumulator (VAR (*OUT*) nextchar: CHAR);

    (* Loads the accumulator with a number, also accepting and keeping	*)
    (* track of any opening parentheses.  Unary operations are also	*)
    (* dealt with by this procedure; and this could lead to the		*)
    (* evaluation of entire subexpressions, because we treat a closing	*)
    (* parenthesis as a unary postfix operator.  On return, nextchar	*)
    (* holds the following keyboard character (usually an operator, but	*)
    (* it could also be Esc, Return, or an illegal keystroke).  Most of	*)
    (* the complexity of this procedure lies in the fact that the user	*)
    (* can also type Backspace at any time, which has the effect of	*)
    (* cancelling the latest number, left parenthesis, or unevaluated	*)
    (* operator, as appropriate.					*)

    (* It is possible that the user will enter no value before the	*)
    (* operator.  In this case, the previous accumulator contents are	*)
    (* retained, unless they have been wiped out by a backspace.	*)

    CONST Backspace = CHR(8);
          Starters = CharSet {"(", ".", "0".."9", "f", "F", "m", "M",
				"p", "P"};
	  Misc = CharSet {CHR(0), "e", "E", EndMarker, Backspace};
	  HandledHere = Starters + Misc + UnaryOperatorSet;

    VAR NumberPresent: BOOLEAN;

    BEGIN
	NumberPresent := FALSE;
	LOOP
	    nextchar := InKey();

	    (* On seeing a "-", we have to decide whether it is a	*)
	    (* unary minus (if so, handle it here) or a binary minus.	*)

	    IF nextchar = "-" THEN
		IF NumberPresent THEN EXIT(*LOOP*)
		ELSE
		    Register[0].value := 0.0;
		    PushStack ("f");
		END (*IF*);

	    ELSIF NOT (nextchar IN HandledHere) THEN
		EXIT (*LOOP*);

	    (* Don't accept a number or similar if a number is	*)
	    (* already present.					*)

	    ELSIF NumberPresent AND (nextchar IN Starters) THEN Beep

	    (* Any character which, by coincidence, has the	*)
	    (* same character code as EndMarker is ignored.	*)

	    ELSIF nextchar = EndMarker THEN (* do nothing *)

	    (* Function key? *)

	    ELSIF nextchar = CHR(0) THEN HandleFunctionKey

	    (* Read prefix unary operator.  We don't evaluate it here;	*)
	    (* it's put on the stack to look like a binary operator.	*)

	    ELSIF CAP(nextchar) = "F" THEN ReadBuiltinFunctionName;

	    (* Handle postfix unary operator.	*)

	    ELSIF nextchar IN UnaryOperatorSet THEN
		PostfixUnaryOperation (nextchar);
		NumberPresent := TRUE;

	    (* Handle opening parenthesis.	*)

	    ELSIF nextchar = "(" THEN
		INC (Register[0].ParenCount);
		DisplayAccumulator;

	    (* P means the constant PI.	*)

	    ELSIF CAP(nextchar) = "P" THEN
		Register[0].value := 3.14159265359;
		DisplayAccumulator;
		NumberPresent := TRUE;

	    (* Fetch a number.	*)

	    ELSIF nextchar IN CharSet {"0".."9", "."} THEN
		AcceptNumber (nextchar);  NumberPresent := TRUE;

	    (* We use the calculator itself to evaluate "E" notation.	*)

	    ELSIF CAP(nextchar) = "E" THEN
		IF NumberPresent THEN
		    PushStack ("x");
		    Register[0].value := 10.0;
		    PutBack ("^");
		ELSE
		    Beep;
		END (*IF*);

	    (* Or an operand from memory.	*)

	    ELSIF (nextchar="m") OR (nextchar="M") THEN
		Register[0].value := MemoryValue[GetMemoryNumber()];
		DisplayAccumulator;
		NumberPresent := TRUE;

	    (* Now the hard part: handle Backspace.	*)

	    ELSIF nextchar = Backspace THEN

		(* The effect of a backspace depends on whether the	*)
		(* accumulator holds a user-supplied number at this	*)
		(* stage.  This depends on things like whether the	*)
		(* user has typed several backspaces in a row.		*)

		IF NumberPresent THEN

		    (* Delete the number in the accumulator.	*)

		    Register[0].value := 0.0;  NumberPresent := FALSE;
		    DisplayAccumulator;

		ELSIF Register[0].ParenCount > 0 THEN

		    (* Remove one left parenthesis.	*)

		    DEC (Register[0].ParenCount);
		    DisplayAccumulator;

		ELSE    (* Delete the last outstanding operator, if any. *)

		    PopStack;
		    NumberPresent := Register[0].operator <> EndMarker;

		END (*IF*);

	    END (*IF*);

	END (*LOOP*);

    END LoadAccumulator;

(************************************************************************)

PROCEDURE PerformCalculation;

    (* This procedure consists of a loop which is repeated until an	*)
    (* Esc character is encountered.  Each time around the loop, we	*)
    (* pick up an operand followed by an operator.  (Fetching the	*)
    (* operand, which is done by procedure LoadAccumulator, may itself	*)
    (* involve some subexpression evaluation, because the operand can	*)
    (* include things like opening and closing parentheses, prefix and	*)
    (* postfix functions, and the like.  Procedure LoadAccumulator also	*)
    (* allows some of the preceding input to be deleted via the		*)
    (* Backspace key.)  The operator may be a binary operator, or Esc,	*)
    (* or Enter, or '='.  (These last two are considered to be		*)
    (* equivalent.)  Anything else is considered to be an unknown	*)
    (* operator, and results in an audible Beep.			*)
    (* A calculation step, or possibly a whole sequence of steps, is	*)
    (* triggered if there are more closing parentheses than opening	*)
    (* parentheses, or if the operator has lower priority than the last	*)
    (* stacked operator.						*)

    CONST Esc = CHR(01BH);

    VAR operator: CHAR;

    BEGIN
	LOOP
	    LoadAccumulator (operator);

	    (* The Esc key drops us out of the calculator.	*)

	    IF operator = Esc THEN EXIT (*LOOP*) END (*IF*);

	    (* Perform any pending operations. *)

	    WHILE TopOperatorPriority() >= priority(operator) DO
		BinaryOperation;
	    END (*WHILE*);

	    (* Push the latest operator, unless it marks the end	*)
	    (* of the calculation.					*)

	    IF priority(operator) = UnknownOperatorPriority THEN
		Beep;
	    ELSIF (operator <> Enter) AND (operator <> "=") THEN
		PushStack (operator);
	    END(*IF*);

	END (*LOOP*);

    END PerformCalculation;

(************************************************************************)
(*			INTERFACE TO THE CALLER				*)
(************************************************************************)

PROCEDURE RunCalculator;

    (* Displays a calculator window on the screen; this can be operated	*)
    (* from the numeric keypad.  On exit, the screen window is closed,	*)
    (* but calculation results are saved for the next invocation of	*)
    (* this procedure.							*)

    VAR KeyboardLocks: CARDINAL;

    BEGIN
	(* Set the NumLock state, if not already set.	*)

	KeyboardLocks := LockStatus();
	IF IAND (KeyboardLocks, NumLockLED) = 0 THEN
	    SetLocks (KeyboardLocks + NumLockLED);
	END (*IF*);

	OpenWindow (calc, yellow, blue,
			baserow, baserow+DisplayedRegisters+MaxMemoryNumber+7,
			    basecol, basecol+numberstart+numberwidth+3,
				simpleframe, doubledivider);
	WriteString (calc, "   Calculator");
	ChangeScrollingRegion (calc, 3, DisplayedRegisters+MaxMemoryNumber+6);
	InitialDisplay;
	PerformCalculation;
	CloseWindow (calc);

	SetLocks (KeyboardLocks);

    END RunCalculator;

(************************************************************************)
(*			    INITIALISATION				*)
(************************************************************************)

PROCEDURE ClearCalculatorState;

    (* Clears all of the working registers of the calculator.	*)

    VAR j: RegisterNumber;  mem: MemoryNumber;

    BEGIN
	FOR j := 0 TO MaxRegisterNumber DO
	    WITH Register[j] DO
		ParenCount := 0;
		value := 0.0;
		operator := EndMarker;
	    END (*WITH*);
	END (*FOR*);
	FOR mem := 0 TO MAX(MemoryNumber) DO
	    MemoryValue[mem] := 0.0;
	END (*FOR*);
    END ClearCalculatorState;

(************************************************************************)

BEGIN
    baserow := 0;  basecol := 60;
    ClearCalculatorState;
END Calculator.
