/*
 *	Intel MP v1.1 specification support routines for multi-pentium 
 *	hosts.
 */

#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/config.h>
#include <linux/timer.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/delay.h>
#include <asm/i82489.h>
#include <asm/smp.h>

extern void *vremap(unsigned long offset, unsigned long size);	/* Linus hasnt put this in the headers yet */

static int smp_found_config=0;

static unsigned long cpu_present_map = 0;
static volatile unsigned long cpu_callin_map[32] = {0,};		/* 0 when not testing */
static unsigned int num_processors = 0;
static unsigned long io_apic_addr = 0;
static unsigned int boot_cpu_id = 0;
static unsigned char *kstack_base,*kstack_end;

/* 
 *	Checksum an MP configuration block.
 */
 
static int mpf_checksum(unsigned char *mp, int len)
{
	int sum=0;
	while(len--)
		sum+=*mp++;
	return sum&0xFF;
}

/*
 *	Processor encoding
 */
 
static char *mpc_family(int family,int model)
{
	static char n[32];
	static char *model_defs[]=
	{
		"80486DX","80486DX",
		"80486SX","80486DX/2 or 80487",
		"80486SL","Intel5X2(tm)",
		"Unknown","Unknown",
		"80486DX/4"
	};
	if(family==0x5)
		return("Pentium(tm)");
	if(family==0x0F && model==0x0F)
		return("Special controller");
	if(family==0x04 && model<9)
		return model_defs[model];
	sprintf(n,"Unknown CPU [%d:%d]",family, model);
	return n;
}

/*
 *	Read the MPC
 */

static int smp_read_mpc(struct mp_config_table *mpc)
{
	char str[16];
	int count=sizeof(*mpc);
	int apics=0;
	unsigned char *mpt=((unsigned char *)mpc)+count;

	if(memcmp(mpc->mpc_signature,MPC_SIGNATURE,4))
	{
		printk("Bad signature [%c%c%c%c].\n",
			mpc->mpc_signature[0],
			mpc->mpc_signature[1],
			mpc->mpc_signature[2],
			mpc->mpc_signature[3]);
		return 1;
	}
	if(mpf_checksum((unsigned char *)mpc,mpc->mpc_length))
	{
		printk("Checksum error.\n");
		return 1;
	}
	if(mpc->mpc_spec!=0x01)
	{
		printk("Unsupported version (%d)\n",mpc->mpc_spec);
		return 1;
	}
	memcpy(str,mpc->mpc_oem,8);
	str[8]=0;
	printk("OEM ID: %s ",str);
	memcpy(str,mpc->mpc_productid,12);
	str[12]=0;
	printk("Product ID: %s ",str);
	printk("APIC at: 0x%lX\n",mpc->mpc_lapic);
	
	/*
	 *	Now process the configuration blocks.
	 */
	 
	while(count<mpc->mpc_length)
	{
		switch(*mpt)
		{
			case MP_PROCESSOR:
			{
				struct mpc_config_processor *m=
					(struct mpc_config_processor *)mpt;
				if(m->mpc_cpuflag&CPU_ENABLED)
				{
					printk("Processor #%d %s APIC version %d\n",
						m->mpc_apicid, 
						mpc_family((m->mpc_cpufeature&
							CPU_FAMILY_MASK)>>8,
							(m->mpc_cpufeature&
								CPU_MODEL_MASK)>>4),
						m->mpc_apicver);
					if(m->mpc_featureflag&(1<<0))
						printk("    Floating point unit present.\n");
					if(m->mpc_featureflag&(1<<7))
						printk("    Machine Exception supported.\n");
					if(m->mpc_featureflag&(1<<8))
						printk("    64 bit compare & exchange supported.\n");
					if(m->mpc_featureflag&(1<<9))
						printk("    Internal APIC present.\n");
					if(m->mpc_cpuflag&CPU_BOOTPROCESSOR)
					{
						printk("    Bootup CPU\n");
						boot_cpu_id=m->mpc_apicid;
					}
					else	/* Boot CPU already counted */
						num_processors++;
						
					if(m->mpc_apicid>31)
						printk("Processor #%d unused. (Max 32 processors).\n",m->mpc_apicid);
					else						
						cpu_present_map|=(1<<m->mpc_apicid);
				}
				mpt+=sizeof(*m);
				count+=sizeof(*m);
				break;
			}
			case MP_BUS:
			{
				struct mpc_config_bus *m=
					(struct mpc_config_bus *)mpt;
				memcpy(str,m->mpc_bustype,6);
				str[6]=0;
				printk("Bus #%d is %s\n",
					m->mpc_busid,
					str);
				mpt+=sizeof(*m);
				count+=sizeof(*m);
				break; 
			}
			case MP_IOAPIC:
			{
				struct mpc_config_ioapic *m=
					(struct mpc_config_ioapic *)mpt;
				if(m->mpc_flags&MPC_APIC_USABLE)
				{
					apics++;
	                                printk("I/O APIC #%d Version %d at 0x%lX.\n",
	                                	m->mpc_apicid,m->mpc_apicver,
	                                	m->mpc_apicaddr);
	                                io_apic_addr = m->mpc_apicaddr;
	                        }
                                mpt+=sizeof(*m);
                                count+=sizeof(*m); 
                                break;
			}
			case MP_INTSRC:
			{
				struct mpc_config_intsrc *m=
					(struct mpc_config_intsrc *)mpt;
				
				mpt+=sizeof(*m);
				count+=sizeof(*m);
				break;
			}
			case MP_LINTSRC:
			{
				struct mpc_config_intlocal *m=
					(struct mpc_config_intlocal *)mpt;
				mpt+=sizeof(*m);
				count+=sizeof(*m);
				break;
			}
		}
	}
	if(apics>1)
		printk("Warning: Multiple APIC's not supported.\n");
	return num_processors;				
}

/*
 *	Scan the memory blocks for an SMP configuration block.
 */
 
void smp_scan_config(unsigned long base, unsigned long length)
{
	unsigned long *bp=(unsigned long *)base;
	struct intel_mp_floating *mpf;
	num_processors = 1;		/* The boot processor */
	
/*	printk("Scan SMP from %p for %ld bytes.\n",
		bp,length);*/
	if(sizeof(*mpf)!=16)
		printk("Error: MPF size\n");
	
	while(length>0)
	{
		if(*bp==SMP_MAGIC_IDENT)
		{
			mpf=(struct intel_mp_floating *)bp;
			if(mpf->mpf_length==1 && 
				!mpf_checksum((unsigned char *)bp,16) &&
				mpf->mpf_specification==1)
			{
				printk("Intel multiprocessing (MPv1.1) available.\n");
				if(mpf->mpf_feature2&(1<<7))
					printk("    IMCR and PIC mode supported.\n");
				smp_found_config=1;
				/*
				 *	Now see if we need to read further.
				 */
				if(mpf->mpf_feature1!=0)
				{
					num_processors=2;
					printk("I/O APIC at 0xFEC00000.\n");
					printk("Bus#0 is ");
				}
				switch(mpf->mpf_feature1)
				{
					case 1:
						printk("ISA");
						break;
					case 2:
						printk("EISA with no IRQ8 chaining");
						break;
					case 3:
						printk("EISA");
						break;
					case 4:
						printk("MCA");
						break;
					case 5:
						printk("ISA\nBus#1 is PCI");
						break;
					case 6:
						printk("EISA\nBus #1 is PCI");
						break;
					case 7:
						printk("MCA\nBus #1 is PCI");
						break;
					case 0:
						break;
					default:
						printk("???\nUnknown standard configuration %d\n",
							mpf->mpf_feature1);
						return;
				}
				if(mpf->mpf_physptr)
					smp_read_mpc((void *)mpf->mpf_physptr);
				printk("Processors: %d\n", num_processors);
			}
		}
		bp+=4;
		length-=16;
	}
}

/*
 *	Trampoline 80x86 program as an array.
 */

static unsigned char trampoline_data[]={ 
#include  "trampoline.hex"
};

/*
 *	Currently trivial. Write the real->protected mode
 *	bootstrap into the page concerned. The caller
 *	has made sure its suitably aligned.
 */
 
static void install_trampoline(unsigned char *mp)
{
	memcpy(mp,trampoline_data,sizeof(trampoline_data));
}

/*
 *	We are called very early to get the low memory for the trampoline/kernel stacks
 *	This has to be done by mm/init.c to parcel us out nice low memory. We allocate
 *	the kernel stacks at 4K, 8K, 12K... currently (0-03FF is preserved for SMM and
 *	other things).
 */
 
unsigned long smp_alloc_memory(unsigned long mem_base)
{
	int size=(num_processors-1)*PAGE_SIZE;		/* Number of stacks needed */
	/*
	 *	Our stacks have to be below the 1Mb line, and mem_base on entry
	 *	is 4K aligned.
	 */
	 
	if(mem_base+size>=0x9F000)
		panic("smp_alloc_memory: Insufficient low memory for kernel stacks.\n");
	kstack_base=(void *)mem_base;
	mem_base+=size;
	kstack_end=(void *)mem_base;
	return mem_base;
}
	
/*
 *	Hand out stacks one at a time.
 */
 
static void *get_kernel_stack(void)
{
	void *stack=kstack_base;
	if(kstack_base>=kstack_end)
		return NULL;
	kstack_base+=PAGE_SIZE;
	return stack;
}



unsigned char *apic_reg;			/* Set to the vremap() of the APIC */

unsigned char *kernel_stacks[32];		/* Kernel stack pointers for CPU's */
 
/*
 *	Architecture specific routine called by the kernel just before init is
 *	fired off. This allows the BP to have everything in order [we hope].
 *	At the end of this all the AP's will hit the system scheduling and off
 *	we go. Each AP will load the system gdt's and jump through the kernel
 *	init into idle(). At this point the scheduler will one day take over 
 * 	and give them jobs to do. smp_processor_callin is a private routine
 *	we use to track CPU's as they power up.
 */
 
void smp_callin(void)
{
	int cpuid=GET_APIC_ID(apic_read(APIC_ID));
/*
 *	This is safe because the other task is waiting for us until
 *	we set our flag, which is after we run printk.
 */	
	printk("OK.[SP=%p]\n",&cpuid);
/*
 *	Allow the master to continue.
 */	
	cpu_callin_map[0]|=(1<<cpuid);
	while(1);
}

void smp_boot_cpus(void)
{
	int i=0;
	int cpucount=0;
	void *stack;
	extern unsigned long init_user_stack[];
	
	
	/*
	 *	Map the local APIC into kernel space
	 */
	
	apic_reg = vremap(0xFEE00000,4096);
	
	
	if(apic_reg == NULL)
		panic("Unable to map local apic.\n");
		
	/*
	 *	Now scan the cpu present map and fire up anything we find.
	 */
	 
	 
	kernel_stacks[boot_cpu_id]=(void *)init_user_stack;	/* Set up for boot processor first */
	
	for(i=0;i<32;i++)
	{
		if((cpu_present_map&(1<<i)) && i!=boot_cpu_id)		/* Rebooting yourself is a bad move */
		{
			unsigned long cfg;
			int timeout;
			
			/*
			 *	We need a kernel stack for each processor.
			 */
			
			stack=get_kernel_stack();	/* We allocated these earlier */
			if(stack==NULL)
				panic("No memory for processor stacks.\n");
			kernel_stacks[i]=stack;
			install_trampoline(stack);
			
			printk("Booting processor %d stack %p: ",i,stack);			/* So we set whats up   */
				
			/*
			 *	Enable the local APIC
			 */
			 
			cfg=apic_read(APIC_SPIV);
			cfg|=(1<<8);		/* Enable APIC */
			apic_write(APIC_SPIV,cfg);
			
			/*
			 *	This gunge sends an IPI (Inter Processor Interrupt) to the
			 *	processor we wish to wake. When the startup IPI is received
			 *	the target CPU does a real mode jump to the stack base.
			 */
			
			cfg=apic_read(APIC_ICR2);
			cfg&=0x00FFFFFF;
			apic_write(APIC_ICR2, cfg|SET_APIC_DEST_FIELD(i));			/* Target chip     	*/
			cfg=apic_read(APIC_ICR);
			cfg&=~0xFDFFF	;							/* Clear bits 		*/
			cfg|=APIC_DEST_FIELD|APIC_DEST_DM_STARTUP|(((unsigned long)stack)>>12);	/* Boot on the stack 	*/		
			apic_write(APIC_ICR, cfg);						/* Kick the second 	*/
			udelay(10);								/* Masses of time 	*/
			cfg=apic_read(APIC_ESR);
			if(cfg&4)		/* Send accept error */
				printk("Processor refused startup request.\n");
			else
			{
				for(timeout=0;timeout<1000;timeout++)
				{
#ifdef SMP_DEBUG
					if(*vp==0xA5 && !f)
					{
						printk("Entered protected mode..");
						f=1;
					}
#endif					
					if(cpu_callin_map[0]&(1<<i))
						break;				/* It has booted */
					udelay(100);				/* Wait 100ms total for a response */
				}
				if(cpu_callin_map[0]&(1<<i))
				{
					cpucount++;
				}
				else
				{
					/*
					 *	At this point we should set up a BIOS warm start and try
					 *	a RESTART IPI. The 486+82489 MP pair don't support STARTUP IPI's
					 */
					printk("Not responding.\n");
					cpu_present_map&=~(1<<i);
				}
			}
		}
	}
	/*
	 *	Allow the user to impress friends.
	 */
	if(cpucount==0)
		printk("Error: only one processor found.\n");
	else
		printk("Total of %d processors activated.\n", cpucount+1);
}
